/*

erni -- run a process as a given user, with an option to chroot

erni can be used by system init scripts to run processes as 
a given user.  This is most useful when the old 

	su - <user> -c <command>

syntax won't work because <user>'s shell has been replaced, i.e.
for security reasons.


erni's core logic is based on the program "runas" which is included 
in the Titan distribution.  To see the original, please visit

	http://www.fish.com/titan

Truth be told, I was about to write this sort of tool and stumbled onto
Titan's "runas" before I'd fleshed out any of my own code.  No sense 
reinventing the wheel, right?  I used the Titan "runas" as a base, then 
built my desired features into it.
-qm


Enhancements to original runas.c, from the Titan distribution:

- different return codes for each error type
	If you exec this program from a script that interprets
	return codes, this is invaluable.

- distributed some of the functionality to functions
	This will simplify code maintenance and overall
	program cleanliness.

- introduced a new struct, "ConfigData," to hold configuration data
	This will simplify code maintenance and overall
	program cleanliness.  We can now pass the single
	"ConfigData" structure around to different functions
	and set/check all configuration params at once.

- use of getopt() for all commandline values, not just the chroot() dir.
	Streamlines program flow.

- use of Doxygen'ated comments throughout the header file
	Simplifies code maintenance -- maintainers can read an
	HTML'ized doc before diving into the source. 
	[see http://www.doxygen.com]

*/

/*
map of values

=============================================================
flag          cf file key         ConfigData member
=============================================================

-u <user>:<group>

              user                user_string

	run process as user "<user>" and group "<group>"

	If user_string is a number, it will be converted
	to a uid_t value; otherwise, we will attempt a
	lookup of the name using getpwnam().  In either
	case, the numeric value is stored in 
	ConfigData.user_uid.


              group               group_string

	If group_string is a number, it will be converted
	to a gid_t value; otherwise, we will attempt a
	lookup of the name using getgrnam().  In either
	case, the numeric value is stored in 
	ConfigData.primary_gid.

-------------------------------------------------------------

-m <mask>     mask [or "umask"]   umask_string

	run process with umask "<mask>"

	This will be converted to a number and stored
	in ConfigData.umask_num.

	This flag is optional; the default value
	is somewhat restrictive (027) but should
	be suitable for most usage.

-------------------------------------------------------------

-E <command>                      command_string

	execute "<command>"

	"-e"/"<command>" works only if the command in question contains
	no quote characters [used to group words, separated
	by spaces, into a single argument].  This is because
	the command string is parsed by string2array -- which
	defines a space as an agrument separator -- to turn 
	it into a command vector for use with the exec[n]() 
	family of functions.

	If such quoting is required, we provide the flag "-E."
	This must be the last argument on the commandline, as
	all arguments thereafter will be considered part of
	the command to run.

-------------------------------------------------------------

-c <chroot>   chroot              chroot_dir

	perform a chroot(2) to directory "<chroot>" before execution

-------------------------------------------------------------

-h            (none)              (none)
	show this help message

-------------------------------------------------------------

-v            (none)              (none)
	Show debug messages

-------------------------------------------------------------

*/


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>		/* dup() , chroot() , setgroups() , */
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>

#include <string.h>		/* strtok() , strerror() , strlen() , strchr() */


#include "Logger.h"		/* logging subsystem */
#include "utils.h"		/* utility functions not specific to Erni */
#include "erni_helpers.h"		/* project macro & function defs */
#include "erni.h"		/* macro and function defs specific to this binary */

extern int errno;

/** name of program */
char *myName_full = NULL ;

/** appwide logging subsystem */
Logger* logger ;

/* - - - - - - - - - - - - - - - - - - - - */


/**
\brief
program entry point


For maintainers, I provide a brief overview of the program flow:

main(){

- init a ConfigData struct, which will hold our configuration
  data and let us pass it around to various and sundry functions

- sanity check: are we running as root?

- parse commandline, placing data in our ConfigData struct

- readConfigFile(): parse a config file, if requested

- checkConfig(): confirm all necessary variables within 
  ConfigData are defined before we continue

- massageConfig(): convert certain strings in ConfigData to
  numbers, and fetch the uid/gid of the specified user
  name/group if words, rather than numbers, were specified.

- perform our chroot(), if requested

- clean up our environment and group memberships

- perform a setuid() to the specified user & set umask

- execute the specified command

- exit

}

*/

/* - - - - - - - - - - - - - - - - - - - - */

int main( int argc , char **argv , char **envp ){

	/* appwide configuration data */
	ConfigData cf = initConfigData() ; 

    myName_full = argv[0];

	/* - - - - - - - - - - - - - - -  */


	Logger_init() ;

	logger = Logger_getInstance() ;

    /* bail if we are not root */

	#ifndef MAINTAINER_MODE
	/* this test is disabled for dev builds ; *not* for production use!  */
    if ( 0 != geteuid() ){
		logger->error( "%s: must be run as root." , myName_full );
		exit( EXIT_ERROR_RUNASNONROOT );
    }
	#endif /* #ifndef MAINTAINER_MODE */

	if( 1 == argc ){
		showUsage() ;
		exit( EXIT_ERROR_NOARGS ) ;
	}


	/*
	disable buffering for Extreme Debug(tm)...
	This must be done before any operations are performed
	on the streams in question.

	The arg count and vector are shifted accordingly, so
	as to not upset getopt().
	*/
	if(
		argc > 1
		&&
		0 == strcmp( "NOBUFFER" , argv[1] )
	){

		setbuf( stdout , NULL ) ;
		setbuf( stderr , NULL ) ;
		++argv ;
		--argc ;

	}


	if( PROCESSARGS_SUCCESS != processArgs( argc , argv , &cf ) ){
		showUsage() ;
		exit( EXIT_ERROR_PROCESSARGS ) ;
	}

	if( READCONFIGFILE_SUCCESS != readConfigFile( &cf ) ){
		showUsage() ;
		exit( EXIT_ERROR_READCONFIGFILE ) ; 
	}

	if( CHECKCONFIG_SUCCESS != checkConfig( &cf ) ){
		showUsage() ;
		exit( EXIT_ERROR_CHECKCONFIG ) ; 
	}



	if( MASSAGECONFIG_SUCCESS != massageConfig( &cf ) ){

		showUsage() ;
		exit( EXIT_ERROR_MASSAGECONFIG ) ;

	}


	dumpConfig( &cf ) ;

	
	/*
	QQQ -- [fork opportunity #1]

	from here to the end should be a separate function.  
	this would also make the code more easily fork()'able, for
	future releases...
	*/

    /* chroot to the new location */
    if( CHROOT_YES == cf.use_chroot ){
		if( 0 > chroot(cf.chroot_dir) ) {
		    logger->error( "%s: unable to chroot to \"%s\": %s", myName_full , cf.chroot_dir, printError(errno) );
		    exit( EXIT_ERROR_CHROOT ) ; 
		}
    }


    /* zap all environment variables */
    envp[0] = NULL;


	switch( cf.which_grouplist ){

		case EXTRAGROUPS_USE_INITGROUPS:
	    	/* eliminate all supplementary groups */
			if( setgroups(0, NULL) < 0 ){
				logger->error( "%s: unable to eliminate supplementary groups: %s", myName_full , printError(errno) );
				exit( EXIT_ERROR_CLEAR_EXTRAGROUPS ) ; 
			}

			if( 0 != initgroups( cf.user_string , cf.primary_gid ) ){
				logger->error( "%s: call to initgroups( %s , %d ) fails: %s", myName_full , cf.user_string , cf.primary_gid , printError(errno) );
				exit( EXIT_ERROR_INITGROUPS ) ;
			}
			break ;

		case EXTRAGROUPS_USE_SETGROUPS:
			if( setgroups(0, NULL) < 0 ){
				logger->error( "%s: unable to eliminate supplementary groups: %s", myName_full , printError(errno) );
				exit( EXIT_ERROR_CLEAR_EXTRAGROUPS ) ; 
			}

			if( setgroups( cf.secondary_gids_length , cf.secondary_gids ) < 0 ){
				logger->error( "%s: unable to set supplementary groups: %s", myName_full , printError(errno) );
				exit( EXIT_ERROR_SET_EXTRAGROUPS ) ; 
			}
			break ;
			

		default:
			logger->error( "%s: logic error: can't tell how to load extra groups (flag is %d)" , myName_full , cf.which_grouplist ) ;
			exit( EXIT_ERROR_CONFUSEGROUPS ) ;

	} /* switch( cf.which_grouplist ) */


	/* switch GID */
	if( setgid( cf.primary_gid ) < 0 ){
		logger->error( "%s: unable to set group ID: %s", myName_full , printError(errno));
		exit( EXIT_ERROR_SGID ) ; 
	}

    /* check that we are not setting the UID to root (0) */
    if( cf.primary_gid == 0 ){
		logger->error( "%s: UID 0 (root) not allowed.", myName_full );
		exit( EXIT_ERROR_SUIDTOROOT ) ; 
    }


    /* switch UID; this must be done last or eliminating supplementary
	groups and switching GIDs will not work.
     */
    if( setuid( cf.user_uid ) < 0 ){
		logger->error( "%s: unable to set user ID: %s", myName_full , printError(errno));
		exit( EXIT_ERROR_SUID ) ; 
    }


    /* set the umask */
    umask( cf.umask_num );


	/*
	QQQ - [fork opportunity #2]

	...or, we could fork here:

	Child runs the execvp and checks for error;
	parent exits.

	That woud keep main() a little more cluttered but
	otherwise the code would flow more... maybe.
	*/

	/*
	fork a child process at the last possible minute -- this lets
	callers catch problems before stdout/stderr are closed, and
	thus log messages disappear
	*/

	if( FORK_YES == cf.call_fork ){

		logger->debug( "will attempt to fork() a child process" ) ;

		const pid_t pid = fork() ;

		switch( pid ){
			case -1: /* fork() problem */
				logger->error( "%s: Unable to fork child process: %s" , myName_full , printError(errno) );
				exit( EXIT_ERROR_FORK ) ;
				break ;

			case 0: /* child */

				/* basic post-fork() cleanup */

				/*
				Instead of closing stdout/stderr, they are redirected to
				/dev/null; this prevents any false positives in truss/strace
				output about writing to bad file descriptors
				*/

				/*
				Logger_setOutputDest( fopen( "/dev/null" , "w" ) ) ;
				Logger_setErrorDest( fopen( "/dev/null" , "w" ) ) ;
				*/

				freopen( "/dev/null" , "w" , stdout ) ;
				freopen( "/dev/null" , "w" , stderr ) ;
				freopen( "/dev/null" , "r" , stdin ) ;

				/* only fails if we're already group leader -- iow, not a problem */
				setsid() ;

				/*
				Notice, we do *not* chdir( "/" ), as recommended by some texts.
				What if the user explicitly calls the command from the current dir
				(aka "./command")?
				In this case, it's up to the user to call the command from "/"
				and avoid the problems of a daemon living in another mounted
				filesystem.
				*/
				break ;

			default: /* parent */
				logger->info( "%s: Command will run in a child process (pid %d)." , myName_full , pid ) ;
				exit( EXIT_SUCCESS ) ;

		} /* switch( pid ) */


	}else{
		logger->debug( "will NOT attempt to fork() a child process" ) ;
	}
	
    /* execute the command */
	logger->debug( "calling exec() on command" ) ;
    execvp( cf.command_vector[0] , &cf.command_vector[0] ) ; 


    /* this point is only reached if execvp() fails */
    logger->error( "%s: unable to execute command \"%s\": %s", myName_full , cf.command_string , printError(errno) ) ; 
	exit( EXIT_ERROR_EXECVP ) ; 

} /* main() */


/* - - - - - - - - - - - - - - - - - - - - */

void dumpConfig( const ConfigData *inConf ){

	const char* defaultPointer = "-" ;
	const char* defaultString = "[null]" ;


	/* used to walk the command vector array */
	int vec_ix ;

	/* logger = Logger_getInstance() ; */

	/* - - - - - - - - - - - - - - - */

	logger->trace( "dumpConfig( %p ): entered" , inConf ) ;

	logger->debug( "========================================" ) ; 

	logger->debug( "" ) ; 

	logger->debug(
		"- config file:         %-10p  %s" , 
		printNull( inConf->config_file , defaultPointer ) ,
		printNull( inConf->config_file , defaultString )
	) ; 

	logger->debug( "" ) ; 


	logger->debug(
		"- fork process?        %-10s  %s" , 
		"" , 
		( inConf->call_fork == FORK_YES ? "yes" : "no" )
	) ;


	logger->debug( "" ) ; 


	logger->debug(
		"- use chroot?          %-10s  %s" , 
		"" , 
		( inConf->use_chroot == CHROOT_YES ? "yes" : "no" )
	) ;


	logger->debug(
		"- chroot dir:          %-10p  %s" , 
		printNull( inConf->chroot_dir , defaultPointer ) ,
		printNull( inConf->chroot_dir , defaultString )
	) ;


	logger->debug( "" ) ; 

	logger->debug(
		"- user_string:         %-10p  %s" , 
		printNull( inConf->user_string , defaultPointer ) ,
		printNull( inConf->user_string , defaultString )
	) ;


	logger->debug(
		"- uid number:          %-10s  %d" ,
		"", 
		inConf->user_uid
	) ;


	logger->debug( "" ) ; 

	logger->debug(
		"- group_string:        %-10p  %s" , 
		printNull( inConf->group_string , defaultPointer ) ,
		printNull( inConf->group_string , defaultString )
	) ;


	logger->debug(
		"- gid number:          %-10s  %d" , 
		"", 
		inConf->primary_gid
	) ;


	logger->debug(
		"- %d secondary groups" ,
		inConf->secondary_gids_length
	) ;

	vec_ix = 0 ;

	while( vec_ix < inConf->secondary_gids_length ){

		logger->debug(
			"  %-2d:                              %d" , 
			vec_ix , 
			inConf->secondary_gids[vec_ix]
		) ;

		++vec_ix ;

	}

	logger->debug(
		"- how to load secondary groups:    "
	) ;

	switch( inConf->which_grouplist ){

		case EXTRAGROUPS_USE_INITGROUPS:
			logger->debug( "call initgroups()" ) ;
			break ;

		case EXTRAGROUPS_USE_SETGROUPS:
			logger->debug( "call setgroups()" ) ;
			break ;

		case EXTRAGROUPS_DONT_KNOW:
		default:
			logger->debug( "unknown (error!)" ) ;
			break ;

	} /* switch( inConf->which_grouplist ) */

	logger->debug( "" ) ; 


	logger->debug(
		"- umask_string:        %-10p  %s" , 
		printNull( inConf->umask_string , defaultPointer ) ,
		printNull( inConf->umask_string , defaultString )
	) ;


	logger->debug(
		"- umask:               %-10s  %o [octal]" , 
		"", 
		inConf->umask_num
	) ;

	logger->debug( "" ) ; 


	vec_ix = 0 ; 

	while( inConf->command_vector[vec_ix] != NULL ){

		logger->debug(
			"  vector  %-2d:          %-10p  %s" , 
			vec_ix , 
			inConf->command_vector[vec_ix] ,
			inConf->command_vector[vec_ix]
		) ;

		++vec_ix ;

	}

	logger->debug( "" ) ; 


	logger->debug(
		"- command:             %-10p  %s" , 
		printNull( inConf->command_string , defaultPointer ) ,
		printNull( inConf->command_string , defaultString )
	) ;

	logger->debug( "" ) ; 

	logger->debug( "========================================" ) ; 

	logger->trace( "dumpConfig(): ...returning to caller" ) ;

	return ;

} /* dumpConfig() */

/* - - - - - - - - - - - - - - - - - - - - */


void showUsage( void ) {

	/*
	we don't use the logging functions here because we want the
	message to be shown no matter what
	*/

	printf( "\n" ) ; 
	printf( "Usage:\n" ) ; 
	printf( "%s -u <user>[:group1,group2,...,groupN]\n" , myName_full ) ;
	printf( "\t[-f <file>] [-c <dir>] [-m <mask>]\n" ) ;
	printf( "\t[-h] [-v] [-V] -E <command>\n" ) ;
	printf( "\n" ) ; 

	printf( "Required flags:\n" ) ; 
	printf( "   -u <u>[:<g>]  run process as user \"<u>\"; Optionally specify\n" ) ;
	printf( "                 up to %d comma-separated groups as \"<g>\"\n" , GROUPLIST_SIZE ) ; 
	printf( "   -E <command>  command to execute; value must be unquoted, and this\n" ) ; 
	printf( "                 must be the last flag on command line)\n" ) ; 
	printf( "\n" ) ; 

	printf( "Optional flags:\n" ) ; 
	printf( "   -m <mask>     run process with umask \"<mask>\" (default \"%o\")\n" , DEFAULT_UMASK ) ;
	printf( "   -d            become a daemon -- that is, run the command in a fork()'d\n") ;
	printf( "                 child process\n" ) ;
	printf( "   -c <dir>      chroot(2) to \"<dir>\" before execution\n" ) ;
	printf( "   -f <file>     read values from \"<file>\"\n" ) ; 
	printf( "   -v            show debugging output\n" ) ; 
	printf( "   -V            print program version\n" ) ; 
	printf( "   -h            show this help message\n" ) ; 
	printf( "\n" ) ; 

	return ;

} /* showUsage */

/* - - - - - - - - - - - - - - - - - - - - */

void showVersion( void ){

	printf(
		"%s version %s, source date %s\n" ,
		PROGRAM_NAME ,
		PROGRAM_VERSION ,
		PROGRAM_SOURCE_DATE 
	) ;

} /* showVersion() */

/* - - - - - - - - - - - - - - - - - - - - */

/* EOF erni.c */
