# Crash Course: runtime reflection system

<!--
@cond TURN_OFF_DOXYGEN
-->
# Table of Contents

* [Introduction](#introduction)
* [Names and identifiers](#names-and-identifiers)
* [Reflection in a nutshell](#reflection-in-a-nutshell)
  * [Any to the rescue](#any-to-the-rescue)
  * [Enjoy the runtime](#enjoy-the-runtime)
  * [Container support](#container-support)
  * [Pointer-like types](#pointer-like-types)
  * [Template information](#template-information)
  * [Automatic conversions](#automatic-conversions)
  * [Implicitly generated default constructor](#implicitly-generated-default-constructor)
  * [From void to any](#from-void-to-any)
  * [Policies: the more, the less](#policies-the-more-the-less)
  * [Named constants and enums](#named-constants-and-enums)
  * [Properties and meta objects](#properties-and-meta-objects)
  * [Unregister types](#unregister-types)
  * [Meta context](#meta-context)
<!--
@endcond TURN_OFF_DOXYGEN
-->

# Introduction

Reflection (or rather, its lack) is a trending topic in the C++ world and a tool
that can unlock a lot of interesting features in the specific case of `EnTT`. I
looked for a third-party library that met my needs on the subject, but I always
came across some details that I didn't like: macros, being intrusive, too many
allocations, and so on.<br/>
I finally decided to write a built-in, non-intrusive and macro-free runtime
reflection system for `EnTT`. Maybe I didn't do better than others or maybe yes,
time will tell me, but at least I can model this tool around the library to
which it belongs and not the opposite.

# Names and identifiers

The meta system doesn't force users to rely on the tools provided by the library
when it comes to working with names and identifiers. It does this by offering an
API that works with opaque identifiers that may or may not be generated by means
of a hashed string.<br/>
This means that users can assign any type of identifier to the meta objects, as
long as they're numeric. It doesn't matter if they're generated at runtime, at
compile-time or with custom functions.

That being said, the examples in the following sections are all based on the
`hashed_string` class as provided by this library. Therefore, where an
identifier is required, it's likely that a user defined literal is used as
follows:

```cpp
auto factory = entt::meta<my_type>().type("reflected_type"_hs);
```

For what it's worth, this is completely equivalent to:

```cpp
auto factory = entt::meta<my_type>().type(42u);
```

Obviously, human-readable identifiers are more convenient to use and highly
recommended.

# Reflection in a nutshell

Reflection always starts from real types (users cannot reflect imaginary types
and it would not make much sense, we wouldn't be talking about reflection
anymore).<br/>
To create a meta node, the library provides the `meta` function that accepts a
type to reflect as a template parameter:

```cpp
auto factory = entt::meta<my_type>();
```

The returned value is a factory object to use to continue building the meta
type.

By default, a meta type is associated with the identifier returned by the
runtime type identification system built-in in `EnTT`.<br/>
However, it's also possible to assign custom identifiers to meta types:

```cpp
auto factory = entt::meta<my_type>().type("reflected_type"_hs);
```

Identifiers are important because users can retrieve meta types at runtime by
searching for them by _name_ other than by type.<br/>
On the other hand, there are cases in which users can be interested in adding
features to a reflected type so that the reflection system can use it correctly
under the hood, but they don't want to also make the type _searchable_. In this
case, it's sufficient not to invoke `type`.

A factory is such that all its member functions return the factory itself or a
decorated version of it. This object can be used to add the following:

* _Constructors_. Actual constructors can be assigned to a reflected type by
  specifying their list of arguments. Free functions (namely, factories) can be
  used as well, as long as the return type is the expected one. From a client's
  point of view, nothing changes if a constructor is a free function or an
  actual constructor.<br/>
  Use the `ctor` member function for this purpose:

  ```cpp
  entt::meta<my_type>().ctor<int, char>().ctor<&factory>();
  ```

* _Destructors_. Free functions and member functions can be used as destructors
  of reflected types. The purpose is to give users the ability to free up
  resources that require special treatment before an object is actually
  destroyed.<br/>
  Use the `dtor` member function for this purpose:

  ```cpp
  entt::meta<my_type>().dtor<&destroy>();
  ```

  A function should neither delete nor explicitly invoke the destructor of a
  given instance.

* _Data members_. Both real data members of the underlying type and static and
  global variables, as well as constants of any kind, can be attached to a meta
  type. From the point of view of the client, all the variables associated with
  the reflected type will appear as if they were part of the type itself.<br/>
  Use the `data` member function for this purpose:

  ```cpp
  entt::meta<my_type>()
      .data<&my_type::static_variable>("static"_hs)
      .data<&my_type::data_member>("member"_hs)
      .data<&global_variable>("global"_hs);
  ```

  The function requires as an argument the identifier to give to the meta data
  once created. Users can then access meta data at runtime by searching for them
  by _name_.<br/>
  Data members can also be defined by means of a setter and getter pair. Setters
  and getters can be either free functions, class members or a mix of them, as
  long as they respect the required signatures. This approach is also convenient
  to create a read-only variable from a non-const data member:

  ```cpp
  entt::meta<my_type>().data<nullptr, &my_type::data_member>("member"_hs);
  ```

  Multiple setters are also supported by means of a `value_list` object:

  ```cpp
  entt::meta<my_type>().data<entt::value_list<&from_int, &from_string>, &my_type::data_member>("member"_hs);
  ```

  Refer to the inline documentation for all the details.

* _Member functions_. Both real member functions of the underlying type and free
  functions can be attached to a meta type. From the point of view of the
  client, all the functions associated with the reflected type will appear as if
  they were part of the type itself.<br/>
  Use the `func` member function for this purpose:

  ```cpp
  entt::meta<my_type>()
      .func<&my_type::static_function>("static"_hs)
      .func<&my_type::member_function>("member"_hs)
      .func<&free_function>("free"_hs);
  ```

  The function requires as an argument the identifier to give to the meta
  function once created. Users can then access meta functions at runtime by
  searching for them by _name_.<br/>
  Overloading of meta functions is supported. Overloaded functions are resolved
  at runtime by the reflection system according to the types of the arguments.

* _Base classes_. A base class is such that the underlying type is actually
  derived from it. In this case, the reflection system tracks the relationship
  and allows for implicit casts at runtime when required.<br/>
  Use the `base` member function for this purpose:

  ```cpp
  entt::meta<derived_type>().base<base_type>();
  ```

  From now on, wherever a `base_type` is required, an instance of `derived_type`
  will also be accepted.

* _Conversion functions_. Actual types can be converted, this is a fact. Just
  think of the relationship between a `double` and an `int` to see it. Similar
  to bases, conversion functions allow users to define conversions that will be
  implicitly performed by the reflection system when required.<br/>
  Use the `conv` member function for this purpose:

  ```cpp
  entt::meta<double>().conv<int>();
  ```

That's all, everything users need to create meta types and enjoy the reflection
system. At first glance it may not seem that much, but users usually learn to
appreciate it over time.<br/>
Also, do not forget what these few lines hide under the hood: a built-in,
non-intrusive and macro-free system for reflection in C++. Features that are
definitely worth the price, at least for me.

## Any to the rescue

The reflection system offers a kind of _extended version_ of the `entt::any`
class (see the core module for more details).<br/>
The purpose is to add some feature on top of those already present, so as to
integrate it with the meta type system without having to duplicate the code.

The API is very similar to that of the `any` type. The class `meta_any` _wraps_
many of the feature to infer a meta node, before forwarding some or all of the
arguments to the underlying storage.<br/>
Among the few relevant differences, `meta_any` adds support for containers and
pointer-like types (see the following sections for more details), while `any`
does not.<br/>
Similar to `any`, this class can also be used to create _aliases_ for unmanaged
objects either with `forward_as_meta` or using the `std::in_place_type<T &>`
disambiguation tag, as well as from an existing object by means of the `as_ref`
member function. However, unlike `any`, `meta_any` treats an empty instance and
one initialized with `void` differently:

```cpp
entt::meta_any empty{};
entt::meta_any other{std::in_place_type<void>};
```

While `any` considers both as empty, `meta_any` treats objects initialized with
`void` as if they were _valid_ ones. This allows to differentiate between failed
function calls and function calls that are successful but return nothing.<br/>
Finally, the member functions `try_cast`, `cast` and `allow_cast` are used to
cast the underlying object to a given type (either a reference or a value type)
or to _convert_ a `meta_any` in such a way that a cast becomes viable for the
resulting object. There is in fact no `any_cast` equivalent for `meta_any`.

## Enjoy the runtime

Once the web of reflected types has been constructed, it's a matter of using it
at runtime where required.<br/>
All this has the great merit that the reflection system stands in fact as a
non-intrusive tool for the runtime, unlike the vast majority of the things
offered by this library and closely linked to the compile-time.

To search for a reflected type there are a few options:

```cpp
// direct access to a reflected type
auto by_type = entt::resolve<my_type>();

// look up a reflected type by identifier
auto by_id = entt::resolve("reflected_type"_hs);

// look up a reflected type by type info
auto by_type_id = entt::resolve(entt::type_id<my_type>());
```

There exists also an overload of the `resolve` function to use to iterate all
the reflected types at once. It returns an iterable object that can be used in a
range-for loop:

```cpp
for(auto &&[id, type]: entt::resolve()) {
    // ...
}
```

In all cases, the returned value is an instance of `meta_type` (possibly with
its id). This kind of objects offer an API to know their _runtime identifiers_,
to iterate all the meta objects associated with them and even to build instances
of the underlying type.<br/>
Refer to the inline documentation for all the details.

Meta data members and functions are accessed by name among the other things:

* Meta data members:

  ```cpp
  auto data = entt::resolve<my_type>().data("member"_hs);
  ```

  The returned type is `meta_data` and may be invalid if there is no meta data
  object associated with the given identifier.<br/>
  A meta data object offers an API to query the underlying type (for example, to
  know if it's a const or a static one), to get the meta type of the variable
  and to set or get the contained value.

* Meta function members:

  ```cpp
  auto func = entt::resolve<my_type>().func("member"_hs);
  ```

  The returned type is `meta_func` and may be invalid if there is no meta
  function object associated with the given identifier.<br/>
  A meta function object offers an API to query the underlying type (for
  example, to know if it's a const or a static function), to know the number of
  arguments, the meta return type and the meta types of the parameters. In
  addition, a meta function object can be used to invoke the underlying function
  and then get the return value in the form of a `meta_any` object.

All the meta objects thus obtained as well as the meta types can be explicitly
converted to a boolean value to check if they are valid:

```cpp
if(auto func = entt::resolve<my_type>().func("member"_hs); func) {
    // ...
}
```

Furthermore, all them (and a few more, like meta basis) are returned by a bunch
of overloads that provide the caller with iterable ranges of top-level elements.
As an example:

```cpp
for(auto &&[id, type]: entt::resolve<my_type>().base()) {
    // ...
}
```

A meta type can also be used to `construct` actual instances of the underlying
type.<br/>
In particular, the `construct` member function accepts a variable number of
arguments and searches for a match. It then returns a `meta_any` object that may
or may not be initialized, depending on whether a suitable constructor has been
found or not.

There is no object that wraps the destructor of a meta type nor a `destroy`
member function in its API. Destructors are invoked implicitly by `meta_any`
behind the scenes and users have not to deal with them explicitly. Furthermore,
they have no name, cannot be searched and wouldn't have member functions to
expose anyway.<br/>
Similarly, conversion functions aren't directly accessible. They are used
internally by `meta_any` and the meta objects when needed.

Meta types and meta objects in general contain much more than what is said: a
plethora of functions in addition to those listed whose purposes and uses go
unfortunately beyond the scope of this document.<br/>
I invite anyone interested in the subject to look at the code, experiment and
read the inline documentation to get the best out of this powerful tool.

## Container support

The runtime reflection system also supports containers of all types.<br/>
Moreover, _containers_ doesn't necessarily mean those offered by the C++
standard library. In fact, user defined data structures can also work with the
meta system in many cases.

To make a container be recognized as such by the meta system, users are required
to provide specializations for either the `meta_sequence_container_traits` class
or the `meta_associative_container_traits` class, according to the actual type
of the container.<br/>
`EnTT` already exports the specializations for some common classes. In
particular:

* `std::vector`, `std::array`, `std::deque` and `std::list` (but not
  `std::forward_list`) are supported as _sequence containers_.

* `std::map`, `std::set` and their unordered counterparts are supported as
  _associative containers_.

It's important to include the header file `container.hpp` to make these
specializations available to the compiler when needed.<br/>
The same file also contains many examples for the users that are interested in
making their own containers available to the meta system.

When a specialization of the `meta_sequence_container_traits` class exists, the
meta system treats the wrapped type as a sequence container. In a similar way,
a type is treated as an associative container if a specialization of the
`meta_associative_container_traits` class is found for it.<br/>
Proxy objects are returned by dedicated members of the `meta_any` class. The
following is a deliberately verbose example of how users can access a proxy
object for a sequence container:

```cpp
std::vector<int> vec{1, 2, 3};
entt::meta_any any = entt::forward_as_meta(vec);

if(any.type().is_sequence_container()) {
    if(auto view = any.as_sequence_container(); view) {
        // ...
    }
}
```

The method to use to get a proxy object for associative containers is
`as_associative_container` instead.<br/>
It goes without saying that it's not necessary to perform a double check.
Instead, it's sufficient to query the meta type or verify that the proxy object
is valid. In fact, proxies are contextually convertible to bool to know if they
are valid. For example, invalid proxies are returned when the wrapped object
isn't a container.<br/>
In all cases, users aren't expected to _reflect_ containers explicitly. It's
sufficient to assign a container for which a specialization of the traits
classes exists to a `meta_any` object to be able to get its proxy object.

The interface of the `meta_sequence_container` proxy object is the same for all
types of sequence containers, although the available features differ from case
to case. In particular:

* The `value_type` member function returns the meta type of the elements.

* The `size` member function returns the number of elements in the container as
  an unsigned integer value:

  ```cpp
  const auto size = view.size();
  ```

* The `resize` member function allows to resize the wrapped container and
  returns true in case of success:

  ```cpp
  const bool ok = view.resize(3u);
  ```

  For example, it's not possible to resize fixed size containers.

* The `clear` member function allows to clear the wrapped container and returns
  true in case of success:

  ```cpp
  const bool ok = view.clear();
  ```

  For example, it's not possible to clear fixed size containers.

* The `begin` and `end` member functions return opaque iterators that can be
  used to iterate the container directly:

  ```cpp
  for(entt::meta_any element: view) {
      // ...
  }
  ```

  In all cases, given an underlying container of type `C`, the returned element
  contains an object of type `C::value_type` which therefore depends on the
  actual container.<br/>
  All meta iterators are input iterators and don't offer an indirection operator
  on purpose.

* The `insert` member function can be used to add elements to the container. It
  accepts a meta iterator and the element to insert:

  ```cpp
  auto last = view.end();
  // appends an integer to the container
  view.insert(last, 42);
  ```

  This function returns a meta iterator pointing to the inserted element and a
  boolean value to indicate whether the operation was successful or not. Note
  that a call to `insert` may silently fail in case of fixed size containers or
  whether the arguments aren't at least convertible to the required types.<br/>
  Since the meta iterators are contextually convertible to bool, users can rely
  on them to know if the operation has failed on the actual container or
  upstream, for example for an argument conversion problem.

* The `erase` member function can be used to remove elements from the container.
  It accepts a meta iterator to the element to remove:

  ```cpp
  auto first = view.begin();
  // removes the first element from the container
  view.erase(first);
  ```

  This function returns a meta iterator following the last removed element and a
  boolean value to indicate whether the operation was successful or not. Note
  that a call to `erase` may silently fail in case of fixed size containers.

* The `operator[]` can be used to access elements in a container. It accepts a
  single argument, that is the position of the element to return:

  ```cpp
  for(std::size_t pos{}, last = view.size(); pos < last; ++pos) {
      entt::meta_any value = view[pos];
      // ...
  }
  ```

  The function returns instances of `meta_any` that directly refer to the actual
  elements. Modifying the returned object will then directly modify the element
  inside the container.<br/>
  Depending on the underlying sequence container, this operation may not be as
  efficient. For example, in the case of an `std::list`, a positional access
  translates to a linear visit of the list itself (probably not what the user
  expects).

Similarly, also the interface of the `meta_associative_container` proxy object
is the same for all types of associative containers. However, there are some
differences in behavior in the case of key-only containers. In particular:

* The `key_only` member function returns true if the wrapped container is a
  key-only one.

* The `key_type` member function returns the meta type of the keys.

* The `mapped_type` member function returns an invalid meta type for key-only
  containers and the meta type of the mapped values for all other types of
  containers.

* The `value_type` member function returns the meta type of the elements.<br/>
  For example, it returns the meta type of `int` for `std::set<int>` while it
  returns the meta type of `std::pair<const int, char>` for
  `std::map<int, char>`.

* The `size` member function returns the number of elements in the container as
  an unsigned integer value:

  ```cpp
  const auto size = view.size();
  ```

* The `clear` member function allows to clear the wrapped container and returns
  true in case of success:

  ```cpp
  const bool ok = view.clear();
  ```

* The `begin` and `end` member functions return opaque iterators that can be
  used to iterate the container directly:

  ```cpp
  for(std::pair<entt::meta_any, entt::meta_any> element: view) {
      // ...
  }
  ```

  In all cases, given an underlying container of type `C`, the returned element
  is a key-value pair where the key has type `C::key_type` and the value has
  type `C::mapped_type`. Since key-only containers don't have a mapped type,
  their _value_ is nothing more than an invalid `meta_any` object.<br/>
  All meta iterators are input iterators and don't offer an indirection operator
  on purpose.

  While the accessed key is usually constant in the associative containers and
  is therefore returned by copy, the value (if any) is wrapped by an instance of
  `meta_any` that directly refers to the actual element. Modifying it will then
  directly modify the element inside the container.

* The `insert` member function can be used to add elements to the container. It
  accepts two arguments, respectively the key and the value to be inserted:

  ```cpp
  auto last = view.end();
  // appends an integer to the container
  view.insert(last.handle(), 42, 'c');
  ```

  This function returns a boolean value to indicate whether the operation was
  successful or not. Note that a call to `insert` may fail when the arguments
  aren't at least convertible to the required types.

* The `erase` member function can be used to remove elements from the container.
  It accepts a single argument, that is the key to be removed:

  ```cpp
  view.erase(42);
  ```

  This function returns a boolean value to indicate whether the operation was
  successful or not. Note that a call to `erase` may fail when the argument
  isn't at least convertible to the required type.

* The `operator[]` can be used to access elements in a container. It accepts a
  single argument, that is the key of the element to return:

  ```cpp
  entt::meta_any value = view[42];
  ```

  The function returns instances of `meta_any` that directly refer to the actual
  elements. Modifying the returned object will then directly modify the element
  inside the container.

Container support is minimal but likely sufficient to satisfy all needs.

## Pointer-like types

As with containers, it's also possible to communicate to the meta system which
types to consider _pointers_. This will allow to dereference instances of
`meta_any`, thus obtaining light _references_ to the pointed objects that are
also correctly associated with their meta types.<br/>
To make the meta system recognize a type as _pointer-like_, users can specialize
the `is_meta_pointer_like` class. `EnTT` already exports the specializations for
some common classes. In particular:

* All types of raw pointers.
* `std::unique_ptr` and `std::shared_ptr`.

It's important to include the header file `pointer.hpp` to make these
specializations available to the compiler when needed.<br/>
The same file also contains many examples for the users that are interested in
making their own pointer-like types available to the meta system.

When a type is recognized as a pointer-like one by the meta system, it's
possible to dereference the instances of `meta_any` that contain these objects.
The following is a deliberately verbose example to show how to use this feature:

```cpp
int value = 42;
// meta type equivalent to that of int *
entt::meta_any any{&value};

if(any.type().is_pointer_like()) {
    // meta type equivalent to that of int
    if(entt::meta_any ref = *any; ref) {
        // ...
    }
}
```

Of course, it's not necessary to perform a double check. Instead, it's enough to
query the meta type or verify that the returned object is valid. For example,
invalid instances are returned when the wrapped object isn't a pointer-like
type.<br/>
Note that dereferencing a pointer-like object returns an instance of `meta_any`
which refers to the pointed object and allows users to modify it directly
(unless the returned element is const, of course).

In general, _dereferencing_ a pointer-like type boils down to a `*ptr`. However,
`EnTT` also supports classes that don't offer an `operator*`. In particular:

* It's possible to exploit a solution based on ADL lookup by offering a function
  (also a template one) named `dereference_meta_pointer_like`:

  ```cpp
  template<typename Type>
  Type & dereference_meta_pointer_like(const custom_pointer_type<Type> &ptr) {
      return ptr.deref();
  }
  ```

* When not in control of the type's namespace, it's possible to inject into the
  `entt` namespace a specialization of the `adl_meta_pointer_like` class
  template to bypass the adl lookup as a whole:

  ```cpp
  template<typename Type>
  struct entt::adl_meta_pointer_like<custom_pointer_type<Type>> {
      static decltype(auto) dereference(const custom_pointer_type<Type> &ptr) {
          return ptr.deref();
      }
  };
  ```

In all other cases, that is, when dereferencing a pointer works as expected and
regardless of the pointed type, no user intervention is required.

## Template information

Meta types also provide a minimal set of information about the nature of the
original type in case it's a class template.<br/>
By default, this works out of the box and requires no user action. However, it's
important to include the header file `template.hpp` to make this information
available to the compiler when needed.

Meta template information are easily found:

```cpp
// this method returns true if the type is recognized as a class template specialization
if(auto type = entt::resolve<std::shared_ptr<my_type>>(); type.is_template_specialization()) {
    // meta type of the class template conveniently wrapped by entt::meta_class_template_tag
    auto class_type = type.template_type();

    // number of template arguments
    std::size_t arity = type.template_arity();

    // meta type of the i-th argument
    auto arg_type = type.template_arg(0u);
}
```

Typically, when template information for a type are required, what the library
provides is sufficient. However, there are some cases where a user may want more
details or a different set of information.<br/>
Consider the case of a class template that is meant to wrap function types:

```cpp
template<typename>
struct function_type;

template<typename Ret, typename... Args>
struct function_type<Ret(Args...)> {};
```

In this case, rather than the function type, the user might want the return type
and unpacked arguments as if they were different template parameters for the
original class template.<br/>
To achieve this, users must enter the library internals and provide their own
specialization for the class template `entt::meta_template_traits`, such as:

```cpp
template<typename Ret, typename... Args>
struct entt::meta_template_traits<function_type<Ret(Args...)>> {
    using class_type = meta_class_template_tag<function_type>;
    using args_type = type_list<Ret, Args...>;
};
```

The reflection system doesn't verify the accuracy of the information nor infer a
correspondence between real types and meta types.<br/>
Therefore, the specialization will be used as is and the information it contains
will be associated with the appropriate type when required.

## Automatic conversions

In C++, there are a number of conversions allowed between arithmetic types that
make it convenient to work with this kind of data.<br/>
If this were to be translated into explicit registrations with the reflection
system, it would result in a long series of instructions such as the following:

```cpp
entt::meta<int>()
    .conv<bool>()
    .conv<char>()
    // ...
    .conv<double>();
```

Repeated for each type eligible to undergo this type of conversions. This is
both error-prone and repetitive.<br/>
Similarly, the language allows users to silently convert unscoped enums to their
underlying types and offers what it takes to do the same for scoped enums. It
would result in the following if it were to be done explicitly:

```cpp
entt::meta<my_enum>()
    .conv<std::underlying_type_t<my_enum>>();
```

Fortunately, all of this can also be avoided. `EnTT` offers implicit support for
these types of conversions:

```cpp
entt::meta_any any{42};
any.allow_cast<double>();
double value = any.cast<double>();
```

With no need for registration, the conversion takes place automatically under
the hood. The same goes for a call to `allow_cast` involving a meta type:

```cpp
entt::meta_type type = entt::resolve<int>();
entt::meta_any any{my_enum::a_value};
any.allow_cast(type);
int value = any.cast<int>();
```

This should make working with arithmetic types and scoped or unscoped enums as
easy as it is in C++.<br/>
It's also worth noting that it's still possible to set up conversion functions
manually and these will always be preferred over the automatic ones.

## Implicitly generated default constructor

In many cases, it's useful to be able to create objects of default constructible
types through the reflection system, while not having to explicitly register the
meta type or the default constructor.<br/>
For example, in the case of primitive types like `int` or `char`, but not just
them.

For this reason and only for default constructible types, default constructors
are automatically defined and associated with their meta types, whether they are
explicitly or implicitly generated.<br/>
Therefore, this is all is needed to construct an integer from its meta type:

```cpp
entt::resolve<int>().construct();
```

Where the meta type can be for example the one returned from a meta container,
useful for building keys without knowing or having to register the actual types.

In all cases, when users register default constructors, they are preferred both
during searches and when the `construct` member function is invoked.

## From void to any

Sometimes all a user has is an opaque pointer to an object of a known meta type.
It would be handy in this case to be able to construct a `meta_any` object from
them.<br/>
For this purpose, the `meta_type` class offers a `from_void` member function
designed to convert an opaque pointer into a `meta_any`:

```cpp
entt::meta_any any = entt::resolve(id).from_void(element);
```

It goes without saying that it's not possible to do a check on the actual type.
Therefore, this call can be considered as a _static cast_ with all the problems
and undefined behaviors of the case following errors.<br/>
On the other hand, the ability to construct a `meta_any` from an opaque pointer
opens the door to some pretty interesting uses that are worth exploring.

## Policies: the more, the less

Policies are a kind of compile-time directives that can be used when registering
reflection information.<br/>
Their purpose is to require slightly different behavior than the default in some
specific cases. For example, when reading a given data member, its value is
returned wrapped in a `meta_any` object which, by default, makes a copy of it.
For large objects or if the caller wants to access the original instance, this
behavior isn't desirable. Policies are there to offer a solution to this and
other problems.

There are a few alternatives available at the moment:

* The _as-is_ policy, associated with the type `entt::as_is_t`.<br/>
  This is the default policy. In general, it should never be used explicitly,
  since it's implicitly selected if no other policy is specified.<br/>
  In this case, the return values of the functions as well as the properties
  exposed as data members are always returned by copy in a dedicated wrapper and
  therefore associated with their original meta types.

* The _as-void_ policy, associated with the type `entt::as_void_t`.<br/>
  Its purpose is to discard the return value of a meta object, whatever it is,
  thus making it appear as if its type were `void`:

  ```cpp
  entt::meta<my_type>().func<&my_type::member_function, entt::as_void_t>("member"_hs);
  ```

  If the use with functions is obvious, it must be said that it's also possible
  to use this policy with constructors and data members. In the first case, the
  constructor will be invoked but the returned wrapper will actually be empty.
  In the second case, instead, the property will not be accessible for reading.

* The _as-ref_ and _as-cref_ policies, associated with the types
  `entt::as_ref_t` and `entt::as_cref_t`.<br/>
  They allow to build wrappers that act as references to unmanaged objects.
  Accessing the object contained in the wrapper for which the _reference_ was
  requested will make it possible to directly access the instance used to
  initialize the wrapper itself:

  ```cpp
  entt::meta<my_type>().data<&my_type::data_member, entt::as_ref_t>("member"_hs);
  ```

  These policies work with constructors (for example, when objects are taken
  from an external container rather than created on demand), data members and
  functions in general.<br/>
  If on the one hand `as_cref_t` always forces the return type to be const,
  `as_ref_t` _adapts_ to the constness of the passed object and to that of the
  return type if any.

Some uses are rather trivial, but it's useful to note that there are some less
obvious corner cases that can in turn be solved with the use of policies.

## Named constants and enums

A special mention should be made for constant values and enums. It wouldn't be
necessary, but it will help distracted readers.

As mentioned, the `data` member function can be used to reflect constants of any
type among the other things.<br/>
This allows users to create meta types for enums that will work exactly like any
other meta type built from a class. Similarly, arithmetic types can be enriched
with constants of special meaning where required.<br/>
Personally, I find it very useful not to export what is the difference between
enums and classes in C++ directly in the space of the reflected types.

All the values thus exported will appear to users as if they were constant data
members of the reflected types.

Exporting constant values or elements from an enum is as simple as ever:

```cpp
entt::meta<my_enum>()
    .data<my_enum::a_value>("a_value"_hs)
    .data<my_enum::another_value>("another_value"_hs);

entt::meta<int>().data<2048>("max_int"_hs);
```

It goes without saying that accessing them is trivial as well. It's a matter of
doing the following, as with any other data member of a meta type:

```cpp
auto value = entt::resolve<my_enum>().data("a_value"_hs).get({}).cast<my_enum>();
auto max = entt::resolve<int>().data("max_int"_hs).get({}).cast<int>();
```

As a side note, remember that all this happens behind the scenes without any
allocation because of the small object optimization performed by the `meta_any`
class.

## Properties and meta objects

Sometimes (for example, when it comes to creating an editor) it might be useful
to attach properties to the meta objects created. Fortunately, this is possible
for most of them.<br/>
For the meta objects that support properties, the member functions of the
factory used for registering them will return an extended version of the factory
itself. The latter can be used to attach properties to the last created meta
object.<br/>
Apparently, it's more difficult to say than to do:

```cpp
entt::meta<my_type>().type("reflected_type"_hs).prop("tooltip"_hs, "message");
```

Properties are always in the key/value form. The key is a numeric identifier,
mostly similar to the identifier used to register meta objects. There are no
restrictions on the type of the value instead, as long as it's movable.<br/>
Key only properties are also supported out of the box:

```cpp
entt::meta<my_type>().type("reflected_type"_hs).prop(my_enum::key_only);
```

To attach multiple properties to a meta object, it's possible to invoke `prop`
more than once.<br/>
It's also possible to invoke `prop` at different times, as long as the factory
is reset to the meta object of interest.

The meta objects for which properties are supported are currently meta types,
meta data and meta functions.<br/>
These types also offer a couple of member functions named `prop` to iterate all
properties at once or to search a specific property by key:

```cpp
// iterate all properties of a meta type
for(auto &&[id, prop]: entt::resolve<my_type>().prop()) {
    // ...
}

// search for a given property by name
auto prop = entt::resolve<my_type>().prop("tooltip"_hs);
```

Meta properties are objects having a fairly poor interface, all in all. They
only provide the `value` member function to retrieve the contained value in the
form of a `meta_any` object.

## Unregister types

A type registered with the reflection system can also be unregistered. This
means unregistering all its data members, member functions, conversion functions
and so on. However, base classes aren't unregistered as well, since they don't
necessarily depend on it.<br/>
Roughly speaking, unregistering a type means disconnecting all associated meta
objects from it and making its identifier no longer available:

```cpp
entt::meta_reset<my_type>();
```

It's also possible to reset types by their unique identifiers:

```cpp
entt::meta_reset("my_type"_hs);
```

Finally, there exists a non-template overload of the `meta_reset` function that
doesn't accept arguments and resets all meta types at once:

```cpp
entt::meta_reset();
```

A type can be re-registered later with a completely different name and form.

## Meta context

All meta types and their parts are created at runtime and stored in a default
_context_. This can be reached via a service locator as:

```cpp
auto &&context = entt::locator<entt::meta_context>::value_or();
```

By itself, a context is an opaque object that the user cannot do much with.
However, users can replace an existing context with another at any time:

```cpp
entt::meta_context other{};
auto &&context = entt::locator<entt::meta_context>::value_or();
std::swap(context, other);
```

This can be useful for testing purposes or to define multiple contexts with
different meta objects to be used as appropriate.

If _replacing_ the default context isn't enough, `EnTT` also offers the ability
to use multiple and externally managed contexts with the runtime reflection
system.<br/>
For example, to create new meta types within a context other than the default
one, simply pass it as an argument to the `meta` call:

```cpp
entt::meta_ctx context{};
auto factory = entt::meta<my_type>(context).type("reflected_type"_hs);
```

By doing so, the new meta type won't be available in the default context but
will be usable by passing around the new context when needed, such as when
creating a new `meta_any` object:

```cpp
entt::meta_any any{context, std::in_place_type<my_type>};
```

Similarly, to search for meta types in a context other than the default one, it
will be necessary to pass it to the `resolve` function:

```cpp
entt::meta_type type = entt::resolve(context, "reflected_type"_hs)
```

More generally, when using externally managed contexts, it's always required to
provide the system with the context to use, at least at the _entry point_.<br/>
For example, once the `meta_type` instant is obtained, it's no longer necessary
to pass the context around as the meta type takes the information with it and
eventually propagates it to all its parts.<br/>
On the other hand, it's necessary to instruct the library on where meta types
are to be fetched when `meta_any`s and `meta_handle`s are constructed, a factory
created or a meta type resolved.
