/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.dataframe.extractor;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.fieldcaps.FieldCapabilitiesResponse;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.document.DocumentField;
import org.elasticsearch.common.regex.Regex;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.fetch.subphase.FetchSourceContext;
import org.elasticsearch.xpack.core.ml.dataframe.DataFrameAnalyticsConfig;
import org.elasticsearch.xpack.core.ml.dataframe.DataFrameAnalyticsDest;
import org.elasticsearch.xpack.core.ml.dataframe.analyses.RequiredField;
import org.elasticsearch.xpack.core.ml.dataframe.analyses.Types;
import org.elasticsearch.xpack.core.ml.job.messages.Messages;
import org.elasticsearch.xpack.core.ml.utils.ExceptionsHelper;
import org.elasticsearch.xpack.core.ml.utils.NameResolver;
import org.elasticsearch.xpack.ml.datafeed.extractor.fields.ExtractedField;
import org.elasticsearch.xpack.ml.datafeed.extractor.fields.ExtractedFields;

public class ExtractedFieldsDetector {
    private static final Logger LOGGER = LogManager.getLogger(ExtractedFieldsDetector.class);
    private static final List<String> IGNORE_FIELDS = Arrays.asList("_id", "_field_names", "_index", "_parent", "_routing", "_seq_no", "_source", "_type", "_uid", "_version", "_feature", "_ignored", "ml__id_copy");
    private final String[] index;
    private final DataFrameAnalyticsConfig config;
    private final String resultsField;
    private final boolean isTaskRestarting;
    private final int docValueFieldsLimit;
    private final FieldCapabilitiesResponse fieldCapabilitiesResponse;

    ExtractedFieldsDetector(String[] index, DataFrameAnalyticsConfig config, String resultsField, boolean isTaskRestarting, int docValueFieldsLimit, FieldCapabilitiesResponse fieldCapabilitiesResponse) {
        this.index = Objects.requireNonNull(index);
        this.config = Objects.requireNonNull(config);
        this.resultsField = resultsField;
        this.isTaskRestarting = isTaskRestarting;
        this.docValueFieldsLimit = docValueFieldsLimit;
        this.fieldCapabilitiesResponse = Objects.requireNonNull(fieldCapabilitiesResponse);
    }

    public ExtractedFields detect() {
        Set<String> fields = this.getIncludedFields();
        if (fields.isEmpty()) {
            throw ExceptionsHelper.badRequestException((String)"No compatible fields could be detected in index {}. Supported types are {}.", (Object[])new Object[]{Arrays.toString(this.index), this.getSupportedTypes()});
        }
        this.checkNoIgnoredFields(fields);
        this.checkFieldsHaveCompatibleTypes(fields);
        this.checkRequiredFields(fields);
        return this.detectExtractedFields(fields);
    }

    private Set<String> getIncludedFields() {
        HashSet<String> fields = new HashSet<String>(this.fieldCapabilitiesResponse.get().keySet());
        this.removeFieldsUnderResultsField(fields);
        FetchSourceContext analyzedFields = this.config.getAnalyzedFields();
        if (analyzedFields == null || analyzedFields.includes().length == 0) {
            fields.removeAll(IGNORE_FIELDS);
            this.removeFieldsWithIncompatibleTypes(fields);
        }
        this.includeAndExcludeFields(fields);
        return fields;
    }

    private void removeFieldsUnderResultsField(Set<String> fields) {
        if (this.resultsField == null) {
            return;
        }
        this.checkResultsFieldIsNotPresent();
        fields.removeIf(field -> field.startsWith(this.resultsField + "."));
    }

    private void checkResultsFieldIsNotPresent() {
        if (this.isTaskRestarting) {
            return;
        }
        Map indexToFieldCaps = this.fieldCapabilitiesResponse.getField(this.resultsField);
        if (indexToFieldCaps != null && !indexToFieldCaps.isEmpty()) {
            throw ExceptionsHelper.badRequestException((String)"A field that matches the {}.{} [{}] already exists; please set a different {}", (Object[])new Object[]{DataFrameAnalyticsConfig.DEST.getPreferredName(), DataFrameAnalyticsDest.RESULTS_FIELD.getPreferredName(), this.resultsField, DataFrameAnalyticsDest.RESULTS_FIELD.getPreferredName()});
        }
    }

    private void removeFieldsWithIncompatibleTypes(Set<String> fields) {
        Iterator<String> fieldsIterator = fields.iterator();
        while (fieldsIterator.hasNext()) {
            String field = fieldsIterator.next();
            if (this.hasCompatibleType(field)) continue;
            fieldsIterator.remove();
        }
    }

    private boolean hasCompatibleType(String field) {
        Map fieldCaps = this.fieldCapabilitiesResponse.getField(field);
        if (fieldCaps == null) {
            LOGGER.debug("[{}] incompatible field [{}] because it is missing from mappings", (Object)this.config.getId(), (Object)field);
            return false;
        }
        Set<String> fieldTypes = fieldCaps.keySet();
        if (Types.numerical().containsAll(fieldTypes)) {
            LOGGER.debug("[{}] field [{}] is compatible as it is numerical", (Object)this.config.getId(), (Object)field);
            return true;
        }
        if (this.config.getAnalysis().supportsCategoricalFields() && Types.categorical().containsAll(fieldTypes)) {
            LOGGER.debug("[{}] field [{}] is compatible as it is categorical", (Object)this.config.getId(), (Object)field);
            return true;
        }
        if (ExtractedFieldsDetector.isBoolean(fieldTypes)) {
            LOGGER.debug("[{}] field [{}] is compatible as it is boolean", (Object)this.config.getId(), (Object)field);
            return true;
        }
        LOGGER.debug("[{}] incompatible field [{}]; types {}; supported {}", (Object)this.config.getId(), (Object)field, fieldTypes, this.getSupportedTypes());
        return false;
    }

    private Set<String> getSupportedTypes() {
        TreeSet<String> supportedTypes = new TreeSet<String>(Types.numerical());
        if (this.config.getAnalysis().supportsCategoricalFields()) {
            supportedTypes.addAll(Types.categorical());
        }
        supportedTypes.add("boolean");
        return supportedTypes;
    }

    private void includeAndExcludeFields(Set<String> fields) {
        FetchSourceContext analyzedFields = this.config.getAnalyzedFields();
        if (analyzedFields == null) {
            return;
        }
        String includes = analyzedFields.includes().length == 0 ? "*" : Strings.arrayToCommaDelimitedString((Object[])analyzedFields.includes());
        String excludes = Strings.arrayToCommaDelimitedString((Object[])analyzedFields.excludes());
        if (Regex.isMatchAllPattern((String)includes) && excludes.isEmpty()) {
            return;
        }
        try {
            SortedSet includedSet = NameResolver.newUnaliased(fields, ex -> new ResourceNotFoundException(Messages.getMessage((String)"No field [{0}] could be detected", (Object[])new Object[]{ex}), new Object[0])).expand(includes, false);
            SortedSet excludedSet = NameResolver.newUnaliased(fields, ex -> new ResourceNotFoundException(Messages.getMessage((String)"No field [{0}] could be detected", (Object[])new Object[]{ex}), new Object[0])).expand(excludes, true);
            fields.retainAll(includedSet);
            fields.removeAll(excludedSet);
        }
        catch (ResourceNotFoundException ex2) {
            throw ExceptionsHelper.badRequestException((String)ex2.getMessage(), (Object[])new Object[0]);
        }
    }

    private void checkNoIgnoredFields(Set<String> fields) {
        Optional<String> ignoreField = IGNORE_FIELDS.stream().filter(fields::contains).findFirst();
        if (ignoreField.isPresent()) {
            throw ExceptionsHelper.badRequestException((String)"field [{}] cannot be analyzed", (Object[])new Object[]{ignoreField.get()});
        }
    }

    private void checkFieldsHaveCompatibleTypes(Set<String> fields) {
        for (String field : fields) {
            Map fieldCaps = this.fieldCapabilitiesResponse.getField(field);
            if (fieldCaps == null) {
                throw ExceptionsHelper.badRequestException((String)"no mappings could be found for field [{}]", (Object[])new Object[]{field});
            }
            if (this.hasCompatibleType(field)) continue;
            throw ExceptionsHelper.badRequestException((String)"field [{}] has unsupported type {}. Supported types are {}.", (Object[])new Object[]{field, fieldCaps.keySet(), this.getSupportedTypes()});
        }
    }

    private void checkRequiredFields(Set<String> fields) {
        List requiredFields = this.config.getAnalysis().getRequiredFields();
        for (RequiredField requiredField : requiredFields) {
            Map fieldCaps = this.fieldCapabilitiesResponse.getField(requiredField.getName());
            if (!fields.contains(requiredField.getName()) || fieldCaps == null || fieldCaps.isEmpty()) {
                List requiredFieldNames = requiredFields.stream().map(RequiredField::getName).collect(Collectors.toList());
                throw ExceptionsHelper.badRequestException((String)"required field [{}] is missing; analysis requires fields {}", (Object[])new Object[]{requiredField.getName(), requiredFieldNames});
            }
            Set fieldTypes = fieldCaps.keySet();
            if (requiredField.getTypes().containsAll(fieldTypes)) continue;
            throw ExceptionsHelper.badRequestException((String)"invalid types {} for required field [{}]; expected types are {}", (Object[])new Object[]{fieldTypes, requiredField.getName(), requiredField.getTypes()});
        }
    }

    private ExtractedFields detectExtractedFields(Set<String> fields) {
        ArrayList<String> sortedFields = new ArrayList<String>(fields);
        Collections.sort(sortedFields);
        ExtractedFields extractedFields = ExtractedFields.build(sortedFields, Collections.emptySet(), this.fieldCapabilitiesResponse);
        if (extractedFields.getDocValueFields().size() > this.docValueFieldsLimit && (extractedFields = this.fetchFromSourceIfSupported(extractedFields)).getDocValueFields().size() > this.docValueFieldsLimit) {
            throw ExceptionsHelper.badRequestException((String)"[{}] fields must be retrieved from doc_values but the limit is [{}]; please adjust the index level setting [{}]", (Object[])new Object[]{extractedFields.getDocValueFields().size(), this.docValueFieldsLimit, IndexSettings.MAX_DOCVALUE_FIELDS_SEARCH_SETTING.getKey()});
        }
        extractedFields = this.fetchBooleanFieldsAsIntegers(extractedFields);
        return extractedFields;
    }

    private ExtractedFields fetchFromSourceIfSupported(ExtractedFields extractedFields) {
        ArrayList<ExtractedField> adjusted = new ArrayList<ExtractedField>(extractedFields.getAllFields().size());
        for (ExtractedField field : extractedFields.getDocValueFields()) {
            adjusted.add(field.supportsFromSource() ? field.newFromSource() : field);
        }
        return new ExtractedFields(adjusted);
    }

    private ExtractedFields fetchBooleanFieldsAsIntegers(ExtractedFields extractedFields) {
        ArrayList<ExtractedField> adjusted = new ArrayList<ExtractedField>(extractedFields.getAllFields().size());
        for (ExtractedField field : extractedFields.getAllFields()) {
            if (ExtractedFieldsDetector.isBoolean(field.getTypes())) {
                adjusted.add(new BooleanAsInteger(field));
                continue;
            }
            adjusted.add(field);
        }
        return new ExtractedFields(adjusted);
    }

    private static boolean isBoolean(Set<String> types) {
        return types.size() == 1 && types.contains("boolean");
    }

    private static class BooleanAsInteger
    extends ExtractedField {
        protected BooleanAsInteger(ExtractedField field) {
            super(field.getAlias(), field.getName(), Collections.singleton("boolean"), ExtractedField.ExtractionMethod.DOC_VALUE);
        }

        @Override
        public Object[] value(SearchHit hit) {
            DocumentField keyValue = hit.field(this.name);
            if (keyValue != null) {
                List<Object> values = keyValue.getValues().stream().map(v -> Boolean.TRUE.equals(v) ? 1 : 0).collect(Collectors.toList());
                return values.toArray(new Object[0]);
            }
            return new Object[0];
        }

        @Override
        public boolean supportsFromSource() {
            return false;
        }
    }
}

