/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.dataframe.extractor;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.search.ClearScrollAction;
import org.elasticsearch.action.search.ClearScrollRequest;
import org.elasticsearch.action.search.ClearScrollResponse;
import org.elasticsearch.action.search.SearchAction;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchScrollAction;
import org.elasticsearch.action.search.SearchScrollRequestBuilder;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.ElasticsearchClient;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.sort.SortOrder;
import org.elasticsearch.xpack.core.ClientHelper;
import org.elasticsearch.xpack.core.ml.dataframe.analyses.Types;
import org.elasticsearch.xpack.ml.datafeed.extractor.fields.ExtractedField;
import org.elasticsearch.xpack.ml.dataframe.extractor.DataFrameDataExtractorContext;

public class DataFrameDataExtractor {
    private static final Logger LOGGER = LogManager.getLogger(DataFrameDataExtractor.class);
    private static final TimeValue SCROLL_TIMEOUT = new TimeValue(30L, TimeUnit.MINUTES);
    private static final String EMPTY_STRING = "";
    private final Client client;
    private final DataFrameDataExtractorContext context;
    private String scrollId;
    private boolean isCancelled;
    private boolean hasNext;
    private boolean searchHasShardFailure;

    DataFrameDataExtractor(Client client, DataFrameDataExtractorContext context) {
        this.client = Objects.requireNonNull(client);
        this.context = Objects.requireNonNull(context);
        this.hasNext = true;
        this.searchHasShardFailure = false;
    }

    public Map<String, String> getHeaders() {
        return Collections.unmodifiableMap(this.context.headers);
    }

    public boolean hasNext() {
        return this.hasNext;
    }

    public boolean isCancelled() {
        return this.isCancelled;
    }

    public void cancel() {
        LOGGER.debug("[{}] Data extractor was cancelled", (Object)this.context.jobId);
        this.isCancelled = true;
    }

    public Optional<List<Row>> next() throws IOException {
        Optional<List<Row>> hits;
        if (!this.hasNext()) {
            throw new NoSuchElementException();
        }
        Optional<List<Row>> optional = hits = this.scrollId == null ? Optional.ofNullable(this.initScroll()) : Optional.ofNullable(this.continueScroll());
        if (!hits.isPresent()) {
            this.hasNext = false;
        }
        return hits;
    }

    protected List<Row> initScroll() throws IOException {
        LOGGER.debug("[{}] Initializing scroll", (Object)this.context.jobId);
        return this.tryRequestWithSearchResponse(() -> this.executeSearchRequest(this.buildSearchRequest()));
    }

    private List<Row> tryRequestWithSearchResponse(Supplier<SearchResponse> request) throws IOException {
        try {
            SearchResponse searchResponse = request.get();
            LOGGER.debug("[{}] Search response was obtained", (Object)this.context.jobId);
            this.searchHasShardFailure = false;
            return this.processSearchResponse(searchResponse);
        }
        catch (Exception e) {
            if (this.searchHasShardFailure) {
                throw e;
            }
            LOGGER.warn((Message)new ParameterizedMessage("[{}] Search resulted to failure; retrying once", (Object)this.context.jobId), (Throwable)e);
            this.markScrollAsErrored();
            return this.initScroll();
        }
    }

    protected SearchResponse executeSearchRequest(SearchRequestBuilder searchRequestBuilder) {
        return (SearchResponse)ClientHelper.executeWithHeaders(this.context.headers, (String)"ml", (Client)this.client, () -> ((SearchRequestBuilder)searchRequestBuilder).get());
    }

    private SearchRequestBuilder buildSearchRequest() {
        SearchRequestBuilder searchRequestBuilder = new SearchRequestBuilder((ElasticsearchClient)this.client, SearchAction.INSTANCE).setScroll(SCROLL_TIMEOUT).setAllowPartialSearchResults(false).addSort("ml__id_copy", SortOrder.ASC).setIndices(this.context.indices).setSize(this.context.scrollSize).setQuery(this.context.query);
        this.setFetchSource(searchRequestBuilder);
        for (ExtractedField docValueField : this.context.extractedFields.getDocValueFields()) {
            searchRequestBuilder.addDocValueField(docValueField.getName(), docValueField.getDocValueFormat());
        }
        return searchRequestBuilder;
    }

    private void setFetchSource(SearchRequestBuilder searchRequestBuilder) {
        if (this.context.includeSource) {
            searchRequestBuilder.setFetchSource(true);
        } else {
            String[] sourceFields = this.context.extractedFields.getSourceFields();
            if (sourceFields.length == 0) {
                searchRequestBuilder.setFetchSource(false);
                searchRequestBuilder.storedFields(new String[]{"_none_"});
            } else {
                searchRequestBuilder.setFetchSource(sourceFields, null);
            }
        }
    }

    private List<Row> processSearchResponse(SearchResponse searchResponse) throws IOException {
        this.scrollId = searchResponse.getScrollId();
        if (searchResponse.getHits().getHits().length == 0) {
            this.hasNext = false;
            this.clearScroll(this.scrollId);
            return null;
        }
        SearchHit[] hits = searchResponse.getHits().getHits();
        ArrayList<Row> rows = new ArrayList<Row>(hits.length);
        for (SearchHit hit : hits) {
            if (this.isCancelled) {
                this.hasNext = false;
                this.clearScroll(this.scrollId);
                break;
            }
            rows.add(this.createRow(hit));
        }
        return rows;
    }

    private Row createRow(SearchHit hit) {
        String[] extractedValues = new String[this.context.extractedFields.getAllFields().size()];
        for (int i = 0; i < extractedValues.length; ++i) {
            ExtractedField field = this.context.extractedFields.getAllFields().get(i);
            Object[] values = field.value(hit);
            if (values.length == 1 && (values[0] instanceof Number || values[0] instanceof String)) {
                extractedValues[i] = Objects.toString(values[0]);
                continue;
            }
            if (values.length == 0 && this.context.includeRowsWithMissingValues) {
                extractedValues[i] = EMPTY_STRING;
                continue;
            }
            extractedValues = null;
            break;
        }
        return new Row(extractedValues, hit);
    }

    private List<Row> continueScroll() throws IOException {
        LOGGER.debug("[{}] Continuing scroll with id [{}]", (Object)this.context.jobId, (Object)this.scrollId);
        return this.tryRequestWithSearchResponse(() -> this.executeSearchScrollRequest(this.scrollId));
    }

    private void markScrollAsErrored() {
        this.scrollId = null;
        this.searchHasShardFailure = true;
    }

    protected SearchResponse executeSearchScrollRequest(String scrollId) {
        return (SearchResponse)ClientHelper.executeWithHeaders(this.context.headers, (String)"ml", (Client)this.client, () -> (SearchResponse)new SearchScrollRequestBuilder((ElasticsearchClient)this.client, SearchScrollAction.INSTANCE).setScroll(SCROLL_TIMEOUT).setScrollId(scrollId).get());
    }

    private void clearScroll(String scrollId) {
        if (scrollId != null) {
            ClearScrollRequest request = new ClearScrollRequest();
            request.addScrollId(scrollId);
            ClientHelper.executeWithHeaders(this.context.headers, (String)"ml", (Client)this.client, () -> (ClearScrollResponse)this.client.execute((ActionType)ClearScrollAction.INSTANCE, (ActionRequest)request).actionGet());
        }
    }

    public List<String> getFieldNames() {
        return this.context.extractedFields.getAllFields().stream().map(ExtractedField::getAlias).collect(Collectors.toList());
    }

    public DataSummary collectDataSummary() {
        SearchRequestBuilder searchRequestBuilder = this.buildDataSummarySearchRequestBuilder();
        SearchResponse searchResponse = this.executeSearchRequest(searchRequestBuilder);
        return new DataSummary(searchResponse.getHits().getTotalHits().value, this.context.extractedFields.getAllFields().size());
    }

    public void collectDataSummaryAsync(ActionListener<DataSummary> dataSummaryActionListener) {
        SearchRequestBuilder searchRequestBuilder = this.buildDataSummarySearchRequestBuilder();
        int numberOfFields = this.context.extractedFields.getAllFields().size();
        ClientHelper.executeWithHeadersAsync(this.context.headers, (String)"ml", (Client)this.client, (ActionType)SearchAction.INSTANCE, (ActionRequest)((SearchRequest)searchRequestBuilder.request()), (ActionListener)ActionListener.wrap(searchResponse -> dataSummaryActionListener.onResponse((Object)new DataSummary(searchResponse.getHits().getTotalHits().value, numberOfFields)), arg_0 -> dataSummaryActionListener.onFailure(arg_0)));
    }

    private SearchRequestBuilder buildDataSummarySearchRequestBuilder() {
        return new SearchRequestBuilder((ElasticsearchClient)this.client, SearchAction.INSTANCE).setIndices(this.context.indices).setSize(0).setQuery(this.context.query).setTrackTotalHits(true);
    }

    public Set<String> getCategoricalFields() {
        HashSet<String> categoricalFields = new HashSet<String>();
        for (ExtractedField extractedField : this.context.extractedFields.getAllFields()) {
            String fieldName = extractedField.getName();
            if (!Types.categorical().containsAll(extractedField.getTypes())) continue;
            categoricalFields.add(fieldName);
        }
        return categoricalFields;
    }

    public static class Row {
        private SearchHit hit;
        @Nullable
        private String[] values;

        private Row(String[] values, SearchHit hit) {
            this.values = values;
            this.hit = hit;
        }

        @Nullable
        public String[] getValues() {
            return this.values;
        }

        public SearchHit getHit() {
            return this.hit;
        }

        public boolean shouldSkip() {
            return this.values == null;
        }

        public int getChecksum() {
            return Arrays.hashCode(this.values);
        }
    }

    public static class DataSummary {
        public final long rows;
        public final int cols;

        public DataSummary(long rows, int cols) {
            this.rows = rows;
            this.cols = cols;
        }
    }
}

