/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.core.indexing;

import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.xpack.core.indexing.IndexerJobStats;
import org.elasticsearch.xpack.core.indexing.IndexerState;
import org.elasticsearch.xpack.core.indexing.IterationResult;

public abstract class AsyncTwoPhaseIndexer<JobPosition, JobStats extends IndexerJobStats> {
    private static final Logger logger = LogManager.getLogger((String)AsyncTwoPhaseIndexer.class.getName());
    private final JobStats stats;
    private final AtomicReference<IndexerState> state;
    private final AtomicReference<JobPosition> position;
    private final Executor executor;

    protected AsyncTwoPhaseIndexer(Executor executor, AtomicReference<IndexerState> initialState, JobPosition initialPosition, JobStats jobStats) {
        this.executor = executor;
        this.state = initialState;
        this.position = new AtomicReference<JobPosition>(initialPosition);
        this.stats = jobStats;
    }

    public IndexerState getState() {
        return this.state.get();
    }

    public JobPosition getPosition() {
        return this.position.get();
    }

    public JobStats getStats() {
        return this.stats;
    }

    public synchronized IndexerState start() {
        this.state.compareAndSet(IndexerState.STOPPED, IndexerState.STARTED);
        return this.state.get();
    }

    public synchronized IndexerState stop() {
        return this.state.updateAndGet(previousState -> {
            if (previousState == IndexerState.INDEXING) {
                return IndexerState.STOPPING;
            }
            if (previousState == IndexerState.STARTED) {
                return IndexerState.STOPPED;
            }
            return previousState;
        });
    }

    public synchronized boolean abort() {
        IndexerState prevState = this.state.getAndUpdate(prev -> IndexerState.ABORTING);
        return prevState == IndexerState.STOPPED || prevState == IndexerState.STARTED;
    }

    public synchronized boolean maybeTriggerAsyncJob(long now) {
        IndexerState currentState = this.state.get();
        switch (currentState) {
            case INDEXING: 
            case STOPPING: 
            case ABORTING: {
                logger.warn("Schedule was triggered for job [" + this.getJobId() + "], but prior indexer is still running (with state [" + (Object)((Object)currentState) + "]");
                return false;
            }
            case STOPPED: {
                logger.debug("Schedule was triggered for job [" + this.getJobId() + "] but job is stopped.  Ignoring trigger.");
                return false;
            }
            case STARTED: {
                logger.debug("Schedule was triggered for job [" + this.getJobId() + "], state: [" + (Object)((Object)currentState) + "]");
                ((IndexerJobStats)this.stats).incrementNumInvocations(1L);
                if (this.state.compareAndSet(IndexerState.STARTED, IndexerState.INDEXING)) {
                    this.executor.execute(() -> this.onStart(now, (ActionListener<Boolean>)ActionListener.wrap(r -> {
                        assert (r != null);
                        if (r.booleanValue()) {
                            this.nextSearch((ActionListener<SearchResponse>)ActionListener.wrap(this::onSearchResponse, this::finishWithSearchFailure));
                        } else {
                            this.onFinish((ActionListener<Void>)ActionListener.wrap(onFinishResponse -> this.doSaveState(this.finishAndSetState(), this.position.get(), () -> {}), onFinishFailure -> this.doSaveState(this.finishAndSetState(), this.position.get(), () -> {})));
                        }
                    }, this::finishWithFailure)));
                    logger.debug("Beginning to index [" + this.getJobId() + "], state: [" + (Object)((Object)currentState) + "]");
                    return true;
                }
                logger.debug("Could not move from STARTED to INDEXING state because current state is [" + (Object)((Object)this.state.get()) + "]");
                return false;
            }
        }
        logger.warn("Encountered unexpected state [" + (Object)((Object)currentState) + "] while indexing");
        throw new IllegalStateException("Job encountered an illegal state [" + (Object)((Object)currentState) + "]");
    }

    protected abstract String getJobId();

    protected abstract IterationResult<JobPosition> doProcess(SearchResponse var1);

    protected abstract SearchRequest buildSearchRequest();

    protected abstract void onStart(long var1, ActionListener<Boolean> var3);

    protected abstract void doNextSearch(SearchRequest var1, ActionListener<SearchResponse> var2);

    protected abstract void doNextBulk(BulkRequest var1, ActionListener<BulkResponse> var2);

    protected abstract void doSaveState(IndexerState var1, JobPosition var2, Runnable var3);

    protected abstract void onFailure(Exception var1);

    protected abstract void onFinish(ActionListener<Void> var1);

    protected void onStop() {
    }

    protected abstract void onAbort();

    private void finishWithSearchFailure(Exception exc) {
        ((IndexerJobStats)this.stats).incrementSearchFailures();
        this.onFailure(exc);
        this.doSaveState(this.finishAndSetState(), this.position.get(), () -> {});
    }

    private void finishWithIndexingFailure(Exception exc) {
        ((IndexerJobStats)this.stats).incrementIndexingFailures();
        this.onFailure(exc);
        this.doSaveState(this.finishAndSetState(), this.position.get(), () -> {});
    }

    private void finishWithFailure(Exception exc) {
        this.onFailure(exc);
        this.finishAndSetState();
    }

    private IndexerState finishAndSetState() {
        AtomicBoolean callOnStop = new AtomicBoolean(false);
        AtomicBoolean callOnAbort = new AtomicBoolean(false);
        IndexerState updatedState = this.state.updateAndGet(prev -> {
            callOnAbort.set(false);
            callOnStop.set(false);
            switch (prev) {
                case INDEXING: {
                    return IndexerState.STARTED;
                }
                case STOPPING: {
                    callOnStop.set(true);
                    return IndexerState.STOPPED;
                }
                case ABORTING: {
                    callOnAbort.set(true);
                    return IndexerState.ABORTING;
                }
                case STOPPED: {
                    return IndexerState.STOPPED;
                }
            }
            throw new IllegalStateException("Indexer job encountered an illegal state [" + (Object)prev + "]");
        });
        if (callOnStop.get()) {
            this.onStop();
        } else if (callOnAbort.get()) {
            this.onAbort();
        }
        return updatedState;
    }

    private void onSearchResponse(SearchResponse searchResponse) {
        ((IndexerJobStats)this.stats).markEndSearch();
        try {
            if (!this.checkState(this.getState())) {
                return;
            }
            assert (searchResponse.getShardFailures().length == 0);
            ((IndexerJobStats)this.stats).incrementNumPages(1L);
            IterationResult<JobPosition> iterationResult = this.doProcess(searchResponse);
            if (iterationResult.isDone()) {
                logger.debug("Finished indexing for job [" + this.getJobId() + "], saving state and shutting down.");
                this.position.set(iterationResult.getPosition());
                this.onFinish((ActionListener<Void>)ActionListener.wrap(r -> this.doSaveState(this.finishAndSetState(), this.position.get(), () -> {}), e -> this.doSaveState(this.finishAndSetState(), this.position.get(), () -> {})));
                return;
            }
            List<IndexRequest> docs = iterationResult.getToIndex();
            if (!docs.isEmpty()) {
                BulkRequest bulkRequest = new BulkRequest();
                docs.forEach(arg_0 -> ((BulkRequest)bulkRequest).add(arg_0));
                ((IndexerJobStats)this.stats).markStartIndexing();
                this.doNextBulk(bulkRequest, (ActionListener<BulkResponse>)ActionListener.wrap(bulkResponse -> {
                    if (bulkResponse.hasFailures()) {
                        logger.warn("Error while attempting to bulk index documents: " + bulkResponse.buildFailureMessage());
                    }
                    ((IndexerJobStats)this.stats).incrementNumOutputDocuments(bulkResponse.getItems().length);
                    if (!this.checkState(this.getState())) {
                        return;
                    }
                    Object newPosition = iterationResult.getPosition();
                    this.position.set(newPosition);
                    this.onBulkResponse((BulkResponse)bulkResponse, newPosition);
                }, this::finishWithIndexingFailure));
            } else {
                try {
                    JobPosition newPosition = iterationResult.getPosition();
                    this.position.set(newPosition);
                    ActionListener listener = ActionListener.wrap(this::onSearchResponse, this::finishWithSearchFailure);
                    this.nextSearch((ActionListener<SearchResponse>)listener);
                }
                catch (Exception e2) {
                    this.finishWithFailure(e2);
                }
            }
        }
        catch (Exception e3) {
            this.finishWithSearchFailure(e3);
        }
    }

    private void onBulkResponse(BulkResponse response, JobPosition position) {
        ((IndexerJobStats)this.stats).markEndIndexing();
        try {
            ActionListener listener = ActionListener.wrap(this::onSearchResponse, this::finishWithSearchFailure);
            if (((IndexerJobStats)this.stats).getNumPages() > 0L && ((IndexerJobStats)this.stats).getNumPages() % 50L == 0L) {
                this.doSaveState(IndexerState.INDEXING, position, () -> this.nextSearch((ActionListener<SearchResponse>)listener));
            } else {
                this.nextSearch((ActionListener<SearchResponse>)listener);
            }
        }
        catch (Exception e) {
            this.finishWithIndexingFailure(e);
        }
    }

    private void nextSearch(ActionListener<SearchResponse> listener) {
        ((IndexerJobStats)this.stats).markStartSearch();
        SearchRequest searchRequest = this.buildSearchRequest().allowPartialSearchResults(false);
        this.doNextSearch(searchRequest, listener);
    }

    private boolean checkState(IndexerState currentState) {
        switch (currentState) {
            case INDEXING: {
                return true;
            }
            case STOPPING: {
                logger.info("Indexer job encountered [" + (Object)((Object)IndexerState.STOPPING) + "] state, halting indexer.");
                this.doSaveState(this.finishAndSetState(), this.getPosition(), () -> {});
                return false;
            }
            case STOPPED: {
                return false;
            }
            case ABORTING: {
                logger.info("Requested shutdown of indexer for job [" + this.getJobId() + "]");
                this.onAbort();
                return false;
            }
        }
        logger.warn("Encountered unexpected state [" + (Object)((Object)currentState) + "] while indexing");
        throw new IllegalStateException("Indexer job encountered an illegal state [" + (Object)((Object)currentState) + "]");
    }
}

