/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authc;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.crypto.SecretKeyFactory;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchSecurityException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionRunnable;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.bulk.BulkAction;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.bulk.TransportSingleItemBulkWriteAction;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.support.ContextPreservingActionListener;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.action.support.replication.ReplicatedWriteRequest;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.cache.RemovalListener;
import org.elasticsearch.common.cache.RemovalNotification;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.logging.DeprecationCategory;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.settings.SecureString;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.common.util.concurrent.FutureUtils;
import org.elasticsearch.common.util.concurrent.ListenableFuture;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.common.xcontent.AbstractObjectParser;
import org.elasticsearch.common.xcontent.ConstructingObjectParser;
import org.elasticsearch.common.xcontent.DeprecationHandler;
import org.elasticsearch.common.xcontent.InstantiatingObjectParser;
import org.elasticsearch.common.xcontent.LoggingDeprecationHandler;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.common.xcontent.ObjectParserHelper;
import org.elasticsearch.common.xcontent.ParseField;
import org.elasticsearch.common.xcontent.XContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.common.xcontent.XContentLocation;
import org.elasticsearch.common.xcontent.XContentParser;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.core.CharArrays;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.license.LicenseUtils;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.search.SearchService;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.core.ClientHelper;
import org.elasticsearch.xpack.core.XPackSettings;
import org.elasticsearch.xpack.core.security.ScrollHelper;
import org.elasticsearch.xpack.core.security.action.ApiKey;
import org.elasticsearch.xpack.core.security.action.ClearSecurityCacheAction;
import org.elasticsearch.xpack.core.security.action.ClearSecurityCacheRequest;
import org.elasticsearch.xpack.core.security.action.ClearSecurityCacheResponse;
import org.elasticsearch.xpack.core.security.action.CreateApiKeyRequest;
import org.elasticsearch.xpack.core.security.action.CreateApiKeyResponse;
import org.elasticsearch.xpack.core.security.action.GetApiKeyResponse;
import org.elasticsearch.xpack.core.security.action.InvalidateApiKeyResponse;
import org.elasticsearch.xpack.core.security.authc.Authentication;
import org.elasticsearch.xpack.core.security.authc.AuthenticationResult;
import org.elasticsearch.xpack.core.security.authc.support.Hasher;
import org.elasticsearch.xpack.core.security.authz.RoleDescriptor;
import org.elasticsearch.xpack.core.security.user.User;
import org.elasticsearch.xpack.security.Security;
import org.elasticsearch.xpack.security.authc.ExpiredApiKeysRemover;
import org.elasticsearch.xpack.security.support.CacheInvalidatorRegistry;
import org.elasticsearch.xpack.security.support.FeatureNotEnabledException;
import org.elasticsearch.xpack.security.support.LockingAtomicCounter;
import org.elasticsearch.xpack.security.support.SecurityIndexManager;

public class ApiKeyService {
    private static final Logger logger = LogManager.getLogger(ApiKeyService.class);
    private static final DeprecationLogger deprecationLogger = DeprecationLogger.getLogger(ApiKeyService.class);
    private static final Version METADATA_INTRODUCED = Version.V_7_13_0;
    public static final String API_KEY_ID_KEY = "_security_api_key_id";
    public static final String API_KEY_NAME_KEY = "_security_api_key_name";
    public static final String API_KEY_METADATA_KEY = "_security_api_key_metadata";
    public static final String API_KEY_REALM_NAME = "_es_api_key";
    public static final String API_KEY_REALM_TYPE = "_es_api_key";
    public static final String API_KEY_CREATOR_REALM_NAME = "_security_api_key_creator_realm_name";
    public static final String API_KEY_CREATOR_REALM_TYPE = "_security_api_key_creator_realm_type";
    public static final Setting<String> PASSWORD_HASHING_ALGORITHM = new Setting("xpack.security.authc.api_key.hashing.algorithm", "pbkdf2", Function.identity(), v -> {
        if (!Hasher.getAvailableAlgoStoredHash().contains(v.toLowerCase(Locale.ROOT))) {
            throw new IllegalArgumentException("Invalid algorithm: " + v + ". Valid values for password hashing are " + Hasher.getAvailableAlgoStoredHash().toString());
        }
        if (v.regionMatches(true, 0, "pbkdf2", 0, "pbkdf2".length())) {
            try {
                SecretKeyFactory.getInstance("PBKDF2withHMACSHA512");
            }
            catch (NoSuchAlgorithmException e) {
                throw new IllegalArgumentException("Support for PBKDF2WithHMACSHA512 must be available in order to use any of the PBKDF2 algorithms for the [xpack.security.authc.api_key.hashing.algorithm] setting.", e);
            }
        }
    }, new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<TimeValue> DELETE_TIMEOUT = Setting.timeSetting((String)"xpack.security.authc.api_key.delete.timeout", (TimeValue)TimeValue.MINUS_ONE, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<TimeValue> DELETE_INTERVAL = Setting.timeSetting((String)"xpack.security.authc.api_key.delete.interval", (TimeValue)TimeValue.timeValueHours((long)24L), (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<String> CACHE_HASH_ALGO_SETTING = Setting.simpleString((String)"xpack.security.authc.api_key.cache.hash_algo", (String)"ssha256", (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<TimeValue> CACHE_TTL_SETTING = Setting.timeSetting((String)"xpack.security.authc.api_key.cache.ttl", (TimeValue)TimeValue.timeValueHours((long)24L), (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<Integer> CACHE_MAX_KEYS_SETTING = Setting.intSetting((String)"xpack.security.authc.api_key.cache.max_keys", (int)25000, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<TimeValue> DOC_CACHE_TTL_SETTING = Setting.timeSetting((String)"xpack.security.authc.api_key.doc_cache.ttl", (TimeValue)TimeValue.timeValueMinutes((long)5L), (TimeValue)TimeValue.timeValueMinutes((long)0L), (TimeValue)TimeValue.timeValueMinutes((long)15L), (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    private static final BytesArray FLEET_SERVER_ROLE_DESCRIPTOR_BYTES_V_7_14 = new BytesArray("{\"elastic/fleet-server\":{\"cluster\":[\"monitor\",\"manage_own_api_key\"],\"indices\":[{\"names\":[\"logs-*\",\"metrics-*\",\"traces-*\",\"synthetics-*\",\".logs-endpoint.diagnostic.collection-*\"],\"privileges\":[\"write\",\"create_index\",\"auto_configure\"],\"allow_restricted_indices\":false},{\"names\":[\".fleet-*\"],\"privileges\":[\"read\",\"write\",\"monitor\",\"create_index\",\"auto_configure\"],\"allow_restricted_indices\":false}],\"applications\":[],\"run_as\":[],\"metadata\":{},\"transient_metadata\":{\"enabled\":true}}}");
    private final Clock clock;
    private final Client client;
    private final XPackLicenseState licenseState;
    private final SecurityIndexManager securityIndex;
    private final ClusterService clusterService;
    private final Hasher hasher;
    private final boolean enabled;
    private final Settings settings;
    private final ExpiredApiKeysRemover expiredApiKeysRemover;
    private final TimeValue deleteInterval;
    private final Cache<String, ListenableFuture<CachedApiKeyHashResult>> apiKeyAuthCache;
    private final Hasher cacheHasher;
    private final ThreadPool threadPool;
    private final ApiKeyDocCache apiKeyDocCache;
    private volatile long lastExpirationRunMs;
    private static final long EVICTION_MONITOR_INTERVAL_SECONDS = 300L;
    private static final long EVICTION_MONITOR_INTERVAL_NANOS = 300000000000L;
    private static final long EVICTION_WARNING_THRESHOLD = 4500L;
    private final AtomicLong lastEvictionCheckedAt = new AtomicLong(0L);
    private final LongAdder evictionCounter = new LongAdder();

    public ApiKeyService(Settings settings, Clock clock, Client client, XPackLicenseState licenseState, SecurityIndexManager securityIndex, ClusterService clusterService, CacheInvalidatorRegistry cacheInvalidatorRegistry, ThreadPool threadPool) {
        this.clock = clock;
        this.client = client;
        this.licenseState = licenseState;
        this.securityIndex = securityIndex;
        this.clusterService = clusterService;
        this.enabled = (Boolean)XPackSettings.API_KEY_SERVICE_ENABLED_SETTING.get(settings);
        this.hasher = Hasher.resolve((String)((String)PASSWORD_HASHING_ALGORITHM.get(settings)));
        this.settings = settings;
        this.deleteInterval = (TimeValue)DELETE_INTERVAL.get(settings);
        this.expiredApiKeysRemover = new ExpiredApiKeysRemover(settings, client);
        this.threadPool = threadPool;
        this.cacheHasher = Hasher.resolve((String)((String)CACHE_HASH_ALGO_SETTING.get(settings)));
        TimeValue ttl = (TimeValue)CACHE_TTL_SETTING.get(settings);
        int maximumWeight = (Integer)CACHE_MAX_KEYS_SETTING.get(settings);
        if (ttl.getNanos() > 0L) {
            this.apiKeyAuthCache = CacheBuilder.builder().setExpireAfterAccess(ttl).setMaximumWeight((long)maximumWeight).removalListener(this.getAuthCacheRemovalListener(maximumWeight)).build();
            TimeValue doc_ttl = (TimeValue)DOC_CACHE_TTL_SETTING.get(settings);
            this.apiKeyDocCache = doc_ttl.getNanos() == 0L ? null : new ApiKeyDocCache(doc_ttl, maximumWeight);
            cacheInvalidatorRegistry.registerCacheInvalidator("api_key", new CacheInvalidatorRegistry.CacheInvalidator(){

                @Override
                public void invalidate(Collection<String> keys) {
                    if (ApiKeyService.this.apiKeyDocCache != null) {
                        ApiKeyService.this.apiKeyDocCache.invalidate(keys);
                    }
                    keys.forEach(arg_0 -> ((Cache)ApiKeyService.this.apiKeyAuthCache).invalidate(arg_0));
                }

                @Override
                public void invalidateAll() {
                    if (ApiKeyService.this.apiKeyDocCache != null) {
                        ApiKeyService.this.apiKeyDocCache.invalidateAll();
                    }
                    ApiKeyService.this.apiKeyAuthCache.invalidateAll();
                }
            });
        } else {
            this.apiKeyAuthCache = null;
            this.apiKeyDocCache = null;
        }
    }

    public void createApiKey(Authentication authentication, CreateApiKeyRequest request, Set<RoleDescriptor> userRoles, ActionListener<CreateApiKeyResponse> listener) {
        Version securityMappingVersion;
        this.ensureEnabled();
        if (request.getMetadata() != null && !request.getMetadata().isEmpty() && (securityMappingVersion = this.securityIndex.getInstallableMappingVersion()).before(METADATA_INTRODUCED)) {
            logger.warn("The security index [{}] mapping is for version [{}] but API Key metadata requires [{}]; the mapping will automatically upgrade to a supported version when the cluster no longer has nodes that are [{}] or earlier", (Object)this.securityIndex.aliasName(), (Object)securityMappingVersion, (Object)METADATA_INTRODUCED, (Object)Security.FLATTENED_FIELD_TYPE_INTRODUCED);
            listener.onFailure((Exception)new IllegalArgumentException("API metadata requires all nodes to be at least [" + Security.FLATTENED_FIELD_TYPE_INTRODUCED + "]"));
            return;
        }
        if (authentication == null) {
            listener.onFailure((Exception)new IllegalArgumentException("authentication must be provided"));
        } else {
            this.createApiKeyAndIndexIt(authentication, request, userRoles, listener);
        }
    }

    private void createApiKeyAndIndexIt(Authentication authentication, CreateApiKeyRequest request, Set<RoleDescriptor> roleDescriptorSet, ActionListener<CreateApiKeyResponse> listener) {
        Instant created = this.clock.instant();
        Instant expiration = this.getApiKeyExpiration(created, request);
        SecureString apiKey = UUIDs.randomBase64UUIDSecureString();
        Version version = this.clusterService.state().nodes().getMinNodeVersion();
        if (version.before(Version.V_6_7_0)) {
            logger.warn("nodes prior to the minimum supported version for api keys {} exist in the cluster; these nodes will not be able to use api keys", (Object)Version.V_6_7_0);
        }
        this.computeHashForApiKey(apiKey, (ActionListener<char[]>)listener.delegateFailure((l, apiKeyHashChars) -> {
            try (XContentBuilder builder = this.newDocument((char[])apiKeyHashChars, request.getName(), authentication, roleDescriptorSet, created, expiration, request.getRoleDescriptors(), version, request.getMetadata());){
                IndexRequest indexRequest = (IndexRequest)((IndexRequestBuilder)this.client.prepareIndex(".security", "_doc").setSource(builder).setRefreshPolicy(request.getRefreshPolicy())).request();
                BulkRequest bulkRequest = TransportSingleItemBulkWriteAction.toSingleItemBulkRequest((ReplicatedWriteRequest)indexRequest);
                this.securityIndex.prepareIndexIfNeededThenExecute(arg_0 -> ((ActionListener)listener).onFailure(arg_0), () -> ClientHelper.executeAsyncWithOrigin((Client)this.client, (String)"security", (ActionType)BulkAction.INSTANCE, (ActionRequest)bulkRequest, (ActionListener)TransportSingleItemBulkWriteAction.wrapBulkResponse((ActionListener)ActionListener.wrap(indexResponse -> {
                    ListenableFuture listenableFuture = new ListenableFuture();
                    listenableFuture.onResponse((Object)new CachedApiKeyHashResult(true, apiKey));
                    this.apiKeyAuthCache.put((Object)indexResponse.getId(), (Object)listenableFuture);
                    listener.onResponse((Object)new CreateApiKeyResponse(request.getName(), indexResponse.getId(), apiKey, expiration));
                }, arg_0 -> ((ActionListener)listener).onFailure(arg_0)))));
            }
            catch (IOException e) {
                listener.onFailure((Exception)e);
            }
            finally {
                Arrays.fill(apiKeyHashChars, '\u0000');
            }
        }));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    XContentBuilder newDocument(char[] apiKeyHashChars, String name, Authentication authentication, Set<RoleDescriptor> userRoles, Instant created, Instant expiration, List<RoleDescriptor> keyRoles, Version version, @Nullable Map<String, Object> metadata) throws IOException {
        XContentBuilder builder = XContentFactory.jsonBuilder();
        builder.startObject().field("doc_type", "api_key").field("creation_time", created.toEpochMilli()).field("expiration_time", expiration == null ? null : Long.valueOf(expiration.toEpochMilli())).field("api_key_invalidated", false);
        byte[] utf8Bytes = null;
        try {
            utf8Bytes = CharArrays.toUtf8Bytes((char[])apiKeyHashChars);
            builder.field("api_key_hash").utf8Value(utf8Bytes, 0, utf8Bytes.length);
        }
        finally {
            if (utf8Bytes != null) {
                Arrays.fill(utf8Bytes, (byte)0);
            }
        }
        builder.startObject("role_descriptors");
        if (keyRoles != null && !keyRoles.isEmpty()) {
            for (RoleDescriptor descriptor : keyRoles) {
                builder.field(descriptor.getName(), (contentBuilder, params) -> descriptor.toXContent(contentBuilder, params, true));
            }
        }
        builder.endObject();
        builder.startObject("limited_by_role_descriptors");
        for (RoleDescriptor descriptor : userRoles) {
            builder.field(descriptor.getName(), (contentBuilder, params) -> descriptor.toXContent(contentBuilder, params, true));
        }
        builder.endObject();
        builder.field("name", name).field("version", version.id);
        if (metadata != null && !metadata.isEmpty()) {
            assert (this.securityIndex.getInstallableMappingVersion().onOrAfter(METADATA_INTRODUCED)) : "API metadata requires all nodes to be at least [" + Security.FLATTENED_FIELD_TYPE_INTRODUCED + "]";
            builder.field("metadata_flattened", metadata);
        }
        builder.startObject("creator").field("principal", authentication.getUser().principal()).field("full_name", authentication.getUser().fullName()).field("email", authentication.getUser().email()).field("metadata", authentication.getUser().metadata()).field("realm", authentication.getSourceRealm().getName()).field("realm_type", authentication.getSourceRealm().getType()).endObject().endObject();
        return builder;
    }

    void authenticateWithApiKeyIfPresent(ThreadContext ctx, ActionListener<AuthenticationResult> listener) {
        if (this.isEnabled()) {
            ApiKeyCredentials credentials;
            try {
                credentials = ApiKeyService.getCredentialsFromHeader(ctx);
            }
            catch (IllegalArgumentException iae) {
                listener.onResponse((Object)AuthenticationResult.unsuccessful((String)iae.getMessage(), (Exception)iae));
                return;
            }
            if (credentials != null) {
                this.loadApiKeyAndValidateCredentials(ctx, credentials, (ActionListener<AuthenticationResult>)ActionListener.wrap(response -> {
                    credentials.close();
                    listener.onResponse(response);
                }, e -> {
                    credentials.close();
                    listener.onFailure(e);
                }));
            } else {
                listener.onResponse((Object)AuthenticationResult.notHandled());
            }
        } else {
            listener.onResponse((Object)AuthenticationResult.notHandled());
        }
    }

    public Authentication createApiKeyAuthentication(AuthenticationResult authResult, String nodeName) {
        if (!authResult.isAuthenticated()) {
            throw new IllegalArgumentException("API Key authn result must be successful");
        }
        User user = authResult.getUser();
        Authentication.RealmRef authenticatedBy = new Authentication.RealmRef("_es_api_key", "_es_api_key", nodeName);
        return new Authentication(user, authenticatedBy, null, Version.CURRENT, Authentication.AuthenticationType.API_KEY, authResult.getMetadata());
    }

    void loadApiKeyAndValidateCredentials(ThreadContext ctx, ApiKeyCredentials credentials, ActionListener<AuthenticationResult> listener) {
        long invalidationCount;
        String docId = credentials.getId();
        Consumer<ApiKeyDoc> validator = apiKeyDoc -> this.validateApiKeyCredentials(docId, (ApiKeyDoc)apiKeyDoc, credentials, this.clock, (ActionListener<AuthenticationResult>)listener.delegateResponse((l, e) -> {
            if (ExceptionsHelper.unwrapCause((Throwable)e) instanceof EsRejectedExecutionException) {
                l.onResponse((Object)AuthenticationResult.terminate((String)"server is too busy to respond", (Exception)e));
            } else {
                l.onFailure(e);
            }
        }));
        if (this.apiKeyDocCache != null) {
            ApiKeyDoc existing = this.apiKeyDocCache.get(docId);
            if (existing != null) {
                validator.accept(existing);
                return;
            }
            invalidationCount = this.apiKeyDocCache.getInvalidationCount();
        } else {
            invalidationCount = -1L;
        }
        GetRequest getRequest = (GetRequest)this.client.prepareGet(".security", "_doc", docId).setFetchSource(true).request();
        ClientHelper.executeAsyncWithOrigin((ThreadContext)ctx, (String)"security", (ActionRequest)getRequest, (ActionListener)ActionListener.wrap(response -> {
            if (response.isExists()) {
                ApiKeyDoc apiKeyDoc;
                try (XContentParser parser = XContentHelper.createParser((NamedXContentRegistry)NamedXContentRegistry.EMPTY, (DeprecationHandler)LoggingDeprecationHandler.INSTANCE, (BytesReference)response.getSourceAsBytesRef(), (XContentType)XContentType.JSON);){
                    apiKeyDoc = ApiKeyDoc.fromXContent(parser);
                }
                if (invalidationCount != -1L) {
                    this.apiKeyDocCache.putIfNoInvalidationSince(docId, apiKeyDoc, invalidationCount);
                }
                validator.accept(apiKeyDoc);
            } else {
                if (this.apiKeyAuthCache != null) {
                    this.apiKeyAuthCache.invalidate((Object)docId);
                }
                listener.onResponse((Object)AuthenticationResult.unsuccessful((String)("unable to find apikey with id " + credentials.getId()), null));
            }
        }, e -> {
            if (ExceptionsHelper.unwrapCause((Throwable)e) instanceof EsRejectedExecutionException) {
                listener.onResponse((Object)AuthenticationResult.terminate((String)"server is too busy to respond", (Exception)e));
            } else {
                listener.onResponse((Object)AuthenticationResult.unsuccessful((String)("apikey authentication for id " + credentials.getId() + " encountered a failure"), (Exception)e));
            }
        }), (arg_0, arg_1) -> ((Client)this.client).get(arg_0, arg_1));
    }

    public void getRoleForApiKey(Authentication authentication, ActionListener<ApiKeyRoleDescriptors> listener) {
        if (authentication.getAuthenticationType() != Authentication.AuthenticationType.API_KEY) {
            throw new IllegalStateException("authentication type must be api key but is " + authentication.getAuthenticationType());
        }
        assert (authentication.getVersion().before(Authentication.VERSION_API_KEY_ROLES_AS_BYTES)) : "This method only applies to authentication objects created before v7.9.0";
        Map metadata = authentication.getMetadata();
        String apiKeyId = (String)metadata.get(API_KEY_ID_KEY);
        Map roleDescriptors = (Map)metadata.get("_security_api_key_role_descriptors");
        Map authnRoleDescriptors = (Map)metadata.get("_security_api_key_limited_by_role_descriptors");
        if (roleDescriptors == null && authnRoleDescriptors == null) {
            listener.onFailure((Exception)((Object)new ElasticsearchSecurityException("no role descriptors found for API key", new Object[0])));
        } else if (roleDescriptors == null || roleDescriptors.isEmpty()) {
            List<RoleDescriptor> authnRoleDescriptorsList = this.parseRoleDescriptors(apiKeyId, authnRoleDescriptors);
            listener.onResponse((Object)new ApiKeyRoleDescriptors(apiKeyId, authnRoleDescriptorsList, null));
        } else {
            List<RoleDescriptor> roleDescriptorList = this.parseRoleDescriptors(apiKeyId, roleDescriptors);
            List<RoleDescriptor> authnRoleDescriptorsList = this.parseRoleDescriptors(apiKeyId, authnRoleDescriptors);
            listener.onResponse((Object)new ApiKeyRoleDescriptors(apiKeyId, roleDescriptorList, authnRoleDescriptorsList));
        }
    }

    public Tuple<String, BytesReference> getApiKeyIdAndRoleBytes(Authentication authentication, boolean limitedBy) {
        if (authentication.getAuthenticationType() != Authentication.AuthenticationType.API_KEY) {
            throw new IllegalStateException("authentication type must be api key but is " + authentication.getAuthenticationType());
        }
        assert (authentication.getVersion().onOrAfter(Authentication.VERSION_API_KEY_ROLES_AS_BYTES)) : "This method only applies to authentication objects created on or after v7.9.0";
        Map metadata = authentication.getMetadata();
        BytesReference bytesReference = (BytesReference)metadata.get(limitedBy ? "_security_api_key_limited_by_role_descriptors" : "_security_api_key_role_descriptors");
        if (limitedBy && bytesReference.length() == 2 && "{}".equals(bytesReference.utf8ToString()) && "_service_account".equals(metadata.get(API_KEY_CREATOR_REALM_NAME)) && "elastic/fleet-server".equals(authentication.getUser().principal())) {
            return new Tuple((Object)((String)metadata.get(API_KEY_ID_KEY)), (Object)FLEET_SERVER_ROLE_DESCRIPTOR_BYTES_V_7_14);
        }
        return new Tuple((Object)((String)metadata.get(API_KEY_ID_KEY)), (Object)bytesReference);
    }

    private List<RoleDescriptor> parseRoleDescriptors(String apiKeyId, Map<String, Object> roleDescriptors) {
        if (roleDescriptors == null) {
            return null;
        }
        return roleDescriptors.entrySet().stream().map(entry -> {
            String name = (String)entry.getKey();
            Map rdMap = (Map)entry.getValue();
            try (XContentBuilder builder = XContentBuilder.builder((XContent)XContentType.JSON.xContent());){
                RoleDescriptor roleDescriptor;
                block14: {
                    builder.map(rdMap);
                    XContentParser parser = XContentType.JSON.xContent().createParser(NamedXContentRegistry.EMPTY, (DeprecationHandler)new ApiKeyLoggingDeprecationHandler(deprecationLogger, apiKeyId), (InputStream)BytesReference.bytes((XContentBuilder)builder).streamInput());
                    try {
                        roleDescriptor = RoleDescriptor.parse((String)name, (XContentParser)parser, (boolean)false);
                        if (parser == null) break block14;
                    }
                    catch (Throwable throwable) {
                        if (parser != null) {
                            try {
                                parser.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    parser.close();
                }
                return roleDescriptor;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }).collect(Collectors.toList());
    }

    public List<RoleDescriptor> parseRoleDescriptors(String apiKeyId, BytesReference bytesReference) {
        if (bytesReference == null) {
            return Collections.emptyList();
        }
        ArrayList<RoleDescriptor> roleDescriptors = new ArrayList<RoleDescriptor>();
        try (XContentParser parser = XContentHelper.createParser((NamedXContentRegistry)NamedXContentRegistry.EMPTY, (DeprecationHandler)new ApiKeyLoggingDeprecationHandler(deprecationLogger, apiKeyId), (BytesReference)bytesReference, (XContentType)XContentType.JSON);){
            parser.nextToken();
            while (parser.nextToken() != XContentParser.Token.END_OBJECT) {
                parser.nextToken();
                String roleName = parser.currentName();
                roleDescriptors.add(RoleDescriptor.parse((String)roleName, (XContentParser)parser, (boolean)false));
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return Collections.unmodifiableList(roleDescriptors);
    }

    void validateApiKeyCredentials(String docId, ApiKeyDoc apiKeyDoc, ApiKeyCredentials credentials, Clock clock, ActionListener<AuthenticationResult> listener) {
        if (!"api_key".equals(apiKeyDoc.docType)) {
            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)("document [" + docId + "] is [" + apiKeyDoc.docType + "] not an api key"), null));
        } else if (apiKeyDoc.invalidated == null) {
            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"api key document is missing invalidated field", null));
        } else if (apiKeyDoc.invalidated.booleanValue()) {
            if (this.apiKeyAuthCache != null) {
                this.apiKeyAuthCache.invalidate((Object)docId);
            }
            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"api key has been invalidated", null));
        } else {
            if (apiKeyDoc.hash == null) {
                throw new IllegalStateException("api key hash is missing");
            }
            if (this.apiKeyAuthCache != null) {
                ListenableFuture listenableCacheEntry;
                AtomicBoolean valueAlreadyInCache = new AtomicBoolean(true);
                try {
                    listenableCacheEntry = (ListenableFuture)this.apiKeyAuthCache.computeIfAbsent((Object)credentials.getId(), k -> {
                        valueAlreadyInCache.set(false);
                        return new ListenableFuture();
                    });
                }
                catch (ExecutionException e) {
                    listener.onFailure((Exception)e);
                    return;
                }
                if (valueAlreadyInCache.get()) {
                    listenableCacheEntry.addListener(ActionListener.wrap(result -> {
                        if (result.success) {
                            if (result.verify(credentials.getKey())) {
                                this.validateApiKeyExpiration(apiKeyDoc, credentials, clock, listener);
                            } else {
                                listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"invalid credentials", null));
                            }
                        } else if (result.verify(credentials.getKey())) {
                            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"invalid credentials", null));
                        } else {
                            this.apiKeyAuthCache.invalidate((Object)credentials.getId(), (Object)listenableCacheEntry);
                            this.validateApiKeyCredentials(docId, apiKeyDoc, credentials, clock, listener);
                        }
                    }, arg_0 -> listener.onFailure(arg_0)), this.threadPool.generic(), this.threadPool.getThreadContext());
                } else {
                    this.verifyKeyAgainstHash(apiKeyDoc.hash, credentials, (ActionListener<Boolean>)ActionListener.wrap(verified -> {
                        listenableCacheEntry.onResponse((Object)new CachedApiKeyHashResult((boolean)verified, credentials.getKey()));
                        if (verified.booleanValue()) {
                            this.validateApiKeyExpiration(apiKeyDoc, credentials, clock, listener);
                        } else {
                            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"invalid credentials", null));
                        }
                    }, arg_0 -> listener.onFailure(arg_0)));
                }
            } else {
                this.verifyKeyAgainstHash(apiKeyDoc.hash, credentials, (ActionListener<Boolean>)ActionListener.wrap(verified -> {
                    if (verified.booleanValue()) {
                        this.validateApiKeyExpiration(apiKeyDoc, credentials, clock, listener);
                    } else {
                        listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"invalid credentials", null));
                    }
                }, arg_0 -> listener.onFailure(arg_0)));
            }
        }
    }

    CachedApiKeyHashResult getFromCache(String id) {
        return this.apiKeyAuthCache == null ? null : (CachedApiKeyHashResult)FutureUtils.get((Future)((Future)this.apiKeyAuthCache.get((Object)id)), (long)0L, (TimeUnit)TimeUnit.MILLISECONDS);
    }

    Cache<String, ListenableFuture<CachedApiKeyHashResult>> getApiKeyAuthCache() {
        return this.apiKeyAuthCache;
    }

    Cache<String, CachedApiKeyDoc> getDocCache() {
        return this.apiKeyDocCache == null ? null : this.apiKeyDocCache.docCache;
    }

    Cache<String, BytesReference> getRoleDescriptorsBytesCache() {
        return this.apiKeyDocCache == null ? null : this.apiKeyDocCache.roleDescriptorsBytesCache;
    }

    void validateApiKeyExpiration(ApiKeyDoc apiKeyDoc, ApiKeyCredentials credentials, Clock clock, ActionListener<AuthenticationResult> listener) {
        if (apiKeyDoc.expirationTime == -1L || Instant.ofEpochMilli(apiKeyDoc.expirationTime).isAfter(clock.instant())) {
            String principal = Objects.requireNonNull((String)apiKeyDoc.creator.get("principal"));
            String fullName = (String)apiKeyDoc.creator.get("full_name");
            String email = (String)apiKeyDoc.creator.get("email");
            Map metadata = (Map)apiKeyDoc.creator.get("metadata");
            User apiKeyUser = new User(principal, Strings.EMPTY_ARRAY, fullName, email, metadata, true);
            HashMap<String, Object> authResultMetadata = new HashMap<String, Object>();
            authResultMetadata.put(API_KEY_CREATOR_REALM_NAME, apiKeyDoc.creator.get("realm"));
            authResultMetadata.put(API_KEY_CREATOR_REALM_TYPE, apiKeyDoc.creator.get("realm_type"));
            authResultMetadata.put("_security_api_key_role_descriptors", apiKeyDoc.roleDescriptorsBytes);
            authResultMetadata.put("_security_api_key_limited_by_role_descriptors", apiKeyDoc.limitedByRoleDescriptorsBytes);
            authResultMetadata.put(API_KEY_ID_KEY, credentials.getId());
            authResultMetadata.put(API_KEY_NAME_KEY, apiKeyDoc.name);
            if (apiKeyDoc.metadataFlattened != null) {
                authResultMetadata.put(API_KEY_METADATA_KEY, apiKeyDoc.metadataFlattened);
            }
            listener.onResponse((Object)AuthenticationResult.success((User)apiKeyUser, authResultMetadata));
        } else {
            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)"api key is expired", null));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static ApiKeyCredentials getCredentialsFromHeader(ThreadContext threadContext) {
        String header = threadContext.getHeader("Authorization");
        if (Strings.hasText((String)header) && header.regionMatches(true, 0, "ApiKey ", 0, "ApiKey ".length()) && header.length() > "ApiKey ".length()) {
            byte[] decodedApiKeyCredBytes = Base64.getDecoder().decode(header.substring("ApiKey ".length()));
            char[] apiKeyCredChars = null;
            try {
                apiKeyCredChars = CharArrays.utf8BytesToChars((byte[])decodedApiKeyCredBytes);
                int colonIndex = -1;
                for (int i = 0; i < apiKeyCredChars.length; ++i) {
                    if (apiKeyCredChars[i] != ':') continue;
                    colonIndex = i;
                    break;
                }
                if (colonIndex < 1) {
                    throw new IllegalArgumentException("invalid ApiKey value");
                }
                ApiKeyCredentials apiKeyCredentials = new ApiKeyCredentials(new String(Arrays.copyOfRange(apiKeyCredChars, 0, colonIndex)), new SecureString(Arrays.copyOfRange(apiKeyCredChars, colonIndex + 1, apiKeyCredChars.length)));
                return apiKeyCredentials;
            }
            finally {
                if (apiKeyCredChars != null) {
                    Arrays.fill(apiKeyCredChars, '\u0000');
                }
            }
        }
        return null;
    }

    public static boolean isApiKeyAuthentication(Authentication authentication) {
        Authentication.AuthenticationType authType = authentication.getAuthenticationType();
        if (Authentication.AuthenticationType.API_KEY == authType) {
            assert ("_es_api_key".equals(authentication.getAuthenticatedBy().getType())) : "API key authentication must have API key realm type";
            return true;
        }
        return false;
    }

    void computeHashForApiKey(SecureString apiKey, ActionListener<char[]> listener) {
        this.threadPool.executor("security-crypto").execute((Runnable)ActionRunnable.supply(listener, () -> this.hasher.hash(apiKey)));
    }

    protected void verifyKeyAgainstHash(String apiKeyHash, ApiKeyCredentials credentials, ActionListener<Boolean> listener) {
        this.threadPool.executor("security-crypto").execute((Runnable)ActionRunnable.supply(listener, () -> {
            Hasher hasher = Hasher.resolveFromHash((char[])apiKeyHash.toCharArray());
            char[] apiKeyHashChars = apiKeyHash.toCharArray();
            try {
                Boolean bl = hasher.verify(credentials.getKey(), apiKeyHashChars);
                return bl;
            }
            finally {
                Arrays.fill(apiKeyHashChars, '\u0000');
            }
        }));
    }

    private Instant getApiKeyExpiration(Instant now, CreateApiKeyRequest request) {
        if (request.getExpiration() != null) {
            return now.plusSeconds(request.getExpiration().getSeconds());
        }
        return null;
    }

    private boolean isEnabled() {
        return this.enabled && this.licenseState.isSecurityEnabled();
    }

    public void ensureEnabled() {
        if (!this.licenseState.isSecurityEnabled()) {
            throw LicenseUtils.newComplianceException((String)"security is not enabled");
        }
        if (!this.enabled) {
            throw new FeatureNotEnabledException(FeatureNotEnabledException.Feature.API_KEY_SERVICE, "api keys are not enabled", new Object[0]);
        }
    }

    public void invalidateApiKeys(String realmName, String username, String apiKeyName, String[] apiKeyIds, ActionListener<InvalidateApiKeyResponse> invalidateListener) {
        this.ensureEnabled();
        if (!(Strings.hasText((String)realmName) || Strings.hasText((String)username) || Strings.hasText((String)apiKeyName) || apiKeyIds != null && apiKeyIds.length != 0)) {
            logger.trace("none of the parameters [api key id, api key name, username, realm name] were specified for invalidation");
            invalidateListener.onFailure((Exception)new IllegalArgumentException("One of [api key id, api key name, username, realm name] must be specified"));
        } else {
            this.findApiKeysForUserRealmApiKeyIdAndNameCombination(realmName, username, apiKeyName, apiKeyIds, true, false, (ActionListener<Collection<ApiKey>>)ActionListener.wrap(apiKeys -> {
                if (apiKeys.isEmpty()) {
                    logger.debug("No active api keys to invalidate for realm [{}], username [{}], api key name [{}] and api key id [{}]", (Object)realmName, (Object)username, (Object)apiKeyName, (Object)Arrays.toString(apiKeyIds));
                    invalidateListener.onResponse((Object)InvalidateApiKeyResponse.emptyResponse());
                } else {
                    this.invalidateAllApiKeys(apiKeys.stream().map(apiKey -> apiKey.getId()).collect(Collectors.toSet()), invalidateListener);
                }
            }, arg_0 -> invalidateListener.onFailure(arg_0)));
        }
    }

    private void invalidateAllApiKeys(Collection<String> apiKeyIds, ActionListener<InvalidateApiKeyResponse> invalidateListener) {
        this.indexInvalidation(apiKeyIds, invalidateListener, null);
    }

    private void findApiKeys(BoolQueryBuilder boolQuery, boolean filterOutInvalidatedKeys, boolean filterOutExpiredKeys, ActionListener<Collection<ApiKey>> listener) {
        if (filterOutInvalidatedKeys) {
            boolQuery.filter((QueryBuilder)QueryBuilders.termQuery((String)"api_key_invalidated", (boolean)false));
        }
        if (filterOutExpiredKeys) {
            BoolQueryBuilder expiredQuery = QueryBuilders.boolQuery();
            expiredQuery.should((QueryBuilder)QueryBuilders.rangeQuery((String)"expiration_time").lte((Object)Instant.now().toEpochMilli()));
            expiredQuery.should((QueryBuilder)QueryBuilders.boolQuery().mustNot((QueryBuilder)QueryBuilders.existsQuery((String)"expiration_time")));
            boolQuery.filter((QueryBuilder)expiredQuery);
        }
        Supplier supplier = this.client.threadPool().getThreadContext().newRestorableContext(false);
        try (ThreadContext.StoredContext ignore = this.client.threadPool().getThreadContext().stashWithOrigin("security");){
            SearchRequest request = (SearchRequest)this.client.prepareSearch(new String[]{".security"}).setScroll((TimeValue)SearchService.DEFAULT_KEEPALIVE_SETTING.get(this.settings)).setQuery((QueryBuilder)boolQuery).setVersion(false).setSize(1000).setFetchSource(true).request();
            this.securityIndex.checkIndexVersionThenExecute(arg_0 -> listener.onFailure(arg_0), () -> this.lambda$findApiKeys$22(request, (Supplier)supplier, listener));
        }
    }

    private void findApiKeysForUserRealmApiKeyIdAndNameCombination(String realmName, String userName, String apiKeyName, String[] apiKeyIds, boolean filterOutInvalidatedKeys, boolean filterOutExpiredKeys, ActionListener<Collection<ApiKey>> listener) {
        SecurityIndexManager frozenSecurityIndex = this.securityIndex.freeze();
        if (!frozenSecurityIndex.indexExists()) {
            listener.onResponse(Collections.emptyList());
        } else if (!frozenSecurityIndex.isAvailable()) {
            listener.onFailure((Exception)((Object)frozenSecurityIndex.getUnavailableReason()));
        } else {
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"doc_type", (String)"api_key"));
            if (Strings.hasText((String)realmName)) {
                boolQuery.filter((QueryBuilder)QueryBuilders.termQuery((String)"creator.realm", (String)realmName));
            }
            if (Strings.hasText((String)userName)) {
                boolQuery.filter((QueryBuilder)QueryBuilders.termQuery((String)"creator.principal", (String)userName));
            }
            if (Strings.hasText((String)apiKeyName) && !"*".equals(apiKeyName)) {
                if (apiKeyName.endsWith("*")) {
                    boolQuery.filter((QueryBuilder)QueryBuilders.prefixQuery((String)"name", (String)apiKeyName.substring(0, apiKeyName.length() - 1)));
                } else {
                    boolQuery.filter((QueryBuilder)QueryBuilders.termQuery((String)"name", (String)apiKeyName));
                }
            }
            if (apiKeyIds != null && apiKeyIds.length > 0) {
                boolQuery.filter((QueryBuilder)QueryBuilders.idsQuery().addIds(apiKeyIds));
            }
            this.findApiKeys(boolQuery, filterOutInvalidatedKeys, filterOutExpiredKeys, listener);
        }
    }

    private void indexInvalidation(Collection<String> apiKeyIds, ActionListener<InvalidateApiKeyResponse> listener, @Nullable InvalidateApiKeyResponse previousResult) {
        this.maybeStartApiKeyRemover();
        if (apiKeyIds.isEmpty()) {
            listener.onFailure((Exception)((Object)new ElasticsearchSecurityException("No api key ids provided for invalidation", new Object[0])));
        } else {
            BulkRequestBuilder bulkRequestBuilder = this.client.prepareBulk();
            for (String apiKeyId : apiKeyIds) {
                UpdateRequest request = (UpdateRequest)this.client.prepareUpdate(".security", "_doc", apiKeyId).setDoc(Collections.singletonMap("api_key_invalidated", true)).request();
                bulkRequestBuilder.add(request);
            }
            bulkRequestBuilder.setRefreshPolicy(WriteRequest.RefreshPolicy.WAIT_UNTIL);
            this.securityIndex.prepareIndexIfNeededThenExecute(ex -> listener.onFailure(this.traceLog("prepare security index", ex)), () -> ClientHelper.executeAsyncWithOrigin((ThreadContext)this.client.threadPool().getThreadContext(), (String)"security", (ActionRequest)((BulkRequest)bulkRequestBuilder.request()), (ActionListener)ActionListener.wrap(bulkResponse -> {
                ArrayList<ElasticsearchException> failedRequestResponses = new ArrayList<ElasticsearchException>();
                ArrayList<String> previouslyInvalidated = new ArrayList<String>();
                ArrayList<String> invalidated = new ArrayList<String>();
                if (null != previousResult) {
                    failedRequestResponses.addAll(previousResult.getErrors());
                    previouslyInvalidated.addAll(previousResult.getPreviouslyInvalidatedApiKeys());
                    invalidated.addAll(previousResult.getInvalidatedApiKeys());
                }
                for (BulkItemResponse bulkItemResponse : bulkResponse.getItems()) {
                    if (bulkItemResponse.isFailed()) {
                        Exception cause = bulkItemResponse.getFailure().getCause();
                        String failedApiKeyId = bulkItemResponse.getFailure().getId();
                        this.traceLog("invalidate api key", failedApiKeyId, cause);
                        failedRequestResponses.add(new ElasticsearchException("Error invalidating api key", (Throwable)cause, new Object[0]));
                        continue;
                    }
                    UpdateResponse updateResponse = (UpdateResponse)bulkItemResponse.getResponse();
                    if (updateResponse.getResult() == DocWriteResponse.Result.UPDATED) {
                        logger.debug("Invalidated api key for doc [{}]", (Object)updateResponse.getId());
                        invalidated.add(updateResponse.getId());
                        continue;
                    }
                    if (updateResponse.getResult() != DocWriteResponse.Result.NOOP) continue;
                    previouslyInvalidated.add(updateResponse.getId());
                }
                InvalidateApiKeyResponse result = new InvalidateApiKeyResponse(invalidated, previouslyInvalidated, failedRequestResponses);
                this.clearCache(result, listener);
            }, e -> {
                Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
                this.traceLog("invalidate api keys", cause);
                listener.onFailure(e);
            }), (arg_0, arg_1) -> ((Client)this.client).bulk(arg_0, arg_1)));
        }
    }

    private void clearCache(final InvalidateApiKeyResponse result, final ActionListener<InvalidateApiKeyResponse> listener) {
        ClearSecurityCacheRequest clearApiKeyCacheRequest = new ClearSecurityCacheRequest().cacheName("api_key").keys(result.getInvalidatedApiKeys().toArray(new String[0]));
        ClientHelper.executeAsyncWithOrigin((Client)this.client, (String)"security", (ActionType)ClearSecurityCacheAction.INSTANCE, (ActionRequest)clearApiKeyCacheRequest, (ActionListener)new ActionListener<ClearSecurityCacheResponse>(){

            public void onResponse(ClearSecurityCacheResponse nodes) {
                listener.onResponse((Object)result);
            }

            public void onFailure(Exception e) {
                logger.error("unable to clear API key cache", (Throwable)e);
                listener.onFailure((Exception)((Object)new ElasticsearchException("clearing the API key cache failed; please clear the caches manually", (Throwable)e, new Object[0])));
            }
        });
    }

    private <E extends Throwable> E traceLog(String action, String identifier, E exception) {
        if (logger.isTraceEnabled()) {
            if (exception instanceof ElasticsearchException) {
                ElasticsearchException esEx = (ElasticsearchException)exception;
                List detail = esEx.getHeader("error_description");
                if (detail != null) {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}] for id [{}] - [{}]", new Object[]{action, identifier, detail}), (Throwable)esEx);
                } else {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}] for id [{}]", (Object)action, (Object)identifier), (Throwable)esEx);
                }
            } else {
                logger.trace(() -> new ParameterizedMessage("Failure in [{}] for id [{}]", (Object)action, (Object)identifier), exception);
            }
        }
        return exception;
    }

    private <E extends Throwable> E traceLog(String action, E exception) {
        if (logger.isTraceEnabled()) {
            if (exception instanceof ElasticsearchException) {
                ElasticsearchException esEx = (ElasticsearchException)exception;
                List detail = esEx.getHeader("error_description");
                if (detail != null) {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}] - [{}]", (Object)action, detail), (Throwable)esEx);
                } else {
                    logger.trace(() -> new ParameterizedMessage("Failure in [{}]", (Object)action), (Throwable)esEx);
                }
            } else {
                logger.trace(() -> new ParameterizedMessage("Failure in [{}]", (Object)action), exception);
            }
        }
        return exception;
    }

    boolean isExpirationInProgress() {
        return this.expiredApiKeysRemover.isExpirationInProgress();
    }

    long lastTimeWhenApiKeysRemoverWasTriggered() {
        return this.lastExpirationRunMs;
    }

    private void maybeStartApiKeyRemover() {
        if (this.securityIndex.isAvailable() && this.client.threadPool().relativeTimeInMillis() - this.lastExpirationRunMs > this.deleteInterval.getMillis()) {
            this.expiredApiKeysRemover.submit(this.client.threadPool());
            this.lastExpirationRunMs = this.client.threadPool().relativeTimeInMillis();
        }
    }

    public void getApiKeys(String realmName, String username, String apiKeyName, String apiKeyId, ActionListener<GetApiKeyResponse> listener) {
        String[] stringArray;
        this.ensureEnabled();
        if (!Strings.hasText((String)apiKeyId)) {
            stringArray = null;
        } else {
            String[] stringArray2 = new String[1];
            stringArray = stringArray2;
            stringArray2[0] = apiKeyId;
        }
        String[] apiKeyIds = stringArray;
        this.findApiKeysForUserRealmApiKeyIdAndNameCombination(realmName, username, apiKeyName, apiKeyIds, false, false, (ActionListener<Collection<ApiKey>>)ActionListener.wrap(apiKeyInfos -> {
            if (apiKeyInfos.isEmpty()) {
                logger.debug("No active api keys found for realm [{}], user [{}], api key name [{}] and api key id [{}]", (Object)realmName, (Object)username, (Object)apiKeyName, (Object)apiKeyId);
                listener.onResponse((Object)GetApiKeyResponse.emptyResponse());
            } else {
                listener.onResponse((Object)new GetApiKeyResponse(apiKeyInfos));
            }
        }, arg_0 -> listener.onFailure(arg_0)));
    }

    private RemovalListener<String, ListenableFuture<CachedApiKeyHashResult>> getAuthCacheRemovalListener(int maximumWeight) {
        return notification -> {
            if (RemovalNotification.RemovalReason.EVICTED == notification.getRemovalReason() && this.getApiKeyAuthCache().count() >= maximumWeight) {
                this.evictionCounter.increment();
                logger.trace("API key with ID [{}] was evicted from the authentication cache, possibly due to cache size limit", notification.getKey());
                long last = this.lastEvictionCheckedAt.get();
                long now = System.nanoTime();
                if (now - last >= 300000000000L && this.lastEvictionCheckedAt.compareAndSet(last, now)) {
                    long sum = this.evictionCounter.sum();
                    this.evictionCounter.add(-sum);
                    if (sum >= 4500L) {
                        logger.warn("Possible thrashing for API key authentication cache, [{}] eviction due to cache size within last [{}] seconds", (Object)sum, (Object)300L);
                    }
                }
            }
        };
    }

    LongAdder getEvictionCounter() {
        return this.evictionCounter;
    }

    AtomicLong getLastEvictionCheckedAt() {
        return this.lastEvictionCheckedAt;
    }

    public static String getCreatorRealmName(Authentication authentication) {
        if (Authentication.AuthenticationType.API_KEY == authentication.getAuthenticationType()) {
            return (String)authentication.getMetadata().get(API_KEY_CREATOR_REALM_NAME);
        }
        return authentication.getSourceRealm().getName();
    }

    public static String getCreatorRealmType(Authentication authentication) {
        if (Authentication.AuthenticationType.API_KEY == authentication.getAuthenticationType()) {
            return (String)authentication.getMetadata().get(API_KEY_CREATOR_REALM_TYPE);
        }
        return authentication.getSourceRealm().getType();
    }

    public static Map<String, Object> getApiKeyMetadata(Authentication authentication) {
        if (Authentication.AuthenticationType.API_KEY != authentication.getAuthenticationType()) {
            throw new IllegalArgumentException("authentication type must be [api_key], got [" + authentication.getAuthenticationType().name().toLowerCase(Locale.ROOT) + "]");
        }
        Object apiKeyMetadata = authentication.getMetadata().get(API_KEY_METADATA_KEY);
        if (apiKeyMetadata != null) {
            Tuple tuple = XContentHelper.convertToMap((BytesReference)((BytesReference)apiKeyMetadata), (boolean)false, (XContentType)XContentType.JSON);
            return (Map)tuple.v2();
        }
        return org.elasticsearch.core.Map.of();
    }

    private /* synthetic */ void lambda$findApiKeys$22(SearchRequest request, Supplier supplier, ActionListener listener) {
        ScrollHelper.fetchAllByEntity((Client)this.client, (SearchRequest)request, (ActionListener)new ContextPreservingActionListener(supplier, listener), hit -> {
            Map source = hit.getSourceAsMap();
            String name = (String)source.get("name");
            String id = hit.getId();
            Long creation = (Long)source.get("creation_time");
            Long expiration = (Long)source.get("expiration_time");
            Boolean invalidated = (Boolean)source.get("api_key_invalidated");
            String username = (String)((Map)source.get("creator")).get("principal");
            String realm = (String)((Map)source.get("creator")).get("realm");
            Map metadata = (Map)source.get("metadata_flattened");
            return new ApiKey(name, id, Instant.ofEpochMilli(creation), expiration != null ? Instant.ofEpochMilli(expiration) : null, invalidated.booleanValue(), username, realm, metadata);
        });
    }

    private static final class ApiKeyDocCache {
        private final Cache<String, CachedApiKeyDoc> docCache;
        private final Cache<String, BytesReference> roleDescriptorsBytesCache;
        private final LockingAtomicCounter lockingAtomicCounter;

        ApiKeyDocCache(TimeValue ttl, int maximumWeight) {
            this.docCache = CacheBuilder.builder().setMaximumWeight((long)maximumWeight).setExpireAfterWrite(ttl).build();
            this.roleDescriptorsBytesCache = CacheBuilder.builder().setExpireAfterAccess(TimeValue.timeValueHours((long)1L)).setMaximumWeight((long)maximumWeight * 2L).build();
            this.lockingAtomicCounter = new LockingAtomicCounter();
        }

        public ApiKeyDoc get(String docId) {
            CachedApiKeyDoc existing = (CachedApiKeyDoc)this.docCache.get((Object)docId);
            if (existing != null) {
                BytesReference roleDescriptorsBytes = (BytesReference)this.roleDescriptorsBytesCache.get((Object)existing.roleDescriptorsHash);
                BytesReference limitedByRoleDescriptorsBytes = (BytesReference)this.roleDescriptorsBytesCache.get((Object)existing.limitedByRoleDescriptorsHash);
                if (roleDescriptorsBytes != null && limitedByRoleDescriptorsBytes != null) {
                    return existing.toApiKeyDoc(roleDescriptorsBytes, limitedByRoleDescriptorsBytes);
                }
            }
            return null;
        }

        public long getInvalidationCount() {
            return this.lockingAtomicCounter.get();
        }

        public void putIfNoInvalidationSince(String docId, ApiKeyDoc apiKeyDoc, long invalidationCount) {
            CachedApiKeyDoc cachedApiKeyDoc = apiKeyDoc.toCachedApiKeyDoc();
            this.lockingAtomicCounter.compareAndRun(invalidationCount, () -> {
                this.docCache.put((Object)docId, (Object)cachedApiKeyDoc);
                try {
                    this.roleDescriptorsBytesCache.computeIfAbsent((Object)cachedApiKeyDoc.roleDescriptorsHash, k -> apiKeyDoc.roleDescriptorsBytes);
                    this.roleDescriptorsBytesCache.computeIfAbsent((Object)cachedApiKeyDoc.limitedByRoleDescriptorsHash, k -> apiKeyDoc.limitedByRoleDescriptorsBytes);
                }
                catch (ExecutionException e) {
                    throw new RuntimeException(e);
                }
            });
        }

        public void invalidate(Collection<String> docIds) {
            this.lockingAtomicCounter.increment();
            logger.debug("Invalidating API key doc cache with ids: [{}]", (Object)Strings.collectionToCommaDelimitedString(docIds));
            docIds.forEach(arg_0 -> this.docCache.invalidate(arg_0));
        }

        public void invalidateAll() {
            this.lockingAtomicCounter.increment();
            logger.debug("Invalidating all API key doc cache and descriptor cache");
            this.docCache.invalidateAll();
            this.roleDescriptorsBytesCache.invalidateAll();
        }
    }

    public static final class ApiKeyCredentials
    implements Closeable {
        private final String id;
        private final SecureString key;

        public ApiKeyCredentials(String id, SecureString key) {
            this.id = id;
            this.key = key;
        }

        String getId() {
            return this.id;
        }

        SecureString getKey() {
            return this.key;
        }

        @Override
        public void close() {
            this.key.close();
        }
    }

    public static final class ApiKeyDoc {
        private static final BytesReference NULL_BYTES = new BytesArray("null");
        static final InstantiatingObjectParser<ApiKeyDoc, Void> PARSER;
        final String docType;
        final long creationTime;
        final long expirationTime;
        final Boolean invalidated;
        final String hash;
        @Nullable
        final String name;
        final int version;
        final BytesReference roleDescriptorsBytes;
        final BytesReference limitedByRoleDescriptorsBytes;
        final Map<String, Object> creator;
        @Nullable
        final BytesReference metadataFlattened;

        public ApiKeyDoc(String docType, long creationTime, long expirationTime, Boolean invalidated, String hash, @Nullable String name, int version, BytesReference roleDescriptorsBytes, BytesReference limitedByRoleDescriptorsBytes, Map<String, Object> creator, @Nullable BytesReference metadataFlattened) {
            this.docType = docType;
            this.creationTime = creationTime;
            this.expirationTime = expirationTime;
            this.invalidated = invalidated;
            this.hash = hash;
            this.name = name;
            this.version = version;
            this.roleDescriptorsBytes = roleDescriptorsBytes;
            this.limitedByRoleDescriptorsBytes = limitedByRoleDescriptorsBytes;
            this.creator = creator;
            this.metadataFlattened = NULL_BYTES.equals(metadataFlattened) ? null : metadataFlattened;
        }

        public CachedApiKeyDoc toCachedApiKeyDoc() {
            MessageDigest digest = MessageDigests.sha256();
            String roleDescriptorsHash = MessageDigests.toHexString((byte[])MessageDigests.digest((BytesReference)this.roleDescriptorsBytes, (MessageDigest)digest));
            digest.reset();
            String limitedByRoleDescriptorsHash = MessageDigests.toHexString((byte[])MessageDigests.digest((BytesReference)this.limitedByRoleDescriptorsBytes, (MessageDigest)digest));
            return new CachedApiKeyDoc(this.creationTime, this.expirationTime, this.invalidated, this.hash, this.name, this.version, this.creator, roleDescriptorsHash, limitedByRoleDescriptorsHash, this.metadataFlattened);
        }

        static ApiKeyDoc fromXContent(XContentParser parser) {
            return (ApiKeyDoc)PARSER.apply(parser, null);
        }

        static {
            InstantiatingObjectParser.Builder builder = InstantiatingObjectParser.builder((String)"api_key_doc", (boolean)true, ApiKeyDoc.class);
            builder.declareString(ConstructingObjectParser.constructorArg(), new ParseField("doc_type", new String[0]));
            builder.declareLong(ConstructingObjectParser.constructorArg(), new ParseField("creation_time", new String[0]));
            builder.declareLongOrNull(ConstructingObjectParser.constructorArg(), -1L, new ParseField("expiration_time", new String[0]));
            builder.declareBoolean(ConstructingObjectParser.constructorArg(), new ParseField("api_key_invalidated", new String[0]));
            builder.declareString(ConstructingObjectParser.constructorArg(), new ParseField("api_key_hash", new String[0]));
            builder.declareStringOrNull(ConstructingObjectParser.optionalConstructorArg(), new ParseField("name", new String[0]));
            builder.declareInt(ConstructingObjectParser.constructorArg(), new ParseField("version", new String[0]));
            ObjectParserHelper parserHelper = new ObjectParserHelper();
            parserHelper.declareRawObject((AbstractObjectParser)builder, ConstructingObjectParser.constructorArg(), new ParseField("role_descriptors", new String[0]));
            parserHelper.declareRawObject((AbstractObjectParser)builder, ConstructingObjectParser.constructorArg(), new ParseField("limited_by_role_descriptors", new String[0]));
            builder.declareObject(ConstructingObjectParser.constructorArg(), (p, c) -> p.map(), new ParseField("creator", new String[0]));
            parserHelper.declareRawObjectOrNull((AbstractObjectParser)builder, ConstructingObjectParser.optionalConstructorArg(), new ParseField("metadata_flattened", new String[0]));
            PARSER = builder.build();
        }
    }

    public static class ApiKeyRoleDescriptors {
        private final String apiKeyId;
        private final List<RoleDescriptor> roleDescriptors;
        private final List<RoleDescriptor> limitedByRoleDescriptors;

        public ApiKeyRoleDescriptors(String apiKeyId, List<RoleDescriptor> roleDescriptors, List<RoleDescriptor> limitedByDescriptors) {
            this.apiKeyId = apiKeyId;
            this.roleDescriptors = roleDescriptors;
            this.limitedByRoleDescriptors = limitedByDescriptors;
        }

        public String getApiKeyId() {
            return this.apiKeyId;
        }

        public List<RoleDescriptor> getRoleDescriptors() {
            return this.roleDescriptors;
        }

        public List<RoleDescriptor> getLimitedByRoleDescriptors() {
            return this.limitedByRoleDescriptors;
        }
    }

    private static class ApiKeyLoggingDeprecationHandler
    implements DeprecationHandler {
        private final DeprecationLogger deprecationLogger;
        private final String apiKeyId;

        private ApiKeyLoggingDeprecationHandler(DeprecationLogger logger, String apiKeyId) {
            this.deprecationLogger = logger;
            this.apiKeyId = apiKeyId;
        }

        public void usedDeprecatedName(String parserName, Supplier<XContentLocation> location, String usedName, String modernName) {
            String prefix = parserName == null ? "" : "[" + parserName + "][" + location.get() + "] ";
            this.deprecationLogger.deprecate(DeprecationCategory.SECURITY, "api_key_field", "{}Deprecated field [{}] used in api key [{}], expected [{}] instead", new Object[]{prefix, usedName, this.apiKeyId, modernName});
        }

        public void usedDeprecatedField(String parserName, Supplier<XContentLocation> location, String usedName, String replacedWith) {
            String prefix = parserName == null ? "" : "[" + parserName + "][" + location.get() + "] ";
            this.deprecationLogger.deprecate(DeprecationCategory.SECURITY, "api_key_field", "{}Deprecated field [{}] used in api key [{}], replaced by [{}]", new Object[]{prefix, usedName, this.apiKeyId, replacedWith});
        }

        public void usedDeprecatedField(String parserName, Supplier<XContentLocation> location, String usedName) {
            String prefix = parserName == null ? "" : "[" + parserName + "][" + location.get() + "] ";
            this.deprecationLogger.deprecate(DeprecationCategory.SECURITY, "api_key_field", "{}Deprecated field [{}] used in api key [{}], which is unused and will be removed entirely", new Object[]{prefix, usedName, this.apiKeyId});
        }
    }

    final class CachedApiKeyHashResult {
        final boolean success;
        final char[] hash;

        CachedApiKeyHashResult(boolean success, SecureString apiKey) {
            this.success = success;
            this.hash = ApiKeyService.this.cacheHasher.hash(apiKey);
        }

        boolean verify(SecureString password) {
            return this.hash != null && ApiKeyService.this.cacheHasher.verify(password, this.hash);
        }
    }

    public static final class CachedApiKeyDoc {
        final long creationTime;
        final long expirationTime;
        final Boolean invalidated;
        final String hash;
        final String name;
        final int version;
        final Map<String, Object> creator;
        final String roleDescriptorsHash;
        final String limitedByRoleDescriptorsHash;
        @Nullable
        final BytesReference metadataFlattened;

        public CachedApiKeyDoc(long creationTime, long expirationTime, Boolean invalidated, String hash, String name, int version, Map<String, Object> creator, String roleDescriptorsHash, String limitedByRoleDescriptorsHash, @Nullable BytesReference metadataFlattened) {
            this.creationTime = creationTime;
            this.expirationTime = expirationTime;
            this.invalidated = invalidated;
            this.hash = hash;
            this.name = name;
            this.version = version;
            this.creator = creator;
            this.roleDescriptorsHash = roleDescriptorsHash;
            this.limitedByRoleDescriptorsHash = limitedByRoleDescriptorsHash;
            this.metadataFlattened = metadataFlattened;
        }

        public ApiKeyDoc toApiKeyDoc(BytesReference roleDescriptorsBytes, BytesReference limitedByRoleDescriptorsBytes) {
            return new ApiKeyDoc("api_key", this.creationTime, this.expirationTime, this.invalidated, this.hash, this.name, this.version, roleDescriptorsBytes, limitedByRoleDescriptorsBytes, this.creator, this.metadataFlattened);
        }
    }
}

