/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authc.pki;

import java.security.MessageDigest;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.net.ssl.X509ExtendedTrustManager;
import javax.net.ssl.X509TrustManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.settings.SecureString;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ReleasableLock;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.env.Environment;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.watcher.ResourceWatcherService;
import org.elasticsearch.xpack.core.security.authc.AuthenticationResult;
import org.elasticsearch.xpack.core.security.authc.AuthenticationToken;
import org.elasticsearch.xpack.core.security.authc.Realm;
import org.elasticsearch.xpack.core.security.authc.RealmConfig;
import org.elasticsearch.xpack.core.security.authc.RealmSettings;
import org.elasticsearch.xpack.core.security.authc.pki.PkiRealmSettings;
import org.elasticsearch.xpack.core.security.user.User;
import org.elasticsearch.xpack.core.ssl.CertParsingUtils;
import org.elasticsearch.xpack.core.ssl.SSLConfigurationSettings;
import org.elasticsearch.xpack.security.authc.BytesKey;
import org.elasticsearch.xpack.security.authc.pki.X509AuthenticationToken;
import org.elasticsearch.xpack.security.authc.support.CachingRealm;
import org.elasticsearch.xpack.security.authc.support.DelegatedAuthorizationSupport;
import org.elasticsearch.xpack.security.authc.support.UserRoleMapper;
import org.elasticsearch.xpack.security.authc.support.mapper.CompositeRoleMapper;
import org.elasticsearch.xpack.security.authc.support.mapper.NativeRoleMappingStore;

public class PkiRealm
extends Realm
implements CachingRealm {
    public static final String PKI_CERT_HEADER_NAME = "__SECURITY_CLIENT_CERTIFICATE";
    private static final String AUTH_TYPE = "UNKNOWN";
    private final ReleasableLock readLock;
    private final ReleasableLock writeLock;
    private final X509TrustManager trustManager;
    private final Pattern principalPattern;
    private final UserRoleMapper roleMapper;
    private final Cache<BytesKey, User> cache;
    private DelegatedAuthorizationSupport delegatedRealms;

    public PkiRealm(RealmConfig config, ResourceWatcherService watcherService, NativeRoleMappingStore nativeRoleMappingStore) {
        this(config, new CompositeRoleMapper(config, watcherService, nativeRoleMappingStore));
    }

    PkiRealm(RealmConfig config, UserRoleMapper roleMapper) {
        super(config);
        ReentrantReadWriteLock iterationLock = new ReentrantReadWriteLock();
        this.readLock = new ReleasableLock(iterationLock.readLock());
        this.writeLock = new ReleasableLock(iterationLock.writeLock());
        this.trustManager = this.trustManagers(config);
        this.principalPattern = (Pattern)config.getSetting(PkiRealmSettings.USERNAME_PATTERN_SETTING);
        this.roleMapper = roleMapper;
        this.roleMapper.refreshRealmOnChange(this);
        this.cache = CacheBuilder.builder().setExpireAfterWrite((TimeValue)config.getSetting(PkiRealmSettings.CACHE_TTL_SETTING)).setMaximumWeight((long)((Integer)config.getSetting(PkiRealmSettings.CACHE_MAX_USERS_SETTING)).intValue()).build();
        this.delegatedRealms = null;
    }

    public void initialize(Iterable<Realm> realms, XPackLicenseState licenseState) {
        if (this.delegatedRealms != null) {
            throw new IllegalStateException("Realm has already been initialized");
        }
        this.delegatedRealms = new DelegatedAuthorizationSupport(realms, this.config, licenseState);
    }

    public boolean supports(AuthenticationToken token) {
        return token instanceof X509AuthenticationToken;
    }

    public X509AuthenticationToken token(ThreadContext context) {
        return PkiRealm.token(context.getTransient(PKI_CERT_HEADER_NAME), this.principalPattern, this.logger);
    }

    public void authenticate(AuthenticationToken authToken, ActionListener<AuthenticationResult> listener) {
        assert (this.delegatedRealms != null) : "Realm has not been initialized correctly";
        X509AuthenticationToken token = (X509AuthenticationToken)authToken;
        try {
            BytesKey fingerprint = PkiRealm.computeFingerprint(token.credentials()[0]);
            User user = (User)this.cache.get((Object)fingerprint);
            if (user != null) {
                if (this.delegatedRealms.hasDelegation()) {
                    this.delegatedRealms.resolve(token.principal(), listener);
                } else {
                    listener.onResponse((Object)AuthenticationResult.success((User)user));
                }
            } else if (!PkiRealm.isCertificateChainTrusted(this.trustManager, token, this.logger)) {
                listener.onResponse((Object)AuthenticationResult.unsuccessful((String)("Certificate for " + token.dn() + " is not trusted"), null));
            } else {
                ActionListener cachingListener = ActionListener.wrap(result -> {
                    if (result.isAuthenticated()) {
                        try (ReleasableLock ignored = this.readLock.acquire();){
                            this.cache.put((Object)fingerprint, (Object)result.getUser());
                        }
                    }
                    listener.onResponse(result);
                }, arg_0 -> listener.onFailure(arg_0));
                if (this.delegatedRealms.hasDelegation()) {
                    this.delegatedRealms.resolve(token.principal(), (ActionListener<AuthenticationResult>)cachingListener);
                } else {
                    this.buildUser(token, (ActionListener<AuthenticationResult>)cachingListener);
                }
            }
        }
        catch (CertificateEncodingException e) {
            listener.onResponse((Object)AuthenticationResult.unsuccessful((String)("Certificate for " + token.dn() + " has encoding issues"), (Exception)e));
        }
    }

    private void buildUser(X509AuthenticationToken token, ActionListener<AuthenticationResult> listener) {
        Map<String, Object> metadata = Collections.singletonMap("pki_dn", token.dn());
        UserRoleMapper.UserData userData = new UserRoleMapper.UserData(token.principal(), token.dn(), Collections.emptySet(), metadata, this.config);
        this.roleMapper.resolveRoles(userData, (ActionListener<Set<String>>)ActionListener.wrap(roles -> {
            User computedUser = new User(token.principal(), roles.toArray(new String[roles.size()]), null, null, metadata, true);
            listener.onResponse((Object)AuthenticationResult.success((User)computedUser));
        }, arg_0 -> listener.onFailure(arg_0)));
    }

    public void lookupUser(String username, ActionListener<User> listener) {
        listener.onResponse(null);
    }

    static X509AuthenticationToken token(Object pkiHeaderValue, Pattern principalPattern, Logger logger) {
        if (pkiHeaderValue == null) {
            return null;
        }
        assert (pkiHeaderValue instanceof X509Certificate[]);
        X509Certificate[] certificates = (X509Certificate[])pkiHeaderValue;
        if (certificates.length == 0) {
            return null;
        }
        String dn = certificates[0].getSubjectX500Principal().toString();
        Matcher matcher = principalPattern.matcher(dn);
        if (!matcher.find()) {
            if (logger.isDebugEnabled()) {
                logger.debug("certificate authentication succeeded for [{}] but could not extract principal from DN", (Object)dn);
            }
            return null;
        }
        String principal = matcher.group(1);
        if (Strings.isNullOrEmpty((String)principal)) {
            if (logger.isDebugEnabled()) {
                logger.debug("certificate authentication succeeded for [{}] but extracted principal was empty", (Object)dn);
            }
            return null;
        }
        return new X509AuthenticationToken(certificates, principal, dn);
    }

    static boolean isCertificateChainTrusted(X509TrustManager trustManager, X509AuthenticationToken token, Logger logger) {
        if (trustManager != null) {
            try {
                trustManager.checkClientTrusted(token.credentials(), AUTH_TYPE);
                return true;
            }
            catch (CertificateException e) {
                if (logger.isTraceEnabled()) {
                    logger.trace(() -> new ParameterizedMessage("failed certificate validation for principal [{}]", (Object)token.principal()), (Throwable)e);
                } else if (logger.isDebugEnabled()) {
                    logger.debug("failed certificate validation for principal [{}]", (Object)token.principal());
                }
                return false;
            }
        }
        return true;
    }

    X509TrustManager trustManagers(RealmConfig realmConfig) {
        List certificateAuthorities = realmConfig.hasSetting(PkiRealmSettings.CAPATH_SETTING) ? (List)realmConfig.getSetting(PkiRealmSettings.CAPATH_SETTING) : null;
        String truststorePath = ((Optional)realmConfig.getSetting(PkiRealmSettings.TRUST_STORE_PATH)).orElse(null);
        if (truststorePath == null && certificateAuthorities == null) {
            return null;
        }
        if (truststorePath != null && certificateAuthorities != null) {
            String pathKey = RealmSettings.getFullSettingKey((RealmConfig)realmConfig, (Setting.AffixSetting)PkiRealmSettings.TRUST_STORE_PATH);
            String caKey = RealmSettings.getFullSettingKey((RealmConfig)realmConfig, (Setting.AffixSetting)PkiRealmSettings.CAPATH_SETTING);
            throw new IllegalArgumentException("[" + pathKey + "] and [" + caKey + "] cannot be used at the same time");
        }
        if (truststorePath != null) {
            X509TrustManager trustManager = PkiRealm.trustManagersFromTruststore(truststorePath, realmConfig);
            if (trustManager.getAcceptedIssuers().length == 0) {
                this.logger.warn("PKI Realm {} uses truststore {} which has no accepted certificate issuers", (Object)this, (Object)truststorePath);
            }
            return trustManager;
        }
        X509TrustManager trustManager = PkiRealm.trustManagersFromCAs(certificateAuthorities, realmConfig.env());
        if (trustManager.getAcceptedIssuers().length == 0) {
            this.logger.warn("PKI Realm {} uses CAs {} with no accepted certificate issuers", (Object)this, (Object)certificateAuthorities);
        }
        return trustManager;
    }

    private static X509TrustManager trustManagersFromTruststore(String truststorePath, RealmConfig realmConfig) {
        if (!realmConfig.hasSetting(PkiRealmSettings.TRUST_STORE_PASSWORD) && !realmConfig.hasSetting(PkiRealmSettings.LEGACY_TRUST_STORE_PASSWORD)) {
            throw new IllegalArgumentException("Neither [" + RealmSettings.getFullSettingKey((RealmConfig)realmConfig, (Setting.AffixSetting)PkiRealmSettings.TRUST_STORE_PASSWORD) + "] or [" + RealmSettings.getFullSettingKey((RealmConfig)realmConfig, (Setting.AffixSetting)PkiRealmSettings.LEGACY_TRUST_STORE_PASSWORD) + "] is configured");
        }
        try (SecureString password = (SecureString)realmConfig.getSetting(PkiRealmSettings.TRUST_STORE_PASSWORD);){
            String trustStoreAlgorithm = (String)realmConfig.getSetting(PkiRealmSettings.TRUST_STORE_ALGORITHM);
            String trustStoreType = SSLConfigurationSettings.getKeyStoreType((Setting)realmConfig.getConcreteSetting(PkiRealmSettings.TRUST_STORE_TYPE), (Settings)realmConfig.settings(), (String)truststorePath);
            try {
                X509ExtendedTrustManager x509ExtendedTrustManager = CertParsingUtils.trustManager((String)truststorePath, (String)trustStoreType, (char[])password.getChars(), (String)trustStoreAlgorithm, (Environment)realmConfig.env());
                return x509ExtendedTrustManager;
            }
            catch (Exception e) {
                throw new IllegalArgumentException("failed to load specified truststore", e);
            }
        }
    }

    private static X509TrustManager trustManagersFromCAs(List<String> certificateAuthorities, Environment env) {
        assert (certificateAuthorities != null);
        try {
            Certificate[] certificates = CertParsingUtils.readCertificates(certificateAuthorities, (Environment)env);
            return CertParsingUtils.trustManager((Certificate[])certificates);
        }
        catch (Exception e) {
            throw new ElasticsearchException("failed to load certificate authorities for PKI realm", (Throwable)e, new Object[0]);
        }
    }

    @Override
    public void expire(String username) {
        try (ReleasableLock ignored = this.writeLock.acquire();){
            Iterator userIterator = this.cache.values().iterator();
            while (userIterator.hasNext()) {
                if (!((User)userIterator.next()).principal().equals(username)) continue;
                userIterator.remove();
            }
        }
    }

    @Override
    public void expireAll() {
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.cache.invalidateAll();
        }
    }

    private static BytesKey computeFingerprint(X509Certificate certificate) throws CertificateEncodingException {
        MessageDigest digest = MessageDigests.sha256();
        digest.update(certificate.getEncoded());
        return new BytesKey(digest.digest());
    }
}

