#include <cmath>
#include <set>
#include <map>
#include <vector>
#include <fstream>
#include <sstream>
#include <cstdlib>
#include "sentence.hpp"
#include "string_indexer.hpp"
#include "models.hpp"
// for parsing
#include "unlabeled_edge_feature_extractor_interface.hpp"
#include "parser_learner_interface.hpp"
#include "eda_unlabeled_edge_feature_extractor.hpp"
#include "chu_liu_edmonds.hpp"
// for labeling
#include "liblinear-1.91/linear.h"
#include "edge_label_feature_extractor_interface.hpp"
#include "simple_edge_label_feature_extractor.hpp"


namespace maeda {


/////////////////////////////////////////////////////////////////////////
// UnlabeledParseModel
/////////////////////////////////////////////////////////////////////////
namespace {
struct ExtractedFeatureData {
  StringIndexer string_indexer;
  std::vector<std::vector<IDString> > feature_string_list_list;
  std::vector<int> gold_standard_index_list;
};


void ExtractUnlabeledEdgeFeatures(
    const Sentence &sentence,
    const UnlabeledEdgeFeatureExtractorInterface* const extractor,
    const bool &left_to_right,
    ExtractedFeatureData *extracted) {
  // ʸñ (ROOTޤ)
  const int length = int(sentence.rows.size());
  // iȯ륨å򸫤
  // i == 0 ROOTǡROOTϽȯʤΤǥå
  for (int i = 1; i < length; i++) {
    const int target = sentence.rows[i].head;
    if (target != Sentence::kInvalidIndex) {
      // targetΤ߽
      std::vector<IDString> feature_string_list;
      // ʸΤɤñΰ֤鸫뤫
      // left_to_rightʤľΰ֤鸫롣Ǥʤʤǽ餫顣
      const int begin = left_to_right ? i + 1 : 0;

      for (int j =  begin; j < length; j++) {
        if (j != i) {
          // ʬˤϤʤ
          std::string row_feature_string;
          extractor->Extract(sentence, i, j, &row_feature_string);

          std::stringstream ss(row_feature_string);
          std::string feature;
          IDString feature_string;
          while (ss >> feature) {
            feature_string.push_back(extracted->string_indexer.Add(feature));
          }
          feature_string_list.push_back(feature_string);
        }
      }
      extracted->feature_string_list_list.push_back(feature_string_list);

      int gold_standard_index;
      if (left_to_right) {
        gold_standard_index = target - begin;
      } else {
        // ʬȤȴΤǡtargetiĶ-1
        gold_standard_index = (i < target) ? target - 1 : target;
      }
      extracted->gold_standard_index_list.push_back(gold_standard_index);
    }
  }
}


void ReportLoadErrorAndDie(const std::string &filename) {
  std::cerr << "Format error: " << filename << std::endl;
  exit(1);
}
} // namespace

UnlabeledParseModel::UnlabeledParseModel()
  : extractor_(NULL), left_to_right_(false) {}

UnlabeledParseModel::UnlabeledParseModel(
   const UnlabeledEdgeFeatureExtractorInterface* const extractor,
   const bool &left_to_right)
  : extractor_(extractor), left_to_right_(left_to_right) {}

void UnlabeledParseModel::Save(const std::string &filename) const {
  std::ofstream stream(filename.c_str());

  if (!stream) {
    std::cerr << "Error: could not open file " << filename << std::endl;
    exit(1);
  }

  std::cerr << "Printing to: " << filename << std::endl;

  stream << "TEXT" << std::endl;

  if (left_to_right_) { stream << "LEFT_TO_RIGHT" << std::endl; }

  stream << "EXTRACTOR" << std::endl;
  stream << extractor_->name() << std::endl;

  stream << "FEATURE_NAME WEIGHT" << std::endl;
  stream << weight_.size() << std::endl;
  for (FeatureID f = 0; f < FeatureID(weight_.size()); f++) {
    stream << feature_indexer_.GetStr(f) << "\t"
           << weight_[f] << std::endl;
  }
}

void UnlabeledParseModel::Load(const std::string &filename) {
  std::ifstream stream(filename.c_str());

  if (!stream) {
    std::cerr << "Error: could not open file " << filename << std::endl;
    exit(1);
  }

  std::string line;

  std::getline(stream, line);
  if (line != "TEXT") {
    ReportLoadErrorAndDie(filename);
  }

  std::getline(stream, line);
  if (line == "LEFT_TO_RIGHT") {
    left_to_right_ = true;
    std::getline(stream, line);  
  }

  if (line == "EXTRACTOR") {
    std::getline(stream, line);
    extractor_ = new EdaUnlabeledEdgeFeatureExtractor;
  } else {
    ReportLoadErrorAndDie(filename);
  }

  std::getline(stream, line);
  if (line ==  "FEATURE_NAME WEIGHT") {
    std::getline(stream, line);
    weight_.resize( size_t(atoi(line.c_str())) );
    std::string f;
    double weight;
    while (std::getline(stream, line)) {
      std::stringstream ss(line);
      ss >> f >> weight;
      FeatureID fid = feature_indexer_.Add(f);
      weight_[fid] = weight;
    }
  } else {
    ReportLoadErrorAndDie(filename);
  }

  return;
}

void UnlabeledParseModel::SaveFeatureVectors(
    const std::string &filename) const {
  // ٥ե¸
  // std::ofstream stream(filename.c_str());

  // stream << extractor_->name() << std::endl;

  // for (int n = 0; n < int(feature_string_list_list_.size()); n++) {
  //   stream << gold_standard_index_list_[n] << std::endl;
  //   for (std::vector<std::string>::const_iterator it =
  //          feature_string_list_list_[n].begin();
  //        it != feature_string_list_list_[n].end(); it++) {
  //     stream << *it << std::endl;
  //   }
  //   stream << std::endl;
  // }
}


void UnlabeledParseModel::PrintArcScores(
    std::ostream &out, const Sentence &sentence) const {
  // (ROOTޤ)ʸñ = դΥΡɿ
  const size_t N = sentence.rows.size();
  // å
  std::vector<std::vector<double> > score(N, std::vector<double>(N, -1.0));

  // iܤȯ륨åΥȤ׻
  // MSTǤϷθȥդθդˤʤ뤳Ȥ
  // ROOTϽȯʤΤǡi = 0 ϥå
  for (size_t i = 1; i < N; i++) {
    const size_t begin = left_to_right_ ? i + 1 : 0;
    double sum = 0;
    for (size_t j = begin; j < N; j++) {
      if (j != i) {
        std::string feature_string;
        extractor_->Extract(sentence, i, j, &feature_string);

        std::stringstream ss(feature_string);
        FeatureVector feature_vector;
        std::string feature;
        while (ss >> feature) {
          FeatureID fid = feature_indexer_.GetID(feature);
          if (fid != kInvalidFeatureID) {
            feature_vector.push_back( Feature(fid, 1.0) );
          }
        }
        // Ȥϵոͤ򥻥å
        score[j][i] = exp(feature_vector.Product(weight_));
        sum += score[j][i];
      }
    }
    // 
    for (size_t j = begin; j < N; j++) {
      if (j != i) { score[j][i] /= sum; }
    }
  }


  // print legend
  out << "src\\dst";
  for (size_t i = 0; i < N; i++) {
    out << "\t" << sentence.rows[i].form;
  }
  out << std::endl;

  for (size_t src = 0; src < N; src++) {
    out << sentence.rows[src].form;
    for (size_t dst = 0; dst < N; dst++) {
      out << "\t" << score[dst][src];
    }
    out << std::endl;
  }
  out << std::endl;

  return;
}

void UnlabeledParseModel::ParseSentenceWithMST(
    const bool &named_entity_p, Sentence *sentence) const {
  // (ROOTޤ)ʸñ = դΥΡɿ
  const size_t N = sentence->rows.size();
  // å
  std::vector<std::vector<double> > score(N, std::vector<double>(N, -1.0));

  // iܤȯ륨åΥȤ׻
  // MSTǤϷθȥդθդˤʤ뤳Ȥ
  // ROOTϽȯʤΤǡi = 0 ϥå
  for (size_t i = 1; i < N; i++) {
    const size_t begin = left_to_right_ ? i + 1 : 0;
    double sum = 0;
    for (size_t j = begin; j < N; j++) {
      if (j != i) {
        std::string feature_string;
        extractor_->Extract(*sentence, i, j, &feature_string);

        std::stringstream ss(feature_string);
        FeatureVector feature_vector;
        std::string feature;
        while (ss >> feature) {
          FeatureID fid = feature_indexer_.GetID(feature);
          if (fid != kInvalidFeatureID) {
            feature_vector.push_back( Feature(fid, 1.0) );
          }
        }
        // Ȥϵոͤ򥻥å
        score[j][i] = exp(feature_vector.Product(weight_));
        sum += score[j][i];
      }
    }
    // 
    for (size_t j = begin; j < N; j++) {
      if (j != i) { score[j][i] /= sum; }
    }
  }


  std::vector<size_t> excluded_indices;
  if (named_entity_p) {
    if (GetExcludedIndicesByNE(*sentence, &excluded_indices)) {
      for (std::vector<size_t>::const_iterator it = excluded_indices.begin();
           it != excluded_indices.end(); ++it) {
        //std::cerr << *it << " " << std::endl;
        for (size_t i = 0; i < N; i++) { score[i][*it] = score[*it][i] = -1; }
      }
    } else {
      std::cerr << "Warning: invalid NE tagging." << std::endl;
    }
  }

  std::vector<int> vs(N, 1);
  std::vector<std::pair<int, int> > edges;
  // ڤ׻
  ChuLiuEdmonds(N, score, vs, &edges);

  for (size_t i = 1; i < N; i++) {
    sentence->rows[i].head = Sentence::kInvalidIndex;
  }

  for (size_t k = 0; k < edges.size(); k++) {
    sentence->rows[edges[k].second].head = edges[k].first;
  }

  for (std::vector<size_t>::const_iterator it = excluded_indices.begin();
       it != excluded_indices.end(); ++it) {
    sentence->rows[*it].head = *it + 1;
  }
}


bool UnlabeledParseModel::Train(const std::vector<Sentence> &sentence_list,
                                ParserLearnerInterface *learner,
                                const int cutoff) {
  ExtractedFeatureData extracted;
   // extract all features
  std::cerr << "Extracting features..."  << std::endl;
  for (size_t i = 0; i < sentence_list.size(); i++) {
    ExtractUnlabeledEdgeFeatures(sentence_list[i],
                                 extractor_,
                                 left_to_right_,
                                 &extracted);
    if (0 < i && i % 1000 == 0) {
      std::cerr << "Read " << i << " of "
                << sentence_list.size() << " sentences." << std::endl;
    }
  }
  std::cerr << "Done." << std::endl;

  const size_t training_data_size = extracted.gold_standard_index_list.size();


  //  <->  map
  std::cerr << "Indexing features..."  << std::endl;
  std::map<FeatureID, int> count;
  std::map<FeatureID, FeatureID> feature_index_indexer;
  for (size_t n = 0; n < training_data_size; n++) {
    for (size_t t = 0;
         t < extracted.feature_string_list_list[n].size(); t++) {
      const IDString &feature_string =
          extracted.feature_string_list_list[n][t];
      for (IDString::const_iterator it = feature_string.begin();
           it != feature_string.end(); it++) {
        if (0 <= count[*it] && cutoff < ++count[*it]) {
          const FeatureID fid = FeatureID(feature_index_indexer.size());
          feature_index_indexer[*it] = fid;
          count[*it] = -1; // `*it' will be ignored from now on.
        }
      }
    }
    if (0 < n && n % 10000 == 0) {
      std::cerr << "Read " << n << " of "
                << training_data_size << " data." << std::endl;
    }
  }
  // ꤹǥХå
  // {
  //   StringIndexer tmp;
  //   tmp.Load("vocab.dat");
  //   for (size_t i = 0; i < tmp.size(); i++) {
  //     std::cerr <<tmp.GetStr(i) << " ("
  //               << extracted.string_indexer.GetID(tmp.GetStr(i))
  //               << ") -> " << i << std::endl;
  //     FeatureID fid = extracted.string_indexer.GetID(tmp.GetStr(i));
  //     feature_index_indexer[fid] = i;
  //   }
  // }
  std::cerr << "Done." << std::endl;
  std::cerr << "#features: " << feature_index_indexer.size() << std::endl;


  // Generate feature vector lists
  std::vector<FeatureVectorList> feature_vector_list_list(training_data_size);

  std::cerr << "Building training data..."  << std::endl;
  for (size_t n = 0; n < training_data_size; n++) {
    // map feature_string -> feature_vector
    const size_t num_targets = extracted.feature_string_list_list[n].size();
    feature_vector_list_list[n].resize(num_targets);
    for (size_t t = 0; t < num_targets; t++) {
      FeatureVector &feature_vector = feature_vector_list_list[n][t];
      const IDString &feature_string =
          extracted.feature_string_list_list[n][t];
      for (IDString::const_iterator it = feature_string.begin();
           it != feature_string.end(); ++it) {
        if (feature_index_indexer.find(*it) != feature_index_indexer.end()) {
          const FeatureID fid = feature_index_indexer[*it];
          feature_vector.push_back(Feature(fid, 1.0));
        }
      }
    }
    if (0 < n && n % 10000 == 0) {
      std::cerr << "Read " << n << " of "
                << training_data_size << " data." << std::endl;
    }
  }
  std::cerr << "Done." << std::endl;


  weight_.resize(feature_index_indexer.size());

  learner->Learn(&feature_vector_list_list,
                 &extracted.gold_standard_index_list,
                 &weight_);

  std::cerr << "Saving features..." << std::endl;
  std::vector<FeatureID> feature_id_list(feature_index_indexer.size());
  for (std::map<FeatureID, FeatureID>::const_iterator it =
         feature_index_indexer.begin();
       it != feature_index_indexer.end(); ++it) {
    feature_id_list[it->second] = it->first;
  }
  for (std::vector<FeatureID>::const_iterator it = feature_id_list.begin();
       it != feature_id_list.end(); it++) {
    feature_indexer_.Add(extracted.string_indexer.GetStr(*it));
  }

  return true;
}




/////////////////////////////////////////////////////////////////////////
// EdgeLabelModel
/////////////////////////////////////////////////////////////////////////

namespace {
void ExtractEdgeLabelFeatures(
    const Sentence &sentence,
    const EdgeLabelFeatureExtractorInterface* const extractor,
    std::vector<std::string> *feature_string_list,
    std::vector<std::string> *gold_standard_label_list) {
  const int length = int(sentence.rows.size());
  for (int i = 0; i < length; i++) {
    if (sentence.rows[i].head != Sentence::kInvalidIndex) {
      std::string feature_string;
      extractor->Extract(sentence, i, sentence.rows[i].head,
                         &feature_string);
      feature_string_list->push_back(feature_string);
      gold_standard_label_list->push_back(sentence.rows[i].deprel);
    }
  }
}
} // namespace


EdgeLabelModel::EdgeLabelModel()
  : extractor_(NULL), model_(NULL) {}

EdgeLabelModel::EdgeLabelModel(
    const EdgeLabelFeatureExtractorInterface* const extractor)
  : extractor_(extractor), model_(NULL) {}


void EdgeLabelModel::LabelEdges(Sentence *sentence) {
  for (int source = 1; source < int(sentence->rows.size()); source++) {
    std::string feature_string;
    extractor_->Extract(*sentence, source, sentence->rows[source].head,
                        &feature_string);

    std::stringstream ss(feature_string);
    std::string str;
    std::vector<FeatureID> features;
    while (ss >> str) {
      FeatureID fid = feature_indexer_.GetID(str);
      if (fid != kInvalidFeatureID) {
        features.push_back(fid);
      }
    }
    std::sort(features.begin(), features.end());

    std::vector<struct feature_node> x(features.size()+2);
    for (size_t j = 0; j < features.size(); j++) {
      x[j].index = int(features[j])+1;
      x[j].value = 1.0;
    }
    x[features.size()].index = -1;

    double predict_label = predict(model_, &x.front());

    sentence->rows[source].deprel =
        label_indexer_.GetStr(int(predict_label)-1);
  }
}


bool EdgeLabelModel::Train(const std::vector<Sentence> &sentence_list) {
  std::cerr << "Extracting features..."  << std::endl;
  for (size_t i = 0; i < sentence_list.size(); i++) {
    ExtractEdgeLabelFeatures(sentence_list[i],
                             extractor_,
                             &feature_string_list_,
                             &gold_standard_label_list_);
    if (0 < i && i % 1000 == 0) {
      std::cerr << "Read " << i << " of "
                << sentence_list.size() << " Sentences." << std::endl;
    }
  }
  std::cerr << "Done." << std::endl;


  const size_t training_data_size = gold_standard_label_list_.size();

  std::cerr << "Indexing features..."  << std::endl;
  //std::map<std::string, int> counter;
  for (size_t i = 0; i < training_data_size; i++) {
    label_indexer_.Add(gold_standard_label_list_[i]);
    std::stringstream ss(feature_string_list_[i]);
    std::string str;
    while (ss >> str) {
      feature_indexer_.Add(str);
    }
    if (0 < i && i % 10000 == 0) {
      std::cerr << "Read " << i << " of "
                << training_data_size << " data." << std::endl;
    }
  }
  //for (size_t i = 0;  i < feature_indexer_.size(); i++) {
  //std::cout << feature_indexer_.GetStr(i) << std::endl;
  //}

  std::cerr << "Done." << std::endl;
  std::cerr << "#features: " << feature_indexer_.size() << std::endl;


  std::cerr << "Building training data..."  << std::endl;
  FeatureID bias_id = kInvalidFeatureID;
  double bias_value = -1.0;
  std::vector<struct feature_node *> xs(training_data_size);
  std::vector<double> ys(training_data_size);
  for (size_t i = 0; i < training_data_size; i++) {
    // +1 because LibLinear uses 1-origin
    ys[i] = label_indexer_.GetID(gold_standard_label_list_[i])+1;
    //std::cout << ys[i];

    std::stringstream ss(feature_string_list_[i]);
    std::string str;
    std::vector<FeatureID> features;
    while (ss >> str) {
      FeatureID fid = feature_indexer_.GetID(str);
      if (fid != kInvalidFeatureID) { features.push_back(fid); }
    }
    std::sort(features.begin(), features.end());
    xs[i] = new struct feature_node[(features.size()+2)];
    for (size_t j = 0; j < features.size(); j++) {
      // +1 because LibLinear uses 1-origin
      xs[i][j].index = int(features[j])+1;
      xs[i][j].value = 1.0;
      //std::cout << " " << xs[i][j].index << ":1";
    }
    //std::cout << std::endl;
    if (0 <= bias_value) {
      xs[i][features.size()].index = bias_id;
      xs[i][features.size()].value = bias_value;
      xs[i][features.size()+1].index = -1;
    } else {
      xs[i][features.size()].index = -1;
    }

    if (0 < i && i % 10000 == 0) {
      std::cerr << "Read " << i << " of "
                << training_data_size << " data." << std::endl;
    }
  }
  std::cerr << "Done." << std::endl;

  struct problem prob;
  prob.l = training_data_size;
  prob.x = &xs.front();
  prob.y = &ys.front();
  prob.n = int(feature_indexer_.size());
  prob.bias = bias_id;

  struct parameter param;
  param.solver_type = L2R_L2LOSS_SVC_DUAL;
  param.C = 1;
  param.eps = 0.1;
  param.p = 0.1;
  param.nr_weight = 0;
  param.weight_label = NULL;
  param.weight = NULL;

  model_ = train(&prob, &param);

  for (size_t i = 0; i < training_data_size; i++) { delete [] xs[i]; }
  destroy_param(&param);

  return true;
}


namespace {
static const char *solver_type_table[]= {
  "L2R_LR", "L2R_L2LOSS_SVC_DUAL",
  "L2R_L2LOSS_SVC", "L2R_L1LOSS_SVC_DUAL", "MCSVM_CS",
  "L1R_L2LOSS_SVC", "L1R_LR", "L2R_LR_DUAL", "", "", "",
  "L2R_L2LOSS_SVR", "L2R_L2LOSS_SVR_DUAL", "L2R_L1LOSS_SVR_DUAL", NULL,
};
} // namespace

void EdgeLabelModel::Save(const std::string &filename) const {
  std::cerr << "Printing to: " << filename << std::endl;

	FILE *fp = fopen(filename.c_str(),"w");
	if(fp==NULL) {
    std::cerr << "Error: could not open file " << filename << std::endl;
    exit(1);
  }

	char *old_locale = strdup(setlocale(LC_ALL, NULL));
	setlocale(LC_ALL, "C");


  fprintf(fp, "TEXT\n");

  fprintf(fp, "EXTRACTOR\n");
  fprintf(fp, "%s\n", extractor_->name().c_str());

  fprintf(fp, "LABEL\n");
  fprintf(fp, "%d\n", int(label_indexer_.size()));
  for (FeatureID f = 0; f < FeatureID(label_indexer_.size()); f++) {
    fprintf(fp, "%s\n", label_indexer_.GetStr(f).c_str());
  }
  fprintf(fp, "FEATURE_NAME WEIGHT\n");
  fprintf(fp, "%d\n", int(feature_indexer_.size()));
  for (FeatureID f = 0; f < FeatureID(feature_indexer_.size()); f++) {
    fprintf(fp, "%s\n", feature_indexer_.GetStr(f).c_str());
  }
  fprintf(fp, "\n");

	int i;
	int nr_feature=model_->nr_feature;
	int n;
	const parameter& param = model_->param;

	if(model_->bias>=0)
		n=nr_feature+1;
	else
		n=nr_feature;
	int w_size = n;

	int nr_w;
	if(model_->nr_class==2 && model_->param.solver_type != MCSVM_CS)
		nr_w=1;
	else
		nr_w=model_->nr_class;

	fprintf(fp, "solver_type %s\n", solver_type_table[param.solver_type]);
	fprintf(fp, "nr_class %d\n", model_->nr_class);
	
	if(model_->label)
	{
		fprintf(fp, "label");
		for(i=0; i<model_->nr_class; i++)
			fprintf(fp, " %d", model_->label[i]);
		fprintf(fp, "\n");
	}

	fprintf(fp, "nr_feature %d\n", nr_feature);

	fprintf(fp, "bias %.16g\n", model_->bias);

	fprintf(fp, "w\n");
	for(i=0; i<w_size; i++)
	{
		int j;
		for(j=0; j<nr_w; j++)
			fprintf(fp, "%.16g ", model_->w[i*nr_w+j]);
		fprintf(fp, "\n");
	}

	setlocale(LC_ALL, old_locale);
	free(old_locale);

	if (ferror(fp) != 0 || fclose(fp) != 0) {
    std::cerr << "Error: error occurs while saving. " << std::endl;
  }
}


void EdgeLabelModel::Load(const std::string &filename) {
	FILE *fp = fopen(filename.c_str(),"r");
	if(fp==NULL) {
    std::cerr << "Error: could not open file " << filename << std::endl;
    exit(1);
  }

	char *old_locale = strdup(setlocale(LC_ALL, NULL));
	setlocale(LC_ALL, "C");

  char row[1024];

  fgets(row, sizeof(row), fp);
  if (strcmp(row, "TEXT\n") != 0) {
    ReportLoadErrorAndDie(filename);
  }

  fgets(row, sizeof(row), fp);
  if (strcmp(row, "EXTRACTOR\n") == 0) {
    fgets(row, sizeof(row), fp);
    extractor_ = new SimpleEdgeLabelFeatureExtractor;
  } else {
    ReportLoadErrorAndDie(filename);
  }

  fgets(row, sizeof(row), fp);
  if (strcmp(row, "LABEL\n") == 0) {
    fgets(row, sizeof(row), fp);
    int num_labels;
    sscanf(row, "%d", &num_labels);
    for (int i = 0; i < num_labels; i++) {
      fgets(row, sizeof(row), fp);
      std::string s(row);
      size_t pos = s.find_last_not_of("\n");
      label_indexer_.Add(s.substr(0, pos+1));
    }
  } else {
    ReportLoadErrorAndDie(filename);
  }

  fgets(row, sizeof(row), fp);
  if (strcmp(row, "FEATURE_NAME WEIGHT\n") == 0) {
    fgets(row, sizeof(row), fp);
    int num_features;
    sscanf(row, "%d", &num_features);
    for (int i = 0; i < num_features; i++) {
      fgets(row, sizeof(row), fp);
      std::string s(row);
      size_t pos = s.find_last_not_of("\n");
      feature_indexer_.Add(s.substr(0, pos+1));
    }
  } else {
    ReportLoadErrorAndDie(filename);
  }

  fgets(row, sizeof(row), fp);
  if (strcmp(row, "\n") != 0) {
    ReportLoadErrorAndDie(filename);
  }


#define Malloc(type,n) (type *)malloc((n)*sizeof(type))

	int i;
	int nr_feature;
	int n;
	int nr_class;
	double bias;
	model_ = Malloc(struct model,1);
	parameter& param = model_->param;

	model_->label = NULL;

	char cmd[81];
	while(1)
	{
		fscanf(fp,"%80s",cmd);
		if(strcmp(cmd,"solver_type")==0)
		{
			fscanf(fp,"%80s",cmd);
			int i;
			for(i=0;solver_type_table[i];i++)
			{
				if(strcmp(solver_type_table[i],cmd)==0)
				{
					param.solver_type=i;
					break;
				}
			}
			if(solver_type_table[i] == NULL)
			{
				fprintf(stderr,"unknown solver type.\n");
				
				setlocale(LC_ALL, old_locale);
				free(model_->label);
				free(model_);
				free(old_locale);
				exit(1);;
			}
		}
		else if(strcmp(cmd,"nr_class")==0)
		{
			fscanf(fp,"%d",&nr_class);
			model_->nr_class=nr_class;
		}
		else if(strcmp(cmd,"nr_feature")==0)
		{
			fscanf(fp,"%d",&nr_feature);
			model_->nr_feature=nr_feature;
		}
		else if(strcmp(cmd,"bias")==0)
		{
			fscanf(fp,"%lf",&bias);
			model_->bias=bias;
		}
		else if(strcmp(cmd,"w")==0)
		{
			break;
		}
		else if(strcmp(cmd,"label")==0)
		{
			int nr_class = model_->nr_class;
			model_->label = Malloc(int,nr_class);
			for(int i=0;i<nr_class;i++)
				fscanf(fp,"%d",&model_->label[i]);
		}
		else
		{
			fprintf(stderr,"unknown text in model file: [%s]\n",cmd);
			setlocale(LC_ALL, old_locale);
			free(model_->label);
			free(model_);
			free(old_locale);
			exit(1);
		}
	}

	nr_feature=model_->nr_feature;
	if(model_->bias>=0)
		n=nr_feature+1;
	else
		n=nr_feature;
	int w_size = n;
	int nr_w;
	if(nr_class==2 && param.solver_type != MCSVM_CS)
		nr_w = 1;
	else
		nr_w = nr_class;

	model_->w=Malloc(double, w_size*nr_w);
	for(i=0; i<w_size; i++)
	{
		int j;
		for(j=0; j<nr_w; j++)
			fscanf(fp, "%lf ", &model_->w[i*nr_w+j]);
		fscanf(fp, "\n");
	}
	
	setlocale(LC_ALL, old_locale);
	free(old_locale);
	
	if (ferror(fp) != 0 || fclose(fp) != 0) {
    std::cerr << "Error: error occurs while saving. " << std::endl;
  }

  return;
}



} // maeda

