/*
 * Copyright (c) 1997-2006  Motoyuki Kasahara
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "build-pre.h"

/*
 * Convert a string from JIS X 0208 to EUC JP.
 */
void
eb_jisx0208_to_euc(char *out_string, const char *in_string)
{
    unsigned char *out_p = (unsigned char *)out_string;
    const unsigned char *in_p = (unsigned char *)in_string;

    while (*in_p != '\0')
	*out_p++ = ((*in_p++) | 0x80);

    *out_p = '\0';
}


/*
 * Convert a string from shift-JIS to EUC JP.
 * (Shift-JIS is used only in the `LANGUAGE' file.)
 */
void
eb_sjis_to_euc(char *out_string, const char *in_string)
{
    unsigned char *out_p = (unsigned char *)out_string;
    const unsigned char *in_p = (unsigned char *)in_string;
    unsigned char c1, c2;

    for (;;) {
	/*
	 * Break at '\0'.
	 */
	c1 = *in_p++;
	if (c1 == '\0')
	    break;

	if (c1 <= 0x7f) {
	    /*
	     * JIS X 0201 Roman character.
	     */
	    *out_p++ = c1;
	} else if (0xa1 <= c1 && c1 <= 0xdf) {
	    /*
	     * JIS X 0201 Kana.
	     */
	    *out_p++ = ' ';
	} else {
	    /*
	     * JIS X 0208 character.
	     */
	    c2 = *in_p++;
	    if (c2 == 0x00)
		break;

	    if (c2 < 0x9f) {
		if (c1 < 0xdf)
		    c1 = ((c1 - 0x30) << 1) - 1;
		else
		    c1 = ((c1 - 0x70) << 1) - 1;

		if (c2 < 0x7f)
		    c2 += 0x61;
		else
		    c2 += 0x60;
	    } else {
		if (c1 < 0xdf)
		    c1 = (c1 - 0x30) << 1;
		else
		    c1 = (c1 - 0x70) << 1;
		c2 += 0x02;
	    }

	    *out_p++ = c1;
	    *out_p++ = c2;
	}
    }

    *out_p = '\0';
}


/*
 * Convert a string from EBCDIC code page 37 to ascii.
 * non-ascii characters are converted into SP.
 */
static const char ebcdic_ascii_table[] = {
  0x00, 0x01, 0x02, 0x03, 0x20, 0x09, 0x20, 0x7f,
  0x20, 0x20, 0x20, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
  0x10, 0x11, 0x12, 0x13, 0x20, 0x20, 0x08, 0x20,
  0x18, 0x19, 0x20, 0x20, 0x1c, 0x1d, 0x1e, 0x1f,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x0a, 0x17, 0x1b,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x05, 0x06, 0x07,
  0x20, 0x20, 0x16, 0x20, 0x20, 0x20, 0x20, 0x04,
  0x20, 0x20, 0x20, 0x20, 0x15, 0x15, 0x20, 0x1a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x2e, 0x3c, 0x28, 0x2b, 0x7c,
  0x26, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x21, 0x24, 0x2a, 0x29, 0x3b, 0x20,
  0x2d, 0x2f, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x2c, 0x25, 0x5f, 0x3e, 0x3f,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x60, 0x3a, 0x23, 0x40, 0x27, 0x3d, 0x22,
  0x20, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
  0x68, 0x69, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70,
  0x71, 0x72, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x7e, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
  0x79, 0x7a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x5e, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x5b, 0x5d, 0x20, 0x20, 0x20, 0x20,
  0x7b, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
  0x48, 0x49, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x7d, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50,
  0x51, 0x52, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x5c, 0x20, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
  0x59, 0x5a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
  0x38, 0x39, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
};

void
eb_ebcdic037_to_ascii(char *out_string, const char *in_string)
{
  while (*in_string != 0) {
    *out_string = ebcdic_ascii_table[(unsigned char)*in_string];
    out_string++;
    in_string++;
  }
}
