#feedster.py
#
# Save this document as feedster.py and put it anywhere in your PYTHONPATH
#
#Provides one function, getFeeds, which uses the Feedster API to get info 
# about a URL:
#
# Example:
#
# import feedster
# feedster.getFeeds(url,type)
#
# getFeeds returns a list of standard python dictionaries.
#
# Read API documentation at http://varchars.com/blog/node/view/138 to 
# understand the "type" argument.
# Get your private key here: http://feedster.com/myfeedster.php?action=showkeys
#
# Copyright 2004 Ross M Karchner
# BSD Licensed: http://www.opensource.org/licenses/bsd-license.php
#
# TODO: Error handling, docstrings, and everything else that make this usable
#
#
#

__copyright__ = 'Copyright 2004 Ross M Karchner'

import rsssax
import os, sys, xml.sax, urllib

LICENSE_KEY = None

_feedsterfile1 = ".feedsterkey"
_feedsterfile2 = "feedsterkey.txt"

_licenseLocations = (
    ( lambda key: key,
      'passed to the function in license_key variable' ),
    ( lambda key: LICENSE_KEY, 
      'module-level LICENSE_KEY variable (call setLicense to set it)' ),
    ( lambda key: os.environ.get( 'FEEDSTER_LICENSE_KEY', None ),
      'an environment variable called FEEDSTER_LICENSE_KEY' ),
    ( lambda key: _contentsOf( os.getcwd(), _feedsterfile1 ), 
      '%s in the current directory' % _feedsterfile1),
    ( lambda key: _contentsOf( os.getcwd(), _feedsterfile2 ),
      '%s in the current directory' % _feedsterfile2),
    ( lambda key: _contentsOf( os.environ.get( 'HOME', '' ), _feedsterfile1 ),
      '%s in your home directory' % _feedsterfile1),
    ( lambda key: _contentsOf( os.environ.get( 'HOME', '' ), _feedsterfile2 ),
      '%s in your home directory' % _feedsterfile2 ),
    ( lambda key: _contentsOf( _getScriptDir(), _feedsterfile1 ),
      '%s in the feedster.py directory' % _feedsterfile1 ),
    ( lambda key: _contentsOf( _getScriptDir(), _feedsterfile2 ),
      '%s in the feedster.py directory' % _feedsterfile2 )
)

class NoLicenseKey(Exception): 
    """
    Thrown when the API is unable to find a valid license key.
    """
    pass

def setLicense(license_key):
    """
    Set the U{Feedster APIs <http://feedster.com/myfeedster.php?action=showkeys>}
     license key
    
    @param license_key: The new key to use
    @type  license_key: String
    @todo: validate the key?
    """
    global LICENSE_KEY
    LICENSE_KEY = license_key
    
def getLicense(license_key = None):
    """
    Get the U{Feedster APIs <http://www.feedster.com/api>} license key
    
    The key can be read from any number of locations.  See the module-leve
    documentation for the search order.
    
    @return: the license key
    @rtype:  String
    @raise NoLicenseKey: if no valid key could be found
    """
    for get, location in _licenseLocations:
        rc = get(license_key)
        if rc: return rc
    raise NoLicenseKey, \
      'get a license key at http://feedster.com/myfeedster.php?action=showkeys'

def _contentsOf(dirname, filename):
    filename = os.path.join(dirname, filename)
    if not os.path.exists(filename): return None
    fsock = open(filename)
    contents = fsock.read()
    fsock.close()
    return contents.strip('\n')
    
class feedsterHandler(xml.sax.ContentHandler):
    def startDocument(self):
        self.data={}
        self.text=""
        self.out=[]

    def startElement(self,tag,attributes):
        self.text=""
        if tag=="feed":
            self.data[u'xmlUrl']=dict(attributes)[u'url']
                    
    def endElement(self,tag):
        if tag=="feed":
            self.out.append(self.data)
            self.data={}
        else:
            self.data[tag]=self.text.strip()
            self.text=''
    
    def characters(self,content):
            self.text=self.text+content    

def getFeeds(url,type):
    if LICENSE_KEY == None: 
        print('No license key found.')
        sys.exit(-1)
    query="http://api.feedster.com/feeds.php?key=%s&type=%s&query=%s" % \
      (LICENSE_KEY,type,url)
    response=urllib.urlopen(query).read()

    p=xml.sax.make_parser()
    handler=feedsterHandler()
    p.setContentHandler(handler)
    p.feed(response)
    return handler.out

def search(terms):
    """get feedster search results via RSS ..."""
    query = 'http://feedster.com/search.php?q=%s&sort=date&ie=UTF-8&hl=&content=full&type=rss&limit=15' % terms
    response = urllib.urlopen(query).read()

    p = xml.sax.make_parser()
    handler = rsssax.rssHandler()
    p.setContentHandler(handler)
    p.feed(response)
    return handler.out
