/*
 * Decompiled with CFR 0.152.
 */
package com.dracoon.sdk.crypto;

import com.dracoon.sdk.crypto.CryptoSystemException;
import com.dracoon.sdk.crypto.CryptoUtils;
import com.dracoon.sdk.crypto.FileDecryptionCipher;
import com.dracoon.sdk.crypto.FileEncryptionCipher;
import com.dracoon.sdk.crypto.InvalidFileKeyException;
import com.dracoon.sdk.crypto.InvalidKeyPairException;
import com.dracoon.sdk.crypto.InvalidPasswordException;
import com.dracoon.sdk.crypto.model.EncryptedFileKey;
import com.dracoon.sdk.crypto.model.PlainFileKey;
import com.dracoon.sdk.crypto.model.UserKeyPair;
import com.dracoon.sdk.crypto.model.UserPrivateKey;
import com.dracoon.sdk.crypto.model.UserPublicKey;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.security.spec.MGF1ParameterSpec;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.OAEPParameterSpec;
import javax.crypto.spec.PSource;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMException;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.PKCS8Generator;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.openssl.jcajce.JcaPKCS8Generator;
import org.bouncycastle.openssl.jcajce.JceOpenSSLPKCS8DecryptorProviderBuilder;
import org.bouncycastle.openssl.jcajce.JceOpenSSLPKCS8EncryptorBuilder;
import org.bouncycastle.operator.InputDecryptorProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.OutputEncryptor;
import org.bouncycastle.pkcs.PKCS8EncryptedPrivateKeyInfo;
import org.bouncycastle.pkcs.PKCSException;
import org.bouncycastle.util.io.pem.PemGenerationException;
import org.bouncycastle.util.io.pem.PemObjectGenerator;

public class Crypto {
    private static final int HASH_ITERATION_COUNT = 10000;
    private static final int FILE_KEY_SIZE = 32;
    private static final int IV_SIZE = 12;

    private Crypto() {
    }

    public static UserKeyPair generateUserKeyPair(String password) throws InvalidKeyPairException, InvalidPasswordException, CryptoSystemException {
        return Crypto.generateUserKeyPair("A", password);
    }

    public static UserKeyPair generateUserKeyPair(String version, String password) throws InvalidKeyPairException, InvalidPasswordException, CryptoSystemException {
        KeyPair keyPair;
        Crypto.validateUserKeyPairVersion(version);
        Crypto.validatePassword(password);
        try {
            KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA");
            keyGen.initialize(2048);
            keyPair = keyGen.generateKeyPair();
        }
        catch (NoSuchAlgorithmException e) {
            throw new CryptoSystemException("Could not generate RSA key pair. Algorithm is missing.", e);
        }
        String privateKeyString = Crypto.encryptPrivateKey(keyPair.getPrivate(), password);
        String publicKeyString = Crypto.getStringFromPublicKey(keyPair.getPublic());
        UserPrivateKey userPrivateKey = new UserPrivateKey();
        userPrivateKey.setVersion(version);
        userPrivateKey.setPrivateKey(privateKeyString);
        UserPublicKey userPublicKey = new UserPublicKey();
        userPublicKey.setVersion(version);
        userPublicKey.setPublicKey(publicKeyString);
        UserKeyPair userKeyPair = new UserKeyPair();
        userKeyPair.setUserPrivateKey(userPrivateKey);
        userKeyPair.setUserPublicKey(userPublicKey);
        return userKeyPair;
    }

    private static String encryptPrivateKey(PrivateKey privateKey, String password) throws InvalidPasswordException, CryptoSystemException {
        JcaPKCS8Generator generator;
        OutputEncryptor encryptor;
        try {
            encryptor = new JceOpenSSLPKCS8EncryptorBuilder(PKCS8Generator.AES_256_CBC).setProvider("BC").setIterationCount(10000).setPasssword(password.toCharArray()).build();
        }
        catch (OperatorCreationException e) {
            throw new CryptoSystemException("Could not encrypt private key. Creation of PKCS8(AES 256 CBC) encryptor failed.", e);
        }
        try {
            generator = new JcaPKCS8Generator(privateKey, encryptor);
        }
        catch (PemGenerationException e) {
            throw new InvalidPasswordException("Could not encrypt private key. Invalid private key password.", e);
        }
        try {
            StringWriter stringWriter = new StringWriter();
            JcaPEMWriter pemWriter = new JcaPEMWriter((Writer)stringWriter);
            pemWriter.writeObject((PemObjectGenerator)generator);
            pemWriter.close();
            return stringWriter.toString();
        }
        catch (IOException e) {
            throw new CryptoSystemException("Could not encrypt private key. PEM encoding failed.", e);
        }
    }

    private static PrivateKey decryptPrivateKey(String privateKey, String password) throws InvalidKeyPairException, InvalidPasswordException, CryptoSystemException {
        PrivateKeyInfo pkInfo;
        Object obj;
        try {
            ByteArrayInputStream in = new ByteArrayInputStream(privateKey.getBytes());
            PEMParser pemReader = new PEMParser((Reader)new InputStreamReader(in));
            obj = pemReader.readObject();
            pemReader.close();
            in.close();
        }
        catch (IOException e) {
            throw new InvalidKeyPairException("Could not decrypt private key. PEM decoding failed.", e);
        }
        try {
            if (!(obj instanceof PKCS8EncryptedPrivateKeyInfo)) {
                throw new InvalidKeyPairException("Could not decrypt private key. Provided key is not a PKCS8 encrypted private key.");
            }
            PKCS8EncryptedPrivateKeyInfo epkInfo = (PKCS8EncryptedPrivateKeyInfo)obj;
            InputDecryptorProvider decryptor = new JceOpenSSLPKCS8DecryptorProviderBuilder().setProvider("BC").build(password.toCharArray());
            pkInfo = epkInfo.decryptPrivateKeyInfo(decryptor);
        }
        catch (OperatorCreationException e) {
            throw new CryptoSystemException("Could not decrypt private key. Creation of PKCS8 decryptor failed.", e);
        }
        catch (PKCSException e) {
            throw new InvalidPasswordException("Could not decrypt private key. Invalid private key password.", e);
        }
        try {
            JcaPEMKeyConverter converter = new JcaPEMKeyConverter().setProvider("BC");
            return converter.getPrivateKey(pkInfo);
        }
        catch (PEMException e) {
            throw new CryptoSystemException("Could not decrypted private key. PEM decoding failed.", e);
        }
    }

    private static String getStringFromPublicKey(PublicKey pubKey) throws InvalidKeyPairException {
        try {
            StringWriter writer = new StringWriter();
            JcaPEMWriter pemWriter = new JcaPEMWriter((Writer)writer);
            pemWriter.writeObject((Object)pubKey);
            pemWriter.close();
            return writer.toString();
        }
        catch (IOException e) {
            throw new InvalidKeyPairException("Could not encode public key. PEM encoding failed.", e);
        }
    }

    private static PublicKey getPublicKeyFromString(String pubKey) throws InvalidKeyPairException, CryptoSystemException {
        Object obj;
        try {
            ByteArrayInputStream in = new ByteArrayInputStream(pubKey.getBytes());
            PEMParser pemReader = new PEMParser((Reader)new InputStreamReader(in));
            obj = pemReader.readObject();
            pemReader.close();
            in.close();
        }
        catch (IOException e) {
            throw new InvalidKeyPairException("Could not decode public key. PEM decoding failed.", e);
        }
        if (!(obj instanceof SubjectPublicKeyInfo)) {
            throw new InvalidKeyPairException("Could not decode public key. Provided key is not PKCS8 public key.");
        }
        SubjectPublicKeyInfo pkInfo = (SubjectPublicKeyInfo)obj;
        try {
            JcaPEMKeyConverter converter = new JcaPEMKeyConverter().setProvider("BC");
            return converter.getPublicKey(pkInfo);
        }
        catch (PEMException e) {
            throw new CryptoSystemException("Could not decode public key. PEM decoding failed.", e);
        }
    }

    public static boolean checkUserKeyPair(UserKeyPair userKeyPair, String password) throws InvalidKeyPairException, CryptoSystemException {
        Crypto.validateUserKeyPair(userKeyPair);
        Crypto.validateUserPrivateKey(userKeyPair.getUserPrivateKey());
        if (password == null || password.isEmpty()) {
            return false;
        }
        try {
            Crypto.decryptPrivateKey(userKeyPair.getUserPrivateKey().getPrivateKey(), password);
            return true;
        }
        catch (InvalidPasswordException e) {
            return false;
        }
        catch (CryptoSystemException | InvalidKeyPairException e) {
            throw e;
        }
    }

    public static EncryptedFileKey encryptFileKey(PlainFileKey plainFileKey, UserPublicKey userPublicKey) throws InvalidFileKeyException, InvalidKeyPairException, CryptoSystemException {
        byte[] eFileKey;
        Cipher cipher;
        Crypto.validatePlainFileKey(plainFileKey);
        Crypto.validateUserPublicKey(userPublicKey);
        PublicKey publicKey = Crypto.getPublicKeyFromString(userPublicKey.getPublicKey());
        try {
            cipher = Cipher.getInstance("RSA/ECB/OAEPWITHSHA-256ANDMGF1PADDING");
            OAEPParameterSpec spec = new OAEPParameterSpec("SHA-256", "MGF1", MGF1ParameterSpec.SHA1, PSource.PSpecified.DEFAULT);
            cipher.init(1, (Key)publicKey, spec);
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchPaddingException e) {
            throw new CryptoSystemException("Could not encrypt file key. Creation of cipher failed.", e);
        }
        catch (InvalidKeyException e) {
            throw new InvalidKeyPairException("Could not encrypt file key. Invalid public key.", e);
        }
        byte[] pFileKey = CryptoUtils.stringToByteArray(plainFileKey.getKey());
        try {
            eFileKey = cipher.doFinal(pFileKey);
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new CryptoSystemException("Could not encrypt file key. Encryption failed.", e);
        }
        EncryptedFileKey encFileKey = new EncryptedFileKey();
        encFileKey.setKey(CryptoUtils.byteArrayToString(eFileKey));
        encFileKey.setIv(plainFileKey.getIv());
        encFileKey.setTag(plainFileKey.getTag());
        encFileKey.setVersion(plainFileKey.getVersion());
        return encFileKey;
    }

    public static PlainFileKey decryptFileKey(EncryptedFileKey encFileKey, UserPrivateKey userPrivateKey, String password) throws InvalidFileKeyException, InvalidKeyPairException, InvalidPasswordException, CryptoSystemException {
        byte[] dFileKey;
        Cipher cipher;
        Crypto.validateEncryptedFileKey(encFileKey);
        Crypto.validateUserPrivateKey(userPrivateKey);
        Crypto.validatePassword(password);
        PrivateKey privateKey = Crypto.decryptPrivateKey(userPrivateKey.getPrivateKey(), password);
        try {
            cipher = Cipher.getInstance("RSA/ECB/OAEPWITHSHA-256ANDMGF1PADDING");
            OAEPParameterSpec spec = new OAEPParameterSpec("SHA-256", "MGF1", MGF1ParameterSpec.SHA1, PSource.PSpecified.DEFAULT);
            cipher.init(2, (Key)privateKey, spec);
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchPaddingException e) {
            throw new CryptoSystemException("Could not decrypt file key. Creation of cipher failed.", e);
        }
        catch (InvalidKeyException e) {
            throw new InvalidKeyPairException("Could not decrypt file key. Invalid private key.", e);
        }
        byte[] eFileKey = CryptoUtils.stringToByteArray(encFileKey.getKey());
        try {
            dFileKey = cipher.doFinal(eFileKey);
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new InvalidFileKeyException("Could not decrypt file key. Encryption failed.", e);
        }
        PlainFileKey plainFileKey = new PlainFileKey();
        plainFileKey.setKey(CryptoUtils.byteArrayToString(dFileKey));
        plainFileKey.setIv(encFileKey.getIv());
        plainFileKey.setTag(encFileKey.getTag());
        plainFileKey.setVersion(encFileKey.getVersion());
        return plainFileKey;
    }

    public static PlainFileKey generateFileKey() {
        try {
            return Crypto.generateFileKey("A");
        }
        catch (InvalidFileKeyException e) {
            return null;
        }
    }

    public static PlainFileKey generateFileKey(String version) throws InvalidFileKeyException {
        Crypto.validateFileKeyVersion(version);
        byte[] key = Crypto.generateSecureRandomByteArray(32);
        byte[] iv = Crypto.generateSecureRandomByteArray(12);
        PlainFileKey fileKey = new PlainFileKey();
        fileKey.setKey(CryptoUtils.byteArrayToString(key));
        fileKey.setIv(CryptoUtils.byteArrayToString(iv));
        fileKey.setTag(null);
        fileKey.setVersion(version);
        return fileKey;
    }

    private static byte[] generateSecureRandomByteArray(int size) {
        SecureRandom sr = new SecureRandom();
        byte[] bytes = new byte[size];
        sr.nextBytes(bytes);
        return bytes;
    }

    public static FileEncryptionCipher createFileEncryptionCipher(PlainFileKey fileKey) throws InvalidFileKeyException, CryptoSystemException {
        Crypto.validatePlainFileKey(fileKey);
        return new FileEncryptionCipher(fileKey);
    }

    public static FileDecryptionCipher createFileDecryptionCipher(PlainFileKey fileKey) throws InvalidFileKeyException, CryptoSystemException {
        Crypto.validatePlainFileKey(fileKey);
        return new FileDecryptionCipher(fileKey);
    }

    private static void validatePassword(String password) throws InvalidPasswordException {
        if (password == null || password.isEmpty()) {
            throw new InvalidPasswordException("Password cannot be null or empty.");
        }
    }

    private static void validateUserKeyPair(UserKeyPair userKeyPair) throws InvalidKeyPairException {
        if (userKeyPair == null) {
            throw new InvalidKeyPairException("User key pair cannot be null.");
        }
    }

    private static void validateUserKeyPairVersion(String version) throws InvalidKeyPairException {
        if (version == null || version.isEmpty() || !version.equals("A")) {
            throw new InvalidKeyPairException("Unknown user key pair version.");
        }
    }

    private static void validateUserPrivateKey(UserPrivateKey privateKey) throws InvalidKeyPairException {
        if (privateKey == null) {
            throw new InvalidKeyPairException("Private key container cannot be null.");
        }
        String version = privateKey.getVersion();
        if (version == null || !version.equals("A")) {
            throw new InvalidKeyPairException("Unknown private key version.");
        }
        String pk = privateKey.getPrivateKey();
        if (pk == null || pk.isEmpty()) {
            throw new InvalidKeyPairException("Private key cannot be null or empty.");
        }
    }

    private static void validateUserPublicKey(UserPublicKey publicKey) throws InvalidKeyPairException {
        if (publicKey == null) {
            throw new InvalidKeyPairException("Public key container cannot be null.");
        }
        String version = publicKey.getVersion();
        if (version == null || !version.equals("A")) {
            throw new InvalidKeyPairException("Unknown public key version.");
        }
        String pk = publicKey.getPublicKey();
        if (pk == null || pk.isEmpty()) {
            throw new InvalidKeyPairException("Public key cannot be null or empty.");
        }
    }

    private static void validateFileKeyVersion(String version) throws InvalidFileKeyException {
        if (version == null || version.isEmpty()) {
            throw new InvalidFileKeyException("Unknown file key version.");
        }
    }

    private static void validatePlainFileKey(PlainFileKey fileKey) throws InvalidFileKeyException {
        if (fileKey == null) {
            throw new InvalidFileKeyException("File key cannot be null.");
        }
        String version = fileKey.getVersion();
        if (version == null || !version.equals("A")) {
            throw new InvalidFileKeyException("Unknown file key version.");
        }
    }

    private static void validateEncryptedFileKey(EncryptedFileKey fileKey) throws InvalidFileKeyException {
        if (fileKey == null) {
            throw new InvalidFileKeyException("File key cannot be null.");
        }
        String version = fileKey.getVersion();
        if (version == null || !version.equals("A")) {
            throw new InvalidFileKeyException("Unknown file key version.");
        }
    }

    static {
        Security.insertProviderAt((Provider)new BouncyCastleProvider(), 1);
    }
}

