## Copyright (C) 2007 Denis Sbragion
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
##(at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##
## This function is part of DRC - Digital room correction

##
## usage: rt = rt60(fs,ir,ic,rtf,rts,rta,fl,at)
##
## Compute the rt60 for the given frequency ranges
## Returns an array with length(rtf) - 1 values
##
## fs: sample rate
## ir: impulse response
## ic: impulse center, -1 for automatic identification
## rtf: array with the rt60 frequency bands
## rts: initial sample after the impulse center where the rt60 computation starts
## rta: attenuation, in dB, where the rt60 computation stops
## fl: length of the filter used for the band filtering
## at: stopband attenuation for the band filters
##
## Example:
##
##  rt = rt60(44100,ir,-1,logspace(log10(20000 / 1024),log10(20000),30),4410,-15,16384,90)
##

function rt = rt60(fs,ir,ic,rtf,rts,rta,fl,at);
	# Find the impulse center if needed
	if (ic < 0)
		[m,ic] = max(abs(hilbert(ir)));
	endif
	
	# Computes the filter window
	fw = chebwin(fl,at)';
	
	# Computes the filtered impulse center
	fic = round(ic + (fl - 1) / 2);
	
	# Cycles on the frequency ranges	
	for cb = 1:(length(rtf) - 1)
		# Computes the current filter
		bf = bandpassfir(fl,2 * rtf(cb) / fs,2 * rtf(cb + 1) / fs) .* fw;
		
		# Computes the band ir
		bir = fftconv(bf,ir);
		
		# Computes the Schroeder backward integration
		iidc(length(bir):-1:1) = cumsum(bir(length(bir):-1:1).^2);
		
		# Convert to log scale
		iidc = 10 * log10(iidc);
		
		# Find the attenuation at the start point
		sat = iidc(fic + rts);
		
		# Find the index of the stop attenuation
		sai = min(find(min(0,iidc((fic + rts):length(iidc)) - (sat + rta))));
		
		# Check if the stop attenuation has been found
		if (sai > 0)		
			# Computes the linear slope
			irs = polyfit((0:sai) / fs,iidc((fic + rts):(fic + rts + sai)),1);
			
			# Assign the rt60
			rt(cb) = -60 / irs(1);
		else
			# Assign the rt60
			rt(cb) = 0;
		endif
	endfor	
endfunction