## Copyright (C) 2005 Denis Sbragion
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
##(at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##
## This function is part of DRC - Digital room correction

## usage: createdrcplots(ir1,ic1,it1,ir2,ic2,it2,outdir,fprefix)
##
## createdrcplots - Create a full set of eps plots for DRC analysis
##
## ir1 = impulse response 1
## ic1 = center sample of impulse response 1, -1 for automatic search
## it1 = title for impulse response 1 plots
## ir2 = impulse response 2
## ic2 = center sample of impulse response 2, -1 for automatic search
## it2 = title for impulse response 2 plots
## outdir = output directory for eps files
## fprefix = prefix for eps files
##
## Sample rate of ir1 and ir2 *must* be 44.1 KHz

function createdrcplots(ir1,ic1,it1,ir2,ic2,it2,outdir,fprefix);

	# Find the impulse center if needed
	if (ic1 < 0)
		[m1,ic1] = max(abs(hilbert(ir1)));	
	else
		m1 = max(abs(hilbert(ir1)));	
	endif
	if (ic2 < 0)
		[m2,ic2] = max(abs(hilbert(ir2)));
	else
		m2 = max(abs(hilbert(ir2)));	
	endif
	
	# Normalize the impulse responses
	ir1 = ir1(:)/m1;
	ir2 = ir2(:)/m2;	
	
	# Report the impulse centers
	disp(sprintf("IR1 impulse center: %d, max: %f",ic1, m1));
	disp(sprintf("IR2 impulse center: %d, max: %f",ic2, m2));
	fflush(stdout);
	
	# Clear the unused variables
	clear m1 m2;
	
	# Save the sample rate
	fs = 44100;
	
	# Enable contour display	when set to > 0
	surcntenable = 0;
	mapcntenable = 0;
	
	# Enable pm3d surface plot when set to > 0
	pm3denable = 1;

	# Resampling factor
	rf = 40;
	
	# Resampling filter length, cutoff frequency and attenuation
	rfl = 16383;
	rcf = 0.95;
	rat = 160;
	
	# About -10 to +30 ms
	irp = 441;
	ira = 882+441;
	
	# Creates the oversampled impulse responses
	ir1os = rf * iupsample(ir1([ic1-irp:ic1+ira]),rf,rfl,rcf,rat);
	ir2os = rf * iupsample(ir2([ic2-irp:ic2+ira]),rf,rfl,rcf,rat);
	
	# Unset the automatic replot
	if (exist("automatic_replot") == 101)
		automatic_replot = 0;
	else
		automatic_replot(0);
	endif
	
	# Reset gnuplot status
	__gnuplot_raw__("reset;\n");

	# Set terminal to postscript eps	
	__gnuplot_raw__("set terminal postscript eps color solid linewidth 1 rounded;\n");

	# Report the stage
	disp("Step response...");
	fflush(stdout);
	
	# Creates the step response
	ir1step = fftconv(ir1os,ones(1,length(ir1os)))/rf;
	ir2step = fftconv(ir2os,ones(1,length(ir2os)))/rf;
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-IRStepResponse.eps" "\"" ";\n" ]);
	
	# Plot the step responses
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-5:25] noreverse;\n");
	__gnuplot_raw__("set yrange [-1.5:1.5] noreverse;\n");
	plot(1000 * ((([0:length(ir1step)-1])/(rf*fs))-((irp/fs) + ((rfl + 1)/(2*rf*fs)))),ir1step,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-5:25] noreverse;\n");
	__gnuplot_raw__("set yrange [-1.5:1.5] noreverse;\n");
	plot(1000 * ((([0:length(ir2step)-1])/(rf*fs))-((irp/fs) + ((rfl + 1)/(2*rf*fs)))),ir2step,["3;" it2 ";"]);
	
	# Reset to single plot
	oneplot();
	
	# Reset the output file		
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Time response, full range...");
	fflush(stdout);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-IRFullRange.eps" "\"" ";\n" ]);
	
	# Plot the oversampled impulse responses
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-2:5] noreverse;\n");
	__gnuplot_raw__("set yrange [-1.2:1.2] noreverse;\n");
	plot(1000 * ((([0:length(ir1os)-1])/(rf*fs))-((irp/fs) + ((rfl + 1)/(2*rf*fs)))),ir1os,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-2:5] noreverse;\n");
	__gnuplot_raw__("set yrange [-1.2:1.2] noreverse;\n");
	plot(1000 * ((([0:length(ir2os)-1])/(rf*fs))-((irp/fs) + ((rfl + 1)/(2*rf*fs)))),ir2os,["3;" it2 ";"]);
	
	# Reset to single plot
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Remove variables no longer used
	clear ir1step ir2step;
	
	# Computes the analytical signal
	ir1osh = abs(hilbert(ir1os));
	ir2osh = abs(hilbert(ir2os));
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-IRFullRangeEnvelope.eps" "\"" ";\n" ]);
		
	# Plot the impulse envelope
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-2:5] noreverse;\n");
	__gnuplot_raw__("set yrange [-1.2:1.2] noreverse;\n");
	plot(1000*((([0:length(ir1osh)-1])/(rf*fs))-((irp/fs) + ((rfl + 1)/(2*rf*fs)))),ir1osh,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-2:5] noreverse;\n");
	__gnuplot_raw__("set yrange [-1.2:1.2] noreverse;\n");
	plot(1000*((([0:length(ir2osh)-1])/(rf*fs))-((irp/fs) + ((rfl + 1)/(2*rf*fs)))),ir2osh,["3;" it2 ";"]);
	
	# Reset to single plot
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-IRFullRangeETC.eps" "\"" ";\n" ]);
	
	# Plot the energy-time curve
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [-2:5] noreverse;\n");
	__gnuplot_raw__("set yrange [-30:5] noreverse;\n");
	plot(1000*((([0:length(ir1osh)-1])/(rf*fs))-((irp/fs) + ((rfl + 1)/(2*rf*fs)))),20 * log10(ir1osh),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [-2:5] noreverse;\n");
	__gnuplot_raw__("set yrange [-30:5] noreverse;\n");
	plot(1000*((([0:length(ir2osh)-1])/(rf*fs))-((irp/fs) + ((rfl + 1)/(2*rf*fs)))),20 * log10(ir2osh),["3;" it2 ";"]);
	
	# Reset to single plot
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Remove variables no longer used
	clear ir1os ir2os ir1osh ir2osh rfl rcf rat;
	
	# Report the stage
	disp("Time response, 2 KHz brickwall...");
	fflush(stdout);
	
	# Create the 2 KHz brickwall filter
	fltlen = 4096;
	fltatt = 160;
	flt = lowpassfir(fltlen,2000/(fs/2)) .* chebwin(fltlen,fltatt)';
	fltdelay = (1 + length(flt)) / 2;
	
	# Creates the 2 KHz brickwall filtered impulse responses
	# about -20 to +50 ms
	ir1bw = fftconv(flt,ir1([ic1-882:ic1+2205]));
	ir2bw = fftconv(flt,ir2([ic2-882:ic2+2205]));
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-IR2KHzBrickwall.eps" "\"" ";\n" ]);
	
	# Plot the brickwall filtered impulse responses
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-20:50] noreverse;\n");
	__gnuplot_raw__("set yrange [-0.15:0.15] noreverse;\n");
	plot(1000*((([0:length(ir1bw)-1])/fs)-((fltdelay + 882)/fs)),ir1bw,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-20:50] noreverse;\n");
	__gnuplot_raw__("set yrange [-0.15:0.15] noreverse;\n");
	plot(1000*((([0:length(ir2bw)-1])/fs)-((fltdelay + 882)/fs)),ir2bw,["3;" it2 ";"]);
	
	# Reset to single plot
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Computes the analytical signal
	ir1bwh = abs(hilbert(ir1bw));
	ir2bwh = abs(hilbert(ir2bw));
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-IR2KHzBrickwallEnvelope.eps" "\"" ";\n" ]);
	
	# Plot the impulse envelope
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-20:50] noreverse;\n");
	__gnuplot_raw__("set yrange [-0.15:0.15] noreverse;\n");
	plot(1000*((([0:length(ir1bwh)-1])/fs)-((fltdelay + 882)/fs)),ir1bwh,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-20:50] noreverse;\n");
	__gnuplot_raw__("set yrange [-0.15:0.15] noreverse;\n");
	plot(1000*((([0:length(ir2bwh)-1])/fs)-((fltdelay + 882)/fs)),ir2bwh,["3;" it2 ";"]);
	
	# Reset to single plot
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-IR2KHzBrickwallETC.eps" "\"" ";\n" ]);
	
	# Plot the energy-time curve
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [-20:50] noreverse;\n");
	__gnuplot_raw__("set yrange [-50:-15] noreverse;\n");
	plot(1000*((([0:length(ir1bwh)-1])/fs)-((fltdelay + 882)/fs)),20 * log10(ir1bwh),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [-20:50] noreverse;\n");
	__gnuplot_raw__("set yrange [-50:-15] noreverse;\n");
	plot(1000*((([0:length(ir2bwh)-1])/fs)-((fltdelay + 882)/fs)),20 * log10(ir2bwh),["3;" it2 ";"]);
	
	# Reset to single plot
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Remove variables no longer used
	clear ir1bw ir2bw ir1bwh ir2bwh;
	
	# Report the stage
	disp("Time response, 200 Hz brickwall...");
	fflush(stdout);
	
	# Create the 200 Hz brickwall filter
	fltlen = 32768;
	fltatt = 160;
	flt = lowpassfir(fltlen,200/(fs/2)) .* chebwin(fltlen,fltatt)';
	fltdelay = (1 + length(flt)) / 2;
	
	# Creates the 200 Hz brickwall filtered impulse responses
	# about -200 to +500 ms
	ir1bw = fftconv(flt,ir1([ic1-8820:ic1+22050]));
	ir2bw = fftconv(flt,ir2([ic2-8820:ic2+22050]));
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-IR200HzBrickwall.eps" "\"" ";\n" ]);
	
	# Plot the brickwall filtered impulse responses
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (s)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-0.2:0.5] noreverse;\n");
	__gnuplot_raw__("set yrange [-0.015:0.015] noreverse;\n");
	plot(((([0:length(ir1bw)-1])/fs)-((fltdelay + 8820)/fs)),ir1bw,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (s)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-0.2:0.5] noreverse;\n");
	__gnuplot_raw__("set yrange [-0.015:0.015] noreverse;\n");
	plot(((([0:length(ir2bw)-1])/fs)-((fltdelay + 8820)/fs)),ir2bw,["3;" it2 ";"]);
	
	# Reset to single plot
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Computes the analytical signal
	ir1bwh = abs(hilbert(ir1bw));
	ir2bwh = abs(hilbert(ir2bw));
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-IR200HzBrickwallEnvelope.eps" "\"" ";\n" ]);
	
	# Plot the impulse envelope
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (s)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-0.2:0.5] noreverse;\n");
	__gnuplot_raw__("set yrange [-0.015:0.015] noreverse;\n");
	plot(((([0:length(ir1bwh)-1])/fs)-((fltdelay + 8820)/fs)),ir1bwh,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (s)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (normalized)\" 0;\n");
	__gnuplot_raw__("set xrange [-0.2:0.5] noreverse;\n");
	__gnuplot_raw__("set yrange [-0.015:0.015] noreverse;\n");
	plot(((([0:length(ir2bwh)-1])/fs)-((fltdelay + 8820)/fs)),ir2bwh,["3;" it2 ";"]);
	
	# Reset to single plot
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-IR200HzBrickwallETC.eps" "\"" ";\n" ]);
	
	# Plot the energy-time curve
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (s)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [-0.2:0.5] noreverse;\n");
	__gnuplot_raw__("set yrange [-70:-35] noreverse;\n");
	plot(((([0:length(ir1bwh)-1])/fs)-((fltdelay + 8820)/fs)),20 * log10(ir1bwh),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (s)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Amplitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [-0.2:0.5] noreverse;\n");
	__gnuplot_raw__("set yrange [-70:-35] noreverse;\n");
	plot(((([0:length(ir2bwh)-1])/fs)-((fltdelay + 8820)/fs)),20 * log10(ir2bwh),["3;" it2 ";"]);
	
	# Reset to single plot
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Remove variables no longer used
	clear ir1bw ir2bw ir1bwh ir2bwh;
	
	# Report the stage
	disp("Unsmoothed frequency response, 1 ms window...");
	fflush(stdout);
	
	# Computes the window, 1 ms
	wind = [ones(1,441) ones(1,22) (blackman(44)(23:44))' zeros(1,8192 - (441 + 44))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(8192 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(8192 - (441 + 1))]) .* wind;
	
	# Rotate the impulse response and transform to the frequency domain
	ir1w = fft([ir1w(442:length(ir1w))' ir1w(1:441)']);
	ir2w = fft([ir2w(442:length(ir2w))' ir2w(1:441)']);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRUnsmoothed1ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [1:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * fs * [0:(length(ir1w)/2)-1] / length(ir1w),20 * log10(abs(ir1w([1:length(ir1w)/2]))),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [1:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * fs * [0:(length(ir2w)/2)-1] / length(ir2w),20 * log10(abs(ir2w([1:length(ir2w)/2]))),["3;" it2 ";"]);
	
	# Reset to single plot
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-PRUnsmoothed1ms.eps" "\"" ";\n" ]);
	
	# Plot the phase response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [1:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(0.001 * fs * [0:(length(ir1w)/2)-1] / length(ir1w),180 * angle(ir1w([1:length(ir1w)/2]))/pi,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [1:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(0.001 * fs * [0:(length(ir2w)/2)-1] / length(ir2w),180 * angle(ir2w([1:length(ir2w)/2]))/pi,["3;" it2 ";"]);

	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Unsmoothed frequency response, 5 ms window...");
	fflush(stdout);
	
	# Computes the window, 5 ms
	wind = [ones(1,441) ones(1,110) (blackman(220)(111:220))' zeros(1,8192 - (441 + 220))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(8192 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(8192 - (441 + 1))]) .* wind;
	
	# Rotate the impulse response and transform to the frequency domain
	ir1w = fft([ir1w(442:length(ir1w))' ir1w(1:441)']);
	ir2w = fft([ir2w(442:length(ir2w))' ir2w(1:441)']);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRUnsmoothed5ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.2:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * fs * [0:(length(ir1w)/2)-1] / length(ir1w),20 * log10(abs(ir1w([1:length(ir1w)/2]))),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.2:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * fs * [0:(length(ir2w)/2)-1] / length(ir2w),20 * log10(abs(ir2w([1:length(ir2w)/2]))),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-PRUnsmoothed5ms.eps" "\"" ";\n" ]);
	
	# Plot the phase response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [0.2:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(0.001 * fs * [0:(length(ir1w)/2)-1] / length(ir1w),180 * angle(ir1w([1:length(ir1w)/2]))/pi,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [0.2:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(0.001 * fs * [0:(length(ir2w)/2)-1] / length(ir2w),180 * angle(ir2w([1:length(ir2w)/2]))/pi,["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Unsmoothed frequency response, 200 ms window...");
	fflush(stdout);
	
	# Computes the window, 200 ms
	wind = [ones(1,8820) ones(1,4410) (blackman(8820)(4411:8820))' zeros(1,32768 - (8820 + 8820))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-8820:ic1+(32768 - (8820 + 1))]) .* wind;
	ir2w = ir2([ic2-8820:ic2+(32768 - (8820 + 1))]) .* wind;
	
	# Rotate and pad the impulse response and transform to the frequency domain
	ir1w = fft([ir1w(8821:length(ir1w))' zeros(1,131072 - length(ir1w)) ir1w(1:8820)']);
	ir2w = fft([ir2w(8821:length(ir2w))' zeros(1,131072 - length(ir2w)) ir2w(1:8820)']);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRUnsmoothed200ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [20:200] noreverse;\n");
	__gnuplot_raw__("set yrange [-30:30] noreverse;\n");
	__gnuplot_raw__("set xtics (20,30,50,70,100,160,200);\n");
	__gnuplot_raw__("set ytics -30,6,30;\n");
	semilogx(fs * [0:(length(ir1w)/2)-1] / length(ir1w),20 * log10(abs(ir1w([1:length(ir1w)/2]))),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [20:200] noreverse;\n");
	__gnuplot_raw__("set yrange [-30:30] noreverse;\n");
	__gnuplot_raw__("set xtics (20,30,50,70,100,160,200);\n");
	__gnuplot_raw__("set ytics -30,6,30;\n");
	semilogx(fs * [0:(length(ir2w)/2)-1] / length(ir2w),20 * log10(abs(ir2w([1:length(ir2w)/2]))),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-PRUnsmoothed200ms.eps" "\"" ";\n" ]);
	
	# Plot the phase response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [20:200] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (20,30,50,70,100,160,200);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(fs * [0:(length(ir1w)/2)-1] / length(ir1w),180 * angle(ir1w([1:length(ir1w)/2]))/pi,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [20:200] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (20,30,50,70,100,160,200);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(fs * [0:(length(ir2w)/2)-1] / length(ir2w),180 * angle(ir2w([1:length(ir2w)/2]))/pi,["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Remove variables no longer used
	clear wind ir1w ir2w;
	
	# Report the stage
	disp("Frequency dependent windowing, magnitude response, 1/3 oct settings...");
	fflush(stdout);
	
	# Estract the windowed impulse responses
	ir1w = ir1(ic1-32767:ic1+32768);
	ir2w = ir2(ic2-32767:ic2+32768);
	
	# Extract the minimum phase impulse response
	[ir1cps, ir1w] = rceps(ir1w);
	[ir2cps, ir2w] = rceps(ir2w);
	clear ir1cps ir2cps;
	
	# Performs the frequency dependant windowing
	ir1w = irslprefilt(ir1w,1,2 * 20/fs,4761,2 * 22000/fs,5,1.0,4096,1.0,5,0);
	ir2w = irslprefilt(ir2w,1,2 * 20/fs,4761,2 * 22000/fs,5,1.0,4096,1.0,5,0);
	
	# Pad the windowed impulse response
	ir1w = [ zeros(1,65536 - length(ir1w)) ir1w(:)' ];
	ir2w = [ zeros(1,65536 - length(ir2w)) ir2w(:)' ];
	
	# Computes the magnitude response
	ir1m = fft(ir1w);
	ir1m = abs(ir1m(1:floor(length(ir1m)/2)));
	ir2m = fft(ir2w);
	ir2m = abs(ir2m(1:floor(length(ir2m)/2)));
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRFDWSmoothed-1-3.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * (0:(length(ir1m)-1)) / length(ir1m),20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * (0:(length(ir2m)-1)) / length(ir2m),20 * log10(ir2m),["3;" it2 ";"]);

	# Reset to single plot		
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, phase response, 1/3 oct settings...");
	fflush(stdout);
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-4096:ic1+32767]);
	ir2w = ir2([ic2-4096:ic2+32767]);
	
	# Performs the frequency dependant windowing
	ir1w = irslprefilt(ir1w,4097,2 * 20/fs,4761,2 * 22000/fs,5,1.0,4096,1.0,5,4096);
	ir2w = irslprefilt(ir2w,4097,2 * 20/fs,4761,2 * 22000/fs,5,1.0,4096,1.0,5,4096);
	
	# Rotate the impulse response and transform to the frequency domain
	ir1w = fft([ir1w(4097:length(ir1w)) ir1w(1:4096)]);
	ir2w = fft([ir2w(4097:length(ir2w)) ir2w(1:4096)]);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-PRFDWSmoothed-1-3.eps" "\"" ";\n" ]);
	
	# Plot the phase response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.03,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(0.001 * fs * [0:(length(ir1w)/2)-1] / length(ir1w),180 * angle(ir1w([1:length(ir1w)/2]))/pi,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.03,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(0.001 * fs * [0:(length(ir2w)/2)-1] / length(ir2w),180 * angle(ir2w([1:length(ir2w)/2]))/pi,["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, magnitude response, 1/6 oct settings...");
	fflush(stdout);
	
	# Estract the windowed impulse responses
	ir1w = ir1(ic1-32767:ic1+32768);
	ir2w = ir2(ic2-32767:ic2+32768);
	
	# Extract the minimum phase impulse response
	[ir1cps, ir1w] = rceps(ir1w);
	[ir2cps, ir2w] = rceps(ir2w);
	clear ir1cps ir2cps;
	
	# Performs the frequency dependant windowing
	ir1w = irslprefilt(ir1w,1,2 * 20/fs,9538,2 * 22000/fs,10,1.0,4096,1.0,10,0);
	ir2w = irslprefilt(ir2w,1,2 * 20/fs,9538,2 * 22000/fs,10,1.0,4096,1.0,10,0);
	
	# Pad the windowed impulse response
	ir1w = [ zeros(1,65536 - length(ir1w)) ir1w(:)' ];
	ir2w = [ zeros(1,65536 - length(ir2w)) ir2w(:)' ];
	
	# Computes the magnitude response
	ir1m = fft(ir1w);
	ir1m = abs(ir1m(1:floor(length(ir1m)/2)));
	ir2m = fft(ir2w);
	ir2m = abs(ir2m(1:floor(length(ir2m)/2)));
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRFDWSmoothed-1-6.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * (0:(length(ir1m)-1)) / length(ir1m),20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * (0:(length(ir2m)-1)) / length(ir2m),20 * log10(ir2m),["3;" it2 ";"]);

	# Reset to single plot		
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, phase response, 1/6 oct settings...");
	fflush(stdout);
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-4096:ic1+32767]);
	ir2w = ir2([ic2-4096:ic2+32767]);
	
	# Performs the frequency dependant windowing
	ir1w = irslprefilt(ir1w,4097,2 * 20/fs,9538,2 * 22000/fs,10,1.0,4096,1.0,10,4096);
	ir2w = irslprefilt(ir2w,4097,2 * 20/fs,9538,2 * 22000/fs,10,1.0,4096,1.0,10,4096);
	
	# Rotate the impulse response and transform to the frequency domain
	ir1w = fft([ir1w(4097:length(ir1w)) ir1w(1:4096)]);
	ir2w = fft([ir2w(4097:length(ir2w)) ir2w(1:4096)]);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-PRFDWSmoothed-1-6.eps" "\"" ";\n" ]);
	
	# Plot the phase response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.03,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(0.001 * fs * [0:(length(ir1w)/2)-1] / length(ir1w),180 * angle(ir1w([1:length(ir1w)/2]))/pi,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.03,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(0.001 * fs * [0:(length(ir2w)/2)-1] / length(ir2w),180 * angle(ir2w([1:length(ir2w)/2]))/pi,["3;" it2 ";"]);

	# Reset to single plot		
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, magnitude response, normal.drc settings...");
	fflush(stdout);
	
	# Estract the windowed impulse responses
	ir1w = ir1(ic1-32767:ic1+32768);
	ir2w = ir2(ic2-32767:ic2+32768);
	
	# Extract the minimum phase impulse response
	[ir1cps, ir1w] = rceps(ir1w);
	[ir2cps, ir2w] = rceps(ir2w);
	clear ir1cps ir2cps;
	
	# Performs the frequency dependant windowing
	ir1w = irslprefilt(ir1w,1,2 * 20/fs,22050,2 * 22000/fs,22,1.0,4096,0.25,22,0);
	ir2w = irslprefilt(ir2w,1,2 * 20/fs,22050,2 * 22000/fs,22,1.0,4096,0.25,22,0);
	
	# Pad the windowed impulse response
	ir1w = [ zeros(1,65536 - length(ir1w)) ir1w(:)' ];
	ir2w = [ zeros(1,65536 - length(ir2w)) ir2w(:)' ];
	
	# Computes the magnitude response
	ir1m = fft(ir1w);
	ir1m = abs(ir1m(1:floor(length(ir1m)/2)));
	ir2m = fft(ir2w);
	ir2m = abs(ir2m(1:floor(length(ir2m)/2)));
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRFDWSmoothed.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * (0:(length(ir1m)-1)) / length(ir1m),20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * (0:(length(ir2m)-1)) / length(ir2m),20 * log10(ir2m),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, phase response, normal.drc settings...");
	fflush(stdout);
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-4096:ic1+32767]);
	ir2w = ir2([ic2-4096:ic2+32767]);
	
	# Performs the frequency dependant windowing
	ir1w = irslprefilt(ir1w,4097,2 * 20/fs,22050,2 * 22000/fs,22,1.0,4096,0.25,22,4096);
	ir2w = irslprefilt(ir2w,4097,2 * 20/fs,22050,2 * 22000/fs,22,1.0,4096,0.25,22,4096);
	
	# Rotate the impulse response and transform to the frequency domain
	ir1w = fft([ir1w(4097:length(ir1w)) ir1w(1:4096)]);
	ir2w = fft([ir2w(4097:length(ir2w)) ir2w(1:4096)]);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-PRFDWSmoothed.eps" "\"" ";\n" ]);
	
	# Plot the phase response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.03,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(0.001 * fs * [0:(length(ir1w)/2)-1] / length(ir1w),180 * angle(ir1w([1:length(ir1w)/2]))/pi,["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Phase (deg)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-180:180] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.03,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -180,30,180;\n");
	semilogx(0.001 * fs * [0:(length(ir2w)/2)-1] / length(ir2w),180 * angle(ir2w([1:length(ir2w)/2]))/pi,["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Warp smoothing window
	wsw = 78;
	
	# Computes the Bark warping
	l = barkwarp(fs);
	
	# Report the stage
	disp("Bark smoothed magnitude response, 1 ms window...");
	fflush(stdout);
	
	# Prewindow the response
	wind = [ones(1,441) ones(1,22) (blackman(44)(23:44))' zeros(1,4096 - (441 + 44))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(4096 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(4096 - (441 + 1))]) .* wind;
	
	# Computes the smoothed magnitude response
	[ir1m, ir1f] = wsmooth(ir1w,l,wsw);
	[ir2m, ir2f] = wsmooth(ir2w,l,wsw);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRBarkSmoothed1ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [1:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir1f,20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [1:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir2f,20 * log10(ir2m),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Bark smoothed magnitude response, 5 ms window...");
	fflush(stdout);
	
	# Prewindow the response
	wind = [ones(1,441) ones(1,110) (blackman(220)(111:220))' zeros(1,4096 - (441 + 220))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(4096 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(4096 - (441 + 1))]) .* wind;
	
	# Computes the smoothed magnitude response
	[ir1m, ir1f] = wsmooth(ir1w,l,wsw);
	[ir2m, ir2f] = wsmooth(ir2w,l,wsw);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRBarkSmoothed5ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.2:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir1f,20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.2:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir2f,20 * log10(ir2m),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Bark smoothed magnitude response, 20 ms window...");
	fflush(stdout);
	
	# Prewindow the response
	wind = [ones(1,441) ones(1,441) (blackman(882)(442:882))' zeros(1,4096 - (441 + 882))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(4096 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(4096 - (441 + 1))]) .* wind;
	
	# Computes the smoothed magnitude response
	[ir1m, ir1f] = wsmooth(ir1w,l,wsw);
	[ir2m, ir2f] = wsmooth(ir2w,l,wsw);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRBarkSmoothed20ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.05:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir1f,20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.05:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir2f,20 * log10(ir2m),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Bark smoothed magnitude response, 50 ms window...");
	fflush(stdout);
	
	# Prewindow the response
	wind = [ones(1,441) ones(1,1102) (blackman(2204)(1103:2204))' zeros(1,4096 - (441 + 2204))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(4096 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(4096 - (441 + 1))]) .* wind;
	
	# Computes the smoothed magnitude response
	[ir1m, ir1f] = wsmooth(ir1w,l,wsw);
	[ir2m, ir2f] = wsmooth(ir2w,l,wsw);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRBarkSmoothed50ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir1f,20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir2f,20 * log10(ir2m),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Bark smoothed magnitude response, 200 ms window...");
	fflush(stdout);
	
	# Prewindow the response
	wind = [ones(1,441) ones(1,4410) (blackman(8820)(4411:8820))' zeros(1,10240 - (441 + 8820))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(10240 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(10240 - (441 + 1))]) .* wind;
	
	# Computes the smoothed magnitude response
	[ir1m, ir1f] = wsmooth(ir1w,l,wsw);
	[ir2m, ir2f] = wsmooth(ir2w,l,wsw);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRBarkSmoothed200ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir1f,20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir2f,20 * log10(ir2m),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Warp smoothing window
	wsw = 92;
	
	# Computes the ERB warping
	l = erbwarp(fs);
	
	# Report the stage
	disp("ERB smoothed magnitude response, 1 ms window...");
	fflush(stdout);
	
	# Prewindow the response
	wind = [ones(1,441) ones(1,22) (blackman(44)(23:44))' zeros(1,4096 - (441 + 44))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(4096 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(4096 - (441 + 1))]) .* wind;
	
	# Computes the smoothed magnitude response
	[ir1m, ir1f] = wsmooth(ir1w,l,wsw);
	[ir2m, ir2f] = wsmooth(ir2w,l,wsw);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRERBSmoothed1ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [1:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir1f,20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [1:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir2f,20 * log10(ir2m),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("ERB smoothed magnitude response, 5 ms window...");
	fflush(stdout);
	
	# Prewindow the response
	wind = [ones(1,441) ones(1,110) (blackman(220)(111:220))' zeros(1,4096 - (441 + 220))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(4096 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(4096 - (441 + 1))]) .* wind;
	
	# Computes the smoothed magnitude response
	[ir1m, ir1f] = wsmooth(ir1w,l,wsw);
	[ir2m, ir2f] = wsmooth(ir2w,l,wsw);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRERBSmoothed5ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.2:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir1f,20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.2:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir2f,20 * log10(ir2m),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("ERB smoothed magnitude response, 20 ms window...");
	fflush(stdout);
	
	# Prewindow the response
	wind = [ones(1,441) ones(1,441) (blackman(882)(442:882))' zeros(1,4096 - (441 + 882))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(4096 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(4096 - (441 + 1))]) .* wind;
	
	# Computes the smoothed magnitude response
	[ir1m, ir1f] = wsmooth(ir1w,l,wsw);
	[ir2m, ir2f] = wsmooth(ir2w,l,wsw);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRERBSmoothed20ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.05:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir1f,20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.05:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir2f,20 * log10(ir2m),["3;" it2 ";"]);

	# Reset to single plot		
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("ERB smoothed magnitude response, 50 ms window...");
	fflush(stdout);
	
	# Prewindow the response
	wind = [ones(1,441) ones(1,1102) (blackman(2204)(1103:2204))' zeros(1,4096 - (441 + 2204))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(4096 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(4096 - (441 + 1))]) .* wind;
	
	# Computes the smoothed magnitude response
	[ir1m, ir1f] = wsmooth(ir1w,l,wsw);
	[ir2m, ir2f] = wsmooth(ir2w,l,wsw);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRERBSmoothed50ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir1f,20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir2f,20 * log10(ir2m),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("ERB smoothed magnitude response, 200 ms window...");
	fflush(stdout);
	
	# Prewindow the response
	wind = [ones(1,441) ones(1,4410) (blackman(8820)(4411:8820))' zeros(1,10240 - (441 + 8820))]';
	
	# Estract the windowed impulse responses
	ir1w = ir1([ic1-441:ic1+(10240 - (441 + 1))]) .* wind;
	ir2w = ir2([ic2-441:ic2+(10240 - (441 + 1))]) .* wind;
	
	# Computes the smoothed magnitude response
	[ir1m, ir1f] = wsmooth(ir1w,l,wsw);
	[ir2m, ir2f] = wsmooth(ir2w,l,wsw);
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MRERBSmoothed200ms.eps" "\"" ";\n" ]);
	
	# Plot the magnitude response
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir1f,20 * log10(ir1m),["1;" it1 ";"]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20] noreverse;\n");
	__gnuplot_raw__("set yrange [-15:15] noreverse;\n");
	__gnuplot_raw__("set xtics (0.02,0.05,0.1,0.2,0.3,0.5,0.7,1,1.5,2,3,5,7,10,16,20);\n");
	__gnuplot_raw__("set ytics -15,3,15;\n");
	semilogx(0.001 * 0.5 * fs * ir2f,20 * log10(ir2m),["3;" it2 ";"]);
	
	# Reset to single plot	
	oneplot();	
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Remove variables no longer used
	clear wind ir1w ir1f ir1m ir2w ir2f ir2m;
	
	# Report the stage
	disp("Spectral decay and formation, high range...");
	fflush(stdout);

	# Spectrogram from -1 to +2 ms
	tbs = 1;
	tbe = 2;
	sbs = 44;
	sas = 88;
	sfs = 64;
	sol = 1;
	
	# 0.5 ms spectrogram window
	swl = 22;
	
	# Computes the spectrogram window, 
	wind = [blackman(swl)' zeros(1,sfs-swl)]';
	
	# Computes the spectrograms
	[S1, f1, t1] = specgram(ir1((ic1 - (-1 + sbs + swl/2)):(ic1 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	[S2, f2, t2] = specgram(ir2((ic2 - (-1 + sbs + swl/2)):(ic2 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	
	# Normalize and truncates the spectral decay
	S1 = abs(S1);
	S1 = S1/max(S1(:));
	S1 = 20 * log10(max(S1,10^(-30/20)));
	
	S2 = abs(S2);
	S2 = S2/max(S2(:));
	S2 = 20 * log10(max(S2,10^(-30/20)));
	
	# Set for surface plotting
	__gnuplot_raw__("unset pm3d;\n");	
	__gnuplot_raw__("set hidden3d;\n");	
	__gnuplot_raw__("set surface;\n");			
	__gnuplot_raw__("set style data lines;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SDHighRange.eps" "\"" ";\n" ]);
	
	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");
	
	# Plot the spectral decay
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [2:20] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t1 - (sbs/fs)),0.001 * f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [2:20] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");
	
	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t2 - (sbs/fs)),0.001 * f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SFHighRange.eps" "\"" ";\n" ]);
	
	# Plot the spectral formation
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [2:20] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cntrparam bspline;\n");
	__gnuplot_raw__("set cntrparam levels auto 8;\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t1 - (sbs/fs)),0.001 * f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [2:20] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t2 - (sbs/fs)),0.001 * f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Spectral decay and formation, mid range...");
	fflush(stdout);
	
	# Spectrogram from -10 to +20 ms
	tbs = 10;
	tbe = 20;
	sbs = 441;
	sas = 882;
	sfs = 1024;
	sol = 20;
	
	# 5 ms spectrogram window
	swl = 220;
	
	# Computes the spectrogram window
	wind = [blackman(swl)' zeros(1,sfs-swl)]';
	
	# Computes the spectrograms
	[S1, f1, t1] = specgram(ir1((ic1 - (-1 + sbs + swl/2)):(ic1 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	[S2, f2, t2] = specgram(ir2((ic2 - (-1 + sbs + swl/2)):(ic2 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	
	# Normalize and truncates the spectral decay
	S1 = abs(S1);
	S1 = S1/max(S1(:));
	S1 = 20 * log10(max(S1,10^(-30/20)));
	
	S2 = abs(S2);
	S2 = S2/max(S2(:));
	S2 = 20 * log10(max(S2,10^(-30/20)));
	
	# Set for surface plotting
	__gnuplot_raw__("unset pm3d;\n");	
	__gnuplot_raw__("set hidden3d;\n");	
	__gnuplot_raw__("set surface;\n");			
	__gnuplot_raw__("set style data lines;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SDMidRange.eps" "\"" ";\n" ]);
	
	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");

	# Plot the spectral decay
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [0.2:2] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t1 - (sbs/fs)),0.001 * f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [0.2:2] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t2 - (sbs/fs)),0.001 * f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SFMidRange.eps" "\"" ";\n" ]);
	
	# Plot the spectral formation
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [0.2:2] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t1 - (sbs/fs)),0.001 * f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [0.2:2] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t2 - (sbs/fs)),0.001 * f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Spectral decay and formation, bass range...");
	fflush(stdout);
	
	# Spectrogram from -0.10 to +0.20 s
	tbs = 0.10;
	tbe = 0.20;
	sbs = 4410;
	sas = 8820;
	sfs = 1024;
	sol = 20;
	
	# 50 ms spctrogram window
	swl = 220;
	
	# Downsample the impulse response 10 times to speed up the spectrogram computation
	dsdf = 10;
	dsfs = fs/dsdf;
	dsfl = 16383;
	dsfd = (dsfl + 1) / (2 * fs); 
	ir1ds = idownsample(ir1(ic1 - (-10 + sbs + swl * 5):ic1 + (sas + sfs * 10)),dsdf,dsfl,0.5,160);
	ir2ds = idownsample(ir2(ic2 - (-10 + sbs + swl * 5):ic2 + (sas + sfs * 10)),dsdf,dsfl,0.5,160);
	
	# Computes the spectrogram window, 
	wind = [blackman(swl)' zeros(1,sfs-swl)]';
	
	# Computes the spectrograms
	[S1, f1, t1] = specgram(ir1ds,sfs,dsfs,wind,sfs - sol);
	[S2, f2, t2] = specgram(ir2ds,sfs,dsfs,wind,sfs - sol);
	
	# Normalize and truncates the spectral decay
	S1 = abs(S1);
	S1 = S1/max(S1(:));
	S1 = 20 * log10(max(S1,10^(-30/20)));
	
	S2 = abs(S2);
	S2 = S2/max(S2(:));
	S2 = 20 * log10(max(S2,10^(-30/20)));

	# Set for surface plotting
	__gnuplot_raw__("unset pm3d;\n");	
	__gnuplot_raw__("set hidden3d;\n");	
	__gnuplot_raw__("set surface;\n");			
	__gnuplot_raw__("set style data lines;\n");
		
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SDBassRange.eps" "\"" ";\n" ]);
	
	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");

	# Plot the spectral decay
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (s)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [20:200] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(t1 - (dsfd + sbs/fs),f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (s)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [20:200] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(t2 - (dsfd + sbs/fs),f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SFBassRange.eps" "\"" ";\n" ]);
	
	# Plot the spectral formation
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (s)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [20:200] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(t1 - (dsfd + sbs/fs),f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (s)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [20:200] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(t2 - (dsfd + sbs/fs),f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Spectral decay and formation, high range, wider window...");
	fflush(stdout);
	
	# Spectrogram from -2 to +4 ms
	tbs = 2;
	tbe = 4;
	sbs = 88;
	sas = 176;
	sfs = 64;
	sol = 2;
	
	# 1 ms spectrogram window
	swl = 44;
	
	# Computes the spectrogram window, 
	wind = [blackman(swl)' zeros(1,sfs-swl)]';
	
	# Computes the spectrograms
	[S1, f1, t1] = specgram(ir1((ic1 - (-1 + sbs + swl/2)):(ic1 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	[S2, f2, t2] = specgram(ir2((ic2 - (-1 + sbs + swl/2)):(ic2 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	
	# Normalize and truncates the spectral decay
	S1 = abs(S1);
	S1 = S1/max(S1(:));
	S1 = 20 * log10(max(S1,10^(-30/20)));
	
	S2 = abs(S2);
	S2 = S2/max(S2(:));
	S2 = 20 * log10(max(S2,10^(-30/20)));

	# Set for surface plotting
	__gnuplot_raw__("unset pm3d;\n");	
	__gnuplot_raw__("set hidden3d;\n");	
	__gnuplot_raw__("set surface;\n");			
	__gnuplot_raw__("set style data lines;\n");
		
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SDHighRangeW.eps" "\"" ";\n" ]);

	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");

	# Plot the spectral decay
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [2:20] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t1 - (sbs/fs)),0.001 * f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [2:20] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t2 - (sbs/fs)),0.001 * f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SFHighRangeW.eps" "\"" ";\n" ]);
	
	# Plot the spectral formation
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [2:20] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t1 - (sbs/fs)),0.001 * f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [2:20] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t2 - (sbs/fs)),0.001 * f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();

	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Spectral decay and formation, mid range, wider window...");
	fflush(stdout);
	
	# Spectrogram from -20 to +40 ms
	tbs = 20;
	tbe = 40;
	sbs = 882;
	sas = 1764;
	sfs = 1024;
	sol = 40;
	
	# 10 ms spectrogram window
	swl = 440;
	
	# Computes the spectrogram window
	wind = [blackman(swl)' zeros(1,sfs-swl)]';
	
	# Computes the spectrograms
	[S1, f1, t1] = specgram(ir1((ic1 - (-1 + sbs + swl/2)):(ic1 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	[S2, f2, t2] = specgram(ir2((ic2 - (-1 + sbs + swl/2)):(ic2 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	
	# Normalize and truncates the spectral decay
	S1 = abs(S1);
	S1 = S1/max(S1(:));
	S1 = 20 * log10(max(S1,10^(-30/20)));
	
	S2 = abs(S2);
	S2 = S2/max(S2(:));
	S2 = 20 * log10(max(S2,10^(-30/20)));

	# Set for surface plotting
	__gnuplot_raw__("unset pm3d;\n");	
	__gnuplot_raw__("set hidden3d;\n");	
	__gnuplot_raw__("set surface;\n");			
	__gnuplot_raw__("set style data lines;\n");
		
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SDMidRangeW.eps" "\"" ";\n" ]);

	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");

	# Plot the spectral decay
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [0.2:2] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t1 - (sbs/fs)),0.001 * f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [0.2:2] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t2 - (sbs/fs)),0.001 * f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();

	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SFMidRangeW.eps" "\"" ";\n" ]);
	
	# Plot the spectral formation
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [0.2:2] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t1 - (sbs/fs)),0.001 * f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [0.2:2] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(1000 * (t2 - (sbs/fs)),0.001 * f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();

	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Report the stage
	disp("Spectral decay and formation, bass range, wider window...");
	fflush(stdout);
	
	# Spectrogram from -0.20 to +0.40 s
	tbs = 0.20;
	tbe = 0.40;
	sbs = 8820;
	sas = 17640;
	sfs = 1024;
	sol = 40;
	
	# 100 ms spctrogram window
	swl = 440;
	
	# Downsample the impulse response 10 times to speed up the spectrogram computation
	dsdf = 10;
	dsfs = fs/dsdf;
	dsfl = 16383;
	dsfd = (dsfl + 1) / (2 * fs); 
	ir1ds = idownsample(ir1(ic1 - (-10 + sbs + swl * 5):ic1 + (sas + sfs * 10)),dsdf,dsfl,0.5,160);
	ir2ds = idownsample(ir2(ic2 - (-10 + sbs + swl * 5):ic2 + (sas + sfs * 10)),dsdf,dsfl,0.5,160);
	
	# Computes the spectrogram window, 
	wind = [blackman(swl)' zeros(1,sfs-swl)]';
	
	# Computes the spectrograms
	[S1, f1, t1] = specgram(ir1ds,sfs,dsfs,wind,sfs - sol);
	[S2, f2, t2] = specgram(ir2ds,sfs,dsfs,wind,sfs - sol);
	
	# Normalize and truncates the spectral decay
	S1 = abs(S1);
	S1 = S1/max(S1(:));
	S1 = 20 * log10(max(S1,10^(-30/20)));
	
	S2 = abs(S2);
	S2 = S2/max(S2(:));
	S2 = 20 * log10(max(S2,10^(-30/20)));

	# Set for surface plotting
	__gnuplot_raw__("unset pm3d;\n");	
	__gnuplot_raw__("set hidden3d;\n");	
	__gnuplot_raw__("set surface;\n");			
	__gnuplot_raw__("set style data lines;\n");
		
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SDBassRangeW.eps" "\"" ";\n" ]);
	
	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");

	# Plot the spectral decay
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (s)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [20:200] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(t1 - (dsfd + sbs/fs),f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (s)\" -3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [20:200] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(t2 - (dsfd + sbs/fs),f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SFBassRangeW.eps" "\"" ";\n" ]);
	
	# Plot the spectral formation
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (s)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [20:200] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(t1 - (dsfd + sbs/fs),f1,S1,[ "title \"" it1 "\" with line 1" ]);
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__("set xlabel \"Time (s)\" 3;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" -3;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",-tbs,tbe) " noreverse;\n" ]);
	__gnuplot_raw__("set yrange [20:200] noreverse;\n");
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set view 60, 330, 1, 1;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Plot the surface
	styledmesh(t2 - (dsfd + sbs/fs),f2,S2,[ "title \"" it2 "\" with line 3" ]);
	
	# Reset to single plot	
	oneplot();

	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Remove variables no longer used
	clear dsdf dsfs dsfl dsfd;
	clear wind S1 f1 t1 S2 f2 t2 ir1ds ir2ds sbs sas sfs sol swl;
	
	# Report the stage
	disp("Spectrogram, 1 ms window...");
	fflush(stdout);
	
	# Spectrogram from -10 to +40 ms
	sbs = 441;
	sas = 1764;
	sfs = 1024;
	sol = 2;
	mlb = 60;
	
	# 1 ms spectrogram window
	swl = 44;
	
	# Computes the spectrogram window
	wind = [blackman(swl)' zeros(1,sfs-swl)]';
	
	# Computes the spectrograms
	S1 = specgram(ir1((ic1 - (-1 + sbs + swl/2)):(ic1 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	S2 = specgram(ir2((ic2 - (-1 + sbs + swl/2)):(ic2 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	
	# Normalize the spectrogram
	S1 = abs(S1);
	S1 = S1/max(S1(:));
	S1 = flipud(20 * log10(S1));
	
	S2 = abs(S2);
	S2 = S2/max(S2(:));
	S2 = flipud(20 * log10(S2));
	
	# Remap the spectrograms for image output
	S1 = round(256 * (mlb + S1)/mlb);
	S2 = round(256 * (mlb + S2)/mlb);
	
	# Write images in postscript format with grayscale colormap
	saveimage([ outdir fprefix "-1Spectrogram01ms.eps" ],S1,"ps",gray(256));
	saveimage([ outdir fprefix "-2Spectrogram01ms.eps" ],S2,"ps",gray(256));
	
	# Write images in ppm format with ocean colormap
	saveimage([ outdir fprefix "-1Spectrogram01ms.ppm" ],S1,"ppm",ocean(256));
	saveimage([ outdir fprefix "-2Spectrogram01ms.ppm" ],S2,"ppm",ocean(256));
	
	# Write images in bmp format with ocean colormap
	bmpwrite(max(S1,1),ocean(256),[ outdir fprefix "-1Spectrogram01ms.bmp" ]);
	bmpwrite(max(S2,1),ocean(256),[ outdir fprefix "-2Spectrogram01ms.bmp" ]);
	
	# Report the stage
	disp("Spectrogram, 20 ms window...");
	fflush(stdout);
	
	# Spectrogram from -100 to +400 ms
	sbs = 4410;
	sas = 17640;
	sfs = 1024;
	sol = 20;
	mlb = 60;
	
	# 20 ms spectrogram window
	swl = 882;
	
	# Computes the spectrogram window
	wind = [blackman(swl)' zeros(1,sfs-swl)]';
	
	# Computes the spectrograms
	S1 = specgram(ir1((ic1 - (-1 + sbs + swl/2)):(ic1 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	S2 = specgram(ir2((ic2 - (-1 + sbs + swl/2)):(ic2 + (sas + sfs))),sfs,fs,wind,sfs - sol);
	
	# Normalize the spectrogram
	S1 = abs(S1);
	S1 = S1/max(S1(:));
	S1 = flipud(20 * log10(S1));
	
	S2 = abs(S2);
	S2 = S2/max(S2(:));
	S2 = flipud(20 * log10(S2));
	
	# Remap the spectrograms for image output
	S1 = round(256 * (mlb + S1)/mlb);
	S2 = round(256 * (mlb + S2)/mlb);
	
	# Write images in postscript format with grayscale colormap
	saveimage([ outdir fprefix "-1Spectrogram20ms.eps" ],S1,"ps",gray(256));
	saveimage([ outdir fprefix "-2Spectrogram20ms.eps" ],S2,"ps",gray(256));
	
	# Write images in ppm format with ocean colormap
	saveimage([ outdir fprefix "-1Spectrogram20ms.ppm" ],S1,"ppm",ocean(256));
	saveimage([ outdir fprefix "-2Spectrogram20ms.ppm" ],S2,"ppm",ocean(256));
	
	# Write images in bmp format with ocean colormap
	bmpwrite(max(S1,1),ocean(256),[ outdir fprefix "-1Spectrogram20ms.bmp" ]);
	bmpwrite(max(S2,1),ocean(256),[ outdir fprefix "-2Spectrogram20ms.bmp" ]);
	
	# Remove variables no longer used
	clear S1 S2;
	
	# Report the stage
	disp("Morlet cycle/octave scalogram, high time resolution...");
	fflush(stdout);
	
	# Morlet scalogram from -10 cicles to +20 cicles
	# 20 Hz to 20 KHz
	# 100 frequency points and 350 cicle points
	cecb = -10;
	cece = 20;
	cenc = 350;
	cesf = 20;
	ceef = 20000;
	cenf = 100;
	
	# Parametri upsampling e filtro
	cefl = 32768;
	ceat = 120;
	ceuf = 8;
	cewm = 0.5;
	
	# Morlet wavelet
	ceft = 2;
			
	# Computes the cicle points
	cecp = linspace(cecb,cece,cenc);
	
	# Computes the frequency points
	cebs = 2 ^ (log2(ceef / cesf) / cenf); 
	cefp = logspace(log10(2 * cesf / (cebs * fs)),log10(2 * ceef / (cebs * fs)),cenf);
	
	# Computes the scalograms
	ce1 = tfir(ir1, ic1, cecp, cefp, cefl, ceat, ceuf, ceft, cewm);
	ce2 = tfir(ir2, ic2, cecp, cefp, cefl, ceat, ceuf, ceft, cewm);
	
	# Remaps the frequency points for plotting
	cefp = logspace(log10(2 * cesf / fs),log10(2 * ceef / fs),cenf);
	
	# Perform the scalogram interpolation
	# Not working yet, disabled.
	cpif = 1;
	fpif = 1;
	
	# Check if interpolation is required
	if (cpif > 1 || fpif > 1)
		# Computes the interpolation cicle points
		itcp = linspace(cecb,cece,cenc * cpif);
		
		# Computes the frequency points
		itfp = logspace(log10(2 * cesf / (cebs * fs)),log10(2 * ceef / (cebs * fs)),cenf * fpif);
		
		# Interpolates the envelope
		ce1 = bicubic(cecp,cefp,ce1,itcp,itfp);
		ce2 = bicubic(cecp,cefp,ce2,itcp,itfp);
		
		# Assign the points
		cecp = itcp;
		cefp = logspace(log10(2 * cesf / fs),log10(2 * ceef / fs),cenf * fpif);
		
		# Delete unused variables
		clear itcp itfp;
	else
		# Remaps the frequency points for plotting
		cefp = logspace(log10(2 * cesf / fs),log10(2 * ceef / fs),cenf * fpif);
	endif
		
	# Computes the envelope
	for I = 1:cenf
		ce1(I,:) = abs(hilbert(ce1(I,:)));
		ce2(I,:) = abs(hilbert(ce2(I,:)));
	endfor
	
	# Normalize the envelope
	ce1 = ce1/max(ce1(:));
	ce2 = ce2/max(ce2(:));
	
	# Check for pm3d parameters
	if (pm3denable > 0)		
		__gnuplot_raw__("unset hidden3d;\n");
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("unset surface;\n");
		__gnuplot_raw__("set style line 1 lt 6 lw 0;\n");
		__gnuplot_raw__("set pm3d at s hidden3d 1;\n");	
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set palette defined ( 0 \"dark-blue\", 1 \"dark-yellow\", 2 \"red\" );\n");
	else
		__gnuplot_raw__("unset pm3d;\n");	
		__gnuplot_raw__("set hidden3d;\n");	
		__gnuplot_raw__("set surface;\n");			
		__gnuplot_raw__("set style data lines;\n");
	endif
	
	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");

	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramHTRENVSD.eps" "\"" ";\n" ]);
		
	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 5;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%16g\";\n");	
	__gnuplot_raw__("set view 60, 30, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%16g\";\n");	
	__gnuplot_raw__("set view 60, 30, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 3");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
		
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramHTRENVSF.eps" "\"" ";\n" ]);
	
	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%-16g\";\n");	
	__gnuplot_raw__("set view 60, 330, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
		
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%-16g\";\n");	
	__gnuplot_raw__("set view 60, 330, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 3");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
		
	# Reset to single plot	
	oneplot();

	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Check for pm3d parameters and set for map view	
	if (pm3denable > 0)
		__gnuplot_raw__("set grid noztics;\n");
		__gnuplot_raw__("unset hidden3d;\n");
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("unset key;\n");
		__gnuplot_raw__("set surface;\n");
		__gnuplot_raw__("set style line 1 lt 6 lw 0;\n");
		__gnuplot_raw__("set pm3d at b nohidden3d;\n");			
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set palette defined ( 0 \"dark-blue\", 1 \"dark-yellow\", 2 \"red\" );\n");			
		__gnuplot_raw__("set ticslevel 0.0;\n");
	else
		__gnuplot_raw__("unset pm3d;\n");	
		__gnuplot_raw__("set hidden3d;\n");	
		__gnuplot_raw__("unset surface;\n");			
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set ticslevel 0.0;\n");
	endif	

	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");

	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramHTRENVMap.eps" "\"" ";\n" ]);
		
	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,0;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 0,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0,2;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%g\";\n");
	__gnuplot_raw__("set view map;\n");
		
	# Check for contour parameters
	if (mapcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set key off;\n");
	else
		__gnuplot_raw__("unset contour;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line palette");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");	
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,0;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 0,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0,2;\n");	
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%g\";\n");
	__gnuplot_raw__("set view map;\n");
		
	# Check for contour parameters
	if (mapcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set key off;\n");
	else
		__gnuplot_raw__("unset contour;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 3");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line palette");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");	
	
	# Convert to log scale
	ce1 = 20 * log10(max(ce1,10^(-30/20)));
	ce2 = 20 * log10(max(ce2,10^(-30/20)));

	# Check for pm3d parameters
	if (pm3denable > 0)		
		__gnuplot_raw__("unset hidden3d;\n");
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("unset surface;\n");
		__gnuplot_raw__("set style line 1 lt 6 lw 0;\n");
		__gnuplot_raw__("set pm3d at s hidden3d 1;\n");	
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set palette defined ( 0 \"dark-blue\", 1 \"dark-yellow\", 2 \"red\" );\n");	
	else
		__gnuplot_raw__("unset pm3d;\n");	
		__gnuplot_raw__("set hidden3d;\n");	
		__gnuplot_raw__("set surface;\n");			
		__gnuplot_raw__("set style data lines;\n");
	endif
	
	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramHTRETCSD.eps" "\"" ";\n" ]);
	
	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%16g\";\n");	
	__gnuplot_raw__("set view 60, 30, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%16g\";\n");	
	__gnuplot_raw__("set view 60, 30, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 3");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	# Reset to single plot	
	oneplot();

	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramHTRETCSF.eps" "\"" ";\n" ]);
	
	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%-16g\";\n");	
	__gnuplot_raw__("set view 60, 330, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%-16g\";\n");	
	__gnuplot_raw__("set view 60, 330, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");

	# Check for pm3d parameters and set for map view
	if (pm3denable > 0)
		__gnuplot_raw__("set grid noztics;\n");
		__gnuplot_raw__("unset hidden3d;\n");
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("unset key;\n");
		__gnuplot_raw__("set surface;\n");
		__gnuplot_raw__("set style line 1 lt 6 lw 0;\n");
		__gnuplot_raw__("set pm3d at b nohidden3d;\n");			
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set palette defined ( 0 \"dark-blue\", 1 \"dark-yellow\", 2 \"red\" );\n");	
		__gnuplot_raw__("set ticslevel 0.0;\n");		
	else
		__gnuplot_raw__("unset pm3d;\n");	
		__gnuplot_raw__("set hidden3d;\n");	
		__gnuplot_raw__("unset surface;\n");			
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set ticslevel 0.0;\n");
	endif
	
	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");

	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramHTRETCMap.eps" "\"" ";\n" ]);
	
	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,0;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 0,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,2;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%g\";\n");
	__gnuplot_raw__("set view map;\n");	
	
	# Check for contour parameters
	if (mapcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set key off;\n");
	else
		__gnuplot_raw__("unset contour;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line palette");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,0;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 0,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,2;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%g\";\n");
	__gnuplot_raw__("set view map;\n");

	# Check for contour parameters
	if (mapcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set key off;\n");
	else
		__gnuplot_raw__("unset contour;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 3");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line palette");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");
	
	# Reset to single plot	
	oneplot();

	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Remove variables no longer used
	clear cecp cefp ce1 ce2;	

	# Report the stage
	disp("Morlet cycle/octave scalogram, medium time resolution...");
	fflush(stdout);
	
	# Morlet cycle/octave scalogram from -10 cicles to +20 cicles
	# 20 Hz to 20 KHz
	# 100 frequency points and 350 cicle points
	cecb = -10;
	cece = 20;
	cenc = 350;
	cesf = 20;
	ceef = 20000;
	cenf = 100;
	
	# Parametri upsampling e filtro
	cefl = 32768;
	ceat = 120;
	ceuf = 8;
	cewm = 2;
	
	# Morlet wavelet
	ceft = 2;
			
	# Computes the cicle points
	cecp = linspace(cecb,cece,cenc);
	
	# Computes the frequency points
	cebs = 2 ^ (log2(ceef / cesf) / cenf); 
	cefp = logspace(log10(2 * cesf / (cebs * fs)),log10(2 * ceef / (cebs * fs)),cenf);
	
	# Computes the Morlet cycle/octave scalogram
	ce1 = tfir(ir1, ic1, cecp, cefp, cefl, ceat, ceuf, ceft, cewm);
	ce2 = tfir(ir2, ic2, cecp, cefp, cefl, ceat, ceuf, ceft, cewm);
	
	# Remaps the frequency points for plotting
	cefp = logspace(log10(2 * cesf / fs),log10(2 * ceef / fs),cenf);
	
	# Computes the envelope
	for I = 1:cenf
		ce1(I,:) = abs(hilbert(ce1(I,:)));
		ce2(I,:) = abs(hilbert(ce2(I,:)));
	endfor
	
	# Normalize the envelope
	ce1 = ce1/max(ce1(:));
	ce2 = ce2/max(ce2(:));
	
	# Check for pm3d parameters
	if (pm3denable > 0)		
		__gnuplot_raw__("unset hidden3d;\n");
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("unset surface;\n");
		__gnuplot_raw__("set style line 1 lt 6 lw 0;\n");
		__gnuplot_raw__("set pm3d at s hidden3d 1;\n");	
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set palette defined ( 0 \"dark-blue\", 1 \"dark-yellow\", 2 \"red\" );\n");
	else
		__gnuplot_raw__("unset pm3d;\n");	
		__gnuplot_raw__("set hidden3d;\n");	
		__gnuplot_raw__("set surface;\n");			
		__gnuplot_raw__("set style data lines;\n");
	endif
	
	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");

	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramMTRENVSD.eps" "\"" ";\n" ]);
		
	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);	
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%16g\";\n");	
	__gnuplot_raw__("set view 60, 30, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%16g\";\n");	
	__gnuplot_raw__("set view 60, 30, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 3");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramMTRENVSF.eps" "\"" ";\n" ]);
	
	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%-16g\";\n");	
	__gnuplot_raw__("set view 60, 330, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%-16g\";\n");	
	__gnuplot_raw__("set view 60, 330, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 3");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	# Reset to single plot	
	oneplot();

	# Reset the output file
	__gnuplot_raw__("set output;\n");
		
	# Check for pm3d parameters and set for map view	
	if (pm3denable > 0)
		__gnuplot_raw__("set grid noztics;\n");
		__gnuplot_raw__("unset hidden3d;\n");
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("unset key;\n");
		__gnuplot_raw__("set surface;\n");
		__gnuplot_raw__("set style line 1 lt 6 lw 0;\n");
		__gnuplot_raw__("set pm3d at b nohidden3d;\n");			
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set palette defined ( 0 \"dark-blue\", 1 \"dark-yellow\", 2 \"red\" );\n");	
		__gnuplot_raw__("set ticslevel 0.0;\n");
	else
		__gnuplot_raw__("unset pm3d;\n");	
		__gnuplot_raw__("set hidden3d;\n");	
		__gnuplot_raw__("unset surface;\n");			
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set ticslevel 0.0;\n");
	endif	

	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramMTRENVMap.eps" "\"" ";\n" ]);

	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,0;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 0,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0,2;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%g\";\n");
	__gnuplot_raw__("set view map;\n");	
	
	# Check for contour parameters
	if (mapcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set key off;\n");
	else
		__gnuplot_raw__("unset contour;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line palette");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,0;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 0,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (norm.)\" 0,2;\n");	
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [0:1] noreverse;\n");
	__gnuplot_raw__("set cbrange [0:1] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set cbtics (0,0.25,0.5,0.75,1.0);\n");
	__gnuplot_raw__("set format y \"%g\";\n");
	__gnuplot_raw__("set view map;\n");
	
	# Check for contour parameters
	if (mapcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set key off;\n");
	else
		__gnuplot_raw__("unset contour;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 3");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line palette");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");	
	
	# Convert to log scale
	ce1 = 20 * log10(max(ce1,10^(-30/20)));
	ce2 = 20 * log10(max(ce2,10^(-30/20)));
	
	# Check for pm3d parameters
	if (pm3denable > 0)		
		__gnuplot_raw__("unset hidden3d;\n");
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("unset surface;\n");
		__gnuplot_raw__("set style line 1 lt 6 lw 0;\n");
		__gnuplot_raw__("set pm3d at s hidden3d 1;\n");	
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set palette defined ( 0 \"dark-blue\", 1 \"dark-yellow\", 2 \"red\" );\n");	
	else
		__gnuplot_raw__("unset pm3d;\n");	
		__gnuplot_raw__("set hidden3d;\n");	
		__gnuplot_raw__("set surface;\n");			
		__gnuplot_raw__("set style data lines;\n");
	endif
	
	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");

	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramMTRETCSD.eps" "\"" ";\n" ]);
	
	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%16g\";\n");
	__gnuplot_raw__("set view 60, 30, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%16g\";\n");	
	__gnuplot_raw__("set view 60, 30, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 3");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	# Reset to single plot	
	oneplot();

	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramMTRETCSF.eps" "\"" ";\n" ]);
	
	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%-16g\";\n");	
	__gnuplot_raw__("set view 60, 330, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
		
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");

	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__("set grid ztics;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,1;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,-1;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" -5,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,0;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%-16g\";\n");	
	__gnuplot_raw__("set view 60, 330, 1, 1.2;\n");

	# Check for contour parameters
	if (surcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set ticslevel 0.5;\n");
	else
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("set ticslevel 0.07;\n");
	endif
		
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 3");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	
	# Reset to single plot	
	oneplot();
	
	# Reset the output file
	__gnuplot_raw__("set output;\n");
		
	# Check for pm3d parameters and set for map view
	if (pm3denable > 0)
		__gnuplot_raw__("set grid noztics;\n");
		__gnuplot_raw__("unset hidden3d;\n");
		__gnuplot_raw__("unset contour;\n");
		__gnuplot_raw__("unset key;\n");
		__gnuplot_raw__("set surface;\n");
		__gnuplot_raw__("set style line 1 lt 6 lw 0;\n");
		__gnuplot_raw__("set pm3d at b nohidden3d;\n");			
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set palette defined ( 0 \"dark-blue\", 1 \"dark-yellow\", 2 \"red\" );\n");	
		__gnuplot_raw__("set ticslevel 0.0;\n");		
	else
		__gnuplot_raw__("unset pm3d;\n");	
		__gnuplot_raw__("set hidden3d;\n");	
		__gnuplot_raw__("unset surface;\n");			
		__gnuplot_raw__("set style data lines;\n");
		__gnuplot_raw__("set ticslevel 0.0;\n");
	endif
	
	# Reset the control parameters
	__gnuplot_raw__("unset logscale;\n");
	__gnuplot_raw__("set xtics autofreq;\n");
	__gnuplot_raw__("set ytics autofreq;\n");
	__gnuplot_raw__("set ztics autofreq;\n");
	
	# Set the output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-MorletScalogramMTRETCMap.eps" "\"" ";\n" ]);
	
	# Plot the scalogram
	subplot(2,1,1);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__([ "set title \"" it1 "\" 0,0;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 0,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,2;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%g\";\n");
	__gnuplot_raw__("set view map;\n");
	
	# Check for contour parameters
	if (mapcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set key off;\n");
	else
		__gnuplot_raw__("unset contour;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line 1");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce1,"with line palette");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");
	
	subplot(2,1,2);
	grid("on");
	__gnuplot_raw__("set logscale y;\n");
	__gnuplot_raw__([ "set title \"" it2 "\" 0,0;\n" ]);
	__gnuplot_raw__("set xlabel \"Time (cycles)\" 0,0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (KHz)\" 0,0;\n");
	__gnuplot_raw__("set zlabel \"Magnitude (dB)\" 0,2;\n");
	__gnuplot_raw__([ "set xrange " sprintf("[%e:%e]",cecb,cece) " noreverse;\n" ]);
	__gnuplot_raw__([ "set yrange " sprintf("[%e:%e]",cesf / 1000,ceef / 1000) " noreverse;\n" ]);
	__gnuplot_raw__("set zrange [-30:2] noreverse;\n");
	__gnuplot_raw__("set cbrange [-30:0] noreverse;\n");
	__gnuplot_raw__("set xtics 5;\n");
	__gnuplot_raw__("set ytics (0.02,0.05,0.1,0.2,0.5,1,2,5,10,20);\n");
	__gnuplot_raw__("set ztics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set cbtics (-30,-25,-20,-15,-10,-5,0,5);\n");
	__gnuplot_raw__("set format y \"%g\";\n");
	__gnuplot_raw__("set view map;\n");
	
	# Check for contour parameters
	if (mapcntenable > 0)		
		__gnuplot_raw__("set contour base;\n");
		__gnuplot_raw__("set cntrparam bspline;\n");
		__gnuplot_raw__("set cntrparam levels auto 8;\n");
		__gnuplot_raw__("set key off;\n");
	else
		__gnuplot_raw__("unset contour;\n");
	endif
	
	# Check for pm3d parameters
	if (pm3denable <= 0)	
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line 3");
	else
		# Plot the surface
		styledmesh(cecp,cefp * fs / 2000,ce2,"with line palette");
	endif
	
	# Reset unwanted parameters
	__gnuplot_raw__("unset title;\n");
	__gnuplot_raw__("set view 60, 30, 1, 1;\n");
	
	# Reset to single plot	
	oneplot();

	# Reset the output file
	__gnuplot_raw__("set output;\n");
	
	# Remove variables no longer used
	clear cecp cefp ce1 ce2;

	# Report the stage
	disp("Completed, wait for GNUPlot to complete before opening output files.");
	fflush(stdout);

endfunction