## Copyright (C) 2005 Denis Sbragion
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
##(at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##
## This function is part of DRC - Digital room correction

## usage: createbasicplots(outdir,fprefix);
##
## createbasicplots - Creates the basic DRC plots
##
## ptfname = psychoacoustic target sample response file name
## outdir = output directory for eps files
## fprefix = prefix for eps files

function createbasicplots(ptfname,outdir,fprefix);

	# Set the frequency dependent windowing basic parameters
	sf = 20;
	sw = 500;
	ef = 20000;
	ew = 0.5;
	np = 32768;
	ptw = 0.3; # 300 ms
	
	# Creates the windows array
	wa = linspace(ew,sw,np);
	
	# Computes the gabor limit
	gl = 1000 ./ (2 * linspace(ew/4,sw,np));
	
	# Unset the automatic replot
	if (exist("automatic_replot") == 101)
		automatic_replot = 0;
	else
		automatic_replot(0);
	endif
	
	# Set terminal to unknown for initial simulated plotting postscript eps
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, linear scale...");
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics 0,50,550;\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics 0,1000,21000;\n");
	__gnuplot_raw__("set key right top;\n");
	for gn = 1:10
		plot(wa,fslprefilt(sf,sw,ef,ew,0.2 * gn,np),sprintf(";WE = %.1f;",0.2 * gn));
		hold on;
	endfor
	hold off;
	
	# Reset the gnuplot status
	__gnuplot_raw__("reset;\n");

	# Set terminal to postscript eps for the real plotting	
	__gnuplot_raw__("set terminal postscript eps color solid linewidth 1 rounded;\n");
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics 0,50,550;\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics 0,1000,21000;\n");
	__gnuplot_raw__("set key right top;\n");
	
	# Set output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-Linear.eps" "\"" "\n" ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__("set output;\n");
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, semilogy scale...");
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics 0,50,550;\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n");
	__gnuplot_raw__("set key right top;\n");
	for gn = 1:10
		semilogy(wa,fslprefilt(sf,sw,ef,ew,0.2 * gn,np),sprintf(";WE = %.1f;",0.2 * gn));
		hold on;
	endfor
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__("set terminal postscript eps color solid linewidth 1 rounded;\n");
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics 0,50,550;\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n");
	__gnuplot_raw__("set key right top;\n");
	
	# Set output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SemiLogY.eps" "\"" "\n" ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__("set output;\n");
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, semilogx scale...");
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics 0,1000,21000;\n");
	__gnuplot_raw__("set key right top;\n");
	for gn = 1:10
		semilogx(wa,fslprefilt(sf,sw,ef,ew,0.2 * gn,np),sprintf(";WE = %.1f;",0.2 * gn));
		hold on;
	endfor
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__("set terminal postscript eps color solid linewidth 1 rounded;\n");
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics 0,1000,21000;\n");
	__gnuplot_raw__("set key right top;\n");
	
	# Set output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SemiLogX.eps" "\"" "\n" ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__("set output;\n");
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, loglog scale...");
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n");
	__gnuplot_raw__("set key right top;\n");
	for gn = 1:10
		loglog(wa,fslprefilt(sf,sw,ef,ew,0.2 * gn,np),sprintf(";WE = %.1f;",0.2 * gn));
		hold on;
	endfor
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__("set terminal postscript eps color solid linewidth 1 rounded;\n");
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n");
	__gnuplot_raw__("set key right top;\n");
	
	# Set output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-LogLog.eps" "\"" "\n" ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__("set output;\n");
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, loglog scale with gabor limit...");
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n");
	__gnuplot_raw__("set key right top;\n");
	for gn = 1:10
		loglog(wa,fslprefilt(sf,sw,ef,ew,0.2 * gn,np),sprintf(";WE = %.1f;",0.2 * gn));
		hold on;
	endfor
	loglog(linspace(ew/4,sw,np),gl,";Gabor;");
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__("set terminal postscript eps color solid linewidth 1 rounded;\n");
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n");
	__gnuplot_raw__("set key right top;\n");
	
	# Set output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-Gabor.eps" "\"" "\n" ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__("set output;\n");
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, time-frequency jail...");
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [-50:550];\n");
	__gnuplot_raw__("set xtics -50,50,550;\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n");
	__gnuplot_raw__("set key right top;\n");
	semilogy(-wa/25,fslprefilt(sf,sw/25,ef,ew,1.0,np),"1;Pre-echo truncation;");
	hold on;
	semilogy(wa,fslprefilt(sf,sw,ef,ew,1.0,np),"3;Ringing truncation;");
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__("set terminal postscript eps color solid linewidth 1 rounded;\n");
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [-50:550];\n");
	__gnuplot_raw__("set xtics -50,50,550;\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n");
	__gnuplot_raw__("set key right top;\n");
	
	# Set output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-Jail.eps" "\"" "\n" ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__("set output;\n");
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, psychoacoustic bandwidth comparison...");
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.01:21];\n");
	__gnuplot_raw__("set xtics (0.01,0.02,0.03,0.05,0.1,0.2,0.3,0.5,1,2,3,5,10,20);\n");
	__gnuplot_raw__("set yrange [0.5:10000];\n");
	__gnuplot_raw__("set ytics (0.5,1,2,3,5,10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000);\n");
	__gnuplot_raw__("set key left top;\n");
	loglog(linspace(sf,ef,np)/1000,(1000 * 0.5) ./ (wslprefilt(sf,sw,ef,ew,1.0,np) + wslprefilt(sf,sw/20,ef,ew,1.0,np)),"1;normal.drc;");
	hold on;
	loglog(linspace(sf,ef,np)/1000,(1000 * 2.0) ./ (wbslprefilt(sf,65,ef,0.46,1.87,np) + wbslprefilt(sf,10,ef,0.46,1.87,np)),"2;erb.drc;");
	hold on;
	loglog(linspace(sf,ef,np)/1000,barkbwidth(linspace(sf,ef,np)),"3;Bark;");
	hold on;
	loglog(linspace(sf,ef,np)/1000,erbbwidth(linspace(sf,ef,np)),"4;ERB;");
	hold on;
	loglog(linspace(sf,ef,np)/1000,foctbwidth(linspace(sf,ef,np),1/3),"5;1-3 oct;");
	hold on;
	loglog(linspace(sf,ef,np)/1000,foctbwidth(linspace(sf,ef,np),1/6),"6;1-6 oct;");
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__("set terminal postscript eps color solid linewidth 1 rounded;\n");
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.01:21];\n");
	__gnuplot_raw__("set xtics (0.01,0.02,0.03,0.05,0.1,0.2,0.3,0.5,1,2,3,5,10,20);\n");
	__gnuplot_raw__("set yrange [0.5:10000];\n");
	__gnuplot_raw__("set ytics (0.5,1,2,3,5,10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000);\n");
	__gnuplot_raw__("set key left top;\n");
	
	# Set output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-BWidthCmp.eps" "\"" "\n" ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__("set output;\n");
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Report the stage
	disp("Frequency dependent windowing, proportional and bilinear comparison...");
	fflush(stdout);
	
	# Plot the frequency dependent windowing curves
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n");
	__gnuplot_raw__("set key right top;\n");
	for gn = 1:7
		loglog(wa,fslprefilt(sf,sw,ef,ew,0.2 + 0.2 * gn,np),sprintf("1;Prop. WE = %.1f;",0.2 + 0.2 * gn));
		hold on;
	endfor
	for gn = 1:7
		loglog(wa,fbslprefilt(sf,sw,ef,ew,0.2 + 0.2 * gn,np),sprintf("3;Bil. WE = %.1f;",0.2 + 0.2 * gn));
		hold on;
	endfor
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__("set terminal postscript eps color solid linewidth 1 rounded;\n");
	grid("on");
	__gnuplot_raw__("set xlabel \"Time (ms)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Frequency (Hz)\" 0;\n");
	__gnuplot_raw__("set xrange [0.3:550];\n");
	__gnuplot_raw__("set xtics (0.3,0.5,1,2,3,5,10,20,30,50,100,200,300,500);\n");
	__gnuplot_raw__("set yrange [10:21000];\n");
	__gnuplot_raw__("set ytics (10,20,30,50,100,200,300,500,1000,2000,3000,5000,10000,20000);\n");
	__gnuplot_raw__("set key right top;\n");
	
	# Set output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-BPComparison.eps" "\"" "\n" ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__("set output;\n");
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Remove variables no longer used
	clear sf sw ef ew wa gl;
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Report the stage
	disp("Transfer functions...");
	fflush(stdout);
	
	# Prepare the target functions
	bkf = [ 0 10 20 400 12800 20000 21500 22050 ];
	bkm = [ -20.0 -10.0 0.00 0.00 -5.00 -6.00 -10.0 -20.0 ];
	
	bk2f = [ 0 10 20 200 12800 20000 21500 22050 ];
	bk2m = [ -20.0 -10.0 0.00 0.00 -3.0 -3.5 -10.0 -20.0 ];
	
	bk2sf = [ 0 16 18 200 12800 20000 21500 22050 ];
	bk2sm = [ -200.0 -120.0 0.00 0.00 -3.0 -3.5 -10.0 -20.0 ];
	
	bk3f = [ 0 10 20 100 12800 20000 21500 22050 ];
	bk3m = [ -20.0 -10.0 0.00 0.00 -3.50 -4.00 -10.0 -20.0 ];
	
	bk3sf = [ 0 16 18 100 12800 20000 21500 22050 ];
	bk3sm = [ -200.0 -120.0 0.00 0.00 -3.50 -4.00 -10.0 -20.0 ];
	
	bk3sfsp = [ 0 14 15 16 17 18 19 20 50 100 150 19900 20000 20100 20200 20300 20400 20500 22050 ];
	bk3smsp = [ -200.0 -200.0 -200.0 -200.0 -100.0 0.00 0.00 0.00 0.00 0.00 0.00 -3.95 -4.00 -4.00 -98.0 -200.0 -200.0 -200.0 -200.0 ];
	
	# Plot the target functions
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.01:22.5];\n");
	__gnuplot_raw__("set xtics (0.01,0.02,0.03,0.05,0.1,0.2,0.3,0.5,1,2,3,5,10,20);\n");
	__gnuplot_raw__("set yrange [-30:9];\n");
	__gnuplot_raw__("set ytics -30,3,9;\n");
	__gnuplot_raw__("set key right top;\n");
	
	semilogx(bkf/1000,bkm,"1;bk;");
	hold on;
	
	semilogx(bk2f/1000,bk2m,"2;bk-2;");
	hold on;
	
	semilogx(bk3f/1000,bk3m,"5;bk-3;");
	hold on;
	
	semilogx(bk2sf/1000,bk2sm,"4;bk-2-sub;");
	hold on;
	
	semilogx(bk3sf/1000,bk3sm,"3;bk-3-sub;");
	hold on;
	
	lbk3sfsp = log10(0.01 + bk3sfsp);
	spx = log10(logspace(log10(0.01),log10(22050.01),np));
	semilogx((10 .^ ub3splinev(lbk3sfsp,lbk3sfsp,spx)) / 1000,
		-6 + ub3splinev(lbk3sfsp,bk3smsp,spx),"8;bk-3-subultra-spline (-6 dB);");	
	hold on;
	
	semilogx(logspace(log10(0.01),log10(22050.01),np) / 1000,
		-9 + interp1(log10(0.01 + bk3sf),bk3sm,spx,'pchip'),
		"9;bk-3-sub-pchip (-9 dB);");	
	clear spx lbk3sfsp;
	hold off;
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__("set terminal postscript eps color solid linewidth 1 rounded;\n");
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.01:22.5];\n");
	__gnuplot_raw__("set xtics (0.01,0.02,0.03,0.05,0.1,0.2,0.3,0.5,1,2,3,5,10,20);\n");
	__gnuplot_raw__("set yrange [-30:9];\n");
	__gnuplot_raw__("set ytics -30,3,9;\n");
	__gnuplot_raw__("set key right top;\n");
	
	# Set output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-DTFCmp.eps" "\"" "\n" ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__("set output;\n");
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Remove variables no longer used
	clear sf sw ef ew wa gl;	
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Report the stage
	disp("Spectral envelope and standard smoothing comparison...");
	fflush(stdout);
	
	# Calcola la finestratura
	ptwl = round(ptw * 44100);
	
	# Carica il file campione
	ptr = loadpcm(ptfname);
	
	# Trova la posizione del picco
	[ptm, pti] = max(abs(ptr));
	
	# Effettua la finestratura
	ptr = (ptr((pti-(ptwl-1)):(pti+ptwl)) .* blackman(2 * ptwl)) / ptm;

	# Calcola l'inviluppo spettrale
	ptev = 20 * log10(ssev(ptr,44100,-2,15));
	
	# Calcola lo spettro unsmoothed
	ptus = 20 * log10(abs(fft(postpad(ptr,2 * length(ptev)))(1:length(ptev))));
	
	# Calcola lo smoothing standard
	ptss = 20 * log10(ssev(ptr,44100,-2,1));
	
	# Calcola l'array risposta in frequenza
	ptf = linspace(0,22050,length(ptev));
	
	# Calcola il punto di riferimento in frequenza
	ptfp = round(100 * 22050 / length(ptf));
	
	# Plot the spectral envelope example
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20];\n");
	__gnuplot_raw__("set xtics (0.02,0.03,0.05,0.1,0.2,0.3,0.5,1,2,3,5,10,20);\n");
	__gnuplot_raw__("set yrange [-35:25];\n");
	__gnuplot_raw__("set ytics -35,5,25;\n");
	__gnuplot_raw__("set key right top;\n");
	
	semilogx(ptf/1000,ptus,"1;Unsmoothed;");
	hold on;
	
	semilogx(ptf/1000,ptss,"2;ERB smoothed;");
	hold on;
	
	semilogx(ptf/1000,ptev,"3;ERB spectral envelope;");
	hold on;
	
	semilogx([20 100 20000] / 1000,[0 0 log2(20000 / 100)] + ptev(ptfp),"5;Reference target;");
	hold off;	
	
	# Set terminal to postscript eps for the real plotting
	__gnuplot_raw__("set terminal postscript eps color solid linewidth 1 rounded;\n");
	grid("on");
	__gnuplot_raw__("set xlabel \"Frequency (KHz)\" 0;\n");
	__gnuplot_raw__("set ylabel \"Magnitude (dB)\" 0;\n");
	__gnuplot_raw__("set xrange [0.02:20];\n");
	__gnuplot_raw__("set xtics (0.02,0.03,0.05,0.1,0.2,0.3,0.5,1,2,3,5,10,20);\n");
	__gnuplot_raw__("set yrange [-35:25];\n");
	__gnuplot_raw__("set ytics -35,5,25;\n");
	__gnuplot_raw__("set key right top;\n");
	
	# Set output file
	__gnuplot_raw__([ "set output " "\"" outdir fprefix "-SpectralEnvelope.eps" "\"" "\n" ]);
	
	# Performs the real plot
	replot;
	
	# Reset output file
	__gnuplot_raw__("set output;\n");
	
	# Reset the terminal to unknown
	__gnuplot_raw__("set terminal unknown;\n");
	
	# Remove variables no longer used
	clear ptf ptr ptev ptus ptss;	
	
	# Report the stage
	disp("Completed, wait for GNUPlot to complete before opening output files.");
	fflush(stdout);

endfunction