## Copyright (C) 2005 Denis Sbragion
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
##(at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##
## This function is part of DRC - Digital room correction

##
## usage: csir = cplxsmooth(s,bw,pds)
##
## Computes the complex smoothed impulse response of signal s on a 
## logarithmic frequency scale with bw bandwidth resolution and 
## pds peak detection strength. 
##
## s   = input signal.
## bw  = bandwidth resolution as a faction of octave, for example 0.25
##       means 1/4 of octave resolution.
## pds = peak detection strength, values above 30 migth lead to 
##       overflow. When set to 1, the algorithm provides the usual 
##       fractional octave smoothing. Values lower than 1 cause 
##       dip detection instead of peak detection.
## csir = output smoothed impulse response, same length of s.
##
## Typical values for psychoacoustic evaluation of the perceived 
## frequency response:
## 
## csir = cplxsmooth(s,1/6,16);
##

function csir = cplxsmooth(s,bw,pds)
	# Get the number of signal samples
	n = length(s);
	
	# Computes the half bandwidth
	hbw = (2 ^ (bw / 2));
	
	# Computes the signal spectrum
	pf = fft(postpad(s(:)',2 * n))(1:n);
	
	# Allocates the envelope array
	csir = zeros(1,n);
	
	# Check if peak detection strength is equal to 1
	if (pds == 1)
		# No peak detection
		# Creates the peak smoothing cumulative sum
		pf = cumsum(pf);		
	else	
		# Creates the peak smoothing cumulative sum
		pf = cumsum(pf .^ pds);		
			
		# Pad the cumulative sum to the bandwitdth bounds
		bu = 1 + round((n - 1) * hbw);
		pf = [0 pf (pf(n) * ones(1,bu - n))];
				
		# Cicles on the signal spectrum to get the peak enhanced smoothing
		for I=1:n
			# Computes the current smoothing bounds
			bl = 1 + round((I - 1) / hbw);
			bu = 1 + round((I - 1) * hbw);		
			
			# Gets the rms value of the peaks		
			csir(I) = ((pf(bu + 1) - pf(bl))/(1 + (bu - bl))) .^ (1 / pds);
		endfor	
		
		# Creates the peak smoothing cumulative sum
		pf = cumsum(csir);
	endif
	
	# Pad the cumulative sum to the bandwitdth bounds
	bu = 1 + round((n - 1) * hbw);
	pf = [0 pf (pf(n) * ones(1,bu - n))];
	
	# Cicles on the signal spectrum to smooth the peak function		
	for I=1:n
		# Computes the current smoothing bounds
		bl = 1 + round((I - 1) / hbw);
		bu = 1 + round((I - 1) * hbw);		
		
		# Gets the rms value of the peaks
		csir(I) = (pf(bu + 1) - pf(bl)) / (1 + (bu - bl));
	endfor	
	
	# Convert the spectrum to an impulse response
	csir = real(ifft([csir conj(csir(n:-1:1))]))(1:n);
endfunction