"""This module defines a Form class that wraps FFC forms and UFC forms
into a cpp.Form (dolfin::Form)."""

# Copyright (C) 2008-2014 Johan Hake
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Anders Logg, 2008-2011
# Modified by Marie E. Rognes, 2011
# Modified by Martin Alnes, 2014

__all__ = ["Form"]

# Import SWIG-generated extension module (DOLFIN C++)
import dolfin.cpp as cpp

# Import ufl and ufc
import ufl, ufc

# Import JIT compiler
from dolfin.compilemodules.jit import jit

from dolfin.cpp import deprecation
from dolfin.cpp import MeshFunctionSizet

# Note that we need to store _compiled_form and _compiled_coefficients
# to prevent Python from garbage-collecting these while still in use.
# FIXME: Figure out how to solve this with shared_ptr

function_space_error = "Error while extracting test and/or trial spaces. "

class Form(cpp.Form):
    def __init__(self, form,
                 form_compiler_parameters=None,
                 subdomains=None):
        "Create JIT-compiled form from any given form (compiled or not)."

        # Check form argument
        if isinstance(form, ufl.Form):
            # Cutoff for better error message than ffc would give
            if not form.empty() and not form.domains():
                raise RuntimeError("Expecting a completed form with domains at this point.")

            # Extract subdomain data
            sd = form.subdomain_data()
            if len(sd) != 1:
                # Until the assembler and ufc has multimesh support, we impose this limitation
                raise RuntimeError("Expecting subdomain data for a single domain only.")
            if subdomains is None:
                self.subdomains, = list(sd.values()) # Assuming single domain
            domain, = list(sd.keys()) # Assuming single domain
            mesh = domain.data()

            # Jit the module, this will take some time...
            self._compiled_form, module, prefix \
                                 = jit(form, form_compiler_parameters,
                                       mpi_comm=mesh.mpi_comm())

            # Extract function spaces of form arguments
            self.function_spaces = [func.function_space() for func in form.arguments()]

            # Extract coefficients from form (pass only the ones that the compiled form actually uses to the assembler)
            original_coefficients = form.coefficients()
            self.coefficients = [original_coefficients[self._compiled_form.original_coefficient_position(i)]
                                 for i in range(self._compiled_form.num_coefficients())]

        elif isinstance(form, cpp.Form):
            cpp.deprecation("Passing a cpp.Form to dolfin.Form constructor",
                            "1.3.0", "1.4.0",
                            "Passing a cpp.Form to dolfin.Form constructor"
                            " will be removed.")
            self._compiled_form = form._compiled_form

            self.function_spaces = form.function_spaces
            self.coefficients = form.coefficients
            self.subdomains = form.subdomains
            mesh = form.mesh()

        else:
            cpp.dolfin_error("form.py",
                             "creating dolfin.Form",
                             "Expected a ufl.Form or a dolfin.Form")

        # This is now a strict requirement
        if mesh is None:
            raise RuntimeError("Expecting to find a Mesh in the form.")

        # Type checking argument function_spaces
        r = self._compiled_form.rank()
        if len(self.function_spaces) != r:
            raise ValueError(function_space_error +
                  " Wrong number of test spaces (should be %d)." % r)
        if not all(isinstance(fs, cpp.FunctionSpace) for fs in self.function_spaces):
            raise ValueError(function_space_error +
                  " Invalid type of test spaces.")

        # Type checking coefficients
        if not all(isinstance(c, cpp.GenericFunction) for c in self.coefficients):
            coefficient_error = "Error while extracting coefficients. "
            raise TypeError(coefficient_error +
                            "Either provide a dict of cpp.GenericFunctions, " +
                            "or use Function to define your form.")

        # Type checking subdomain data
        # Check that we have no additional subdomain data (user misspelling)
        # TODO: Get from ufl list?
        integral_types = ("cell", "exterior_facet", "interior_facet", "point") 
        for k in list(self.subdomains.keys()):
            if self.subdomains[k] is None:
                del self.subdomains[k]
        additional_keys = set(self.subdomains.keys()) - set(integral_types)
        if additional_keys:
            cpp.warning("Invalid keys in subdomains: %s" % additional_keys)
        # Check that we have only MeshFunctions
        for data in list(self.subdomains.values()):
            # TODO: This wasn't here before, disable if it's too restrictive:
            if not (data is None or isinstance(data, MeshFunctionSizet)):
                cpp.warning("Invalid subdomain data type %s, expecting a MeshFunction" % type(data))

        # Initialize base class
        cpp.Form.__init__(self, self._compiled_form,
                          self.function_spaces, self.coefficients)

        # Attach subdomains if we have them
        subdomains = self.subdomains.get("cell")
        if subdomains is not None:
            self.set_cell_domains(subdomains)
        subdomains = self.subdomains.get("exterior_facet")
        if subdomains is not None:
            self.set_exterior_facet_domains(subdomains)
        subdomains = self.subdomains.get("interior_facet")
        if subdomains is not None:
            self.set_interior_facet_domains(subdomains)
        subdomains = self.subdomains.get("point")

        if subdomains is not None:
            self.set_vertex_domains(subdomains)

        # Attach mesh
        self.set_mesh(mesh)

def _extract_function_spaces_from_compiled_form(compiled_form, given_function_spaces):
    function_spaces = []
    r = compiled_form.rank()

    # Given function spaces are necessary for assembling pure UFC forms,
    # and possibly also convenient for other rare scenarios.
    if isinstance(given_function_spaces, cpp.FunctionSpace):
        if r > 0:
            # Repeat a single given space
            function_spaces = [given_function_spaces]*r
        else:
            raise ValueError(function_space_error +
                             "Superfluous test spaces given for a functional.")
    elif isinstance(given_function_spaces, (list,tuple)):
        if all(isinstance(fs, cpp.FunctionSpace) for fs in given_function_spaces):
            # Take given list of spaces
            function_spaces = list(given_function_spaces)
        else:
            raise ValueError(function_space_error +
                             "Invalid given test spaces.")
    else:
        raise ValueError(function_space_error +
                         "Expecting given_function_spaces to be one or more function spaces.")

    return function_spaces
