"""This module defines a jit function that wraps the jit function for
the chosen form compiler."""

# Copyright (C) 2008-2014 Johan Hake
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Anders Logg, 2008-2009.
# Modified by Garth N. Wells, 2011.
# Modified by Martin S. Alnes, 2014.

from __future__ import print_function

__all__ = ["jit"]

from functools import wraps

# Import SWIG-generated extension module (DOLFIN C++)
import dolfin.cpp as cpp
from dolfin.cpp import MPI, info, parameters, warning, info

import ufl
import six

def mpi_jit_decorator(local_jit, *args, **kwargs):
    """A decorator for jit compilation

    Use this function as a decorator to any jit compiler function.  In
    a parallel run, this function will first call the jit compilation
    function on the first process. When this is done, and the module
    is in the cache, it will call the jit compiler on the remaining
    processes, which will then use the cached module.

    *Example*
        .. code-block:: python

            def jit_something(something):
                ....

    """
    @wraps(local_jit)
    def mpi_jit(*args, **kwargs):

        # Create MPI_COMM_WORLD wrapper
        mpi_comm = kwargs.get("mpi_comm")
        if mpi_comm is None:
            mpi_comm = cpp.mpi_comm_world()

        # Just call JIT compiler when running in serial
        if MPI.size(mpi_comm) == 1:
            return local_jit(*args, **kwargs)

        # Compile first on process 0
        if MPI.rank(mpi_comm) == 0:
            output = local_jit(*args, **kwargs)

        # Wait for the first process to finish
        MPI.barrier(mpi_comm)

        # Then compile on all other processes (which may then just
        # read the cache)
        if not MPI.rank(mpi_comm) == 0:
            output = local_jit(*args,**kwargs)

        return output

    # Return the decorated jit function
    return mpi_jit

@mpi_jit_decorator
def jit(form, form_compiler_parameters=None, mpi_comm=None):
    """Just-in-time compile any provided form.

    It uses the jit function from the form compiler registered by
    parameters["form_compiler"]["name"].
    """

    # Check that form is not empty
    if isinstance(form, ufl.Form):
        if form.empty():
            raise RuntimeError("Form is empty. Cannot pass to JIT compiler.")

    # Import form compiler
    form_compiler_name = cpp.parameters["form_compiler"]["name"]
    try:
        form_compiler = __import__(form_compiler_name)
    except ImportError as message:
        print(message)
        warning("Could not import %s form compiler, falling back to FFC." \
                % form_compiler_name)
        try:
            form_compiler = __import__("ffc")
        except:
            cpp.dolfin_error("jit.py",
                             "perform just-in-time compilation of form",
                             "Could not import FFC form compiler")

    # Checks on form compiler interface
    if not (hasattr(form_compiler, 'default_parameters')
            and hasattr(form_compiler, 'jit')):
        raise RuntimeError("Form compiler must implement the default_parameters and jit functions.")

    # Prepare form compiler parameters
    p = form_compiler.default_parameters()

    # Set parameters from global DOLFIN parameter set
    for key, value in six.iteritems(parameters["form_compiler"]):
        p[key] = value

    # Override with local parameters if any
    if form_compiler_parameters:
        p.update(form_compiler_parameters)

    # Execute!
    return form_compiler.jit(form, parameters=p)
