
/* Compiler implementation of the D programming language
 * Copyright (C) 1999-2018 by The D Language Foundation, All Rights Reserved
 * written by Walter Bright
 * http://www.digitalmars.com
 * Distributed under the Boost Software License, Version 1.0.
 * http://www.boost.org/LICENSE_1_0.txt
 * https://github.com/dlang/dmd/blob/master/src/mars.h
 */

#ifndef DMD_MARS_H
#define DMD_MARS_H

#ifdef __DMC__
#pragma once
#endif

/*
It is very important to use version control macros correctly - the
idea is that host and target are independent. If these are done
correctly, cross compilers can be built.
The host compiler and host operating system are also different,
and are predefined by the host compiler. The ones used in
dmd are:

Macros defined by the compiler, not the code:

    Compiler:
        __DMC__             Digital Mars compiler
        _MSC_VER            Microsoft compiler
        __GNUC__            Gnu compiler
        __clang__           Clang compiler

    Host operating system:
        _WIN32              Microsoft NT, Windows 95, Windows 98, Win32s,
                            Windows 2000, Win XP, Vista
        _WIN64              Windows for AMD64
        __linux__           Linux
        __APPLE__           Mac OSX
        __FreeBSD__         FreeBSD
        __OpenBSD__         OpenBSD
        __DragonFly__       DragonFlyBSD
        __sun               Solaris, OpenSolaris, SunOS, OpenIndiana, etc

For the target systems, there are the target operating system and
the target object file format:

    Target operating system:
        TARGET_WINDOS       Covers 32 bit windows and 64 bit windows
        TARGET_LINUX        Covers 32 and 64 bit linux
        TARGET_OSX          Covers 32 and 64 bit Mac OSX
        TARGET_FREEBSD      Covers 32 and 64 bit FreeBSD
        TARGET_OPENBSD      Covers 32 and 64 bit OpenBSD
        TARGET_DRAGONFLYBSD Covers 64 bit DragonFlyBSD
        TARGET_SOLARIS      Covers 32 and 64 bit Solaris

    It is expected that the compiler for each platform will be able
    to generate 32 and 64 bit code from the same compiler binary.

    There are currently no macros for byte endianness order.
 */


#include <stdio.h>
#include <stdint.h>
#include <stdarg.h>

#ifdef __DMC__
#ifdef DEBUG
#undef assert
#define assert(e) (static_cast<void>((e) || (printf("assert %s(%d) %s\n", __FILE__, __LINE__, #e), halt())))
#endif
#endif

void unittests();

struct OutBuffer;

#include "globals.h"

#include "ctfloat.h"

#include "complex_t.h"

#include "errors.h"

class Dsymbol;
class Library;
struct File;
void obj_start(char *srcfile);
void obj_end(Library *library, File *objfile);
void obj_append(Dsymbol *s);
void obj_write_deferred(Library *library);

/// Utility functions used by both main and frontend.
void readFile(Loc loc, File *f);
void writeFile(Loc loc, File *f);
void ensurePathToNameExists(Loc loc, const char *name);

const char *importHint(const char *s);
/// Little helper function for writing out deps.
void escapePath(OutBuffer *buf, const char *fname);

#endif /* DMD_MARS_H */
