/*
 * Utility functions implemented using GStreamer
 *
 * Copyright (C) 2009 W. Michael Petullo <mike@flyn.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdlib.h>
#include <string.h>
#include <gst/gst.h>
#include <gst/pbutils/pbutils.h>

#include "util-gst.h"
#include "dmapd-daap-record.h"
#include "av-meta-reader-gst.h"

const GstClockTime DISCOVER_TIMEOUT = 5 * GST_SECOND;

struct AVMetaReaderGstPrivate {
	GMainLoop  *loop;
	GstDiscoverer *discoverer;
};

static void
av_meta_reader_gst_set_property (GObject *object,
				 guint prop_id,
				 const GValue *value,
				 GParamSpec *pspec)
{
	G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
}

static void
av_meta_reader_gst_get_property (GObject *object,
				 guint prop_id,
				 GValue *value,
				 GParamSpec *pspec)
{
	G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
}

static GOptionGroup *
av_meta_reader_gst_get_option_group (AVMetaReader *reader)
{
	static GOptionGroup *option_group = NULL;

	if (! gst_is_initialized ()) {
		option_group = gst_init_get_option_group ();
	}

	return option_group;
}

static void
av_meta_reader_gst_finalize (GObject *self)
{
	// G_OBJECT_CLASS (av_meta_reader_gst_parent_class)->finalize (self);
}

static void
av_meta_reader_gst_class_finalize (AVMetaReaderGstClass *klass)
{
}

static gchar *
determine_format (DAAPRecord *record, const gchar *description)
{
	gchar *format = NULL;

	if (g_strrstr (description, "MP3"))
		format = "mp3";
	else if (g_strrstr (description, "MPEG-4 AAC"))
		format = "aac";
	else if (g_strrstr (description, "Vorbis"))
		format = "ogg";
	else if (g_strrstr (description, "FLAC"))
		format = "flac";
	else {
		gchar *ext = NULL, *location = NULL;

		g_debug ("Failed to get type from stream, using filename");

		g_object_get (record, "location", &location, NULL);
		if (NULL == location) {
			g_debug ("Failed to get type from filename, guessing");
			format = "mp3";
			goto done;
		}

		ext = strrchr (location, '.');
		if (ext == NULL || 0x00 == *(ext + 1)) {
			g_debug ("Failed to get type from filename, guessing");
			format = "mp3";
			goto done;
		}

		format = ++ext;
	}

	g_assert (NULL != format);

	g_debug ("    Format is %s.", format);

done:
	return format;
}

static void
insert_tag (const GstTagList *list, const gchar *tag, DAAPRecord *record)
{
	gint i;

	g_assert (tag);

	for (i = 0; i < gst_tag_list_get_tag_size (list, tag); i++) {
		gchar *val;

		if (G_TYPE_STRING == gst_tag_get_type (tag)) {
			if (! gst_tag_list_get_string_index (list, tag, i, &val))
				g_assert_not_reached ();
		} else {
			val = g_strdup_value_contents (gst_tag_list_get_value_index (list, tag, i));
			if (NULL == val) {
				g_warning ("Failed to get value contents");
				goto done;
			}
		}

		g_debug ("    Tag %s is %s.", tag, val);

		if (! strcmp ("title", tag)) {
			g_object_set (record, "title", val, NULL);
		} else if (! strcmp ("artist", tag)) {
			g_object_set (record, "songartist", val, NULL);
		} else if (! strcmp ("album", tag)) {
			g_object_set (record, "songalbum", val, NULL);
		} else if (! strcmp ("album-disc-number", tag)) {
			errno = 0;
			long disc = strtol (val, NULL, 10);
			if (! errno) {
				g_object_set (record, "disc", disc, NULL);
			} else {
				g_warning ("Error parsing disc: %s", val);
			}
		} else if (! strcmp ("date", tag)) {
			// val should be "1985-01-01."
			if (strlen (val) < 4) {
				g_warning ("Error parsing date: %s", val);
			} else {
				val[4] = 0x00;
				errno = 0;
				long year = strtol (val, NULL, 10);
				if (! errno) {
					g_object_set (record, "year", year, NULL);
				} else {
					g_warning ("Error parsing year: %s", val);
				}
			}
		} else if (! strcmp ("genre", tag)) {
			g_object_set (record, "songgenre", val, NULL);
		} else if (! strcmp ("audio-codec", tag)) {
			gboolean has_video;
			g_object_get (record, "has-video", &has_video, NULL);
			g_debug ("    %s video.", has_video ? "Has" : "Does not have");
			if (has_video) {
				g_object_set (record, "mediakind", DMAP_MEDIA_KIND_MOVIE, NULL);
				/* FIXME: get from video stream. */
				gchar *ext, *location;

				g_object_get (record, "location", &location, NULL);
				ext = strrchr (location, '.');
				if (ext == NULL) {
					ext = "mov";
				} else {
					ext++;
				}
				g_object_set (record, "format", ext, NULL);
			} else {
				g_object_set (record, "mediakind", DMAP_MEDIA_KIND_MUSIC, NULL);
				gchar *format = determine_format (record, val);
				g_assert (format);
				g_object_set (record, "format", format, NULL);
			}
		} else if (! strcmp ("track-number", tag)) {
			errno = 0;
			long track = strtol (val, NULL, 10);
			if (! errno) {
				g_object_set (record, "track", track, NULL);
			} else {
				g_warning ("Error parsing track: %s", val);
			}
		} else {
			g_debug ("    Unused metadata %s.", tag);
		}
		g_free (val);
	}
done:
	return;
}

static void on_discovered_cb (GstDiscoverer *discoverer, GstDiscovererInfo *info, GError *error, DAAPRecord *record)
{
	const gchar *uri;
	GList *video_streams;
	const GstTagList *tags;
	GstClockTime duration = -1;
	GstDiscovererResult result;
	GstDiscovererStreamInfo *sinfo;

	uri = gst_discoverer_info_get_uri (info);
	g_assert (NULL != uri);

	result = gst_discoverer_info_get_result (info);
	if (GST_DISCOVERER_OK != result) {
		g_warning ("Could not read metadata from %s\n", uri);
		goto done;
	}

	tags = gst_discoverer_info_get_tags (info);
	if (NULL != tags) {
		gst_tag_list_foreach (tags, (GstTagForeachFunc) insert_tag, record);
	}

	duration = gst_discoverer_info_get_duration (info);
	g_object_set (record, "duration", (gint32) (duration / GST_SECOND), NULL);

	video_streams = gst_discoverer_info_get_video_streams (info);
	if (NULL != video_streams) {
		g_debug ("Has video component");
		g_object_set (record, "has-video", TRUE, NULL);
	}

done:
	if (NULL != video_streams) {
		 gst_discoverer_stream_info_list_free (video_streams);
	}

	return;
}

static void on_finished_cb (GstDiscoverer *discoverer, AVMetaReaderGstPrivate *priv)
{
	g_main_loop_quit (priv->loop);
}

static gboolean
av_meta_reader_gst_read (AVMetaReader *_reader, DAAPRecord *record, const gchar *path)
{
	AVMetaReaderGst *reader = AV_META_READER_GST (_reader);

	g_assert (NULL != reader);
	g_assert (NULL != reader->priv);
	g_assert (NULL != record);
	g_assert (NULL != path);

	gboolean fnval = FALSE;	
	GError *error = NULL;
	gchar *uri = NULL;

	uri = g_filename_to_uri (path, NULL, NULL);
	if (NULL == uri) {
		g_warning ("Error converting %s to URI", path);
		goto done;
	}

	reader->priv->discoverer = gst_discoverer_new (DISCOVER_TIMEOUT, &error);
	if (NULL != error) {
		g_warning ("Error creating discoverer instance: %s\n", error->message);
		goto done;
	}

	g_signal_connect (reader->priv->discoverer, "discovered", G_CALLBACK (on_discovered_cb), record);
	g_signal_connect (reader->priv->discoverer, "finished",   G_CALLBACK (on_finished_cb),   reader->priv);

	gst_discoverer_start (reader->priv->discoverer);

	if (! gst_discoverer_discover_uri_async (reader->priv->discoverer, uri)) {
		g_warning ("Failed to start discovering URI '%s'\n", uri);
		goto done;
	}

	g_main_loop_run (reader->priv->loop);

	gst_discoverer_stop (reader->priv->discoverer);

	fnval = TRUE;

done:
	if (NULL != uri) {
		g_free (uri);
	}

	if (NULL != error) {
		g_error_free (error);
	}

	if (reader->priv->discoverer) {
		g_object_unref (reader->priv->discoverer);
	}

	return fnval;
}

static void av_meta_reader_gst_register_type (GTypeModule *module);

G_MODULE_EXPORT gboolean
dmapd_module_load (GTypeModule *module)
{
	av_meta_reader_gst_register_type (module);
	return TRUE;
}

G_MODULE_EXPORT gboolean
dmapd_module_unload (GTypeModule *module)
{
	return TRUE;
}

static void av_meta_reader_gst_init (AVMetaReaderGst *reader)
{
	reader->priv = AV_META_READER_GST_GET_PRIVATE (reader);

	reader->priv->loop       = g_main_loop_new (NULL, FALSE);
	reader->priv->discoverer = NULL;
}

static void av_meta_reader_gst_class_init (AVMetaReaderGstClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
	AVMetaReaderClass *av_meta_reader_class = AV_META_READER_CLASS (klass);

	g_type_class_add_private (klass, sizeof (AVMetaReaderGstPrivate));

	gobject_class->set_property = av_meta_reader_gst_set_property;
	gobject_class->get_property = av_meta_reader_gst_get_property;
	gobject_class->finalize = av_meta_reader_gst_finalize;

	av_meta_reader_class->read = av_meta_reader_gst_read;
	av_meta_reader_class->get_option_group = av_meta_reader_gst_get_option_group;
}

G_DEFINE_DYNAMIC_TYPE (AVMetaReaderGst,
		       av_meta_reader_gst,
		       TYPE_AV_META_READER)
