/*
Sakila for DB2 is a port of the Sakila example database available for MySQL, which was originally developed by Mike Hillyer of the MySQL AB documentation team.
This project is designed to help database administrators to decide which database to use for development of new products
The user can run the same SQL against different kind of databases and compare the performance

License: BSD
Copyright DB Software Laboratory and Data Geekery GmbH (DB2 port)
http://www.etl-tools.com / http://www.datageekery.com


*/

CREATE TABLE actor (
  actor_id INTEGER NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  first_name VARCHAR(45) NOT NULL,
  last_name VARCHAR(45) NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_actor PRIMARY KEY (actor_id)
);

CREATE INDEX idx_actor_last_name ON actor(last_name);

CREATE OR REPLACE TRIGGER actor_before_update
BEFORE UPDATE ON actor
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE country (
  country_id SMALLINT NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  country VARCHAR(50) NOT NULL,
  last_update DATE,
  CONSTRAINT pk_country PRIMARY KEY (country_id)
);

CREATE OR REPLACE TRIGGER country_before_update
BEFORE UPDATE ON country
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE city (
  city_id INTEGER NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  city VARCHAR(50) NOT NULL,
  country_id SMALLINT NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_city PRIMARY KEY (city_id),
  CONSTRAINT fk_city_country FOREIGN KEY (country_id) REFERENCES country (country_id)
);

CREATE OR REPLACE TRIGGER city_before_update
BEFORE UPDATE ON city
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE address (
  address_id INTEGER NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  address VARCHAR(50) NOT NULL,
  address2 VARCHAR(50) DEFAULT NULL,
  district VARCHAR(20) NOT NULL,
  city_id INT NOT NULL,
  postal_code VARCHAR(10) DEFAULT NULL,
  phone VARCHAR(20) NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_address PRIMARY KEY (address_id),
  CONSTRAINT fk_address_city FOREIGN KEY (city_id) REFERENCES city (city_id)
);

CREATE INDEX idx_fk_city_id ON address(city_id);

CREATE OR REPLACE TRIGGER address_before_update
BEFORE UPDATE ON address
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE language (
  language_id SMALLINT NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  name CHAR(20) NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_language PRIMARY KEY (language_id)
);

CREATE OR REPLACE TRIGGER language_before_update
BEFORE UPDATE ON language
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE category (
  category_id SMALLINT NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  name VARCHAR(25) NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_category PRIMARY KEY (category_id)
);

CREATE OR REPLACE TRIGGER category_before_update
BEFORE UPDATE ON category
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE customer (
  customer_id INT NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  store_id INT NOT NULL,
  first_name VARCHAR(45) NOT NULL,
  last_name VARCHAR(45) NOT NULL,
  email VARCHAR(50) DEFAULT NULL,
  address_id INT NOT NULL,
  active CHAR(1) DEFAULT 'Y' NOT NULL,
  create_date DATE NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_customer PRIMARY KEY (customer_id),
  CONSTRAINT fk_customer_address FOREIGN KEY (address_id) REFERENCES address(address_id)
);

CREATE INDEX idx_customer_fk_store_id ON customer(store_id);
CREATE INDEX idx_customer_fk_address_id ON customer(address_id);
CREATE INDEX idx_customer_last_name ON customer(last_name);

CREATE OR REPLACE TRIGGER customer_before_update
BEFORE UPDATE ON customer
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE film (
  film_id int NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  title VARCHAR(255) NOT NULL,
  description CLOB DEFAULT NULL,
  release_year VARCHAR(4) DEFAULT NULL,
  language_id SMALLINT NOT NULL,
  original_language_id SMALLINT DEFAULT NULL,
  rental_duration SMALLINT  DEFAULT 3 NOT NULL,
  rental_rate DECIMAL(4,2) DEFAULT 4.99 NOT NULL,
  length SMALLINT DEFAULT NULL,
  replacement_cost DECIMAL(5,2) DEFAULT 19.99 NOT NULL,
  rating VARCHAR(10) DEFAULT 'G',
  special_features VARCHAR(100) DEFAULT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_film PRIMARY KEY (film_id),
  CONSTRAINT fk_film_language FOREIGN KEY (language_id) REFERENCES language (language_id) ,
  CONSTRAINT fk_film_language_original FOREIGN KEY (original_language_id) REFERENCES language (language_id)
);

ALTER TABLE film ADD CONSTRAINT CHECK_special_features CHECK (
  special_features IS NULL OR
  special_features LIKE '%Trailers%' OR
  special_features LIKE '%Commentaries%' OR
  special_features LIKE '%Deleted Scenes%' OR
  special_features LIKE '%Behind the Scenes%'
);

ALTER TABLE film ADD CONSTRAINT CHECK_special_rating CHECK (rating IN ('G','PG','PG-13','R','NC-17'));

CREATE INDEX idx_fk_language_id ON film (language_id);
CREATE INDEX idx_fk_original_language_id ON film (original_language_id);

CREATE OR REPLACE TRIGGER film_before_update
BEFORE UPDATE ON film
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE film_actor (
  actor_id INT NOT NULL,
  film_id INT NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_film_actor PRIMARY KEY (actor_id,film_id),
  CONSTRAINT fk_film_actor_actor FOREIGN KEY (actor_id) REFERENCES actor (actor_id),
  CONSTRAINT fk_film_actor_film FOREIGN KEY (film_id) REFERENCES film (film_id)
);

CREATE INDEX idx_fk_film_actor_film ON film_actor (film_id);
CREATE INDEX idx_fk_film_actor_actor ON film_actor (actor_id);

CREATE OR REPLACE TRIGGER film_actor_before_update
BEFORE UPDATE ON film_actor
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE film_category (
  film_id INT NOT NULL,
  category_id SMALLINT NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_film_category PRIMARY KEY (film_id, category_id),
  CONSTRAINT fk_film_category_film FOREIGN KEY (film_id) REFERENCES film (film_id),
  CONSTRAINT fk_film_category_category FOREIGN KEY (category_id) REFERENCES category (category_id)
);

CREATE INDEX idx_fk_film_category_film ON film_category(film_id);
CREATE INDEX idx_fk_film_category_category ON film_category(category_id);

CREATE OR REPLACE TRIGGER film_category_before_update
BEFORE UPDATE ON film_category
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE film_text (
  film_id SMALLINT NOT NULL,
  title VARCHAR(255) NOT NULL,
  description CLOB,
  CONSTRAINT pk_film_text PRIMARY KEY (film_id)
);

CREATE TABLE inventory (
  inventory_id INT NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  film_id INT NOT NULL,
  store_id INT NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_inventory PRIMARY KEY (inventory_id),
  CONSTRAINT fk_inventory_film FOREIGN KEY (film_id) REFERENCES film (film_id)
);

CREATE INDEX idx_fk_film_id ON inventory(film_id);
CREATE INDEX idx_fk_film_id_store_id ON inventory(store_id,film_id);

CREATE OR REPLACE TRIGGER inventory_before_update
BEFORE UPDATE ON inventory
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE staff (
  staff_id SMALLINT NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  first_name VARCHAR(45) NOT NULL,
  last_name VARCHAR(45) NOT NULL,
  address_id INT NOT NULL,
  picture BLOB DEFAULT NULL,
  email VARCHAR(50) DEFAULT NULL,
  store_id INT NOT NULL,
  active SMALLINT DEFAULT 1 NOT NULL,
  username VARCHAR(16) NOT NULL,
  password VARCHAR(40) DEFAULT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_staff PRIMARY KEY (staff_id),
  CONSTRAINT fk_staff_address FOREIGN KEY (address_id) REFERENCES address (address_id)
);

CREATE INDEX idx_fk_staff_store_id ON staff(store_id);
CREATE INDEX idx_fk_staff_address_id ON staff(address_id);

CREATE OR REPLACE TRIGGER staff_before_update
BEFORE UPDATE ON staff
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE store (
  store_id INT NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  manager_staff_id SMALLINT NOT NULL,
  address_id INT NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_store PRIMARY KEY (store_id),
  CONSTRAINT fk_store_staff FOREIGN KEY (manager_staff_id) REFERENCES staff (staff_id),
  CONSTRAINT fk_store_address FOREIGN KEY (address_id) REFERENCES address (address_id)
);

CREATE INDEX idx_store_fk_manager_staff_id ON store(manager_staff_id);
CREATE INDEX idx_fk_store_address ON store(address_id);

CREATE OR REPLACE TRIGGER store_before_update
BEFORE UPDATE ON store
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE payment (
  payment_id int NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  customer_id INT NOT NULL,
  staff_id SMALLINT NOT NULL,
  rental_id INT DEFAULT NULL,
  amount DECIMAL(5,2) NOT NULL,
  payment_date DATE NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_payment PRIMARY KEY (payment_id),
  CONSTRAINT fk_payment_customer FOREIGN KEY (customer_id) REFERENCES customer (customer_id) ,
  CONSTRAINT fk_payment_staff FOREIGN KEY (staff_id) REFERENCES staff (staff_id)
);

CREATE INDEX idx_fk_staff_id ON payment(staff_id);
CREATE INDEX idx_fk_customer_id ON payment(customer_id);

CREATE OR REPLACE TRIGGER payment_before_update
BEFORE UPDATE ON payment
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

CREATE TABLE rental (
  rental_id INT NOT NULL GENERATED BY DEFAULT AS IDENTITY,
  rental_date DATE NOT NULL,
  inventory_id INT NOT NULL,
  customer_id INT NOT NULL,
  return_date DATE DEFAULT NULL,
  staff_id SMALLINT NOT NULL,
  last_update DATE NOT NULL DEFAULT CURRENT_DATE,
  CONSTRAINT pk_rental PRIMARY KEY (rental_id),
  CONSTRAINT fk_rental_staff FOREIGN KEY (staff_id) REFERENCES staff (staff_id) ,
  CONSTRAINT fk_rental_inventory FOREIGN KEY (inventory_id) REFERENCES inventory (inventory_id) ,
  CONSTRAINT fk_rental_customer FOREIGN KEY (customer_id) REFERENCES customer (customer_id)
);

CREATE INDEX idx_rental_fk_inventory_id ON rental(inventory_id);
CREATE INDEX idx_rental_fk_customer_id ON rental(customer_id);
CREATE INDEX idx_rental_fk_staff_id ON rental(staff_id);
CREATE UNIQUE INDEX   idx_rental_uq  ON rental (rental_date,inventory_id,customer_id);

CREATE OR REPLACE TRIGGER rental_before_update
BEFORE UPDATE ON rental
REFERENCING NEW AS new
FOR EACH ROW
BEGIN
  SET new.last_update = current_date;
END
/

ALTER TABLE customer ADD CONSTRAINT fk_customer_store FOREIGN KEY (store_id) REFERENCES store (store_id);
ALTER TABLE inventory ADD CONSTRAINT fk_inventory_store FOREIGN KEY (store_id) REFERENCES store (store_id);
ALTER TABLE staff ADD CONSTRAINT fk_staff_store FOREIGN KEY (store_id) REFERENCES store (store_id);
ALTER TABLE payment ADD CONSTRAINT fk_payment_rental FOREIGN KEY (rental_id) REFERENCES rental (rental_id) ON DELETE SET NULL;

CREATE OR REPLACE VIEW customer_list
AS
SELECT cu.customer_id AS ID,
       cu.first_name||' '||cu.last_name AS name,
       a.address AS address,
       a.postal_code AS zip_code,
       a.phone AS phone,
       city.city AS city,
       country.country AS country,
       decode(cu.active, 1,'active','') AS notes,
       cu.store_id AS SID
FROM customer cu JOIN address a ON cu.address_id = a.address_id JOIN city ON a.city_id = city.city_id
    JOIN country ON city.country_id = country.country_id;

CREATE OR REPLACE VIEW film_list
AS
SELECT film.film_id AS FID,
       film.title AS title,
       film.description AS description,
       category.name AS category,
       film.rental_rate AS price,
       film.length AS length,
       film.rating AS rating,
       actor.first_name||' '||actor.last_name AS actors
FROM category LEFT JOIN film_category ON category.category_id = film_category.category_id LEFT JOIN film ON film_category.film_id = film.film_id
        JOIN film_actor ON film.film_id = film_actor.film_id
    JOIN actor ON film_actor.actor_id = actor.actor_id;

CREATE OR REPLACE VIEW staff_list
AS
SELECT s.staff_id AS ID,
       s.first_name||' '||s.last_name AS name,
       a.address AS address,
       a.postal_code AS zip_code,
       a.phone AS phone,
       city.city AS city,
       country.country AS country,
       s.store_id AS SID
FROM staff s JOIN address a ON s.address_id = a.address_id JOIN city ON a.city_id = city.city_id
    JOIN country ON city.country_id = country.country_id;

CREATE OR REPLACE VIEW sales_by_store
AS
SELECT
  s.store_id
 ,c.city||','||cy.country AS store
 ,m.first_name||' '||m.last_name AS manager
 ,SUM(p.amount) AS total_sales
FROM payment p
INNER JOIN rental r ON p.rental_id = r.rental_id
INNER JOIN inventory i ON r.inventory_id = i.inventory_id
INNER JOIN store s ON i.store_id = s.store_id
INNER JOIN address a ON s.address_id = a.address_id
INNER JOIN city c ON a.city_id = c.city_id
INNER JOIN country cy ON c.country_id = cy.country_id
INNER JOIN staff m ON s.manager_staff_id = m.staff_id
GROUP BY
  s.store_id
, c.city||','||cy.country
, m.first_name||' '||m.last_name;

CREATE OR REPLACE VIEW sales_by_film_category
AS
SELECT
c.name AS category
, SUM(p.amount) AS total_sales
FROM payment p
INNER JOIN rental r ON p.rental_id = r.rental_id
INNER JOIN inventory i ON r.inventory_id = i.inventory_id
INNER JOIN film f ON i.film_id = f.film_id
INNER JOIN film_category fc ON f.film_id = fc.film_id
INNER JOIN category c ON fc.category_id = c.category_id
GROUP BY c.name;