//
//  docx.cc
//
//  Copyright (c) 2022 Yuji Hirose. All rights reserved.
//  MIT License
//

#include <cstdlib>
#include <iostream>
#include <peglib.h>

using namespace peg;

int main(void) {
  parser parser(R"(ROOT       <- DECL CONTENT
DECL       <- '<?' (!('?>') .)+ '?>'
CONTENT    <- (ELEMENT / TEXT)*
ELEMENT    <- $(STAG CONTENT ETAG) / ATAG
STAG       <- '<' $tag< TAG_NAME > (' ' ATTRIBUTES)?'>'
ETAG       <- '</' $tag '>'
ATAG       <- '<' TAG_NAME (' ' ATTRIBUTES)? '/>'
TAG_NAME   <- < [a-zA-Z:]+ >
ATTRIBUTES <- (!('>' / '/>') .)+
TEXT       <- < TEXT_DATA+ >
TEXT_DATA  <- ![<] .
  )");

  std::vector<std::string> stack;
  stack.push_back("[ROOT]");

  parser["STAG"] = [&](const SemanticValues &vs) {
    auto tag = std::any_cast<std::string>(vs[0]);
    if (tag == "w:pPr" || tag == "w:t") {
      std::cout << "<" << tag << ">" << std::endl;
      stack.push_back(tag);
    }
  };

  parser["ETAG"] = [&](const SemanticValues & /*vs*/) {
    auto tag = stack.back();
    if (tag == "w:pPr" || tag == "w:t") {
      std::cout << "</" << tag << ">" << std::endl;
      stack.pop_back();
    }
  };

  parser["TAG_NAME"] = [&](const SemanticValues &vs) {
    return vs.token_to_string();
  };

  parser["TEXT"] = [&](const SemanticValues &vs) {
    auto tag = stack.back();
    if (tag == "w:t") { std::cout << vs.sv(); }
  };

  const auto source = R"(<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
<w:document xmlns:wpc="http://schemas.microsoft.com/office/word/2010/wordprocessingCanvas" xmlns:cx="http://schemas.microsoft.com/office/drawing/2014/chartex" xmlns:cx1="http://schemas.microsoft.com/office/drawing/2015/9/8/chartex" xmlns:cx2="http://schemas.microsoft.com/office/drawing/2015/10/21/chartex" xmlns:cx3="http://schemas.microsoft.com/office/drawing/2016/5/9/chartex" xmlns:cx4="http://schemas.microsoft.com/office/drawing/2016/5/10/chartex" xmlns:cx5="http://schemas.microsoft.com/office/drawing/2016/5/11/chartex" xmlns:cx6="http://schemas.microsoft.com/office/drawing/2016/5/12/chartex" xmlns:cx7="http://schemas.microsoft.com/office/drawing/2016/5/13/chartex" xmlns:cx8="http://schemas.microsoft.com/office/drawing/2016/5/14/chartex" xmlns:mc="http://schemas.openxmlformats.org/markup-compatibility/2006" xmlns:aink="http://schemas.microsoft.com/office/drawing/2016/ink" xmlns:am3d="http://schemas.microsoft.com/office/drawing/2017/model3d" xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships" xmlns:m="http://schemas.openxmlformats.org/officeDocument/2006/math" xmlns:v="urn:schemas-microsoft-com:vml" xmlns:wp14="http://schemas.microsoft.com/office/word/2010/wordprocessingDrawing" xmlns:wp="http://schemas.openxmlformats.org/drawingml/2006/wordprocessingDrawing" xmlns:w10="urn:schemas-microsoft-com:office:word" xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main" xmlns:w14="http://schemas.microsoft.com/office/word/2010/wordml" xmlns:w15="http://schemas.microsoft.com/office/word/2012/wordml" xmlns:w16cid="http://schemas.microsoft.com/office/word/2016/wordml/cid" xmlns:w16se="http://schemas.microsoft.com/office/word/2015/wordml/symex" xmlns:wpg="http://schemas.microsoft.com/office/word/2010/wordprocessingGroup" xmlns:wpi="http://schemas.microsoft.com/office/word/2010/wordprocessingInk" xmlns:wne="http://schemas.microsoft.com/office/word/2006/wordml" xmlns:wps="http://schemas.microsoft.com/office/word/2010/wordprocessingShape" mc:Ignorable="w14 w15 w16se w16cid wp14"><w:body><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>研究記事</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>３６</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>「</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>魚ではなく人を集める漁師」になりますか</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:i/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>「恐れることはありません。今後，あなたは人を生きたまま捕るのです」。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>ルカ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>５：１０</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>７３番の歌</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>私たちに大胆さを与えてください</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>何を学ぶか</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>イエスは自分の</w:t></w:r><w:bookmarkStart w:id="0" w:name="_GoBack"/><w:bookmarkEnd w:id="0"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>弟子になるよう，謙遜で働き者の漁師たちを招きました。イエスは現在でも，謙遜でよく働く人たちに，</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>「</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>魚ではなく人を集める漁師」になるよう勧めています。この記事は，聖書を学んではいても，イエスの勧めの通りに行動するのをためらっている人が何をするとよいかを説明しています。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>ペテロ，アンデレ，ヤコブ，ヨハネは漁師でした。イエスから，「私に付いてきなさい。魚ではなく人を集める漁師</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>にしてあげましょう」と言われた時，驚いたことでしょう。どうしましたか。「直ちに網を捨てて後に従った」と聖書は述べています。（マタ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>４：１８</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２２</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）その決定により，彼らの生活は大きく変わりました。魚ではなく「人を生きたまま捕る」ことになったのです。（ルカ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>５：１０</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）現在イエスは，真理を愛する誠実な人たちに同じように呼び掛けています。（マタ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２８：１９，２０</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）あなたはその呼び掛けに応じて，人を集める漁師になりましたか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>イエスは４人の漁師に何と言いましたか。４人はどうしましたか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>語句の説明：「</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>魚ではなく人を集める漁師」</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>とは，良い知らせを伝え，イエスの弟子になるよう他の人を教える人全てのことです。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>２</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>あなたは聖書を学んでよく進歩し，</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>伝道者になることを考えているかもしれません。なかなかその一歩を踏み出せないとしても，自分は駄目だとは考えないでください。慎重なのは，この決定の大切さを理解している証拠なのかもしれません。聖書には，ペテロと仲間たちが網を「直ちに」捨てた，と書かれています。でもペテロとアンデレはその場の勢いで行動したのではありません。イエスと出会ってメシアと認めたのは，それより６カ月以上前のことでした。（ヨハ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１：３５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>４２</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）あなたもエホバとイエスについてすでに多くのことを学び，これからもエホバと親しくなりたいと思っていることでしょう。それでも，伝道者になることを決める前に，よく考える必要があります。ペテロたちはなぜそのような決定ができましたか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>２．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>人を集める漁師になることを，時間をかけて決定する必要があるのはなぜですか。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>そのステップを踏む上でどんなことが助けになりますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>３</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>イエス</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>の最初の弟子になった４人の漁師たちは，意欲を持ち，漁の仕方に詳しく，勇気があり，自分を甘やかさない人でした。こうした要素は，人を</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>集める漁師になる上でも役立ったに違いありません。私たちがイエスの弟子として働く上でも役立ちます。この記事では，どうすればこうした点で成長できるかを考えます。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>３．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>何があると，伝道者になりたいという願いが強まりますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>意欲を高める</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>４</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>ペテロは家族を養うために魚を集めました。でも漁は単なる生計の手段ではありませんでした。漁師の仕事に愛着を抱いていたようです。（ヨハ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２１：３，９</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）人を集める漁師としての活動にも愛着を持ちました。エホバの助けにより，その活動に熟達しました。（使徒</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２：１４，４１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>４．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>ペテロは漁師の仕事をどう思っていましたか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>私たちが伝道するのは，エホバを愛しているからです。それが伝道する何よりの理由です。エホバへの愛があれば，自分には無理だという気持ちを克服できます。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>イエスは，人を集める漁師になるようペテロに勧めた時，「恐れることはありません」とも言いました。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（ルカ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>５：８</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>を読む。）</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>ペテロは，イエスの弟子になったらどうなるのかと考えて萎縮したのではありません。イエスの奇跡によって魚が大量に捕れたことに圧倒され，イエスと共に働く資格などないと感じたのです。あなたの場合はどうですか。イエス</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:lastRenderedPageBreak/><w:t>の弟子になることに何が伴うかを知って，気後れしていますか。もしそうであれば，エホバとイエスと隣人への愛を強めましょう。そうすれば，イエスの呼び掛けに応えて人を集める漁師になりたいと思うでしょう。（マタ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２２：３７，３９</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>。ヨハ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１４：１５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>５．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>ルカ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>章</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>８</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>節によれば，ペテロが恐れたのはなぜですか。私たちはどうすれば恐れの気持ちを克服できますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>６</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>ほかにもどんな理由で私たちは伝道しますか。「</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>行って，</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>……</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>人々を弟子としなさい」というイエスの命令に従いたいと思っています。（マタ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２８：１９，２０</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）さらに，人々が「痛めつけられ，放り出され」，神の王国についての真理を切実に必要としているからです。（マタ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>９：３６</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）エホバはあらゆる人が真理の正確な知識を得て救われることを望んでいます。（テモ一</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２：４</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>６．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>私たちはほかにもどんな理由で伝道しますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>７</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>伝道によって何が成し遂げられるかを考えるなら，命を救う活動に参加するよう動かされます。漁師が魚を集めるのは売るか食べるかするためですが，私たちが人々を集めるのはその人たちの命を救うためです。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（ローマ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１０：１３</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>を読む。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>テモ一</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>４：１６</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>７．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>ローマ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１０</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>章</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１３</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>節によれば，伝道が大切なのはなぜですか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>伝道の仕方を学ぶ</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>８</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>イエスの時代，</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>イスラエルの漁師は集めて良いのはどんな魚かを知っていなければなりませんでした。（レビ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１１：９</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１２</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）魚がどこにいるかも知る必要がありました。魚は，水質が合い，餌がたくさんある所に生息します。いつ漁に出るかも重要です。太平洋のある島で奉仕する宣教者は，そのことを知りました。地元の兄弟から，一緒に漁に行くよう誘われました。それで，「明日の朝９時に行きますね」と言いました。すると地元の兄弟から，「兄弟は漁のことが分かっていないね。自分に都合のいい時間ではなく，魚が捕れる時間に行かないと」と言われました。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>９</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>世紀</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>の，人を集める漁師も，</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>“</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>魚</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>”</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>がいる場所に，会えそうな時間に行きました。例えば</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>イエスの弟子たちは，神殿や会堂や広場で，また</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="000000"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>家から家に行って伝道しました。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（使徒</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>５：４２；１７：１７；１８：４</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）私たちも，区域の人々の生活スタイルに通じておきましょう。人に合わせ，人がいそうな場所で，会えそうな時間に伝道する必要があります。（コリ一</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>９：１９</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２３</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>８</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>９．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>漁師はどんなことを知っていなければなりませんか。なぜですか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１０</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>漁師は，どんな道具が漁に必要で，どう使うかを知っていなければなりません。私たちも，伝道に必要な道具を持っていなければなりません。使い方も知らなければなりません。イエスは，人を集めるための明確な指示を弟子たちに与えました。何を持っていき，どこで伝道し，何と言うかを教えました。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（マタ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１０：５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>７</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>。ルカ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１０：１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）現在，エホバの組織は，使いやすい道具を収めた宣教ツールボックスを用意しています。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>各種の道具の使い方も教えています。使い方の訓練を受けた私たちは，自信を持って上手に使えるようになっています。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（テモ二</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２：１５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１０．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>エホバの組織はどんな道具を用意していますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>「ものみの塔」２０１８年１０月号１１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１６ページの「真理を教えましょう」という記事を参照。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>勇気を持つ</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>漁師には勇気が必要です。天候が急に変わって海が荒れることがあるからです。夜の暗い中で働かなければならないこともあります。同じように，人を集める漁師にも勇気が求められます。伝道に出始めてエホバの証人であることを明らかにすると，家族に反対され，友達に冷やかされ，伝えようとしても断られるかもしれません。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>“</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>嵐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>”</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>の中にいるように感じるでしょう。でも，それは意外なことではありません。イエスは，弟子たちが敵対する人たちの中で伝道することになる，と予告しました。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（マタ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１０：１６</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１１．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>人を集める漁師に勇気が要るのはなぜですか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１２</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>どうすれば勇気を持てますか。まず，イエスが今もこの活動を天から導いていることを意識してください。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（ヨハ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１６：３３</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>。啓</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１４：１４</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１６</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）さらに，あなたを必ず助けるというエホバの約束に対する信仰を強めてください。（マタ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>６：３２</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:lastRenderedPageBreak/><w:t>３４</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）信仰が強まれば，勇気も強まります。ペテロ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>と仲間たちは</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="000000"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>イエスの弟子になるためにそれまでの仕事を辞めて，強い信仰を示しました。あなたも，聖書レッスンを始めて集会に出席し始めたことを友達や家族に伝えた時，強い信仰を示したのです。エホバの基準に合わせるために，行動や生活を大きく変えてきたことでしょう。それにも信仰や勇気が要ります。これからも勇気を示していくなら，「</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>あなたがどこに行っても，あなたの神エホバが共にいる」ということを実感できるでしょう。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（ヨシュア</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１：７</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>９</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>を読む。）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１２．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>ヨシュア</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>章</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>７</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>９</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>節によれば，どうすれば勇気を持てますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１３</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>勇気を持つため，ほかにも何ができますか。勇気と大胆さを与えてください，と祈りましょう。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（使徒</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>４：２９，３１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）エホバは祈りに答え，必ずあなたのそばにいて，支え続けてくださいます。エホバが過去にご自分に仕える人をどのように助けたかを思い巡らしましょう。さらに，あなたが問題を乗り越え，生活を改めるための力をエホバが与えてくださったということも考えてください。エホバはご自分の民が紅海を渡れるよう助けたのですから，あなたがイエスの弟子になるようにも助けてくださいます。（出</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１４：１３</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）次のように述べた詩編作者と同じ確信を持ちましょう。「エホバは私の側にいてくださる。私は恐れない。人が私に何を行えるだろう」。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>詩</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１１８：６</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１３．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>エホバの助けを思い巡らし，祈ると勇気を持てます。なぜそういえますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１４</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>勇気を持つのに，もともと内気だった人が大胆に語れるようエホバがどのように助けたかを知ることも役立ちます。真＜まさ＞枝＜え＞姉妹の経験を考えてみてください。姉妹は控えめな人で，信じていることを人に話すなんてとてもできない，と思っていました。知らない人に話すことだけでも，自分では登れない壁のように感じました。それで姉妹は，エホバと隣人に対する愛を深めることを特に意識しました。伝道が急務であることを考え，伝道したいという意欲を抱けるよう助けてください，と祈りました。恐れを克服し，</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>正規開拓奉仕を行えるまでになりました。エホバは伝道を始めたばかりの人が「勇気を出」せるようにもしてくださいます。知＜とも＞余＜よ＞姉妹の経験を取り上げましょう。初めて野外奉仕に出た時，戸口に出てきた女性に「エホバの証人とは関わりたくない」と怒鳴られ，追い返されてしまいました。でも，気を落とすどころか，一緒にいた姉妹に，「うれしいわ。何も言わないのに，あの人は私がエホバの証人だと思ってくれた」と言いました。知余は現在，正規開拓奉仕をしています。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１４．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>真枝と</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>知余の経験からどんなことを学べますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>自分を甘やかさない</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>漁師として生計を立てている人は自分を甘やかしません。自分を甘やかさない人は，すべき事をしっかり果たします。漁師は朝早く起き，仕事を最後まで果たし，天候が崩れても漁を続けます。同じように，私たちが伝道をしっかり続け，最後まで果たすには，自分を甘やかさない態度が必要です。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（マタ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１０：２２</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１５．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>自分を甘やかさない態度がクリスチャンに必要なのはなぜですか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１６</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>私たちには皆，自分に楽な道を選ぶ傾向があります。自分の力だけでは，その傾向を克服し，自分にとって難しい事柄を最後まで行うことはできません。自分を甘やかさないためには自制が必要です。エホバが聖なる力によって助けてくださいます。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（ガラ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>５：２２，２３</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１６．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>どうすれば難しく感じる事柄をきちんと行えますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１７</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>パウロは自分を甘やかすことはしませんでした。正しいことを行うのに「自分の体を打ちたた」</w:t></w:r><w:proofErr w:type="gramStart"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>か</w:t></w:r><w:proofErr w:type="gramEnd"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>なければならないことを認めていました。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（コリント第一</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>９：２５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２７</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>を読む。）</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>パウロは他の人に，自分を甘やかさず，「</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP"/></w:rPr><w:t>全てのことを適正に，取り決めに沿って行</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>いましょう</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP"/></w:rPr><w:t>」と勧めました。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（コリ一</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１４：４０</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）私たちも自分を甘やかさず，良い知らせを定期的に伝えることも含め，エホバの崇拝をきちんと続けましょう。（使徒</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２：４６</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１７．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>コリント第一</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>９</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>章</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>２７</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>節でパウロは，自分を甘やかさないためにどんな努力をしていると言っていますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>遅らせない</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:lastRenderedPageBreak/><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１８</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>漁の場合，成功の尺度となるのは集めた魚の量です。私たちの場合，エホバの証人になるよう何人の人を助けたかによって成功が計られることはありません。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（ルカ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>８：１１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１５</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）良い知らせを伝え，人々を教えることを続ける限り，エホバは私たちが成功していると見てくださいます。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>なぜですか。エホバとイエスに従っていることになるからです。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（マル</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１３：１０</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>。使徒</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>５：２８，２９</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１８．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>エホバはどんなことを成功と見てくださいますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１９</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>国によっては，漁が許可されているのは１年のうち数カ月の間だけです。そのような場合，漁を行える時期の終わりが近づくと，漁師はますます漁に集中します。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>人を集める漁師である私たちにも，今一層伝道に力を入れる理由があります。今の体制の終わりが非常に近いからです。命を救う活動を行う期間は限られています。この重要な活動に参加するのを遅らせたり，状況が整うのを待った方がよいと考えたりしないでください。（伝</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>１１：４</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:vertAlign w:val="superscript"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>２０</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>意欲を高め，聖書について伝える方法を学び，勇気を持ち，自分を甘やかさない態度を持つため，今行動しましょう。人を集める漁師は，今８００万人以上います。その人たちに加わってください。そうすれば，エホバからの喜びを経験できるでしょう。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>（ネヘ</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>８：１０</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>）伝道を十分に行い，エホバが良いとご覧になるまで行い続けることを決意しましょう。次の記事では，人を集める漁師として，王国について伝道し続ける決意を強めるための３つの方法を学びます。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１９</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>２０．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:color w:val="050505"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>今伝道に力を入れるどんな理由がありますか。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>どのようにできますか</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>意欲を高める</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>勇気を持つ</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>自分を甘やかさない</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t xml:space="preserve"> </w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>６６番の歌</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t xml:space="preserve"> </w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>良い知らせを伝える</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>ペテロと仲間たちは</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>人を集める漁師になった。この大切な活動は現在も行われている。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>（４</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>５節を参照。）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>漁に打ち込む人は・・・</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:val="ja-JP" w:eastAsia="ja-JP" w:bidi="ar-SA"/></w:rPr><w:t>魚が捕れる時間に，捕れる場所で働く。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>（８</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>９節を参照。）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>２．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>漁に必要な道具の使い方を知っている。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>（１０節を参照。）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:bCs/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>３．</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:b/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>海が荒れても勇気を持って働く。</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>（１１</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsia="MS Mincho" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:u w:val="dotted"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１２節を参照。）</w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>image.cnt01:label</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsiaTheme="minorEastAsia" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１．ペテロと３人の弟子たちが，魚でいっぱいになった網を一生懸命に引き上げている。２．２人の姉妹が楽しそうに話しながら伝道している。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>image.cnt02:label</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsiaTheme="minorEastAsia" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>漁に打ち込んでいる人を描いた３つの場面。８</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="Times New Roman" w:eastAsiaTheme="minorEastAsia" w:hAnsi="Times New Roman" w:cs="Times New Roman"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>‐</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsiaTheme="minorEastAsia" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>１２節で取り上げられている。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>image.cnt03:label</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsiaTheme="minorEastAsia" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>夜明けに１人の漁師が網を投げている。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>image.cnt04:label</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsiaTheme="minorEastAsia" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>夕暮れに２人の漁師が網を修理している。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>image.cnt05:label</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsiaTheme="minorEastAsia" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>嵐の夜に２人の漁師が網を手繰り寄せている。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:proofErr w:type="spellStart"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>image.lsr:label</w:t></w:r><w:proofErr w:type="spellEnd"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Gothic" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>ペテロと３人の弟子たちが，魚でいっぱいになった網を一生懸命に引き上げている。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:proofErr w:type="spellStart"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>image.lss:label</w:t></w:r><w:proofErr w:type="spellEnd"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Gothic" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>ペテロと３人の弟子たちが，魚でいっぱいになった網を一生懸命に引き上げている。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:proofErr w:type="spellStart"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>image.sqr:label</w:t></w:r><w:proofErr w:type="spellEnd"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Gothic" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>ペテロと３人の弟子たちが，魚でいっぱいになった網を一生懸命に引き上げている。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:proofErr w:type="spellStart"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>image.sqs:label</w:t></w:r><w:proofErr w:type="spellEnd"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Gothic" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>ペテロと３人の弟子たち</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Gothic" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:lastRenderedPageBreak/><w:t>が，魚でいっぱいになった網を一生懸命に引き上げている。</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:proofErr w:type="spellStart"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>MEPS:Title</w:t></w:r><w:proofErr w:type="spellEnd"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Gothic" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>「魚ではなく人を集める漁師」になりますか</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:proofErr w:type="spellStart"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>WEB:MetaTitle</w:t></w:r><w:proofErr w:type="spellEnd"/><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Gothic" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>「魚ではなく人を集める漁師」になりますか</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t xml:space="preserve">  </w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Gothic" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t>「ものみの塔」研究</w:t></w:r><w:r w:rsidRPr="00360EA4"><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="Arial" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr><w:t></w:t></w:r></w:p><w:p w:rsidR="00425316" w:rsidRPr="00360EA4" w:rsidRDefault="00425316" w:rsidP="00425316"><w:pPr><w:spacing w:line="240" w:lineRule="auto"/><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Mincho" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:sz w:val="22"/><w:szCs w:val="22"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr></w:p><w:p w:rsidR="0004652D" w:rsidRPr="00360EA4" w:rsidRDefault="0004652D"><w:pPr><w:rPr><w:rFonts w:ascii="MEPS Bookman Universal" w:eastAsia="MS Gothic" w:hAnsi="MEPS Bookman Universal" w:cs="MEPS Bookman Universal"/><w:lang w:eastAsia="ja-JP"/></w:rPr></w:pPr></w:p><w:sectPr w:rsidR="0004652D" w:rsidRPr="00360EA4" w:rsidSect="001F4FC2"><w:pgSz w:w="11907" w:h="16840" w:code="9"/><w:pgMar w:top="1134" w:right="1418" w:bottom="1134" w:left="1418" w:header="709" w:footer="709" w:gutter="0"/><w:cols w:space="708"/><w:titlePg/><w:docGrid w:linePitch="360"/></w:sectPr></w:body></w:document>
)";

  if (parser.parse(source)) { return 0; }

  std::cout << "syntax error..." << std::endl;
  return -1;
}
