# Consodoc Publishing Server
# Copyright (c) Oleg Parashchenko, <olpa consodoc com>
import sys, os, os.path, re
import SCons.Builder, SCons.Errors, SCons.Action, SCons.Sig.MD5
import Consodoc, Consodoc.compat, Consodoc.Builders.texloginfo

def _updateVars(vars, xxx):
  def key(k):
    return 'tmp%s_%s' % (xxx, k)
  key_in = key('in')
  if vars.get(key_in, None) == None:
    vars[key_in] = vars['patch_tex']
  if vars.get(key('out'), None) == None:
    vars[key('out')] = os.path.splitext(vars[key_in])[0] + '.' + xxx
  basename = os.path.splitext(vars[key_in])[0]
  for ext in ['log', 'aux', 'toc', 'out']:
    if ext == 'out':
      k = key('outout')
    else:
      k = key(ext)
    if vars.get(k, None) == None:
      vars[k] = '%s.%s' % (basename, ext)
  if vars.get(key('include_dirs'), None) == None:
    vars[key('include_dirs')] = [vars['supportdir']]
  if vars.get(key('image_dirs'), None) == None:
    sfx = xxx
    if sfx == 'dvi':
      sfx = 'eps'
    vars[key('image_dirs')] = ['images/%s' % sfx]
  key_max = key('rerun_max')
  if vars.get(key_max, None) == None:
    vars[key_max] = 4 # Change also the test "tmpout/rerun/max_times"
  if vars.get(key('rerun_message'), None) == None:
    vars[key('rerun_message')] = 're-running TeX'
  if vars.get(key('rerun_nomore_message'), None) == None:
    vars[key('rerun_nomore_message')] = 'After %i attempts, re-run is still required. Human inspection is required.' % vars[key_max]

def updateVars(vars):
  _updateVars(vars, 'pdf')
  _updateVars(vars, 'dvi')

#
# Execute LaTeX
#
def run_xxxlatex(target, source, env, xxx):
  def key(k):
    return 'tmp%s_%s' % (xxx, k)
  vars = env['CONSODOC']
  # Check that patch lock is absent
  try:
    os.stat(vars['patch_error'])
    raise Consodoc.ConsodocError(target, vars['patch_error_message'])
  except OSError:
    pass
  # Setup TeX environment
  texin = env['ENV'].get('TEXINPUTS', None)
  if texin == None:
    texin = os.getenv('TEXINPUTS')
  if texin == None:
    texin = ''
  else:
    texin = texin + os.pathsep
  for directory in vars[key('include_dirs')]:
    texin = texin + os.path.abspath(directory) + os.pathsep
  for directory in vars[key('image_dirs')]:
    texin = texin + os.path.abspath(directory) + os.pathsep
  if 'texenv2' == str(target[0]):
    ee = env['ENV']
    oe = os.environ
    for k in oe:
      if (not ee.has_key(k)) or ('PATH' == k):
        ee[k] = oe[k]
  env['ENV']['TEXINPUTS'] = texin
  env['ENV']['HOME']      = os.getenv('HOME')
  # Undocumented behaviour: getting shell with TeX environemnt
  if str(target[0]) in ('texenv', 'texenv2'):
    action = SCons.Action.CommandAction('bash')
    return action.execute(target, source, env)
  # TeX command-line action
  tex_file = str(source[0])
  tex_dir  = os.path.dirname(tex_file)
  tex_file = os.path.basename(tex_file)
  redir    = '>/dev/null'
  try:
    redir = ' >' + os.devnull
  except:
    if '\r' == os.linesep: # Mac
      redir = '>Dev:Nul'
    elif '\r\n' == os.linesep: # Windows
      redir = '>NUL'
  cmdline  = '%s --src --interaction batchmode %s%s' % (vars['%slatex' % xxx], tex_file, redir)
  action   = SCons.Action.CommandAction(cmdline)
  #
  # Run TeX
  #
  reruns    = 0
  rerun_why = ''
  while reruns < vars[key('rerun_max')]:
    reruns = reruns + 1
    if reruns > 1:
      print '*** %s: %s' % (vars[key('rerun_message')], rerun_why)
    #
    # Remember auxiliary files
    #
    aux_file = vars[key('aux')] # Used later
    aux_list = [aux_file, vars[key('toc')], vars[key('outout')]]
    def get_aux_signatures(flist):
      ret = {}
      for node in env.arg2nodes(flist):
        ret[str(node)] = SCons.Sig.MD5.signature(node)
        node.clear()
      return ret
    old_aux = get_aux_signatures(aux_list)
    #
    # Run LaTeX
    #
    old_dir = os.getcwd()
    os.chdir(tex_dir)
    ccode = action.execute(target, source, env)
    os.chdir(old_dir)
    if ccode > 1:
      return ccode
    #
    # Parse TeX log, decide if re-run is required
    #
    tli = Consodoc.Builders.texloginfo.texloginfo()
    tli.parse_log_file(vars[key('log')])
    tex_warnings = tli.get_warnings()
    tex_errors   = tli.get_errors()
    need_rerun   = tli.get_rerun()
    #
    # 1) If error, no re-run. 2) If the log says so, re-run
    #
    if tex_errors != '': break
    if 0 != need_rerun:
      rerun_why = 'advice from the log file'
      continue
    #
    # Yet another re-run decision, based on auxiliary files.
    # Heuristic: ignore change of .aux file on the first run
    #
    new_aux = get_aux_signatures(aux_list)
    if 1 == reruns:
      new_aux[aux_file] = old_aux[aux_file]
    for k in aux_list:
      if old_aux[k] != new_aux[k]:
        rerun_why = '''the auxiliary file `%s' has changed''' % k
        break
    if old_aux == new_aux: break # no re-run if auxiliary files still the same
  else:
    #
    # Too much re-runs
    #
    try:
      os.remove(vars[key('out')])
    except:
      pass
    raise SCons.Errors.BuildError(target, vars[key('rerun_nomore_message')])
  if tex_warnings:
    print tex_warnings,
  if tex_errors != '':
    try:
      os.remove(vars[key('out')])
    except:
      pass
    raise Consodoc.ConsodocError(target, "\n" + tex_errors)
  return 0

def run_pdflatex(target, source, env):
  return run_xxxlatex(target, source, env, 'pdf')

def run_dvilatex(target, source, env):
  return run_xxxlatex(target, source, env, 'dvi')

#
# Parsing LaTeX log file
#
def grep_log(target, source, env):
  log_file   = str(source[0])
  str_target = str(target[0])
  tli = Consodoc.Builders.texloginfo.texloginfo()
  tli.parse_log_file(log_file)
  if 'warnings' == str_target:
    print tli.get_warnings(),
    return None
  if 'errors' == str_target:
    print tli.get_errors(),
    return None
  assert 'missed' == str_target
  print tli.get_missed(),

#
# Consodoc setup
#
def updateEnvironment(env):
  vars = env['CONSODOC']
  env['BUILDERS']['cdoc_pdflatex'] = SCons.Builder.Builder(action = run_pdflatex)
  env['BUILDERS']['cdoc_dvilatex'] = SCons.Builder.Builder(action = run_dvilatex)
  pdf_file = vars['tmppdf_out']
  dvi_file = vars['tmpdvi_out']
  env.cdoc_pdflatex(pdf_file, vars['tmppdf_in'])
  env.cdoc_dvilatex(dvi_file, vars['tmpdvi_in'])
  env.Alias('tmppdf', pdf_file)
  env.Alias('tmpdvi', dvi_file)
  for key in ['aux', 'log', 'outout', 'toc']:
    env.Clean(pdf_file, vars['tmppdf_%s' % key])
    env.Clean(dvi_file, vars['tmpdvi_%s' % key])
  # Undocumented target
  env.cdoc_pdflatex('texenv',  None)
  env.cdoc_pdflatex('texenv2', None)
  # Yet another undocumented target
  targets = map (lambda node: str(node), Consodoc.compat.get_build_targets())
  if ('repdf' in targets) or ('redvi' in targets):
    def always_ok(target, source, env):
      return None
    def gen_explain_func(s):
      return lambda: '''rebuilding `%s' due to user demand\n''' % s
    env['BUILDERS']['always_ok'] = SCons.Builder.Builder(action = always_ok)
    if 'repdf' in targets:
      out_pdf = vars['pdf_out']
      for node in env.arg2nodes([pdf_file, out_pdf]):
        env.AlwaysBuild(node)
        node.explain = gen_explain_func(str(node))
      env.always_ok('repdf', out_pdf)
    if 'redvi' in targets:
      out_dvi = vars['dvi_out']
      for node in env.arg2nodes([dvi_file, out_dvi]):
        env.AlwaysBuild(node)
        node.explain = gen_explain_func(str(node))
      env.always_ok('redvi', out_dvi)
  # Even more undocumented targets
  log_file = env['CONSODOC']['tmppdf_log']
  env['BUILDERS']['grep_log'] = SCons.Builder.Builder(action = grep_log)
  env.grep_log('missed',   log_file)
  env.grep_log('errors',   log_file)
  env.grep_log('warnings', log_file)
