# Consodoc Publishing Server
# Copyright (c) Oleg Parashchenko, <olpa consodoc com>
import os, os.path
import Consodoc, Consodoc.compat
import SCons.Builder, SCons.Script, SCons.Action
import shutil

def updateVars(vars):
  if None == vars.get('patch_file', None):
    vars['patch_file'] = vars['in_patch']
  if None == vars.get('patch_tex_orig', None):
    vars['patch_tex_orig'] = vars['pretex_out']
  if None == vars.get('patch_tex', None):
    basename = os.path.basename(vars['patch_tex_orig'])
    basename = os.path.splitext(basename)[0]
    vars['patch_tex'] = os.path.join(vars['tmpdir'], basename)
  if None == vars.get('patch_tex_copy', None):
    vars['patch_tex_copy'] = vars['patch_tex'] + '.copy'
  if None == vars.get('patch_tex_rej', None):
    vars['patch_tex_rej'] = vars['patch_tex'] + '.rej'
  if None == vars.get('patch_error', None):
    basename = os.path.basename(vars['patch_file'])
    basename = os.path.splitext(basename)[0]
    vars['patch_error'] = os.path.join(vars['tmpdir'], basename + '.patch-lock')
  if None == vars.get('patch_error_message', None):
    vars['patch_error_message'] = 'File \'%s\' is patched with problems. Correct the rejections manually, delete the file \'%s\' and re-run generation.' % (vars['patch_tex'], vars['patch_error'])
  if None == vars.get('patch_reminder_message', None):
    vars['patch_reminder_message'] = 'TeX code in \'%s\' was changed. Consider generating a patch.' % vars['patch_tex']

class FileopWrapper:
  def exists(self, fname):
    try:
      os.stat(fname)
      return 1
    except:
      return 0
  def timestamp(self, fname):
    return os.stat(fname).st_mtime

# TeX file is indended to be edited by the user, but when it is changed,
# SCons doesn't re-generate PDF file. Why? Because TeX file is considered
# to be intermediate, and its signature is the signature of the build
# process. To track user changes, we should override calculation of
# the signature.
import SCons.Node.FS
class DerivedButSourceFile(SCons.Node.FS.File):
  # 0.96.90 and branch-core compatibility
  def __init__(self, *ls, **kw):
    SCons.Node.FS.File.__init__(self, *ls, **kw)
    if None == getattr(self, 'calc_bsig', None):
      self.calc_bsig = self.get_bsig
      self.calc_csig = self.get_csig
  # Actual signature calculation. Hope it's ok for the SCons cvs version.
  def calc_signature(self, calc=None):
    s = self.calc_bsig(calc)
    if self.rexists():
      s = s + '_' + self.calc_csig(calc)
    return s

def assert_no_patch_error(vars, target, h=FileopWrapper()):
  if h.exists(vars['patch_error']):
    raise Consodoc.ConsodocError(target, vars['patch_error_message'])

def no_patch(target, source, env):
  vars = env['CONSODOC']
  assert_no_patch_error(vars, target)
  shutil.copyfile(str(source[0]), str(target[0]))
  tex_copy = vars['patch_tex_copy']
  shutil.copyfile(str(target[0]), tex_copy)

def apply_patch(target, source, env):
  vars = env['CONSODOC']
  assert_no_patch_error(vars, target)
  #
  # Perform patching
  #
  cmdline = '%s %s <%s' % (vars['patch'], target[0], vars['patch_file'])
  shutil.copyfile(str(source[0]), str(target[0]))
  action = SCons.Action.CommandAction(cmdline)
  ccode  = action.execute(target, source, env)
  #
  # Regardless if patching is succesfull or not, make a copy of result.
  # (User can fix the errors and delete lock-file, then the environment
  # should be sane.)
  #
  tex_copy = vars['patch_tex_copy']
  shutil.copyfile(str(target[0]), tex_copy)
  #
  # Now check the result. If patching is failed, create the lock
  # and raise an error
  #
  if ccode != 0:
    try: 
      fsock = open(vars['patch_error'], 'w')
      fsock.close()
    except Exception, e:
      print 'Error creating patch error file: ', str(e)
    target[0].built() # See "tests/patch/no_repatch/readme"
    raise Consodoc.ConsodocError(target, vars['patch_error_message'])
  #
  # If patching is successfull, remove "bad" files
  #
  err_file = vars['patch_error']
  rej_file = vars['patch_tex_rej']
  try:
    os.remove(err_file)
    os.remove(rej_file)
  except:
    pass

def generate_patch(target, source, env):
  vars       = env['CONSODOC']
  tex_file   = vars['patch_tex']
  patch_file = vars['patch_file']
  tex_orig   = vars['patch_tex_orig']
  tex_copy   = vars['patch_tex_copy']
  assert_no_patch_error(vars, target)
  # Generate a patch
  diffstring='%s -u %s %s >%s' % (vars['diff'], tex_orig, tex_file, patch_file)
  action = SCons.Action.CommandAction(diffstring)
  ccode  = action.execute(target, source, env)
  if ccode > 1:
    return ccode
  shutil.copyfile(tex_file, tex_copy)

def updateEnvironment(env, fileop=FileopWrapper()):
  vars = env['CONSODOC']
  tex_file   = vars['patch_tex']
  patch_file = vars['patch_file']
  #
  # To avoid circular dependencies, the step of generating patch
  # is added only if requested from command line
  #
  # Check if reverse order (generating a patch)
  #
  reverse_order = 0
  for tlist in (Consodoc.compat.get_build_targets(), Consodoc.compat.get_default_targets()):
    for target in tlist:
      # In functional tests, to test the reverse patching step,
      # we need to set 'patch' the default target. When done so,
      # SCons, for some reasons, says "Do not know how to make
      # target `patch'.  Stop.". So, before Consodoc inittialization,
      # tests set the default to 'patch-hack', and after -- re-set
      # to 'patch'.
      if str(target) in ['patch', 'patch-hack']:
        reverse_order = 1
        break
  #
  # Check if the TeX code is changed
  #
  tex_code_changed = 0
  copy_file = vars['patch_tex_copy']
  if fileop.exists(tex_file) and fileop.exists(copy_file):
    h1 = open(tex_file)
    h2 = open(copy_file)
    tex_code_changed = h1.read() != h2.read()
    h2.close()
    h1.close()
  #
  # Reverse order, generate a patch
  #
  if reverse_order:
    if '__main__.FileopProxy' == str(fileop.__class__):
      if fileop.exists('unit_test_wants_error'):
        raise Consodoc.ConsodocError(None, vars['patch_error_message'])
    env['BUILDERS']['cdoc_patch_generate'] = SCons.Builder.Builder(action = generate_patch)
    env.cdoc_patch_generate(patch_file, None)
    if tex_code_changed:
      env.AlwaysBuild(patch_file)
      patch_file_node = env.arg2nodes(patch_file)[0]
      re_str = ''
      if fileop.exists(patch_file):
        re_str = 're'
      def my_explain(self = None):
        return '''%sbuilding `%s' because TeX code in `%s' is changed (comparing to `%s')\n''' % (re_str, patch_file, tex_file, copy_file)
      patch_file_node.explain = my_explain
    env.Alias('patch', patch_file)
    env.Precious(patch_file)
    env.Clean(patch_file, vars['patch_tex_rej'])
  else:
    tex_file_node = env.fs.Entry(tex_file, klass=DerivedButSourceFile)
    #
    # Forward order, apply a patch or copy file
    #
    if tex_code_changed:
      print '*** ' + vars['patch_reminder_message']
    if fileop.exists(patch_file):
      env['BUILDERS']['cdoc_patch_apply'] = SCons.Builder.Builder(action = apply_patch)
      env.cdoc_patch_apply(tex_file_node, [vars['patch_tex_orig'], patch_file])
      env.Precious(tex_file_node)
    else:
      env['BUILDERS']['cdoc_patch_copy'] = SCons.Builder.Builder(action = no_patch)
      env.cdoc_patch_copy(tex_file_node, vars['patch_tex_orig'])
    if fileop.exists(vars['patch_error']):
      env.Precious(tex_file_node)
      env.AlwaysBuild(tex_file_node)
      def my_explain(self = tex_file_node):
        return '''pseudo-rebuilding `%s' to raise an error on the patch step\n''' % self
      tex_file_node.explain = my_explain
    env.Alias('tex',   tex_file_node)
    env.Clean(tex_file_node, vars['patch_tex_rej'])
    env.Clean(tex_file_node, vars['patch_tex_copy'])
    env.Clean(tex_file_node, vars['patch_error'])
  #
  # In case of no-exec build (-n), notify user about error lock
  #
  if not SCons.Action.execute_actions:
    if fileop.exists(vars['patch_error']):
      print '*** ' + vars['patch_error_message']
