/*
  COnfigurator for Gnome (COG)
  Copyright (C) 2002 Mads Villadsen <maxximum@krakoa.dk>
   
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "gconf-convenience.h"

const gchar *gconf_entry_get_short_desc (GConfEntry *entry, GConfClient *client)
{
  GConfSchema *schema;
  GConfEntry *schema_entry;
  
  schema_entry = gconf_client_get_entry (client, gconf_entry_get_schema_name (entry), NULL, TRUE, NULL);
  schema = gconf_value_get_schema (gconf_entry_get_value (schema_entry)); /* FIXME - check for value == NULL */
  return gconf_schema_get_short_desc (schema);
}

const gchar *gconf_entry_get_long_desc (GConfEntry *entry, GConfClient *client)
{
  GConfSchema *schema;
  GConfEntry *schema_entry;
  
  schema_entry = gconf_client_get_entry (client, gconf_entry_get_schema_name (entry), NULL, TRUE, NULL);
  schema = gconf_value_get_schema (gconf_entry_get_value (schema_entry)); /* FIXME - check for value == NULL */
  return gconf_schema_get_long_desc (schema);
}

const gchar *gconf_key_get_short_desc (gchar *key)
{
  GConfClient *client;
  GConfEntry *entry;
  const gchar *return_val = NULL;

  /* Get the default client */
  client = gconf_client_get_default ();

  entry = gconf_client_get_entry (client, key, NULL, TRUE, NULL);
  if (entry != NULL)
    return_val = gconf_entry_get_short_desc (entry, client);
  
  g_object_unref (G_OBJECT (client));

  return return_val;
}

const gchar *gconf_key_get_long_desc (gchar *key)
{
  GConfClient *client;
  GConfEntry *entry;
  const gchar *return_val = NULL;

  /* Get the default client */
  client = gconf_client_get_default ();

  entry = gconf_client_get_entry (client, key, NULL, TRUE, NULL);
  if (entry != NULL)
    return_val = gconf_entry_get_long_desc (entry, client);
  
  g_object_unref (G_OBJECT (client));

  return return_val;
}

gboolean gconf_key_get_bool (gchar *key)
{
  GConfClient *client;
  GConfEntry *entry;
  gboolean return_val = FALSE;

  /* Get the default client */
  client = gconf_client_get_default ();

  entry = gconf_client_get_entry (client, key, NULL, TRUE, NULL);
  if (entry != NULL)
    return_val = gconf_value_get_bool (gconf_entry_get_value (entry));  
  
  g_object_unref (G_OBJECT (client));

  return return_val;
}

gint gconf_key_get_int (gchar *key)
{
  GConfClient *client;
  GConfEntry *entry;
  gint return_val = 0;

  /* Get the default client */
  client = gconf_client_get_default ();

  entry = gconf_client_get_entry (client, key, NULL, TRUE, NULL);
  if (entry != NULL)
    return_val = gconf_value_get_int (gconf_entry_get_value (entry));  
  
  g_object_unref (G_OBJECT (client));

  return return_val;
}

gchar *gconf_key_get_string (gchar *key)
{
  GConfClient *client;
  GConfEntry *entry;
  gchar *return_val = NULL;

  /* Get the default client */
  client = gconf_client_get_default ();

  entry = gconf_client_get_entry (client, key, NULL, TRUE, NULL);
  if (entry != NULL)
    return_val = gconf_value_get_string (gconf_entry_get_value (entry));  
  
  g_object_unref (G_OBJECT (client));

  return return_val;
}

gboolean gconf_key_is_writable (gchar *key)
{
  GConfClient *client;
  GConfEntry *entry;
  gboolean return_val = FALSE;

  /* Get the default client */
  client = gconf_client_get_default ();

  entry = gconf_client_get_entry (client, key, NULL, TRUE, NULL);
  if (entry != NULL)
    return_val = gconf_client_key_is_writable (client, key, NULL);
  
  g_object_unref (G_OBJECT (client));

  return return_val;
}

gboolean gconf_key_is_valid (gchar *key)
{
  GConfClient *client;
  GConfEntry *entry;
  gboolean return_val = FALSE;

  /* Get the default client */
  client = gconf_client_get_default ();

  entry = gconf_client_get_entry (client, key, NULL, TRUE, NULL);
  if ((gconf_entry_get_value (entry) != NULL))
    /*if ((gconf_entry_get_value (entry) != NULL) && (gconf_entry_get_schema_name (entry) != NULL))*/
    /*does the entry actually contain anything and does it have a schema associated with it */
    return_val = TRUE;
  
  g_object_unref (G_OBJECT (client));

  return return_val;
}

gboolean gconf_key_is_bool (gchar *key)
{
  GConfClient *client;
  GConfEntry *entry;
  gboolean return_val = FALSE;

  /* Check key validity */
  if (gconf_key_is_valid (key))
    {
      /* Get the default client */
      client = gconf_client_get_default ();
      
      entry = gconf_client_get_entry (client, key, NULL, TRUE, NULL);
      if (gconf_entry_get_value (entry)->type == GCONF_VALUE_BOOL)
	return_val = TRUE;
      
      g_object_unref (G_OBJECT (client));
    }

  return return_val;
}

gboolean gconf_key_is_int (gchar *key)
{
  GConfClient *client;
  GConfEntry *entry;
  gboolean return_val = FALSE;

  if (key != NULL)
    {
      /* Check key validity */
      if (gconf_key_is_valid (key))
	{
	  /* Get the default client */
	  client = gconf_client_get_default ();
	  
	  entry = gconf_client_get_entry (client, key, NULL, TRUE, NULL);
	  if (gconf_entry_get_value (entry)->type == GCONF_VALUE_INT)
	    return_val = TRUE;
	  
	  g_object_unref (G_OBJECT (client));
	}
    }

  return return_val;
}

gboolean gconf_key_is_string (gchar *key)
{
  GConfClient *client;
  GConfEntry *entry;
  gboolean return_val = FALSE;

  /* Check key validity */
  if (gconf_key_is_valid (key))
    {
      /* Get the default client */
      client = gconf_client_get_default ();
      
      entry = gconf_client_get_entry (client, key, NULL, TRUE, NULL);
      if (gconf_entry_get_value (entry)->type == GCONF_VALUE_STRING)
	return_val = TRUE;
      
      g_object_unref (G_OBJECT (client));
    }

  return return_val;
}


/* Since the callback and config notification is the same 
 * for all boolean widgets they are put in here
 */
/* FIXME - they don't really belong here */
void entry_bool_toggled_callback(GtkWidget* w, gpointer user_data)
{
  GConfClient *client;
  gboolean b;

  client = GCONF_CLIENT(g_object_get_data (G_OBJECT (w), "client"));
  b = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(w));

  gconf_client_set_bool (client, (gchar *) user_data, b, NULL);
}

void bool_config_notify (GConfClient *client,
			 guint        cnxn_id,
			 GConfEntry  *entry,
			 gpointer     user_data)
{
  GtkWidget *w = user_data;
 
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), gconf_value_get_bool (gconf_entry_get_value (entry)));
}

void int_spin_button_config_notify (GConfClient *client,
				    guint        cnxn_id,
				    GConfEntry  *entry,
				    gpointer     user_data)
{
  GtkWidget *w = user_data;
 
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(w), gconf_value_get_int (gconf_entry_get_value (entry)));
}

void entry_spin_button_value_changed_callback (GtkWidget* w, gpointer user_data)
{
  GConfClient *client;
  gint i;

  client = GCONF_CLIENT(g_object_get_data (G_OBJECT (w), "client"));
  i = gtk_spin_button_get_value (GTK_SPIN_BUTTON(w));

  gconf_client_set_int (client, (gchar *) user_data, i, NULL);
}


/* Remove the notification callback when the widget monitoring
 * notifications is destroyed
 */
static void setting_widget_destroy_callback (GtkWidget *widget,
					     gpointer   data)
{
  guint notify_id;
  GConfClient *client;

  client = g_object_get_data (G_OBJECT (widget), "client");
  notify_id = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (widget), "notify_id"));

  if (notify_id != 0)
    gconf_client_notify_remove (client, notify_id);
}

void setup_setting_widget (GtkWidget *setting_widget,
			   gchar *gconf_key,
			   GConfClientNotifyFunc func)			   
{
  GConfClient *client;
  GConfEntry *entry;
  guint notify_id;
  
  /* Get the default client */
  client = gconf_client_get_default ();
  gconf_client_add_dir (client, "/", GCONF_CLIENT_PRELOAD_NONE, NULL); /* We want to monitor _all_ directories */
  
  entry = gconf_client_get_entry (client, gconf_key, NULL, TRUE, NULL);
  
  notify_id = gconf_client_notify_add (client,
				       gconf_key,
				       func,
				       setting_widget,
				       NULL, NULL);
	  
  /* Note that notify_id will be 0 if there was an error,
   * so we handle that in our destroy callback.
   */
  g_object_set_data (G_OBJECT (setting_widget), "notify_id", GUINT_TO_POINTER (notify_id));
  g_object_set_data (G_OBJECT (setting_widget), "client", client);
	  
  g_signal_connect (G_OBJECT (setting_widget), "destroy",
		    G_CALLBACK (setting_widget_destroy_callback),
		    NULL);
	  
  /* If key isn't writable, then set insensitive */
  gtk_widget_set_sensitive (setting_widget, gconf_key_is_writable (gconf_key));

  g_object_unref (G_OBJECT (client));
}

