# 7 "lexer_cocci.mll"
 
open Parser_cocci_menhir
module D = Data
module Ast = Ast_cocci
module Ast0 = Ast0_cocci
module P = Parse_aux
module FC = Flag_parsing_cocci
exception Lexical of string
let tok = Lexing.lexeme

let line = ref 1
let logical_line = ref 0

(* ---------------------------------------------------------------------- *)
(* control codes *)

(* Defined in data.ml
type line_type = MINUS | OPTMINUS | PLUS | CONTEXT | OPT
*)

let current_line_type = ref (D.CONTEXT,!line,!logical_line)

let prev_plus = ref false
let line_start = ref 0 (* offset of the beginning of the line *)
let get_current_line_type lexbuf =
  let (c,l,ll) = !current_line_type in
  let lex_start = Lexing.lexeme_start lexbuf in
  let preceeding_spaces =
    if !line_start < 0 then 0 else lex_start - !line_start in
  (*line_start := -1;*)
  prev_plus := (c = D.PLUS) || (c = D.PLUSPLUS);
  (c,l,ll,ll,lex_start,preceeding_spaces,[],[],[],"")
let current_line_started = ref false
let col_zero = ref true

let contextify (c,l,ll,lle,lex_start,preceeding_spaces,bef,aft,pos,ws) =
  (D.CONTEXT,l,ll,lle,lex_start,preceeding_spaces,bef,aft,pos,ws)

let reset_line lexbuf =
  line := !line + 1;
  current_line_type := (D.CONTEXT,!line,!logical_line);
  current_line_started := false;
  (if not(!Data.in_meta || !Data.in_rule_name) then col_zero := true);
  line_start := Lexing.lexeme_start lexbuf + 1

let started_line = ref (-1)

let start_line seen_char =
  current_line_started := true;
  col_zero := false;
  (if seen_char && not(!line = !started_line)
  then
    begin
      started_line := !line;
      logical_line := !logical_line + 1
    end)

let pass_zero _ = col_zero := false

let lexerr s1 s2 = raise (Lexical (Printf.sprintf "%s%s" s1 s2))

let opt_reverse_token token =
  if !FC.interpret_inverted
  then match token with
         D.MINUS        -> D.PLUSPLUS  (* maybe too liberal *)
       | D.OPTMINUS     -> lexerr "cannot invert token ?- (an optional minus line), which is needed for reversing the patch" ""
       | D.PLUS         -> D.MINUS
       | D.PLUSPLUS     -> D.MINUS (* may not be sufficient *)
       | _              -> token
  else token

let add_current_line_type x =
  match (opt_reverse_token x,!current_line_type) with
    (D.MINUS,(D.CONTEXT,ln,lln))  ->
      current_line_type := (D.MINUS,ln,lln)
  | (D.MINUS,(D.OPT,ln,lln))      ->
      current_line_type := (D.OPTMINUS,ln,lln)
  | (D.PLUS,(D.CONTEXT,ln,lln))   ->
      current_line_type := (D.PLUS,ln,lln)
  | (D.PLUSPLUS,(D.CONTEXT,ln,lln))   ->
      current_line_type := (D.PLUSPLUS,ln,lln)
  | (D.OPT,(D.CONTEXT,ln,lln))    ->
      current_line_type := (D.OPT,ln,lln)
  | _ -> lexerr "invalid control character combination" ""

let check_minus_context_linetype s =
  match !current_line_type with
    (D.PLUS,_,_) | (D.PLUSPLUS,_,_) -> lexerr "invalid in a + context: " s
  | _ -> ()

let check_context_linetype s =
  match !current_line_type with
    (D.CONTEXT,_,_) -> ()
  | _ -> lexerr "invalid in a nonempty context: " s

let check_plus_linetype s =
  match !current_line_type with
    (D.PLUS,_,_) | (D.PLUSPLUS,_,_) -> ()
  | _ -> lexerr "invalid in a non + context: " s

let check_arity_context_linetype s =
  match !current_line_type with
    (D.CONTEXT,_,_) | (D.PLUS,_,_) | (D.PLUSPLUS,_,_)
  | (D.OPT,_,_) -> ()
  | _ -> lexerr "invalid in a nonempty context: " s

let check_comment s =
  if not !current_line_started
  then lexerr "+ expected at the beginning of the line" s

let process_include start finish str =
  (match !current_line_type with
    (D.PLUS,_,_) | (D.PLUSPLUS,_,_) ->
      (try
	let _ = Str.search_forward (Str.regexp "\\.\\.\\.") str start in
	lexerr "... not allowed in + include" ""
      with Not_found -> ())
  | _ -> ());
  String.sub str (start + 1) (finish - start - 1)

(* ---------------------------------------------------------------------- *)
type pm = PATCH | MATCH | UNKNOWN

let pm = ref UNKNOWN

let patch_or_match = function
    PATCH ->
      if not !D.ignore_patch_or_match
      then
	(match !pm with
	  MATCH ->
	    lexerr "- or + not allowed in the first column for a match" ""
	| PATCH -> ()
	| UNKNOWN -> Flag.sgrep_mode2 := false; pm := PATCH)
  | MATCH ->
      if not !D.ignore_patch_or_match
      then
	(match !pm with
	  PATCH -> lexerr "* not allowed in the first column for a patch" ""
	| MATCH -> ()
	| UNKNOWN -> Flag.sgrep_mode2 := true; pm := MATCH)
  | _ -> failwith "unexpected argument"

(* ---------------------------------------------------------------------- *)
(* identifiers, including metavariables *)

let metavariables = (Hashtbl.create(100) : (string, D.clt -> token) Hashtbl.t)

let all_metavariables =
  (Hashtbl.create(100) : (string,(string * (D.clt -> token)) list) Hashtbl.t)

let type_names = (Hashtbl.create(100) : (string, D.clt -> token) Hashtbl.t)

let attr_names = (Hashtbl.create(100) : (string, D.clt -> token) Hashtbl.t)

let declarer_names = (Hashtbl.create(100) : (string, D.clt -> token) Hashtbl.t)

let iterator_names = (Hashtbl.create(100) : (string, D.clt -> token) Hashtbl.t)

let symbol_names = (Hashtbl.create(15) : (string, D.clt -> token) Hashtbl.t)

let rule_names = (Hashtbl.create(100) : (string, unit) Hashtbl.t)

let check_var s linetype =
  let fail _ =
    if (!Data.in_prolog || !Data.in_rule_name) &&
      Str.string_match (Str.regexp "<.*>") s 0
    then TPathIsoFile s
    else
      try (Hashtbl.find metavariables s) linetype
      with Not_found ->
	(try (Hashtbl.find type_names s) linetype
	with Not_found ->
	  (try (Hashtbl.find attr_names s) linetype
	  with Not_found ->
	    (try (Hashtbl.find declarer_names s) linetype
	    with Not_found ->
	      (try (Hashtbl.find iterator_names s) linetype
	      with Not_found ->
		(try (Hashtbl.find symbol_names s) linetype
		with Not_found ->
                  TIdent (s,linetype)))))) in
  if !Data.in_meta || !Data.in_rule_name
  then (try Hashtbl.find rule_names s; TRuleName s with Not_found -> fail())
  else fail()

let id_tokens lexbuf =
  let s = tok lexbuf in
  let linetype = get_current_line_type lexbuf in
  let in_rule_name = !Data.in_rule_name in
  let in_meta = !Data.in_meta && not !Data.saw_struct in
  let in_iso = !Data.in_iso in
  let in_prolog = !Data.in_prolog in
  (if s = "identifer" && in_meta
  then Common.pr2 "Warning: should identifer be identifier?");
  match s with
    "metavariable" when in_meta -> check_arity_context_linetype s; TMetavariable
  | "identifier" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TIdentifier
  | "type" when in_meta ->       check_arity_context_linetype s; TType
  | "parameter" when in_meta ->  check_arity_context_linetype s; TParameter
  | "operator" when in_meta ->   check_arity_context_linetype s; TOperator
  | "binary" when in_meta ->   check_arity_context_linetype s; TBinary
  | "assignment" when in_meta ->   check_arity_context_linetype s; TAssignment
  | "constant"  when in_meta ->  check_arity_context_linetype s; TConstant
  | "generated" when in_rule_name && not (!Flag.make_hrule = None) ->
      check_arity_context_linetype s; TGenerated
  | "expression" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TExpression
  | "declaration" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TDeclaration
  | "field" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TField
  | "initialiser" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TInitialiser
  | "initializer" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TInitialiser
  | "idexpression" when in_meta ->
      check_arity_context_linetype s; TIdExpression
  | "statement" when in_meta ->  check_arity_context_linetype s; TStatement
  | "function"  when in_meta ->  check_arity_context_linetype s; TFunction
  | "local" when in_meta ->      check_arity_context_linetype s; TLocal
  | "global" when in_meta ->     check_arity_context_linetype s; TGlobal
  | "list" when in_meta ->       check_arity_context_linetype s; Tlist
  | "fresh" when in_meta ->      check_arity_context_linetype s; TFresh
  | "typedef" when in_meta ->    check_arity_context_linetype s; TTypedef
  | "attribute" when in_meta ->  check_arity_context_linetype s; TAttribute
  | "declarer" when in_meta ->   check_arity_context_linetype s; TDeclarer
  | "iterator" when in_meta ->   check_arity_context_linetype s; TIterator
  | "name" when in_meta ->       check_arity_context_linetype s; TName
  | "position" when in_meta ->   check_arity_context_linetype s; TPosition
  | "format" when in_meta ->     check_arity_context_linetype s; TFormat
  | "analysis" when in_meta ->   check_arity_context_linetype s; TAnalysis
  | "any" when in_meta ->        check_arity_context_linetype s; TPosAny
  | "pure" when in_meta && in_iso ->
      check_arity_context_linetype s; TPure
  | "context" when in_meta && in_iso ->
      check_arity_context_linetype s; TContext
  | "error" when in_meta ->      check_arity_context_linetype s; TError
  | "words" when in_meta ->      check_context_linetype s; TWords
  | "symbol" when in_meta ->     check_arity_context_linetype s; TSymbol

  | "using" when in_rule_name || in_prolog ->  check_context_linetype s; TUsing
  | "virtual" when in_prolog || in_rule_name || in_meta ->
      (* don't want to allow virtual as a rule name *)
      check_context_linetype s; TVirtual
  | "disable" when in_rule_name ->  check_context_linetype s; TDisable
  | "extends" when in_rule_name -> check_context_linetype s; TExtends
  | "depends" when in_rule_name -> check_context_linetype s; TDepends
  | "on" when in_rule_name      -> check_context_linetype s; TOn
  | "ever" when in_rule_name    -> check_context_linetype s; TEver
  | "never" when in_rule_name   -> check_context_linetype s; TNever
  | "file" when in_rule_name    -> check_context_linetype s; TFile
  | "in" when in_rule_name      -> check_context_linetype s; TIn
  (* exists and forall for when are reparsed in parse_cocci.ml *)
  | "exists" when in_rule_name  -> check_context_linetype s; TExists
  | "forall" when in_rule_name  -> check_context_linetype s; TForall
  | "script" when (in_rule_name || in_meta) ->
      check_context_linetype s; TScript
  | "initialize" when in_rule_name -> check_context_linetype s; TInitialize
  | "finalize" when in_rule_name   -> check_context_linetype s; TFinalize

  | "char" ->       Tchar     linetype
  | "short" ->      Tshort    linetype
  | "int" ->        Tint      linetype
  | "double" ->     Tdouble   linetype
  | "float" ->      Tfloat    linetype
  | "long" ->       Tlong     linetype
  | "void" ->       Tvoid     linetype
  | "size_t" ->     Tsize_t   linetype
  | "ssize_t" ->    Tssize_t  linetype
  | "ptrdiff_t" ->  Tptrdiff_t linetype
  (* in_meta is only for the first keyword; drop it now to allow any type
     name *)
  | "struct" ->     Data.saw_struct := true; Tstruct   linetype
  | "union" ->      Data.saw_struct := true; Tunion    linetype
  | "enum" ->       Data.saw_struct := true; Tenum     linetype
  | "unsigned" ->   Tunsigned linetype
  | "signed" ->     Tsigned   linetype
  | "decimal" when !Flag.ibm -> Tdecimal linetype
  | "EXEC" when !Flag.ibm -> Texec linetype

  | "auto"  ->      Tauto     linetype
  | "register" ->   Tregister linetype
  | "extern" ->     Textern   linetype
  | "static" ->     Tstatic   linetype
  | "inline" ->     Tinline   linetype
  | "typedef" ->    Ttypedef  linetype

  | "const" ->      Tconst    linetype
  | "volatile" ->   Tvolatile linetype

  | "if" ->         TIf       linetype
  | "else" ->       TElse     linetype
  | "while" ->      TWhile    linetype
  | "do" ->         TDo       linetype
  | "for" ->        TFor      linetype
  | "switch" ->     TSwitch   linetype
  | "case" ->       TCase     linetype
  | "default" ->    TDefault  linetype
  | "return" ->     TReturn   linetype
  | "break" ->      TBreak    linetype
  | "continue" ->   TContinue linetype
  | "goto" ->       TGoto     linetype

  | "sizeof" ->     TSizeof   linetype

  | "Expression"       when !Data.in_iso -> TIsoExpression
  | "ArgExpression"    when !Data.in_iso -> TIsoArgExpression
  | "TestExpression"   when !Data.in_iso -> TIsoTestExpression
  | "ToTestExpression" when !Data.in_iso -> TIsoToTestExpression
  | "Statement"        when !Data.in_iso -> TIsoStatement
  | "Declaration"      when !Data.in_iso -> TIsoDeclaration
  | "Type"             when !Data.in_iso -> TIsoType
  | "TopLevel"         when !Data.in_iso -> TIsoTopLevel

  | "_" when !Data.in_meta -> TUnderscore

  | s -> check_var s linetype

let mkassign op lexbuf =
  TOpAssign (op, (get_current_line_type lexbuf))

let init _ =
  line := 1;
  logical_line := 0;
  prev_plus := false;
  line_start := 0;
  current_line_started := false;
  current_line_type := (D.CONTEXT,0,0);
  col_zero := true;
  pm := UNKNOWN;
  Data.in_rule_name := false;
  Data.in_meta := false;
  Data.in_prolog := false;
  Data.saw_struct := false;
  Data.inheritable_positions := [];
  Hashtbl.clear all_metavariables;
  Hashtbl.clear Data.all_metadecls;
  Hashtbl.clear metavariables;
  Hashtbl.clear type_names;
  Hashtbl.clear rule_names;
  Hashtbl.clear iterator_names;
  Hashtbl.clear declarer_names;
  Hashtbl.clear symbol_names;
  let get_name (_,x) = x in
  Data.add_meta_meta :=
    (fun name pure ->
      let fn clt = TMeta(name,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_id_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaId(name,constraints,Ast.NoVal,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_virt_id_meta_found :=
    (fun name vl ->
      let fn clt = TIdent(vl,clt) in
      Hashtbl.replace metavariables name fn);
  Data.add_virt_id_meta_not_found :=
    (fun name pure ->
      let fn clt = TMetaId(name,Ast.CstrTrue,Ast.NoVal,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_fresh_id_meta :=
    (fun name seed ->
      let fn clt = TMetaId(name,Ast.CstrTrue,seed,Ast0.Impure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_type_meta :=
    (fun name pure ->
      let fn clt = TMetaType(name,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_init_meta :=
    (fun name pure ->
      let fn clt = TMetaInit(name,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_initlist_meta :=
    (function name -> function lenname -> function pure ->
      let fn clt = TMetaInitList(name,lenname,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_param_meta :=
    (function name -> function pure ->
      let fn clt = TMetaParam(name,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_paramlist_meta :=
    (function name -> function lenname -> function pure ->
      let fn clt = TMetaParamList(name,lenname,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_const_meta :=
    (fun tyopt name constraints pure ->
      let fn clt = TMetaConst(name,constraints,pure,tyopt,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_err_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaErr(name,constraints,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_exp_meta :=
    (fun tyopt name constraints pure ->
      let fn clt = TMetaExp(name,constraints,pure,tyopt,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_idexp_meta :=
    (fun tyopt name constraints pure ->
      let fn clt = TMetaIdExp(name,constraints,pure,tyopt,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_local_idexp_meta :=
    (fun tyopt name constraints pure ->
      let fn clt = TMetaLocalIdExp(name,constraints,pure,tyopt,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_global_idexp_meta :=
    (fun tyopt name constraints pure ->
      let fn clt = TMetaGlobalIdExp(name,constraints,pure,tyopt,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_explist_meta :=
    (function name -> function lenname -> function pure ->
      let fn clt = TMetaExpList(name,lenname,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_decl_meta :=
    (function name -> function pure ->
      let fn clt = TMetaDecl(name,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_field_meta :=
    (function name -> function pure ->
      let fn clt = TMetaField(name,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_field_list_meta :=
    (function name -> function lenname -> function pure ->
      let fn clt = TMetaFieldList(name,lenname,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_fmt_meta :=
    (function name -> function iconstraints ->
      let fn clt = failwith "format metavariable only allowed in a string" in
      Data.format_metavariables :=
	(get_name name,(name,iconstraints)) :: !Data.format_metavariables;
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_fmtlist_meta :=
    (function name -> function lenname ->
      let fn clt =
	failwith "format list metavariable only allowed in a string" in
      Data.format_list_metavariables :=
	(get_name name,(name,lenname)) :: !Data.format_list_metavariables;
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_stm_meta :=
    (function name -> function pure ->
      let fn clt = TMetaStm(name,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_stmlist_meta :=
    (function name -> function lenname -> function pure ->
      let fn clt = TMetaStmList(name,lenname,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_dparamlist_meta :=
    (function name -> function lenname -> function pure ->
      let fn clt = TMetaDParamList(name,lenname,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_func_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaFunc(name,constraints,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_local_func_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaLocalFunc(name,constraints,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_iterator_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaIterator(name,constraints,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_declarer_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaDeclarer(name,constraints,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_pos_meta :=
    (fun name constraints any ->
      let fn ((d,ln,_,_,_,_,_,_,_,_) as clt) =
	(if d = Data.PLUS
	then
	  failwith
	    (Printf.sprintf "%d: positions only allowed in minus code" ln));
	TMetaPos(name,constraints,any,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_assignOp_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaAssignOp (name, constraints, pure, clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_binaryOp_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaBinaryOp (name, constraints, pure, clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_type_name :=
    (function name ->
      let fn clt = TTypeId(name,clt) in
      Hashtbl.replace type_names name fn);
  Data.add_attribute :=
    (function name ->
      let fn clt = Tattr (name, clt) in
      Hashtbl.replace attr_names name fn);
  Data.add_declarer_name :=
    (function name ->
      let fn clt = TDeclarerId(name,clt) in
      Hashtbl.replace declarer_names name fn);
  Data.add_iterator_name :=
    (function name ->
      let fn clt = TIteratorId(name,clt) in
      Hashtbl.replace iterator_names name fn);
  Data.add_symbol_meta :=
    (function name ->
      let fn clt = TSymId (name,clt) in
      Hashtbl.replace symbol_names name fn);
  Data.init_rule := (function _ -> Hashtbl.clear metavariables);
  Data.install_bindings :=
    (function parent ->
      List.iter (function (name,fn) -> Hashtbl.add metavariables name fn)
	(Hashtbl.find all_metavariables parent))

(* initialization for each cocci rule *)
let reinit _ =
  Data.format_metavariables := [];
  Data.format_list_metavariables := []

(* the following is needed to properly tokenize include files.  Because an
include file is included after seeing a @, so current_line_started is true.
Current_line_started is not important for parsing the name of a rule, so we
don't have to reset this value to true after parsing an included file. *)
let include_init _ =
  current_line_started := false

let drop_spaces s =
  let len = String.length s in
  let rec loop n =
    if n = len
    then n
    else
      if List.mem (String.get s n) [' ';'\t']
      then loop (n+1)
      else n in
  let start = loop 0 in
  String.sub s start (len - start)

# 537 "lexer_cocci.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\161\255\162\255\081\000\119\000\168\255\169\255\192\000\
    \023\001\038\000\067\000\101\000\079\000\218\255\081\000\082\000\
    \085\000\224\255\226\255\229\255\230\255\231\255\232\255\234\255\
    \086\000\192\000\238\255\087\000\240\255\116\000\118\000\140\000\
    \100\001\111\001\198\001\087\000\029\002\058\002\150\000\255\255\
    \144\002\119\000\106\000\174\000\175\000\178\255\211\255\049\000\
    \050\000\089\000\041\000\055\000\051\000\060\002\188\255\078\000\
    \089\000\091\000\185\000\153\002\228\002\094\000\101\000\098\000\
    \108\000\203\000\086\003\169\003\185\255\116\000\111\000\106\000\
    \119\000\184\255\064\002\119\000\113\000\105\000\123\000\123\000\
    \087\003\179\000\172\000\203\000\204\000\183\255\001\001\182\255\
    \180\000\181\255\241\000\243\000\072\001\192\001\244\000\066\002\
    \110\002\245\000\115\000\131\000\066\002\181\000\183\000\075\001\
    \195\001\180\000\184\000\197\001\177\000\203\000\249\001\208\000\
    \219\000\063\002\007\001\227\003\058\004\145\004\232\004\063\005\
    \150\005\237\005\068\006\155\006\242\006\073\007\160\007\018\008\
    \026\001\164\002\194\001\198\001\199\001\105\008\009\001\192\008\
    \023\009\081\009\168\009\226\009\010\001\199\000\057\010\012\001\
    \026\001\144\010\231\010\028\001\062\011\149\011\207\011\038\012\
    \250\255\125\012\212\012\043\013\130\013\217\013\135\004\068\001\
    \140\002\069\001\054\001\245\255\243\255\071\001\225\255\086\003\
    \049\003\221\004\057\001\097\001\109\001\190\001\191\001\192\001\
    \246\255\196\001\214\001\244\255\205\255\204\255\164\255\214\255\
    \216\001\228\255\111\002\213\255\219\001\112\002\200\255\004\000\
    \179\255\190\255\233\255\235\255\237\255\194\255\208\255\212\255\
    \199\255\228\001\229\001\206\255\203\255\210\255\163\255\201\255\
    \195\255\209\255\207\255\019\014\106\014\233\001\164\014\251\014\
    \234\001\082\015\169\015\227\015\058\016\165\255\093\002\076\002\
    \137\016\053\005\241\007\184\008\164\016\139\005\226\005\234\016\
    \106\002\166\255\072\017\196\255\152\017\190\017\200\255\201\255\
    \008\018\095\018\031\002\010\002\009\002\064\002\142\002\128\002\
    \010\002\027\002\062\002\145\002\239\255\240\255\057\006\146\002\
    \243\255\244\255\245\255\246\255\247\255\248\255\004\003\052\003\
    \123\002\067\003\255\255\213\016\146\002\098\003\209\255\216\255\
    \252\255\236\255\251\255\235\255\250\255\219\255\242\255\145\006\
    \139\010\231\006\062\007\234\255\139\002\211\255\217\255\218\255\
    \215\255\225\255\148\002\210\255\214\255\222\255\213\255\221\255\
    \212\255\153\018\240\018\153\002\042\019\129\019\155\002\216\019\
    \047\020\105\020\192\020\208\255\157\002\023\021\110\021\158\002\
    \197\021\028\022\086\022\173\022\231\022\159\002\118\002\062\023\
    \187\002\188\002\149\023\236\023\189\002\067\024\154\024\212\024\
    \043\025\197\255\124\016\216\002\122\025\150\007\057\011\074\015\
    \149\025\094\008\194\008\219\025\040\003\198\255\119\003\255\255\
    \153\010\252\255\033\026\015\009\157\009\254\255\056\026\253\255\
    \043\007\254\255\225\016\255\255\251\255\094\026\049\010\067\011\
    \253\255\117\026\252\255\217\016\253\255\048\017\254\255\255\255\
    \250\255\155\026\138\011\030\012\252\255\178\026\251\255\176\012\
    \252\255\253\255\254\255\239\002\255\255";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\089\000\089\000\255\255\255\255\081\000\
    \080\000\053\000\066\000\064\000\040\000\255\255\036\000\059\000\
    \032\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \019\000\094\000\255\255\016\000\255\255\014\000\013\000\058\000\
    \028\000\080\000\080\000\006\000\080\000\094\000\033\000\255\255\
    \001\000\255\255\255\255\002\000\003\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\068\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\069\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\075\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\255\255\255\255\255\255\255\255\255\255\075\000\
    \075\000\255\255\255\255\075\000\255\255\255\255\075\000\255\255\
    \255\255\075\000\255\255\255\255\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \255\255\255\255\255\255\255\255\004\000\255\255\083\000\255\255\
    \084\000\255\255\082\000\255\255\255\255\255\255\255\255\082\000\
    \255\255\255\255\255\255\084\000\255\255\084\000\255\255\255\255\
    \255\255\080\000\080\000\007\000\080\000\080\000\088\000\255\255\
    \008\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \088\000\255\255\057\000\063\000\034\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \039\000\255\255\078\000\255\255\038\000\079\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\062\000\035\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\085\000\255\255\255\255\255\255\
    \085\000\255\255\085\000\255\255\255\255\255\255\089\000\089\000\
    \255\255\088\000\255\255\255\255\089\000\088\000\255\255\089\000\
    \089\000\255\255\255\255\255\255\057\000\057\000\255\255\255\255\
    \049\000\048\000\059\000\035\000\029\000\028\000\027\000\032\000\
    \025\000\022\000\018\000\031\000\255\255\255\255\014\000\023\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\059\000\006\000\
    \059\000\024\000\255\255\001\000\255\255\002\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\056\000\
    \255\255\056\000\255\255\255\255\017\000\255\255\255\255\255\255\
    \255\255\255\255\026\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\053\000\255\255\255\255\255\255\053\000\255\255\
    \053\000\255\255\255\255\255\255\255\255\255\255\255\255\051\000\
    \255\255\052\000\255\255\050\000\255\255\255\255\255\255\255\255\
    \050\000\255\255\255\255\255\255\052\000\255\255\052\000\255\255\
    \255\255\255\255\057\000\057\000\255\255\056\000\255\255\255\255\
    \057\000\056\000\255\255\057\000\057\000\255\255\255\255\255\255\
    \000\000\255\255\003\000\001\000\001\000\255\255\002\000\255\255\
    \255\255\255\255\001\000\255\255\255\255\004\000\002\000\002\000\
    \255\255\003\000\255\255\255\255\255\255\002\000\255\255\255\255\
    \255\255\005\000\003\000\003\000\255\255\004\000\255\255\255\255\
    \255\255\255\255\255\255\004\000\255\255";
  Lexing.lex_default = 
   "\001\000\000\000\000\000\255\255\255\255\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\255\255\000\000\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\043\000\044\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\000\000\090\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\086\000\084\000\084\000\000\000\086\000\000\000\
    \255\255\000\000\090\000\090\000\090\000\090\000\094\000\090\000\
    \090\000\097\000\255\255\255\255\255\255\255\255\255\255\103\000\
    \104\000\255\255\255\255\107\000\255\255\255\255\110\000\255\255\
    \255\255\113\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\130\000\130\000\130\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\186\000\000\000\255\255\189\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\235\000\000\000\255\255\255\255\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\013\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\000\000\255\255\000\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\087\001\000\000\
    \089\001\000\000\255\255\255\255\255\255\000\000\255\255\000\000\
    \097\001\000\000\100\001\000\000\000\000\255\255\255\255\255\255\
    \000\000\255\255\000\000\108\001\000\000\112\001\000\000\000\000\
    \000\000\255\255\255\255\255\255\000\000\255\255\000\000\120\001\
    \000\000\000\000\000\000\255\255\000\000";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\040\000\039\000\039\000\039\000\039\000\192\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \040\000\027\000\005\000\037\000\007\000\014\000\011\000\006\000\
    \026\000\023\000\016\000\029\000\018\000\030\000\032\000\038\000\
    \004\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\009\000\017\000\031\000\012\000\015\000\028\000\
    \035\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\034\000\
    \008\000\008\000\008\000\022\000\025\000\021\000\010\000\036\000\
    \211\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\033\000\
    \008\000\008\000\008\000\020\000\024\000\019\000\013\000\225\000\
    \210\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\208\000\207\000\206\000\205\000\200\000\
    \201\000\202\000\199\000\198\000\190\000\221\000\226\000\152\000\
    \044\000\043\000\111\000\069\000\061\000\222\000\100\000\188\000\
    \099\000\055\000\209\000\184\000\098\000\225\000\223\000\228\000\
    \228\000\228\000\228\000\228\000\228\000\228\000\228\000\227\000\
    \227\000\187\000\056\000\183\000\185\000\221\000\226\000\173\000\
    \255\255\255\255\174\000\221\000\226\000\222\000\057\000\074\000\
    \045\000\058\000\059\000\222\000\062\000\043\000\223\000\075\000\
    \171\000\170\000\192\000\172\000\223\000\191\000\063\000\224\000\
    \064\000\065\000\197\000\046\000\066\000\070\000\071\000\072\000\
    \073\000\059\000\076\000\221\000\226\000\077\000\078\000\079\000\
    \080\000\088\000\089\000\222\000\007\000\108\000\193\000\105\000\
    \196\000\194\000\255\255\066\000\223\000\255\255\085\000\224\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\114\000\255\255\115\000\255\255\255\255\255\255\
    \002\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\104\000\103\000\106\000\107\000\007\000\
    \109\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\195\000\110\000\112\000\087\000\
    \113\000\135\000\130\000\114\000\141\000\137\000\140\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\114\000\255\255\115\000\148\000\255\255\144\000\095\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\160\000\165\000\164\000\166\000\008\000\182\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\159\000\007\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\181\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\114\000\180\000\115\000\255\255\093\000\255\255\255\255\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\255\255\255\255\255\255\255\255\008\000\255\255\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\156\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\007\000\131\000\177\000\175\000\176\000\132\000\
    \132\000\255\255\178\000\255\255\255\255\255\255\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \114\000\255\255\115\000\255\255\179\000\186\000\189\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\153\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\204\000\203\000\217\000\213\000\008\000\094\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\007\000\051\001\053\000\041\001\053\000\040\001\032\001\
    \255\255\255\255\255\255\255\255\255\255\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\114\000\
    \031\001\115\000\053\000\044\000\053\000\054\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \255\255\255\255\255\255\030\001\116\000\038\001\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \232\000\040\000\039\000\039\000\039\000\039\000\051\000\048\000\
    \051\000\048\000\059\000\049\000\092\000\049\000\047\000\096\000\
    \047\000\233\000\050\000\101\000\050\000\129\000\091\000\052\000\
    \040\000\052\000\233\000\041\000\037\001\102\000\233\000\162\000\
    \232\000\059\000\161\000\016\001\039\001\033\001\034\001\042\000\
    \255\255\013\001\255\255\255\255\129\000\255\255\255\255\255\255\
    \029\001\233\000\163\000\036\001\128\000\028\001\027\001\021\001\
    \022\001\035\001\233\000\047\001\097\000\043\001\233\000\056\001\
    \052\001\062\001\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\058\001\061\001\069\001\065\001\
    \060\000\255\255\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\124\001\000\000\
    \000\000\000\000\000\000\000\000\084\001\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\255\255\
    \255\255\019\001\255\255\060\000\084\001\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\066\000\
    \080\000\168\000\168\000\168\000\168\000\168\000\168\000\168\000\
    \168\000\168\000\168\000\000\000\255\255\014\001\255\255\255\255\
    \255\255\017\001\013\001\000\000\085\001\000\000\066\000\080\000\
    \000\000\083\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \015\001\169\000\020\001\169\000\000\000\081\000\168\000\168\000\
    \168\000\168\000\168\000\168\000\168\000\168\000\168\000\168\000\
    \000\000\000\000\000\000\082\000\085\001\000\000\000\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\000\000\018\001\000\000\000\000\067\000\000\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\068\000\000\000\088\001\000\000\000\000\000\000\000\000\
    \000\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\000\000\000\000\000\000\133\000\
    \067\000\000\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\000\000\000\000\
    \000\000\000\000\133\000\000\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\007\000\000\000\
    \000\000\133\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\114\000\000\000\115\000\255\255\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\008\000\000\000\117\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\007\000\000\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\114\000\167\000\115\000\000\000\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\167\000\000\000\000\000\000\000\
    \008\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\118\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\007\000\168\000\168\000\168\000\
    \168\000\168\000\168\000\168\000\168\000\168\000\168\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\114\000\000\000\115\000\000\000\000\000\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\000\000\000\000\000\000\000\000\008\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\119\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\007\000\000\000\225\000\225\000\225\000\
    \225\000\225\000\225\000\225\000\225\000\225\000\225\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\114\000\167\000\115\000\000\000\000\000\000\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\167\000\000\000\000\000\000\000\008\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\120\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\007\000\229\000\229\000\229\000\229\000\229\000\
    \229\000\229\000\229\000\229\000\229\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \114\000\000\000\115\000\000\000\000\000\000\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\000\000\000\000\000\000\000\000\008\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\121\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\007\000\229\000\229\000\229\000\229\000\229\000\229\000\
    \229\000\229\000\229\000\229\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\114\000\
    \000\000\115\000\000\000\000\000\000\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\000\000\000\000\008\000\000\000\008\000\122\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \007\000\023\001\023\001\023\001\023\001\023\001\023\001\023\001\
    \023\001\023\001\023\001\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\114\000\000\000\
    \115\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\123\000\008\000\008\000\008\000\008\000\008\000\007\000\
    \000\000\023\001\023\001\023\001\023\001\023\001\023\001\023\001\
    \023\001\023\001\023\001\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\114\000\024\001\115\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\024\001\000\000\
    \000\000\000\000\008\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\124\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\007\000\025\001\
    \025\001\025\001\025\001\025\001\025\001\025\001\025\001\025\001\
    \025\001\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\114\000\000\000\115\000\000\000\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\008\000\099\001\008\000\008\000\008\000\008\000\125\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\007\000\025\001\025\001\
    \025\001\025\001\025\001\025\001\025\001\025\001\025\001\025\001\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\114\000\000\000\115\000\000\000\098\001\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\000\000\000\000\000\000\000\000\
    \126\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\007\000\000\000\077\001\077\001\
    \077\001\077\001\077\001\077\001\077\001\077\001\077\001\077\001\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\114\000\024\001\115\000\000\000\000\000\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\024\001\000\000\000\000\000\000\127\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\129\000\230\000\000\000\230\000\000\000\
    \000\000\229\000\229\000\229\000\229\000\229\000\229\000\229\000\
    \229\000\229\000\229\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\129\000\000\000\000\000\000\000\007\000\000\000\
    \000\000\000\000\128\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\114\000\000\000\115\000\000\000\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\008\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\133\000\081\001\081\001\
    \081\001\081\001\081\001\081\001\081\001\081\001\081\001\081\001\
    \000\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\000\000\000\000\000\000\000\000\134\000\
    \000\000\000\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\000\000\000\000\000\000\000\000\
    \133\000\000\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\136\000\000\000\225\000\133\000\
    \227\000\227\000\227\000\227\000\227\000\227\000\227\000\227\000\
    \227\000\227\000\081\001\081\001\081\001\081\001\081\001\081\001\
    \081\001\081\001\081\001\081\001\000\000\226\000\000\000\000\000\
    \000\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\000\000\000\000\226\000\000\000\136\000\
    \000\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\000\000\000\000\137\000\092\001\
    \092\001\092\001\092\001\092\001\092\001\092\001\092\001\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\144\000\000\000\145\000\000\000\000\000\000\000\000\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\000\000\000\000\000\000\138\000\136\000\000\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\000\000\000\000\000\000\000\000\
    \138\000\000\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\093\001\093\001\093\001\
    \093\001\093\001\093\001\093\001\093\001\000\000\000\000\000\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\140\000\000\000\139\000\000\000\000\000\000\000\
    \000\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\000\000\000\000\000\000\142\000\138\000\
    \000\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\000\000\000\000\000\000\
    \000\000\142\000\000\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\000\000\000\000\
    \142\000\103\001\103\001\103\001\103\001\103\001\103\001\103\001\
    \103\001\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\000\000\000\000\000\000\000\000\143\000\
    \000\000\000\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\000\000\000\000\000\000\000\000\
    \142\000\000\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\146\000\000\000\026\001\142\000\
    \026\001\000\000\000\000\025\001\025\001\025\001\025\001\025\001\
    \025\001\025\001\025\001\025\001\025\001\000\000\000\000\000\000\
    \000\000\091\001\091\001\091\001\091\001\091\001\091\001\091\001\
    \091\001\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\000\000\000\000\000\000\000\000\146\000\
    \000\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\000\000\000\000\146\000\000\000\
    \000\000\090\001\000\000\000\000\000\000\000\000\000\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\000\000\000\000\000\000\000\000\147\000\000\000\000\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\000\000\000\000\000\000\000\000\146\000\000\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\149\000\000\000\082\001\146\000\082\001\000\000\
    \000\000\081\001\081\001\081\001\081\001\081\001\081\001\081\001\
    \081\001\081\001\081\001\104\001\104\001\104\001\104\001\104\001\
    \104\001\104\001\104\001\000\000\000\000\000\000\000\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\255\255\000\000\000\000\000\000\149\000\000\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\115\001\115\001\115\001\115\001\115\001\115\001\
    \115\001\115\001\000\000\000\000\000\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\144\000\
    \000\000\150\000\000\000\000\000\000\000\000\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \000\000\000\000\000\000\151\000\149\000\000\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\000\000\000\000\000\000\000\000\151\000\000\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\000\000\000\000\151\000\116\001\116\001\
    \116\001\116\001\116\001\116\001\116\001\116\001\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \000\000\000\000\000\000\000\000\147\000\000\000\000\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\000\000\000\000\000\000\000\000\151\000\000\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\007\000\000\000\000\000\151\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\114\000\
    \000\000\115\000\122\001\122\001\122\001\122\001\008\000\008\000\
    \008\000\008\000\154\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\123\001\121\001\008\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \007\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\114\000\000\000\
    \115\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\155\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\007\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\114\000\000\000\115\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\008\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\007\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\114\000\000\000\115\000\000\000\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\008\000\000\000\008\000\008\000\008\000\008\000\157\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\007\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\114\000\000\000\115\000\000\000\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\000\000\000\000\000\000\212\000\
    \008\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\155\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\000\000\000\000\
    \000\000\000\000\212\000\000\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\213\000\000\000\214\000\000\000\
    \000\000\000\000\000\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\000\000\000\000\000\000\
    \215\000\212\000\000\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\000\000\
    \000\000\000\000\000\000\215\000\000\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \000\000\000\000\215\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\000\000\000\000\000\000\
    \000\000\216\000\000\000\000\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\000\000\000\000\
    \000\000\000\000\215\000\000\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\218\000\000\000\
    \077\001\215\000\079\001\079\001\079\001\079\001\079\001\079\001\
    \079\001\079\001\079\001\079\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\078\001\
    \000\000\000\000\000\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\000\000\000\000\078\001\
    \000\000\218\000\000\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\213\000\000\000\219\000\000\000\000\000\
    \000\000\000\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\000\000\000\000\000\000\220\000\
    \218\000\000\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\000\000\000\000\
    \000\000\000\000\220\000\000\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\000\000\
    \000\000\220\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\000\000\000\000\000\000\000\000\
    \216\000\000\000\000\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\000\000\000\000\000\000\
    \000\000\220\000\000\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\000\000\000\000\000\000\
    \220\000\231\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \231\000\231\000\231\000\000\000\000\000\000\000\000\000\000\000\
    \085\001\000\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \000\000\085\001\225\000\000\000\228\000\228\000\228\000\228\000\
    \228\000\228\000\228\000\228\000\227\000\227\000\011\001\010\001\
    \010\001\010\001\010\001\110\001\110\001\110\001\110\001\000\000\
    \085\001\226\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \222\000\085\001\000\000\000\000\000\000\011\001\000\000\000\000\
    \000\000\223\000\000\000\111\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\012\001\000\000\000\000\000\000\
    \000\000\226\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \222\000\102\001\102\001\102\001\102\001\102\001\102\001\102\001\
    \102\001\223\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \231\000\231\000\231\000\231\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\231\000\231\000\231\000\231\000\231\000\
    \231\000\000\000\000\000\000\000\000\000\109\001\222\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\223\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\231\000\231\000\231\000\231\000\231\000\
    \231\000\011\001\010\001\010\001\010\001\010\001\222\000\000\000\
    \000\000\101\001\000\000\000\000\000\000\000\000\000\000\223\000\
    \114\001\114\001\114\001\114\001\114\001\114\001\114\001\114\001\
    \011\001\006\001\238\000\242\000\240\000\248\000\246\000\239\000\
    \005\001\004\001\250\000\249\000\253\000\255\000\254\000\009\001\
    \237\000\236\000\236\000\236\000\236\000\236\000\236\000\236\000\
    \236\000\236\000\243\000\252\000\251\000\007\001\247\000\000\000\
    \008\001\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\003\001\000\000\002\001\244\000\241\000\
    \113\001\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\001\001\245\000\000\001\077\001\000\000\
    \236\000\236\000\236\000\236\000\236\000\236\000\236\000\236\000\
    \236\000\236\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\073\001\078\001\000\000\000\000\
    \000\000\255\255\000\000\000\000\074\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\077\001\075\001\080\001\080\001\
    \080\001\080\001\080\001\080\001\080\001\080\001\079\001\079\001\
    \000\000\000\000\000\000\000\000\073\001\078\001\000\000\000\000\
    \000\000\000\000\073\001\078\001\074\001\000\000\000\000\000\000\
    \000\000\000\000\074\001\000\000\000\000\075\001\000\000\000\000\
    \000\000\000\000\000\000\075\001\000\000\000\000\076\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\073\001\078\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\074\001\000\000\240\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\075\001\000\000\000\000\076\001\000\000\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\052\001\000\000\053\001\000\000\000\000\000\000\
    \255\255\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\000\000\000\000\000\000\000\000\240\000\
    \000\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\240\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\052\001\000\000\053\001\000\000\000\000\000\000\000\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\000\000\000\000\000\000\042\001\241\000\000\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\000\000\000\000\000\000\000\000\
    \042\001\000\000\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\043\001\000\000\044\001\000\000\000\000\000\000\
    \000\000\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\000\000\000\000\000\000\045\001\042\001\
    \000\000\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\000\000\000\000\000\000\
    \000\000\045\001\000\000\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\000\000\000\000\
    \045\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\000\000\000\000\000\000\000\000\046\001\
    \000\000\000\000\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\000\000\000\000\000\000\000\000\
    \045\001\000\000\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\048\001\000\000\000\000\045\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\000\000\000\000\000\000\000\000\048\001\
    \000\000\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\043\001\000\000\049\001\000\000\000\000\000\000\000\000\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\000\000\000\000\000\000\050\001\048\001\000\000\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\000\000\000\000\000\000\000\000\
    \050\001\000\000\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\000\000\000\000\050\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\000\000\000\000\000\000\000\000\046\001\000\000\
    \000\000\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\000\000\000\000\000\000\000\000\050\001\
    \000\000\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\054\001\000\000\000\000\050\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\000\000\000\000\000\000\000\000\054\001\000\000\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\000\000\000\000\054\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \000\000\000\000\000\000\000\000\055\001\000\000\000\000\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\000\000\000\000\000\000\000\000\054\001\000\000\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\057\001\000\000\000\000\054\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \000\000\000\000\000\000\000\000\057\001\000\000\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\000\000\000\000\058\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\065\001\000\000\
    \066\001\000\000\000\000\000\000\000\000\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\000\000\
    \000\000\000\000\059\001\057\001\000\000\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\000\000\000\000\000\000\000\000\059\001\000\000\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\061\001\
    \000\000\060\001\000\000\000\000\000\000\000\000\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \000\000\000\000\000\000\063\001\059\001\000\000\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\000\000\000\000\000\000\000\000\063\001\000\000\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\000\000\000\000\063\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \000\000\000\000\000\000\000\000\064\001\000\000\000\000\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\000\000\000\000\000\000\000\000\063\001\000\000\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\067\001\000\000\000\000\063\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \000\000\000\000\000\000\000\000\067\001\000\000\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\000\000\000\000\067\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\000\000\000\000\
    \000\000\000\000\068\001\000\000\000\000\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\000\000\
    \000\000\000\000\000\000\067\001\000\000\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\070\001\
    \000\000\000\000\067\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\000\000\000\000\
    \000\000\000\000\070\001\000\000\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\065\001\000\000\071\001\000\000\
    \000\000\000\000\000\000\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\000\000\000\000\000\000\
    \072\001\070\001\000\000\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\000\000\
    \000\000\000\000\000\000\072\001\000\000\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \000\000\000\000\072\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\000\000\000\000\000\000\
    \000\000\068\001\000\000\000\000\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\000\000\000\000\
    \000\000\000\000\072\001\000\000\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\000\000\000\000\
    \000\000\072\001\083\001\083\001\083\001\083\001\083\001\083\001\
    \083\001\083\001\083\001\083\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\083\001\083\001\083\001\083\001\083\001\
    \083\001\000\000\000\000\077\001\000\000\080\001\080\001\080\001\
    \080\001\080\001\080\001\080\001\080\001\079\001\079\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\078\001\083\001\083\001\083\001\083\001\083\001\
    \083\001\074\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\075\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\078\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\074\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\075\001\083\001\083\001\083\001\083\001\083\001\
    \083\001\083\001\083\001\083\001\083\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\083\001\083\001\083\001\083\001\
    \083\001\083\001\000\000\000\000\000\000\000\000\000\000\074\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \075\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\083\001\083\001\083\001\083\001\
    \083\001\083\001\000\000\000\000\000\000\000\000\000\000\074\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \075\001\094\001\094\001\094\001\094\001\094\001\094\001\094\001\
    \094\001\094\001\094\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\094\001\094\001\094\001\094\001\094\001\094\001\
    \095\001\095\001\095\001\095\001\095\001\095\001\095\001\095\001\
    \095\001\095\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\095\001\095\001\095\001\095\001\095\001\095\001\000\000\
    \000\000\000\000\094\001\094\001\094\001\094\001\094\001\094\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\105\001\105\001\
    \105\001\105\001\105\001\105\001\105\001\105\001\105\001\105\001\
    \000\000\095\001\095\001\095\001\095\001\095\001\095\001\105\001\
    \105\001\105\001\105\001\105\001\105\001\106\001\106\001\106\001\
    \106\001\106\001\106\001\106\001\106\001\106\001\106\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\106\001\106\001\
    \106\001\106\001\106\001\106\001\000\000\000\000\000\000\105\001\
    \105\001\105\001\105\001\105\001\105\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\117\001\117\001\117\001\117\001\117\001\
    \117\001\117\001\117\001\117\001\117\001\000\000\106\001\106\001\
    \106\001\106\001\106\001\106\001\117\001\117\001\117\001\117\001\
    \117\001\117\001\118\001\118\001\118\001\118\001\118\001\118\001\
    \118\001\118\001\118\001\118\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\118\001\118\001\118\001\118\001\118\001\
    \118\001\000\000\000\000\000\000\117\001\117\001\117\001\117\001\
    \117\001\117\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\118\001\118\001\118\001\118\001\118\001\
    \118\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\191\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \009\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \010\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\011\000\012\000\012\000\014\000\015\000\
    \015\000\015\000\016\000\024\000\027\000\003\000\003\000\035\000\
    \041\000\042\000\047\000\050\000\051\000\003\000\048\000\029\000\
    \048\000\052\000\011\000\030\000\048\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\029\000\055\000\030\000\030\000\003\000\003\000\031\000\
    \043\000\044\000\031\000\004\000\004\000\003\000\056\000\049\000\
    \038\000\057\000\058\000\004\000\061\000\038\000\003\000\049\000\
    \031\000\031\000\025\000\031\000\004\000\025\000\062\000\004\000\
    \063\000\064\000\024\000\038\000\065\000\069\000\070\000\071\000\
    \072\000\058\000\075\000\004\000\004\000\076\000\077\000\078\000\
    \079\000\081\000\088\000\004\000\007\000\098\000\025\000\099\000\
    \025\000\025\000\082\000\065\000\004\000\083\000\084\000\004\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\090\000\007\000\091\000\094\000\097\000\
    \000\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\101\000\102\000\105\000\106\000\007\000\
    \108\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\008\000\025\000\109\000\111\000\086\000\
    \112\000\114\000\128\000\134\000\140\000\141\000\143\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\092\000\008\000\144\000\103\000\147\000\091\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\159\000\161\000\162\000\165\000\008\000\170\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\032\000\033\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\171\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\172\000\033\000\082\000\092\000\043\000\044\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\093\000\083\000\084\000\104\000\033\000\107\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\034\000\130\000\173\000\174\000\175\000\131\000\
    \132\000\090\000\177\000\091\000\094\000\097\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\086\000\034\000\110\000\178\000\184\000\188\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\201\000\202\000\213\000\216\000\034\000\093\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\036\000\242\000\037\000\243\000\053\000\244\000\248\000\
    \092\000\113\000\074\000\103\000\095\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \249\000\036\000\037\000\037\000\053\000\037\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \096\000\186\000\189\000\250\000\036\000\245\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \223\000\040\000\040\000\040\000\040\000\040\000\037\000\037\000\
    \053\000\053\000\059\000\037\000\074\000\053\000\037\000\095\000\
    \053\000\222\000\037\000\100\000\053\000\129\000\074\000\037\000\
    \040\000\053\000\222\000\040\000\246\000\100\000\232\000\160\000\
    \223\000\059\000\160\000\008\001\245\000\247\000\247\000\040\000\
    \093\000\012\001\130\000\104\000\129\000\107\000\131\000\132\000\
    \028\001\222\000\160\000\246\000\129\000\251\000\251\000\255\000\
    \255\000\034\001\222\000\043\001\096\000\046\001\232\000\052\001\
    \055\001\061\001\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\062\001\064\001\065\001\068\001\
    \059\000\110\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\123\001\255\255\
    \255\255\255\255\255\255\255\255\075\001\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\113\000\
    \074\000\006\001\095\000\060\000\075\001\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\066\000\
    \080\000\168\000\168\000\168\000\168\000\168\000\168\000\168\000\
    \168\000\168\000\168\000\255\255\013\001\009\001\096\000\186\000\
    \189\000\007\001\009\001\255\255\084\001\255\255\066\000\080\000\
    \255\255\080\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \009\001\167\000\006\001\167\000\255\255\080\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \255\255\255\255\255\255\080\000\084\001\255\255\255\255\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\255\255\007\001\255\255\255\255\066\000\255\255\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\067\000\255\255\086\001\255\255\255\255\255\255\255\255\
    \255\255\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\255\255\255\255\255\255\115\000\
    \067\000\255\255\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\255\255\255\255\
    \255\255\255\255\115\000\255\255\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\116\000\255\255\
    \255\255\115\000\013\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\255\255\116\000\086\001\
    \255\255\255\255\255\255\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\255\255\255\255\255\255\
    \255\255\116\000\255\255\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\117\000\255\255\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\158\000\117\000\255\255\255\255\
    \255\255\255\255\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\158\000\255\255\255\255\255\255\
    \117\000\255\255\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\118\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\255\255\
    \118\000\118\000\118\000\118\000\118\000\118\000\118\000\118\000\
    \118\000\118\000\118\000\255\255\118\000\255\255\255\255\255\255\
    \255\255\118\000\118\000\118\000\118\000\118\000\118\000\118\000\
    \118\000\118\000\118\000\118\000\118\000\118\000\118\000\118\000\
    \118\000\118\000\118\000\118\000\118\000\118\000\118\000\118\000\
    \118\000\118\000\118\000\255\255\255\255\255\255\255\255\118\000\
    \255\255\118\000\118\000\118\000\118\000\118\000\118\000\118\000\
    \118\000\118\000\118\000\118\000\118\000\118\000\118\000\118\000\
    \118\000\118\000\118\000\118\000\118\000\118\000\118\000\118\000\
    \118\000\118\000\118\000\119\000\255\255\225\000\225\000\225\000\
    \225\000\225\000\225\000\225\000\225\000\225\000\225\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\225\000\119\000\255\255\255\255\255\255\255\255\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\225\000\255\255\255\255\255\255\119\000\255\255\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\120\000\229\000\229\000\229\000\229\000\229\000\
    \229\000\229\000\229\000\229\000\229\000\255\255\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\255\255\120\000\255\255\255\255\255\255\255\255\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\255\255\255\255\255\255\255\255\120\000\255\255\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\121\000\230\000\230\000\230\000\230\000\230\000\230\000\
    \230\000\230\000\230\000\230\000\255\255\121\000\121\000\121\000\
    \121\000\121\000\121\000\121\000\121\000\121\000\121\000\121\000\
    \255\255\121\000\255\255\255\255\255\255\255\255\121\000\121\000\
    \121\000\121\000\121\000\121\000\121\000\121\000\121\000\121\000\
    \121\000\121\000\121\000\121\000\121\000\121\000\121\000\121\000\
    \121\000\121\000\121\000\121\000\121\000\121\000\121\000\121\000\
    \255\255\255\255\255\255\255\255\121\000\255\255\121\000\121\000\
    \121\000\121\000\121\000\121\000\121\000\121\000\121\000\121\000\
    \121\000\121\000\121\000\121\000\121\000\121\000\121\000\121\000\
    \121\000\121\000\121\000\121\000\121\000\121\000\121\000\121\000\
    \122\000\254\000\254\000\254\000\254\000\254\000\254\000\254\000\
    \254\000\254\000\254\000\255\255\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\255\255\
    \122\000\255\255\255\255\255\255\255\255\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\255\255\
    \255\255\255\255\255\255\122\000\255\255\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\123\000\
    \255\255\023\001\023\001\023\001\023\001\023\001\023\001\023\001\
    \023\001\023\001\023\001\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\023\001\123\000\
    \255\255\255\255\255\255\255\255\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\023\001\255\255\
    \255\255\255\255\123\000\255\255\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\124\000\025\001\
    \025\001\025\001\025\001\025\001\025\001\025\001\025\001\025\001\
    \025\001\255\255\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\255\255\124\000\255\255\
    \255\255\255\255\255\255\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\255\255\255\255\255\255\
    \255\255\124\000\096\001\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\125\000\026\001\026\001\
    \026\001\026\001\026\001\026\001\026\001\026\001\026\001\026\001\
    \255\255\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\255\255\125\000\255\255\096\001\
    \255\255\255\255\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\255\255\255\255\255\255\255\255\
    \125\000\255\255\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\126\000\255\255\077\001\077\001\
    \077\001\077\001\077\001\077\001\077\001\077\001\077\001\077\001\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\077\001\126\000\255\255\255\255\255\255\
    \255\255\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\077\001\255\255\255\255\255\255\126\000\
    \255\255\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\127\000\226\000\255\255\226\000\255\255\
    \255\255\226\000\226\000\226\000\226\000\226\000\226\000\226\000\
    \226\000\226\000\226\000\096\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\127\000\255\255\255\255\255\255\127\000\255\255\
    \255\255\255\255\127\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\255\255\127\000\255\255\
    \255\255\255\255\255\255\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\255\255\255\255\255\255\
    \255\255\127\000\255\255\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\133\000\081\001\081\001\
    \081\001\081\001\081\001\081\001\081\001\081\001\081\001\081\001\
    \255\255\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\255\255\255\255\255\255\255\255\133\000\
    \255\255\255\255\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\255\255\255\255\255\255\255\255\
    \133\000\255\255\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\135\000\255\255\227\000\133\000\
    \227\000\227\000\227\000\227\000\227\000\227\000\227\000\227\000\
    \227\000\227\000\082\001\082\001\082\001\082\001\082\001\082\001\
    \082\001\082\001\082\001\082\001\255\255\227\000\255\255\255\255\
    \255\255\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\255\255\255\255\227\000\255\255\135\000\
    \255\255\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\136\000\255\255\255\255\135\000\091\001\
    \091\001\091\001\091\001\091\001\091\001\091\001\091\001\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\255\255\136\000\255\255\255\255\255\255\255\255\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\255\255\255\255\255\255\137\000\136\000\255\255\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\255\255\255\255\255\255\255\255\
    \137\000\255\255\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\138\000\092\001\092\001\092\001\
    \092\001\092\001\092\001\092\001\092\001\255\255\255\255\255\255\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\255\255\138\000\255\255\255\255\255\255\
    \255\255\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\255\255\255\255\255\255\139\000\138\000\
    \255\255\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\255\255\255\255\255\255\
    \255\255\139\000\255\255\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\142\000\255\255\255\255\
    \139\000\102\001\102\001\102\001\102\001\102\001\102\001\102\001\
    \102\001\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\255\255\255\255\255\255\255\255\142\000\
    \255\255\255\255\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\255\255\255\255\255\255\255\255\
    \142\000\255\255\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\145\000\255\255\024\001\142\000\
    \024\001\255\255\255\255\024\001\024\001\024\001\024\001\024\001\
    \024\001\024\001\024\001\024\001\024\001\255\255\255\255\255\255\
    \255\255\088\001\088\001\088\001\088\001\088\001\088\001\088\001\
    \088\001\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\255\255\255\255\255\255\255\255\145\000\
    \255\255\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\146\000\255\255\255\255\145\000\255\255\
    \255\255\088\001\255\255\255\255\255\255\255\255\255\255\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\255\255\255\255\255\255\255\255\146\000\255\255\255\255\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\255\255\255\255\255\255\255\255\146\000\255\255\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\148\000\255\255\078\001\146\000\078\001\255\255\
    \255\255\078\001\078\001\078\001\078\001\078\001\078\001\078\001\
    \078\001\078\001\078\001\103\001\103\001\103\001\103\001\103\001\
    \103\001\103\001\103\001\255\255\255\255\255\255\255\255\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\088\001\255\255\255\255\255\255\148\000\255\255\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\149\000\114\001\114\001\114\001\114\001\114\001\114\001\
    \114\001\114\001\255\255\255\255\255\255\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \255\255\149\000\255\255\255\255\255\255\255\255\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \255\255\255\255\255\255\150\000\149\000\255\255\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\255\255\255\255\255\255\255\255\150\000\255\255\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\151\000\255\255\255\255\150\000\115\001\115\001\
    \115\001\115\001\115\001\115\001\115\001\115\001\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \255\255\255\255\255\255\255\255\151\000\255\255\255\255\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\255\255\255\255\255\255\255\255\151\000\255\255\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\153\000\255\255\255\255\151\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \255\255\153\000\119\001\119\001\119\001\119\001\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \255\255\255\255\119\001\119\001\153\000\255\255\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \154\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\255\255\
    \154\000\255\255\255\255\255\255\255\255\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\255\255\
    \255\255\255\255\255\255\154\000\255\255\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\155\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\255\255\155\000\
    \255\255\255\255\255\255\255\255\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\255\255\255\255\
    \255\255\255\255\155\000\255\255\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\156\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \119\001\255\255\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\255\255\156\000\255\255\
    \255\255\255\255\255\255\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\255\255\255\255\255\255\
    \255\255\156\000\255\255\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\157\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\255\255\157\000\255\255\255\255\
    \255\255\255\255\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\255\255\255\255\255\255\211\000\
    \157\000\255\255\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\211\000\211\000\211\000\211\000\
    \211\000\211\000\211\000\211\000\211\000\211\000\211\000\211\000\
    \211\000\211\000\211\000\211\000\211\000\211\000\211\000\211\000\
    \211\000\211\000\211\000\211\000\211\000\211\000\255\255\255\255\
    \255\255\255\255\211\000\255\255\211\000\211\000\211\000\211\000\
    \211\000\211\000\211\000\211\000\211\000\211\000\211\000\211\000\
    \211\000\211\000\211\000\211\000\211\000\211\000\211\000\211\000\
    \211\000\211\000\211\000\211\000\211\000\211\000\212\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\255\255\212\000\255\255\
    \255\255\255\255\255\255\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\255\255\255\255\255\255\
    \214\000\212\000\255\255\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\212\000\212\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\214\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\214\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\214\000\214\000\214\000\255\255\
    \255\255\255\255\255\255\214\000\255\255\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\214\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\214\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\214\000\214\000\214\000\215\000\
    \255\255\255\255\214\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\255\255\255\255\255\255\
    \255\255\215\000\255\255\255\255\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\255\255\255\255\
    \255\255\255\255\215\000\255\255\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\217\000\255\255\
    \079\001\215\000\079\001\079\001\079\001\079\001\079\001\079\001\
    \079\001\079\001\079\001\079\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\079\001\
    \255\255\255\255\255\255\217\000\217\000\217\000\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\255\255\255\255\079\001\
    \255\255\217\000\255\255\217\000\217\000\217\000\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\218\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\255\255\218\000\255\255\255\255\
    \255\255\255\255\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\255\255\255\255\255\255\219\000\
    \218\000\255\255\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\255\255\255\255\
    \255\255\255\255\219\000\255\255\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\220\000\255\255\
    \255\255\219\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\255\255\255\255\255\255\255\255\
    \220\000\255\255\255\255\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\255\255\255\255\255\255\
    \255\255\220\000\255\255\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\255\255\255\255\255\255\
    \220\000\224\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\224\000\255\255\255\255\255\255\255\255\255\255\
    \074\001\255\255\224\000\224\000\224\000\224\000\224\000\224\000\
    \255\255\074\001\228\000\255\255\228\000\228\000\228\000\228\000\
    \228\000\228\000\228\000\228\000\228\000\228\000\011\001\011\001\
    \011\001\011\001\011\001\107\001\107\001\107\001\107\001\255\255\
    \074\001\228\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \228\000\074\001\255\255\255\255\255\255\011\001\255\255\255\255\
    \255\255\228\000\255\255\107\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\011\001\255\255\255\255\255\255\
    \255\255\228\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \228\000\098\001\098\001\098\001\098\001\098\001\098\001\098\001\
    \098\001\228\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \231\000\231\000\231\000\231\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\231\000\231\000\231\000\231\000\231\000\
    \231\000\255\255\255\255\255\255\255\255\107\001\231\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\231\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\231\000\231\000\231\000\231\000\231\000\
    \231\000\234\000\234\000\234\000\234\000\234\000\231\000\255\255\
    \255\255\098\001\255\255\255\255\255\255\255\255\255\255\231\000\
    \109\001\109\001\109\001\109\001\109\001\109\001\109\001\109\001\
    \234\000\234\000\234\000\234\000\234\000\234\000\234\000\234\000\
    \234\000\234\000\234\000\234\000\234\000\234\000\234\000\234\000\
    \234\000\234\000\234\000\234\000\234\000\234\000\234\000\234\000\
    \234\000\234\000\234\000\234\000\234\000\234\000\234\000\255\255\
    \234\000\234\000\234\000\234\000\234\000\234\000\234\000\234\000\
    \234\000\234\000\234\000\234\000\234\000\234\000\234\000\234\000\
    \234\000\234\000\234\000\234\000\234\000\234\000\234\000\234\000\
    \234\000\234\000\234\000\234\000\255\255\234\000\234\000\234\000\
    \109\001\234\000\234\000\234\000\234\000\234\000\234\000\234\000\
    \234\000\234\000\234\000\234\000\234\000\234\000\234\000\234\000\
    \234\000\234\000\234\000\234\000\234\000\234\000\234\000\234\000\
    \234\000\234\000\234\000\234\000\234\000\234\000\236\000\255\255\
    \236\000\236\000\236\000\236\000\236\000\236\000\236\000\236\000\
    \236\000\236\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\107\001\255\255\255\255\236\000\236\000\255\255\255\255\
    \255\255\098\001\255\255\255\255\236\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\237\000\236\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \255\255\255\255\255\255\255\255\236\000\236\000\255\255\255\255\
    \255\255\255\255\237\000\237\000\236\000\255\255\255\255\255\255\
    \255\255\255\255\237\000\255\255\255\255\236\000\255\255\255\255\
    \255\255\255\255\255\255\237\000\255\255\255\255\237\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\237\000\237\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\237\000\255\255\240\000\255\255\255\255\255\255\
    \109\001\255\255\255\255\237\000\255\255\255\255\237\000\255\255\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\255\255\240\000\255\255\255\255\255\255\
    \234\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\255\255\255\255\255\255\255\255\240\000\
    \255\255\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\241\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\255\255\241\000\255\255\255\255\255\255\255\255\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\255\255\255\255\255\255\041\001\241\000\255\255\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\041\001\041\001\041\001\041\001\041\001\041\001\
    \041\001\041\001\041\001\041\001\041\001\041\001\041\001\041\001\
    \041\001\041\001\041\001\041\001\041\001\041\001\041\001\041\001\
    \041\001\041\001\041\001\041\001\255\255\255\255\255\255\255\255\
    \041\001\255\255\041\001\041\001\041\001\041\001\041\001\041\001\
    \041\001\041\001\041\001\041\001\041\001\041\001\041\001\041\001\
    \041\001\041\001\041\001\041\001\041\001\041\001\041\001\041\001\
    \041\001\041\001\041\001\041\001\042\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\255\255\042\001\255\255\255\255\255\255\
    \255\255\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\255\255\255\255\255\255\044\001\042\001\
    \255\255\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\042\001\042\001\042\001\042\001\042\001\
    \042\001\042\001\042\001\044\001\044\001\044\001\044\001\044\001\
    \044\001\044\001\044\001\044\001\044\001\044\001\044\001\044\001\
    \044\001\044\001\044\001\044\001\044\001\044\001\044\001\044\001\
    \044\001\044\001\044\001\044\001\044\001\255\255\255\255\255\255\
    \255\255\044\001\255\255\044\001\044\001\044\001\044\001\044\001\
    \044\001\044\001\044\001\044\001\044\001\044\001\044\001\044\001\
    \044\001\044\001\044\001\044\001\044\001\044\001\044\001\044\001\
    \044\001\044\001\044\001\044\001\044\001\045\001\255\255\255\255\
    \044\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\255\255\255\255\255\255\255\255\045\001\
    \255\255\255\255\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\255\255\255\255\255\255\255\255\
    \045\001\255\255\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\045\001\045\001\045\001\045\001\
    \045\001\045\001\045\001\045\001\047\001\255\255\255\255\045\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\047\001\047\001\047\001\047\001\047\001\047\001\047\001\
    \047\001\047\001\047\001\047\001\047\001\047\001\047\001\047\001\
    \047\001\047\001\047\001\047\001\047\001\047\001\047\001\047\001\
    \047\001\047\001\047\001\255\255\255\255\255\255\255\255\047\001\
    \255\255\047\001\047\001\047\001\047\001\047\001\047\001\047\001\
    \047\001\047\001\047\001\047\001\047\001\047\001\047\001\047\001\
    \047\001\047\001\047\001\047\001\047\001\047\001\047\001\047\001\
    \047\001\047\001\047\001\048\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\255\255\048\001\255\255\255\255\255\255\255\255\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\255\255\255\255\255\255\049\001\048\001\255\255\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \048\001\048\001\049\001\049\001\049\001\049\001\049\001\049\001\
    \049\001\049\001\049\001\049\001\049\001\049\001\049\001\049\001\
    \049\001\049\001\049\001\049\001\049\001\049\001\049\001\049\001\
    \049\001\049\001\049\001\049\001\255\255\255\255\255\255\255\255\
    \049\001\255\255\049\001\049\001\049\001\049\001\049\001\049\001\
    \049\001\049\001\049\001\049\001\049\001\049\001\049\001\049\001\
    \049\001\049\001\049\001\049\001\049\001\049\001\049\001\049\001\
    \049\001\049\001\049\001\049\001\050\001\255\255\255\255\049\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\255\255\255\255\255\255\255\255\050\001\255\255\
    \255\255\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\255\255\255\255\255\255\255\255\050\001\
    \255\255\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\050\001\053\001\255\255\255\255\050\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \053\001\053\001\053\001\053\001\053\001\053\001\053\001\053\001\
    \053\001\053\001\053\001\053\001\053\001\053\001\053\001\053\001\
    \053\001\053\001\053\001\053\001\053\001\053\001\053\001\053\001\
    \053\001\053\001\255\255\255\255\255\255\255\255\053\001\255\255\
    \053\001\053\001\053\001\053\001\053\001\053\001\053\001\053\001\
    \053\001\053\001\053\001\053\001\053\001\053\001\053\001\053\001\
    \053\001\053\001\053\001\053\001\053\001\053\001\053\001\053\001\
    \053\001\053\001\054\001\255\255\255\255\053\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \255\255\255\255\255\255\255\255\054\001\255\255\255\255\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\255\255\255\255\255\255\255\255\054\001\255\255\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\054\001\054\001\054\001\
    \054\001\056\001\255\255\255\255\054\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\056\001\056\001\
    \056\001\056\001\056\001\056\001\056\001\056\001\056\001\056\001\
    \056\001\056\001\056\001\056\001\056\001\056\001\056\001\056\001\
    \056\001\056\001\056\001\056\001\056\001\056\001\056\001\056\001\
    \255\255\255\255\255\255\255\255\056\001\255\255\056\001\056\001\
    \056\001\056\001\056\001\056\001\056\001\056\001\056\001\056\001\
    \056\001\056\001\056\001\056\001\056\001\056\001\056\001\056\001\
    \056\001\056\001\056\001\056\001\056\001\056\001\056\001\056\001\
    \057\001\255\255\255\255\056\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\255\255\
    \057\001\255\255\255\255\255\255\255\255\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\255\255\
    \255\255\255\255\058\001\057\001\255\255\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\057\001\
    \057\001\057\001\057\001\057\001\057\001\057\001\057\001\058\001\
    \058\001\058\001\058\001\058\001\058\001\058\001\058\001\058\001\
    \058\001\058\001\058\001\058\001\058\001\058\001\058\001\058\001\
    \058\001\058\001\058\001\058\001\058\001\058\001\058\001\058\001\
    \058\001\255\255\255\255\255\255\255\255\058\001\255\255\058\001\
    \058\001\058\001\058\001\058\001\058\001\058\001\058\001\058\001\
    \058\001\058\001\058\001\058\001\058\001\058\001\058\001\058\001\
    \058\001\058\001\058\001\058\001\058\001\058\001\058\001\058\001\
    \058\001\059\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \255\255\059\001\255\255\255\255\255\255\255\255\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \255\255\255\255\255\255\060\001\059\001\255\255\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \060\001\060\001\060\001\060\001\060\001\060\001\060\001\060\001\
    \060\001\060\001\060\001\060\001\060\001\060\001\060\001\060\001\
    \060\001\060\001\060\001\060\001\060\001\060\001\060\001\060\001\
    \060\001\060\001\255\255\255\255\255\255\255\255\060\001\255\255\
    \060\001\060\001\060\001\060\001\060\001\060\001\060\001\060\001\
    \060\001\060\001\060\001\060\001\060\001\060\001\060\001\060\001\
    \060\001\060\001\060\001\060\001\060\001\060\001\060\001\060\001\
    \060\001\060\001\063\001\255\255\255\255\060\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \255\255\255\255\255\255\255\255\063\001\255\255\255\255\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\255\255\255\255\255\255\255\255\063\001\255\255\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\066\001\255\255\255\255\063\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\066\001\066\001\
    \066\001\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \066\001\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \066\001\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \255\255\255\255\255\255\255\255\066\001\255\255\066\001\066\001\
    \066\001\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \066\001\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \066\001\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \067\001\255\255\255\255\066\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\255\255\255\255\
    \255\255\255\255\067\001\255\255\255\255\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\255\255\
    \255\255\255\255\255\255\067\001\255\255\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\069\001\
    \255\255\255\255\067\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\069\001\069\001\069\001\069\001\
    \069\001\069\001\069\001\069\001\069\001\069\001\069\001\069\001\
    \069\001\069\001\069\001\069\001\069\001\069\001\069\001\069\001\
    \069\001\069\001\069\001\069\001\069\001\069\001\255\255\255\255\
    \255\255\255\255\069\001\255\255\069\001\069\001\069\001\069\001\
    \069\001\069\001\069\001\069\001\069\001\069\001\069\001\069\001\
    \069\001\069\001\069\001\069\001\069\001\069\001\069\001\069\001\
    \069\001\069\001\069\001\069\001\069\001\069\001\070\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\255\255\070\001\255\255\
    \255\255\255\255\255\255\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\255\255\255\255\255\255\
    \071\001\070\001\255\255\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\070\001\070\001\070\001\
    \070\001\070\001\070\001\070\001\070\001\071\001\071\001\071\001\
    \071\001\071\001\071\001\071\001\071\001\071\001\071\001\071\001\
    \071\001\071\001\071\001\071\001\071\001\071\001\071\001\071\001\
    \071\001\071\001\071\001\071\001\071\001\071\001\071\001\255\255\
    \255\255\255\255\255\255\071\001\255\255\071\001\071\001\071\001\
    \071\001\071\001\071\001\071\001\071\001\071\001\071\001\071\001\
    \071\001\071\001\071\001\071\001\071\001\071\001\071\001\071\001\
    \071\001\071\001\071\001\071\001\071\001\071\001\071\001\072\001\
    \255\255\255\255\071\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\255\255\255\255\255\255\
    \255\255\072\001\255\255\255\255\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\255\255\255\255\
    \255\255\255\255\072\001\255\255\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\255\255\255\255\
    \255\255\072\001\076\001\076\001\076\001\076\001\076\001\076\001\
    \076\001\076\001\076\001\076\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\076\001\076\001\076\001\076\001\076\001\
    \076\001\255\255\255\255\080\001\255\255\080\001\080\001\080\001\
    \080\001\080\001\080\001\080\001\080\001\080\001\080\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\080\001\076\001\076\001\076\001\076\001\076\001\
    \076\001\080\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\080\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\080\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\080\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\080\001\083\001\083\001\083\001\083\001\083\001\
    \083\001\083\001\083\001\083\001\083\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\083\001\083\001\083\001\083\001\
    \083\001\083\001\255\255\255\255\255\255\255\255\255\255\083\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \083\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\083\001\083\001\083\001\083\001\
    \083\001\083\001\255\255\255\255\255\255\255\255\255\255\083\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \083\001\090\001\090\001\090\001\090\001\090\001\090\001\090\001\
    \090\001\090\001\090\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\090\001\090\001\090\001\090\001\090\001\090\001\
    \094\001\094\001\094\001\094\001\094\001\094\001\094\001\094\001\
    \094\001\094\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\094\001\094\001\094\001\094\001\094\001\094\001\255\255\
    \255\255\255\255\090\001\090\001\090\001\090\001\090\001\090\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\101\001\101\001\
    \101\001\101\001\101\001\101\001\101\001\101\001\101\001\101\001\
    \255\255\094\001\094\001\094\001\094\001\094\001\094\001\101\001\
    \101\001\101\001\101\001\101\001\101\001\105\001\105\001\105\001\
    \105\001\105\001\105\001\105\001\105\001\105\001\105\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\105\001\105\001\
    \105\001\105\001\105\001\105\001\255\255\255\255\255\255\101\001\
    \101\001\101\001\101\001\101\001\101\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\113\001\113\001\113\001\113\001\113\001\
    \113\001\113\001\113\001\113\001\113\001\255\255\105\001\105\001\
    \105\001\105\001\105\001\105\001\113\001\113\001\113\001\113\001\
    \113\001\113\001\117\001\117\001\117\001\117\001\117\001\117\001\
    \117\001\117\001\117\001\117\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\117\001\117\001\117\001\117\001\117\001\
    \117\001\255\255\255\255\255\255\113\001\113\001\113\001\113\001\
    \113\001\113\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\117\001\117\001\117\001\117\001\117\001\
    \117\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\002\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\004\000\005\000\000\000\038\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\006\000\009\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\018\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\033\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\001\000\015\000\015\000\028\000\028\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\001\000\015\000\015\000\028\000\028\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\023\000\012\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\040\000\058\000\059\000\065\000\066\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\040\000\058\000\059\000\065\000\066\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\064\000\057\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_code = 
   "\255\003\255\002\255\255\000\002\255\000\003\255\004\255\255\005\
    \255\255\000\004\001\005\255\007\255\006\255\255\009\255\008\255\
    \255\000\006\001\008\255\000\007\001\009\255";
}

let rec token lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 10 (-1) ; (* L=2 [3] <- p ; [2] <- p ;  *)
  lexbuf.Lexing.lex_mem.(3) <- lexbuf.Lexing.lex_curr_pos ;
  lexbuf.Lexing.lex_mem.(2) <- lexbuf.Lexing.lex_curr_pos ;
  __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 565 "lexer_cocci.mll"
    ( let cls = !current_line_started in
      if not cls
      then
	begin
	  match !current_line_type with
	    (D.PLUS,_,_) | (D.PLUSPLUS,_,_) ->
	      (* increment the logical line even though nothing seen *)
	      start_line true;
	      let info = get_current_line_type lexbuf in
	      reset_line lexbuf;
	      TDirective (Ast.Noindent "", info)
	  | _ -> reset_line lexbuf; token lexbuf
	end
      else (reset_line lexbuf; token lexbuf) )
# 2706 "lexer_cocci.ml"

  | 1 ->
let
# 580 "lexer_cocci.mll"
                      w
# 2712 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 580 "lexer_cocci.mll"
                         ( (* collect whitespaces only when inside a rule *)
    start_line false;
    if !Data.in_rule_name || !Data.in_prolog || !Data.in_iso
    then token lexbuf
    else TWhitespace w )
# 2720 "lexer_cocci.ml"

  | 2 ->
let
# 586 "lexer_cocci.mll"
                                       after
# 2726 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 586 "lexer_cocci.mll"
                                              (
    match !current_line_type with
      (D.PLUS,_,_) | (D.PLUSPLUS,_,_) ->
	let str =
	  if !current_line_started
	  then (tok lexbuf)
	  else after in
	start_line true;
	TDirective (Ast.Indent str, get_current_line_type lexbuf)
    | _ -> start_line false; token lexbuf )
# 2739 "lexer_cocci.ml"

  | 3 ->
let
# 597 "lexer_cocci.mll"
                                       after
# 2745 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 597 "lexer_cocci.mll"
                                              (
     check_context_linetype after;
     start_line false; token lexbuf )
# 2751 "lexer_cocci.ml"

  | 4 ->
# 602 "lexer_cocci.mll"
   ( match !current_line_type with
      (D.PLUS,_,_) | (D.PLUSPLUS,_,_) ->
	start_line true;
	TDirective (Ast.Space (tok lexbuf), get_current_line_type lexbuf)
    | _ -> failwith "attributes only allowed in + code" )
# 2760 "lexer_cocci.ml"

  | 5 ->
# 608 "lexer_cocci.mll"
         ( start_line true; TArobArob )
# 2765 "lexer_cocci.ml"

  | 6 ->
# 609 "lexer_cocci.mll"
         ( pass_zero();
	   if !Data.in_rule_name || not !current_line_started
	   then (start_line true; TArob)
	   else (check_minus_context_linetype "@";
		 TPArob (get_current_line_type lexbuf)) )
# 2774 "lexer_cocci.ml"

  | 7 ->
# 616 "lexer_cocci.mll"
      ( start_line true; check_minus_context_linetype (tok lexbuf);
	TWhen (get_current_line_type lexbuf) )
# 2780 "lexer_cocci.ml"

  | 8 ->
# 620 "lexer_cocci.mll"
      ( start_line true; check_minus_context_linetype (tok lexbuf);
	TEllipsis (get_current_line_type lexbuf) )
# 2786 "lexer_cocci.ml"

  | 9 ->
# 622 "lexer_cocci.mll"
           ( start_line true; check_context_linetype (tok lexbuf);
	     TOEllipsis (get_current_line_type lexbuf) )
# 2792 "lexer_cocci.ml"

  | 10 ->
# 624 "lexer_cocci.mll"
           ( start_line true; check_context_linetype (tok lexbuf);
	     TCEllipsis (get_current_line_type lexbuf) )
# 2798 "lexer_cocci.ml"

  | 11 ->
# 626 "lexer_cocci.mll"
            ( start_line true; check_minus_context_linetype (tok lexbuf);
	     TPOEllipsis (get_current_line_type lexbuf) )
# 2804 "lexer_cocci.ml"

  | 12 ->
# 628 "lexer_cocci.mll"
            ( start_line true; check_minus_context_linetype (tok lexbuf);
	     TPCEllipsis (get_current_line_type lexbuf) )
# 2810 "lexer_cocci.ml"

  | 13 ->
# 630 "lexer_cocci.mll"
        ( pass_zero();
	  if !current_line_started
	  then (start_line true; TMinus (get_current_line_type lexbuf))
          else (patch_or_match PATCH;
		add_current_line_type D.MINUS; token lexbuf) )
# 2819 "lexer_cocci.ml"

  | 14 ->
# 635 "lexer_cocci.mll"
        ( pass_zero();
	  if !current_line_started
	  then (start_line true; TPlus (get_current_line_type lexbuf))
          else (patch_or_match PATCH;
		add_current_line_type D.PLUS; token lexbuf) )
# 2828 "lexer_cocci.ml"

  | 15 ->
# 640 "lexer_cocci.mll"
        ( pass_zero();
	  if !current_line_started
	  then (start_line true; TWhy (get_current_line_type lexbuf))
          else (add_current_line_type D.OPT; token lexbuf) )
# 2836 "lexer_cocci.ml"

  | 16 ->
# 644 "lexer_cocci.mll"
        ( start_line true; TBang (get_current_line_type lexbuf) )
# 2841 "lexer_cocci.ml"

  | 17 ->
# 645 "lexer_cocci.mll"
        ( if not !col_zero
	  then (start_line true; TOPar (get_current_line_type lexbuf))
          else
            (start_line true; check_context_linetype (tok lexbuf);
	     TOPar0 ("(",get_current_line_type lexbuf)))
# 2850 "lexer_cocci.ml"

  | 18 ->
# 650 "lexer_cocci.mll"
          ( start_line true;
	    TOPar0 ("\\(",contextify(get_current_line_type lexbuf)) )
# 2856 "lexer_cocci.ml"

  | 19 ->
# 652 "lexer_cocci.mll"
        ( if not (!col_zero)
	  then (start_line true; TOr(get_current_line_type lexbuf))
          else (start_line true;
		check_context_linetype (tok lexbuf);
		TMid0 ("|",get_current_line_type lexbuf)))
# 2865 "lexer_cocci.ml"

  | 20 ->
# 657 "lexer_cocci.mll"
          ( start_line true;
	    TMid0 ("\\|",contextify(get_current_line_type lexbuf)) )
# 2871 "lexer_cocci.ml"

  | 21 ->
# 659 "lexer_cocci.mll"
        ( if not !col_zero
	  then (start_line true; TCPar (get_current_line_type lexbuf))
          else
            (start_line true; check_context_linetype (tok lexbuf);
	     TCPar0 (")",get_current_line_type lexbuf)))
# 2880 "lexer_cocci.ml"

  | 22 ->
# 664 "lexer_cocci.mll"
          ( start_line true;
	    TCPar0 ("\\)",contextify(get_current_line_type lexbuf)) )
# 2886 "lexer_cocci.ml"

  | 23 ->
# 667 "lexer_cocci.mll"
        ( start_line true; TOCro (get_current_line_type lexbuf)   )
# 2891 "lexer_cocci.ml"

  | 24 ->
# 668 "lexer_cocci.mll"
        ( start_line true; TCCro (get_current_line_type lexbuf)   )
# 2896 "lexer_cocci.ml"

  | 25 ->
# 669 "lexer_cocci.mll"
        ( start_line true; TOBrace (get_current_line_type lexbuf) )
# 2901 "lexer_cocci.ml"

  | 26 ->
# 670 "lexer_cocci.mll"
        ( start_line true; TCBrace (get_current_line_type lexbuf) )
# 2906 "lexer_cocci.ml"

  | 27 ->
# 672 "lexer_cocci.mll"
                   ( start_line true; TPtrOp (get_current_line_type lexbuf)  )
# 2911 "lexer_cocci.ml"

  | 28 ->
# 673 "lexer_cocci.mll"
                   ( start_line true; TDot (get_current_line_type lexbuf)    )
# 2916 "lexer_cocci.ml"

  | 29 ->
# 674 "lexer_cocci.mll"
                   ( start_line true; TComma (get_current_line_type lexbuf)  )
# 2921 "lexer_cocci.ml"

  | 30 ->
# 675 "lexer_cocci.mll"
               ( start_line true; TVAEllipsis (get_current_line_type lexbuf) )
# 2926 "lexer_cocci.ml"

  | 31 ->
# 676 "lexer_cocci.mll"
                   ( start_line true; TPtVirg (get_current_line_type lexbuf) )
# 2931 "lexer_cocci.ml"

  | 32 ->
# 679 "lexer_cocci.mll"
                   ( pass_zero();
		     if !current_line_started
		     then
		       (start_line true; TMul (get_current_line_type lexbuf))
		     else
		       (patch_or_match MATCH;
			add_current_line_type D.MINUS; token lexbuf) )
# 2942 "lexer_cocci.ml"

  | 33 ->
# 686 "lexer_cocci.mll"
                   ( start_line true;
		     TDmOp (Ast.Div,get_current_line_type lexbuf) )
# 2948 "lexer_cocci.ml"

  | 34 ->
# 688 "lexer_cocci.mll"
                    ( start_line true;
		     TDmOp (Ast.Min,get_current_line_type lexbuf) )
# 2954 "lexer_cocci.ml"

  | 35 ->
# 690 "lexer_cocci.mll"
                    ( start_line true;
		     TDmOp (Ast.Max,get_current_line_type lexbuf) )
# 2960 "lexer_cocci.ml"

  | 36 ->
# 692 "lexer_cocci.mll"
                   ( start_line true;
		     TDmOp (Ast.Mod,get_current_line_type lexbuf) )
# 2966 "lexer_cocci.ml"

  | 37 ->
# 694 "lexer_cocci.mll"
                   ( start_line true;  TTilde (get_current_line_type lexbuf) )
# 2971 "lexer_cocci.ml"

  | 38 ->
# 696 "lexer_cocci.mll"
                   ( pass_zero();
 		     if !current_line_started
 		     then
 		       (start_line true; TInc (get_current_line_type lexbuf))
 		     else (patch_or_match PATCH;
 			   add_current_line_type D.PLUSPLUS; token lexbuf) )
# 2981 "lexer_cocci.ml"

  | 39 ->
# 702 "lexer_cocci.mll"
                   ( start_line true;  TDec (get_current_line_type lexbuf) )
# 2986 "lexer_cocci.ml"

  | 40 ->
# 704 "lexer_cocci.mll"
                   ( start_line true; TEq (get_current_line_type lexbuf) )
# 2991 "lexer_cocci.ml"

  | 41 ->
# 706 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Minus lexbuf )
# 2996 "lexer_cocci.ml"

  | 42 ->
# 707 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Plus lexbuf )
# 3001 "lexer_cocci.ml"

  | 43 ->
# 709 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Mul lexbuf )
# 3006 "lexer_cocci.ml"

  | 44 ->
# 710 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Div lexbuf )
# 3011 "lexer_cocci.ml"

  | 45 ->
# 711 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Mod lexbuf )
# 3016 "lexer_cocci.ml"

  | 46 ->
# 713 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.And lexbuf )
# 3021 "lexer_cocci.ml"

  | 47 ->
# 714 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Or lexbuf )
# 3026 "lexer_cocci.ml"

  | 48 ->
# 715 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Xor lexbuf )
# 3031 "lexer_cocci.ml"

  | 49 ->
# 716 "lexer_cocci.mll"
                    ( start_line true; mkassign Ast.Max lexbuf )
# 3036 "lexer_cocci.ml"

  | 50 ->
# 717 "lexer_cocci.mll"
                    ( start_line true; mkassign Ast.Min lexbuf )
# 3041 "lexer_cocci.ml"

  | 51 ->
# 719 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.DecLeft lexbuf )
# 3046 "lexer_cocci.ml"

  | 52 ->
# 720 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.DecRight lexbuf )
# 3051 "lexer_cocci.ml"

  | 53 ->
# 722 "lexer_cocci.mll"
                   ( start_line true; TDotDot (get_current_line_type lexbuf) )
# 3056 "lexer_cocci.ml"

  | 54 ->
# 724 "lexer_cocci.mll"
                   ( start_line true; TEqEq    (get_current_line_type lexbuf) )
# 3061 "lexer_cocci.ml"

  | 55 ->
# 725 "lexer_cocci.mll"
                   ( start_line true; TNotEq   (get_current_line_type lexbuf) )
# 3066 "lexer_cocci.ml"

  | 56 ->
# 726 "lexer_cocci.mll"
                   ( start_line true;
		     TLogOp(Ast.SupEq,get_current_line_type lexbuf) )
# 3072 "lexer_cocci.ml"

  | 57 ->
# 728 "lexer_cocci.mll"
                   ( start_line true;
		     TLogOp(Ast.InfEq,get_current_line_type lexbuf) )
# 3078 "lexer_cocci.ml"

  | 58 ->
# 730 "lexer_cocci.mll"
                   ( start_line true;
		     TLogOp(Ast.Inf,get_current_line_type lexbuf) )
# 3084 "lexer_cocci.ml"

  | 59 ->
# 732 "lexer_cocci.mll"
                   ( start_line true;
		     TLogOp(Ast.Sup,get_current_line_type lexbuf) )
# 3090 "lexer_cocci.ml"

  | 60 ->
# 735 "lexer_cocci.mll"
                   ( start_line true; TAndLog (get_current_line_type lexbuf) )
# 3095 "lexer_cocci.ml"

  | 61 ->
# 736 "lexer_cocci.mll"
                   ( start_line true; TOrLog  (get_current_line_type lexbuf) )
# 3100 "lexer_cocci.ml"

  | 62 ->
# 738 "lexer_cocci.mll"
                   ( start_line true;
		     TShROp(Ast.DecRight,get_current_line_type lexbuf) )
# 3106 "lexer_cocci.ml"

  | 63 ->
# 740 "lexer_cocci.mll"
                   ( start_line true;
		     TShLOp(Ast.DecLeft,get_current_line_type lexbuf) )
# 3112 "lexer_cocci.ml"

  | 64 ->
# 743 "lexer_cocci.mll"
                   ( if not (!col_zero)
                     then (start_line true; TAnd(get_current_line_type lexbuf))
                     else (start_line true;
	                   check_context_linetype (tok lexbuf);
		           TAnd0 ("&",get_current_line_type lexbuf)))
# 3121 "lexer_cocci.ml"

  | 65 ->
# 748 "lexer_cocci.mll"
                   ( start_line true;
	             TAnd0 ("\\&",contextify(get_current_line_type lexbuf)) )
# 3127 "lexer_cocci.ml"

  | 66 ->
# 751 "lexer_cocci.mll"
                   ( start_line true; TXor(get_current_line_type lexbuf) )
# 3132 "lexer_cocci.ml"

  | 67 ->
# 753 "lexer_cocci.mll"
                    ( start_line true; TCppConcatOp )
# 3137 "lexer_cocci.ml"

  | 68 ->
let
# 754 "lexer_cocci.mll"
                                               wss
# 3143 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 754 "lexer_cocci.mll"
                                                        def
# 3148 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(1)
and
# 755 "lexer_cocci.mll"
                                  ident
# 3153 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_curr_pos in
# 756 "lexer_cocci.mll"
      ( start_line true;
	let (arity,line,lline,llend,offset,col,strbef,straft,pos,ws) as lt =
	  get_current_line_type lexbuf in
	let off = String.length def in
	(* -1 in the code below because the ident is not at the line start *)
	TUndef
	  (lt,
	   check_var ident
	     (arity,line,lline,llend,offset+off,col+off,[],[],[],wss)) )
# 3165 "lexer_cocci.ml"

  | 69 ->
let
# 765 "lexer_cocci.mll"
                                                  wss
# 3171 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 765 "lexer_cocci.mll"
                                                            def
# 3176 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(1)
and
# 766 "lexer_cocci.mll"
                                   ident
# 3181 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_curr_pos in
# 767 "lexer_cocci.mll"
      ( start_line true;
	let (arity,line,lline,llend,offset,col,strbef,straft,pos,ws) as lt =
	  get_current_line_type lexbuf in
	let off = String.length def in
	(* -1 in the code below because the ident is not at the line start *)
	TDefine
	  (lt,
	   check_var ident
	     (arity,line,lline,llend,offset+off,col+off,[],[],[],wss)) )
# 3193 "lexer_cocci.ml"

  | 70 ->
let
# 776 "lexer_cocci.mll"
                                                  wss
# 3199 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 776 "lexer_cocci.mll"
                                                            def
# 3204 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(1)
and
# 777 "lexer_cocci.mll"
                                    ident
# 3209 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 779 "lexer_cocci.mll"
      ( start_line true;
	let (arity,line,lline,llend,offset,col,strbef,straft,pos,ws) as lt =
	  get_current_line_type lexbuf in
	let off = String.length def in
	TDefineParam
        (lt,
	 check_var ident
	   (* why pos here but not above? *)
	   (arity,line,lline,llend,offset+off,col+off,strbef,straft,pos,wss),
	 offset + off + (String.length ident),
	 col + off + (String.length ident)) )
# 3223 "lexer_cocci.ml"

  | 71 ->
# 791 "lexer_cocci.mll"
      ( start_line true; TPragma(get_current_line_type lexbuf) )
# 3228 "lexer_cocci.ml"

  | 72 ->
# 796 "lexer_cocci.mll"
      ( TIncludeL
	  (let str = tok lexbuf in
	  let start = String.index str '\"' in
	  let finish = String.rindex str '\"' in
	  start_line true;
	  (process_include start finish str, get_current_line_type lexbuf)) )
# 3238 "lexer_cocci.ml"

  | 73 ->
# 803 "lexer_cocci.mll"
      ( TIncludeNL
	  (let str = tok lexbuf in
	  let start = String.index str '<' in
	  let finish = String.rindex str '>' in
	  start_line true;
	  (process_include start finish str,get_current_line_type lexbuf)) )
# 3248 "lexer_cocci.ml"

  | 74 ->
# 810 "lexer_cocci.mll"
      ( check_minus_context_linetype (tok lexbuf);
	start_line true;
	TIncludeAny("...",get_current_line_type lexbuf) )
# 3255 "lexer_cocci.ml"

  | 75 ->
# 821 "lexer_cocci.mll"
      ( start_line true; check_plus_linetype (tok lexbuf);
	TDirective (Ast.Noindent(tok lexbuf), get_current_line_type lexbuf) )
# 3261 "lexer_cocci.ml"

  | 76 ->
# 825 "lexer_cocci.mll"
      ( start_line true;
	let res = TCppEscapedNewline (get_current_line_type lexbuf) in
	reset_line lexbuf;
	res )
# 3269 "lexer_cocci.ml"

  | 77 ->
# 831 "lexer_cocci.mll"
      (
       match !current_line_type with
        (D.PLUS,_,_) | (D.PLUSPLUS,_,_) ->
        start_line true;
	(* second argument to TDirective is not quite right, because
	   it represents only the first token of the comment, but that
	   should be good enough *)
	TDirective (Ast.Indent("/*"^(comment check_comment lexbuf)),
		 get_current_line_type lexbuf)
      |	_ -> let _ = comment (fun _ -> ()) lexbuf in token lexbuf )
# 3283 "lexer_cocci.ml"

  | 78 ->
# 842 "lexer_cocci.mll"
      ( (if !current_line_started
      then lexerr "--- must be at the beginning of the line" "");
	start_line true;
	TMinusFile
	  (let str = tok lexbuf in
	  (drop_spaces(String.sub str 3 (String.length str - 3)),
	   (get_current_line_type lexbuf))) )
# 3294 "lexer_cocci.ml"

  | 79 ->
# 850 "lexer_cocci.mll"
      ( (if !current_line_started
      then lexerr "+++ must be at the beginning of the line" "");
	start_line true;
	TPlusFile
	  (let str = tok lexbuf in
	  (drop_spaces(String.sub str 3 (String.length str - 3)),
	   (get_current_line_type lexbuf))) )
# 3305 "lexer_cocci.ml"

  | 80 ->
# 859 "lexer_cocci.mll"
      ( start_line true; id_tokens lexbuf )
# 3310 "lexer_cocci.ml"

  | 81 ->
# 863 "lexer_cocci.mll"
      ( start_line true; id_tokens lexbuf )
# 3315 "lexer_cocci.ml"

  | 82 ->
# 870 "lexer_cocci.mll"
      (
	start_line true;
	if not !Flag.c_plus_plus
	then
	  Common.pr2_once
	    "< and > not allowed in C identifiers, try -c++ option";
	id_tokens lexbuf
      )
# 3327 "lexer_cocci.ml"

  | 83 ->
# 881 "lexer_cocci.mll"
      (
	start_line true;
	if not !Flag.c_plus_plus
	then
	  Common.pr2_once
	    "< and > not allowed in C identifiers, try -c++ option";
	id_tokens lexbuf
      )
# 3339 "lexer_cocci.ml"

  | 84 ->
# 897 "lexer_cocci.mll"
      (
	start_line true;
	if not !Flag.c_plus_plus
	then
	  Common.pr2_once
	    "~ and :: not allowed in C identifiers, try -c++ option";
	id_tokens lexbuf
      )
# 3351 "lexer_cocci.ml"

  | 85 ->
# 910 "lexer_cocci.mll"
      (
	start_line true;
	if not !Flag.c_plus_plus
	then
	  Common.pr2_once
	    "~ and :: not allowed in C identifiers, try -c++ option";
	id_tokens lexbuf
      )
# 3363 "lexer_cocci.ml"

  | 86 ->
# 921 "lexer_cocci.mll"
        ( start_line true;
	  TChar(char lexbuf,get_current_line_type lexbuf) )
# 3369 "lexer_cocci.ml"

  | 87 ->
# 923 "lexer_cocci.mll"
         ( start_line true;
	  TString(string lexbuf,(get_current_line_type lexbuf)) )
# 3375 "lexer_cocci.ml"

  | 88 ->
let
# 925 "lexer_cocci.mll"
             x
# 3381 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 925 "lexer_cocci.mll"
                   ( start_line true;
		     TFloat(x,(get_current_line_type lexbuf)) )
# 3386 "lexer_cocci.ml"

  | 89 ->
let
# 935 "lexer_cocci.mll"
         x
# 3392 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 935 "lexer_cocci.mll"
            ( start_line true; TInt(x,(get_current_line_type lexbuf)) )
# 3396 "lexer_cocci.ml"

  | 90 ->
let
# 937 "lexer_cocci.mll"
                           x
# 3402 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 938 "lexer_cocci.mll"
      ( if !Flag.ibm
      then
	begin
	  start_line true;
	  let len = string_of_int(String.length x - 1) in
          TDecimalCst(x,len,"0",(get_current_line_type lexbuf))
	end
      else failwith "unrecognized constant modifier d/D" )
# 3413 "lexer_cocci.ml"

  | 91 ->
# 947 "lexer_cocci.mll"
                   ( TIso )
# 3418 "lexer_cocci.ml"

  | 92 ->
# 948 "lexer_cocci.mll"
                   ( TRightIso )
# 3423 "lexer_cocci.ml"

  | 93 ->
# 950 "lexer_cocci.mll"
                   ( EOF )
# 3428 "lexer_cocci.ml"

  | 94 ->
# 952 "lexer_cocci.mll"
      ( lexerr "unrecognised symbol, in token rule: " (tok lexbuf) )
# 3433 "lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and metavariable_decl_token lexbuf =
    __ocaml_lex_metavariable_decl_token_rec lexbuf 234
and __ocaml_lex_metavariable_decl_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 957 "lexer_cocci.mll"
    ( reset_line lexbuf; metavariable_decl_token lexbuf )
# 3444 "lexer_cocci.ml"

  | 1 ->
# 959 "lexer_cocci.mll"
                  (
    start_line false; metavariable_decl_token lexbuf )
# 3450 "lexer_cocci.ml"

  | 2 ->
# 962 "lexer_cocci.mll"
                                   (
    start_line false; metavariable_decl_token lexbuf )
# 3456 "lexer_cocci.ml"

  | 3 ->
# 965 "lexer_cocci.mll"
         ( start_line true; TArobArob )
# 3461 "lexer_cocci.ml"

  | 4 ->
# 967 "lexer_cocci.mll"
         ( start_line true; TTildeEq (get_current_line_type lexbuf) )
# 3466 "lexer_cocci.ml"

  | 5 ->
# 968 "lexer_cocci.mll"
         ( start_line true; TTildeExclEq (get_current_line_type lexbuf) )
# 3471 "lexer_cocci.ml"

  | 6 ->
# 969 "lexer_cocci.mll"
         ( start_line true; TEq (get_current_line_type lexbuf) )
# 3476 "lexer_cocci.ml"

  | 7 ->
# 970 "lexer_cocci.mll"
        ( start_line true; TOPar (get_current_line_type lexbuf) )
# 3481 "lexer_cocci.ml"

  | 8 ->
# 971 "lexer_cocci.mll"
        ( start_line true; TCPar (get_current_line_type lexbuf) )
# 3486 "lexer_cocci.ml"

  | 9 ->
# 973 "lexer_cocci.mll"
        ( start_line true; TOCro (get_current_line_type lexbuf)   )
# 3491 "lexer_cocci.ml"

  | 10 ->
# 974 "lexer_cocci.mll"
        ( start_line true; TCCro (get_current_line_type lexbuf)   )
# 3496 "lexer_cocci.ml"

  | 11 ->
# 975 "lexer_cocci.mll"
        ( start_line true; TOBrace (get_current_line_type lexbuf) )
# 3501 "lexer_cocci.ml"

  | 12 ->
# 976 "lexer_cocci.mll"
        ( start_line true; TCBrace (get_current_line_type lexbuf) )
# 3506 "lexer_cocci.ml"

  | 13 ->
# 978 "lexer_cocci.mll"
                   ( start_line true; TPtrOp (get_current_line_type lexbuf)  )
# 3511 "lexer_cocci.ml"

  | 14 ->
# 979 "lexer_cocci.mll"
                   ( start_line true; TDot (get_current_line_type lexbuf)    )
# 3516 "lexer_cocci.ml"

  | 15 ->
# 980 "lexer_cocci.mll"
                   ( start_line true; TComma (get_current_line_type lexbuf)  )
# 3521 "lexer_cocci.ml"

  | 16 ->
# 981 "lexer_cocci.mll"
                   ( start_line true;
		     TMPtVirg (* works better with tokens_all *) )
# 3527 "lexer_cocci.ml"

  | 17 ->
# 983 "lexer_cocci.mll"
                   ( start_line true;
		     TShLOp(Ast.DecLeft,get_current_line_type lexbuf) )
# 3533 "lexer_cocci.ml"

  | 18 ->
# 986 "lexer_cocci.mll"
                   ( pass_zero();
		     if !current_line_started
		     then
		       (start_line true; TMul (get_current_line_type lexbuf))
		     else
		       (patch_or_match MATCH;
			add_current_line_type D.MINUS;
			metavariable_decl_token lexbuf) )
# 3545 "lexer_cocci.ml"

  | 19 ->
# 995 "lexer_cocci.mll"
                   ( start_line true; TEqEq    (get_current_line_type lexbuf) )
# 3550 "lexer_cocci.ml"

  | 20 ->
# 996 "lexer_cocci.mll"
                   ( start_line true; TNotEq   (get_current_line_type lexbuf) )
# 3555 "lexer_cocci.ml"

  | 21 ->
# 997 "lexer_cocci.mll"
                   ( start_line true; TSub     (get_current_line_type lexbuf) )
# 3560 "lexer_cocci.ml"

  | 22 ->
# 998 "lexer_cocci.mll"
        ( (start_line true; TPlus (get_current_line_type lexbuf)) )
# 3565 "lexer_cocci.ml"

  | 23 ->
# 999 "lexer_cocci.mll"
        ( (start_line true; TMinus (get_current_line_type lexbuf)) )
# 3570 "lexer_cocci.ml"

  | 24 ->
# 1000 "lexer_cocci.mll"
        ( start_line true; TDmOp (Ast.Div,get_current_line_type lexbuf) )
# 3575 "lexer_cocci.ml"

  | 25 ->
# 1001 "lexer_cocci.mll"
        ( start_line true; TDmOp (Ast.Mod,get_current_line_type lexbuf) )
# 3580 "lexer_cocci.ml"

  | 26 ->
# 1002 "lexer_cocci.mll"
         ( start_line true; TShROp(Ast.DecRight,get_current_line_type lexbuf) )
# 3585 "lexer_cocci.ml"

  | 27 ->
# 1003 "lexer_cocci.mll"
        ( start_line true; TAnd (get_current_line_type lexbuf) )
# 3590 "lexer_cocci.ml"

  | 28 ->
# 1004 "lexer_cocci.mll"
        (  (start_line true; TOr(get_current_line_type lexbuf)) )
# 3595 "lexer_cocci.ml"

  | 29 ->
# 1005 "lexer_cocci.mll"
        ( start_line true; TXor(get_current_line_type lexbuf) )
# 3600 "lexer_cocci.ml"

  | 30 ->
# 1006 "lexer_cocci.mll"
         ( start_line true; TLogOp(Ast.SupEq,get_current_line_type lexbuf) )
# 3605 "lexer_cocci.ml"

  | 31 ->
# 1007 "lexer_cocci.mll"
        ( start_line true; TLogOp(Ast.Inf,get_current_line_type lexbuf) )
# 3610 "lexer_cocci.ml"

  | 32 ->
# 1008 "lexer_cocci.mll"
        ( start_line true; TLogOp(Ast.Sup,get_current_line_type lexbuf) )
# 3615 "lexer_cocci.ml"

  | 33 ->
# 1009 "lexer_cocci.mll"
         ( start_line true; TAndLog (get_current_line_type lexbuf) )
# 3620 "lexer_cocci.ml"

  | 34 ->
# 1010 "lexer_cocci.mll"
         ( start_line true; TOrLog (get_current_line_type lexbuf) )
# 3625 "lexer_cocci.ml"

  | 35 ->
# 1011 "lexer_cocci.mll"
         ( start_line true; TDotDot (get_current_line_type lexbuf) )
# 3630 "lexer_cocci.ml"

  | 36 ->
# 1012 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Minus lexbuf )
# 3635 "lexer_cocci.ml"

  | 37 ->
# 1013 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Plus lexbuf )
# 3640 "lexer_cocci.ml"

  | 38 ->
# 1014 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Mul lexbuf )
# 3645 "lexer_cocci.ml"

  | 39 ->
# 1015 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Div lexbuf )
# 3650 "lexer_cocci.ml"

  | 40 ->
# 1016 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Mod lexbuf )
# 3655 "lexer_cocci.ml"

  | 41 ->
# 1017 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.And lexbuf )
# 3660 "lexer_cocci.ml"

  | 42 ->
# 1018 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Or lexbuf )
# 3665 "lexer_cocci.ml"

  | 43 ->
# 1019 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Xor lexbuf )
# 3670 "lexer_cocci.ml"

  | 44 ->
# 1020 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.DecLeft lexbuf )
# 3675 "lexer_cocci.ml"

  | 45 ->
# 1021 "lexer_cocci.mll"
                   ( start_line true; mkassign Ast.DecRight lexbuf )
# 3680 "lexer_cocci.ml"

  | 46 ->
# 1023 "lexer_cocci.mll"
      (match !current_line_type with
        (D.PLUS,_,_) | (D.PLUSPLUS,_,_) ->
        start_line true;
	(* second argument to TDirective is not quite right, because
	   it represents only the first token of the comment, but that
	   should be good enough *)
	TDirective (Ast.Indent("/*"^(comment check_comment lexbuf)),
		 get_current_line_type lexbuf)
      |	_ -> let _ = comment (fun _ -> ()) lexbuf in
	     metavariable_decl_token lexbuf )
# 3694 "lexer_cocci.ml"

  | 47 ->
# 1034 "lexer_cocci.mll"
                    ( start_line true; TCppConcatOp (* for fresh vars *) )
# 3699 "lexer_cocci.ml"

  | 48 ->
# 1037 "lexer_cocci.mll"
      ( start_line true; id_tokens lexbuf )
# 3704 "lexer_cocci.ml"

  | 49 ->
# 1041 "lexer_cocci.mll"
      ( start_line true; id_tokens lexbuf )
# 3709 "lexer_cocci.ml"

  | 50 ->
# 1048 "lexer_cocci.mll"
      ( start_line true;
	if not !Flag.c_plus_plus
	then
	  Common.pr2_once
	    "< and > not allowed in C identifiers, try -c++ option";
	id_tokens lexbuf
      )
# 3720 "lexer_cocci.ml"

  | 51 ->
# 1058 "lexer_cocci.mll"
      ( start_line true;
	if not !Flag.c_plus_plus
	then
	  Common.pr2_once
	    "< and > not allowed in C identifiers, try -c++ option";
	id_tokens lexbuf
      )
# 3731 "lexer_cocci.ml"

  | 52 ->
# 1073 "lexer_cocci.mll"
      ( start_line true;
	if not !Flag.c_plus_plus
	then
	  Common.pr2_once
	    "~ and :: not allowed in C identifiers, try -c++ option";
	id_tokens lexbuf
      )
# 3742 "lexer_cocci.ml"

  | 53 ->
# 1085 "lexer_cocci.mll"
      ( start_line true;
	if not !Flag.c_plus_plus
	then
	  Common.pr2_once
	    "~ and :: not allowed in C identifiers, try -c++ option";
	id_tokens lexbuf
      )
# 3753 "lexer_cocci.ml"

  | 54 ->
# 1095 "lexer_cocci.mll"
        ( start_line true;
	  TChar(char lexbuf,get_current_line_type lexbuf) )
# 3759 "lexer_cocci.ml"

  | 55 ->
# 1097 "lexer_cocci.mll"
         ( start_line true;
	  TString(string lexbuf,(get_current_line_type lexbuf)) )
# 3765 "lexer_cocci.ml"

  | 56 ->
let
# 1099 "lexer_cocci.mll"
             x
# 3771 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1099 "lexer_cocci.mll"
                   ( Printf.printf "36\n"; start_line true;
		     TFloat(x,(get_current_line_type lexbuf)) )
# 3776 "lexer_cocci.ml"

  | 57 ->
let
# 1109 "lexer_cocci.mll"
         x
# 3782 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1109 "lexer_cocci.mll"
            ( start_line true; TInt(x,(get_current_line_type lexbuf)) )
# 3786 "lexer_cocci.ml"

  | 58 ->
let
# 1111 "lexer_cocci.mll"
                           x
# 3792 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1112 "lexer_cocci.mll"
      ( if !Flag.ibm
      then
	begin
	  start_line true;
	  let len = string_of_int(String.length x - 1) in
          TDecimalCst(x,len,"0",(get_current_line_type lexbuf))
	end
      else failwith "unrecognized constant modifier d/D" )
# 3803 "lexer_cocci.ml"

  | 59 ->
# 1121 "lexer_cocci.mll"
      ( lexerr
	 "metavariables: unrecognised symbol in metavariable_decl_token rule: "
	 (tok lexbuf) )
# 3810 "lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_metavariable_decl_token_rec lexbuf __ocaml_lex_state

and char lexbuf =
    __ocaml_lex_char_rec lexbuf 342
and __ocaml_lex_char_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 1127 "lexer_cocci.mll"
          x
# 3822 "lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 1127 "lexer_cocci.mll"
                                       ( String.make 1 x ^ restchars lexbuf )
# 3826 "lexer_cocci.ml"

  | 1 ->
let
# 1129 "lexer_cocci.mll"
                                             x
# 3832 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1129 "lexer_cocci.mll"
                                                     ( x ^ restchars lexbuf )
# 3836 "lexer_cocci.ml"

  | 2 ->
let
# 1133 "lexer_cocci.mll"
                                  x
# 3842 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1133 "lexer_cocci.mll"
                                                     ( x ^ restchars lexbuf )
# 3846 "lexer_cocci.ml"

  | 3 ->
let
# 1134 "lexer_cocci.mll"
                 v
# 3852 "lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 1134 "lexer_cocci.mll"
                                  x
# 3857 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_start_pos + 2) in
# 1135 "lexer_cocci.mll"
 (
          (match v with (* Machine specific ? *)
          | 'n' -> ()  | 't' -> ()   | 'v' -> ()  | 'b' -> () | 'r' -> ()
          | 'f' -> () | 'a' -> ()
	  | '\\' -> () | '?'  -> () | '\'' -> ()  | '\"' -> ()
          | 'e' -> () (* linuxext: ? *)
	  | _ ->
              Common.pr2 ("LEXER: unrecognised symbol in char:"^tok lexbuf);
	  );
          x ^ restchars lexbuf
	)
# 3871 "lexer_cocci.ml"

  | 4 ->
# 1147 "lexer_cocci.mll"
      ( Common.pr2 ("LEXER: unrecognised symbol in char:"^tok lexbuf);
        tok lexbuf ^ restchars lexbuf
      )
# 3878 "lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_char_rec lexbuf __ocaml_lex_state

and restchars lexbuf =
    __ocaml_lex_restchars_rec lexbuf 352
and __ocaml_lex_restchars_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 1152 "lexer_cocci.mll"
                                       ( "" )
# 3889 "lexer_cocci.ml"

  | 1 ->
let
# 1153 "lexer_cocci.mll"
          x
# 3895 "lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 1153 "lexer_cocci.mll"
                                       ( String.make 1 x ^ restchars lexbuf )
# 3899 "lexer_cocci.ml"

  | 2 ->
let
# 1155 "lexer_cocci.mll"
                                             x
# 3905 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1155 "lexer_cocci.mll"
                                                     ( x ^ restchars lexbuf )
# 3909 "lexer_cocci.ml"

  | 3 ->
let
# 1159 "lexer_cocci.mll"
                                  x
# 3915 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1159 "lexer_cocci.mll"
                                                     ( x ^ restchars lexbuf )
# 3919 "lexer_cocci.ml"

  | 4 ->
let
# 1160 "lexer_cocci.mll"
                 v
# 3925 "lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 1160 "lexer_cocci.mll"
                                  x
# 3930 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_start_pos + 2) in
# 1161 "lexer_cocci.mll"
 (
          (match v with (* Machine specific ? *)
          | 'n' -> ()  | 't' -> ()   | 'v' -> ()  | 'b' -> () | 'r' -> ()
          | 'f' -> () | 'a' -> ()
	  | '\\' -> () | '?'  -> () | '\'' -> ()  | '\"' -> ()
          | 'e' -> () (* linuxext: ? *)
	  | _ ->
              Common.pr2 ("LEXER: unrecognised symbol in char:"^tok lexbuf);
	  );
          x ^ restchars lexbuf
	)
# 3944 "lexer_cocci.ml"

  | 5 ->
# 1173 "lexer_cocci.mll"
      ( Common.pr2 ("LEXER: unrecognised symbol in char:"^tok lexbuf);
        tok lexbuf ^ restchars lexbuf
      )
# 3951 "lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_restchars_rec lexbuf __ocaml_lex_state

and string lexbuf =
    __ocaml_lex_string_rec lexbuf 363
and __ocaml_lex_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 1178 "lexer_cocci.mll"
                                               ( "" )
# 3962 "lexer_cocci.ml"

  | 1 ->
let
# 1179 "lexer_cocci.mll"
                                 x
# 3968 "lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 1180 "lexer_cocci.mll"
    ( line := !line + 1; (Printf.sprintf "%c" x) ^ string lexbuf )
# 3972 "lexer_cocci.ml"

  | 2 ->
let
# 1181 "lexer_cocci.mll"
          x
# 3978 "lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 1181 "lexer_cocci.mll"
                               ( Common.string_of_char x ^ string lexbuf )
# 3982 "lexer_cocci.ml"

  | 3 ->
let
# 1182 "lexer_cocci.mll"
                                            x
# 3988 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1182 "lexer_cocci.mll"
                                              ( x ^ string lexbuf )
# 3992 "lexer_cocci.ml"

  | 4 ->
let
# 1183 "lexer_cocci.mll"
                               x
# 3998 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1183 "lexer_cocci.mll"
                                              ( x ^ string lexbuf )
# 4002 "lexer_cocci.ml"

  | 5 ->
let
# 1184 "lexer_cocci.mll"
                v
# 4008 "lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 1184 "lexer_cocci.mll"
                       x
# 4013 "lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_start_pos + 2) in
# 1185 "lexer_cocci.mll"
       (
         (match v with
	    | 'n' -> ()  | 't' -> ()   | 'v' -> ()  | 'b' -> () | 'r' -> ()
	    | 'f' -> () | 'a' -> ()
	    | '\\' -> () | '?'  -> () | '\'' -> ()  | '\"' -> ()
	    | 'e' -> ()
	    | '\n' -> ()
	    | '(' -> () | '|' -> () | ')' -> ()
	    | _ -> lexerr "unrecognised symbol:" (tok lexbuf)
	 );
          x ^ string lexbuf
       )
# 4028 "lexer_cocci.ml"

  | 6 ->
# 1197 "lexer_cocci.mll"
      ( lexerr "unrecognised symbol: " (tok lexbuf) )
# 4033 "lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_string_rec lexbuf __ocaml_lex_state

and comment check_comment lexbuf =
    __ocaml_lex_comment_rec check_comment lexbuf 375
and __ocaml_lex_comment_rec check_comment lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 1200 "lexer_cocci.mll"
         ( let s = tok lexbuf in check_comment s; start_line true; s )
# 4044 "lexer_cocci.ml"

  | 1 ->
# 1202 "lexer_cocci.mll"
      ( let s = tok lexbuf in
        (* even blank line should have a + *)
        check_comment s;
        reset_line lexbuf; s ^ comment check_comment lexbuf )
# 4052 "lexer_cocci.ml"

  | 2 ->
# 1206 "lexer_cocci.mll"
        ( pass_zero();
	  if !current_line_started
	  then (start_line true;
		let s = tok lexbuf in s^(comment check_comment lexbuf))
	  else (start_line true; comment check_comment lexbuf) )
# 4061 "lexer_cocci.ml"

  | 3 ->
# 1213 "lexer_cocci.mll"
      ( let s = tok lexbuf in
        check_comment s; start_line true; s ^ comment check_comment lexbuf )
# 4067 "lexer_cocci.ml"

  | 4 ->
# 1216 "lexer_cocci.mll"
      ( let s = tok lexbuf in
        check_comment s; start_line true; s ^ comment check_comment lexbuf )
# 4073 "lexer_cocci.ml"

  | 5 ->
# 1219 "lexer_cocci.mll"
      ( start_line true; let s = tok lexbuf in
        Common.pr2 ("LEXER: unrecognised symbol in comment:"^s);
        s ^ comment check_comment lexbuf
      )
# 4081 "lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_comment_rec check_comment lexbuf __ocaml_lex_state

;;

