# Copyright (C) 2004 Scott W. Dunlop <swdunlop at users.sourceforge.net>
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

from codex import Codex
from md5 import md5 
from database import DatabaseLogger

class Authenticator( object ):
    __slots__ = 'wiki',
    
    def __init__( self, wiki ):
        self.wiki = wiki 
       
    def getWiki( self ):
        return self.wiki

    def getDatabase( self ):
        return self.getWiki().getDatabase()
        
    def getUserForToken( self, token ):
        session = Codex.getSession( key=token )
        if session is None: return None
        
        return session.getName()
    
    def authenticate( self, username, password ):
        result = self.check( username, password )

        if result:
            return result, Codex.createSession( result ).getKey()
        else:
            return None

    def shouldMentionUsers( self ):
        return True        

class NoneAuthenticator( Authenticator ):
    def check( self, username, attempt ):
        return False
        
    def shouldMentionUsers( self ):
        return False
        
class CloudAuthenticator( Authenticator ):
    def check( self, username, attempt ):
        answer = self.getDatabase().getPassword( username )
        if answer is None: return False
        if answer == attempt:
            return username
        else:
            return False

class Md5Authenticator( Authenticator ):
    def check( self, username, attempt ):
        answer = self.getDatabase().getPassword( username )
        if answer is None: return False
        if answer == md5( attempt ).hexdigest():
            return username
        else:
            return False

class DatabaseAuthenticator( Authenticator ):
    __slots__ = 'connection', 'cursor'
    def __init__( self, database ):
        Authenticator.__init__( self, database )
        self.connect( )

class KabukiAuthenticator( DatabaseAuthenticator ):
    def connect( self ):
        import sqlite

        path = self.getDatabase().getConfig( 'auth-db-path' )
        if path:
            self.connection = sqlite.connect( 
                path, 
                command_logfile=DatabaseLogger( self.getWiki() ),
                autocommit=1
            )

    def check( self, username, attempt ):
        try:
            cursor = self.connection.cursor()
            
            cursor.execute( 
                "SELECT username, password FROM KabukiUsers "
                "WHERE userid == %s;",
                username.lower()
            )
            
            truename, password = cursor.fetchone()

            if password == attempt:
                return truename
            else:
                return False
        except:
            return False

authentication_schemes = {
    'cloud' : CloudAuthenticator,
    'no' : NoneAuthenticator,
    'none' : NoneAuthenticator,
    'false' : NoneAuthenticator,
    'f': NoneAuthenticator,
    'md5': Md5Authenticator,
    'kabuki' : KabukiAuthenticator
}
