/* cLog - cross platform logging library
 * Copyright (c) 2003 WebGroup Media, LLC <support@webgroupmedia.com>
 *
 * The MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdarg.h> // va_*
#include <string.h> // memset
#include <stdio.h> // vsnprintf
#include <time.h> // localtime
 
#include "clog.h"

#ifdef MEMWATCH
#include <memwatch.h>
#warning "USING MEMWATCH"
#endif

char *(g_text[]) = { "MARK ", "FATAL", "ERROR", "WARN ", "DEBUG", "TRACE" };

void clog(CLOG_INFO* info, int level, const char *format, ...)
{
  // cannot put logging in this function due to recursion

  if(NULL==info) return;

  if(!(-1 < level && 6 > level)) {
    level = CMARK;
  }

  // if we are logging to the function or file
  if(level <= info->s_logfile_level || level <= info->s_callback_level) {
    va_list ap; // for the variable args
    time_t    t;
    struct tm *ts;
    char message[1024];

    memset((char*)message, 0, 1024);
    va_start(ap, format);
    vsnprintf(message, 1023, format, ap);
    va_end(ap);


    // log to function
    if(NULL!=info->s_callback && level <= info->s_callback_level) {
      info->s_callback(level, message);
    }

    // log to file
    if(NULL!=info->s_logfile && level <= info->s_logfile_level) {
      if(NULL!=info->s_logfile) { // log to text file
        time(&t);
        ts = localtime(&t);

        if(NULL!=ts) { // add the time to the beginning
          fprintf(info->s_logfile, "[%d.%.2d.%.2d %.2d:%.2d:%.2d] %s ", ts->tm_year + 1900, ts->tm_mon + 1, ts->tm_mday, ts->tm_hour, ts->tm_min, ts->tm_sec, g_text[level]);
        }

        fprintf(info->s_logfile, message);
#ifdef WIN32
        fprintf(info->s_logfile, "\r");
#endif
        fprintf(info->s_logfile, "\n");
        fflush(info->s_logfile);
        ts = NULL;
      }
    }
  }
}
