(in-package #:djula)

(defvar *auto-escape* t
  "Controls auto escaping.")

;;; truncatechars:"30" => (:truncatechars 30)
(defun parse-filter-string (string)
  (if-let ((colon (position #\: string)))
    (list (make-keyword (string-upcase (subseq string 0 colon)))
          (string-trim '(#\") (subseq string (1+ colon))))
    (list (make-keyword (string-upcase string)))))

(defun integer-or-keyword (string)
  "If the STRING is an integer return an integer, otherwise return STRING as a
keyword."
  (if (every 'digit-char-p string)
      (parse-integer string)
      (make-keyword (string-upcase string))))

;;; foo.bar.baz.2 => (:foo :bar :baz 2)
(defun parse-variable-phrase (string)
  (if-let ((dot (position #\. string)))
    (cons (integer-or-keyword (subseq string 0 dot))
          (parse-variable-phrase (subseq string (1+ dot))))
    (list (integer-or-keyword string))))


(defun escape-string-split (char string &optional (escape #\\) (quotes (list #\" #\')))
  (let (escaped
        quoted)
    (flet ((escaped-char (delimiter char)
             (when (not escaped)
               (cond ((and quoted (eql char quoted))
                      (setf quoted nil))
                     ((and (not quoted) (member char quotes))
                      (setf quoted char))))
             (prog1
                 (and (not escaped)
                      (not quoted)
                      (eql delimiter char))
               (if (and (not escaped) ;; Check for escaped escape char
                        (eql escape char))
                   (setf escaped t)
                   (setf escaped nil)))))
      (split-sequence:split-sequence char string :test #'escaped-char))))

;;; foo.bar.baz.2 | truncatechars:"30" | upper => ((:foo :bar :baz 2) (:truncatechars 30) (:upper))
(defun parse-variable-clause (unparsed-string)
  (destructuring-bind (var . filter-strings)
      (mapcar (lambda (s)
                (string-trim '(#\space #\tab #\newline #\return) s))
              (escape-string-split #\| unparsed-string))
    (cons (parse-variable-phrase var)
          (mapcar #'parse-filter-string filter-strings))))

(def-token-processor :unparsed-variable (unparsed-string) rest
  ":PARSED-VARIABLE tokens are parsed into :VARIABLE tokens by PROCESS-TOKENS"
  (cons (list* :variable (parse-variable-clause unparsed-string))
        (process-tokens rest)))

(let ((var-not-bound (gensym "VAR-NOT-BOUND-")))
  (defun template-var-boundp (varname)
    (not (eq (getf *template-arguments* varname var-not-bound)
             var-not-bound))))

(defun check-template-variable-boundp (varname)
  (when *strict-mode*
    (unless (template-var-boundp varname)
      (error "Variable not bound: ~a" varname))))

(defun ensure-access (object key)
  (multiple-value-bind (val accessed-p)
      (access:access object key)
    (when (and (not accessed-p) *strict-mode*)
      (error "Can't access ~s of ~a" key object))
    val))

(defun apply-keys/indexes (thing keys/indexes)
  (let ((*package* (find-package *template-package*)))
    (reduce (lambda (thing key)
              (handler-case
                  (cond
                    ((numberp key) (elt thing key))
                    ((keywordp key)
                     ;; Try to access with keyword
                     (multiple-value-bind (val accessed-p)
                         (ignore-errors (access:access thing key))
                       (if (and accessed-p (not (typep accessed-p 'error)))
                           val
                           ;; else, try to access with interned symbol
                           (ensure-access thing (intern (symbol-name key))))))
                    (t (ensure-access thing key)))
                (error (e)
                  (error
                   "There was an error while accessing the ~A ~S of the object ~S: ~a"
                   (if (numberp key)
                       "index"
                       "attribute")
                   key thing e))))
            keys/indexes
            :initial-value thing)))

(let ((no-default (gensym "NO-DEFAULT")))
  (defun get-variable (name &optional (default no-default))
    "takes a variable `NAME' and returns:
   1. the value of `NAME'
   2. any error string generated by the lookup (if there is an error string then the
      lookup was unsuccessful)"
    (when (eq default no-default)
      (check-template-variable-boundp name))
    (or (access:access *template-arguments* name)
        (when (not (eq default no-default))
          default))))

(defun resolve-variable-phrase (list)
  "takes a list starting wise a variable and ending with 0 or more keys or indexes [this
is a direct translation from the dot (.) syntax] and returns two values:

   1. the result [looking up the var and applying index/keys]
   2. an error string if something went wrond [note: if there is an error string then
the result probably shouldn't be considered useful."
  (when-let (v (get-variable (first list)))
    (apply-keys/indexes v (rest list))))

(def-token-compiler :variable (variable-phrase &rest filters)
  ;; Output the value of a variable access.
  ;; check to see if the "dont-escape" filter is used
  ;; "safe" takes precedence before "escape"
  (let ((dont-escape
          (or
           (find '(:safe) filters :test #'equal) ; safe filter used
           (and (not *auto-escape*)              ; autoescape off and no escape filter used
                (not (find '(:escape) filters :test #'equal))))))
    ;; return a function that resolves the variable-phase and applies the filters
    (lambda (stream)
      ;; if the variable is not bound, signal error
      (check-template-variable-boundp (first variable-phrase))
      (multiple-value-bind (ret error-string)
          (resolve-variable-phrase variable-phrase)
        (cond
          (error-string
           (with-template-error error-string
             (error error-string)))
          (t
           (let ((filtered-ret
                   (template-print-object
                    (or
                     (apply-filters
                      ret filters)
                     ""))))
             (princ (if dont-escape
                        filtered-ret
                        (escape-for-html filtered-ret))
                    stream))))))))
