(in-package #:cl-user)

(defpackage #:cl-colors2-tests
  (:use
   #:alexandria
   #:common-lisp
   #:cl-colors2
   #:clunit)
  (:export #:run))

(in-package #:cl-colors2-tests)

(defsuite cl-colors-suite ())

(defmacro first-value (a)
  `(first (multiple-value-list ,a)))

(defun run (&key (use-debugger nil))
  "Run all the tests for CL-COLORS2-TESTS."
  (clunit:run-suite 'cl-colors-suite :use-debugger use-debugger))

(defun rgb= (rgb1 rgb2 &optional (epsilon 1e-10))
  "Compare RGB colors for (numerical) equality."
  (color-equals rgb1 rgb2 :tolerance epsilon))

(defun random-rgb ()
  (rgb (random 1d0) (random 1d0) (random 1d0)))

(deftest rgb<->hsv-test (cl-colors-suite)
  (loop repeat 100 do
    (let ((rgb (random-rgb)))
      (assert-true (rgb= rgb (as-rgb (as-hsv rgb))))
      (assert-true (color-equals (as-hsv rgb) rgb))
      (assert-true (rgb= rgb (as-rgb (as-hsv* rgb))))
      (assert-true (color-equals (as-hsv* rgb) (as-hsv rgb))))))

(deftest rgb<->hsl-test (cl-colors-suite)
  (loop repeat 100 do
    (let ((rgb (random-rgb)))
      (assert-true (rgb= rgb (as-rgb (as-hsl rgb))))
      (assert-true (color-equals (as-hsl rgb) rgb))
      (assert-true (rgb= rgb (as-rgb (as-hsl rgb))))
      (assert-true (color-equals (as-hsl rgb) (as-hsl rgb))))))

(deftest conversions-tests (cl-colors-suite)
  (let ((*clunit-equality-test* #'color-equals))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (as-rgb (as-hsv (rgb (/ 169 256) (/ 104.22 256) (/ 54 256)))))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (as-rgb (as-hsl (as-hsv (rgb (/ 169 256) (/ 104.22 256) (/ 54 256))))))
    (assert-equality* (rgb 53/128 27/32 97/256)
        (as-hsl (as-hsv (rgb (/ 106 256) (/ 216 256) (/ 97 256)))))))

(deftest parse-name (cl-colors-suite)
  (assert-true (as-rgb "snow"))
  (assert-condition error (as-rgb "foo-bar"))
  (assert-true (as-rgb "teal"))
  (assert-condition error (as-rgb "teal" :colors-list *x11-colors-list*))
  (assert-false (as-rgb "teal" :colors-list *x11-colors-list* :errorp nil)))

;; (defun test-hue-combination (from to positivep)
;;   (dotimes (i 21)
;;     (format t "~a " (hue-combination from to (/ i 20) positivep))))

(deftest print-hex-rgb-test (cl-colors-suite)
  (let ((rgb (rgb 0.070 0.203 0.337)))
    (assert-equalp "#123456" (print-hex-rgb rgb))
    (assert-equalp "123456" (print-hex-rgb rgb :hash nil))
    (assert-equalp "#135" (print-hex-rgb rgb :short t))
    (assert-equalp "135" (print-hex-rgb rgb :hash nil :short t))
    (assert-equalp "#12345678" (print-hex-rgb rgb :alpha 0.47))
    (assert-equalp "12345678" (print-hex-rgb rgb :alpha 0.47 :hash nil))
    (assert-equalp "#1357" (print-hex-rgb rgb :alpha 0.47 :short t))
    (assert-equalp "1357" (print-hex-rgb rgb :alpha 0.47 :hash nil :short t))))

(deftest print-hex-test (cl-colors-suite)
  (assert-equalp "#c86414" (print-hex (rgb 200/255 100/255 20/255)))
  (assert-equalp "#c86414" (print-hex (hsv 26.67 90/100  7843/10000)))
  (assert-equalp "#c86414" (print-hex (hsl 26.67 8182/10000 4314/10000))))

(deftest parse-hex-rgb-test (cl-colors-suite)
  (let ((rgb                     (rgb 0.070 0.203 0.337))
        (*clunit-equality-test*  (rcurry #'rgb= 0.01)))
    (assert-equality* rgb (first-value (parse-hex-rgb "#123456")))
    (assert-equality* rgb (first-value (parse-hex-rgb "123456")))
    (assert-equality* rgb (first-value (parse-hex-rgb "#135")))
    (assert-equality* rgb (first-value (parse-hex-rgb "135")))
    (let ((*clunit-equality-test* #'(lambda (list1 list2)
                                      (and (rgb= (car list1) (car list2) 0.01)
                                           (cl-colors2::eps= (cadr list1) (cadr list2)
                                                             0.01)))))
      (assert-equality* (list rgb 0.47) (multiple-value-list (parse-hex-rgb "#12345678")))
      (assert-equality* (list rgb 0.47) (multiple-value-list (parse-hex-rgb "12345678")))
      (assert-equality* (list rgb 0.47) (multiple-value-list (parse-hex-rgb "#1357")))
      (assert-equality* (list rgb 0.47) (multiple-value-list (parse-hex-rgb "1357")))
      (assert-equality* (list (rgb 1 1 0) 2/3)
          (multiple-value-list (parse-hex-rgb "FF0A")))
      (assert-equality* (list (rgb 1 1 0) 2/3)
          (multiple-value-list (parse-hex-rgb "#FF0A")))
      (assert-equality* (list (RGB 50/51 251/255 84/85) 1/255)
          (multiple-value-list (parse-hex-rgb "#FAFBFC01"))))))

(deftest print-hex-rgb/format-test (cl-colors-suite)
  (assert-equalp "#123456"
                 (with-output-to-string (*standard-output*)
                             (print-hex-rgb (rgb 0.070 0.203 0.337)
                                            :destination T))))

(deftest hex<->rgb-test (cl-colors-suite)
  (loop repeat 100 do
       (let ((rgb                     (random-rgb))
             (*clunit-equality-test*  (rcurry #'rgb= 0.01)))
         (assert-equality* rgb (first-value (parse-hex-rgb (print-hex-rgb rgb)))))))

(deftest parse-hex-rgb-ranges-test (cl-colors-suite)
  (let ((*clunit-equality-test* (rcurry #'rgb= 0.001)))
    (assert-equality* (rgb 0.070 0.203 0.337)
        (first-value (parse-hex-rgb "foo#123456zzz" :start 3 :end 10)))))

(deftest parse-css-hsv (cl-colors-suite)
  (let ((rgb                    (rgb 169/256 0.40710938 27/128))
        (*clunit-equality-test* #'color-equals)
        (*epsilon*              1e-4))
    (assert-true (color-equals (cl-colors2::parse-hsv-to-rgb (print-css-hsv rgb))
                               rgb))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsv-to-rgb "hsv(26.20 68.05% 66.02%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsv-to-rgb "hsv(26.20,68.05%,66.02%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsv-to-rgb "hsv(26.20     68.05%      66.02%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsv-to-rgb "hsv(26.20  ,   68.05%   ,   66.02%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsv-to-rgb "hsv(26.20     68.05      66.02%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsv-to-rgb "hsv(26.20     68.05%      66.02)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsv-to-rgb "hsv(26.20  ,   68.05%,66.02%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsv-to-rgb "hsv(26.20  ,   68.05%      66.02%)"))))

(deftest parse-css-hsl (cl-colors-suite)
  (let ((rgb                    (rgb 169/256 0.40710938 27/128))
        (*clunit-equality-test* #'color-equals)
        (*epsilon*              1e-4))
    (assert-true (color-equals (cl-colors2::parse-hsl-to-rgb (print-css-hsl rgb))
                               rgb))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsl-to-rgb "hsl(26.20 51.57% 43.55%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsl-to-rgb "hsl(26.20,51.57%,43.55%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsl-to-rgb "hsl(26.20     51.57%      43.55%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsl-to-rgb "hsl(26.20  ,   51.57%   ,   43.55%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsl-to-rgb "hsl(26.20     51.57      43.55%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsl-to-rgb "hsl(    26.20     51.57%      43.55)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsl-to-rgb "hsl(26.20  ,   51.57%,43.55%)"))
    (assert-equality* (rgb 169/256 0.40710938 27/128)
        (cl-colors2::parse-hsl-to-rgb "hsl(26.20  ,   51.57%      43.55%)"))))

(deftest parse-css-rgb (cl-colors-suite)
  (let ((rgb                    (rgb 169/256 0.40710938 27/128))
        (*clunit-equality-test* (lambda (a b)
                                  (if (cl-colors2::rgb-p a)
                                      (color-equals a b)
                                      (equalp a b))))
        (*epsilon*              1e-4))
    (assert-true (color-equals (cl-colors2:parse-rgb/a-to-rgb (print-css-rgb/a rgb))
                               rgb))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 1 nil)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(168.34 103.81 53.79)"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 1 nil)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(168.34,103.81,53.79)"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 1 nil)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(168.34       103.81       53.79)"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 1 nil)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(168.34    ,  103.81     ,  53.79)"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 1 nil)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(  168.34      103.81       53.79   )"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 1 nil)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(168.34    ,  103.81,53.79)"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 1 nil)
        (cl-colors2:parse-rgb/a-to-rgb "rgb (168.34    ,  103.81   53.79)"))))

(deftest parse-css-rgba (cl-colors-suite)
  (let ((rgb                    (rgb 169/256 0.40710938 27/128))
        (*clunit-equality-test* (lambda (a b)
                                  (if (cl-colors2::rgb-p a)
                                      (color-equals a b)
                                      (equalp a b))))
        (*epsilon*              1e-4))
    (assert-true (color-equals (cl-colors2:parse-rgb/a-to-rgb (print-css-rgb/a rgb))
                               rgb))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 1 nil)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(168.34 103.81 53.79 / none)"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 1 nil)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(168.34,103.81,53.79 /none)"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 0.5 t)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(168.34       103.81       53.79 / 0.5)"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 0.2 t)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(168.34    ,  103.81     ,  53.79, 0.2)"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 1 nil)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(  168.34      103.81       53.79 / foo )"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 1 t)
        (cl-colors2:parse-rgb/a-to-rgb "rgb(168.34    ,  103.81,53.79, 9.9)"))
    (assert-equality* (values (rgb 169/256 0.40710938 27/128) 0.3 t)
        (cl-colors2:parse-rgb/a-to-rgb "rgb (168.34    ,  103.81   53.79 / 0.3)"))))

(deftest print-css-rgb-test (cl-colors-suite)
  (flet ((color->string (r g b alpha)
           (with-output-to-string (*standard-output*)
             (print-css-rgb/a (rgb r g b)
                              :alpha       alpha
                              :destination *standard-output*))))
    (assert-equalp "rgb(17.85 51.76 85.94)"
        (color->string 0.070 0.203 0.337 nil))
    (assert-equalp "rgb(17.85 51.76 85.94)"
        (color->string 0.070 0.203 0.337 nil))
    (assert-equalp "rgb(255.00 170.00 255.00)"
        (cl-colors2:print-css-rgb/a "#ffaaff"))
    (assert-equalp "rgba(255.00 170.00 102.00 / 204.00)"
        (let ((rgb (hex-to-rgb "FFAA66")))
          (color->string (rgb-red   rgb)
                         (rgb-green rgb)
                         (rgb-blue  rgb)
                         .8)))))

(deftest print-css-hsl-test (cl-colors-suite)
  (flet ((color->string (h s l)
           (with-output-to-string (*standard-output*)
             (print-css-hsl (hsl h s l)
                            :destination *standard-output*))))
    (assert-equalp "hsl(26.67 81.82% 43.14%)"
        (color->string 26.67 8182/10000 4314/10000))))

(deftest print-css-hsv-test (cl-colors-suite)
  (flet ((color->string (h s v)
           (with-output-to-string (*standard-output*)
             (print-css-hsv (hsv h s v)
                            :destination *standard-output*))))
    (assert-equalp "hsv(26.67 81.82% 43.14%)"
        (color->string 26.67 8182/10000 4314/10000))))

(run)
