/******************************************************************************
**
** Copyright (C) 2009-2011 Kyle Lutz <kyle.r.lutz@gmail.com>
** All rights reserved.
**
** This file is a part of the chemkit project. For more information
** see <http://www.chemkit.org>.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
**
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in the
**     documentation and/or other materials provided with the distribution.
**   * Neither the name of the chemkit project nor the names of its
**     contributors may be used to endorse or promote products derived
**     from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
******************************************************************************/

#include "amberparameters.h"

#include <cstring>

namespace {

// --- Bond Parameters ----------------------------------------------------- //
struct BondParameters
{
    const char *typeA;
    const char *typeB;
    AmberBondParameters parameters;
};

const struct BondParameters BondParameters[] = {
    {"OW", "HW", {553.0, 0.9572}},
    {"HW", "HW", {553.0, 1.5136}},
    {"C", "C", {310.0, 1.525}},
    {"C", "CA", {469.0, 1.409}},
    {"C", "CB", {447.0, 1.419}},
    {"C", "CM", {410.0, 1.444}},
    {"C", "CT", {317.0, 1.522}},
    {"C", "N", {490.0, 1.335}},
    {"C", "N*", {424.0, 1.383}},
    {"C", "NA", {418.0, 1.388}},
    {"C", "NC", {457.0, 1.358}},
    {"C", "O", {570.0, 1.229}},
    {"C", "O2", {656.0, 1.250}},
    {"C", "OH", {450.0, 1.364}},
    {"C", "OS", {450.0, 1.323}},
    {"C", "H4", {367.0, 1.080}},
    {"C", "H5", {367.0, 1.080}},
    {"CA", "CA", {469.0, 1.400}},
    {"CA", "CB", {469.0, 1.404}},
    {"CA", "CM", {427.0, 1.433}},
    {"CA", "CN", {469.0, 1.400}},
    {"CA", "CT", {317.0, 1.510}},
    {"CA", "HA", {367.0, 1.080}},
    {"CA", "H4", {367.0, 1.080}},
    {"CA", "N2", {481.0, 1.340}},
    {"CA", "NA", {427.0, 1.381}},
    {"CA", "NC", {483.0, 1.339}},
    {"CA", "OH", {450.0, 1.364}},
    {"CB", "CB", {520.0, 1.370}},
    {"CB", "N*", {436.0, 1.374}},
    {"CB", "NB", {414.0, 1.391}},
    {"CB", "NC", {461.0, 1.354}},
    {"CD", "HA", {367.0, 1.080}},
    {"CD", "CD", {469.0, 1.400}},
    {"CD", "CM", {549.0, 1.350}},
    {"CD", "CT", {317.0, 1.510}},
    {"CK", "H5", {367.0, 1.080}},
    {"CK", "N*", {440.0, 1.371}},
    {"CK", "NB", {529.0, 1.304}},
    {"CM", "CM", {549.0, 1.350}},
    {"CM", "CT", {317.0, 1.510}},
    {"CM", "HA", {367.0, 1.080}},
    {"CM", "H4", {367.0, 1.080}},
    {"CM", "H5", {367.0, 1.080}},
    {"CM", "N*", {448.0, 1.365}},
    {"CM", "OS", {480.0, 1.240}},
    {"CQ", "H5", {367.0, 1.080}},
    {"CQ", "NC", {502.0, 1.324}},
    {"CT", "CT", {310.0, 1.526}},
    {"CT", "HC", {340.0, 1.090}},
    {"CT", "H1", {340.0, 1.090}},
    {"CT", "H2", {340.0, 1.090}},
    {"CT", "H3", {340.0, 1.090}},
    {"CT", "HP", {340.0, 1.090}},
    {"CT", "N*", {337.0, 1.475}},
    {"CT", "N2", {337.0, 1.463}},
    {"CT", "OH", {320.0, 1.410}},
    {"CT", "OS", {320.0, 1.410}},
    {"C*", "HC", {367.0, 1.080}},
    {"C*", "CB", {388.0, 1.459}},
    {"C*", "CT", {317.0, 1.495}},
    {"C*", "CW", {546.0, 1.352}},
    {"CB", "CN", {447.0, 1.419}},
    {"CC", "CT", {317.0, 1.504}},
    {"CC", "CV", {512.0, 1.375}},
    {"CC", "CW", {518.0, 1.371}},
    {"CC", "NA", {422.0, 1.385}},
    {"CC", "NB", {410.0, 1.394}},
    {"CN", "NA", {428.0, 1.380}},
    {"CR", "H5", {367.0, 1.080}},
    {"CR", "NA", {477.0, 1.343}},
    {"CR", "NB", {488.0, 1.335}},
    {"CT", "N", {337.0, 1.449}},
    {"CT", "N3", {367.0, 1.471}},
    {"CT", "NT", {367.0, 1.471}},
    {"CT", "S", {227.0, 1.810}},
    {"CT", "SH", {237.0, 1.810}},
    {"CT", "CY", {400.0, 1.458}},
    {"CT", "CZ", {400.0, 1.459}},
    {"CV", "H4", {367.0, 1.080}},
    {"CV", "NB", {410.0, 1.394}},
    {"CW", "H4", {367.0, 1.080}},
    {"CW", "NA", {427.0, 1.381}},
    {"CY", "NY", {600.0, 1.150}},
    {"CZ", "CZ", {600.0, 1.206}},
    {"CZ", "HZ", {400.0, 1.056}},
    {"O2", "P", {525.0, 1.480}},
    {"OH", "P", {230.0, 1.610}},
    {"OS", "P", {230.0, 1.610}},
    {"H", "N2", {434.0, 1.010}},
    {"H", "N*", {434.0, 1.010}},
    {"H", "NA", {434.0, 1.010}},
    {"H", "N", {434.0, 1.010}},
    {"H", "N3", {434.0, 1.010}},
    {"H", "NT", {434.0, 1.010}},
    {"HO", "OH", {553.0, 0.960}},
    {"HO", "OS", {553.0, 0.960}},
    {"HS", "SH", {274.0, 1.336}},
    {"S", "S", {166.0, 2.038}},
    {"F", "CT", {367.0, 1.380}},
    {"Cl", "CT", {232.0, 1.766}},
    {"Br", "CT", {159.0, 1.944}},
    {"I", "CT", {148.0, 2.166}},
    {"F", "CA", {386.0, 1.359}},
    {"Cl", "CA", {193.0, 1.727}},
    {"I", "CA", {171.0, 2.075}},
    {"Br", "CA", {172.0, 1.890}}
};

const int BondParametersCount = sizeof(BondParameters) / sizeof(*BondParameters);

// --- Angle Parameters ---------------------------------------------------- //
struct AngleParameters
{
    const char *typeA;
    const char *typeB;
    const char *typeC;
    AmberAngleParameters parameters;
};

const struct AngleParameters AngleParameters[] = {
    {"HW", "OW", "HW", {100., 104.52}},
    {"HW", "HW", "OW", {0., 127.74}},
    {"C", "C", "O", {80.0, 120.00}},
    {"C", "C", "OH", {80.0, 120.00}},
    {"CA", "C", "CA", {63.0, 120.00}},
    {"CA", "C", "OH", {70.0, 120.00}},
    {"CB", "C", "NA", {70.0, 111.30}},
    {"CB", "C", "O", {80.0, 128.80}},
    {"CM", "C", "NA", {70.0, 114.10}},
    {"CM", "C", "O", {80.0, 125.30}},
    {"CT", "C", "O", {80.0, 120.40}},
    {"CT", "C", "O2", {70.0, 117.00}},
    {"CT", "C", "N", {70.0, 116.60}},
    {"CT", "C", "CT", {63.0, 117.00}},
    {"CT", "C", "OS", {80.0, 115.00}},
    {"CT", "C", "OH", {80.0, 110.00}},
    {"N*", "C", "NA", {70.0, 115.40}},
    {"N*", "C", "NC", {70.0, 118.60}},
    {"N*", "C", "O", {80.0, 120.90}},
    {"NA", "C", "O", {80.0, 120.60}},
    {"NC", "C", "O", {80.0, 122.50}},
    {"N", "C", "O", {80.0, 122.90}},
    {"O", "C", "O", {80.0, 126.00}},
    {"O", "C", "OH", {80.0, 120.00}},
    {"O", "C", "OS", {80.0, 125.00}},
    {"O2", "C", "O2", {80.0, 126.00}},
    {"H4", "C", "C", {50.0, 120.00}},
    {"H4", "C", "CM", {50.0, 115.00}},
    {"H4", "C", "CT", {50.0, 115.00}},
    {"H4", "C", "O", {50.0, 120.00}},
    {"H4", "C", "OH", {50.0, 120.00}},
    {"H5", "C", "N", {50.0, 120.00}},
    {"H5", "C", "O", {50.0, 119.00}},
    {"H5", "C", "OH", {50.0, 107.00}},
    {"H5", "C", "OS", {50.0, 107.00}},
    {"C", "CA", "CA", {63.0, 120.00}},
    {"C", "CA", "HA", {50.0, 120.00}},
    {"CA", "CA", "CA", {63.0, 120.00}},
    {"CA", "CA", "CB", {63.0, 120.00}},
    {"CA", "CA", "CT", {70.0, 120.00}},
    {"CA", "CA", "HA", {50.0, 120.00}},
    {"CA", "CA", "H4", {50.0, 120.00}},
    {"CA", "CA", "OH", {70.0, 120.00}},
    {"CA", "CA", "CN", {63.0, 120.00}},
    {"CB", "CA", "HA", {50.0, 120.00}},
    {"CB", "CA", "H4", {50.0, 120.00}},
    {"CB", "CA", "N2", {70.0, 123.50}},
    {"CB", "CA", "NC", {70.0, 117.30}},
    {"CM", "CA", "N2", {70.0, 120.10}},
    {"CM", "CA", "NC", {70.0, 121.50}},
    {"CN", "CA", "HA", {50.0, 120.00}},
    {"NA", "CA", "NC", {70.0, 123.30}},
    {"N2", "CA", "NA", {70.0, 116.00}},
    {"N2", "CA", "NC", {70.0, 119.30}},
    {"N2", "CA", "N2", {70.0, 120.00}},
    {"F", "CA", "CA", {70.0, 121.00}},
    {"Cl", "CA", "CA", {70.0, 118.80}},
    {"Br", "CA", "CA", {70.0, 118.80}},
    {"I", "CA", "CA", {70.0, 118.80}},
    {"C", "CB", "CB", {63.0, 119.20}},
    {"C", "CB", "NB", {70.0, 130.00}},
    {"CA", "CB", "CB", {63.0, 117.30}},
    {"CA", "CB", "NB", {70.0, 132.40}},
    {"CB", "CB", "N*", {70.0, 106.20}},
    {"CB", "CB", "NB", {70.0, 110.40}},
    {"CB", "CB", "NC", {70.0, 127.70}},
    {"C*", "CB", "CA", {63.0, 134.90}},
    {"C*", "CB", "CN", {63.0, 108.80}},
    {"CA", "CB", "CN", {63.0, 116.20}},
    {"N*", "CB", "NC", {70.0, 126.20}},
    {"CD", "CD", "CM", {63.0, 120.00}},
    {"CD", "CD", "CT", {70.0, 120.00}},
    {"CM", "CD", "CT", {70.0, 120.00}},
    {"HA", "CD", "HA", {35.0, 119.00}},
    {"HA", "CD", "CD", {50.0, 120.00}},
    {"HA", "CD", "CM", {50.0, 120.00}},
    {"H5", "CK", "N*", {50.0, 123.05}},
    {"H5", "CK", "NB", {50.0, 123.05}},
    {"N*", "CK", "NB", {70.0, 113.90}},
    {"C", "CM", "CM", {63.0, 120.70}},
    {"C", "CM", "CT", {70.0, 119.70}},
    {"C", "CM", "HA", {50.0, 119.70}},
    {"C", "CM", "H4", {50.0, 119.70}},
    {"CA", "CM", "CM", {63.0, 117.00}},
    {"CA", "CM", "HA", {50.0, 123.30}},
    {"CA", "CM", "H4", {50.0, 123.30}},
    {"CM", "CM", "CT", {70.0, 119.70}},
    {"CM", "CM", "HA", {50.0, 119.70}},
    {"CM", "CM", "H4", {50.0, 119.70}},
    {"CM", "CM", "N*", {70.0, 121.20}},
    {"CM", "CM", "OS", {80.0, 125.00}},
    {"H4", "CM", "N*", {50.0, 119.10}},
    {"H4", "CM", "OS", {50.0, 113.00}},
    {"HA", "CM", "HA", {35.0, 120.00}},
    {"HA", "CM", "CD", {50.0, 120.00}},
    {"HA", "CM", "CT", {50.0, 120.00}},
    {"NC", "CQ", "NC", {70.0, 129.10}},
    {"H5", "CQ", "NC", {50.0, 115.45}},
    {"H1", "CT", "H1", {35.0, 109.50}},
    {"H1", "CT", "N*", {50.0, 109.50}},
    {"H1", "CT", "OH", {50.0, 109.50}},
    {"H1", "CT", "OS", {50.0, 109.50}},
    {"H1", "CT", "CM", {50.0, 109.50}},
    {"H1", "CT", "CY", {50.0, 110.00}},
    {"H1", "CT", "CZ", {50.0, 110.00}},
    {"H1", "CT", "N", {50.0, 109.50}},
    {"H1", "CT", "S", {50.0, 109.50}},
    {"H1", "CT", "SH", {50.0, 109.50}},
    {"H1", "CT", "N2", {50.0, 109.50}},
    {"H1", "CT", "NT", {50.0, 109.50}},
    {"H2", "CT", "H2", {35.0, 109.50}},
    {"H2", "CT", "N*", {50.0, 109.50}},
    {"H2", "CT", "OS", {50.0, 109.50}},
    {"HP", "CT", "HP", {35.0, 109.50}},
    {"HP", "CT", "N3", {50.0, 109.50}},
    {"HC", "CT", "HC", {35.0, 109.50}},
    {"HC", "CT", "CM", {50.0, 109.50}},
    {"HC", "CT", "CD", {50.0, 109.50}},
    {"HC", "CT", "CZ", {50.0, 110.00}},
    {"C", "CT", "H1", {50.0, 109.50}},
    {"C", "CT", "HP", {50.0, 109.50}},
    {"C", "CT", "HC", {50.0, 109.50}},
    {"C", "CT", "N", {63.0, 110.10}},
    {"C", "CT", "N3", {80.0, 111.20}},
    {"C", "CT", "CT", {63.0, 111.10}},
    {"C", "CT", "OS", {60.0, 109.50}},
    {"CA", "CT", "HC", {50.0, 109.50}},
    {"CC", "CT", "CT", {63.0, 113.10}},
    {"CC", "CT", "HC", {50.0, 109.50}},
    {"CM", "CT", "CT", {63.0, 111.00}},
    {"CM", "CT", "OS", {50.0, 109.50}},
    {"CT", "CT", "CT", {40.0, 109.50}},
    {"CT", "CT", "HC", {50.0, 109.50}},
    {"CT", "CT", "H1", {50.0, 109.50}},
    {"CT", "CT", "H2", {50.0, 109.50}},
    {"CT", "CT", "HP", {50.0, 109.50}},
    {"CT", "CT", "N*", {50.0, 109.50}},
    {"CT", "CT", "OH", {50.0, 109.50}},
    {"CT", "CT", "OS", {50.0, 109.50}},
    {"CT", "CT", "S", {50.0, 114.70}},
    {"CT", "CT", "SH", {50.0, 108.60}},
    {"CT", "CT", "CA", {63.0, 114.00}},
    {"CT", "CT", "N2", {80.0, 111.20}},
    {"CT", "CT", "N", {80.0, 109.70}},
    {"CT", "CT", "N3", {80.0, 111.20}},
    {"CT", "CT", "NT", {80.0, 111.20}},
    {"CT", "CT", "CY", {63.0, 110.00}},
    {"CT", "CT", "CZ", {63.0, 110.00}},
    {"C*", "CT", "CT", {63.0, 115.60}},
    {"C*", "CT", "HC", {50.0, 109.50}},
    {"OS", "CT", "OS", {160.0, 101.00}},
    {"OS", "CT", "CY", {50.0, 110.00}},
    {"OS", "CT", "CZ", {50.0, 110.00}},
    {"OS", "CT", "N*", {50.0, 109.50}},
    {"F", "CT", "F", {77.0, 109.10}},
    {"F", "CT", "H1", {50.0, 109.50}},
    {"F", "CT", "CT", {50.0, 109.00}},
    {"F", "CT", "H2", {50.0, 109.50}},
    {"Cl", "CT", "CT", {50.0, 108.50}},
    {"Cl", "CT", "H1", {50.0, 108.50}},
    {"Br", "CT", "CT", {50.0, 108.00}},
    {"Br", "CT", "H1", {50.0, 106.50}},
    {"I", "CT", "CT", {50.0, 106.00}},
    {"CT", "CC", "NA", {70.0, 120.00}},
    {"CT", "CC", "CV", {70.0, 120.00}},
    {"CT", "CC", "NB", {70.0, 120.00}},
    {"CV", "CC", "NA", {70.0, 120.00}},
    {"CW", "CC", "NA", {70.0, 120.00}},
    {"CW", "CC", "NB", {70.0, 120.00}},
    {"CT", "CC", "CW", {70.0, 120.00}},
    {"H5", "CR", "NA", {50.0, 120.00}},
    {"H5", "CR", "NB", {50.0, 120.00}},
    {"NA", "CR", "NA", {70.0, 120.00}},
    {"NA", "CR", "NB", {70.0, 120.00}},
    {"CC", "CV", "H4", {50.0, 120.00}},
    {"CC", "CV", "NB", {70.0, 120.00}},
    {"H4", "CV", "NB", {50.0, 120.00}},
    {"CC", "CW", "H4", {50.0, 120.00}},
    {"CC", "CW", "NA", {70.0, 120.00}},
    {"C*", "CW", "H4", {50.0, 120.00}},
    {"C*", "CW", "NA", {70.0, 108.70}},
    {"H4", "CW", "NA", {50.0, 120.00}},
    {"CB", "C*", "CT", {70.0, 128.60}},
    {"CB", "C*", "CW", {63.0, 106.40}},
    {"CT", "C*", "CW", {70.0, 125.00}},
    {"CA", "CN", "CB", {63.0, 122.70}},
    {"CA", "CN", "NA", {70.0, 132.80}},
    {"CB", "CN", "NA", {70.0, 104.40}},
    {"CT", "CY", "NY", {80.0, 180.00}},
    {"CT", "CZ", "CZ", {80.0, 180.00}},
    {"CZ", "CZ", "HZ", {50.0, 180.00}},
    {"C", "N", "CT", {50.0, 121.90}},
    {"C", "N", "H", {50.0, 120.00}},
    {"CT", "N", "H", {50.0, 118.04}},
    {"CT", "N", "CT", {50.0, 118.00}},
    {"H", "N", "H", {35.0, 120.00}},
    {"C", "N*", "CM", {70.0, 121.60}},
    {"C", "N*", "CT", {70.0, 117.60}},
    {"C", "N*", "H", {50.0, 119.20}},
    {"CB", "N*", "CK", {70.0, 105.40}},
    {"CB", "N*", "CT", {70.0, 125.80}},
    {"CB", "N*", "H", {50.0, 125.80}},
    {"CK", "N*", "CT", {70.0, 128.80}},
    {"CK", "N*", "H", {50.0, 128.80}},
    {"CM", "N*", "CT", {70.0, 121.20}},
    {"CM", "N*", "H", {50.0, 121.20}},
    {"CA", "N2", "H", {50.0, 120.00}},
    {"CA", "N2", "CT", {50.0, 123.20}},
    {"CT", "N2", "H", {50.0, 118.40}},
    {"H", "N2", "H", {35.0, 120.00}},
    {"CT", "N3", "H", {50.0, 109.50}},
    {"CT", "N3", "CT", {50.0, 109.50}},
    {"H", "N3", "H", {35.0, 109.50}},
    {"CT", "NT", "H", {50.0, 109.50}},
    {"CT", "NT", "CT", {50.0, 109.50}},
    {"H", "NT", "H", {35.0, 109.50}},
    {"C", "NA", "C", {70.0, 126.40}},
    {"C", "NA", "CA", {70.0, 125.20}},
    {"C", "NA", "H", {50.0, 116.80}},
    {"CA", "NA", "H", {50.0, 118.00}},
    {"CC", "NA", "CR", {70.0, 120.00}},
    {"CC", "NA", "H", {50.0, 120.00}},
    {"CR", "NA", "CW", {70.0, 120.00}},
    {"CR", "NA", "H", {50.0, 120.00}},
    {"CW", "NA", "H", {50.0, 120.00}},
    {"CN", "NA", "CW", {70.0, 111.60}},
    {"CN", "NA", "H", {50.0, 123.10}},
    {"CB", "NB", "CK", {70.0, 103.80}},
    {"CC", "NB", "CR", {70.0, 117.00}},
    {"CR", "NB", "CV", {70.0, 117.00}},
    {"C", "NC", "CA", {70.0, 120.50}},
    {"CA", "NC", "CB", {70.0, 112.20}},
    {"CA", "NC", "CQ", {70.0, 118.60}},
    {"CB", "NC", "CQ", {70.0, 111.00}},
    {"C", "OH", "HO", {50.0, 113.00}},
    {"CA", "OH", "HO", {50.0, 113.00}},
    {"CT", "OH", "HO", {55.0, 108.50}},
    {"HO", "OH", "P", {45.0, 108.50}},
    {"C", "OS", "CT", {60.0, 117.00}},
    {"CM", "OS", "CT", {60.0, 117.00}},
    {"CT", "OS", "CT", {60.0, 109.50}},
    {"CT", "OS", "P", {100.0, 120.50}},
    {"P", "OS", "P", {100.0, 120.50}},
    {"O2", "P", "OH", {45.0, 108.23}},
    {"O2", "P", "O2", {140.0, 119.90}},
    {"O2", "P", "OS", {100.0, 108.23}},
    {"OH", "P", "OS", {45.0, 102.60}},
    {"OS", "P", "OS", {45.0, 102.60}},
    {"CT", "S", "CT", {62.0, 98.90}},
    {"CT", "S", "S", {68.0, 103.70}},
    {"CT", "SH", "HS", {43.0, 96.00}},
    {"HS", "SH", "HS", {35.0, 92.07}}
};

const int AngleParametersCount = sizeof(AngleParameters) / sizeof(*AngleParameters);

// --- Torsion Parameters -------------------------------------------------- //
struct TorsionParameters
{
    const char *typeA;
    const char *typeB;
    const char *typeC;
    const char *typeD;
    AmberTorsionParameters parameters;
};

const struct TorsionParameters TorsionParameters[] = {
    {"X", "C", "C", "X", {0.000, 0.276, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "C", "CA", "X", {0.000, 0.276, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "C", "CB", "X", {0.000, 0.333, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "C", "CM", "X", {0.000, 0.460, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "C", "CT", "X", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "C", "N", "X", {0.000, 0.400, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "C", "N*", "X", {0.000, 0.690, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "C", "NA", "X", {0.000, 0.741, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "C", "NC", "X", {0.000, 0.250, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "C", "O", "X", {0.000, 0.357, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "C", "OH", "X", {0.000, 0.435, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "C", "OS", "X", {0.000, 0.370, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CA", "CA", "X", {0.000, 0.276, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CA", "CB", "X", {0.000, 0.286, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CA", "CM", "X", {0.000, 0.392, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CA", "CN", "X", {0.000, 0.276, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CA", "CT", "X", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CA", "N2", "X", {0.000, 0.417, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CA", "NA", "X", {0.000, 0.667, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CA", "NC", "X", {0.000, 0.208, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CA", "OH", "X", {0.000, 1.111, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CB", "CB", "X", {0.000, 0.183, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CB", "CN", "X", {0.000, 0.333, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CB", "N*", "X", {0.000, 0.606, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CB", "NB", "X", {0.000, 0.392, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CB", "NC", "X", {0.000, 0.241, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CC", "CT", "X", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CC", "CV", "X", {0.000, 0.194, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CC", "CW", "X", {0.000, 0.186, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CC", "NA", "X", {0.000, 0.714, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CC", "NB", "X", {0.000, 0.417, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CD", "CD", "X", {0.000, 1.000, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CD", "CT", "X", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CD", "CM", "X", {0.000, 0.150, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CK", "N*", "X", {0.000, 0.588, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CK", "NB", "X", {0.000, 0.100, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CM", "CM", "X", {0.000, 0.150, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CM", "CT", "X", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CM", "N*", "X", {0.000, 0.541, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CM", "OS", "X", {0.000, 0.952, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CN", "NA", "X", {0.000, 0.656, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CQ", "NC", "X", {0.000, 0.147, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CT", "CT", "X", {0.000, 0.000, 6.429, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CT", "CY", "X", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CT", "CZ", "X", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CT", "N", "X", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CT", "N*", "X", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CT", "N2", "X", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CT", "NT", "X", {0.000, 0.000, 3.333, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CT", "N3", "X", {0.000, 0.000, 6.429, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CT", "OH", "X", {0.000, 0.000, 6.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CT", "OS", "X", {0.000, 0.000, 2.609, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CT", "S", "X", {0.000, 0.000, 3.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "CT", "SH", "X", {0.000, 0.000, 4.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "C*", "CB", "X", {0.000, 0.597, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "C*", "CT", "X", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "C*", "CW", "X", {0.000, 0.153, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CR", "NA", "X", {0.000, 0.430, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CR", "NB", "X", {0.000, 0.200, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CV", "NB", "X", {0.000, 0.417, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "CW", "NA", "X", {0.000, 0.667, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"X", "OH", "P", "X", {0.000, 0.000, 4.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"X", "OS", "P", "X", {0.000, 0.000, 4.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"N", "CT", "C", "N", {0.588, 0.500, 0.000, 0.000, 180.0, 180.0, 0.0, 0.0}},
    {"C", "N", "CT", "C", {1.250, 1.176, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"CT", "CT", "N", "C", {1.887, 0.000, 6.667, 2.000, 0.0, 0.0, 180.0, 180.0}},
    {"CT", "CT", "C", "N", {0.000, 14.286, 0.000, 10.000, 0.0, 0.0, 0.0, 0.0}},
    {"H", "N", "C", "O", {0.500, 0.400, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"CT", "S", "S", "CT", {0.000, 0.286, 1.667, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"OH", "P", "OS", "CT", {0.000, 0.833, 4.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"OS", "P", "OS", "CT", {0.000, 0.833, 4.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"H1", "CT", "C", "O", {1.250, 0.000, 12.500, 0.000, 0.0, 0.0, 180.0, 0.0}},
    {"HC", "CT", "C", "O", {1.250, 0.000, 12.500, 0.000, 0.0, 0.0, 180.0, 0.0}},
    {"HC", "CT", "C", "O", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"HC", "CT", "CT", "HC", {0.000, 0.000, 6.667, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"HC", "CT", "CT", "CT", {0.000, 0.000, 6.250, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"HC", "CT", "CM", "CM", {0.870, 0.000, 2.632, 0.000, 0.0, 0.0, 180.0, 0.0}},
    {"HO", "OH", "CT", "CT", {4.000, 0.000, 6.250, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"HO", "OH", "C", "O", {0.526, 0.435, 0.000, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"CM", "CM", "C", "O", {0.000, 0.460, 3.333, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"CT", "CM", "CM", "CT", {0.526, 0.150, 0.000, 0.000, 180.0, 180.0, 0.0, 0.0}},
    {"CT", "CT", "CT", "CT", {5.000, 4.000, 5.556, 0.000, 180.0, 180.0, 0.0, 0.0}},
    {"CT", "CT", "NT", "CT", {0.000, 2.083, 3.333, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"CT", "CT", "OS", "CT", {0.000, 10.000, 2.611, 0.000, 0.0, 180.0, 0.0, 0.0}},
    {"CT", "CT", "OS", "C", {1.250, 0.000, 2.611, 0.000, 180.0, 0.0, 0.0, 0.0}},
    {"CT", "OS", "CT", "OS", {0.741, 1.176, 10.000, 0.000, 180.0, 180.0, 0.0, 0.0}},
    {"CT", "OS", "CT", "N*", {0.000, 1.538, 2.611, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"CT", "CZ", "CZ", "HZ", {0.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"O", "C", "OS", "CT", {0.714, 0.370, 0.000, 0.000, 180.0, 180.0, 0.0, 0.0}},
    {"OS", "CT", "N*", "CK", {0.400, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"OS", "CT", "N*", "CM", {0.400, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"OS", "CT", "CT", "OS", {0.000, 0.851, 6.944, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"OS", "CT", "CT", "OH", {0.000, 0.851, 6.944, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"OH", "CT", "CT", "OH", {0.000, 0.851, 6.944, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"F", "CT", "CT", "F", {0.833, 0.000, 0.000, 0.000, 180.0, 0.0, 0.0, 0.0}},
    {"Cl", "CT", "CT", "Cl", {2.222, 0.000, 0.000, 0.000, 180.0, 0.0, 0.0, 0.0}},
    {"Br", "CT", "CT", "Br", {0.000, 0.000, 0.000, 0.000, 180.0, 0.0, 0.0, 0.0}},
    {"H1", "CT", "CT", "OS", {4.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"H1", "CT", "CT", "OH", {4.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"H1", "CT", "CT", "F", {5.263, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"H1", "CT", "CT", "Cl", {4.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"H1", "CT", "CT", "Br", {1.818, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"HC", "CT", "CT", "OS", {4.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"HC", "CT", "CT", "OH", {4.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"HC", "CT", "CT", "F", {5.263, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"HC", "CT", "CT", "Cl", {4.000, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}},
    {"HC", "CT", "CT", "Br", {1.818, 0.000, 0.000, 0.000, 0.0, 0.0, 0.0, 0.0}}
};

const int TorsionParametersCount = sizeof(TorsionParameters) / sizeof(*TorsionParameters);

// --- Nonbonded Parameters ------------------------------------------------ //
struct NonbondedParameters
{
    const char *type;
    AmberNonbondedParameters parameters;
};

const struct NonbondedParameters NonbondedParameters[] = {
    {"H", {0.6000, 0.0157}},
    {"HO", {0.0000, 0.0000}},
    {"HS", {0.6000, 0.0157}},
    {"HC", {1.4870, 0.0157}},
    {"H1", {1.3870, 0.0157}},
    {"H2", {1.2870, 0.0157}},
    {"H3", {1.1870, 0.0157}},
    {"HP", {1.1000, 0.0157}},
    {"HA", {1.4590, 0.0150}},
    {"H4", {1.4090, 0.0150}},
    {"H5", {1.3590, 0.0150}},
    {"HW", {0.0000, 0.0000}},
    {"HZ", {1.4590, 0.0150}},
    {"O", {1.6612, 0.2100}},
    {"O2", {1.6612, 0.2100}},
    {"OW", {1.7683, 0.1520}},
    {"OH", {1.7210, 0.2104}},
    {"OS", {1.6837, 0.1700}},
    {"C*", {1.9080, 0.0860}},
    {"CA", {1.9080, 0.0860}},
    {"CB", {1.9080, 0.0860}},
    {"CC", {1.9080, 0.0860}},
    {"CD", {1.9080, 0.0860}},
    {"CK", {1.9080, 0.0860}},
    {"CM", {1.9080, 0.0860}},
    {"CN", {1.9080, 0.0860}},
    {"CQ", {1.9080, 0.0860}},
    {"CR", {1.9080, 0.0860}},
    {"CV", {1.9080, 0.0860}},
    {"CW", {1.9080, 0.0860}},
    {"CY", {1.9080, 0.0860}},
    {"CZ", {1.9080, 0.0860}},
    {"CT", {1.9080, 0.1094}},
    {"C", {1.9080, 0.0860}},
    {"N", {1.8240, 0.1700}},
    {"NA", {1.8240, 0.1700}},
    {"N2", {1.8240, 0.1700}},
    {"N*", {1.8240, 0.1700}},
    {"NC", {1.8240, 0.1700}},
    {"NB", {1.8240, 0.1700}},
    {"NT", {1.8240, 0.1700}},
    {"NY", {1.8240, 0.1700}},
    {"N3", {1.8240, 0.1700}},
    {"NY", {1.8240, 0.1700}},
    {"S", {2.0000, 0.2500}},
    {"SH", {2.0000, 0.2500}},
    {"P", {2.1000, 0.2000}},
    {"IM", {2.47, 0.1}},
    {"Li", {1.1370, 0.0183}},
    {"IP", {1.8680, 0.00277}},
    {"Na", {1.8680, 0.00277}},
    {"K", {2.6580, 0.000328}},
    {"Rb", {2.9560, 0.00017}},
    {"Cs", {3.3950, 0.0000806}},
    {"MG", {0.7926, 0.8947}},
    {"C0", {1.7131, 0.459789}},
    {"Zn", {1.10, 0.0125}},
    {"F", {1.75, 0.061}},
    {"Cl", {1.948, 0.265}},
    {"Br", {2.22, 0.320}},
    {"I", {2.35, 0.40}},
    {"IB", {5.0, 0.1}}
};

const int NonbondedParametersCount = sizeof(NonbondedParameters) / sizeof(*NonbondedParameters);

} // end anonymous namespace

// === AmberParameters ===================================================== //
// --- Construction and Destruction ---------------------------------------- //
AmberParameters::AmberParameters()
{
}

AmberParameters::~AmberParameters()
{
}

// --- Parameters ---------------------------------------------------------- //
const AmberBondParameters* AmberParameters::bondParameters(const std::string &typeA, const std::string &typeB) const
{
    for(int i = 0; i < BondParametersCount; i++){
        const struct BondParameters *parameters = &BondParameters[i];

        if((typeA == parameters->typeA && typeB == parameters->typeB) ||
           (typeA == parameters->typeB && (typeB == parameters->typeA))){
            return &parameters->parameters;
        }
    }

    return 0;
}

const AmberAngleParameters* AmberParameters::angleParameters(const std::string &typeA, const std::string &typeB, const std::string &typeC) const
{
    for(int i = 0; i < AngleParametersCount; i++){
        const struct AngleParameters *parameters = &AngleParameters[i];

        if(typeB == parameters->typeB){
            if((typeA == parameters->typeA && typeC == parameters->typeC) ||
               (typeA == parameters->typeC && typeC == parameters->typeA)){
                return &parameters->parameters;
            }
        }
    }

    return 0;
}

const AmberTorsionParameters* AmberParameters::torsionParameters(const std::string &typeA, const std::string &typeB, const std::string &typeC, const std::string &typeD) const
{
    for(int i = 0; i < TorsionParametersCount; i++){
        const struct TorsionParameters *parameters = &TorsionParameters[i];

        if((typeB == parameters->typeB && typeC == parameters->typeC) ||
           (typeC == parameters->typeB && typeB == parameters->typeC)){
            if(strcmp("X", parameters->typeA) == 0){
                return &parameters->parameters;
            }
            else if((typeA == parameters->typeA && typeD == parameters->typeD) ||
                    (typeA == parameters->typeD && typeD == parameters->typeA)){
                return &parameters->parameters;
            }
        }
    }

    return 0;
}

const AmberNonbondedParameters* AmberParameters::nonbondedParameters(const std::string &type) const
{
    for(int i = 0; i < NonbondedParametersCount; i++){
        const struct NonbondedParameters *parameters = &NonbondedParameters[i];

        if(type == parameters->type){
            return &parameters->parameters;
        }
    }

    return 0;
}
