/* The Computer Language Benchmarks Game
   http://benchmarksgame.alioth.debian.org/

   contributed by Hannah Hemmaplardh
   modified by Lydia Duncan
*/

config const numMeetings : int = 6000000;  // number of meetings to take place
config const numChameneos1 : int(32) = 3;  // size of population 1
config const numChameneos2 : int(32) = 10; // size of population 2
enum Color {blue=0, red=1, yellow=2};
enum Digit {zero, one, two, three, four,
            five, six, seven, eight, nine};
config const verbose = false;
// if verbose is true, prints out non-det output, otherwise prints det output
config param CHAMENEOS_IDX_MASK = 0xFF: uint(32);
config param MEET_COUNT_SHIFT = 8;

class MeetingPlace {
  var state : atomic uint(32);

  /* constructor for MeetingPlace, sets the
     number of meetings to take place */
  proc MeetingPlace() {
    state.write((numMeetings << MEET_COUNT_SHIFT) : uint(32));
  }

  /* reset must be called after meet,
     to reset numMeetings for a subsequent call of meet */
  proc reset() {
    state.write((numMeetings << MEET_COUNT_SHIFT) : uint(32));
  }
}



/* getComplement returns the complement of this and another color:
   if this and the other color are of the same value, return own value
   otherwise return the color that is neither this or the other color */
inline proc getComplement(myColor : Color, otherColor : Color) {
  if (myColor == otherColor) {
    return myColor;
  }
  return (3 - myColor - otherColor) : Color;
}

class Chameneos {
  var id: int(32);
  var color : Color;
  var meetings : int;
  var meetingsWithSelf : int;
  var meetingCompleted : atomic bool;

  /* start tells a Chameneos to go to a given MeetingPlace, where it may meet
     with another Chameneos.  If it does, it will get the other's color and
     use this color and its own to compute the color both will have after the
     meeting has ended, setting both of their colors to this value. */
  proc start(population : [] Chameneos, meetingPlace: MeetingPlace) {
    var stateTemp : uint(32);
    var peer_idx : uint(32);
    var xchg : uint(32);

    stateTemp = meetingPlace.state.read(memory_order_acquire);

    while (true) {
      peer_idx = stateTemp & CHAMENEOS_IDX_MASK;
      if (peer_idx) {
        xchg = stateTemp - peer_idx - (1 << MEET_COUNT_SHIFT):uint(32);
      } else if (stateTemp) {
        xchg = stateTemp | id;
      } else {
        break;
      }
      if (meetingPlace.state.compareExchangeStrong(stateTemp, xchg, memory_order_acq_rel)) {
        if (peer_idx) {
          runMeeting(population, peer_idx);
        } else {
          // Attend meeting

          // This version uses waitFor instead of chpl_task_yield
          meetingCompleted.waitFor(true);

          meetingCompleted.write(false, memory_order_release);
          stateTemp = meetingPlace.state.read(memory_order_acquire);
        }
      } else {
        stateTemp = meetingPlace.state.read(memory_order_acquire);
      }
    }
  }

  /* Given the id of its peer, finds and updates the data of its peer and
     itself */
  proc runMeeting (population : [] Chameneos, peer_idx : uint(32)) {
    var peer : Chameneos;
    var is_same : int;
    var newColor : Color;
    if (id == peer_idx) {
      is_same = 1;
      halt("halt: chameneos met with self");
    }
    peer = population[peer_idx:int(32)];
    newColor = getComplement(color, peer.color);
    peer.color = newColor;
    peer.meetings += 1;
    peer.meetingsWithSelf += is_same;
    peer.meetingCompleted.write(true, memory_order_release);
    
    color = newColor;
    meetings += 1;
    meetingsWithSelf += is_same;
  }
}

/* printColorChanges prints the result of getComplement for all possible
   pairs of colors */
proc printColorChanges() {
  const colors : [1..3] Color = (Color.blue, Color.red, Color.yellow);
  for color1 in colors {
    for color2 in colors {
      writeln(color1, " + ", color2, " -> ", getComplement(color1, color2));
    }
  }
  writeln();
}

/* populate takes an parameter of type int, size, and returns a population of
   chameneos of that size. if population size is set to 10, will use preset
   array of colors  */
proc populate (size : int(32)) {
  const colorsDefault10  = (Color.blue, Color.red, Color.yellow, Color.red,
                            Color.yellow, Color.blue, Color.red, Color.yellow,
                            Color.red, Color.blue);
  const D : domain(1, int(32)) = {1..size};
  var population : [D] Chameneos;

  if (size == 10) {
    for i in D {
      population(i) = new Chameneos(i, colorsDefault10(i));
    }
  } else {
    for i in D {
      population(i) = new Chameneos(i, ((i-1) % 3):Color);
    }
  }
  return population;
}

/* run takes a population of Chameneos and a MeetingPlace, then allows the
   Chameneos to meet. It then prints out the number of times each Chameneos
   met another Chameneos, spells out the number of times it met with itself,
   then spells out the total number of times all the Chameneos met
   another Chameneos. */
proc run(population : [] Chameneos, meetingPlace : MeetingPlace) {
  for i in population {
    write(" ", i.color);
  }
  writeln();

  coforall i in population {
    i.start(population, meetingPlace);
  }

  meetingPlace.reset();
}

proc runQuiet(population : [] Chameneos, meetingPlace : MeetingPlace) {
  coforall i in population {
    i.start(population, meetingPlace);
  }
  meetingPlace.reset();

  const totalMeetings = + reduce population.meetings;
  const totalMeetingsWithSelf = + reduce population.meetingsWithSelf;
  if (totalMeetings == numMeetings*2) {
    writeln("total meetings PASS");
  } else {
    writeln("total meetings actual = ", totalMeetings, ", total meetings expected = ", numMeetings*2);
  }

  if (totalMeetingsWithSelf == 0) {
    writeln("total meetings with self PASS");
  } else {
    writeln("total meetings with self actual = ", totalMeetingsWithSelf, ", total meetings with self expected = 0");
  }

  writeln();
}

proc printInfo(population : [] Chameneos) {
  for i in population {
    write(i.meetings);
    spellInt(i.meetingsWithSelf);
  }
  const totalMeetings = + reduce population.meetings;
  spellInt(totalMeetings);
  writeln();
}

/* spellInt takes an integer, and spells each of its digits out in English */
proc spellInt(n : int) {
  var s : string = n:string;
  for i in 1..s.length {
    write(" ", (s.substring(i):int + 1):Digit);
  }
  writeln();
}

proc main() {
  if (numChameneos1 < 2 || numChameneos2 < 2 || numMeetings < 0) {
    writeln("Please specify numChameneos1 and numChameneos2 of at least 2, and numMeetings of at least 0.");
  } else  {
    printColorChanges();

    const forest : MeetingPlace = new MeetingPlace();

    const population1 = populate(numChameneos1);
    const population2 = populate(numChameneos2);

    if (verbose) {
      run(population1, forest);
      printInfo(population1);

      run(population2, forest);
      printInfo(population2);
    } else {
      runQuiet(population1, forest);
      runQuiet(population2, forest);
    }
  }
}

