/*
 * Copyright 2004-2016 Cray Inc.
 * Other additional copyright holders may be indicated within.
 *
 * The entirety of this work is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 *
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "astutil.h"
#include "stlUtil.h"
#include "build.h"
#include "expr.h"
#include "passes.h"
#include "stmt.h"
#include "stringutil.h"
#include "symbol.h"
#include "config.h"

static bool mainReturnsInt;

static void build_chpl_entry_points();
static void build_accessor(AggregateType* ct, Symbol* field, bool setter);
static void build_accessors(AggregateType* ct, Symbol* field);
static void build_union_assignment_function(AggregateType* ct);
static void build_enum_assignment_function(EnumType* et);
static void build_record_assignment_function(AggregateType* ct);
static void build_extern_init_function(Type* type);
static void build_extern_assignment_function(Type* type);
static void build_record_cast_function(AggregateType* ct);
static void build_record_copy_function(AggregateType* ct);
static void build_record_hash_function(AggregateType* ct);
static void build_record_init_function(AggregateType* ct);
static void build_record_equality_function(AggregateType* ct);
static void build_record_inequality_function(AggregateType* ct);
static void build_enum_cast_function(EnumType* et);
static void build_enum_first_function(EnumType* et);
static void build_enum_enumerate_function(EnumType* et);

//static void buildDefaultReadFunction(AggregateType* type);
//static void buildDefaultReadFunction(EnumType* type);

static void buildDefaultReadWriteFunctions(AggregateType* type);

static void buildStringCastFunction(EnumType* type);

static void buildFieldAccessorFunctions(AggregateType* at);


void buildDefaultFunctions() {
  build_chpl_entry_points();

  SET_LINENO(rootModule); // todo - remove reset_ast_loc() calls below?

  std::vector<BaseAST*> asts;
  collect_asts(rootModule, asts);
  for_vector(BaseAST, ast, asts) {
    if (TypeSymbol* type = toTypeSymbol(ast)) {
      // Here we build default functions that are always generated (even when
      // the type symbol has FLAG_NO_DEFAULT_FUNCTIONS attached).
      if (AggregateType* ct = toAggregateType(type->type)) {
        buildFieldAccessorFunctions(ct);

        if (!ct->symbol->hasFlag(FLAG_REF))
          buildDefaultDestructor(ct);
        // Classes should use the nil:<type> _defaultOf method unless they
        // do not inherit from object.  For those types and records, call
        // we need a more complicated _defaultOf method generated by the
        // compiler
        if (!ct->isClass() || ct->symbol->hasFlag(FLAG_NO_OBJECT))
          build_record_init_function(ct);
      }

      if (type->hasFlag(FLAG_NO_DEFAULT_FUNCTIONS))
        continue;

      // Here we build default functions that respect the "no default
      // functions" pragma.
      if (AggregateType* ct = toAggregateType(type->type))
      {
        buildDefaultReadWriteFunctions(ct);

        if (isRecord(ct)) {
          if (!isRecordWrappedType(ct)) {
            build_record_equality_function(ct);
            build_record_inequality_function(ct);
          }
          build_record_assignment_function(ct);
          build_record_cast_function(ct);
          build_record_copy_function(ct);
          build_record_hash_function(ct);
        }
        if (isUnion(ct))
          build_union_assignment_function(ct);
      }
      else if (EnumType* et = toEnumType(type->type)) {
        //buildDefaultReadFunction(et);
        buildStringCastFunction(et);

        build_enum_cast_function(et);
        build_enum_assignment_function(et);
        build_enum_first_function(et);
        build_enum_enumerate_function(et);
      }
      else
      {
        // The type is a simple type.

        // Other simple types are handled explicitly in the module code.
        // But to avoid putting a catch-all case there to implement assignment
        // for extern types that are simple (as far as we can tell), we build
        // definitions for those assignments here.
        if (type->hasFlag(FLAG_EXTERN)) {
          build_extern_init_function(type->type);
          build_extern_assignment_function(type->type);
        }
      }
    }
  }
}


static void buildFieldAccessorFunctions(AggregateType* at)
{
  for_fields(field, at) {
    if (!field->hasFlag(FLAG_IMPLICIT_ALIAS_FIELD)) {
      if (isVarSymbol(field)) {
        if (strcmp(field->name, "_promotionType")) {
          build_accessors(at, field);
        }
      } else if (isEnumType(field->type)) {
        build_accessors(at, field);
      }
    }
  }
}


// function_exists returns true iff
//  function's name matches name
//  function's number of formals matches numFormals
//  function's first formal's type matches formalType1 if not NULL
//  function's second formal's type matches formalType2 if not NULL
//  function's third formal's type matches formalType3 if not NULL

static bool type_match(Type* type, Symbol* sym) {
  if (type == dtAny)
    return true;
  if (sym->type == type)
    return true;
  SymExpr* se = toSymExpr(sym->defPoint->exprType);
  if (se && se->var->type == type)
    return true;
  return false;
}

typedef enum {
  FIND_EITHER = 0,
  FIND_REF,
  FIND_NOT_REF
} fuction_exists_kind_t;

static FnSymbol* function_exists(const char* name,
                                 int numFormals,
                                 Type* formalType1 = NULL,
                                 Type* formalType2 = NULL,
                                 Type* formalType3 = NULL,
                                 fuction_exists_kind_t kind=FIND_EITHER)
{
  switch(numFormals)
  {
   default:
    INT_FATAL("function_exists checks at most 3 argument types.  Add more if needed.");
    break;
   case 3:  if (!formalType3)   INT_FATAL("Missing argument formalType3");  break;
   case 2:  if (!formalType2)   INT_FATAL("Missing argument formalType2");  break;
   case 1:  if (!formalType1)   INT_FATAL("Missing argument formalType1");  break;
   case 0:  break;
  }

  forv_Vec(FnSymbol, fn, gFnSymbols)
  {
    if (strcmp(name, fn->name))
      continue;

    // numFormals must match exactly.
    if (numFormals != fn->numFormals())
        continue;

    if (formalType1)
      if (!type_match(formalType1, fn->getFormal(1)))
        continue;

    if (formalType2)
      if (!type_match(formalType2, fn->getFormal(2)))
        continue;

    if (formalType3)
      if (!type_match(formalType3, fn->getFormal(3)))
        continue;

    if (kind == FIND_REF && fn->retTag != RET_REF)
      continue;

    if (kind == FIND_NOT_REF && fn->retTag == RET_REF)
      continue;

    return fn;
  }

  // No matching function found.
  return NULL;
}


static void fixup_accessor(AggregateType* ct, Symbol *field,
                           bool fieldIsConst, bool recordLike,
                           FnSymbol* fn)
{
  std::vector<BaseAST*> asts;
  collect_asts(fn, asts);
  for_vector(BaseAST, ast, asts) {
    if (CallExpr* call = toCallExpr(ast)) {
      if (call->isNamed(field->name) && call->numActuals() == 2) {
        if (call->get(1)->typeInfo() == dtMethodToken &&
            call->get(2)->typeInfo() == ct) {
          Expr* arg2 = call->get(2);
          call->replace(new CallExpr(PRIM_GET_MEMBER,
                                     arg2->remove(),
                                     new_CStringSymbol(field->name)));
        }
      }
    }
  }
  fn->addFlag(FLAG_FIELD_ACCESSOR);
  if (fieldIsConst)
    fn->addFlag(FLAG_REF_TO_CONST);
  else if (recordLike)
    fn->addFlag(FLAG_REF_TO_CONST_WHEN_CONST_THIS);
}

// This function builds the getter or the setter, depending on the
// 'setter' argument.
static void build_accessor(AggregateType* ct, Symbol *field, bool setter) {
  const bool fieldIsConst = field->hasFlag(FLAG_CONST);
  const bool recordLike = ct->isRecord() || ct->isUnion();

  // Only build a 'ref' version for records and classes.
  // Unions need a special getter and setter.
  if (!isUnion(ct))
    if (!setter)
      return;

  FnSymbol* fn = new FnSymbol(field->name);
  fn->addFlag(FLAG_NO_IMPLICIT_COPY);
  fn->addFlag(FLAG_INLINE);

  if (ct->symbol->hasFlag(FLAG_SYNC))
    fn->addFlag(FLAG_SYNC);

  if (ct->symbol->hasFlag(FLAG_SINGLE))
    fn->addFlag(FLAG_SINGLE);

  if (ct->symbol->hasFlag(FLAG_ATOMIC_TYPE))
    fn->addFlag(FLAG_ATOMIC_TYPE);

  fn->addFlag(FLAG_FIELD_ACCESSOR);

  if (fieldIsConst)
    fn->addFlag(FLAG_REF_TO_CONST);
  else if (recordLike)
    fn->addFlag(FLAG_REF_TO_CONST_WHEN_CONST_THIS);

  fn->insertFormalAtTail(new ArgSymbol(INTENT_BLANK, "_mt", dtMethodToken));
  fn->addFlag(FLAG_METHOD);

  ArgSymbol* _this = new ArgSymbol(INTENT_BLANK, "this", ct);
  _this->addFlag(FLAG_ARG_THIS);
  fn->insertFormalAtTail(_this);
  if (field->isParameter())
    fn->retTag = RET_PARAM;
  else if (field->hasFlag(FLAG_TYPE_VARIABLE))
    fn->retTag = RET_TYPE;
  else if (field->hasFlag(FLAG_SUPER_CLASS)) {
    fn->retTag = RET_VALUE;
  } else {
    if (fieldIsConst || !setter)
      fn->retTag = RET_CONST_REF;
    else
      fn->retTag = RET_REF;
  }

  if (isUnion(ct)) {
    if (setter)  {
      // Set the union ID in the setter.
      fn->insertAtTail(
          new CallExpr(PRIM_SET_UNION_ID,
                       _this,
                       new CallExpr(PRIM_FIELD_NAME_TO_NUM,
                                    ct->symbol,
                                    new_CStringSymbol(field->name))));
    } else {
      // Check the union ID in the getter.
      fn->insertAtTail(new CondStmt(
          new CallExpr("!=",
            new CallExpr(PRIM_GET_UNION_ID, _this),
            new CallExpr(PRIM_FIELD_NAME_TO_NUM,
                         ct->symbol,
                         new_CStringSymbol(field->name))),
          new CallExpr("halt", new_CStringSymbol("illegal union access"))));
    }
  }
  if (isTypeSymbol(field) && isEnumType(field->type)) {
    fn->insertAtTail(new CallExpr(PRIM_RETURN, field));
    // better flatten enumerated types now
    ct->symbol->defPoint->insertBefore(field->defPoint->remove());
  } else if (field->hasFlag(FLAG_TYPE_VARIABLE) || field->hasFlag(FLAG_SUPER_CLASS))
    fn->insertAtTail(new CallExpr(PRIM_RETURN,
                                  new CallExpr(PRIM_GET_MEMBER_VALUE,
                                               new SymExpr(_this),
                                               new SymExpr(new_CStringSymbol(field->name)))));
  else
    fn->insertAtTail(new CallExpr(PRIM_RETURN,
                                  new CallExpr(PRIM_GET_MEMBER,
                                               new SymExpr(_this),
                                               new SymExpr(new_CStringSymbol(field->name)))));

  DefExpr* def = new DefExpr(fn);
  ct->symbol->defPoint->insertBefore(def);
  reset_ast_loc(fn, field);
  normalize(fn);
  ct->methods.add(fn);
  fn->addFlag(FLAG_METHOD);
  fn->addFlag(FLAG_METHOD_PRIMARY);
  fn->cname = astr("chpl_get_", ct->symbol->cname, "_", fn->cname);
  fn->addFlag(FLAG_NO_PARENS);
  fn->_this = _this;
}

// Getter and setter functions are provided by the compiler if not supplied by
// the user.
// These functions have the same binding strength as if they were user-defined.
// This function calls build_accessor twice, passing
// true (to build the setter) and false (to build the getter).
static void build_accessors(AggregateType* ct, Symbol *field) {
  const bool fieldIsConst = field->hasFlag(FLAG_CONST);
  const bool recordLike = ct->isRecord() || ct->isUnion();

  FnSymbol *setter = function_exists(field->name, 2,
                                     dtMethodToken, ct, NULL, FIND_REF);
  FnSymbol *getter = function_exists(field->name, 2,
                                     dtMethodToken, ct, NULL, FIND_NOT_REF);
  if (setter)
    fixup_accessor(ct, field, fieldIsConst, recordLike, setter);
  if (getter)
    fixup_accessor(ct, field, fieldIsConst, recordLike, getter);
  if (getter || setter)
    return;

  // Otherwise, build compiler-default getter and setter.
  build_accessor(ct, field, true);
  build_accessor(ct, field, false);
}

static FnSymbol* chpl_gen_main_exists() {
  FnSymbol* match = NULL;
  ModuleSymbol* matchMod = NULL;
  ModuleSymbol* module = NULL;

  if (strlen(mainModuleName) != 0) {
    forv_Vec(ModuleSymbol, mod, allModules) {
      if (!strcmp(mainModuleName, mod->name))
        module = mod;
    }
    if (!module)
      USR_FATAL("Couldn't find module %s", mainModuleName);
  }

  bool firstProblem = true;

  forv_Vec(FnSymbol, fn, gFnSymbols) {
    if (!strcmp("main", fn->name)) {
      if (fn->numFormals() == 0 ||
          (fn->numFormals() == 1 &&
           fn->getFormal(1)->typeInfo() == dtArray) ) {
        mainHasArgs = (fn->numFormals() > 0);

        CallExpr* ret = toCallExpr(fn->body->body.last());

        if (!ret || !ret->isPrimitive(PRIM_RETURN))
          INT_FATAL(fn, "function is not normalized");

        SymExpr* sym = toSymExpr(ret->get(1));

        if (!sym)
          INT_FATAL(fn, "function is not normalized");

        if( sym->var != gVoid ) {
          mainReturnsInt = true;
        } else {
          mainReturnsInt = false;
        }

        ModuleSymbol* fnMod = fn->getModule();

        if ((module == NULL && 
             fnMod->hasFlag(FLAG_MODULE_FROM_COMMAND_LINE_FILE)) ||
            fnMod == module) {
          if (!match) {
            match = fn;
            matchMod = fnMod;
          } else {
            if (firstProblem) {
              firstProblem = false;
              USR_FATAL_CONT("Ambiguous main() function%s:",
                             fnMod == matchMod ? "" : " (use --main-module to disambiguate)");
              USR_PRINT(match, "in module %s", matchMod->name);
            }
            USR_PRINT(fn, "in module %s", fnMod->name);
          } // else, this is not a candidate for the main module
        }
      } else {
        USR_FATAL_CONT("main() function with invalid signature");
        USR_PRINT(fn, "in module %s", fn->getModule()->name);
      }
    }
  }
  if (!firstProblem) {
    USR_STOP();
  }
  return match;
}


static void build_chpl_entry_points() {
  //
  // chpl_user_main is the (user) programmatic portion of the app
  //
  FnSymbol* chpl_user_main = chpl_gen_main_exists();

  if (fLibraryCompile) {
    if (chpl_user_main)
      INT_FATAL(chpl_user_main, "'main' found when compiling a library");
  }

  if (!chpl_user_main) {
    if (strlen(mainModuleName) != 0) {
      forv_Vec(ModuleSymbol, mod, userModules) {
        if (!strcmp(mod->name, mainModuleName))
          mainModule = mod;
      }
      if (!mainModule)
        USR_FATAL("unknown module specified in '--main-module=%s'", mainModuleName);
    } else {
      for_alist(expr, theProgram->block->body) {
        if (DefExpr* def = toDefExpr(expr)) {
          if (ModuleSymbol* mod = toModuleSymbol(def->sym)) {
            if (mod->hasFlag(FLAG_MODULE_FROM_COMMAND_LINE_FILE)) {
              if (mainModule) {
                USR_FATAL_CONT("a program with multiple user modules requires a main function");
                USR_PRINT("alternatively, specify a main module with --main-module");
                USR_STOP();
              }
              mainModule = mod;
            }
          }
        }
      }
    }
    SET_LINENO(mainModule);
    chpl_user_main = new FnSymbol("main");
    chpl_user_main->retType = dtVoid;
    mainModule->block->insertAtTail(new DefExpr(chpl_user_main));
    normalize(chpl_user_main);
  } else {
    if (!isModuleSymbol(chpl_user_main->defPoint->parentSymbol)) {
      USR_FATAL(chpl_user_main, "main function must be defined at module scope");
    }
    mainModule = chpl_user_main->getModule();
  }

  SET_LINENO(chpl_user_main);
  chpl_user_main->cname = "chpl_user_main";

  //
  // chpl_gen_main is the entry point for the compiler-generated code.
  // It invokes the user's code.
  //
  chpl_gen_main = new FnSymbol("chpl_gen_main");

  ArgSymbol* arg = new ArgSymbol(INTENT_BLANK, "_arg", dtMainArgument);
  chpl_gen_main->insertFormalAtTail(arg);
  chpl_gen_main->retType = dtInt[INT_SIZE_64];

  chpl_gen_main->cname = "chpl_gen_main";
  chpl_gen_main->addFlag(FLAG_EXPORT);  // chpl_gen_main is always exported.
  chpl_gen_main->addFlag(FLAG_LOCAL_ARGS);
  chpl_gen_main->addFlag(FLAG_COMPILER_GENERATED);
  mainModule->block->insertAtTail(new DefExpr(chpl_gen_main));
  VarSymbol* main_ret = newTemp("_main_ret", dtInt[INT_SIZE_64]);
  VarSymbol* endCount = newTemp("_endCount");
  chpl_gen_main->insertAtTail(new DefExpr(main_ret));
  chpl_gen_main->insertAtTail(new DefExpr(endCount));

  //
  // In --minimal-modules compilation mode, we won't have any
  // parallelism, so no need for end counts (or atomic/sync types to
  // support them).
  //
  if (fMinimalModules == false) {
    chpl_gen_main->insertAtTail(new CallExpr(PRIM_MOVE, endCount, new CallExpr("_endCountAlloc", /* forceLocalTypes= */gFalse)));
    chpl_gen_main->insertAtTail(new CallExpr(PRIM_SET_END_COUNT, endCount));
  }

  chpl_gen_main->insertAtTail(new CallExpr("chpl_rt_preUserCodeHook"));

  // We have to initialize the main module explicitly.
  // It will initialize all the modules it uses, recursively.
  chpl_gen_main->insertAtTail(new CallExpr(mainModule->initFn));

  bool main_ret_set = false;

  if (!fLibraryCompile) {
    SET_LINENO(chpl_gen_main);
    if (mainHasArgs) {
      VarSymbol* converted_args = newTemp("_main_args");

      chpl_gen_main->insertAtTail(new DefExpr(converted_args));
      chpl_gen_main->insertAtTail(new CallExpr(PRIM_MOVE,
                                               converted_args,
                                               new CallExpr("chpl_convert_args", arg)));

      if (mainReturnsInt) {
        chpl_gen_main->insertAtTail(new CallExpr(PRIM_MOVE,
                                                 main_ret,
                                                 new CallExpr("main", converted_args)));
        main_ret_set = true;
      } else {
        chpl_gen_main->insertAtTail(new CallExpr("main", converted_args));
      }
    } else {
      if (mainReturnsInt) {
        chpl_gen_main->insertAtTail(new CallExpr(PRIM_MOVE,
                                                 main_ret,
                                                 new CallExpr("main")));
        main_ret_set = true;
      } else {
        chpl_gen_main->insertAtTail(new CallExpr("main"));
      }
    }
  }

  if (!main_ret_set) {
    chpl_gen_main->insertAtTail(new CallExpr(PRIM_MOVE,
                                             main_ret,
                                             new_IntSymbol(0, INT_SIZE_64)));
  }

  chpl_gen_main->insertAtTail(new CallExpr("chpl_rt_postUserCodeHook"));

  //
  // In --minimal-modules compilation mode, we won't be waiting on an
  // endcount (see comment above)
  //
  if (fMinimalModules == false) {
    chpl_gen_main->insertAtTail(new CallExpr("_waitEndCount"));
  }

  chpl_gen_main->insertAtTail(new CallExpr(PRIM_RETURN, main_ret));

  normalize(chpl_gen_main);
}

static void build_record_equality_function(AggregateType* ct) {
  if (function_exists("==", 2, ct, ct))
    return;

  FnSymbol* fn = new FnSymbol("==");
  fn->addFlag(FLAG_COMPILER_GENERATED);
  ArgSymbol* arg1 = new ArgSymbol(INTENT_BLANK, "_arg1", ct);
  arg1->addFlag(FLAG_MARKED_GENERIC);
  ArgSymbol* arg2 = new ArgSymbol(INTENT_BLANK, "_arg2", ct);
  fn->insertFormalAtTail(arg1);
  fn->insertFormalAtTail(arg2);
  fn->retType = dtBool;
  for_fields(tmp, ct) {
    if (!tmp->hasFlag(FLAG_IMPLICIT_ALIAS_FIELD)) {
      if (strcmp(tmp->name, "_promotionType")) {
        Expr* left = new CallExpr(tmp->name, gMethodToken, arg1);
        Expr* right = new CallExpr(tmp->name, gMethodToken, arg2);
        fn->insertAtTail(new CondStmt(new CallExpr("!=", left, right), new CallExpr(PRIM_RETURN, gFalse)));
      }
    }
  }
  fn->insertAtTail(new CallExpr(PRIM_RETURN, gTrue));
  DefExpr* def = new DefExpr(fn);
  ct->symbol->defPoint->insertBefore(def);
  reset_ast_loc(def, ct->symbol);
  normalize(fn);
}


static void build_record_inequality_function(AggregateType* ct) {
  if (function_exists("!=", 2, ct, ct))
    return;

  FnSymbol* fn = new FnSymbol("!=");
  fn->addFlag(FLAG_COMPILER_GENERATED);
  ArgSymbol* arg1 = new ArgSymbol(INTENT_BLANK, "_arg1", ct);
  arg1->addFlag(FLAG_MARKED_GENERIC);
  ArgSymbol* arg2 = new ArgSymbol(INTENT_BLANK, "_arg2", ct);
  fn->insertFormalAtTail(arg1);
  fn->insertFormalAtTail(arg2);
  fn->retType = dtBool;
  for_fields(tmp, ct) {
    if (!tmp->hasFlag(FLAG_IMPLICIT_ALIAS_FIELD)) {
      if (strcmp(tmp->name, "_promotionType")) {
        Expr* left = new CallExpr(tmp->name, gMethodToken, arg1);
        Expr* right = new CallExpr(tmp->name, gMethodToken, arg2);
        fn->insertAtTail(new CondStmt(new CallExpr("!=", left, right),
                                      new CallExpr(PRIM_RETURN, gTrue)));
      }
    }
  }
  fn->insertAtTail(new CallExpr(PRIM_RETURN, gFalse));
  DefExpr* def = new DefExpr(fn);
  ct->symbol->defPoint->insertBefore(def);
  reset_ast_loc(def, ct->symbol);
  normalize(fn);
}

static void build_enum_first_function(EnumType* et) {
  if (function_exists("chpl_enum_first", 1, et))
    return;
  // Build a function that returns the first option for the enum
  // specified, also known as the default.
  FnSymbol* fn = new FnSymbol("chpl_enum_first");
  fn->addFlag(FLAG_COMPILER_GENERATED);
  // Making this compiler generated allows users to define what the
  // default is for a particular enum.  They can also redefine the
  // _defaultOf function for the enum to obtain this functionality (and
  // that is the encouraged path to take).
  fn->addFlag(FLAG_INLINE);
  ArgSymbol* arg = new ArgSymbol(INTENT_BLANK, "t", dtAny);
  arg->addFlag(FLAG_MARKED_GENERIC);
  arg->addFlag(FLAG_TYPE_VARIABLE);
  fn->insertFormalAtTail(arg);
  fn->retTag = RET_PARAM;

  fn->where = new BlockStmt(new CallExpr("==", arg, et->symbol));

  DefExpr* defExpr = toDefExpr(et->constants.head);
  if (defExpr)
    fn->insertAtTail(new CallExpr(PRIM_RETURN, defExpr->sym));
  else
    fn->insertAtTail(new CallExpr(PRIM_RETURN, dtVoid));
  // If there are one or more enumerators for this type, return the first one
  // listed.  Otherwise return void.

  DefExpr* fnDef = new DefExpr(fn);
  // needs to go in the base module because when called from _defaultOf(et),
  // they are automatically inserted
  baseModule->block->insertAtTail(fnDef);
  reset_ast_loc(fnDef, et->symbol);

  normalize(fn);
}

static void build_enum_enumerate_function(EnumType* et) {
  // Build a function that returns a tuple of the enum's values
  // Each enum type has its own chpl_enum_enumerate function.
  FnSymbol* fn = new FnSymbol("chpl_enum_enumerate");
  fn->addFlag(FLAG_COMPILER_GENERATED);
  ArgSymbol* arg = new ArgSymbol(INTENT_BLANK, "t", dtAny);
  arg->addFlag(FLAG_TYPE_VARIABLE);
  fn->insertFormalAtTail(arg);
  fn->where = new BlockStmt(new CallExpr("==", arg, et->symbol));

  et->symbol->defPoint->insertAfter(new DefExpr(fn));

  // Generate the tuple of enum values for the given enum type
  CallExpr* call = new CallExpr("_construct__tuple");
  for_enums(constant, et) {
    call->insertAtTail(constant->sym);
  }
  call->insertAtHead(new_IntSymbol(call->numActuals()));

  fn->insertAtTail(new CallExpr(PRIM_RETURN, call));

  normalize(fn);
}

static void build_enum_cast_function(EnumType* et) {
  // integral value to enumerated type cast function
  FnSymbol* fn = new FnSymbol("_cast");
  fn->addFlag(FLAG_COMPILER_GENERATED);
  ArgSymbol* arg1 = new ArgSymbol(INTENT_BLANK, "t", dtAny);
  arg1->addFlag(FLAG_TYPE_VARIABLE);
  ArgSymbol* arg2 = new ArgSymbol(INTENT_BLANK, "_arg2", dtIntegral);
  fn->insertFormalAtTail(arg1);
  fn->insertFormalAtTail(arg2);
  fn->where = new BlockStmt(new CallExpr("==", arg1, et->symbol));
  if (fNoBoundsChecks) {
    fn->addFlag(FLAG_INLINE);
    fn->insertAtTail(new CallExpr(PRIM_RETURN, new CallExpr(PRIM_CAST, et->symbol, arg2)));
  } else {
    // Generate a select statement with when clauses for each of the
    // enumeration constants, and an otherwise clause that calls halt.
    int64_t count = 0;
    BlockStmt* whenstmts = buildChapelStmt();
    for_enums(constant, et) {
      if (!get_int(constant->init, &count)) {
        count++;
      }
      CondStmt* when =
        new CondStmt(new CallExpr(PRIM_WHEN,
                                  new SymExpr(new_IntSymbol(count))),
                     new CallExpr(PRIM_RETURN,
                                  new CallExpr(PRIM_CAST,
                                               et->symbol, arg2)));
      whenstmts->insertAtTail(when);
    }
    const char * errorString = "enumerated type out of bounds";
    CondStmt* otherwise =
      new CondStmt(new CallExpr(PRIM_WHEN),
                   new BlockStmt(new CallExpr("halt",
                                 new_StringSymbol(errorString))));
    whenstmts->insertAtTail(otherwise);
    fn->insertAtTail(buildSelectStmt(new SymExpr(arg2), whenstmts));
  }
  DefExpr* def = new DefExpr(fn);
  //
  // these cast functions need to go in the base module because they
  // are automatically inserted to handle implicit coercions
  //
  baseModule->block->insertAtTail(def);
  reset_ast_loc(def, et->symbol);
  normalize(fn);

  // string to enumerated type cast function
  fn = new FnSymbol("_cast");
  fn->addFlag(FLAG_COMPILER_GENERATED);
  arg1 = new ArgSymbol(INTENT_BLANK, "t", dtAny);
  arg1->addFlag(FLAG_TYPE_VARIABLE);
  arg2 = new ArgSymbol(INTENT_BLANK, "_arg2", dtString);
  fn->insertFormalAtTail(arg1);
  fn->insertFormalAtTail(arg2);

  CondStmt* cond = NULL;
  for_enums(constant, et) {
    cond = new CondStmt(
             new CallExpr("==", arg2, new_StringSymbol(constant->sym->name)),
             new CallExpr(PRIM_RETURN, constant->sym),
             cond);
    cond = new CondStmt(
             new CallExpr("==", arg2,
                          new_StringSymbol(
                            astr(et->symbol->name, ".", constant->sym->name))),
             new CallExpr(PRIM_RETURN, constant->sym),
             cond);
  }

  fn->insertAtTail(cond);

  fn->insertAtTail(new CallExpr("halt",
                                new_StringSymbol("illegal conversion of string \\\""),
                                arg2,
                                new_StringSymbol("\\\" to "),
                                new_StringSymbol(et->symbol->name)));

  fn->insertAtTail(new CallExpr(PRIM_RETURN,
                                toDefExpr(et->constants.first())->sym));

  fn->where = new BlockStmt(new CallExpr("==", arg1, et->symbol));
  def = new DefExpr(fn);
  //
  // these cast functions need to go in the base module because they
  // are automatically inserted to handle implicit coercions
  //
  baseModule->block->insertAtTail(def);
  reset_ast_loc(def, et->symbol);
  normalize(fn);
}


static void build_enum_assignment_function(EnumType* et) {
  if (function_exists("=", 2, et, et))
    return;

  FnSymbol* fn = new FnSymbol("=");
  fn->addFlag(FLAG_ASSIGNOP);
  fn->addFlag(FLAG_COMPILER_GENERATED);
  fn->addFlag(FLAG_INLINE);
  ArgSymbol* arg1 = new ArgSymbol(INTENT_REF, "_arg1", et);
  ArgSymbol* arg2 = new ArgSymbol(INTENT_BLANK, "_arg2", et);
  fn->insertFormalAtTail(arg1);
  fn->insertFormalAtTail(arg2);
  fn->insertAtTail(new CallExpr(PRIM_ASSIGN, arg1, arg2));
  DefExpr* def = new DefExpr(fn);
  et->symbol->defPoint->insertBefore(def);
  reset_ast_loc(def, et->symbol);
  normalize(fn);
}


static void build_record_assignment_function(AggregateType* ct) {
  if (function_exists("=", 2, ct, ct))
    return;

  FnSymbol* fn = new FnSymbol("=");
  fn->addFlag(FLAG_ASSIGNOP);
  fn->addFlag(FLAG_COMPILER_GENERATED);

  ArgSymbol* arg1 = new ArgSymbol(INTENT_REF, "_arg1", ct);
  arg1->addFlag(FLAG_MARKED_GENERIC); // TODO: Check if we really want this.

  bool externRecord = ct->symbol->hasFlag(FLAG_EXTERN);
  // If the LHS is extern, the RHS must be of matching type; otherwise
  // Chapel permits matches that have the same names
  ArgSymbol* arg2 = new ArgSymbol(INTENT_BLANK, "_arg2",
                                  (externRecord ? ct : dtAny));
  fn->insertFormalAtTail(arg1);
  fn->insertFormalAtTail(arg2);

  // This is required because coercion from a subtype to a base record type
  // inserts a cast.  Such a cast is implemented using assignment.  The
  // resolution errors out because the return type of the cast is recursively
  // defined.  (See classes/marybeth/test_dispatch_record.chpl)
  if (! externRecord)
    fn->where = new BlockStmt(new CallExpr(PRIM_IS_SUBTYPE, ct->symbol, arg2));

  fn->retType = dtVoid;

  if (externRecord) {
    fn->insertAtTail(new CallExpr(PRIM_ASSIGN, arg1, arg2));
    ct->symbol->addFlag(FLAG_POD);
    fn->addFlag(FLAG_INLINE);
  } else {
    for_fields(tmp, ct) {
      if (!tmp->hasFlag(FLAG_IMPLICIT_ALIAS_FIELD)) {
        if (!tmp->hasFlag(FLAG_TYPE_VARIABLE) &&
            !tmp->isParameter()               &&
            strcmp(tmp->name, "_promotionType"))
          fn->insertAtTail(new CallExpr("=",
                                        new CallExpr(".", arg1, new_CStringSymbol(tmp->name)),
                                        new CallExpr(".", arg2, new_CStringSymbol(tmp->name))));
      }
    }
  }

  DefExpr* def = new DefExpr(fn);
  ct->symbol->defPoint->insertBefore(def);
  reset_ast_loc(def, ct->symbol);
  normalize(fn);
}

static void build_extern_init_function(Type* type)
{
  if (function_exists("_defaultOf", 1, type))
    return;

  // In the world where initialization lived entirely within the compiler,
  // externs did not initialize themselves except in very rare cases (see
  // c_strings).  This is the same as using noinit with the type instance.
  // In the world of module default initializers, externs can be specified to
  // use noinit here, while specific externs that differ can define their own
  // default initialization in the modules.  If we had a way to determine that
  // a type was extern from just the provided type, this code could also move
  // to the modules.
  FnSymbol* fn = new FnSymbol("_defaultOf");
  fn->addFlag(FLAG_INLINE);
  fn->addFlag(FLAG_COMPILER_GENERATED);
  ArgSymbol* arg = new ArgSymbol(INTENT_BLANK, "t", type);
  arg->addFlag(FLAG_TYPE_VARIABLE);
  fn->insertFormalAtTail(arg);
  VarSymbol* ret = new VarSymbol("ret", type);
  fn->insertAtTail(new CallExpr(PRIM_MOVE, ret,
                                new CallExpr(PRIM_NO_INIT, new SymExpr(type->symbol))));
  fn->insertAtTail(new CallExpr(PRIM_RETURN, ret));
  fn->insertAtHead(new DefExpr(ret));
  fn->retType = type;
  DefExpr* def = new DefExpr(fn);
  type->symbol->defPoint->insertBefore(def);
  reset_ast_loc(def, type->symbol);
  normalize(fn);
}

static void build_extern_assignment_function(Type* type)
{
  if (function_exists("=", 2, type, type))
    return;

  FnSymbol* fn = new FnSymbol("=");
  fn->addFlag(FLAG_ASSIGNOP);
  type->symbol->addFlag(FLAG_POD);
  fn->addFlag(FLAG_COMPILER_GENERATED);
  fn->addFlag(FLAG_INLINE);

  ArgSymbol* arg1 = new ArgSymbol(INTENT_REF, "_arg1", type);
  ArgSymbol* arg2 = new ArgSymbol(INTENT_BLANK, "_arg2", type);
  fn->insertFormalAtTail(arg1);
  fn->insertFormalAtTail(arg2);
  fn->insertAtTail(new CallExpr(PRIM_ASSIGN, arg1, arg2));
  DefExpr* def = new DefExpr(fn);
  if (type->symbol->defPoint->parentSymbol == rootModule)
    baseModule->block->body.insertAtTail(def);
  else
    type->symbol->defPoint->insertBefore(def);
  reset_ast_loc(def, type->symbol);

  normalize(fn);
}


// _cast is automatically called to perform coercions.
// If the coercion is permissible, then the operand can be statically cast to
// the target type.
static void build_record_cast_function(AggregateType* ct) {
  FnSymbol* fn = new FnSymbol("_cast");
  fn->addFlag(FLAG_COMPILER_GENERATED);
  fn->addFlag(FLAG_INLINE);
  ArgSymbol* t = new ArgSymbol(INTENT_BLANK, "t", dtAny);
  t->addFlag(FLAG_TYPE_VARIABLE);
  ArgSymbol* arg = new ArgSymbol(INTENT_BLANK, "arg", dtAny);
  fn->insertFormalAtTail(t);
  fn->insertFormalAtTail(arg);
  fn->where = new BlockStmt(new CallExpr(PRIM_IS_SUBTYPE, ct->symbol, t));
  VarSymbol* ret = newTemp();
  fn->insertAtTail(new DefExpr(ret));
  fn->insertAtTail(new CallExpr(PRIM_MOVE, ret, new CallExpr(PRIM_INIT, t)));
  fn->insertAtTail(new CallExpr("=", ret, arg));
  fn->insertAtTail(new CallExpr(PRIM_RETURN, ret));
  DefExpr* def = new DefExpr(fn);
  ct->symbol->defPoint->insertBefore(def);
  reset_ast_loc(def, ct->symbol);
  normalize(fn);
}

// TODO: we should know what field is active after assigning unions
static void build_union_assignment_function(AggregateType* ct) {
  if (function_exists("=", 2, ct, ct))
    return;

  FnSymbol* fn = new FnSymbol("=");
  fn->addFlag(FLAG_ASSIGNOP);
  fn->addFlag(FLAG_COMPILER_GENERATED);
  ArgSymbol* arg1 = new ArgSymbol(INTENT_REF, "_arg1", ct);
  ArgSymbol* arg2 = new ArgSymbol(INTENT_BLANK, "_arg2", ct);
  fn->insertFormalAtTail(arg1);
  fn->insertFormalAtTail(arg2);
  fn->retType = dtUnknown;
  fn->insertAtTail(new CallExpr(PRIM_SET_UNION_ID, arg1, new_IntSymbol(0)));
  for_fields(tmp, ct) {
    if (!tmp->hasFlag(FLAG_IMPLICIT_ALIAS_FIELD)) {
      if (!tmp->hasFlag(FLAG_TYPE_VARIABLE)) {
        fn->insertAtTail(new CondStmt(
            new CallExpr("==",
              new CallExpr(PRIM_GET_UNION_ID, arg2),
              new CallExpr(PRIM_FIELD_NAME_TO_NUM,
                           ct->symbol,
                           new_CStringSymbol(tmp->name))),
            new CallExpr("=",
              new CallExpr(".", arg1, new_StringSymbol(tmp->name)),
              new CallExpr(".", arg2, new_StringSymbol(tmp->name)))));
      }
    }
  }
  DefExpr* def = new DefExpr(fn);
  ct->symbol->defPoint->insertBefore(def);
  reset_ast_loc(def, ct->symbol);
  normalize(fn);
}


static void build_record_copy_function(AggregateType* ct) {
  if (function_exists("chpl__initCopy", 1, ct))
    return;

  FnSymbol* fn = new FnSymbol("chpl__initCopy");
  fn->addFlag(FLAG_INIT_COPY_FN);
  fn->addFlag(FLAG_COMPILER_GENERATED);
  ArgSymbol* arg = new ArgSymbol(INTENT_BLANK, "x", ct);
  arg->addFlag(FLAG_MARKED_GENERIC);
  fn->insertFormalAtTail(arg);
  CallExpr* call = new CallExpr(ct->defaultInitializer);
  for_fields(tmp, ct)
  {
    // Weed out implicit alias and promotion type fields.
    if (tmp->hasFlag(FLAG_IMPLICIT_ALIAS_FIELD))
      continue;
    // TODO: This needs to be done uniformly, using an ignore_field flag or...
    if (!strcmp("_promotionType", tmp->name))
      continue;

    CallExpr* init = new CallExpr(".", arg, new_CStringSymbol(tmp->name));
    call->insertAtTail(new NamedExpr(tmp->name, init));

    // Special handling for nested record types:
    // We need to convert the constructor call into a method call.
    if (!strcmp(tmp->name, "outer"))
      call->insertAtHead(gMethodToken);
  }
  if (ct->symbol->hasFlag(FLAG_EXTERN)) {
    int actualsNeeded = ct->defaultInitializer->formals.length;
    int actualsGiven = call->argList.length;
    // This code assumes that in the case where an extern has not been fully
    // specified in Chapel code, its default constructor will require one more
    // actual than the fields specified: another object of that extern type
    // whose contents can be used.
    if (actualsNeeded == actualsGiven + 1) {
      call->insertAtTail(arg);
    } else if (actualsNeeded != actualsGiven) {
      // The user didn't partially specify the type in Chapel code, but the
      // number of actuals provided didn't match the expected number.  This is
      // an internal error.
      INT_FATAL(arg, "Extern type's constructor call didn't create expected # of actuals");
    }
  }
  fn->insertAtTail(new CallExpr(PRIM_RETURN, call));
  DefExpr* def = new DefExpr(fn);
  ct->symbol->defPoint->insertBefore(def);
  reset_ast_loc(def, ct->symbol);
  normalize(fn);
}


static void build_record_hash_function(AggregateType *ct) {
  if (function_exists("chpl__defaultHash", 1, ct))
    return;

  FnSymbol *fn = new FnSymbol("chpl__defaultHash");
  fn->addFlag(FLAG_COMPILER_GENERATED);
  ArgSymbol *arg = new ArgSymbol(INTENT_BLANK, "r", ct);
  arg->addFlag(FLAG_MARKED_GENERIC);
  fn->insertFormalAtTail(arg);

  if (ct->fields.length == 0) {
    fn->insertAtTail(new CallExpr(PRIM_RETURN, new_IntSymbol(0)));
    fn->addFlag(FLAG_INLINE);
  } else {
    CallExpr *call = NULL;
    bool first = true;
    for_fields(field, ct) {
      if (!field->hasFlag(FLAG_IMPLICIT_ALIAS_FIELD)) {
        CallExpr *field_access = new CallExpr(field->name, gMethodToken, arg);
        if (first) {
          call = new CallExpr("chpl__defaultHash", field_access);
          first = false;
        } else {
          call = new CallExpr("^",
                              new CallExpr("chpl__defaultHash",
                                           field_access),
                              new CallExpr("<<",
                                           call,
                                           new_IntSymbol(17)));
        }
      }
    }
    fn->insertAtTail(new CallExpr(PRIM_RETURN, call));
  }
  DefExpr *def = new DefExpr(fn);
  ct->symbol->defPoint->insertBefore(def);
  reset_ast_loc(def, ct->symbol);
  normalize(fn);
}

static void build_record_init_function(AggregateType* ct) {
  if (function_exists("_defaultOf", 1, ct))
    return;

  // These cases are already handled by the module catch-all definition of
  // _defaultOf, so skip.  Most of those should be weeded out at the call
  // site by the check that if the type in question is a class, it does not
  // inherit from object (and so does not have a default value of gNil), but
  // not all will be.
  if (ct->symbol->hasFlag(FLAG_ITERATOR_CLASS) ||
      (ct->defaultValue && ct->defaultValue == gNil))
    return;

  FnSymbol* fn = new FnSymbol("_defaultOf");

  fn->addFlag(FLAG_COMPILER_GENERATED);
  fn->addFlag(FLAG_INLINE);

  ArgSymbol* arg = new ArgSymbol(INTENT_BLANK, "t", ct);
  arg->addFlag(FLAG_MARKED_GENERIC);
  arg->addFlag(FLAG_TYPE_VARIABLE);

  fn->insertFormalAtTail(arg);

  if (ct->symbol->hasFlag(FLAG_ITERATOR_RECORD)) {
    // Preserving previous functionality
    fn->insertAtTail(new CallExpr(PRIM_RETURN, arg));
  } else if (ct->symbol->hasFlag(FLAG_TUPLE)) {
    // Tuples are handled by generics.cpp because some of their information is
    // not known at this point.  The function needs to exist prior to generic
    // resolution, which is why its outer shell is created here, but its body
    // is not capable of being created until then.
  } else {
    // To default initialize, call the type specified default constructor (by
    // name), passing in all generic arguments.
    CallExpr* call = new CallExpr(ct->defaultInitializer->name);
    // Need to insert all required arguments into this call
    for_formals(formal, ct->defaultInitializer) {
      if (formal->hasFlag(FLAG_IS_MEME))
        // Do not pass in the meme argument.
        // Resolution creates a wrapper and arranges for "this" to be passed in
        // in this position
        continue;

      // If the initializer function is a method, we must call it as a
      // method.  So just pass the method token along.  (No need to create a
      // formal temp, etc.)
      if (formal->type == dtMethodToken)
      {
        call->insertAtTail(gMethodToken);
        continue;
      }

      VarSymbol* tmp = newTemp(formal->name);

      if (formal->isParameter()) {
        // Param and type fields are specific to the generic instantiation
        // of the type, so they cannot be known when we create this method.
        // However, they are knowable when the method is resolved.  This
        // utilizes the field query primitives to get the correct values
        // when those substitutions are known
        tmp->addFlag(FLAG_PARAM);

        fn->insertAtTail(new CallExpr(PRIM_MOVE,
                                      tmp,
                                      new CallExpr(PRIM_QUERY_PARAM_FIELD,
                                                   arg,
                                                   new_CStringSymbol(formal->name))));
        fn->insertAtHead(new DefExpr(tmp));
        call->insertAtTail(new NamedExpr(formal->name, new SymExpr(tmp)));

      } else if (formal->hasFlag(FLAG_TYPE_VARIABLE)) {
        tmp->addFlag(FLAG_TYPE_VARIABLE);
        fn->insertAtTail(new CallExpr(PRIM_MOVE,
                                      tmp,
                                      new CallExpr(PRIM_QUERY_TYPE_FIELD,
                                                   arg,
                                                   new_CStringSymbol(formal->name))));
        fn->insertAtHead(new DefExpr(tmp));
        call->insertAtTail(new NamedExpr(formal->name, new SymExpr(tmp)));

      } else {
        if (!formal->defaultExpr) {
          // There are no substitutions for var fields, so if a defaultExpr
          // has been provided, we don't need to worry about copying values
          // over.
          if (formal->type && formal->type != dtAny &&
              strcmp(formal->name, "outer") != 0) {
            // We know the type already, make use of it
            fn->insertAtTail(new CallExpr(PRIM_MOVE, tmp, new CallExpr(PRIM_INIT, formal->type->symbol)));
          } else {
            // Type wasn't currently able to be determined, so we'll have to
            // get it the long way.
            VarSymbol* typeTemp = newTemp ("type_tmp");
            VarSymbol* callTemp = newTemp ("call_tmp");

            typeTemp->addFlag(FLAG_TYPE_VARIABLE);

            fn->insertAtTail(new CallExpr(PRIM_MOVE,
                                          callTemp,
                                          new CallExpr(PRIM_GET_MEMBER_VALUE,
                                                       arg,
                                                       new_CStringSymbol(formal->name))));

            fn->insertAtTail(new CallExpr(PRIM_MOVE,
                                          typeTemp,
                                          new CallExpr(PRIM_TYPEOF, callTemp)));

            fn->insertAtTail(new CallExpr(PRIM_MOVE, tmp, new CallExpr(PRIM_INIT, typeTemp)));
            fn->insertAtHead(new DefExpr(callTemp));
            fn->insertAtHead(new DefExpr(typeTemp));
          }

          fn->insertAtHead(new DefExpr(tmp));

          call->insertAtTail(new NamedExpr(formal->name, new SymExpr(tmp)));
        }
      }
    }
    fn->insertAtTail(new CallExpr(PRIM_RETURN, call));
  }

  DefExpr* def = new DefExpr(fn);

  ct->symbol->defPoint->insertBefore(def);
  reset_ast_loc(def, ct->symbol);

  normalize(fn);
}

static void buildDefaultReadWriteFunctions(AggregateType* ct) {
  bool hasReadWriteThis = false;
  bool hasReadThis = false;
  bool hasWriteThis = false;
  bool makeReadThisAndWriteThis = true;

  //
  // We have no QIO when compiling with --minimal-modules, so no need
  // to build default R/W functions.
  //
  if (fMinimalModules == true) {
    return;
  }

  // If we have a readWriteThis, we'll call it from readThis/writeThis.
  if (function_exists("readWriteThis", 3, dtMethodToken, ct, dtAny)) {
    hasReadWriteThis = true;
  }
  // We'll make a writeThis and a readThis if neither exist.
  // If only one exists, we leave just one (as some types
  // can be written but not read, for example).
  if (function_exists("writeThis", 3, dtMethodToken, ct, dtAny)) {
    hasWriteThis = true;
    makeReadThisAndWriteThis = false;
  }
  if (function_exists("readThis", 3, dtMethodToken, ct, dtAny)) {
    hasReadThis = true;
    makeReadThisAndWriteThis = false;
  }

  // Make writeThis if we have neither writeThis nor readThis.
  if ( makeReadThisAndWriteThis && ! hasWriteThis ) {
    FnSymbol* fn = new FnSymbol("writeThis");
    fn->addFlag(FLAG_COMPILER_GENERATED);
    fn->addFlag(FLAG_INLINE);
    fn->cname = astr("_auto_", ct->symbol->name, "_write");
    fn->_this = new ArgSymbol(INTENT_BLANK, "this", ct);
    fn->_this->addFlag(FLAG_ARG_THIS);
    ArgSymbol* fileArg = new ArgSymbol(INTENT_BLANK, "f", dtAny);
    fileArg->addFlag(FLAG_MARKED_GENERIC);
    fn->insertFormalAtTail(new ArgSymbol(INTENT_BLANK, "_mt", dtMethodToken));
    fn->addFlag(FLAG_METHOD);
    fn->insertFormalAtTail(fn->_this);
    fn->insertFormalAtTail(fileArg);
    fn->retType = dtVoid;

    if( hasReadWriteThis ) {
      fn->insertAtTail(new CallExpr(buildDotExpr(fn->_this, "readWriteThis"), fileArg));
    } else {
      fn->insertAtTail(new CallExpr("writeThisDefaultImpl", fileArg, fn->_this));
    }

    DefExpr* def = new DefExpr(fn);
    ct->symbol->defPoint->insertBefore(def);
    fn->addFlag(FLAG_METHOD);
    fn->addFlag(FLAG_METHOD_PRIMARY);
    reset_ast_loc(def, ct->symbol);
    normalize(fn);
    ct->methods.add(fn);
  }
  if ( makeReadThisAndWriteThis && ! hasReadThis ) {
    FnSymbol* fn = new FnSymbol("readThis");
    fn->addFlag(FLAG_COMPILER_GENERATED);
    fn->addFlag(FLAG_INLINE);
    fn->cname = astr("_auto_", ct->symbol->name, "_read");
    fn->_this = new ArgSymbol(INTENT_BLANK, "this", ct);
    fn->_this->addFlag(FLAG_ARG_THIS);
    ArgSymbol* fileArg = new ArgSymbol(INTENT_BLANK, "f", dtAny);
    fileArg->addFlag(FLAG_MARKED_GENERIC);
    fn->insertFormalAtTail(new ArgSymbol(INTENT_BLANK, "_mt", dtMethodToken));
    fn->addFlag(FLAG_METHOD);
    fn->insertFormalAtTail(fn->_this);
    fn->insertFormalAtTail(fileArg);
    fn->retType = dtVoid;

    if( hasReadWriteThis ) {
      fn->insertAtTail(new CallExpr(buildDotExpr(fn->_this, "readWriteThis"), fileArg));
    } else {
      fn->insertAtTail(new CallExpr("readThisDefaultImpl", fileArg, fn->_this));
    }

    DefExpr* def = new DefExpr(fn);
    ct->symbol->defPoint->insertBefore(def);
    // ? ct->methods.add(fn) ? in old code
    fn->addFlag(FLAG_METHOD);
    fn->addFlag(FLAG_METHOD_PRIMARY);
    reset_ast_loc(def, ct->symbol);
    normalize(fn);
    ct->methods.add(fn);
  }
}


static void buildStringCastFunction(EnumType* et) {
  if (function_exists("_cast", 2, dtString, et))
    return;

  FnSymbol* fn = new FnSymbol("_cast");
  fn->addFlag(FLAG_COMPILER_GENERATED);
  ArgSymbol* t = new ArgSymbol(INTENT_BLANK, "t", dtAny);
  t->addFlag(FLAG_TYPE_VARIABLE);
  fn->insertFormalAtTail(t);
  ArgSymbol* arg = new ArgSymbol(INTENT_BLANK, "this", et);
  arg->addFlag(FLAG_ARG_THIS);
  fn->insertFormalAtTail(arg);
  fn->where = new BlockStmt(new CallExpr("==", t, dtString->symbol));

  for_enums(constant, et) {
    fn->insertAtTail(
      new CondStmt(
        new CallExpr("==", arg, constant->sym),
        new CallExpr(PRIM_RETURN, new_StringSymbol(constant->sym->name))));
  }
  fn->insertAtTail(new CallExpr(PRIM_RETURN, new_StringSymbol("")));

  DefExpr* def = new DefExpr(fn);
  //
  // these cast functions need to go in the base module because they
  // are automatically inserted to handle implicit coercions
  //
  baseModule->block->insertAtTail(def);
  reset_ast_loc(def, et->symbol);
  normalize(fn);
}


void buildDefaultDestructor(AggregateType* ct) {
  if (function_exists("~chpl_destroy", 2, dtMethodToken, ct))
    return;

  SET_LINENO(ct->symbol);

  FnSymbol* fn = new FnSymbol("~chpl_destroy");
  fn->addFlag(FLAG_COMPILER_GENERATED);
  fn->addFlag(FLAG_DESTRUCTOR);
  fn->addFlag(FLAG_INLINE);
  fn->cname = astr("chpl__auto_destroy_", ct->symbol->name);
  fn->insertFormalAtTail(new ArgSymbol(INTENT_BLANK, "_mt", dtMethodToken));
  fn->addFlag(FLAG_METHOD);
  fn->_this = new ArgSymbol(INTENT_BLANK, "this", ct);
  fn->_this->addFlag(FLAG_ARG_THIS);
  fn->insertFormalAtTail(fn->_this);
  fn->retType = dtVoid;
  fn->insertAtTail(new CallExpr(PRIM_RETURN, gVoid));
  ct->symbol->defPoint->insertBefore(new DefExpr(fn));
  fn->addFlag(FLAG_METHOD_PRIMARY);
  ct->methods.add(fn);
}
