/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Intyear    intyear         Year interpolation
*/

#include <cdi.h>

#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "listarray.h"
#include "util_files.h"


void *
Intyear(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss1, nmiss2, nmiss3;
  char filesuffix[32];
  char filename[8192];

  cdoInitialize(process);

  operatorInputArg("years");

  ListArray<int> listArrayInt;
  const auto nyears = listArrayInt.argvToInt(cdoGetOperArgv());
  auto iyears = listArrayInt.data();

  std::vector<CdoStreamID> streamIDs(nyears);

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);
  const auto vlistID3 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_ALL);

  auto gridsize = vlistGridsizeMax(vlistID1);
  Varray<double> array1(gridsize);
  Varray<double> array2(gridsize);
  Varray<double> array3(gridsize);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = vlistInqTaxis(vlistID2);
  const auto taxisID3 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID3)) taxisDeleteBounds(taxisID3);
  vlistDefTaxis(vlistID3, taxisID3);

  strcpy(filename, cdoGetStreamName(2));
  const int nchars = strlen(filename);

  const char *refname = cdoGetObase();
  filesuffix[0] = 0;
  cdoGenFileSuffix(filesuffix, sizeof(filesuffix), cdoInqFiletype(streamID1), vlistID1, refname);

  for (int iy = 0; iy < nyears; iy++)
    {
      sprintf(filename + nchars, "%04d", iyears[iy]);
      if (filesuffix[0]) sprintf(filename + nchars + 4, "%s", filesuffix);

      streamIDs[iy] = cdoOpenWrite(filename);
      cdoDefVlist(streamIDs[iy], vlistID3);
    }

  int tsID = 0;
  while (true)
    {
      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;
      nrecs = cdoStreamInqTimestep(streamID2, tsID);
      if (nrecs == 0) cdoAbort("Too few timesteps in second inputfile!");

      const auto vtime = taxisInqVtime(taxisID1);
      const auto vdate1 = taxisInqVdate(taxisID1);
      const auto year1 = vdate1 / 10000;
      const auto vdate2 = taxisInqVdate(taxisID2);
      const auto year2 = vdate2 / 10000;

      for (int iy = 0; iy < nyears; iy++)
        {
          if (iyears[iy] < year1 || iyears[iy] > year2)
            cdoAbort("Year %d out of bounds (first year %d; last year %d)!", iyears[iy], year1, year2);
          const auto vdate3 = vdate1 - year1 * 10000 + iyears[iy] * 10000;
          taxisDefVdate(taxisID3, vdate3);
          taxisDefVtime(taxisID3, vtime);
          cdoDefTimestep(streamIDs[iy], tsID);
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoInqRecord(streamID2, &varID, &levelID);

          cdoReadRecord(streamID1, array1.data(), &nmiss1);
          cdoReadRecord(streamID2, array2.data(), &nmiss2);

          gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));

          for (int iy = 0; iy < nyears; iy++)
            {
              const double fac1 = ((double) year2 - iyears[iy]) / (year2 - year1);
              const double fac2 = ((double) iyears[iy] - year1) / (year2 - year1);

              nmiss3 = 0;

              if (nmiss1 || nmiss2)
                {
                  const auto missval1 = vlistInqVarMissval(vlistID1, varID);
                  const auto missval2 = vlistInqVarMissval(vlistID2, varID);

                  for (size_t i = 0; i < gridsize; i++)
                    {
                      if (!DBL_IS_EQUAL(array1[i], missval1) && !DBL_IS_EQUAL(array2[i], missval2))
                        array3[i] = array1[i] * fac1 + array2[i] * fac2;
                      /* 2010-04-19 Uwe Schulzweida: removed
                      else if ( DBL_IS_EQUAL(array1[i], missval1) &&
                                !DBL_IS_EQUAL(array2[i], missval2) && fac2 >= 0.5 )
                        array3[i] = array2[i];
                      else if ( DBL_IS_EQUAL(array2[i], missval2) &&
                                !DBL_IS_EQUAL(array1[i], missval1) && fac1 >= 0.5 )
                        array3[i] = array1[i];
                      */
                      else
                        {
                          array3[i] = missval1;
                          nmiss3++;
                        }
                    }
                }
              else
                {
                  for (size_t i = 0; i < gridsize; i++) array3[i] = array1[i] * fac1 + array2[i] * fac2;
                }

              cdoDefRecord(streamIDs[iy], varID, levelID);
              cdoWriteRecord(streamIDs[iy], array3.data(), nmiss3);
            }
        }

      tsID++;
    }

  for (int iy = 0; iy < nyears; iy++) cdoStreamClose(streamIDs[iy]);

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return 0;
}
