/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      FC         fc2sp           Fourier to spectral
      FC         sp2fc           Spectral to fourier
      FC         fc2gp           Fourier to gridpoint
      FC         gp2fc           Gridpoint to fourier
*/

#include <cdi.h>

#include "cdo_vlist.h"
#include "process_int.h"
#include <mpim_grid.h>
#include "specspace.h"
#include "griddes.h"


void *
FC(void *process)
{
  int nrecs;
  int varID, levelID;
  int gridID1 = -1, gridID2 = -1;
  size_t nmiss;
  size_t nlon = 0, nlat = 0;
  int ntr = 0;
  int nsp = 0, nfc = 0;

  cdoInitialize(process);

  const auto lcopy = unchangedRecord();

  const auto FC2SP = cdoOperatorAdd("fc2sp", 0, 0, nullptr);
  const auto SP2FC = cdoOperatorAdd("sp2fc", 0, 0, nullptr);
  const auto FC2GP = cdoOperatorAdd("fc2gp", 0, 0, nullptr);
  const auto GP2FC = cdoOperatorAdd("gp2fc", 0, 0, nullptr);

  const auto operatorID = cdoOperatorID();

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  auto gridIDsp = vlistGetFirstSpectralGrid(vlistID1);
  auto gridIDgp = vlistGetFirstGaussianGrid(vlistID1);
  auto gridIDfc = vlistGetFirstFourierGrid(vlistID1);

  // define output grid
  if (operatorID == FC2SP)
    {
      if (gridIDfc == -1) cdoWarning("No fourier data found!");

      gridID1 = gridIDfc;

      if (gridID1 != -1)
        {
          nfc = gridInqSize(gridID1);
          ntr = gridInqTrunc(gridID1);
          nlat = nfc_to_nlat(nfc, ntr);

          if (gridIDsp != -1)
            if (ntr != gridInqTrunc(gridIDsp)) gridIDsp = -1;

          if (gridIDsp == -1)
            {
              nsp = (ntr + 1) * (ntr + 2);
              gridIDsp = gridCreate(GRID_SPECTRAL, nsp);
              gridDefTrunc(gridIDsp, ntr);
              gridDefComplexPacking(gridIDsp, 1);
            }

          gridID2 = gridIDsp;
          nlon = 2 * nlat;
          ntr = gridInqTrunc(gridID2);
        }
    }
  else if (operatorID == SP2FC)
    {
      if (gridIDsp == -1) cdoWarning("No spectral data found!");

      gridID1 = gridIDsp;

      if (gridID1 != -1)
        {
          ntr = gridInqTrunc(gridID1);
          nlat = ntr_to_nlat(ntr);

          if (gridIDfc != -1)
            {
              if (ntr != gridInqTrunc(gridIDfc)) gridIDfc = -1;
            }

          if (gridIDfc == -1)
            {
              nfc = 2 * nlat * (ntr + 1);
              gridIDfc = gridCreate(GRID_FOURIER, nfc);
              gridDefTrunc(gridIDfc, ntr);
            }

          gridID2 = gridIDfc;
          nlon = 2 * nlat;
        }
    }
  else if (operatorID == GP2FC)
    {
      if (gridIDgp == -1) cdoWarning("No Gaussian grid data found!");

      gridID1 = gridIDgp;

      if (gridID1 != -1)
        {
          nlon = gridInqXsize(gridID1);
          nlat = gridInqYsize(gridID1);
          ntr = nlat_to_ntr(nlat);

          if (gridIDfc != -1)
            if (ntr != gridInqTrunc(gridIDfc)) gridIDfc = -1;

          if (gridIDfc == -1)
            {
              nfc = 2 * nlat * (ntr + 1);
              gridIDfc = gridCreate(GRID_FOURIER, nfc);
              gridDefTrunc(gridIDfc, ntr);
            }

          gridID2 = gridIDfc;
        }
    }
  else if (operatorID == FC2GP)
    {
      if (gridIDfc == -1) cdoWarning("No fourier data found!");

      gridID1 = gridIDfc;

      if (gridID1 != -1)
        {
          nfc = gridInqSize(gridID1);
          ntr = gridInqTrunc(gridID1);
          nlat = nfc_to_nlat(nfc, ntr);

          if (gridIDgp != -1)
            {
              if (nlat != gridInqYsize(gridIDgp)) gridIDgp = -1;
            }

          if (gridIDgp == -1)
            {
              char gridname[20];
              snprintf(gridname, sizeof(gridname), "t%dgrid", ntr);

              gridIDgp = gridFromName(gridname);
            }

          gridID2 = gridIDgp;
          nlon = gridInqXsize(gridID2);
          nlat = gridInqYsize(gridID2);
        }
    }

  FC_Transformation fcTrans;
  SP_Transformation spTrans;
  if (nlon > 0)
    {
      if (operatorID == GP2FC || operatorID == FC2GP)
        fcTrans.init(nlon, nlat, ntr);
      else
        spTrans.init(nlon, nlat, ntr, 0);
    }

  // printf("nfc %d, ntr %d, nlat %zu, nlon %zu\n", nfc, ntr, nlat, nlon);

  const auto nvars = vlistNvars(vlistID2);
  std::vector<bool> vars(nvars);
  for (varID = 0; varID < nvars; varID++) vars[varID] = gridID1 == vlistInqVarGrid(vlistID1, varID);

  if (gridID1 != -1) vlistChangeGrid(vlistID2, gridID1, gridID2);

  const auto streamID2 = cdoOpenWrite(1);

  cdoDefVlist(streamID2, vlistID2);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);
  Varray<double> array1(gridsizemax), array2;

  if (gridID2 != -1)
    {
      const auto gridsize = gridInqSize(gridID2);
      array2.resize(gridsize);
    }

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (vars[varID])
            {
              cdoReadRecord(streamID1, array1.data(), &nmiss);
              if (nmiss) cdoAbort("Missing values unsupported for spectral/fourier data!");

              gridID1 = vlistInqVarGrid(vlistID1, varID);
              if (operatorID == FC2SP)
                four2spec(spTrans, gridID1, array1.data(), gridID2, array2.data());
              else if (operatorID == SP2FC)
                spec2four(spTrans, gridID1, array1.data(), gridID2, array2.data());
              else if (operatorID == FC2GP)
                four2grid(fcTrans, gridID1, array1.data(), gridID2, array2.data());
              else if (operatorID == GP2FC)
                grid2four(fcTrans, gridID1, array1.data(), gridID2, array2.data());

              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, array2.data(), nmiss);
            }
          else
            {
              cdoDefRecord(streamID2, varID, levelID);
              if (lcopy)
                {
                  cdoCopyRecord(streamID2, streamID1);
                }
              else
                {
                  cdoReadRecord(streamID1, array1.data(), &nmiss);
                  cdoWriteRecord(streamID2, array1.data(), nmiss);
                }
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
