/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

     Subtrend   subtrend        Subtract trend
*/

#include <cdi.h>

#include "cdo_int.h"
#include "pstream_int.h"

void *
Subtrend(void *process)
{
  int varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  const int streamID1 = cdoStreamOpenRead(cdoStreamName(0));
  const int streamID2 = cdoStreamOpenRead(cdoStreamName(1));
  const int streamID3 = cdoStreamOpenRead(cdoStreamName(2));

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = cdoStreamInqVlist(streamID2);
  const int vlistID3 = cdoStreamInqVlist(streamID3);
  const int vlistID4 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_DIM);
  vlistCompare(vlistID1, vlistID3, CMP_DIM);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID4 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID4, taxisID4);

  const int streamID4 = cdoStreamOpenWrite(cdoStreamName(3));
  pstreamDefVlist(streamID4, vlistID4);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field1, field4;
  fieldInit(field1);
  fieldInit(field4);
  field1.ptr = (double *) Malloc(gridsizemax * sizeof(double));
  field4.ptr = (double *) Malloc(gridsizemax * sizeof(double));

  FieldVector2D vars2, vars3;
  fieldsFromVlist(vlistID1, vars2, FIELD_PTR);
  fieldsFromVlist(vlistID1, vars3, FIELD_PTR);

  int tsID = 0;
  int nrecs = cdoStreamInqTimestep(streamID2, tsID);

  for (int recID = 0; recID < nrecs; recID++)
    {
      pstreamInqRecord(streamID2, &varID, &levelID);
      pstreamReadRecord(streamID2, vars2[varID][levelID].ptr, &nmiss);
    }

  tsID = 0;
  nrecs = cdoStreamInqTimestep(streamID3, tsID);

  for (int recID = 0; recID < nrecs; recID++)
    {
      pstreamInqRecord(streamID3, &varID, &levelID);
      pstreamReadRecord(streamID3, vars3[varID][levelID].ptr, &nmiss);
    }

  tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID4, taxisID1);
      pstreamDefTimestep(streamID4, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          pstreamInqRecord(streamID1, &varID, &levelID);
          pstreamReadRecord(streamID1, field1.ptr, &nmiss);

          const int gridID = vlistInqVarGrid(vlistID1, varID);
          const size_t gridsize = gridInqSize(gridID);

          const double missval = vlistInqVarMissval(vlistID1, varID);
          const double missval1 = missval;
          const double missval2 = missval;
          for (size_t i = 0; i < gridsize; i++)
            field4.ptr[i] = SUBMN(field1.ptr[i], ADDMN(vars2[varID][levelID].ptr[i], MULMN(vars3[varID][levelID].ptr[i], tsID)));

          nmiss = arrayNumMV(gridsize, field4.ptr, missval);
          pstreamDefRecord(streamID4, varID, levelID);
          pstreamWriteRecord(streamID4, field4.ptr, nmiss);
        }

      tsID++;
    }

  fieldsFree(vlistID1, vars2);
  fieldsFree(vlistID1, vars3);

  if (field1.ptr) Free(field1.ptr);
  if (field4.ptr) Free(field4.ptr);

  cdoStreamClose(streamID4);
  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
