import sys, os

def writeConstraintList(stream, constraintList):
  """ Write out a MODULE format constraint list, 
  using only unambiguous, stereospecifically assigned constraints 
  Input: stream: A writable stream,  
  constraintList: CCPN ConstraintList, Distance Hbond, Jcoupling, or RDC 
  -- Based on rhf code for PALES (=NMRWISH?), djo35 19th August 2010
  """

  atomDict = {
    'H' : 'HN',
  }
  
  if constraintList.className not in ('DistanceConstraintList', 
                                      'HBondConstraintList', 
                                      'JCouplingConstraintList', 
                                      'RdcConstraintList'):
    raise Exception(
     "writeConstraintList: incorrect type of input constraint list: %s"
     % constraintList.className)
  
  # make data arrays, and get MolSystem
  iAtoms = []
  jAtoms = []
  constraints = []
  molSystems = set()
  for constraint in constraintList.sortedConstraints():

    # get non-ambiuous ConstraintItem 
    items = constraint.sortedItems()
    if len(items) != 1:
      # skip ambiguous constraints
      continue
    item = items[0]

    # get resonances associated with restraint
    resonances = item.orderedResonances or item.sortedResonances()

    # get the assigned atoms associated with resonance
    assignAtoms = []
    for resonance in resonances:
      resonanceSet = resonance.resonanceSet
      if resonanceSet is not None:
        atomSets = list(resonanceSet.getAtomSets())
        if len(atomSets) == 1:
          atoms = atomSets[0].sortedAtoms()
          if len(atoms) == 1:
            assignAtoms.append(atoms[0])

    iAtom, jAtom = assignAtoms

    value = constraint.targetValue
    if value is None:
      continue

    iAtoms.append(iAtom)
    jAtoms.append(jAtom)
    constraints.append(constraint)

    molSystems.add(iAtom.topObject)
    molSystems.add(jAtom.topObject)

  # check for unique MolSystem
  if len(molSystems) == 1:
    molSystem = molSystems.pop()
  else:
    raise Exception(
     "writeConstraintList: input constraint list %s includes more than one MolSystem"
     % constraintList)

  # write header comment
  stream.write("# File generated by CcpNmr Analysis\n")
  stream.write("# RESNO ATNAME RESNO ATNAME DIST UNC\n")
  format = "%4d %2s %4d %2s  % 13.9f %8.3f\n"
  stream.write("# FORMAT: %s" % format)

  # write actual data
  for ii, constraint in enumerate(constraints):
    ll = []
    for atom in (iAtoms[ii], jAtoms[ii]):
      residue = atom.residue
      chain = residue.chain
      chainCode = chain.pdbOneLetterCode
      if not chainCode.strip():
        chainCode = chain.code[0]
      atomName = atomDict.get(atom.name, atom.name)
      ll.extend( [ residue.seqCode, atomName ] )

    ll.append(constraint.targetValue)
    ll.append(constraint.error)
    #
    stream.write(format % tuple(ll))

def BlackledgeToPdbConverter( inFile, outFile ):
  """ It would seem that MODULE doesn't produce acceptable PDB code.
      Hopefully this function will fix it. """

  if not os.path.isfile( inFile ):
    print '%s cannot find input file %s, exiting.' % ( __name__, inFile )
    sys.exit()

  newLines = []
  for line in open( inFile, 'r' ).readlines():
    if line.startswith( 'ATOM' ):
      # insert extra space
      newLine = line[:55] + ' ' + line[55:60] + ' ' + line[60:65] + '          ' + line[65:]
      newLines.append( newLine )
    else:
      newLines.append( line )

  open( outFile, 'w' ).writelines( newLines )

def getFixedResonance(chain, nmrConstraintStore, seqCode, atomName):
  """ Wayne's utility function for getting a fixed resonance from RDC value stuff """

  atomDict = {
    'HN': 'H',
  }

  seqCode = int(seqCode)
  residue = chain.findFirstResidue(seqCode=seqCode)
  atomName = atomDict.get(atomName, atomName)
  atom = residue.findFirstAtom(name=atomName)
  #if not atom:
  #  print 'HERE11', atomName, seqCode, residue.ccpCode, [x.name for x in residue.sortedAtoms()]

  atoms = (atom,)
  fixedAtomSet = nmrConstraintStore.findFirstFixedAtomSet(atoms=atoms)
  if fixedAtomSet:
    atomSets = (fixedAtomSet,)
    fixedResonanceSet = nmrConstraintStore.findFirstFixedResonanceSet(atomSets=atomSets)
    fixedResonance = fixedResonanceSet.findFirstResonance()
  else:
    fixedAtomSet = nmrConstraintStore.newFixedAtomSet(atoms=atoms)

    isotopeCode = atomName[0]
    fixedResonance = nmrConstraintStore.newFixedResonance(isotopeCode=isotopeCode)

    atomSets = (fixedAtomSet,)
    resonances = (fixedResonance,)
    fixedResonanceSet = nmrConstraintStore.newFixedResonanceSet(atomSets=atomSets, resonances=resonances)

  return fixedResonance

def getBackValuesListFromFile( rdcFile, chain, nmrConstraintStore ):
  """ Wayne's read a MODULE rdc file to CCPN code """

  rdcConstraintList = nmrConstraintStore.newRdcConstraintList()

  fp = open( rdcFile )

  for line in fp.readlines():
    splitline = line.strip().split()
    # (seq1, at1, seq2, at2, dist, err) = [t.strip() for t in line.split()]

    if len(splitline) < 8: continue

    try: int(splitline[0])
    except: continue

    (seq1, at1, seq2, at2, dist, back, diff, err ) = [t.strip() for t in splitline][:8]

    resonance1 = getFixedResonance(chain, nmrConstraintStore, seq1, at1)
    resonance2 = getFixedResonance(chain, nmrConstraintStore, seq2, at2)

    dist = float(back)
    err = float(err)
    rdcConstraint = rdcConstraintList.newRdcConstraint(targetValue=dist, error=err)
    resonances = (resonance1, resonance2)
    rdcConstraintItem = rdcConstraint.newRdcConstraintItem(resonances=resonances)

    line = fp.readline().strip()
  fp.close()

  return rdcConstraintList

def getRawBackValuesFromFile( rdcFile ):

  rdcData = []

  fp = open( rdcFile )
  
  for line in fp.readlines():
    splitline = line.strip().split()

    if len(splitline) > 8: 

      (seq1, at1, seq2, at2, dist, back, diff, err ) = [t.strip() for t in splitline][:8]
  
      rdcData.append( (seq1, at1, seq2, at2, dist, back, diff, err ) )
  
  fp.close()
  
  return rdcData


if __name__ == '__main__':

  # BlackledgeToPdbConverter( 'module/tempC.pdb', 'fileIo.pdb' )

  for line in getRawBackValuesFromFile( 'tmpqfl7PJ.tab.back' ):
    print line














