/*
This tool is part of the WhiteboxTools geospatial analysis library.
Authors: Dr. John Lindsay
Created: 22/06/2017
Last Modified: 14/02/2020
License: MIT
*/

use crate::raster::*;
use crate::structures::Array2D;
use crate::tools::*;
use crate::vector::*;
use std::env;
use std::f64;
use std::io::{Error, ErrorKind};
use std::path;

/// This tool will perform a watershedding operation based on a group of input vector pour points (`--pour_pts`),
/// i.e. outlets or points-of-interest, or a raster containing point points. Watershedding is a procedure that identifies
/// all of the cells upslope of a cell of interest (pour point) that are connected to the pour point by a flow-path. The
/// user must specify the name of a D8-derived flow pointer (flow direction) raster (`--d8_pntr`), a vector pour point file
/// (`--pour_pts`), and the output raster (`--output`). The pour points must be of a Point ShapeType (i.e. Point, PointZ, PointM,
/// MultiPoint, MultiPointZ, MultiPointM). Watersheds will be assigned the input pour point FID value. The flow
/// pointer raster must be generated using the D8 algorithm, `D8Pointer`.
///
/// Pour point vectors can be attained by on-screen digitizing to designate these points-of-interest locations.
/// Because pour points are usually, although not always, situated on a stream network, it is recommended that you
/// use Jenson's method (`JensonSnapPourPoints`) to snap pour points on the stream network. This will ensure that
/// the digitized outlets are coincident with the digital stream contained within the DEM flowpaths. If this is not
/// done prior to inputting a pour-point set to the `Watershed` tool, anomalously small watersheds may be ouput, as
/// pour points that fall off of the main flow path (even by one cell) in the D8 pointer will yield very different
/// catchment areas.
///
/// If a raster pour point is specified instead of vector points, the watershed labels will derive their IDs from the
/// grid cell values of all non-zero, non-NoData valued grid cells in the pour points file. Notice that this file can
/// contain any integer data. For example, if a lakes raster, with each lake possessing a unique ID, is used as the
/// pour points raster, the tool will map the watersheds draining to each of the input lake features. Similarly,
/// a pour points raster may actually be a streams file, such as what is generated by the `StreamLinkIdentifier` tool.
///
/// By default, the pointer raster is assumed to use the clockwise indexing method used by WhiteboxTools.
/// If the pointer file contains ESRI flow direction values instead, the `--esri_pntr` parameter must be specified.
///
/// There are several tools that perform similar watershedding operations in WhiteboxTools. `Watershed` is appropriate
/// to use when you have a set of specific locations for which you need to derive the watershed areas. Use the `Basins`
/// tool instead when you simply want to find the watersheds draining to each outlet situated along the edge of a
/// DEM. The `Isobasins` tool can be used to divide a landscape into roughly equally sized watersheds. The `Subbasins`
/// and `StrahlerOrderBasins` are useful when you need to find the areas draining to each link within a stream network.
/// Finally, `Hillslopes` can be used to idenfity the areas draining the each of the left and right banks of a stream
/// network.
///
/// # Reference
/// Jenson, S. K. (1991), Applications of hydrological information automati-cally extracted from digital elevation
/// models, Hydrological Processes, 5, 31–44, doi:10.1002/hyp.3360050104.
///
/// Lindsay JB, Rothwell JJ, and Davies H. 2008. Mapping outlet points used for watershed delineation onto DEM-derived
/// stream networks, Water Resources Research, 44, W08442, doi:10.1029/2007WR006507.
///
/// # See Also
/// `D8Pointer`, `Basins`, `Subbasins`, `Isobasins`, `StrahlerOrderBasins`, `Hillslopes`, `JensonSnapPourPoints`,
/// `BreachDepressions`, `FillDepressions`
pub struct Watershed {
    name: String,
    description: String,
    toolbox: String,
    parameters: Vec<ToolParameter>,
    example_usage: String,
}

impl Watershed {
    pub fn new() -> Watershed {
        // public constructor
        let name = "Watershed".to_string();
        let toolbox = "Hydrological Analysis".to_string();
        let description =
            "Identifies the watershed, or drainage basin, draining to a set of target cells."
                .to_string();

        let mut parameters = vec![];
        parameters.push(ToolParameter {
            name: "Input D8 Pointer File".to_owned(),
            flags: vec!["--d8_pntr".to_owned()],
            description: "Input D8 pointer raster file.".to_owned(),
            parameter_type: ParameterType::ExistingFile(ParameterFileType::Raster),
            default_value: None,
            optional: false,
        });

        parameters.push(ToolParameter {
            name: "Input Pour Points (Outlet) File".to_owned(),
            flags: vec!["--pour_pts".to_owned()],
            // description: "Input vector pour points (outlet) file.".to_owned(),
            // parameter_type: ParameterType::ExistingFile(ParameterFileType::Vector(
            //     VectorGeometryType::Point,
            // )),
            description: "Input pour points (outlet) file.".to_owned(),
            parameter_type: ParameterType::ExistingFile(ParameterFileType::RasterAndVector(
                VectorGeometryType::Point,
            )),
            default_value: None,
            optional: false,
        });

        parameters.push(ToolParameter {
            name: "Output File".to_owned(),
            flags: vec!["-o".to_owned(), "--output".to_owned()],
            description: "Output raster file.".to_owned(),
            parameter_type: ParameterType::NewFile(ParameterFileType::Raster),
            default_value: None,
            optional: false,
        });

        parameters.push(ToolParameter {
            name: "Does the pointer file use the ESRI pointer scheme?".to_owned(),
            flags: vec!["--esri_pntr".to_owned()],
            description: "D8 pointer uses the ESRI style scheme.".to_owned(),
            parameter_type: ParameterType::Boolean,
            default_value: Some("false".to_owned()),
            optional: true,
        });

        let sep: String = path::MAIN_SEPARATOR.to_string();
        let p = format!("{}", env::current_dir().unwrap().display());
        let e = format!("{}", env::current_exe().unwrap().display());
        let mut short_exe = e
            .replace(&p, "")
            .replace(".exe", "")
            .replace(".", "")
            .replace(&sep, "");
        if e.contains(".exe") {
            short_exe += ".exe";
        }
        let usage = format!(">>.*{0} -r={1} -v --wd=\"*path*to*data*\" --d8_pntr='d8pntr.tif' --pour_pts='pour_pts.shp' -o='output.tif'", short_exe, name).replace("*", &sep);

        Watershed {
            name: name,
            description: description,
            toolbox: toolbox,
            parameters: parameters,
            example_usage: usage,
        }
    }
}

impl WhiteboxTool for Watershed {
    fn get_source_file(&self) -> String {
        String::from(file!())
    }

    fn get_tool_name(&self) -> String {
        self.name.clone()
    }

    fn get_tool_description(&self) -> String {
        self.description.clone()
    }

    fn get_tool_parameters(&self) -> String {
        match serde_json::to_string(&self.parameters) {
            Ok(json_str) => return format!("{{\"parameters\":{}}}", json_str),
            Err(err) => return format!("{:?}", err),
        }
    }

    fn get_example_usage(&self) -> String {
        self.example_usage.clone()
    }

    fn get_toolbox(&self) -> String {
        self.toolbox.clone()
    }

    fn run<'a>(
        &self,
        args: Vec<String>,
        working_directory: &'a str,
        verbose: bool,
    ) -> Result<(), Error> {
        let mut d8_file = String::new();
        let mut pourpts_file = String::new();
        let mut output_file = String::new();
        let mut esri_style = false;

        if args.len() == 0 {
            return Err(Error::new(
                ErrorKind::InvalidInput,
                "Tool run with no parameters.",
            ));
        }
        for i in 0..args.len() {
            let mut arg = args[i].replace("\"", "");
            arg = arg.replace("\'", "");
            let cmd = arg.split("="); // in case an equals sign was used
            let vec = cmd.collect::<Vec<&str>>();
            let mut keyval = false;
            if vec.len() > 1 {
                keyval = true;
            }
            let flag_val = vec[0].to_lowercase().replace("--", "-");
            if flag_val == "-d8_pntr" {
                d8_file = if keyval {
                    vec[1].to_string()
                } else {
                    args[i + 1].to_string()
                };
            } else if flag_val == "-pour_pts" {
                pourpts_file = if keyval {
                    vec[1].to_string()
                } else {
                    args[i + 1].to_string()
                };
            } else if flag_val == "-o" || flag_val == "-output" {
                output_file = if keyval {
                    vec[1].to_string()
                } else {
                    args[i + 1].to_string()
                };
            } else if flag_val == "-esri_pntr" || flag_val == "-esri_style" {
                if vec.len() == 1 || !vec[1].to_string().to_lowercase().contains("false") {
                    esri_style = true;
                }
            }
        }

        if verbose {
            println!("***************{}", "*".repeat(self.get_tool_name().len()));
            println!("* Welcome to {} *", self.get_tool_name());
            println!("***************{}", "*".repeat(self.get_tool_name().len()));
        }

        let sep: String = path::MAIN_SEPARATOR.to_string();

        let mut progress: usize;
        let mut old_progress: usize = 1;

        if !d8_file.contains(&sep) && !d8_file.contains("/") {
            d8_file = format!("{}{}", working_directory, d8_file);
        }
        if !pourpts_file.contains(&sep) && !pourpts_file.contains("/") {
            pourpts_file = format!("{}{}", working_directory, pourpts_file);
        }
        if !output_file.contains(&sep) && !output_file.contains("/") {
            output_file = format!("{}{}", working_directory, output_file);
        }

        if verbose {
            println!("Reading data...")
        };

        let pntr = Raster::new(&d8_file, "r")?;

        let start = Instant::now();

        let rows = pntr.configs.rows as isize;
        let columns = pntr.configs.columns as isize;
        let nodata = -32768f64;
        let pntr_nodata = pntr.configs.nodata;

        // Create a mapping from the pointer values to cells offsets.
        // This may seem wasteful, using only 8 of 129 values in the array,
        // but the mapping method is far faster than calculating z.ln() / ln(2.0).
        // It's also a good way of allowing for different point styles.
        let mut pntr_matches: [i8; 129] = [0i8; 129];
        if !esri_style {
            // This maps Whitebox-style D8 pointer values
            // onto the cell offsets in dx and dy.
            pntr_matches[1] = 0i8;
            pntr_matches[2] = 1i8;
            pntr_matches[4] = 2i8;
            pntr_matches[8] = 3i8;
            pntr_matches[16] = 4i8;
            pntr_matches[32] = 5i8;
            pntr_matches[64] = 6i8;
            pntr_matches[128] = 7i8;
        } else {
            // This maps Esri-style D8 pointer values
            // onto the cell offsets in dx and dy.
            pntr_matches[1] = 1i8;
            pntr_matches[2] = 2i8;
            pntr_matches[4] = 3i8;
            pntr_matches[8] = 4i8;
            pntr_matches[16] = 5i8;
            pntr_matches[32] = 6i8;
            pntr_matches[64] = 7i8;
            pntr_matches[128] = 0i8;
        }

        let dx = [1, 1, 1, 0, -1, -1, -1, 0];
        let dy = [-1, 0, 1, 1, 1, 0, -1, -1];
        let mut z: f64;

        let mut flow_dir: Array2D<i8> = Array2D::new(rows, columns, -2, -2)?;
        let mut output = Raster::initialize_using_file(&output_file, &pntr);
        output.configs.nodata = nodata;
        output.configs.data_type = DataType::I32;
        output.configs.photometric_interp = PhotometricInterpretation::Categorical;
        output.configs.palette = "qual.pal".to_string(); //palette;
        let low_value = f64::MIN;
        output.reinitialize_values(low_value);

        if pourpts_file.to_lowercase().ends_with(".shp") {
            // Note that this only works because at the moment, Shapefiles are the only supported vector.
            // If additional vector formats are added in the future, this will need updating.
            let pourpts = Shapefile::read(&pourpts_file)?;

            // make sure the input vector file is of points type
            if pourpts.header.shape_type.base_shape_type() != ShapeType::Point {
                return Err(Error::new(
                    ErrorKind::InvalidInput,
                    "The input vector data must be of point base shape type.",
                ));
            }
            for record_num in 0..pourpts.num_records {
                let record = pourpts.get_record(record_num);
                let row = pntr.get_row_from_y(record.points[0].y);
                let col = pntr.get_column_from_x(record.points[0].x);
                output.set_value(row, col, (record_num + 1) as f64);

                if verbose {
                    progress =
                        (100.0_f64 * record_num as f64 / (pourpts.num_records - 1) as f64) as usize;
                    if progress != old_progress {
                        println!("Locating pour points: {}%", progress);
                        old_progress = progress;
                    }
                }
            }

            for row in 0..rows {
                for col in 0..columns {
                    z = pntr[(row, col)];
                    if z != pntr_nodata {
                        if z > 0.0 {
                            flow_dir.set_value(row, col, pntr_matches[z as usize]);
                        } else {
                            flow_dir.set_value(row, col, -1i8);
                        }
                    } else {
                        output.set_value(row, col, nodata);
                    }
                }
                if verbose {
                    progress = (100.0_f64 * row as f64 / (rows - 1) as f64) as usize;
                    if progress != old_progress {
                        println!("Initializing: {}%", progress);
                        old_progress = progress;
                    }
                }
            }
        } else {
            // it's a raster
            let pourpts = Raster::new(&pourpts_file, "r")?;
            output.configs.palette = pourpts.configs.palette.clone();
            // make sure the input files have the same size
            if pourpts.configs.rows != pntr.configs.rows
                || pourpts.configs.columns != pntr.configs.columns
            {
                return Err(Error::new(ErrorKind::InvalidInput,
                                    "The input files must have the same number of rows and columns and spatial extent."));
            }

            for row in 0..rows {
                for col in 0..columns {
                    z = pntr.get_value(row, col);
                    if z != pntr_nodata {
                        if z > 0.0 {
                            flow_dir.set_value(row, col, pntr_matches[z as usize]);
                        } else {
                            flow_dir.set_value(row, col, -1i8);
                        }
                    } else {
                        output.set_value(row, col, nodata);
                    }
                    z = pourpts.get_value(row, col);
                    if z != nodata && z > 0.0 {
                        output.set_value(row, col, z);
                    }
                }
                if verbose {
                    progress = (100.0_f64 * row as f64 / (rows - 1) as f64) as usize;
                    if progress != old_progress {
                        println!("Initializing: {}%", progress);
                        old_progress = progress;
                    }
                }
            }
        };

        let mut flag: bool;
        let (mut x, mut y): (isize, isize);
        let mut dir: i8;
        let mut outlet_id: f64;
        for row in 0..rows {
            for col in 0..columns {
                if output[(row, col)] == low_value {
                    flag = false;
                    x = col;
                    y = row;
                    outlet_id = nodata;
                    while !flag {
                        // find its downslope neighbour
                        dir = flow_dir[(y, x)];
                        if dir >= 0 {
                            // move x and y accordingly
                            x += dx[dir as usize];
                            y += dy[dir as usize];

                            // if the new cell already has a value in the output, use that as the outletID
                            z = output[(y, x)];
                            if z != low_value {
                                outlet_id = z;
                                flag = true;
                            }
                        } else {
                            flag = true;
                        }
                    }

                    flag = false;
                    x = col;
                    y = row;
                    output[(y, x)] = outlet_id;
                    while !flag {
                        // find its downslope neighbour
                        dir = flow_dir[(y, x)];
                        if dir >= 0 {
                            // move x and y accordingly
                            x += dx[dir as usize];
                            y += dy[dir as usize];

                            // if the new cell already has a value in the output, use that as the outletID
                            if output[(y, x)] != low_value {
                                flag = true;
                            }
                        } else {
                            flag = true;
                        }
                        output[(y, x)] = outlet_id;
                    }
                }
            }
            if verbose {
                progress = (100.0_f64 * row as f64 / (rows - 1) as f64) as usize;
                if progress != old_progress {
                    println!("Progress: {}%", progress);
                    old_progress = progress;
                }
            }
        }

        let elapsed_time = get_formatted_elapsed_time(start);
        output.add_metadata_entry(format!(
            "Created by whitebox_tools\' {} tool",
            self.get_tool_name()
        ));
        output.add_metadata_entry(format!("D8 pointer file: {}", d8_file));
        output.add_metadata_entry(format!("Pour-points file: {}", pourpts_file));
        output.add_metadata_entry(format!("Elapsed Time (excluding I/O): {}", elapsed_time));

        if verbose {
            println!("Saving data...")
        };
        let _ = match output.write() {
            Ok(_) => {
                if verbose {
                    println!("Output file written")
                }
            }
            Err(e) => return Err(e),
        };
        if verbose {
            println!(
                "{}",
                &format!("Elapsed Time (excluding I/O): {}", elapsed_time)
            );
        }

        Ok(())
    }
}
