//! Utility functions for commands.
#![allow(clippy::redundant_closure)]

use failure;
use std::fs;
use std::path::{Path, PathBuf};
use std::time::Duration;
use walkdir::WalkDir;

/// If an explicit path is given, then use it, otherwise assume the current
/// directory is the crate path.
pub fn get_crate_path(path: Option<PathBuf>) -> Result<PathBuf, failure::Error> {
    match path {
        Some(p) => Ok(p),
        None => find_manifest_from_cwd(),
    }
}

/// Search up the path for the manifest file from the current working directory
/// If we don't find the manifest file then return back the current working directory
/// to provide the appropriate error
fn find_manifest_from_cwd() -> Result<PathBuf, failure::Error> {
    let mut parent_path = std::env::current_dir()?;
    let mut manifest_path = parent_path.join("Cargo.toml");
    loop {
        if !manifest_path.is_file() {
            if parent_path.pop() {
                manifest_path = parent_path.join("Cargo.toml");
            } else {
                return Ok(PathBuf::from("."));
            }
        } else {
            return Ok(parent_path);
        }
    }
}

/// Construct our `pkg` directory in the crate.
pub fn create_pkg_dir(out_dir: &Path) -> Result<(), failure::Error> {
    let _ = fs::remove_file(out_dir.join("package.json")); // Clean up package.json from previous runs
    fs::create_dir_all(&out_dir)?;
    fs::write(out_dir.join(".gitignore"), "*")?;
    Ok(())
}

/// Locates the pkg directory from a specific path
/// Returns None if unable to find the 'pkg' directory
pub fn find_pkg_directory(path: &Path) -> Option<PathBuf> {
    if is_pkg_directory(path) {
        return Some(path.to_owned());
    }

    WalkDir::new(path)
        .into_iter()
        .filter_map(|x| x.ok().map(|e| e.into_path()))
        .find(|x| is_pkg_directory(&x))
}

fn is_pkg_directory(path: &Path) -> bool {
    path.exists() && path.is_dir() && path.ends_with("pkg")
}

/// Render a `Duration` to a form suitable for display on a console
pub fn elapsed(duration: Duration) -> String {
    let secs = duration.as_secs();

    if secs >= 60 {
        format!("{}m {:02}s", secs / 60, secs % 60)
    } else {
        format!("{}.{:02}s", secs, duration.subsec_nanos() / 10_000_000)
    }
}
