use anyhow::{anyhow, Result};
use chrono::{DateTime, Local};
use rand::{thread_rng, RngCore};
use scrypt::Params;
use serde::{Deserialize, Serialize};

use crate::backend::{FileType, ReadBackend};
use crate::crypto::{CryptoKey, Key};
use crate::id::Id;

#[serde_with::apply(Option => #[serde(default, skip_serializing_if = "Option::is_none")])]
#[derive(Debug, Serialize, Deserialize)]
pub struct KeyFile {
    hostname: Option<String>,
    username: Option<String>,
    created: Option<DateTime<Local>>,
    kdf: String,
    #[serde(rename = "N")]
    n: u32,
    r: u32,
    p: u32,
    data: String,
    salt: String,
}

impl KeyFile {
    /// Generate a Key using the key derivation function from KeyFile and a given password
    fn kdf_key(&self, passwd: &impl AsRef<[u8]>) -> Result<Key> {
        let params = Params::new(log_2(self.n), self.r, self.p)
            .map_err(|_| anyhow!("invalid scrypt paramters"))?;
        let salt = base64::decode(&self.salt)?;

        let mut key = [0; 64];
        scrypt::scrypt(passwd.as_ref(), &salt, &params, &mut key).expect("output length invalid?");

        Ok(Key::from_slice(&key))
    }

    /// Extract a key from the data of the KeyFile using the given key.
    /// The key usually should be the key generated by kdf_key
    fn key_from_data(&self, key: &Key) -> Result<Key> {
        let dec_data = key
            .decrypt_data(&base64::decode(&self.data)?)
            .map_err(|_| anyhow!("decryption failed"))?;
        serde_json::from_slice::<MasterKey>(&dec_data)?.key()
    }

    /// Extract a key from the data of the KeyFile using the key
    /// from the derivation function in combination with the given password.
    pub fn key_from_password(&self, passwd: &impl AsRef<[u8]>) -> Result<Key> {
        self.key_from_data(&self.kdf_key(passwd)?)
    }

    /// Generate a new KeyFile from a given key and password.
    pub fn generate(
        key: Key,
        passwd: &impl AsRef<[u8]>,
        hostname: Option<String>,
        username: Option<String>,
        with_created: bool,
    ) -> Result<Self> {
        let masterkey = MasterKey::from_key(key);
        let params = Params::recommended();
        let mut salt = [0; 64];
        thread_rng().fill_bytes(&mut salt);

        let mut key = [0; 64];
        scrypt::scrypt(passwd.as_ref(), &salt, &params, &mut key).expect("output length invalid?");

        let key = Key::from_slice(&key);
        let data = key.encrypt_data(&serde_json::to_vec(&masterkey)?)?;

        Ok(Self {
            hostname,
            username,
            kdf: "scrypt".to_string(),
            n: 2_u32.pow(params.log_n() as u32),
            r: params.r(),
            p: params.p(),
            created: with_created.then(Local::now),
            data: base64::encode(data),
            salt: base64::encode(salt),
        })
    }
}

impl KeyFile {
    /// Get a KeyFile from the backend
    pub fn from_backend<B: ReadBackend>(be: &B, id: &Id) -> Result<Self> {
        let data = be.read_full(FileType::Key, id)?;
        Ok(serde_json::from_slice(&data)?)
    }
}

const fn num_bits<T>() -> usize {
    std::mem::size_of::<T>() * 8
}

fn log_2(x: u32) -> u8 {
    assert!(x > 0);
    num_bits::<u32>() as u8 - x.leading_zeros() as u8 - 1
}

#[derive(Debug, Serialize, Deserialize)]
struct Mac {
    k: String,
    r: String,
}

#[derive(Debug, Serialize, Deserialize)]
struct MasterKey {
    mac: Mac,
    encrypt: String,
}

impl MasterKey {
    fn from_key(key: Key) -> Self {
        let (encrypt, k, r) = key.to_keys();
        Self {
            encrypt: base64::encode(encrypt),
            mac: Mac {
                k: base64::encode(k),
                r: base64::encode(r),
            },
        }
    }

    fn key(&self) -> Result<Key> {
        Ok(Key::from_keys(
            &base64::decode(&self.encrypt)?,
            &base64::decode(&self.mac.k)?,
            &base64::decode(&self.mac.r)?,
        ))
    }
}

fn key_from_backend<B: ReadBackend>(be: &B, id: &Id, passwd: &impl AsRef<[u8]>) -> Result<Key> {
    KeyFile::from_backend(be, id)?.key_from_password(passwd)
}

/// Find a KeyFile in the backend that fits to the given password and return the contained key.
/// If a key hint is given, only this key is tested.
/// This is recommended for a large number of keys.
pub fn find_key_in_backend<B: ReadBackend>(
    be: &B,
    passwd: &impl AsRef<[u8]>,
    hint: Option<&Id>,
) -> Result<Key> {
    match hint {
        Some(id) => key_from_backend(be, id, passwd),
        None => {
            for id in be.list(FileType::Key)? {
                if let Ok(key) = key_from_backend(be, &id, passwd) {
                    return Ok(key);
                }
            }
            Err(anyhow!("no suitable key found!"))
        }
    }
}
