use clap;
use clap::ArgMatches;
pub type StaticSubcommand = clap::App<'static, 'static>;

mod fs_operation;
mod ssh_auth_attempts;
pub mod remote;
mod ask;

pub mod hooks;
pub mod info;
pub mod init;
pub mod record;
pub mod add;
pub mod pull;
pub mod push;
pub mod apply;
pub mod clone;
pub mod remove;
pub mod mv;
pub mod ls;
pub mod revert;
pub mod unrecord;
pub mod log;
pub mod patch;
pub mod fork;
pub mod branches;
pub mod delete_branch;
pub mod checkout;
pub mod diff;
pub mod credit;
pub mod dist;
pub mod key;
pub mod rollback;
pub mod status;
pub mod show_dependencies;
pub mod tag;
pub mod sign;
pub mod challenge;
pub mod generate_completions;

use libpijul::Hash;

use rand;
use std::fs::{File, canonicalize, metadata, create_dir};
use std::path::{Path, PathBuf};
use std::env::current_dir;
use libpijul::{DEFAULT_BRANCH, Repository, fs_representation, Inode, Txn};
use error::{Result, ErrorKind};
use std::io::{Read, Write, stderr};
use std::process::exit;
use std::borrow::Cow;
use std::env::var;

pub fn all_command_invocations() -> Vec<StaticSubcommand> {
    return vec![log::invocation(),
                info::invocation(),
                init::invocation(),
                record::invocation(),
                unrecord::invocation(),
                add::invocation(),
                pull::invocation(),
                push::invocation(),
                apply::invocation(),
                clone::invocation(),
                remove::invocation(),
                mv::invocation(),
                ls::invocation(),
                revert::invocation(),
                patch::invocation(),
                fork::invocation(),
                branches::invocation(),
                delete_branch::invocation(),
                checkout::invocation(),
                diff::invocation(),
                credit::invocation(),
                dist::invocation(),
                key::invocation(),
                rollback::invocation(),
                status::invocation(),
                show_dependencies::invocation(),
                tag::invocation(),
                sign::invocation(),
                challenge::invocation(),
                generate_completions::invocation(),
    ];
}

pub fn get_wd(repository_path: Option<&Path>) -> Result<PathBuf> {
    debug!("get_wd: {:?}", repository_path);
    match repository_path {
        None => Ok(canonicalize(current_dir()?)?),
        Some(a) if a.is_relative() => Ok(canonicalize(current_dir()?.join(a))?),
        Some(a) => Ok(canonicalize(a)?)
    }
}

pub fn get_current_branch(root: &Path) -> Result<String> {
    debug!("path: {:?}", root);
    let mut path = fs_representation::repo_dir(&canonicalize(root)?);
    path.push("current_branch");
    if let Ok(mut f) = File::open(&path) {
        let mut s = String::new();
        f.read_to_string(&mut s)?;
        Ok(s.trim().to_string())
    } else {
        Ok(DEFAULT_BRANCH.to_string())
    }
}

pub fn set_current_branch(root: &Path, branch: &str) -> Result<()> {
    debug!("set current branch: root={:?}, branch={:?}", root, branch);
    let mut path = fs_representation::repo_dir(&canonicalize(root)?);
    path.push("current_branch");
    let mut f = File::create(&path)?;
    f.write_all(branch.trim().as_ref())?;
    f.write_all(b"\n")?;
    Ok(())
}

/// Returns an error if the `dir` is contained in a repository.
pub fn assert_no_containing_repo(dir: &Path) -> Result<()> {
    if metadata(dir).is_ok() {
        if fs_representation::find_repo_root(&canonicalize(dir)?).is_some() {
            return Err(ErrorKind::InARepository(dir.to_owned()).into())
        }
    }
    Ok(())
}

/// Creates an empty pijul repository in the given directory.
pub fn create_repo(dir: &Path) -> Result<()> {
    // Check that a repository does not already exist.
    if metadata(dir).is_err() {
        create_dir(dir)?;
    }
    let dir = canonicalize(dir)?;
    if fs_representation::find_repo_root(&dir).is_some() {
        return Err(ErrorKind::InARepository(dir.to_owned()).into());
    }

    fs_representation::create(&dir, rand::thread_rng())?;
    let pristine_dir = fs_representation::pristine_dir(&dir);
    let repo = Repository::open(&pristine_dir, None)?;
    repo.mut_txn_begin(rand::thread_rng())?
        .commit()?;
    Ok(())
}

fn default_explain<R>(command_result: Result<R>) {
    debug!("default_explain");
    match command_result {
        Ok(_) => (),
        Err(e) => {
            writeln!(stderr(), "error: {}", e).unwrap();
            exit(1)
        }
    }
}

fn validate_base58(x: String) -> ::std::result::Result<(), String> {
    if Hash::from_base58(&x).is_some() {
        Ok(())
    } else {
        Err(format!("\"{}\" is invalid base58", x))
    }
}

/// Almost all commands want to know the current directory and the repository root.  This struct
/// fills that need, and also provides methods for other commonly-used tasks.
pub struct BasicOptions<'a> {
    /// This isn't 100% the same as the actual current working directory, so pay attention: this
    /// will be the current directory, unless the user specifies `--repository`, in which case
    /// `cwd` will actually be the path of the repository root. In other words, specifying
    /// `--repository` has the same effect as changing directory to the repository root before
    /// running `pijul`.
    pub cwd: PathBuf,
    pub repo_root: PathBuf,
    args: &'a ArgMatches<'a>,
}



impl<'a> BasicOptions<'a> {

    /// Reads the options from command line arguments.
    pub fn from_args(args: &'a ArgMatches<'a>) -> Result<BasicOptions<'a>> {
        let wd = get_wd(args.value_of("repository").map(Path::new))?;
        let repo_root = if let Some(r) = fs_representation::find_repo_root(&canonicalize(&wd)?) {
            r
        } else {
            return Err(ErrorKind::NotInARepository.into())
        };
        Ok(BasicOptions {
            cwd: wd,
            repo_root: repo_root,
            args: args,
        })
    }

    /// Gets the name of the desired branch.
    pub fn branch(&self) -> String {
        if let Some(b) = self.args.value_of("branch") {
            b.to_string()
        } else if let Ok(b) = get_current_branch(&self.repo_root) {
            b
        } else {
            DEFAULT_BRANCH.to_string()
        }
    }

    pub fn repo_dir(&self) -> PathBuf {
        fs_representation::repo_dir(&self.repo_root)
    }

    pub fn repo_root(&self) -> PathBuf {
        self.repo_root.clone()
    }

    pub fn open_repo(&self) -> Result<Repository> {
        Repository::open(self.pristine_dir(), None).map_err(|e| e.into())
    }

    pub fn open_and_grow_repo(&self, increase: u64) -> Result<Repository> {
        Repository::open(self.pristine_dir(), Some(increase)).map_err(|e| e.into())
    }

    pub fn pristine_dir(&self) -> PathBuf {
        fs_representation::pristine_dir(&self.repo_root)
    }

    pub fn patches_dir(&self) -> PathBuf {
        fs_representation::patches_dir(&self.repo_root)
    }

    fn dir_inode(&self, txn: &Txn) -> Result<Inode> {
        use libpijul::ROOT_INODE;
        if let Some(dir) = self.args.value_of("dir") {
            let dir = if Path::new(dir).is_relative() {
                let root = if let Some(root) = self.args.value_of("repository") {
                    Path::new(root).to_path_buf()
                } else {
                    current_dir()?
                };
                root.join(&dir).canonicalize()?
            } else {
                Path::new(dir).canonicalize()?
            };
            let prefix = self.repo_root();
            let dir = dir.strip_prefix(&prefix)?;
            debug!("{:?}", dir);
            let inode = txn.find_inode(&dir)?;
            debug!("{:?}", inode);
            Ok(inode)
        } else {
            Ok(ROOT_INODE)
        }
    }

}

fn remote_pijul_cmd() -> Cow<'static, str> {
    if let Ok(cmd) = var("REMOTE_PIJUL") {
        Cow::Owned(cmd)
    } else {
        Cow::Borrowed("pijul")
    }
}

#[cfg(unix)]
fn setup_pager() {
    use pager::Pager;
    Pager::with_pager("less -r").setup()
}

#[cfg(not(unix))]
fn setup_pager() {
}
