use clap::{SubCommand, Arg, ArgMatches};
use error::{ErrorKind, Error};
use commands::{BasicOptions, StaticSubcommand};
use std::io::{Write,stderr};
use std::process::exit;
use meta;
use meta::KeyType;
use thrussh;
use bincode;
use futures;
use thrussh::{client, ChannelId};
use thrussh_keys::key;
use futures::Future;
use std::sync::Arc;
use regex::Regex;
use thrussh_keys;
use std;
use super::ask;
use std::path::PathBuf;
use std::borrow::Cow;
use username;
use cryptovec;

pub fn invocation() -> StaticSubcommand {
    return SubCommand::with_name("key")
        .about("Manage signing and SSH keys")
        .subcommand(
            SubCommand::with_name("upload")
                .about("Upload keys to a remote server")
                .arg(Arg::with_name("port")
                     .long("port")
                     .short("p")
                     .help("Port of the SSH server.")
                     .takes_value(true)
                     .required(false))
                .arg(Arg::with_name("repository")
                     .long("repository")
                     .help("The repository where the signing key is, if the key was generated with --for-repository.")
                     .takes_value(true)
                     .required(false))
                .arg(Arg::with_name("address")
                     .help("Address to use, for instance pijul_org@nest.pijul.com.")
                     .takes_value(true)
                     .required(true))
        )
        .subcommand(
            SubCommand::with_name("gen")
                .about("Generate keys. If neither --ssh nor --signing is given, both key types are generated.")
                .arg(Arg::with_name("ssh")
                     .long("ssh")
                     .help("Generate an SSH key")
                     .takes_value(false))
                .arg(Arg::with_name("signing")
                     .long("signing")
                     .help("Generate a signing key")
                     .takes_value(false))
                .arg(Arg::with_name("local")
                     .long("local")
                     .help("Save keys for the local repository only")
                     .takes_value(false)
                     .required(false))
                .arg(Arg::with_name("repository")
                     .long("for-repository")
                     .help("Save keys for the given repository only")
                     .takes_value(true)
                     .required(false))
        )
}

pub enum Params<'a> {
    Upload {
        address: &'a str,
        port: u16,
        repository: Option<PathBuf>,
        remote_cmd: Cow<'static, str>,
    },
    Gen {
        signing: bool,
        ssh: bool,
        local: Option<PathBuf>,
    },
    None
}

pub fn parse_args<'a>(args: &'a ArgMatches) -> Result<Params<'a>, Error> {
    match args.subcommand() {
        ("upload", Some(args)) =>
            Ok(Params::Upload {
                address: args.value_of("address").unwrap(),
                port: args.value_of("port").and_then(|x| x.parse().ok()).unwrap_or(22),
                repository: if args.is_present("repository") {
                    Some(BasicOptions::from_args(args)?.repo_dir())
                } else {
                    None
                },
                remote_cmd: super::remote_pijul_cmd(),
            }),
        ("gen", Some(args)) =>
            Ok(Params::Gen {
                signing: args.is_present("signing") || !args.is_present("ssh"),
                ssh: args.is_present("ssh") || !args.is_present("signing"),
                local: if args.is_present("repository") || args.is_present("local") {
                    Some(BasicOptions::from_args(args)?.repo_dir())
                } else {
                    None
                },
            }),
        _ => Ok(Params::None)
    }
}

pub fn run(arg_matches: &ArgMatches) -> Result<(), Error> {
    match parse_args(arg_matches)? {
        Params::Upload { address, port, repository, remote_cmd }=> {
            match meta::load_global_or_local_signing_key(repository.as_ref()) {
                Ok(key) => {
                    let config = Arc::new(thrussh::client::Config::default());
                    let ssh_user_host = Regex::new(r"^([^@]*)@(.*)$").unwrap();
                    let (user, server) =
                        if let Some(cap) = ssh_user_host.captures(&address) {
                            (cap[1].to_string(), cap[2].to_string())
                        } else {
                            (username::get_user_name().unwrap(), address.to_string())
                        };

                    let mut l = tokio_core::reactor::Core::new().unwrap();
                    let h = l.handle();

                    let client = SshClient::new(port, &server, key, &h);

                    use super::ssh_auth_attempts::{AuthAttempts, AuthAttemptFuture};
                    let use_agent = client.agent.is_some();

                    l.run(thrussh::client::connect_future(
                        h,
                        (server.as_str(), port), config, None, client,
                        |connection| {
                            AuthAttemptFuture::new(connection, AuthAttempts::new(user.to_string(), repository, use_agent), user)
                                .from_err()
                                .and_then(|session| {

                                    session.channel_open_session().and_then(|(mut session, channelid)| {
                                        session.exec(channelid, false, &format!("{} challenge", remote_cmd));
                                        session.flush().unwrap();
                                        session.wait(move |session| {
                                            session.handler().exit_status.is_some()
                                        })
                                    })
                                })
                        }
                    )?).unwrap();
                }
                Err(e) => return Err(e)


            }
        }
        Params::Gen { signing, ssh, local } => {
            if let Some(ref dot_pijul) = local {
                if ssh {
                    meta::generate_key(dot_pijul, None, KeyType::SSH)?
                }
                if signing {
                    meta::generate_key(dot_pijul, None, KeyType::Signing)?
                }
            } else {
                if ssh {
                    meta::generate_global_key(KeyType::SSH)?
                }
                if signing {
                    meta::generate_global_key(KeyType::Signing)?
                }
            }
        }
        Params::None => {}
    }
    Ok(())
}

pub fn explain(r: Result<(), Error>) {
    if let Err(Error(kind, _)) = r {
        if let ErrorKind::InARepository(p) = kind {
            writeln!(stderr(), "Repository {} already exists", p.display()).unwrap();
        } else {
            writeln!(stderr(), "error: {}", kind).unwrap();
        }
        exit(1)
    }
}

#[cfg(unix)]
use thrussh_keys::agent::client::AgentClient;
#[cfg(unix)]
use tokio_uds::UnixStream;

use tokio_core;
use thrussh_keys::key::KeyPair;

struct SshClient {
    exit_status: Option<u32>,
    key_pair: KeyPair,
    host: String,
    port: u16,
    #[cfg(unix)]
    agent: Option<AgentClient<UnixStream>>,
    #[cfg(windows)]
    agent: Option<()>,
}

impl SshClient {
    #[cfg(unix)]
    fn new(port: u16, host: &str, key_pair: KeyPair, h: &tokio_core::reactor::Handle) -> Self {
        let agent = if let Ok(path) = std::env::var("SSH_AUTH_SOCK") {
            UnixStream::connect(path, h).ok().map(thrussh_keys::agent::client::AgentClient::connect)
        } else {
            None
        };
        debug!("agent = {:?}", agent.is_some());
        SshClient {
            exit_status: None,
            host: host.to_string(),
            key_pair,
            port,
            agent,
        }
    }

    #[cfg(windows)]
    fn new(port: u16, host: &str, key_pair: KeyPair, _: &tokio_core::reactor::Handle) -> Self {
        SshClient {
            exit_status: None,
            host: host.to_string(),
            key_pair,
            port,
            agent: None,
        }
    }
}


impl client::Handler for SshClient {
    type Error = Error;
    type FutureBool = futures::Finished<(Self, bool), Self::Error>;
    type FutureUnit = futures::Finished<Self, Self::Error>;
    type SessionUnit = futures::Finished<(Self, client::Session), Self::Error>;
    type FutureSign = Box<futures::Future<Item = (Self, cryptovec::CryptoVec), Error = Self::Error>>;

    #[cfg(unix)]
    fn auth_publickey_sign(mut self, key: &thrussh_keys::key::PublicKey, mut to_sign: cryptovec::CryptoVec) -> Self::FutureSign {
        debug!("auth_publickey_sign");
        if let Some(agent) = self.agent.take() {
            use thrussh_keys::encoding::Encoding;
            debug!("using agent");
            Box::new(
                agent.sign_request(key, &to_sign)
                    .and_then(move |(client, sig)| {
                        debug!("sig = {:?}", sig);
                        if let Some(sig) = sig {
                            to_sign.extend_ssh_string(&sig[..]);
                        }
                        self.agent = Some(client);
                        futures::finished((self, to_sign))
                    })
                    .from_err()
            )
        } else {
            debug!("no agent");
            Box::new(futures::finished((self, to_sign)))
        }
    }

    fn check_server_key(self, server_public_key: &key::PublicKey) -> Self::FutureBool {
        let path = std::env::home_dir().unwrap().join(".ssh").join("known_hosts");
        match thrussh_keys::check_known_hosts_path(&self.host, self.port, &server_public_key, &path) {
            Ok(true) => futures::done(Ok((self, true))),
            Ok(false) => {
                if let Ok(false) = ask::ask_learn_ssh(&self.host,
                                                      self.port, "") {
                    futures::done(Ok((self, false)))
                } else {
                    thrussh_keys::learn_known_hosts_path(&self.host,
                                                         self.port,
                                                         &server_public_key,
                                                         &path)
                        .unwrap();
                    futures::done(Ok((self, true)))
                }
            }
            Err(e) => if let thrussh_keys::ErrorKind::KeyChanged(line) = *e.kind() {
                println!("Host key changed! Someone might be eavesdropping this communication, \
                          refusing to continue. Previous key found line {}",
                         line);
                futures::done(Ok((self, false)))
            } else {
                futures::done(Err(From::from(e)))
            }
        }
    }
    fn data(self, channel: ChannelId, _: Option<u32>, data: &[u8], mut session: client::Session)
            -> Self::SessionUnit {
        use thrussh_keys::PublicKeyBase64;
        let response = (
            self.key_pair.public_key_base64(),
            self.key_pair.sign_detached(data).unwrap()
        );
        session.data(channel, None, &bincode::serialize(&response, bincode::Infinite).unwrap());
        session.channel_eof(channel);
        futures::finished((self, session))
    }
    fn exit_status(mut self,
                   channel: thrussh::ChannelId,
                   exit_status: u32,
                   session: thrussh::client::Session)
                   -> Self::SessionUnit {
        debug!("exit_status received on channel {:?}: {:?}:", channel, exit_status);
        self.exit_status = Some(exit_status);
        debug!("self.exit_status = {:?}", self.exit_status);
        futures::finished((self, session))
    }
}
