// SPDX-License-Identifier: (Apache-2.0 OR MIT)

use std::ptr::NonNull;

pub struct PyDictIter {
    dict_ptr: *mut pyo3_ffi::PyObject,
    pos: isize,
}

impl PyDictIter {
    #[inline]
    pub fn from_pyobject(obj: *mut pyo3_ffi::PyObject) -> Self {
        unsafe {
            PyDictIter {
                dict_ptr: obj,
                pos: 0,
            }
        }
    }
}

impl Iterator for PyDictIter {
    type Item = (NonNull<pyo3_ffi::PyObject>, NonNull<pyo3_ffi::PyObject>);

    #[cfg(Py_3_13)]
    #[inline]
    fn next(&mut self) -> Option<Self::Item> {
        let mut key: *mut pyo3_ffi::PyObject = std::ptr::null_mut();
        let mut value: *mut pyo3_ffi::PyObject = std::ptr::null_mut();
        unsafe {
            if pyo3_ffi::PyDict_Next(self.dict_ptr, &mut self.pos, &mut key, &mut value) == 1 {
                Some((nonnull!(key), nonnull!(value)))
            } else {
                None
            }
        }
    }

    #[cfg(not(Py_3_13))]
    #[inline]
    fn next(&mut self) -> Option<Self::Item> {
        let mut key: *mut pyo3_ffi::PyObject = std::ptr::null_mut();
        let mut value: *mut pyo3_ffi::PyObject = std::ptr::null_mut();
        unsafe {
            if pyo3_ffi::_PyDict_Next(
                self.dict_ptr,
                &mut self.pos,
                &mut key,
                &mut value,
                std::ptr::null_mut(),
            ) == 1
            {
                Some((nonnull!(key), nonnull!(value)))
            } else {
                None
            }
        }
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        let len = ffi!(Py_SIZE(self.dict_ptr)) as usize;
        (len, Some(len))
    }
}

#[cfg(feature = "trusted_len")]
unsafe impl std::iter::TrustedLen for PyDictIter {}
