use nu_test_support::fs::Stub::FileWithContent;
use nu_test_support::playground::Playground;
use nu_test_support::{nu, nu_repl_code, pipeline};

#[test]
fn table_0() {
    let actual = nu!(r#"[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table"#);
    assert_eq!(
        actual.out,
        "╭───┬───┬───┬────────────────╮\
         │ # │ a │ b │       c        │\
         ├───┼───┼───┼────────────────┤\
         │ 0 │ 1 │ 2 │              3 │\
         │ 1 │ 4 │ 5 │ [list 3 items] │\
         ╰───┴───┴───┴────────────────╯"
    );
}

#[test]
fn table_collapse_0() {
    let actual = nu!(r#"[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table --collapse"#);
    assert_eq!(
        actual.out,
        "╭───┬───┬───╮\
         │ a │ b │ c │\
         ├───┼───┼───┤\
         │ 1 │ 2 │ 3 │\
         ├───┼───┼───┤\
         │ 4 │ 5 │ 1 │\
         │   │   ├───┤\
         │   │   │ 2 │\
         │   │   ├───┤\
         │   │   │ 3 │\
         ╰───┴───┴───╯"
    );
}

#[test]
fn table_collapse_basic() {
    let actual = nu!(nu_repl_code(&[
        "let-env config = { table: { mode: basic }};",
        "[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table --collapse"
    ]));
    assert_eq!(
        actual.out,
        "+---+---+---+\
         | a | b | c |\
         +---+---+---+\
         | 1 | 2 | 3 |\
         +---+---+---+\
         | 4 | 5 | 1 |\
         |   |   +---+\
         |   |   | 2 |\
         |   |   +---+\
         |   |   | 3 |\
         +---+---+---+"
    );
}

#[test]
fn table_collapse_heavy() {
    let actual = nu!(nu_repl_code(&[
        "let-env config = { table: { mode: heavy }};",
        "[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table --collapse"
    ]));
    assert_eq!(
        actual.out,
        "┏━━━┳━━━┳━━━┓\
         ┃ a ┃ b ┃ c ┃\
         ┣━━━╋━━━╋━━━┫\
         ┃ 1 ┃ 2 ┃ 3 ┃\
         ┣━━━╋━━━╋━━━┫\
         ┃ 4 ┃ 5 ┃ 1 ┃\
         ┃   ┃   ┣━━━┫\
         ┃   ┃   ┃ 2 ┃\
         ┃   ┃   ┣━━━┫\
         ┃   ┃   ┃ 3 ┃\
         ┗━━━┻━━━┻━━━┛"
    );
}

#[test]
fn table_collapse_compact() {
    let actual = nu!(nu_repl_code(&[
        "let-env config = { table: { mode: compact }};",
        "[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table --collapse"
    ]));
    assert_eq!(
        actual.out,
        "┌───┬───┬───┐\
         │ a │ b │ c │\
         ├───┼───┼───┤\
         │ 1 │ 2 │ 3 │\
         ├───┼───┼───┤\
         │ 4 │ 5 │ 1 │\
         │   │   ├───┤\
         │   │   │ 2 │\
         │   │   ├───┤\
         │   │   │ 3 │\
         └───┴───┴───┘"
    );
}

#[test]
fn table_collapse_compact_double() {
    let actual = nu!(nu_repl_code(&[
        "let-env config = { table: { mode: compact_double }};",
        "[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table --collapse"
    ]));
    assert_eq!(
        actual.out,
        "╔═══╦═══╦═══╗\
         ║ a ║ b ║ c ║\
         ╠═══╬═══╬═══╣\
         ║ 1 ║ 2 ║ 3 ║\
         ╠═══╬═══╬═══╣\
         ║ 4 ║ 5 ║ 1 ║\
         ║   ║   ╠═══╣\
         ║   ║   ║ 2 ║\
         ║   ║   ╠═══╣\
         ║   ║   ║ 3 ║\
         ╚═══╩═══╩═══╝"
    );
}

#[test]
fn table_collapse_compact_light() {
    let actual = nu!(nu_repl_code(&[
        "let-env config = { table: { mode: light }};",
        "[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table --collapse"
    ]));
    assert_eq!(
        actual.out,
        "┌───┬───┬───┐\
         │ a │ b │ c │\
         ├───┼───┼───┤\
         │ 1 │ 2 │ 3 │\
         ├───┼───┼───┤\
         │ 4 │ 5 │ 1 │\
         │   │   ├───┤\
         │   │   │ 2 │\
         │   │   ├───┤\
         │   │   │ 3 │\
         └───┴───┴───┘"
    );
}

#[test]
fn table_collapse_none() {
    let actual = nu!(nu_repl_code(&[
        "let-env config = { table: { mode: none }};",
        "[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table --collapse"
    ]));
    assert_eq!(
        actual.out,
        concat!(
            " a  b  c ",
            " 1  2  3 ",
            " 4  5  1 ",
            "       2 ",
            "       3 ",
        )
    );
}

#[test]
fn table_collapse_compact_reinforced() {
    let actual = nu!(nu_repl_code(&[
        "let-env config = { table: { mode: reinforced }};",
        "[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table --collapse"
    ]));
    assert_eq!(
        actual.out,
        "┏───┬───┬───┓\
         │ a │ b │ c │\
         ├───┼───┼───┤\
         │ 1 │ 2 │ 3 │\
         ├───┼───┼───┤\
         │ 4 │ 5 │ 1 │\
         │   │   ├───┤\
         │   │   │ 2 │\
         │   │   ├───┤\
         │   │   │ 3 │\
         ┗───┴───┴───┛"
    );
}

#[test]
fn table_collapse_compact_thin() {
    let actual = nu!(nu_repl_code(&[
        "let-env config = { table: { mode: thin }};",
        "[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table --collapse"
    ]));
    assert_eq!(
        actual.out,
        "┌───┬───┬───┐\
         │ a │ b │ c │\
         ├───┼───┼───┤\
         │ 1 │ 2 │ 3 │\
         ├───┼───┼───┤\
         │ 4 │ 5 │ 1 │\
         │   │   ├───┤\
         │   │   │ 2 │\
         │   │   ├───┤\
         │   │   │ 3 │\
         └───┴───┴───┘"
    );
}

#[test]
fn table_collapse_hearts() {
    let actual = nu!(nu_repl_code(&[
        "let-env config = { table: { mode: with_love }};",
        "[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table --collapse"
    ]));
    assert_eq!(
        actual.out,
        concat!(
            "❤❤❤❤❤❤❤❤❤❤❤❤❤",
            "❤ a ❤ b ❤ c ❤",
            "❤❤❤❤❤❤❤❤❤❤❤❤❤",
            "❤ 1 ❤ 2 ❤ 3 ❤",
            "❤❤❤❤❤❤❤❤❤❤❤❤❤",
            "❤ 4 ❤ 5 ❤ 1 ❤",
            "❤   ❤   ❤❤❤❤❤",
            "❤   ❤   ❤ 2 ❤",
            "❤   ❤   ❤❤❤❤❤",
            "❤   ❤   ❤ 3 ❤",
            "❤❤❤❤❤❤❤❤❤❤❤❤❤",
        )
    );
}

#[test]
fn table_collapse_does_wrapping_for_long_strings() {
    let actual = nu!(
        r#"[[a]; [11111111111111111111111111111111111111111111111111111111111111111111111111111111]] | table --collapse"#
    );
    assert_eq!(
        actual.out,
        "╭────────────────────────────────╮\
         │ a                              │\
         ├────────────────────────────────┤\
         │ 111111111111111109312339230430 │\
         │ 179149313814687359833671239329 │\
         │ 01313323321729744896.0000      │\
         ╰────────────────────────────────╯"
    );
}

#[test]
fn table_expand_0() {
    let actual = nu!(r#"[[a b, c]; [1 2 3] [4 5 [1 2 3]]] | table --expand"#);
    assert_eq!(
        actual.out,
        "╭───┬───┬───┬───────────╮\
         │ # │ a │ b │     c     │\
         ├───┼───┼───┼───────────┤\
         │ 0 │ 1 │ 2 │         3 │\
         │ 1 │ 4 │ 5 │ ╭───┬───╮ │\
         │   │   │   │ │ 0 │ 1 │ │\
         │   │   │   │ │ 1 │ 2 │ │\
         │   │   │   │ │ 2 │ 3 │ │\
         │   │   │   │ ╰───┴───╯ │\
         ╰───┴───┴───┴───────────╯"
    );
}

// I am not sure whether the test is platform dependent, cause we don't set a term_width on our own
#[test]
fn table_expand_exceed_overlap_0() {
    // no expand

    let actual = nu!(r#"[[a b, c]; [xxxxxxxxxxxxxxxxxxxxxx 2 3] [4 5 [1 2 3]]] | table --expand"#);

    assert_eq!(
        actual.out,
        "╭───┬────────────────────────┬───┬───────────╮\
         │ # │           a            │ b │     c     │\
         ├───┼────────────────────────┼───┼───────────┤\
         │ 0 │ xxxxxxxxxxxxxxxxxxxxxx │ 2 │         3 │\
         │ 1 │                      4 │ 5 │ ╭───┬───╮ │\
         │   │                        │   │ │ 0 │ 1 │ │\
         │   │                        │   │ │ 1 │ 2 │ │\
         │   │                        │   │ │ 2 │ 3 │ │\
         │   │                        │   │ ╰───┴───╯ │\
         ╰───┴────────────────────────┴───┴───────────╯",
    );

    // expand

    let actual = nu!(
        r#"[[a b, c]; [xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx 2 3] [4 5 [1 2 3]]] | table --expand"#
    );

    assert_eq!(
        actual.out,
        "╭──────┬───────────────────────────────────────────────────┬─────┬─────────────╮\
         │    # │                         a                         │  b  │      c      │\
         ├──────┼───────────────────────────────────────────────────┼─────┼─────────────┤\
         │    0 │ xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx    │   2 │           3 │\
         │    1 │                                                 4 │   5 │ ╭───┬───╮   │\
         │      │                                                   │     │ │ 0 │ 1 │   │\
         │      │                                                   │     │ │ 1 │ 2 │   │\
         │      │                                                   │     │ │ 2 │ 3 │   │\
         │      │                                                   │     │ ╰───┴───╯   │\
         ╰──────┴───────────────────────────────────────────────────┴─────┴─────────────╯"
    );
}

#[test]
fn table_expand_deep_0() {
    let actual = nu!(r#"[[a b, c]; [1 2 3] [4 5 [1 2 [1 2 3]]]] | table --expand --expand-deep=1"#);
    assert_eq!(
        actual.out,
        "╭───┬───┬───┬────────────────────────╮\
         │ # │ a │ b │           c            │\
         ├───┼───┼───┼────────────────────────┤\
         │ 0 │ 1 │ 2 │                      3 │\
         │ 1 │ 4 │ 5 │ ╭───┬────────────────╮ │\
         │   │   │   │ │ 0 │              1 │ │\
         │   │   │   │ │ 1 │              2 │ │\
         │   │   │   │ │ 2 │ [list 3 items] │ │\
         │   │   │   │ ╰───┴────────────────╯ │\
         ╰───┴───┴───┴────────────────────────╯"
    );
}

#[test]
fn table_expand_deep_1() {
    let actual = nu!(r#"[[a b, c]; [1 2 3] [4 5 [1 2 [1 2 3]]]] | table --expand --expand-deep=0"#);
    assert_eq!(
        actual.out,
        "╭───┬───┬───┬────────────────╮\
         │ # │ a │ b │       c        │\
         ├───┼───┼───┼────────────────┤\
         │ 0 │ 1 │ 2 │              3 │\
         │ 1 │ 4 │ 5 │ [list 3 items] │\
         ╰───┴───┴───┴────────────────╯"
    );
}

#[test]
fn table_expand_flatten_0() {
    let actual = nu!(r#"[[a b, c]; [1 2 3] [4 5 [1 2 [1 1 1]]]] | table --expand --flatten "#);
    assert_eq!(
        actual.out,
        "╭───┬───┬───┬───────────────╮\
         │ # │ a │ b │       c       │\
         ├───┼───┼───┼───────────────┤\
         │ 0 │ 1 │ 2 │             3 │\
         │ 1 │ 4 │ 5 │ ╭───┬───────╮ │\
         │   │   │   │ │ 0 │     1 │ │\
         │   │   │   │ │ 1 │     2 │ │\
         │   │   │   │ │ 2 │ 1 1 1 │ │\
         │   │   │   │ ╰───┴───────╯ │\
         ╰───┴───┴───┴───────────────╯"
    );
}

#[test]
fn table_expand_flatten_1() {
    let actual = nu!(
        r#"[[a b, c]; [1 2 3] [4 5 [1 2 [1 1 1]]]] | table --expand --flatten --flatten-separator=,"#
    );
    assert_eq!(
        actual.out,
        "╭───┬───┬───┬───────────────╮\
         │ # │ a │ b │       c       │\
         ├───┼───┼───┼───────────────┤\
         │ 0 │ 1 │ 2 │             3 │\
         │ 1 │ 4 │ 5 │ ╭───┬───────╮ │\
         │   │   │   │ │ 0 │     1 │ │\
         │   │   │   │ │ 1 │     2 │ │\
         │   │   │   │ │ 2 │ 1,1,1 │ │\
         │   │   │   │ ╰───┴───────╯ │\
         ╰───┴───┴───┴───────────────╯"
    );
}

#[test]
fn table_expand_flatten_and_deep_1() {
    let actual = nu!(
        r#"[[a b, c]; [1 2 3] [4 5 [1 2 [1 [1 1 1] 1]]]] | table --expand --expand-deep=2 --flatten --flatten-separator=,"#
    );

    assert_eq!(
        actual.out,
        "╭───┬───┬───┬────────────────────────────────╮\
         │ # │ a │ b │               c                │\
         ├───┼───┼───┼────────────────────────────────┤\
         │ 0 │ 1 │ 2 │                              3 │\
         │ 1 │ 4 │ 5 │ ╭───┬────────────────────────╮ │\
         │   │   │   │ │ 0 │                      1 │ │\
         │   │   │   │ │ 1 │                      2 │ │\
         │   │   │   │ │ 2 │ ╭───┬────────────────╮ │ │\
         │   │   │   │ │   │ │ 0 │              1 │ │ │\
         │   │   │   │ │   │ │ 1 │ [list 3 items] │ │ │\
         │   │   │   │ │   │ │ 2 │              1 │ │ │\
         │   │   │   │ │   │ ╰───┴────────────────╯ │ │\
         │   │   │   │ ╰───┴────────────────────────╯ │\
         ╰───┴───┴───┴────────────────────────────────╯"
    );
}

#[test]
fn table_expand_record_0() {
    let actual = nu!(r#"[{c: {d: 1}}] | table --expand"#);

    assert_eq!(
        actual.out,
        "╭───┬───────────╮\
         │ # │     c     │\
         ├───┼───────────┤\
         │ 0 │ ╭───┬───╮ │\
         │   │ │ d │ 1 │ │\
         │   │ ╰───┴───╯ │\
         ╰───┴───────────╯"
    );
}

#[test]
fn table_expand_record_1() {
    let actual =
        nu!(r#"[[a b, c]; [1 2 3] [4 5 [1 2 {a: 123, b: 234, c: 345}]]] | table --expand"#);

    assert_eq!(
        actual.out,
        "╭───┬───┬───┬─────────────────────╮\
         │ # │ a │ b │          c          │\
         ├───┼───┼───┼─────────────────────┤\
         │ 0 │ 1 │ 2 │                   3 │\
         │ 1 │ 4 │ 5 │ ╭───┬─────────────╮ │\
         │   │   │   │ │ 0 │           1 │ │\
         │   │   │   │ │ 1 │           2 │ │\
         │   │   │   │ │ 2 │ ╭───┬─────╮ │ │\
         │   │   │   │ │   │ │ a │ 123 │ │ │\
         │   │   │   │ │   │ │ b │ 234 │ │ │\
         │   │   │   │ │   │ │ c │ 345 │ │ │\
         │   │   │   │ │   │ ╰───┴─────╯ │ │\
         │   │   │   │ ╰───┴─────────────╯ │\
         ╰───┴───┴───┴─────────────────────╯"
    );
}

#[test]
fn table_expand_record_2() {
    let structure = "{\
        field1: [ a, b, c ],\
        field2: [ 123, 234, 345 ],\
        field3: [ [ head1, head2, head3 ]; [ 1 2 3 ] [ 79 79 79 ] [ { f1: 'a string', f2: 1000 }, 1, 2 ] ],\
        field4: { f1: 1, f2: 3, f3: { f1: f1, f2: f2, f3: f3 } }\
    }";
    let actual = nu!(format!("{structure} | table --expand"));

    assert_eq!(
        actual.out,
        "╭────────┬───────────────────────────────────────────╮\
         │        │ ╭───┬───╮                                 │\
         │ field1 │ │ 0 │ a │                                 │\
         │        │ │ 1 │ b │                                 │\
         │        │ │ 2 │ c │                                 │\
         │        │ ╰───┴───╯                                 │\
         │        │ ╭───┬─────╮                               │\
         │ field2 │ │ 0 │ 123 │                               │\
         │        │ │ 1 │ 234 │                               │\
         │        │ │ 2 │ 345 │                               │\
         │        │ ╰───┴─────╯                               │\
         │        │ ╭───┬───────────────────┬───────┬───────╮ │\
         │ field3 │ │ # │       head1       │ head2 │ head3 │ │\
         │        │ ├───┼───────────────────┼───────┼───────┤ │\
         │        │ │ 0 │                 1 │     2 │     3 │ │\
         │        │ │ 1 │                79 │    79 │    79 │ │\
         │        │ │ 2 │ ╭────┬──────────╮ │     1 │     2 │ │\
         │        │ │   │ │ f1 │ a string │ │       │       │ │\
         │        │ │   │ │ f2 │ 1000     │ │       │       │ │\
         │        │ │   │ ╰────┴──────────╯ │       │       │ │\
         │        │ ╰───┴───────────────────┴───────┴───────╯ │\
         │        │ ╭────┬─────────────╮                      │\
         │ field4 │ │ f1 │ 1           │                      │\
         │        │ │ f2 │ 3           │                      │\
         │        │ │    │ ╭────┬────╮ │                      │\
         │        │ │ f3 │ │ f1 │ f1 │ │                      │\
         │        │ │    │ │ f2 │ f2 │ │                      │\
         │        │ │    │ │ f3 │ f3 │ │                      │\
         │        │ │    │ ╰────┴────╯ │                      │\
         │        │ ╰────┴─────────────╯                      │\
         ╰────────┴───────────────────────────────────────────╯"
    );
}

#[test]
#[cfg(not(windows))]
fn external_with_too_much_stdout_should_not_hang_nu() {
    use nu_test_support::fs::Stub::FileWithContent;
    use nu_test_support::pipeline;
    use nu_test_support::playground::Playground;
    Playground::setup("external with too much stdout", |dirs, sandbox| {
        let bytes: usize = 81920;
        let mut large_file_body = String::with_capacity(bytes);
        for _ in 0..bytes {
            large_file_body.push('a');
        }
        sandbox.with_files(vec![FileWithContent("a_large_file.txt", &large_file_body)]);

        let actual = nu!(
            cwd: dirs.test(), pipeline(
            r#"
                cat a_large_file.txt | table
            "#
        ));

        assert_eq!(actual.out, large_file_body);
    })
}

#[test]
fn table_pagging_row_offset_overlap() {
    let actual = nu!(r#"0..1000"#);
    assert_eq!(actual.out, "╭─────┬─────╮│   0 │   0 ││   1 │   1 ││   2 │   2 ││   3 │   3 ││   4 │   4 ││   5 │   5 ││   6 │   6 ││   7 │   7 ││   8 │   8 ││   9 │   9 ││  10 │  10 ││  11 │  11 ││  12 │  12 ││  13 │  13 ││  14 │  14 ││  15 │  15 ││  16 │  16 ││  17 │  17 ││  18 │  18 ││  19 │  19 ││  20 │  20 ││  21 │  21 ││  22 │  22 ││  23 │  23 ││  24 │  24 ││  25 │  25 ││  26 │  26 ││  27 │  27 ││  28 │  28 ││  29 │  29 ││  30 │  30 ││  31 │  31 ││  32 │  32 ││  33 │  33 ││  34 │  34 ││  35 │  35 ││  36 │  36 ││  37 │  37 ││  38 │  38 ││  39 │  39 ││  40 │  40 ││  41 │  41 ││  42 │  42 ││  43 │  43 ││  44 │  44 ││  45 │  45 ││  46 │  46 ││  47 │  47 ││  48 │  48 ││  49 │  49 ││  50 │  50 ││  51 │  51 ││  52 │  52 ││  53 │  53 ││  54 │  54 ││  55 │  55 ││  56 │  56 ││  57 │  57 ││  58 │  58 ││  59 │  59 ││  60 │  60 ││  61 │  61 ││  62 │  62 ││  63 │  63 ││  64 │  64 ││  65 │  65 ││  66 │  66 ││  67 │  67 ││  68 │  68 ││  69 │  69 ││  70 │  70 ││  71 │  71 ││  72 │  72 ││  73 │  73 ││  74 │  74 ││  75 │  75 ││  76 │  76 ││  77 │  77 ││  78 │  78 ││  79 │  79 ││  80 │  80 ││  81 │  81 ││  82 │  82 ││  83 │  83 ││  84 │  84 ││  85 │  85 ││  86 │  86 ││  87 │  87 ││  88 │  88 ││  89 │  89 ││  90 │  90 ││  91 │  91 ││  92 │  92 ││  93 │  93 ││  94 │  94 ││  95 │  95 ││  96 │  96 ││  97 │  97 ││  98 │  98 ││  99 │  99 ││ 100 │ 100 ││ 101 │ 101 ││ 102 │ 102 ││ 103 │ 103 ││ 104 │ 104 ││ 105 │ 105 ││ 106 │ 106 ││ 107 │ 107 ││ 108 │ 108 ││ 109 │ 109 ││ 110 │ 110 ││ 111 │ 111 ││ 112 │ 112 ││ 113 │ 113 ││ 114 │ 114 ││ 115 │ 115 ││ 116 │ 116 ││ 117 │ 117 ││ 118 │ 118 ││ 119 │ 119 ││ 120 │ 120 ││ 121 │ 121 ││ 122 │ 122 ││ 123 │ 123 ││ 124 │ 124 ││ 125 │ 125 ││ 126 │ 126 ││ 127 │ 127 ││ 128 │ 128 ││ 129 │ 129 ││ 130 │ 130 ││ 131 │ 131 ││ 132 │ 132 ││ 133 │ 133 ││ 134 │ 134 ││ 135 │ 135 ││ 136 │ 136 ││ 137 │ 137 ││ 138 │ 138 ││ 139 │ 139 ││ 140 │ 140 ││ 141 │ 141 ││ 142 │ 142 ││ 143 │ 143 ││ 144 │ 144 ││ 145 │ 145 ││ 146 │ 146 ││ 147 │ 147 ││ 148 │ 148 ││ 149 │ 149 ││ 150 │ 150 ││ 151 │ 151 ││ 152 │ 152 ││ 153 │ 153 ││ 154 │ 154 ││ 155 │ 155 ││ 156 │ 156 ││ 157 │ 157 ││ 158 │ 158 ││ 159 │ 159 ││ 160 │ 160 ││ 161 │ 161 ││ 162 │ 162 ││ 163 │ 163 ││ 164 │ 164 ││ 165 │ 165 ││ 166 │ 166 ││ 167 │ 167 ││ 168 │ 168 ││ 169 │ 169 ││ 170 │ 170 ││ 171 │ 171 ││ 172 │ 172 ││ 173 │ 173 ││ 174 │ 174 ││ 175 │ 175 ││ 176 │ 176 ││ 177 │ 177 ││ 178 │ 178 ││ 179 │ 179 ││ 180 │ 180 ││ 181 │ 181 ││ 182 │ 182 ││ 183 │ 183 ││ 184 │ 184 ││ 185 │ 185 ││ 186 │ 186 ││ 187 │ 187 ││ 188 │ 188 ││ 189 │ 189 ││ 190 │ 190 ││ 191 │ 191 ││ 192 │ 192 ││ 193 │ 193 ││ 194 │ 194 ││ 195 │ 195 ││ 196 │ 196 ││ 197 │ 197 ││ 198 │ 198 ││ 199 │ 199 ││ 200 │ 200 ││ 201 │ 201 ││ 202 │ 202 ││ 203 │ 203 ││ 204 │ 204 ││ 205 │ 205 ││ 206 │ 206 ││ 207 │ 207 ││ 208 │ 208 ││ 209 │ 209 ││ 210 │ 210 ││ 211 │ 211 ││ 212 │ 212 ││ 213 │ 213 ││ 214 │ 214 ││ 215 │ 215 ││ 216 │ 216 ││ 217 │ 217 ││ 218 │ 218 ││ 219 │ 219 ││ 220 │ 220 ││ 221 │ 221 ││ 222 │ 222 ││ 223 │ 223 ││ 224 │ 224 ││ 225 │ 225 ││ 226 │ 226 ││ 227 │ 227 ││ 228 │ 228 ││ 229 │ 229 ││ 230 │ 230 ││ 231 │ 231 ││ 232 │ 232 ││ 233 │ 233 ││ 234 │ 234 ││ 235 │ 235 ││ 236 │ 236 ││ 237 │ 237 ││ 238 │ 238 ││ 239 │ 239 ││ 240 │ 240 ││ 241 │ 241 ││ 242 │ 242 ││ 243 │ 243 ││ 244 │ 244 ││ 245 │ 245 ││ 246 │ 246 ││ 247 │ 247 ││ 248 │ 248 ││ 249 │ 249 ││ 250 │ 250 ││ 251 │ 251 ││ 252 │ 252 ││ 253 │ 253 ││ 254 │ 254 ││ 255 │ 255 ││ 256 │ 256 ││ 257 │ 257 ││ 258 │ 258 ││ 259 │ 259 ││ 260 │ 260 ││ 261 │ 261 ││ 262 │ 262 ││ 263 │ 263 ││ 264 │ 264 ││ 265 │ 265 ││ 266 │ 266 ││ 267 │ 267 ││ 268 │ 268 ││ 269 │ 269 ││ 270 │ 270 ││ 271 │ 271 ││ 272 │ 272 ││ 273 │ 273 ││ 274 │ 274 ││ 275 │ 275 ││ 276 │ 276 ││ 277 │ 277 ││ 278 │ 278 ││ 279 │ 279 ││ 280 │ 280 ││ 281 │ 281 ││ 282 │ 282 ││ 283 │ 283 ││ 284 │ 284 ││ 285 │ 285 ││ 286 │ 286 ││ 287 │ 287 ││ 288 │ 288 ││ 289 │ 289 ││ 290 │ 290 ││ 291 │ 291 ││ 292 │ 292 ││ 293 │ 293 ││ 294 │ 294 ││ 295 │ 295 ││ 296 │ 296 ││ 297 │ 297 ││ 298 │ 298 ││ 299 │ 299 ││ 300 │ 300 ││ 301 │ 301 ││ 302 │ 302 ││ 303 │ 303 ││ 304 │ 304 ││ 305 │ 305 ││ 306 │ 306 ││ 307 │ 307 ││ 308 │ 308 ││ 309 │ 309 ││ 310 │ 310 ││ 311 │ 311 ││ 312 │ 312 ││ 313 │ 313 ││ 314 │ 314 ││ 315 │ 315 ││ 316 │ 316 ││ 317 │ 317 ││ 318 │ 318 ││ 319 │ 319 ││ 320 │ 320 ││ 321 │ 321 ││ 322 │ 322 ││ 323 │ 323 ││ 324 │ 324 ││ 325 │ 325 ││ 326 │ 326 ││ 327 │ 327 ││ 328 │ 328 ││ 329 │ 329 ││ 330 │ 330 ││ 331 │ 331 ││ 332 │ 332 ││ 333 │ 333 ││ 334 │ 334 ││ 335 │ 335 ││ 336 │ 336 ││ 337 │ 337 ││ 338 │ 338 ││ 339 │ 339 ││ 340 │ 340 ││ 341 │ 341 ││ 342 │ 342 ││ 343 │ 343 ││ 344 │ 344 ││ 345 │ 345 ││ 346 │ 346 ││ 347 │ 347 ││ 348 │ 348 ││ 349 │ 349 ││ 350 │ 350 ││ 351 │ 351 ││ 352 │ 352 ││ 353 │ 353 ││ 354 │ 354 ││ 355 │ 355 ││ 356 │ 356 ││ 357 │ 357 ││ 358 │ 358 ││ 359 │ 359 ││ 360 │ 360 ││ 361 │ 361 ││ 362 │ 362 ││ 363 │ 363 ││ 364 │ 364 ││ 365 │ 365 ││ 366 │ 366 ││ 367 │ 367 ││ 368 │ 368 ││ 369 │ 369 ││ 370 │ 370 ││ 371 │ 371 ││ 372 │ 372 ││ 373 │ 373 ││ 374 │ 374 ││ 375 │ 375 ││ 376 │ 376 ││ 377 │ 377 ││ 378 │ 378 ││ 379 │ 379 ││ 380 │ 380 ││ 381 │ 381 ││ 382 │ 382 ││ 383 │ 383 ││ 384 │ 384 ││ 385 │ 385 ││ 386 │ 386 ││ 387 │ 387 ││ 388 │ 388 ││ 389 │ 389 ││ 390 │ 390 ││ 391 │ 391 ││ 392 │ 392 ││ 393 │ 393 ││ 394 │ 394 ││ 395 │ 395 ││ 396 │ 396 ││ 397 │ 397 ││ 398 │ 398 ││ 399 │ 399 ││ 400 │ 400 ││ 401 │ 401 ││ 402 │ 402 ││ 403 │ 403 ││ 404 │ 404 ││ 405 │ 405 ││ 406 │ 406 ││ 407 │ 407 ││ 408 │ 408 ││ 409 │ 409 ││ 410 │ 410 ││ 411 │ 411 ││ 412 │ 412 ││ 413 │ 413 ││ 414 │ 414 ││ 415 │ 415 ││ 416 │ 416 ││ 417 │ 417 ││ 418 │ 418 ││ 419 │ 419 ││ 420 │ 420 ││ 421 │ 421 ││ 422 │ 422 ││ 423 │ 423 ││ 424 │ 424 ││ 425 │ 425 ││ 426 │ 426 ││ 427 │ 427 ││ 428 │ 428 ││ 429 │ 429 ││ 430 │ 430 ││ 431 │ 431 ││ 432 │ 432 ││ 433 │ 433 ││ 434 │ 434 ││ 435 │ 435 ││ 436 │ 436 ││ 437 │ 437 ││ 438 │ 438 ││ 439 │ 439 ││ 440 │ 440 ││ 441 │ 441 ││ 442 │ 442 ││ 443 │ 443 ││ 444 │ 444 ││ 445 │ 445 ││ 446 │ 446 ││ 447 │ 447 ││ 448 │ 448 ││ 449 │ 449 ││ 450 │ 450 ││ 451 │ 451 ││ 452 │ 452 ││ 453 │ 453 ││ 454 │ 454 ││ 455 │ 455 ││ 456 │ 456 ││ 457 │ 457 ││ 458 │ 458 ││ 459 │ 459 ││ 460 │ 460 ││ 461 │ 461 ││ 462 │ 462 ││ 463 │ 463 ││ 464 │ 464 ││ 465 │ 465 ││ 466 │ 466 ││ 467 │ 467 ││ 468 │ 468 ││ 469 │ 469 ││ 470 │ 470 ││ 471 │ 471 ││ 472 │ 472 ││ 473 │ 473 ││ 474 │ 474 ││ 475 │ 475 ││ 476 │ 476 ││ 477 │ 477 ││ 478 │ 478 ││ 479 │ 479 ││ 480 │ 480 ││ 481 │ 481 ││ 482 │ 482 ││ 483 │ 483 ││ 484 │ 484 ││ 485 │ 485 ││ 486 │ 486 ││ 487 │ 487 ││ 488 │ 488 ││ 489 │ 489 ││ 490 │ 490 ││ 491 │ 491 ││ 492 │ 492 ││ 493 │ 493 ││ 494 │ 494 ││ 495 │ 495 ││ 496 │ 496 ││ 497 │ 497 ││ 498 │ 498 ││ 499 │ 499 ││ 500 │ 500 ││ 501 │ 501 ││ 502 │ 502 ││ 503 │ 503 ││ 504 │ 504 ││ 505 │ 505 ││ 506 │ 506 ││ 507 │ 507 ││ 508 │ 508 ││ 509 │ 509 ││ 510 │ 510 ││ 511 │ 511 ││ 512 │ 512 ││ 513 │ 513 ││ 514 │ 514 ││ 515 │ 515 ││ 516 │ 516 ││ 517 │ 517 ││ 518 │ 518 ││ 519 │ 519 ││ 520 │ 520 ││ 521 │ 521 ││ 522 │ 522 ││ 523 │ 523 ││ 524 │ 524 ││ 525 │ 525 ││ 526 │ 526 ││ 527 │ 527 ││ 528 │ 528 ││ 529 │ 529 ││ 530 │ 530 ││ 531 │ 531 ││ 532 │ 532 ││ 533 │ 533 ││ 534 │ 534 ││ 535 │ 535 ││ 536 │ 536 ││ 537 │ 537 ││ 538 │ 538 ││ 539 │ 539 ││ 540 │ 540 ││ 541 │ 541 ││ 542 │ 542 ││ 543 │ 543 ││ 544 │ 544 ││ 545 │ 545 ││ 546 │ 546 ││ 547 │ 547 ││ 548 │ 548 ││ 549 │ 549 ││ 550 │ 550 ││ 551 │ 551 ││ 552 │ 552 ││ 553 │ 553 ││ 554 │ 554 ││ 555 │ 555 ││ 556 │ 556 ││ 557 │ 557 ││ 558 │ 558 ││ 559 │ 559 ││ 560 │ 560 ││ 561 │ 561 ││ 562 │ 562 ││ 563 │ 563 ││ 564 │ 564 ││ 565 │ 565 ││ 566 │ 566 ││ 567 │ 567 ││ 568 │ 568 ││ 569 │ 569 ││ 570 │ 570 ││ 571 │ 571 ││ 572 │ 572 ││ 573 │ 573 ││ 574 │ 574 ││ 575 │ 575 ││ 576 │ 576 ││ 577 │ 577 ││ 578 │ 578 ││ 579 │ 579 ││ 580 │ 580 ││ 581 │ 581 ││ 582 │ 582 ││ 583 │ 583 ││ 584 │ 584 ││ 585 │ 585 ││ 586 │ 586 ││ 587 │ 587 ││ 588 │ 588 ││ 589 │ 589 ││ 590 │ 590 ││ 591 │ 591 ││ 592 │ 592 ││ 593 │ 593 ││ 594 │ 594 ││ 595 │ 595 ││ 596 │ 596 ││ 597 │ 597 ││ 598 │ 598 ││ 599 │ 599 ││ 600 │ 600 ││ 601 │ 601 ││ 602 │ 602 ││ 603 │ 603 ││ 604 │ 604 ││ 605 │ 605 ││ 606 │ 606 ││ 607 │ 607 ││ 608 │ 608 ││ 609 │ 609 ││ 610 │ 610 ││ 611 │ 611 ││ 612 │ 612 ││ 613 │ 613 ││ 614 │ 614 ││ 615 │ 615 ││ 616 │ 616 ││ 617 │ 617 ││ 618 │ 618 ││ 619 │ 619 ││ 620 │ 620 ││ 621 │ 621 ││ 622 │ 622 ││ 623 │ 623 ││ 624 │ 624 ││ 625 │ 625 ││ 626 │ 626 ││ 627 │ 627 ││ 628 │ 628 ││ 629 │ 629 ││ 630 │ 630 ││ 631 │ 631 ││ 632 │ 632 ││ 633 │ 633 ││ 634 │ 634 ││ 635 │ 635 ││ 636 │ 636 ││ 637 │ 637 ││ 638 │ 638 ││ 639 │ 639 ││ 640 │ 640 ││ 641 │ 641 ││ 642 │ 642 ││ 643 │ 643 ││ 644 │ 644 ││ 645 │ 645 ││ 646 │ 646 ││ 647 │ 647 ││ 648 │ 648 ││ 649 │ 649 ││ 650 │ 650 ││ 651 │ 651 ││ 652 │ 652 ││ 653 │ 653 ││ 654 │ 654 ││ 655 │ 655 ││ 656 │ 656 ││ 657 │ 657 ││ 658 │ 658 ││ 659 │ 659 ││ 660 │ 660 ││ 661 │ 661 ││ 662 │ 662 ││ 663 │ 663 ││ 664 │ 664 ││ 665 │ 665 ││ 666 │ 666 ││ 667 │ 667 ││ 668 │ 668 ││ 669 │ 669 ││ 670 │ 670 ││ 671 │ 671 ││ 672 │ 672 ││ 673 │ 673 ││ 674 │ 674 ││ 675 │ 675 ││ 676 │ 676 ││ 677 │ 677 ││ 678 │ 678 ││ 679 │ 679 ││ 680 │ 680 ││ 681 │ 681 ││ 682 │ 682 ││ 683 │ 683 ││ 684 │ 684 ││ 685 │ 685 ││ 686 │ 686 ││ 687 │ 687 ││ 688 │ 688 ││ 689 │ 689 ││ 690 │ 690 ││ 691 │ 691 ││ 692 │ 692 ││ 693 │ 693 ││ 694 │ 694 ││ 695 │ 695 ││ 696 │ 696 ││ 697 │ 697 ││ 698 │ 698 ││ 699 │ 699 ││ 700 │ 700 ││ 701 │ 701 ││ 702 │ 702 ││ 703 │ 703 ││ 704 │ 704 ││ 705 │ 705 ││ 706 │ 706 ││ 707 │ 707 ││ 708 │ 708 ││ 709 │ 709 ││ 710 │ 710 ││ 711 │ 711 ││ 712 │ 712 ││ 713 │ 713 ││ 714 │ 714 ││ 715 │ 715 ││ 716 │ 716 ││ 717 │ 717 ││ 718 │ 718 ││ 719 │ 719 ││ 720 │ 720 ││ 721 │ 721 ││ 722 │ 722 ││ 723 │ 723 ││ 724 │ 724 ││ 725 │ 725 ││ 726 │ 726 ││ 727 │ 727 ││ 728 │ 728 ││ 729 │ 729 ││ 730 │ 730 ││ 731 │ 731 ││ 732 │ 732 ││ 733 │ 733 ││ 734 │ 734 ││ 735 │ 735 ││ 736 │ 736 ││ 737 │ 737 ││ 738 │ 738 ││ 739 │ 739 ││ 740 │ 740 ││ 741 │ 741 ││ 742 │ 742 ││ 743 │ 743 ││ 744 │ 744 ││ 745 │ 745 ││ 746 │ 746 ││ 747 │ 747 ││ 748 │ 748 ││ 749 │ 749 ││ 750 │ 750 ││ 751 │ 751 ││ 752 │ 752 ││ 753 │ 753 ││ 754 │ 754 ││ 755 │ 755 ││ 756 │ 756 ││ 757 │ 757 ││ 758 │ 758 ││ 759 │ 759 ││ 760 │ 760 ││ 761 │ 761 ││ 762 │ 762 ││ 763 │ 763 ││ 764 │ 764 ││ 765 │ 765 ││ 766 │ 766 ││ 767 │ 767 ││ 768 │ 768 ││ 769 │ 769 ││ 770 │ 770 ││ 771 │ 771 ││ 772 │ 772 ││ 773 │ 773 ││ 774 │ 774 ││ 775 │ 775 ││ 776 │ 776 ││ 777 │ 777 ││ 778 │ 778 ││ 779 │ 779 ││ 780 │ 780 ││ 781 │ 781 ││ 782 │ 782 ││ 783 │ 783 ││ 784 │ 784 ││ 785 │ 785 ││ 786 │ 786 ││ 787 │ 787 ││ 788 │ 788 ││ 789 │ 789 ││ 790 │ 790 ││ 791 │ 791 ││ 792 │ 792 ││ 793 │ 793 ││ 794 │ 794 ││ 795 │ 795 ││ 796 │ 796 ││ 797 │ 797 ││ 798 │ 798 ││ 799 │ 799 ││ 800 │ 800 ││ 801 │ 801 ││ 802 │ 802 ││ 803 │ 803 ││ 804 │ 804 ││ 805 │ 805 ││ 806 │ 806 ││ 807 │ 807 ││ 808 │ 808 ││ 809 │ 809 ││ 810 │ 810 ││ 811 │ 811 ││ 812 │ 812 ││ 813 │ 813 ││ 814 │ 814 ││ 815 │ 815 ││ 816 │ 816 ││ 817 │ 817 ││ 818 │ 818 ││ 819 │ 819 ││ 820 │ 820 ││ 821 │ 821 ││ 822 │ 822 ││ 823 │ 823 ││ 824 │ 824 ││ 825 │ 825 ││ 826 │ 826 ││ 827 │ 827 ││ 828 │ 828 ││ 829 │ 829 ││ 830 │ 830 ││ 831 │ 831 ││ 832 │ 832 ││ 833 │ 833 ││ 834 │ 834 ││ 835 │ 835 ││ 836 │ 836 ││ 837 │ 837 ││ 838 │ 838 ││ 839 │ 839 ││ 840 │ 840 ││ 841 │ 841 ││ 842 │ 842 ││ 843 │ 843 ││ 844 │ 844 ││ 845 │ 845 ││ 846 │ 846 ││ 847 │ 847 ││ 848 │ 848 ││ 849 │ 849 ││ 850 │ 850 ││ 851 │ 851 ││ 852 │ 852 ││ 853 │ 853 ││ 854 │ 854 ││ 855 │ 855 ││ 856 │ 856 ││ 857 │ 857 ││ 858 │ 858 ││ 859 │ 859 ││ 860 │ 860 ││ 861 │ 861 ││ 862 │ 862 ││ 863 │ 863 ││ 864 │ 864 ││ 865 │ 865 ││ 866 │ 866 ││ 867 │ 867 ││ 868 │ 868 ││ 869 │ 869 ││ 870 │ 870 ││ 871 │ 871 ││ 872 │ 872 ││ 873 │ 873 ││ 874 │ 874 ││ 875 │ 875 ││ 876 │ 876 ││ 877 │ 877 ││ 878 │ 878 ││ 879 │ 879 ││ 880 │ 880 ││ 881 │ 881 ││ 882 │ 882 ││ 883 │ 883 ││ 884 │ 884 ││ 885 │ 885 ││ 886 │ 886 ││ 887 │ 887 ││ 888 │ 888 ││ 889 │ 889 ││ 890 │ 890 ││ 891 │ 891 ││ 892 │ 892 ││ 893 │ 893 ││ 894 │ 894 ││ 895 │ 895 ││ 896 │ 896 ││ 897 │ 897 ││ 898 │ 898 ││ 899 │ 899 ││ 900 │ 900 ││ 901 │ 901 ││ 902 │ 902 ││ 903 │ 903 ││ 904 │ 904 ││ 905 │ 905 ││ 906 │ 906 ││ 907 │ 907 ││ 908 │ 908 ││ 909 │ 909 ││ 910 │ 910 ││ 911 │ 911 ││ 912 │ 912 ││ 913 │ 913 ││ 914 │ 914 ││ 915 │ 915 ││ 916 │ 916 ││ 917 │ 917 ││ 918 │ 918 ││ 919 │ 919 ││ 920 │ 920 ││ 921 │ 921 ││ 922 │ 922 ││ 923 │ 923 ││ 924 │ 924 ││ 925 │ 925 ││ 926 │ 926 ││ 927 │ 927 ││ 928 │ 928 ││ 929 │ 929 ││ 930 │ 930 ││ 931 │ 931 ││ 932 │ 932 ││ 933 │ 933 ││ 934 │ 934 ││ 935 │ 935 ││ 936 │ 936 ││ 937 │ 937 ││ 938 │ 938 ││ 939 │ 939 ││ 940 │ 940 ││ 941 │ 941 ││ 942 │ 942 ││ 943 │ 943 ││ 944 │ 944 ││ 945 │ 945 ││ 946 │ 946 ││ 947 │ 947 ││ 948 │ 948 ││ 949 │ 949 ││ 950 │ 950 ││ 951 │ 951 ││ 952 │ 952 ││ 953 │ 953 ││ 954 │ 954 ││ 955 │ 955 ││ 956 │ 956 ││ 957 │ 957 ││ 958 │ 958 ││ 959 │ 959 ││ 960 │ 960 ││ 961 │ 961 ││ 962 │ 962 ││ 963 │ 963 ││ 964 │ 964 ││ 965 │ 965 ││ 966 │ 966 ││ 967 │ 967 ││ 968 │ 968 ││ 969 │ 969 ││ 970 │ 970 ││ 971 │ 971 ││ 972 │ 972 ││ 973 │ 973 ││ 974 │ 974 ││ 975 │ 975 ││ 976 │ 976 ││ 977 │ 977 ││ 978 │ 978 ││ 979 │ 979 ││ 980 │ 980 ││ 981 │ 981 ││ 982 │ 982 ││ 983 │ 983 ││ 984 │ 984 ││ 985 │ 985 ││ 986 │ 986 ││ 987 │ 987 ││ 988 │ 988 ││ 989 │ 989 ││ 990 │ 990 ││ 991 │ 991 ││ 992 │ 992 ││ 993 │ 993 ││ 994 │ 994 ││ 995 │ 995 ││ 996 │ 996 ││ 997 │ 997 ││ 998 │ 998 ││ 999 │ 999 │╰─────┴─────╯╭──────┬──────╮│ 1000 │ 1000 │╰──────┴──────╯");
}

#[test]
fn table_index_0() {
    let actual = nu!(r#"[1 3 1 3 2 1 1]"#);
    assert_eq!(
        actual.out,
        "╭───┬───╮│ 0 │ 1 ││ 1 │ 3 ││ 2 │ 1 ││ 3 │ 3 ││ 4 │ 2 ││ 5 │ 1 ││ 6 │ 1 │╰───┴───╯"
    );
}

#[test]
fn test_expand_big_0() {
    Playground::setup("test_expand_big_0", |dirs, sandbox| {
        sandbox.with_files(vec![FileWithContent(
            "sample.toml",
            r#"
            [package]
            authors = ["The Nushell Project Developers"]
            default-run = "nu"
            description = "A new type of shell"
            documentation = "https://www.nushell.sh/book/"
            edition = "2021"
            exclude = ["images"]
            homepage = "https://www.nushell.sh"
            license = "MIT"
            name = "nu"
            repository = "https://github.com/nushell/nushell"
            rust-version = "1.60"
            version = "0.74.1"

            # See more keys and their definitions at https://doc.rust-lang.org/cargo/reference/manifest.html

            [package.metadata.binstall]
            pkg-url = "{ repo }/releases/download/{ version }/{ name }-{ version }-{ target }.{ archive-format }"
            pkg-fmt = "tgz"

            [package.metadata.binstall.overrides.x86_64-pc-windows-msvc]
            pkg-fmt = "zip"

            [workspace]
            members = [
                "crates/nu-cli",
                "crates/nu-engine",
                "crates/nu-parser",
                "crates/nu-system",
                "crates/nu-command",
                "crates/nu-protocol",
                "crates/nu-plugin",
                "crates/nu_plugin_inc",
                "crates/nu_plugin_gstat",
                "crates/nu_plugin_example",
                "crates/nu_plugin_query",
                "crates/nu_plugin_custom_values",
                "crates/nu-utils",
            ]

            [dependencies]
            chrono = { version = "0.4.23", features = ["serde"] }
            crossterm = "0.24.0"
            ctrlc = "3.2.1"
            log = "0.4"
            miette = { version = "5.5.0", features = ["fancy-no-backtrace"] }
            nu-ansi-term = "0.46.0"
            nu-cli = { path = "./crates/nu-cli", version = "0.74.1" }
            nu-engine = { path = "./crates/nu-engine", version = "0.74.1" }
            reedline = { version = "0.14.0", features = ["bashisms", "sqlite"] }

            rayon = "1.6.1"
            is_executable = "1.0.1"
            simplelog = "0.12.0"
            time = "0.3.12"

            [target.'cfg(not(target_os = "windows"))'.dependencies]
            # Our dependencies don't use OpenSSL on Windows
            openssl = { version = "0.10.38", features = ["vendored"], optional = true }
            signal-hook = { version = "0.3.14", default-features = false }


            [target.'cfg(windows)'.build-dependencies]
            winres = "0.1"

            [target.'cfg(target_family = "unix")'.dependencies]
            nix = { version = "0.25", default-features = false, features = ["signal", "process", "fs", "term"] }
            atty = "0.2"

            [dev-dependencies]
            nu-test-support = { path = "./crates/nu-test-support", version = "0.74.1" }
            tempfile = "3.2.0"
            assert_cmd = "2.0.2"
            criterion = "0.4"
            pretty_assertions = "1.0.0"
            serial_test = "0.10.0"
            hamcrest2 = "0.3.0"
            rstest = { version = "0.15.0", default-features = false }
            itertools = "0.10.3"

            [features]
            plugin = [
                "nu-plugin",
                "nu-cli/plugin",
                "nu-parser/plugin",
                "nu-command/plugin",
                "nu-protocol/plugin",
                "nu-engine/plugin",
            ]
            # extra used to be more useful but now it's the same as default. Leaving it in for backcompat with existing build scripts
            extra = ["default"]
            default = ["plugin", "which-support", "trash-support", "sqlite"]
            stable = ["default"]
            wasi = []

            # Enable to statically link OpenSSL; otherwise the system version will be used. Not enabled by default because it takes a while to build
            static-link-openssl = ["dep:openssl"]

            # Stable (Default)
            which-support = ["nu-command/which-support"]
            trash-support = ["nu-command/trash-support"]

            # Main nu binary
            [[bin]]
            name = "nu"
            path = "src/main.rs"

            # To use a development version of a dependency please use a global override here
            # changing versions in each sub-crate of the workspace is tedious
            [patch.crates-io]
            reedline = { git = "https://github.com/nushell/reedline.git", branch = "main" }

            # Criterion benchmarking setup
            # Run all benchmarks with `cargo bench`
            # Run individual benchmarks like `cargo bench -- <regex>` e.g. `cargo bench -- parse`
            [[bench]]
            name = "benchmarks"
            harness = false
            "#,
        )]);

        let actual = nu!(
            cwd: dirs.test(), pipeline(
            "open sample.toml | table --expand"
        ));

        let expected = join_lines([
            "╭──────────────────┬───────────────────────────────────────────────────────────╮",
            "│                  │ ╭───┬─────────┬────────────╮                              │",
            "│ bench            │ │ # │ harness │    name    │                              │",
            "│                  │ ├───┼─────────┼────────────┤                              │",
            "│                  │ │ 0 │ false   │ benchmarks │                              │",
            "│                  │ ╰───┴─────────┴────────────╯                              │",
            "│                  │ ╭───┬──────┬─────────────╮                                │",
            "│ bin              │ │ # │ name │    path     │                                │",
            "│                  │ ├───┼──────┼─────────────┤                                │",
            "│                  │ │ 0 │ nu   │ src/main.rs │                                │",
            "│                  │ ╰───┴──────┴─────────────╯                                │",
            "│                  │ ╭───────────────┬───────────────────────────────────────╮ │",
            "│ dependencies     │ │               │ ╭──────────┬───────────────╮          │ │",
            "│                  │ │ chrono        │ │          │ ╭───┬───────╮ │          │ │",
            "│                  │ │               │ │ features │ │ 0 │ serde │ │          │ │",
            "│                  │ │               │ │          │ ╰───┴───────╯ │          │ │",
            "│                  │ │               │ │ version  │ 0.4.23        │          │ │",
            "│                  │ │               │ ╰──────────┴───────────────╯          │ │",
            "│                  │ │ crossterm     │ 0.24.0                                │ │",
            "│                  │ │ ctrlc         │ 3.2.1                                 │ │",
            "│                  │ │ is_executable │ 1.0.1                                 │ │",
            "│                  │ │ log           │ 0.4                                   │ │",
            "│                  │ │               │ ╭──────────┬────────────────────────╮ │ │",
            "│                  │ │ miette        │ │          │ ╭───┬────────────────╮ │ │ │",
            "│                  │ │               │ │ features │ │ 0 │ fancy-no-backt │ │ │ │",
            "│                  │ │               │ │          │ │   │ race           │ │ │ │",
            "│                  │ │               │ │          │ ╰───┴────────────────╯ │ │ │",
            "│                  │ │               │ │ version  │ 5.5.0                  │ │ │",
            "│                  │ │               │ ╰──────────┴────────────────────────╯ │ │",
            "│                  │ │ nu-ansi-term  │ 0.46.0                                │ │",
            "│                  │ │               │ ╭─────────┬─────────────────╮         │ │",
            "│                  │ │ nu-cli        │ │ path    │ ./crates/nu-cli │         │ │",
            "│                  │ │               │ │ version │ 0.74.1          │         │ │",
            "│                  │ │               │ ╰─────────┴─────────────────╯         │ │",
            "│                  │ │               │ ╭────────────┬──────────────────────╮ │ │",
            "│                  │ │ nu-engine     │ │ path       │ ./crates/nu-engine   │ │ │",
            "│                  │ │               │ │ version    │ 0.74.1               │ │ │",
            "│                  │ │               │ ╰────────────┴──────────────────────╯ │ │",
            "│                  │ │ rayon         │ 1.6.1                                 │ │",
            "│                  │ │               │ ╭─────────────┬─────────────────────╮ │ │",
            "│                  │ │ reedline      │ │             │ ╭───┬──────────╮    │ │ │",
            "│                  │ │               │ │ features    │ │ 0 │ bashisms │    │ │ │",
            "│                  │ │               │ │             │ │ 1 │ sqlite   │    │ │ │",
            "│                  │ │               │ │             │ ╰───┴──────────╯    │ │ │",
            "│                  │ │               │ │ version     │ 0.14.0              │ │ │",
            "│                  │ │               │ ╰─────────────┴─────────────────────╯ │ │",
            "│                  │ │ simplelog     │ 0.12.0                                │ │",
            "│                  │ │ time          │ 0.3.12                                │ │",
            "│                  │ ╰───────────────┴───────────────────────────────────────╯ │",
            "│                  │ ╭───────────────────┬───────────────────────────────────╮ │",
            "│ dev-dependencies │ │ assert_cmd        │ 2.0.2                             │ │",
            "│                  │ │ criterion         │ 0.4                               │ │",
            "│                  │ │ hamcrest2         │ 0.3.0                             │ │",
            "│                  │ │ itertools         │ 0.10.3                            │ │",
            "│                  │ │                   │ ╭─────────┬─────────────────────╮ │ │",
            "│                  │ │ nu-test-support   │ │ path    │ ./crates/nu-test-su │ │ │",
            "│                  │ │                   │ │         │ pport               │ │ │",
            "│                  │ │                   │ │ version │ 0.74.1              │ │ │",
            "│                  │ │                   │ ╰─────────┴─────────────────────╯ │ │",
            "│                  │ │ pretty_assertions │ 1.0.0                             │ │",
            "│                  │ │                   │ ╭────────────────────┬──────────╮ │ │",
            "│                  │ │ rstest            │ │ default-features   │ false    │ │ │",
            "│                  │ │                   │ │ version            │ 0.15.0   │ │ │",
            "│                  │ │                   │ ╰────────────────────┴──────────╯ │ │",
            "│                  │ │ serial_test       │ 0.10.0                            │ │",
            "│                  │ │ tempfile          │ 3.2.0                             │ │",
            "│                  │ ╰───────────────────┴───────────────────────────────────╯ │",
            "│                  │ ╭─────────────────────┬─────────────────────────────────╮ │",
            "│ features         │ │                     │ ╭───┬───────────────╮           │ │",
            "│                  │ │ default             │ │ 0 │ plugin        │           │ │",
            "│                  │ │                     │ │ 1 │ which-support │           │ │",
            "│                  │ │                     │ │ 2 │ trash-support │           │ │",
            "│                  │ │                     │ │ 3 │ sqlite        │           │ │",
            "│                  │ │                     │ ╰───┴───────────────╯           │ │",
            "│                  │ │                     │ ╭───┬─────────╮                 │ │",
            "│                  │ │ extra               │ │ 0 │ default │                 │ │",
            "│                  │ │                     │ ╰───┴─────────╯                 │ │",
            "│                  │ │                     │ ╭───┬────────────────────╮      │ │",
            "│                  │ │ plugin              │ │ 0 │ nu-plugin          │      │ │",
            "│                  │ │                     │ │ 1 │ nu-cli/plugin      │      │ │",
            "│                  │ │                     │ │ 2 │ nu-parser/plugin   │      │ │",
            "│                  │ │                     │ │ 3 │ nu-command/plugin  │      │ │",
            "│                  │ │                     │ │ 4 │ nu-protocol/plugin │      │ │",
            "│                  │ │                     │ │ 5 │ nu-engine/plugin   │      │ │",
            "│                  │ │                     │ ╰───┴────────────────────╯      │ │",
            "│                  │ │                     │ ╭───┬─────────╮                 │ │",
            "│                  │ │ stable              │ │ 0 │ default │                 │ │",
            "│                  │ │                     │ ╰───┴─────────╯                 │ │",
            "│                  │ │                     │ ╭───┬─────────────╮             │ │",
            "│                  │ │ static-link-openssl │ │ 0 │ dep:openssl │             │ │",
            "│                  │ │                     │ ╰───┴─────────────╯             │ │",
            "│                  │ │                     │ ╭───┬─────────────────────────╮ │ │",
            "│                  │ │ trash-support       │ │ 0 │ nu-command/trash-suppor │ │ │",
            "│                  │ │                     │ │   │ t                       │ │ │",
            "│                  │ │                     │ ╰───┴─────────────────────────╯ │ │",
            "│                  │ │ wasi                │ [list 0 items]                  │ │",
            "│                  │ │                     │ ╭───┬─────────────────────────╮ │ │",
            "│                  │ │ which-support       │ │ 0 │ nu-command/which-suppor │ │ │",
            "│                  │ │                     │ │   │ t                       │ │ │",
            "│                  │ │                     │ ╰───┴─────────────────────────╯ │ │",
            "│                  │ ╰─────────────────────┴─────────────────────────────────╯ │",
            "│                  │ ╭───────────────┬───────────────────────────────────────╮ │",
            "│ package          │ │               │ ╭───┬───────────────────────────────╮ │ │",
            "│                  │ │ authors       │ │ 0 │ The Nushell Project           │ │ │",
            "│                  │ │               │ │   │ Developers                    │ │ │",
            "│                  │ │               │ ╰───┴───────────────────────────────╯ │ │",
            "│                  │ │ default-run   │ nu                                    │ │",
            "│                  │ │ description   │ A new type of shell                   │ │",
            "│                  │ │ documentation │ https://www.nushell.sh/book/          │ │",
            "│                  │ │ edition       │ 2021                                  │ │",
            "│                  │ │               │ ╭───┬────────╮                        │ │",
            "│                  │ │ exclude       │ │ 0 │ images │                        │ │",
            "│                  │ │               │ ╰───┴────────╯                        │ │",
            "│                  │ │ homepage      │ https://www.nushell.sh                │ │",
            "│                  │ │ license       │ MIT                                   │ │",
            "│                  │ │               │ ╭──────────┬────────────────────────╮ │ │",
            "│                  │ │ metadata      │ │          │ ╭───────────┬────────╮ │ │ │",
            "│                  │ │               │ │ binstall │ │ overrides │ {recor │ │ │ │",
            "│                  │ │               │ │          │ │           │ d 1    │ │ │ │",
            "│                  │ │               │ │          │ │           │ field} │ │ │ │",
            "│                  │ │               │ │          │ │ pkg-fmt   │ tgz    │ │ │ │",
            "│                  │ │               │ │          │ │ pkg-url   │ { repo │ │ │ │",
            "│                  │ │               │ │          │ │           │  }/rel │ │ │ │",
            "│                  │ │               │ │          │ │           │ eases/ │ │ │ │",
            "│                  │ │               │ │          │ │           │ downlo │ │ │ │",
            "│                  │ │               │ │          │ │           │ ad/{ v │ │ │ │",
            "│                  │ │               │ │          │ │           │ ersion │ │ │ │",
            "│                  │ │               │ │          │ │           │  }/{   │ │ │ │",
            "│                  │ │               │ │          │ │           │ name   │ │ │ │",
            "│                  │ │               │ │          │ │           │ }-{ ve │ │ │ │",
            "│                  │ │               │ │          │ │           │ rsion  │ │ │ │",
            "│                  │ │               │ │          │ │           │ }-{    │ │ │ │",
            "│                  │ │               │ │          │ │           │ target │ │ │ │",
            "│                  │ │               │ │          │ │           │  }.{ a │ │ │ │",
            "│                  │ │               │ │          │ │           │ rchive │ │ │ │",
            "│                  │ │               │ │          │ │           │ -forma │ │ │ │",
            "│                  │ │               │ │          │ │           │ t }    │ │ │ │",
            "│                  │ │               │ │          │ ╰───────────┴────────╯ │ │ │",
            "│                  │ │               │ ╰──────────┴────────────────────────╯ │ │",
            "│                  │ │ name          │ nu                                    │ │",
            "│                  │ │ repository    │ https://github.com/nushell/nushell    │ │",
            "│                  │ │ rust-version  │ 1.60                                  │ │",
            "│                  │ │ version       │ 0.74.1                                │ │",
            "│                  │ ╰───────────────┴───────────────────────────────────────╯ │",
            "│                  │ ╭───────────┬───────────────────────────────────────────╮ │",
            "│ patch            │ │           │ ╭──────────┬────────────────────────────╮ │ │",
            "│                  │ │ crates-io │ │          │ ╭────────┬───────────────╮ │ │ │",
            "│                  │ │           │ │ reedline │ │ branch │ main          │ │ │ │",
            "│                  │ │           │ │          │ │ git    │ https://githu │ │ │ │",
            "│                  │ │           │ │          │ │        │ b.com/nushell │ │ │ │",
            "│                  │ │           │ │          │ │        │ /reedline.git │ │ │ │",
            "│                  │ │           │ │          │ ╰────────┴───────────────╯ │ │ │",
            "│                  │ │           │ ╰──────────┴────────────────────────────╯ │ │",
            "│                  │ ╰───────────┴───────────────────────────────────────────╯ │",
            "│                  │ ╭─────────────────────────────────┬─────────────────────╮ │",
            "│ target           │ │                                 │ ╭──────────────┬──╮ │ │",
            "│                  │ │ cfg(not(target_os = \"windows\")) │ │ dependencies │  │ │ │",
            "│                  │ │                                 │ ╰──────────────┴──╯ │ │",
            "│                  │ │                                 │ ╭──────────────┬──╮ │ │",
            "│                  │ │ cfg(target_family = \"unix\")     │ │ dependencies │  │ │ │",
            "│                  │ │                                 │ ╰──────────────┴──╯ │ │",
            "│                  │ │ cfg(windows)                    │ {record 1 field}    │ │",
            "│                  │ ╰─────────────────────────────────┴─────────────────────╯ │",
            "│                  │ ╭───────────┬───────────────────────────────────────────╮ │",
            "│ workspace        │ │           │ ╭────┬────────────────────────────────╮   │ │",
            "│                  │ │ members   │ │  0 │ crates/nu-cli                  │   │ │",
            "│                  │ │           │ │  1 │ crates/nu-engine               │   │ │",
            "│                  │ │           │ │  2 │ crates/nu-parser               │   │ │",
            "│                  │ │           │ │  3 │ crates/nu-system               │   │ │",
            "│                  │ │           │ │  4 │ crates/nu-command              │   │ │",
            "│                  │ │           │ │  5 │ crates/nu-protocol             │   │ │",
            "│                  │ │           │ │  6 │ crates/nu-plugin               │   │ │",
            "│                  │ │           │ │  7 │ crates/nu_plugin_inc           │   │ │",
            "│                  │ │           │ │  8 │ crates/nu_plugin_gstat         │   │ │",
            "│                  │ │           │ │  9 │ crates/nu_plugin_example       │   │ │",
            "│                  │ │           │ │ 10 │ crates/nu_plugin_query         │   │ │",
            "│                  │ │           │ │ 11 │ crates/nu_plugin_custom_values │   │ │",
            "│                  │ │           │ │ 12 │ crates/nu-utils                │   │ │",
            "│                  │ │           │ ╰────┴────────────────────────────────╯   │ │",
            "│                  │ ╰───────────┴───────────────────────────────────────────╯ │",
            "╰──────────────────┴───────────────────────────────────────────────────────────╯",
        ]);

        assert_eq!(actual.out, expected);

        let actual = nu!(
            cwd: dirs.test(), pipeline(
            "open sample.toml | table --expand --width=120"
        ));

        let expected = join_lines([
            "╭──────────────────┬───────────────────────────────────────────────────────────────────────────────────────────────────╮",
            "│                  │ ╭───┬─────────┬────────────╮                                                                      │",
            "│ bench            │ │ # │ harness │    name    │                                                                      │",
            "│                  │ ├───┼─────────┼────────────┤                                                                      │",
            "│                  │ │ 0 │ false   │ benchmarks │                                                                      │",
            "│                  │ ╰───┴─────────┴────────────╯                                                                      │",
            "│                  │ ╭───┬──────┬─────────────╮                                                                        │",
            "│ bin              │ │ # │ name │    path     │                                                                        │",
            "│                  │ ├───┼──────┼─────────────┤                                                                        │",
            "│                  │ │ 0 │ nu   │ src/main.rs │                                                                        │",
            "│                  │ ╰───┴──────┴─────────────╯                                                                        │",
            "│                  │ ╭───────────────┬───────────────────────────────────────────╮                                     │",
            "│ dependencies     │ │               │ ╭──────────┬───────────────╮              │                                     │",
            "│                  │ │ chrono        │ │          │ ╭───┬───────╮ │              │                                     │",
            "│                  │ │               │ │ features │ │ 0 │ serde │ │              │                                     │",
            "│                  │ │               │ │          │ ╰───┴───────╯ │              │                                     │",
            "│                  │ │               │ │ version  │ 0.4.23        │              │                                     │",
            "│                  │ │               │ ╰──────────┴───────────────╯              │                                     │",
            "│                  │ │ crossterm     │ 0.24.0                                    │                                     │",
            "│                  │ │ ctrlc         │ 3.2.1                                     │                                     │",
            "│                  │ │ is_executable │ 1.0.1                                     │                                     │",
            "│                  │ │ log           │ 0.4                                       │                                     │",
            "│                  │ │               │ ╭──────────┬────────────────────────────╮ │                                     │",
            "│                  │ │ miette        │ │          │ ╭───┬────────────────────╮ │ │                                     │",
            "│                  │ │               │ │ features │ │ 0 │ fancy-no-backtrace │ │ │                                     │",
            "│                  │ │               │ │          │ ╰───┴────────────────────╯ │ │                                     │",
            "│                  │ │               │ │ version  │ 5.5.0                      │ │                                     │",
            "│                  │ │               │ ╰──────────┴────────────────────────────╯ │                                     │",
            "│                  │ │ nu-ansi-term  │ 0.46.0                                    │                                     │",
            "│                  │ │               │ ╭─────────┬─────────────────╮             │                                     │",
            "│                  │ │ nu-cli        │ │ path    │ ./crates/nu-cli │             │                                     │",
            "│                  │ │               │ │ version │ 0.74.1          │             │                                     │",
            "│                  │ │               │ ╰─────────┴─────────────────╯             │                                     │",
            "│                  │ │               │ ╭─────────┬────────────────────╮          │                                     │",
            "│                  │ │ nu-engine     │ │ path    │ ./crates/nu-engine │          │                                     │",
            "│                  │ │               │ │ version │ 0.74.1             │          │                                     │",
            "│                  │ │               │ ╰─────────┴────────────────────╯          │                                     │",
            "│                  │ │ rayon         │ 1.6.1                                     │                                     │",
            "│                  │ │               │ ╭──────────┬──────────────────╮           │                                     │",
            "│                  │ │ reedline      │ │          │ ╭───┬──────────╮ │           │                                     │",
            "│                  │ │               │ │ features │ │ 0 │ bashisms │ │           │                                     │",
            "│                  │ │               │ │          │ │ 1 │ sqlite   │ │           │                                     │",
            "│                  │ │               │ │          │ ╰───┴──────────╯ │           │                                     │",
            "│                  │ │               │ │ version  │ 0.14.0           │           │                                     │",
            "│                  │ │               │ ╰──────────┴──────────────────╯           │                                     │",
            "│                  │ │ simplelog     │ 0.12.0                                    │                                     │",
            "│                  │ │ time          │ 0.3.12                                    │                                     │",
            "│                  │ ╰───────────────┴───────────────────────────────────────────╯                                     │",
            "│                  │ ╭───────────────────┬────────────────────────────────────────╮                                    │",
            "│ dev-dependencies │ │ assert_cmd        │ 2.0.2                                  │                                    │",
            "│                  │ │ criterion         │ 0.4                                    │                                    │",
            "│                  │ │ hamcrest2         │ 0.3.0                                  │                                    │",
            "│                  │ │ itertools         │ 0.10.3                                 │                                    │",
            "│                  │ │                   │ ╭─────────┬──────────────────────────╮ │                                    │",
            "│                  │ │ nu-test-support   │ │ path    │ ./crates/nu-test-support │ │                                    │",
            "│                  │ │                   │ │ version │ 0.74.1                   │ │                                    │",
            "│                  │ │                   │ ╰─────────┴──────────────────────────╯ │                                    │",
            "│                  │ │ pretty_assertions │ 1.0.0                                  │                                    │",
            "│                  │ │                   │ ╭──────────────────┬────────╮          │                                    │",
            "│                  │ │ rstest            │ │ default-features │ false  │          │                                    │",
            "│                  │ │                   │ │ version          │ 0.15.0 │          │                                    │",
            "│                  │ │                   │ ╰──────────────────┴────────╯          │                                    │",
            "│                  │ │ serial_test       │ 0.10.0                                 │                                    │",
            "│                  │ │ tempfile          │ 3.2.0                                  │                                    │",
            "│                  │ ╰───────────────────┴────────────────────────────────────────╯                                    │",
            "│                  │ ╭─────────────────────┬──────────────────────────────────╮                                        │",
            "│ features         │ │                     │ ╭───┬───────────────╮            │                                        │",
            "│                  │ │ default             │ │ 0 │ plugin        │            │                                        │",
            "│                  │ │                     │ │ 1 │ which-support │            │                                        │",
            "│                  │ │                     │ │ 2 │ trash-support │            │                                        │",
            "│                  │ │                     │ │ 3 │ sqlite        │            │                                        │",
            "│                  │ │                     │ ╰───┴───────────────╯            │                                        │",
            "│                  │ │                     │ ╭───┬─────────╮                  │                                        │",
            "│                  │ │ extra               │ │ 0 │ default │                  │                                        │",
            "│                  │ │                     │ ╰───┴─────────╯                  │                                        │",
            "│                  │ │                     │ ╭───┬────────────────────╮       │                                        │",
            "│                  │ │ plugin              │ │ 0 │ nu-plugin          │       │                                        │",
            "│                  │ │                     │ │ 1 │ nu-cli/plugin      │       │                                        │",
            "│                  │ │                     │ │ 2 │ nu-parser/plugin   │       │                                        │",
            "│                  │ │                     │ │ 3 │ nu-command/plugin  │       │                                        │",
            "│                  │ │                     │ │ 4 │ nu-protocol/plugin │       │                                        │",
            "│                  │ │                     │ │ 5 │ nu-engine/plugin   │       │                                        │",
            "│                  │ │                     │ ╰───┴────────────────────╯       │                                        │",
            "│                  │ │                     │ ╭───┬─────────╮                  │                                        │",
            "│                  │ │ stable              │ │ 0 │ default │                  │                                        │",
            "│                  │ │                     │ ╰───┴─────────╯                  │                                        │",
            "│                  │ │                     │ ╭───┬─────────────╮              │                                        │",
            "│                  │ │ static-link-openssl │ │ 0 │ dep:openssl │              │                                        │",
            "│                  │ │                     │ ╰───┴─────────────╯              │                                        │",
            "│                  │ │                     │ ╭───┬──────────────────────────╮ │                                        │",
            "│                  │ │ trash-support       │ │ 0 │ nu-command/trash-support │ │                                        │",
            "│                  │ │                     │ ╰───┴──────────────────────────╯ │                                        │",
            "│                  │ │ wasi                │ [list 0 items]                   │                                        │",
            "│                  │ │                     │ ╭───┬──────────────────────────╮ │                                        │",
            "│                  │ │ which-support       │ │ 0 │ nu-command/which-support │ │                                        │",
            "│                  │ │                     │ ╰───┴──────────────────────────╯ │                                        │",
            "│                  │ ╰─────────────────────┴──────────────────────────────────╯                                        │",
            "│                  │ ╭───────────────┬───────────────────────────────────────────────────────────────────────────────╮ │",
            "│ package          │ │               │ ╭───┬────────────────────────────────╮                                        │ │",
            "│                  │ │ authors       │ │ 0 │ The Nushell Project Developers │                                        │ │",
            "│                  │ │               │ ╰───┴────────────────────────────────╯                                        │ │",
            "│                  │ │ default-run   │ nu                                                                            │ │",
            "│                  │ │ description   │ A new type of shell                                                           │ │",
            "│                  │ │ documentation │ https://www.nushell.sh/book/                                                  │ │",
            "│                  │ │ edition       │ 2021                                                                          │ │",
            "│                  │ │               │ ╭───┬────────╮                                                                │ │",
            "│                  │ │ exclude       │ │ 0 │ images │                                                                │ │",
            "│                  │ │               │ ╰───┴────────╯                                                                │ │",
            "│                  │ │ homepage      │ https://www.nushell.sh                                                        │ │",
            "│                  │ │ license       │ MIT                                                                           │ │",
            "│                  │ │               │ ╭──────────┬────────────────────────────────────────────────────────────────╮ │ │",
            "│                  │ │ metadata      │ │          │ ╭───────────┬────────────────────────────────────────────────╮ │ │ │",
            "│                  │ │               │ │ binstall │ │           │ ╭────────────────────────┬───────────────────╮ │ │ │ │",
            "│                  │ │               │ │          │ │ overrides │ │                        │ ╭─────────┬─────╮ │ │ │ │ │",
            "│                  │ │               │ │          │ │           │ │ x86_64-pc-windows-msvc │ │ pkg-fmt │ zip │ │ │ │ │ │",
            "│                  │ │               │ │          │ │           │ │                        │ ╰─────────┴─────╯ │ │ │ │ │",
            "│                  │ │               │ │          │ │           │ ╰────────────────────────┴───────────────────╯ │ │ │ │",
            "│                  │ │               │ │          │ │ pkg-fmt   │ tgz                                            │ │ │ │",
            "│                  │ │               │ │          │ │ pkg-url   │ { repo }/releases/download/{ version }/{ name  │ │ │ │",
            "│                  │ │               │ │          │ │           │ }-{ version }-{ target }.{ archive-format }    │ │ │ │",
            "│                  │ │               │ │          │ ╰───────────┴────────────────────────────────────────────────╯ │ │ │",
            "│                  │ │               │ ╰──────────┴────────────────────────────────────────────────────────────────╯ │ │",
            "│                  │ │ name          │ nu                                                                            │ │",
            "│                  │ │ repository    │ https://github.com/nushell/nushell                                            │ │",
            "│                  │ │ rust-version  │ 1.60                                                                          │ │",
            "│                  │ │ version       │ 0.74.1                                                                        │ │",
            "│                  │ ╰───────────────┴───────────────────────────────────────────────────────────────────────────────╯ │",
            "│                  │ ╭───────────┬───────────────────────────────────────────────────────────────────────────────────╮ │",
            "│ patch            │ │           │ ╭─────────────────┬─────────────────────────────────────────────────────────────╮ │ │",
            "│                  │ │ crates-io │ │                 │ ╭────────┬─────────────────────────────────────────╮        │ │ │",
            "│                  │ │           │ │ reedline        │ │ branch │ main                                    │        │ │ │",
            "│                  │ │           │ │                 │ │ git    │ https://github.com/nushell/reedline.git │        │ │ │",
            "│                  │ │           │ │                 │ ╰────────┴─────────────────────────────────────────╯        │ │ │",
            "│                  │ │           │ ╰─────────────────┴─────────────────────────────────────────────────────────────╯ │ │",
            "│                  │ ╰───────────┴───────────────────────────────────────────────────────────────────────────────────╯ │",
            "│                  │ ╭─────────────────────────────────┬─────────────────────────────────────────────────────────────╮ │",
            "│ target           │ │                                 │ ╭──────────────┬──────────────────────────────────────────╮ │ │",
            "│                  │ │ cfg(not(target_os = \"windows\")) │ │              │ ╭─────────────┬────────────────────────╮ │ │ │",
            "│                  │ │                                 │ │ dependencies │ │             │ ╭──────────┬─────────╮ │ │ │ │",
            "│                  │ │                                 │ │              │ │ openssl     │ │ features │ [list 1 │ │ │ │ │",
            "│                  │ │                                 │ │              │ │             │ │          │  item]  │ │ │ │ │",
            "│                  │ │                                 │ │              │ │             │ │ optional │ true    │ │ │ │ │",
            "│                  │ │                                 │ │              │ │             │ │ version  │ 0.10.38 │ │ │ │ │",
            "│                  │ │                                 │ │              │ │             │ ╰──────────┴─────────╯ │ │ │ │",
            "│                  │ │                                 │ │              │ │ signal-hook │ {record 2 fields}      │ │ │ │",
            "│                  │ │                                 │ │              │ ╰─────────────┴────────────────────────╯ │ │ │",
            "│                  │ │                                 │ ╰──────────────┴──────────────────────────────────────────╯ │ │",
            "│                  │ │                                 │ ╭──────────────┬──────────────────────────────────────────╮ │ │",
            "│                  │ │ cfg(target_family = \"unix\")     │ │              │ ╭──────┬───────────────────────────────╮ │ │ │",
            "│                  │ │                                 │ │ dependencies │ │ atty │ 0.2                           │ │ │ │",
            "│                  │ │                                 │ │              │ │      │ ╭──────────────────┬────────╮ │ │ │ │",
            "│                  │ │                                 │ │              │ │ nix  │ │ default-features │ false  │ │ │ │ │",
            "│                  │ │                                 │ │              │ │      │ │ features         │ [list  │ │ │ │ │",
            "│                  │ │                                 │ │              │ │      │ │                  │ 4      │ │ │ │ │",
            "│                  │ │                                 │ │              │ │      │ │                  │ items] │ │ │ │ │",
            "│                  │ │                                 │ │              │ │      │ │ version          │ 0.25   │ │ │ │ │",
            "│                  │ │                                 │ │              │ │      │ ╰──────────────────┴────────╯ │ │ │ │",
            "│                  │ │                                 │ │              │ ╰──────┴───────────────────────────────╯ │ │ │",
            "│                  │ │                                 │ ╰──────────────┴──────────────────────────────────────────╯ │ │",
            "│                  │ │                                 │ ╭────────────────────┬──────────────────╮                   │ │",
            "│                  │ │ cfg(windows)                    │ │                    │ ╭────────┬─────╮ │                   │ │",
            "│                  │ │                                 │ │ build-dependencies │ │ winres │ 0.1 │ │                   │ │",
            "│                  │ │                                 │ │                    │ ╰────────┴─────╯ │                   │ │",
            "│                  │ │                                 │ ╰────────────────────┴──────────────────╯                   │ │",
            "│                  │ ╰─────────────────────────────────┴─────────────────────────────────────────────────────────────╯ │",
            "│                  │ ╭─────────┬─────────────────────────────────────────╮                                             │",
            "│ workspace        │ │         │ ╭────┬────────────────────────────────╮ │                                             │",
            "│                  │ │ members │ │  0 │ crates/nu-cli                  │ │                                             │",
            "│                  │ │         │ │  1 │ crates/nu-engine               │ │                                             │",
            "│                  │ │         │ │  2 │ crates/nu-parser               │ │                                             │",
            "│                  │ │         │ │  3 │ crates/nu-system               │ │                                             │",
            "│                  │ │         │ │  4 │ crates/nu-command              │ │                                             │",
            "│                  │ │         │ │  5 │ crates/nu-protocol             │ │                                             │",
            "│                  │ │         │ │  6 │ crates/nu-plugin               │ │                                             │",
            "│                  │ │         │ │  7 │ crates/nu_plugin_inc           │ │                                             │",
            "│                  │ │         │ │  8 │ crates/nu_plugin_gstat         │ │                                             │",
            "│                  │ │         │ │  9 │ crates/nu_plugin_example       │ │                                             │",
            "│                  │ │         │ │ 10 │ crates/nu_plugin_query         │ │                                             │",
            "│                  │ │         │ │ 11 │ crates/nu_plugin_custom_values │ │                                             │",
            "│                  │ │         │ │ 12 │ crates/nu-utils                │ │                                             │",
            "│                  │ │         │ ╰────┴────────────────────────────────╯ │                                             │",
            "│                  │ ╰─────────┴─────────────────────────────────────────╯                                             │",
            "╰──────────────────┴───────────────────────────────────────────────────────────────────────────────────────────────────╯",
        ]);

        assert_eq!(actual.out, expected);

        let actual = nu!(
            cwd: dirs.test(), pipeline(
            "open sample.toml | table --expand --width=60"
        ));

        _print_lines(&actual.out, 60);

        let expected = join_lines([
            "╭──────────────────┬───────────────────────────────────────╮",
            "│                  │ ╭───┬─────────┬────────────╮          │",
            "│ bench            │ │ # │ harness │    name    │          │",
            "│                  │ ├───┼─────────┼────────────┤          │",
            "│                  │ │ 0 │ false   │ benchmarks │          │",
            "│                  │ ╰───┴─────────┴────────────╯          │",
            "│                  │ ╭───┬──────┬─────────────╮            │",
            "│ bin              │ │ # │ name │    path     │            │",
            "│                  │ ├───┼──────┼─────────────┤            │",
            "│                  │ │ 0 │ nu   │ src/main.rs │            │",
            "│                  │ ╰───┴──────┴─────────────╯            │",
            "│                  │ ╭───────────────┬───────────────────╮ │",
            "│ dependencies     │ │               │ ╭──────────┬────╮ │ │",
            "│                  │ │ chrono        │ │ features │ [l │ │ │",
            "│                  │ │               │ │          │ is │ │ │",
            "│                  │ │               │ │          │ t  │ │ │",
            "│                  │ │               │ │          │ 1  │ │ │",
            "│                  │ │               │ │          │ it │ │ │",
            "│                  │ │               │ │          │ em │ │ │",
            "│                  │ │               │ │          │ ]  │ │ │",
            "│                  │ │               │ │ version  │ 0. │ │ │",
            "│                  │ │               │ │          │ 4. │ │ │",
            "│                  │ │               │ │          │ 23 │ │ │",
            "│                  │ │               │ ╰──────────┴────╯ │ │",
            "│                  │ │ crossterm     │ 0.24.0            │ │",
            "│                  │ │ ctrlc         │ 3.2.1             │ │",
            "│                  │ │ is_executable │ 1.0.1             │ │",
            "│                  │ │ log           │ 0.4               │ │",
            "│                  │ │               │ ╭──────────┬────╮ │ │",
            "│                  │ │ miette        │ │ features │ [l │ │ │",
            "│                  │ │               │ │          │ is │ │ │",
            "│                  │ │               │ │          │ t  │ │ │",
            "│                  │ │               │ │          │ 1  │ │ │",
            "│                  │ │               │ │          │ it │ │ │",
            "│                  │ │               │ │          │ em │ │ │",
            "│                  │ │               │ │          │ ]  │ │ │",
            "│                  │ │               │ │ version  │ 5. │ │ │",
            "│                  │ │               │ │          │ 5. │ │ │",
            "│                  │ │               │ │          │ 0  │ │ │",
            "│                  │ │               │ ╰──────────┴────╯ │ │",
            "│                  │ │ nu-ansi-term  │ 0.46.0            │ │",
            "│                  │ │               │ ╭─────────┬─────╮ │ │",
            "│                  │ │ nu-cli        │ │ path    │ ./c │ │ │",
            "│                  │ │               │ │         │ rat │ │ │",
            "│                  │ │               │ │         │ es/ │ │ │",
            "│                  │ │               │ │         │ nu- │ │ │",
            "│                  │ │               │ │         │ cli │ │ │",
            "│                  │ │               │ │ version │ 0.7 │ │ │",
            "│                  │ │               │ │         │ 4.1 │ │ │",
            "│                  │ │               │ ╰─────────┴─────╯ │ │",
            "│                  │ │               │ ╭─────────┬─────╮ │ │",
            "│                  │ │ nu-engine     │ │ path    │ ./c │ │ │",
            "│                  │ │               │ │         │ rat │ │ │",
            "│                  │ │               │ │         │ es/ │ │ │",
            "│                  │ │               │ │         │ nu- │ │ │",
            "│                  │ │               │ │         │ eng │ │ │",
            "│                  │ │               │ │         │ ine │ │ │",
            "│                  │ │               │ │ version │ 0.7 │ │ │",
            "│                  │ │               │ │         │ 4.1 │ │ │",
            "│                  │ │               │ ╰─────────┴─────╯ │ │",
            "│                  │ │ rayon         │ 1.6.1             │ │",
            "│                  │ │               │ ╭──────────┬────╮ │ │",
            "│                  │ │ reedline      │ │ features │ [l │ │ │",
            "│                  │ │               │ │          │ is │ │ │",
            "│                  │ │               │ │          │ t  │ │ │",
            "│                  │ │               │ │          │ 2  │ │ │",
            "│                  │ │               │ │          │ it │ │ │",
            "│                  │ │               │ │          │ em │ │ │",
            "│                  │ │               │ │          │ s] │ │ │",
            "│                  │ │               │ │ version  │ 0. │ │ │",
            "│                  │ │               │ │          │ 14 │ │ │",
            "│                  │ │               │ │          │ .0 │ │ │",
            "│                  │ │               │ ╰──────────┴────╯ │ │",
            "│                  │ │ simplelog     │ 0.12.0            │ │",
            "│                  │ │ time          │ 0.3.12            │ │",
            "│                  │ ╰───────────────┴───────────────────╯ │",
            "│                  │ ╭───────────────────┬───────────────╮ │",
            "│ dev-dependencies │ │ assert_cmd        │ 2.0.2         │ │",
            "│                  │ │ criterion         │ 0.4           │ │",
            "│                  │ │ hamcrest2         │ 0.3.0         │ │",
            "│                  │ │ itertools         │ 0.10.3        │ │",
            "│                  │ │ nu-test-support   │ {record 2     │ │",
            "│                  │ │                   │ fields}       │ │",
            "│                  │ │ pretty_assertions │ 1.0.0         │ │",
            "│                  │ │ rstest            │ {record 2     │ │",
            "│                  │ │                   │ fields}       │ │",
            "│                  │ │ serial_test       │ 0.10.0        │ │",
            "│                  │ │ tempfile          │ 3.2.0         │ │",
            "│                  │ ╰───────────────────┴───────────────╯ │",
            "│                  │ ╭─────────────────────┬─────────────╮ │",
            "│ features         │ │                     │ ╭───┬─────╮ │ │",
            "│                  │ │ default             │ │ 0 │ plu │ │ │",
            "│                  │ │                     │ │   │ gin │ │ │",
            "│                  │ │                     │ │ 1 │ whi │ │ │",
            "│                  │ │                     │ │   │ ch- │ │ │",
            "│                  │ │                     │ │   │ sup │ │ │",
            "│                  │ │                     │ │   │ por │ │ │",
            "│                  │ │                     │ │   │ t   │ │ │",
            "│                  │ │                     │ │ 2 │ tra │ │ │",
            "│                  │ │                     │ │   │ sh- │ │ │",
            "│                  │ │                     │ │   │ sup │ │ │",
            "│                  │ │                     │ │   │ por │ │ │",
            "│                  │ │                     │ │   │ t   │ │ │",
            "│                  │ │                     │ │ 3 │ sql │ │ │",
            "│                  │ │                     │ │   │ ite │ │ │",
            "│                  │ │                     │ ╰───┴─────╯ │ │",
            "│                  │ │                     │ ╭───┬─────╮ │ │",
            "│                  │ │ extra               │ │ 0 │ def │ │ │",
            "│                  │ │                     │ │   │ aul │ │ │",
            "│                  │ │                     │ │   │ t   │ │ │",
            "│                  │ │                     │ ╰───┴─────╯ │ │",
            "│                  │ │                     │ ╭───┬─────╮ │ │",
            "│                  │ │ plugin              │ │ 0 │ nu- │ │ │",
            "│                  │ │                     │ │   │ plu │ │ │",
            "│                  │ │                     │ │   │ gin │ │ │",
            "│                  │ │                     │ │ 1 │ nu- │ │ │",
            "│                  │ │                     │ │   │ cli │ │ │",
            "│                  │ │                     │ │   │ /pl │ │ │",
            "│                  │ │                     │ │   │ ugi │ │ │",
            "│                  │ │                     │ │   │ n   │ │ │",
            "│                  │ │                     │ │ 2 │ nu- │ │ │",
            "│                  │ │                     │ │   │ par │ │ │",
            "│                  │ │                     │ │   │ ser │ │ │",
            "│                  │ │                     │ │   │ /pl │ │ │",
            "│                  │ │                     │ │   │ ugi │ │ │",
            "│                  │ │                     │ │   │ n   │ │ │",
            "│                  │ │                     │ │ 3 │ nu- │ │ │",
            "│                  │ │                     │ │   │ com │ │ │",
            "│                  │ │                     │ │   │ man │ │ │",
            "│                  │ │                     │ │   │ d/p │ │ │",
            "│                  │ │                     │ │   │ lug │ │ │",
            "│                  │ │                     │ │   │ in  │ │ │",
            "│                  │ │                     │ │ 4 │ nu- │ │ │",
            "│                  │ │                     │ │   │ pro │ │ │",
            "│                  │ │                     │ │   │ toc │ │ │",
            "│                  │ │                     │ │   │ ol/ │ │ │",
            "│                  │ │                     │ │   │ plu │ │ │",
            "│                  │ │                     │ │   │ gin │ │ │",
            "│                  │ │                     │ │ 5 │ nu- │ │ │",
            "│                  │ │                     │ │   │ eng │ │ │",
            "│                  │ │                     │ │   │ ine │ │ │",
            "│                  │ │                     │ │   │ /pl │ │ │",
            "│                  │ │                     │ │   │ ugi │ │ │",
            "│                  │ │                     │ │   │ n   │ │ │",
            "│                  │ │                     │ ╰───┴─────╯ │ │",
            "│                  │ │                     │ ╭───┬─────╮ │ │",
            "│                  │ │ stable              │ │ 0 │ def │ │ │",
            "│                  │ │                     │ │   │ aul │ │ │",
            "│                  │ │                     │ │   │ t   │ │ │",
            "│                  │ │                     │ ╰───┴─────╯ │ │",
            "│                  │ │                     │ ╭───┬─────╮ │ │",
            "│                  │ │ static-link-openssl │ │ 0 │ dep │ │ │",
            "│                  │ │                     │ │   │ :op │ │ │",
            "│                  │ │                     │ │   │ ens │ │ │",
            "│                  │ │                     │ │   │ sl  │ │ │",
            "│                  │ │                     │ ╰───┴─────╯ │ │",
            "│                  │ │                     │ ╭───┬─────╮ │ │",
            "│                  │ │ trash-support       │ │ 0 │ nu- │ │ │",
            "│                  │ │                     │ │   │ com │ │ │",
            "│                  │ │                     │ │   │ man │ │ │",
            "│                  │ │                     │ │   │ d/t │ │ │",
            "│                  │ │                     │ │   │ ras │ │ │",
            "│                  │ │                     │ │   │ h-s │ │ │",
            "│                  │ │                     │ │   │ upp │ │ │",
            "│                  │ │                     │ │   │ ort │ │ │",
            "│                  │ │                     │ ╰───┴─────╯ │ │",
            "│                  │ │ wasi                │ [list 0     │ │",
            "│                  │ │                     │ items]      │ │",
            "│                  │ │                     │ ╭───┬─────╮ │ │",
            "│                  │ │ which-support       │ │ 0 │ nu- │ │ │",
            "│                  │ │                     │ │   │ com │ │ │",
            "│                  │ │                     │ │   │ man │ │ │",
            "│                  │ │                     │ │   │ d/w │ │ │",
            "│                  │ │                     │ │   │ hic │ │ │",
            "│                  │ │                     │ │   │ h-s │ │ │",
            "│                  │ │                     │ │   │ upp │ │ │",
            "│                  │ │                     │ │   │ ort │ │ │",
            "│                  │ │                     │ ╰───┴─────╯ │ │",
            "│                  │ ╰─────────────────────┴─────────────╯ │",
            "│                  │ ╭───────────────┬───────────────────╮ │",
            "│ package          │ │               │ ╭───┬───────────╮ │ │",
            "│                  │ │ authors       │ │ 0 │ The       │ │ │",
            "│                  │ │               │ │   │ Nushell   │ │ │",
            "│                  │ │               │ │   │ Project D │ │ │",
            "│                  │ │               │ │   │ evelopers │ │ │",
            "│                  │ │               │ ╰───┴───────────╯ │ │",
            "│                  │ │ default-run   │ nu                │ │",
            "│                  │ │ description   │ A new type of     │ │",
            "│                  │ │               │ shell             │ │",
            "│                  │ │ documentation │ https://www.nushe │ │",
            "│                  │ │               │ ll.sh/book/       │ │",
            "│                  │ │ edition       │ 2021              │ │",
            "│                  │ │               │ ╭───┬────────╮    │ │",
            "│                  │ │ exclude       │ │ 0 │ images │    │ │",
            "│                  │ │               │ ╰───┴────────╯    │ │",
            "│                  │ │ homepage      │ https://www.nushe │ │",
            "│                  │ │               │ ll.sh             │ │",
            "│                  │ │ license       │ MIT               │ │",
            "│                  │ │               │ ╭──────────┬────╮ │ │",
            "│                  │ │ metadata      │ │ binstall │ {r │ │ │",
            "│                  │ │               │ │          │ ec │ │ │",
            "│                  │ │               │ │          │ or │ │ │",
            "│                  │ │               │ │          │ d  │ │ │",
            "│                  │ │               │ │          │ 3  │ │ │",
            "│                  │ │               │ │          │ fi │ │ │",
            "│                  │ │               │ │          │ el │ │ │",
            "│                  │ │               │ │          │ ds │ │ │",
            "│                  │ │               │ │          │ }  │ │ │",
            "│                  │ │               │ ╰──────────┴────╯ │ │",
            "│                  │ │ name          │ nu                │ │",
            "│                  │ │ repository    │ https://github.co │ │",
            "│                  │ │               │ m/nushell/nushell │ │",
            "│                  │ │ rust-version  │ 1.60              │ │",
            "│                  │ │ version       │ 0.74.1            │ │",
            "│                  │ ╰───────────────┴───────────────────╯ │",
            "│                  │ ╭───────────┬───────────────────────╮ │",
            "│ patch            │ │           │ ╭──────────┬────────╮ │ │",
            "│                  │ │ crates-io │ │ reedline │ {recor │ │ │",
            "│                  │ │           │ │          │ d 2 fi │ │ │",
            "│                  │ │           │ │          │ elds}  │ │ │",
            "│                  │ │           │ ╰──────────┴────────╯ │ │",
            "│                  │ ╰───────────┴───────────────────────╯ │",
            "│ target           │ {record 3 fields}                     │",
            "│                  │ ╭─────────┬─────────────────────────╮ │",
            "│ workspace        │ │         │ ╭────┬────────────────╮ │ │",
            "│                  │ │ members │ │  0 │ crates/nu-cli  │ │ │",
            "│                  │ │         │ │  1 │ crates/nu-engi │ │ │",
            "│                  │ │         │ │    │ ne             │ │ │",
            "│                  │ │         │ │  2 │ crates/nu-pars │ │ │",
            "│                  │ │         │ │    │ er             │ │ │",
            "│                  │ │         │ │  3 │ crates/nu-syst │ │ │",
            "│                  │ │         │ │    │ em             │ │ │",
            "│                  │ │         │ │  4 │ crates/nu-comm │ │ │",
            "│                  │ │         │ │    │ and            │ │ │",
            "│                  │ │         │ │  5 │ crates/nu-prot │ │ │",
            "│                  │ │         │ │    │ ocol           │ │ │",
            "│                  │ │         │ │  6 │ crates/nu-plug │ │ │",
            "│                  │ │         │ │    │ in             │ │ │",
            "│                  │ │         │ │  7 │ crates/nu_plug │ │ │",
            "│                  │ │         │ │    │ in_inc         │ │ │",
            "│                  │ │         │ │  8 │ crates/nu_plug │ │ │",
            "│                  │ │         │ │    │ in_gstat       │ │ │",
            "│                  │ │         │ │  9 │ crates/nu_plug │ │ │",
            "│                  │ │         │ │    │ in_example     │ │ │",
            "│                  │ │         │ │ 10 │ crates/nu_plug │ │ │",
            "│                  │ │         │ │    │ in_query       │ │ │",
            "│                  │ │         │ │ 11 │ crates/nu_plug │ │ │",
            "│                  │ │         │ │    │ in_custom_valu │ │ │",
            "│                  │ │         │ │    │ es             │ │ │",
            "│                  │ │         │ │ 12 │ crates/nu-util │ │ │",
            "│                  │ │         │ │    │ s              │ │ │",
            "│                  │ │         │ ╰────┴────────────────╯ │ │",
            "│                  │ ╰─────────┴─────────────────────────╯ │",
            "╰──────────────────┴───────────────────────────────────────╯",
        ]);

        assert_eq!(actual.out, expected);

        let actual = nu!(
            cwd: dirs.test(), pipeline(
            "open sample.toml | table --expand --width=40"
        ));

        _print_lines(&actual.out, 40);

        let expected = join_lines([
            "╭──────────────────┬───────────────────╮",
            "│ bench            │ [table 1 row]     │",
            "│ bin              │ [table 1 row]     │",
            "│ dependencies     │ {record 13        │",
            "│                  │ fields}           │",
            "│ dev-dependencies │ {record 9 fields} │",
            "│ features         │ {record 8 fields} │",
            "│ package          │ {record 13        │",
            "│                  │ fields}           │",
            "│                  │ ╭───────────┬───╮ │",
            "│ patch            │ │ crates-io │ { │ │",
            "│                  │ │           │ r │ │",
            "│                  │ │           │ e │ │",
            "│                  │ │           │ c │ │",
            "│                  │ │           │ o │ │",
            "│                  │ │           │ r │ │",
            "│                  │ │           │ d │ │",
            "│                  │ │           │   │ │",
            "│                  │ │           │ 1 │ │",
            "│                  │ │           │   │ │",
            "│                  │ │           │ f │ │",
            "│                  │ │           │ i │ │",
            "│                  │ │           │ e │ │",
            "│                  │ │           │ l │ │",
            "│                  │ │           │ d │ │",
            "│                  │ │           │ } │ │",
            "│                  │ ╰───────────┴───╯ │",
            "│ target           │ {record 3 fields} │",
            "│                  │ ╭─────────┬─────╮ │",
            "│ workspace        │ │ members │ [li │ │",
            "│                  │ │         │ st  │ │",
            "│                  │ │         │ 13  │ │",
            "│                  │ │         │ ite │ │",
            "│                  │ │         │ ms] │ │",
            "│                  │ ╰─────────┴─────╯ │",
            "╰──────────────────┴───────────────────╯",
        ]);

        assert_eq!(actual.out, expected);
    })
}

#[test]
fn table_expande_with_no_header_internally_0() {
    let nu_value = r##"{ "config            ": { "ls": { "use_ls_colors": true, "clickable_links": false }, "rm": { "always_trash": false }, "cd": { "abbreviations": false }, "table": { "mode": "rounded", "index_mode": "always", "trim": { "methodology": "wrapping", "wrapping_try_keep_words": true, "truncating_suffix": "..." } }, "explore": { "help_banner": true, "exit_esc": true, "command_bar_text": "#C4C9C6", "status_bar_background": { "fg": "#1D1F21", "bg": "#C4C9C6" }, "highlight": { "bg": "yellow", "fg": "black" }, "status": {}, "try": {}, "table": { "split_line": "#404040", "cursor": true, "line_index": true, "line_shift": true, "line_head_top": true, "line_head_bottom": true, "show_head": true, "show_index": true }, "config": { "cursor_color": { "bg": "yellow", "fg": "black" } } }, "history": { "max_size": 10000, "sync_on_enter": true, "file_format": "plaintext" }, "completions": { "case_sensitive": false, "quick": true, "partial": true, "algorithm": "prefix", "external": { "enable": true, "max_results": 100, "completer": null } }, "filesize": { "metric": true, "format": "auto" }, "cursor_shape": { "emacs": "line", "vi_insert": "block", "vi_normal": "underscore" }, "color_config": { "separator": "white", "leading_trailing_space_bg": { "attr": "n" }, "header": "green_bold", "empty": "blue", "bool": null, "int": "white", "filesize": null, "duration": "white", "date": null, "range": "white", "float": "white", "string": "white", "nothing": "white", "binary": "white", "cellpath": "white", "row_index": "green_bold", "record": "white", "list": "white", "block": "white", "hints": "dark_gray", "search_result": {"fg": "white", "bg": "red"}, "shape_and": "purple_bold", "shape_binary": "purple_bold", "shape_block": "blue_bold", "shape_bool": "light_cyan", "shape_custom": "green", "shape_datetime": "cyan_bold", "shape_directory": "cyan", "shape_external": "cyan", "shape_externalarg": "green_bold", "shape_filepath": "cyan", "shape_flag": "blue_bold", "shape_float": "purple_bold", "shape_garbage": { "fg": "#FFFFFF", "bg": "#FF0000", "attr": "b" }, "shape_globpattern": "cyan_bold", "shape_int": "purple_bold", "shape_internalcall": "cyan_bold", "shape_list": "cyan_bold", "shape_literal": "blue", "shape_matching_brackets": { "attr": "u" }, "shape_nothing": "light_cyan", "shape_operator": "yellow", "shape_or": "purple_bold", "shape_pipe": "purple_bold", "shape_range": "yellow_bold", "shape_record": "cyan_bold", "shape_redirection": "purple_bold", "shape_signature": "green_bold", "shape_string": "green", "shape_string_interpolation": "cyan_bold", "shape_table": "blue_bold", "shape_variable": "purple" }, "use_grid_icons": true, "footer_mode": "25", "float_precision": 2, "use_ansi_coloring": true, "edit_mode": "emacs", "shell_integration": true, "show_banner": true, "render_right_prompt_on_last_line": false, "hooks": { "pre_prompt": [ null ], "pre_execution": [ null ], "env_change": { "PWD": [ null ] }, "display_output": null }, "menus": [ { "name": "completion_menu", "only_buffer_difference": false, "marker": "| ", "type": { "layout": "columnar", "columns": 4, "col_width": 20, "col_padding": 2 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" } }, { "name": "history_menu", "only_buffer_difference": true, "marker": "? ", "type": { "layout": "list", "page_size": 10 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" } }, { "name": "help_menu", "only_buffer_difference": true, "marker": "? ", "type": { "layout": "description", "columns": 4, "col_width": 20, "col_padding": 2, "selection_rows": 4, "description_rows": 10 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" } }, { "name": "commands_menu", "only_buffer_difference": false, "marker": "# ", "type": { "layout": "columnar", "columns": 4, "col_width": 20, "col_padding": 2 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" }, "source": null }, { "name": "vars_menu", "only_buffer_difference": true, "marker": "# ", "type": { "layout": "list", "page_size": 10 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" }, "source": null }, { "name": "commands_with_description", "only_buffer_difference": true, "marker": "# ", "type": { "layout": "description", "columns": 4, "col_width": 20, "col_padding": 2, "selection_rows": 4, "description_rows": 10 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" }, "source": null } ], "keybindings": [ { "name": "completion_menu", "modifier": "none", "keycode": "tab", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "until": [ { "send": "menu", "name": "completion_menu" }, { "send": "menunext" } ] } }, { "name": "completion_previous", "modifier": "shift", "keycode": "backtab", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "send": "menuprevious" } }, { "name": "history_menu", "modifier": "control", "keycode": "char_r", "mode": "emacs", "event": { "send": "menu", "name": "history_menu" } }, { "name": "next_page", "modifier": "control", "keycode": "char_x", "mode": "emacs", "event": { "send": "menupagenext" } }, { "name": "undo_or_previous_page", "modifier": "control", "keycode": "char_z", "mode": "emacs", "event": { "until": [ { "send": "menupageprevious" }, { "edit": "undo" } ] } }, { "name": "yank", "modifier": "control", "keycode": "char_y", "mode": "emacs", "event": { "until": [ { "edit": "pastecutbufferafter" } ] } }, { "name": "unix-line-discard", "modifier": "control", "keycode": "char_u", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "until": [ { "edit": "cutfromlinestart" } ] } }, { "name": "kill-line", "modifier": "control", "keycode": "char_k", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "until": [ { "edit": "cuttolineend" } ] } }, { "name": "commands_menu", "modifier": "control", "keycode": "char_t", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "send": "menu", "name": "commands_menu" } }, { "name": "vars_menu", "modifier": "alt", "keycode": "char_o", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "send": "menu", "name": "vars_menu" } }, { "name": "commands_with_description", "modifier": "control", "keycode": "char_s", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "send": "menu", "name": "commands_with_description" } } ] } }"##;

    let actual = nu!(format!("{} | table --expand --width 141", nu_value.trim()));

    assert_eq!(
        actual.out,
        join_lines([
            "╭────────────────────┬──────────────────────────────────────────────────────────────────────────────────────────────────────────────────────╮",
            "│                    │ ╭──────────────────────────────────┬───────────────────────────────────────────────────────────────────────────────╮ │",
            "│ config             │ │                                  │ ╭─────────────────┬───────╮                                                   │ │",
            "│                    │ │ ls                               │ │ use_ls_colors   │ true  │                                                   │ │",
            "│                    │ │                                  │ │ clickable_links │ false │                                                   │ │",
            "│                    │ │                                  │ ╰─────────────────┴───────╯                                                   │ │",
            "│                    │ │                                  │ ╭──────────────┬───────╮                                                      │ │",
            "│                    │ │ rm                               │ │ always_trash │ false │                                                      │ │",
            "│                    │ │                                  │ ╰──────────────┴───────╯                                                      │ │",
            "│                    │ │                                  │ ╭───────────────┬───────╮                                                     │ │",
            "│                    │ │ cd                               │ │ abbreviations │ false │                                                     │ │",
            "│                    │ │                                  │ ╰───────────────┴───────╯                                                     │ │",
            "│                    │ │                                  │ ╭────────────┬────────────────────────────────────────╮                       │ │",
            "│                    │ │ table                            │ │ mode       │ rounded                                │                       │ │",
            "│                    │ │                                  │ │ index_mode │ always                                 │                       │ │",
            "│                    │ │                                  │ │            │ ╭─────────────────────────┬──────────╮ │                       │ │",
            "│                    │ │                                  │ │ trim       │ │ methodology             │ wrapping │ │                       │ │",
            "│                    │ │                                  │ │            │ │ wrapping_try_keep_words │ true     │ │                       │ │",
            "│                    │ │                                  │ │            │ │ truncating_suffix       │ ...      │ │                       │ │",
            "│                    │ │                                  │ │            │ ╰─────────────────────────┴──────────╯ │                       │ │",
            "│                    │ │                                  │ ╰────────────┴────────────────────────────────────────╯                       │ │",
            "│                    │ │                                  │ ╭───────────────────────────────┬───────────────────────────────────────────╮ │ │",
            "│                    │ │ explore                          │ │ help_banner                   │ true                                      │ │ │",
            "│                    │ │                                  │ │ exit_esc                      │ true                                      │ │ │",
            "│                    │ │                                  │ │ command_bar_text              │ #C4C9C6                                   │ │ │",
            "│                    │ │                                  │ │                               │ ╭────┬─────────╮                          │ │ │",
            "│                    │ │                                  │ │ status_bar_background         │ │ fg │ #1D1F21 │                          │ │ │",
            "│                    │ │                                  │ │                               │ │ bg │ #C4C9C6 │                          │ │ │",
            "│                    │ │                                  │ │                               │ ╰────┴─────────╯                          │ │ │",
            "│                    │ │                                  │ │                               │ ╭────┬────────╮                           │ │ │",
            "│                    │ │                                  │ │ highlight                     │ │ bg │ yellow │                           │ │ │",
            "│                    │ │                                  │ │                               │ │ fg │ black  │                           │ │ │",
            "│                    │ │                                  │ │                               │ ╰────┴────────╯                           │ │ │",
            "│                    │ │                                  │ │ status                        │ {record 0 fields}                         │ │ │",
            "│                    │ │                                  │ │ try                           │ {record 0 fields}                         │ │ │",
            "│                    │ │                                  │ │                               │ ╭──────────────────┬─────────╮            │ │ │",
            "│                    │ │                                  │ │ table                         │ │ split_line       │ #404040 │            │ │ │",
            "│                    │ │                                  │ │                               │ │ cursor           │ true    │            │ │ │",
            "│                    │ │                                  │ │                               │ │ line_index       │ true    │            │ │ │",
            "│                    │ │                                  │ │                               │ │ line_shift       │ true    │            │ │ │",
            "│                    │ │                                  │ │                               │ │ line_head_top    │ true    │            │ │ │",
            "│                    │ │                                  │ │                               │ │ line_head_bottom │ true    │            │ │ │",
            "│                    │ │                                  │ │                               │ │ show_head        │ true    │            │ │ │",
            "│                    │ │                                  │ │                               │ │ show_index       │ true    │            │ │ │",
            "│                    │ │                                  │ │                               │ ╰──────────────────┴─────────╯            │ │ │",
            "│                    │ │                                  │ │                               │ ╭──────────────┬─────────────────╮        │ │ │",
            "│                    │ │                                  │ │ config                        │ │              │ ╭────┬────────╮ │        │ │ │",
            "│                    │ │                                  │ │                               │ │ cursor_color │ │ bg │ yellow │ │        │ │ │",
            "│                    │ │                                  │ │                               │ │              │ │ fg │ black  │ │        │ │ │",
            "│                    │ │                                  │ │                               │ │              │ ╰────┴────────╯ │        │ │ │",
            "│                    │ │                                  │ │                               │ ╰──────────────┴─────────────────╯        │ │ │",
            "│                    │ │                                  │ ╰───────────────────────────────┴───────────────────────────────────────────╯ │ │",
            "│                    │ │                                  │ ╭───────────────┬───────────╮                                                 │ │",
            "│                    │ │ history                          │ │ max_size      │ 10000     │                                                 │ │",
            "│                    │ │                                  │ │ sync_on_enter │ true      │                                                 │ │",
            "│                    │ │                                  │ │ file_format   │ plaintext │                                                 │ │",
            "│                    │ │                                  │ ╰───────────────┴───────────╯                                                 │ │",
            "│                    │ │                                  │ ╭────────────────┬────────────────────────╮                                   │ │",
            "│                    │ │ completions                      │ │ case_sensitive │ false                  │                                   │ │",
            "│                    │ │                                  │ │ quick          │ true                   │                                   │ │",
            "│                    │ │                                  │ │ partial        │ true                   │                                   │ │",
            "│                    │ │                                  │ │ algorithm      │ prefix                 │                                   │ │",
            "│                    │ │                                  │ │                │ ╭─────────────┬──────╮ │                                   │ │",
            "│                    │ │                                  │ │ external       │ │ enable      │ true │ │                                   │ │",
            "│                    │ │                                  │ │                │ │ max_results │ 100  │ │                                   │ │",
            "│                    │ │                                  │ │                │ │ completer   │      │ │                                   │ │",
            "│                    │ │                                  │ │                │ ╰─────────────┴──────╯ │                                   │ │",
            "│                    │ │                                  │ ╰────────────────┴────────────────────────╯                                   │ │",
            "│                    │ │                                  │ ╭────────┬──────╮                                                             │ │",
            "│                    │ │ filesize                         │ │ metric │ true │                                                             │ │",
            "│                    │ │                                  │ │ format │ auto │                                                             │ │",
            "│                    │ │                                  │ ╰────────┴──────╯                                                             │ │",
            "│                    │ │                                  │ ╭───────────┬────────────╮                                                    │ │",
            "│                    │ │ cursor_shape                     │ │ emacs     │ line       │                                                    │ │",
            "│                    │ │                                  │ │ vi_insert │ block      │                                                    │ │",
            "│                    │ │                                  │ │ vi_normal │ underscore │                                                    │ │",
            "│                    │ │                                  │ ╰───────────┴────────────╯                                                    │ │",
            "│                    │ │                                  │ ╭────────────────────────────┬────────────────────╮                           │ │",
            "│                    │ │ color_config                     │ │ separator                  │ white              │                           │ │",
            "│                    │ │                                  │ │                            │ ╭──────┬───╮       │                           │ │",
            "│                    │ │                                  │ │ leading_trailing_space_bg  │ │ attr │ n │       │                           │ │",
            "│                    │ │                                  │ │                            │ ╰──────┴───╯       │                           │ │",
            "│                    │ │                                  │ │ header                     │ green_bold         │                           │ │",
            "│                    │ │                                  │ │ empty                      │ blue               │                           │ │",
            "│                    │ │                                  │ │ bool                       │                    │                           │ │",
            "│                    │ │                                  │ │ int                        │ white              │                           │ │",
            "│                    │ │                                  │ │ filesize                   │                    │                           │ │",
            "│                    │ │                                  │ │ duration                   │ white              │                           │ │",
            "│                    │ │                                  │ │ date                       │                    │                           │ │",
            "│                    │ │                                  │ │ range                      │ white              │                           │ │",
            "│                    │ │                                  │ │ float                      │ white              │                           │ │",
            "│                    │ │                                  │ │ string                     │ white              │                           │ │",
            "│                    │ │                                  │ │ nothing                    │ white              │                           │ │",
            "│                    │ │                                  │ │ binary                     │ white              │                           │ │",
            "│                    │ │                                  │ │ cellpath                   │ white              │                           │ │",
            "│                    │ │                                  │ │ row_index                  │ green_bold         │                           │ │",
            "│                    │ │                                  │ │ record                     │ white              │                           │ │",
            "│                    │ │                                  │ │ list                       │ white              │                           │ │",
            "│                    │ │                                  │ │ block                      │ white              │                           │ │",
            "│                    │ │                                  │ │ hints                      │ dark_gray          │                           │ │",
            "│                    │ │                                  │ │                            │ ╭────┬───────╮     │                           │ │",
            "│                    │ │                                  │ │ search_result              │ │ fg │ white │     │                           │ │",
            "│                    │ │                                  │ │                            │ │ bg │ red   │     │                           │ │",
            "│                    │ │                                  │ │                            │ ╰────┴───────╯     │                           │ │",
            "│                    │ │                                  │ │ shape_and                  │ purple_bold        │                           │ │",
            "│                    │ │                                  │ │ shape_binary               │ purple_bold        │                           │ │",
            "│                    │ │                                  │ │ shape_block                │ blue_bold          │                           │ │",
            "│                    │ │                                  │ │ shape_bool                 │ light_cyan         │                           │ │",
            "│                    │ │                                  │ │ shape_custom               │ green              │                           │ │",
            "│                    │ │                                  │ │ shape_datetime             │ cyan_bold          │                           │ │",
            "│                    │ │                                  │ │ shape_directory            │ cyan               │                           │ │",
            "│                    │ │                                  │ │ shape_external             │ cyan               │                           │ │",
            "│                    │ │                                  │ │ shape_externalarg          │ green_bold         │                           │ │",
            "│                    │ │                                  │ │ shape_filepath             │ cyan               │                           │ │",
            "│                    │ │                                  │ │ shape_flag                 │ blue_bold          │                           │ │",
            "│                    │ │                                  │ │ shape_float                │ purple_bold        │                           │ │",
            "│                    │ │                                  │ │                            │ ╭──────┬─────────╮ │                           │ │",
            "│                    │ │                                  │ │ shape_garbage              │ │ fg   │ #FFFFFF │ │                           │ │",
            "│                    │ │                                  │ │                            │ │ bg   │ #FF0000 │ │                           │ │",
            "│                    │ │                                  │ │                            │ │ attr │ b       │ │                           │ │",
            "│                    │ │                                  │ │                            │ ╰──────┴─────────╯ │                           │ │",
            "│                    │ │                                  │ │ shape_globpattern          │ cyan_bold          │                           │ │",
            "│                    │ │                                  │ │ shape_int                  │ purple_bold        │                           │ │",
            "│                    │ │                                  │ │ shape_internalcall         │ cyan_bold          │                           │ │",
            "│                    │ │                                  │ │ shape_list                 │ cyan_bold          │                           │ │",
            "│                    │ │                                  │ │ shape_literal              │ blue               │                           │ │",
            "│                    │ │                                  │ │                            │ ╭──────┬───╮       │                           │ │",
            "│                    │ │                                  │ │ shape_matching_brackets    │ │ attr │ u │       │                           │ │",
            "│                    │ │                                  │ │                            │ ╰──────┴───╯       │                           │ │",
            "│                    │ │                                  │ │ shape_nothing              │ light_cyan         │                           │ │",
            "│                    │ │                                  │ │ shape_operator             │ yellow             │                           │ │",
            "│                    │ │                                  │ │ shape_or                   │ purple_bold        │                           │ │",
            "│                    │ │                                  │ │ shape_pipe                 │ purple_bold        │                           │ │",
            "│                    │ │                                  │ │ shape_range                │ yellow_bold        │                           │ │",
            "│                    │ │                                  │ │ shape_record               │ cyan_bold          │                           │ │",
            "│                    │ │                                  │ │ shape_redirection          │ purple_bold        │                           │ │",
            "│                    │ │                                  │ │ shape_signature            │ green_bold         │                           │ │",
            "│                    │ │                                  │ │ shape_string               │ green              │                           │ │",
            "│                    │ │                                  │ │ shape_string_interpolation │ cyan_bold          │                           │ │",
            "│                    │ │                                  │ │ shape_table                │ blue_bold          │                           │ │",
            "│                    │ │                                  │ │ shape_variable             │ purple             │                           │ │",
            "│                    │ │                                  │ ╰────────────────────────────┴────────────────────╯                           │ │",
            "│                    │ │ use_grid_icons                   │ true                                                                          │ │",
            "│                    │ │ footer_mode                      │ 25                                                                            │ │",
            "│                    │ │ float_precision                  │ 2                                                                             │ │",
            "│                    │ │ use_ansi_coloring                │ true                                                                          │ │",
            "│                    │ │ edit_mode                        │ emacs                                                                         │ │",
            "│                    │ │ shell_integration                │ true                                                                          │ │",
            "│                    │ │ show_banner                      │ true                                                                          │ │",
            "│                    │ │ render_right_prompt_on_last_line │ false                                                                         │ │",
            "│                    │ │                                  │ ╭────────────────┬────────────────────╮                                       │ │",
            "│                    │ │ hooks                            │ │                │ ╭───┬──╮           │                                       │ │",
            "│                    │ │                                  │ │ pre_prompt     │ │ 0 │  │           │                                       │ │",
            "│                    │ │                                  │ │                │ ╰───┴──╯           │                                       │ │",
            "│                    │ │                                  │ │                │ ╭───┬──╮           │                                       │ │",
            "│                    │ │                                  │ │ pre_execution  │ │ 0 │  │           │                                       │ │",
            "│                    │ │                                  │ │                │ ╰───┴──╯           │                                       │ │",
            "│                    │ │                                  │ │                │ ╭─────┬──────────╮ │                                       │ │",
            "│                    │ │                                  │ │ env_change     │ │     │ ╭───┬──╮ │ │                                       │ │",
            "│                    │ │                                  │ │                │ │ PWD │ │ 0 │  │ │ │                                       │ │",
            "│                    │ │                                  │ │                │ │     │ ╰───┴──╯ │ │                                       │ │",
            "│                    │ │                                  │ │                │ ╰─────┴──────────╯ │                                       │ │",
            "│                    │ │                                  │ │ display_output │                    │                                       │ │",
            "│                    │ │                                  │ ╰────────────────┴────────────────────╯                                       │ │",
            "│                    │ │                                  │ ╭───┬───────────────────────────┬────────────────────────┬────────┬─────╮     │ │",
            "│                    │ │ menus                            │ │ # │           name            │ only_buffer_difference │ marker │ ... │     │ │",
            "│                    │ │                                  │ ├───┼───────────────────────────┼────────────────────────┼────────┼─────┤     │ │",
            "│                    │ │                                  │ │ 0 │ completion_menu           │ false                  │ |      │ ... │     │ │",
            "│                    │ │                                  │ │ 1 │ history_menu              │ true                   │ ?      │ ... │     │ │",
            "│                    │ │                                  │ │ 2 │ help_menu                 │ true                   │ ?      │ ... │     │ │",
            "│                    │ │                                  │ │ 3 │ commands_menu             │ false                  │ #      │ ... │     │ │",
            "│                    │ │                                  │ │ 4 │ vars_menu                 │ true                   │ #      │ ... │     │ │",
            "│                    │ │                                  │ │ 5 │ commands_with_description │ true                   │ #      │ ... │     │ │",
            "│                    │ │                                  │ ╰───┴───────────────────────────┴────────────────────────┴────────┴─────╯     │ │",
            "│                    │ │                                  │ ╭────┬───────────────────────────┬──────────┬─────────┬───────────────┬─────╮ │ │",
            "│                    │ │ keybindings                      │ │  # │           name            │ modifier │ keycode │     mode      │ ... │ │ │",
            "│                    │ │                                  │ ├────┼───────────────────────────┼──────────┼─────────┼───────────────┼─────┤ │ │",
            "│                    │ │                                  │ │  0 │ completion_menu           │ none     │ tab     │ ╭───┬───────╮ │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 0 │ emacs │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 1 │ vi_no │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ rmal  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 2 │ vi_in │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ sert  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ ╰───┴───────╯ │     │ │ │",
            "│                    │ │                                  │ │  1 │ completion_previous       │ shift    │ backtab │ ╭───┬───────╮ │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 0 │ emacs │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 1 │ vi_no │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ rmal  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 2 │ vi_in │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ sert  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ ╰───┴───────╯ │     │ │ │",
            "│                    │ │                                  │ │  2 │ history_menu              │ control  │ char_r  │ emacs         │ ... │ │ │",
            "│                    │ │                                  │ │  3 │ next_page                 │ control  │ char_x  │ emacs         │ ... │ │ │",
            "│                    │ │                                  │ │  4 │ undo_or_previous_page     │ control  │ char_z  │ emacs         │ ... │ │ │",
            "│                    │ │                                  │ │  5 │ yank                      │ control  │ char_y  │ emacs         │ ... │ │ │",
            "│                    │ │                                  │ │  6 │ unix-line-discard         │ control  │ char_u  │ ╭───┬───────╮ │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 0 │ emacs │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 1 │ vi_no │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ rmal  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 2 │ vi_in │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ sert  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ ╰───┴───────╯ │     │ │ │",
            "│                    │ │                                  │ │  7 │ kill-line                 │ control  │ char_k  │ ╭───┬───────╮ │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 0 │ emacs │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 1 │ vi_no │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ rmal  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 2 │ vi_in │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ sert  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ ╰───┴───────╯ │     │ │ │",
            "│                    │ │                                  │ │  8 │ commands_menu             │ control  │ char_t  │ ╭───┬───────╮ │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 0 │ emacs │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 1 │ vi_no │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ rmal  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 2 │ vi_in │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ sert  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ ╰───┴───────╯ │     │ │ │",
            "│                    │ │                                  │ │  9 │ vars_menu                 │ alt      │ char_o  │ ╭───┬───────╮ │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 0 │ emacs │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 1 │ vi_no │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ rmal  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 2 │ vi_in │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ sert  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ ╰───┴───────╯ │     │ │ │",
            "│                    │ │                                  │ │ 10 │ commands_with_description │ control  │ char_s  │ ╭───┬───────╮ │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 0 │ emacs │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 1 │ vi_no │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ rmal  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │ 2 │ vi_in │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ │   │ sert  │ │     │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ ╰───┴───────╯ │     │ │ │",
            "│                    │ │                                  │ ╰────┴───────────────────────────┴──────────┴─────────┴───────────────┴─────╯ │ │",
            "│                    │ ╰──────────────────────────────────┴───────────────────────────────────────────────────────────────────────────────╯ │",
            "╰────────────────────┴──────────────────────────────────────────────────────────────────────────────────────────────────────────────────────╯",
        ])
    );
}

#[test]
fn table_expande_with_no_header_internally_1() {
    let nu_value = r##"{ "config            ": { "ls": { "use_ls_colors": true, "clickable_links": false }, "rm": { "always_trash": false }, "cd": { "abbreviations": false }, "table": { "mode": "rounded", "index_mode": "always", "trim": { "methodology": "wrapping", "wrapping_try_keep_words": true, "truncating_suffix": "..." } }, "explore": { "help_banner": true, "exit_esc": true, "command_bar_text": "#C4C9C6", "status_bar_background": { "fg": "#1D1F21", "bg": "#C4C9C6" }, "highlight": { "bg": "yellow", "fg": "black" }, "status": {}, "try": {}, "table": { "split_line": "#404040", "cursor": true, "line_index": true, "line_shift": true, "line_head_top": true, "line_head_bottom": true, "show_head": true, "show_index": true }, "config": { "cursor_color": { "bg": "yellow", "fg": "black" } } }, "history": { "max_size": 10000, "sync_on_enter": true, "file_format": "plaintext" }, "completions": { "case_sensitive": false, "quick": true, "partial": true, "algorithm": "prefix", "external": { "enable": true, "max_results": 100, "completer": null } }, "filesize": { "metric": true, "format": "auto" }, "cursor_shape": { "emacs": "line", "vi_insert": "block", "vi_normal": "underscore" }, "color_config": { "separator": "white", "leading_trailing_space_bg": { "attr": "n" }, "header": "green_bold", "empty": "blue", "bool": null, "int": "white", "filesize": null, "duration": "white", "date": null, "range": "white", "float": "white", "string": "white", "nothing": "white", "binary": "white", "cellpath": "white", "row_index": "green_bold", "record": "white", "list": "white", "block": "white", "hints": "dark_gray", "search_result": {"fg": "white", "bg": "red"}, "shape_and": "purple_bold", "shape_binary": "purple_bold", "shape_block": "blue_bold", "shape_bool": "light_cyan", "shape_custom": "green", "shape_datetime": "cyan_bold", "shape_directory": "cyan", "shape_external": "cyan", "shape_externalarg": "green_bold", "shape_filepath": "cyan", "shape_flag": "blue_bold", "shape_float": "purple_bold", "shape_garbage": { "fg": "#FFFFFF", "bg": "#FF0000", "attr": "b" }, "shape_globpattern": "cyan_bold", "shape_int": "purple_bold", "shape_internalcall": "cyan_bold", "shape_list": "cyan_bold", "shape_literal": "blue", "shape_matching_brackets": { "attr": "u" }, "shape_nothing": "light_cyan", "shape_operator": "yellow", "shape_or": "purple_bold", "shape_pipe": "purple_bold", "shape_range": "yellow_bold", "shape_record": "cyan_bold", "shape_redirection": "purple_bold", "shape_signature": "green_bold", "shape_string": "green", "shape_string_interpolation": "cyan_bold", "shape_table": "blue_bold", "shape_variable": "purple" }, "use_grid_icons": true, "footer_mode": "25", "float_precision": 2, "use_ansi_coloring": true, "edit_mode": "emacs", "shell_integration": true, "show_banner": true, "render_right_prompt_on_last_line": false, "hooks": { "pre_prompt": [ null ], "pre_execution": [ null ], "env_change": { "PWD": [ null ] }, "display_output": null }, "menus": [ { "name": "completion_menu", "only_buffer_difference": false, "marker": "| ", "type": { "layout": "columnar", "columns": 4, "col_width": 20, "col_padding": 2 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" } }, { "name": "history_menu", "only_buffer_difference": true, "marker": "? ", "type": { "layout": "list", "page_size": 10 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" } }, { "name": "help_menu", "only_buffer_difference": true, "marker": "? ", "type": { "layout": "description", "columns": 4, "col_width": 20, "col_padding": 2, "selection_rows": 4, "description_rows": 10 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" } }, { "name": "commands_menu", "only_buffer_difference": false, "marker": "# ", "type": { "layout": "columnar", "columns": 4, "col_width": 20, "col_padding": 2 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" }, "source": null }, { "name": "vars_menu", "only_buffer_difference": true, "marker": "# ", "type": { "layout": "list", "page_size": 10 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" }, "source": null }, { "name": "commands_with_description", "only_buffer_difference": true, "marker": "# ", "type": { "layout": "description", "columns": 4, "col_width": 20, "col_padding": 2, "selection_rows": 4, "description_rows": 10 }, "style": { "text": "green", "selected_text": "green_reverse", "description_text": "yellow" }, "source": null } ], "keybindings": [ { "name": "completion_menu", "modifier": "none", "keycode": "tab", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "until": [ { "send": "menu", "name": "completion_menu" }, { "send": "menunext" } ] } }, { "name": "completion_previous", "modifier": "shift", "keycode": "backtab", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "send": "menuprevious" } }, { "name": "history_menu", "modifier": "control", "keycode": "char_r", "mode": "emacs", "event": { "send": "menu", "name": "history_menu" } }, { "name": "next_page", "modifier": "control", "keycode": "char_x", "mode": "emacs", "event": { "send": "menupagenext" } }, { "name": "undo_or_previous_page", "modifier": "control", "keycode": "char_z", "mode": "emacs", "event": { "until": [ { "send": "menupageprevious" }, { "edit": "undo" } ] } }, { "name": "yank", "modifier": "control", "keycode": "char_y", "mode": "emacs", "event": { "until": [ { "edit": "pastecutbufferafter" } ] } }, { "name": "unix-line-discard", "modifier": "control", "keycode": "char_u", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "until": [ { "edit": "cutfromlinestart" } ] } }, { "name": "kill-line", "modifier": "control", "keycode": "char_k", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "until": [ { "edit": "cuttolineend" } ] } }, { "name": "commands_menu", "modifier": "control", "keycode": "char_t", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "send": "menu", "name": "commands_menu" } }, { "name": "vars_menu", "modifier": "alt", "keycode": "char_o", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "send": "menu", "name": "vars_menu" } }, { "name": "commands_with_description", "modifier": "control", "keycode": "char_s", "mode": [ "emacs", "vi_normal", "vi_insert" ], "event": { "send": "menu", "name": "commands_with_description" } } ] } }"##;

    let actual = nu!(format!("{} | table --expand --width 136", nu_value.trim()));

    assert_eq!(
        actual.out,
        join_lines([
            "╭────────────────────┬─────────────────────────────────────────────────────────────────────────────────────────────────────────────────╮",
            "│                    │ ╭──────────────────────────────────┬──────────────────────────────────────────────────────────────────────────╮ │",
            "│ config             │ │                                  │ ╭─────────────────┬───────╮                                              │ │",
            "│                    │ │ ls                               │ │ use_ls_colors   │ true  │                                              │ │",
            "│                    │ │                                  │ │ clickable_links │ false │                                              │ │",
            "│                    │ │                                  │ ╰─────────────────┴───────╯                                              │ │",
            "│                    │ │                                  │ ╭──────────────┬───────╮                                                 │ │",
            "│                    │ │ rm                               │ │ always_trash │ false │                                                 │ │",
            "│                    │ │                                  │ ╰──────────────┴───────╯                                                 │ │",
            "│                    │ │                                  │ ╭───────────────┬───────╮                                                │ │",
            "│                    │ │ cd                               │ │ abbreviations │ false │                                                │ │",
            "│                    │ │                                  │ ╰───────────────┴───────╯                                                │ │",
            "│                    │ │                                  │ ╭────────────┬────────────────────────────────────────╮                  │ │",
            "│                    │ │ table                            │ │ mode       │ rounded                                │                  │ │",
            "│                    │ │                                  │ │ index_mode │ always                                 │                  │ │",
            "│                    │ │                                  │ │            │ ╭─────────────────────────┬──────────╮ │                  │ │",
            "│                    │ │                                  │ │ trim       │ │ methodology             │ wrapping │ │                  │ │",
            "│                    │ │                                  │ │            │ │ wrapping_try_keep_words │ true     │ │                  │ │",
            "│                    │ │                                  │ │            │ │ truncating_suffix       │ ...      │ │                  │ │",
            "│                    │ │                                  │ │            │ ╰─────────────────────────┴──────────╯ │                  │ │",
            "│                    │ │                                  │ ╰────────────┴────────────────────────────────────────╯                  │ │",
            "│                    │ │                                  │ ╭────────────────────────────┬─────────────────────────────────────────╮ │ │",
            "│                    │ │ explore                          │ │ help_banner                │ true                                    │ │ │",
            "│                    │ │                                  │ │ exit_esc                   │ true                                    │ │ │",
            "│                    │ │                                  │ │ command_bar_text           │ #C4C9C6                                 │ │ │",
            "│                    │ │                                  │ │                            │ ╭────┬─────────╮                        │ │ │",
            "│                    │ │                                  │ │ status_bar_background      │ │ fg │ #1D1F21 │                        │ │ │",
            "│                    │ │                                  │ │                            │ │ bg │ #C4C9C6 │                        │ │ │",
            "│                    │ │                                  │ │                            │ ╰────┴─────────╯                        │ │ │",
            "│                    │ │                                  │ │                            │ ╭────┬────────╮                         │ │ │",
            "│                    │ │                                  │ │ highlight                  │ │ bg │ yellow │                         │ │ │",
            "│                    │ │                                  │ │                            │ │ fg │ black  │                         │ │ │",
            "│                    │ │                                  │ │                            │ ╰────┴────────╯                         │ │ │",
            "│                    │ │                                  │ │ status                     │ {record 0 fields}                       │ │ │",
            "│                    │ │                                  │ │ try                        │ {record 0 fields}                       │ │ │",
            "│                    │ │                                  │ │                            │ ╭──────────────────┬─────────╮          │ │ │",
            "│                    │ │                                  │ │ table                      │ │ split_line       │ #404040 │          │ │ │",
            "│                    │ │                                  │ │                            │ │ cursor           │ true    │          │ │ │",
            "│                    │ │                                  │ │                            │ │ line_index       │ true    │          │ │ │",
            "│                    │ │                                  │ │                            │ │ line_shift       │ true    │          │ │ │",
            "│                    │ │                                  │ │                            │ │ line_head_top    │ true    │          │ │ │",
            "│                    │ │                                  │ │                            │ │ line_head_bottom │ true    │          │ │ │",
            "│                    │ │                                  │ │                            │ │ show_head        │ true    │          │ │ │",
            "│                    │ │                                  │ │                            │ │ show_index       │ true    │          │ │ │",
            "│                    │ │                                  │ │                            │ ╰──────────────────┴─────────╯          │ │ │",
            "│                    │ │                                  │ │                            │ ╭──────────────┬─────────────────╮      │ │ │",
            "│                    │ │                                  │ │ config                     │ │              │ ╭────┬────────╮ │      │ │ │",
            "│                    │ │                                  │ │                            │ │ cursor_color │ │ bg │ yellow │ │      │ │ │",
            "│                    │ │                                  │ │                            │ │              │ │ fg │ black  │ │      │ │ │",
            "│                    │ │                                  │ │                            │ │              │ ╰────┴────────╯ │      │ │ │",
            "│                    │ │                                  │ │                            │ ╰──────────────┴─────────────────╯      │ │ │",
            "│                    │ │                                  │ ╰────────────────────────────┴─────────────────────────────────────────╯ │ │",
            "│                    │ │                                  │ ╭───────────────┬───────────╮                                            │ │",
            "│                    │ │ history                          │ │ max_size      │ 10000     │                                            │ │",
            "│                    │ │                                  │ │ sync_on_enter │ true      │                                            │ │",
            "│                    │ │                                  │ │ file_format   │ plaintext │                                            │ │",
            "│                    │ │                                  │ ╰───────────────┴───────────╯                                            │ │",
            "│                    │ │                                  │ ╭────────────────┬────────────────────────╮                              │ │",
            "│                    │ │ completions                      │ │ case_sensitive │ false                  │                              │ │",
            "│                    │ │                                  │ │ quick          │ true                   │                              │ │",
            "│                    │ │                                  │ │ partial        │ true                   │                              │ │",
            "│                    │ │                                  │ │ algorithm      │ prefix                 │                              │ │",
            "│                    │ │                                  │ │                │ ╭─────────────┬──────╮ │                              │ │",
            "│                    │ │                                  │ │ external       │ │ enable      │ true │ │                              │ │",
            "│                    │ │                                  │ │                │ │ max_results │ 100  │ │                              │ │",
            "│                    │ │                                  │ │                │ │ completer   │      │ │                              │ │",
            "│                    │ │                                  │ │                │ ╰─────────────┴──────╯ │                              │ │",
            "│                    │ │                                  │ ╰────────────────┴────────────────────────╯                              │ │",
            "│                    │ │                                  │ ╭────────┬──────╮                                                        │ │",
            "│                    │ │ filesize                         │ │ metric │ true │                                                        │ │",
            "│                    │ │                                  │ │ format │ auto │                                                        │ │",
            "│                    │ │                                  │ ╰────────┴──────╯                                                        │ │",
            "│                    │ │                                  │ ╭───────────┬────────────╮                                               │ │",
            "│                    │ │ cursor_shape                     │ │ emacs     │ line       │                                               │ │",
            "│                    │ │                                  │ │ vi_insert │ block      │                                               │ │",
            "│                    │ │                                  │ │ vi_normal │ underscore │                                               │ │",
            "│                    │ │                                  │ ╰───────────┴────────────╯                                               │ │",
            "│                    │ │                                  │ ╭────────────────────────────┬────────────────────╮                      │ │",
            "│                    │ │ color_config                     │ │ separator                  │ white              │                      │ │",
            "│                    │ │                                  │ │                            │ ╭──────┬───╮       │                      │ │",
            "│                    │ │                                  │ │ leading_trailing_space_bg  │ │ attr │ n │       │                      │ │",
            "│                    │ │                                  │ │                            │ ╰──────┴───╯       │                      │ │",
            "│                    │ │                                  │ │ header                     │ green_bold         │                      │ │",
            "│                    │ │                                  │ │ empty                      │ blue               │                      │ │",
            "│                    │ │                                  │ │ bool                       │                    │                      │ │",
            "│                    │ │                                  │ │ int                        │ white              │                      │ │",
            "│                    │ │                                  │ │ filesize                   │                    │                      │ │",
            "│                    │ │                                  │ │ duration                   │ white              │                      │ │",
            "│                    │ │                                  │ │ date                       │                    │                      │ │",
            "│                    │ │                                  │ │ range                      │ white              │                      │ │",
            "│                    │ │                                  │ │ float                      │ white              │                      │ │",
            "│                    │ │                                  │ │ string                     │ white              │                      │ │",
            "│                    │ │                                  │ │ nothing                    │ white              │                      │ │",
            "│                    │ │                                  │ │ binary                     │ white              │                      │ │",
            "│                    │ │                                  │ │ cellpath                   │ white              │                      │ │",
            "│                    │ │                                  │ │ row_index                  │ green_bold         │                      │ │",
            "│                    │ │                                  │ │ record                     │ white              │                      │ │",
            "│                    │ │                                  │ │ list                       │ white              │                      │ │",
            "│                    │ │                                  │ │ block                      │ white              │                      │ │",
            "│                    │ │                                  │ │ hints                      │ dark_gray          │                      │ │",
            "│                    │ │                                  │ │                            │ ╭────┬───────╮     │                      │ │",
            "│                    │ │                                  │ │ search_result              │ │ fg │ white │     │                      │ │",
            "│                    │ │                                  │ │                            │ │ bg │ red   │     │                      │ │",
            "│                    │ │                                  │ │                            │ ╰────┴───────╯     │                      │ │",
            "│                    │ │                                  │ │ shape_and                  │ purple_bold        │                      │ │",
            "│                    │ │                                  │ │ shape_binary               │ purple_bold        │                      │ │",
            "│                    │ │                                  │ │ shape_block                │ blue_bold          │                      │ │",
            "│                    │ │                                  │ │ shape_bool                 │ light_cyan         │                      │ │",
            "│                    │ │                                  │ │ shape_custom               │ green              │                      │ │",
            "│                    │ │                                  │ │ shape_datetime             │ cyan_bold          │                      │ │",
            "│                    │ │                                  │ │ shape_directory            │ cyan               │                      │ │",
            "│                    │ │                                  │ │ shape_external             │ cyan               │                      │ │",
            "│                    │ │                                  │ │ shape_externalarg          │ green_bold         │                      │ │",
            "│                    │ │                                  │ │ shape_filepath             │ cyan               │                      │ │",
            "│                    │ │                                  │ │ shape_flag                 │ blue_bold          │                      │ │",
            "│                    │ │                                  │ │ shape_float                │ purple_bold        │                      │ │",
            "│                    │ │                                  │ │                            │ ╭──────┬─────────╮ │                      │ │",
            "│                    │ │                                  │ │ shape_garbage              │ │ fg   │ #FFFFFF │ │                      │ │",
            "│                    │ │                                  │ │                            │ │ bg   │ #FF0000 │ │                      │ │",
            "│                    │ │                                  │ │                            │ │ attr │ b       │ │                      │ │",
            "│                    │ │                                  │ │                            │ ╰──────┴─────────╯ │                      │ │",
            "│                    │ │                                  │ │ shape_globpattern          │ cyan_bold          │                      │ │",
            "│                    │ │                                  │ │ shape_int                  │ purple_bold        │                      │ │",
            "│                    │ │                                  │ │ shape_internalcall         │ cyan_bold          │                      │ │",
            "│                    │ │                                  │ │ shape_list                 │ cyan_bold          │                      │ │",
            "│                    │ │                                  │ │ shape_literal              │ blue               │                      │ │",
            "│                    │ │                                  │ │                            │ ╭──────┬───╮       │                      │ │",
            "│                    │ │                                  │ │ shape_matching_brackets    │ │ attr │ u │       │                      │ │",
            "│                    │ │                                  │ │                            │ ╰──────┴───╯       │                      │ │",
            "│                    │ │                                  │ │ shape_nothing              │ light_cyan         │                      │ │",
            "│                    │ │                                  │ │ shape_operator             │ yellow             │                      │ │",
            "│                    │ │                                  │ │ shape_or                   │ purple_bold        │                      │ │",
            "│                    │ │                                  │ │ shape_pipe                 │ purple_bold        │                      │ │",
            "│                    │ │                                  │ │ shape_range                │ yellow_bold        │                      │ │",
            "│                    │ │                                  │ │ shape_record               │ cyan_bold          │                      │ │",
            "│                    │ │                                  │ │ shape_redirection          │ purple_bold        │                      │ │",
            "│                    │ │                                  │ │ shape_signature            │ green_bold         │                      │ │",
            "│                    │ │                                  │ │ shape_string               │ green              │                      │ │",
            "│                    │ │                                  │ │ shape_string_interpolation │ cyan_bold          │                      │ │",
            "│                    │ │                                  │ │ shape_table                │ blue_bold          │                      │ │",
            "│                    │ │                                  │ │ shape_variable             │ purple             │                      │ │",
            "│                    │ │                                  │ ╰────────────────────────────┴────────────────────╯                      │ │",
            "│                    │ │ use_grid_icons                   │ true                                                                     │ │",
            "│                    │ │ footer_mode                      │ 25                                                                       │ │",
            "│                    │ │ float_precision                  │ 2                                                                        │ │",
            "│                    │ │ use_ansi_coloring                │ true                                                                     │ │",
            "│                    │ │ edit_mode                        │ emacs                                                                    │ │",
            "│                    │ │ shell_integration                │ true                                                                     │ │",
            "│                    │ │ show_banner                      │ true                                                                     │ │",
            "│                    │ │ render_right_prompt_on_last_line │ false                                                                    │ │",
            "│                    │ │                                  │ ╭────────────────┬────────────────────╮                                  │ │",
            "│                    │ │ hooks                            │ │                │ ╭───┬──╮           │                                  │ │",
            "│                    │ │                                  │ │ pre_prompt     │ │ 0 │  │           │                                  │ │",
            "│                    │ │                                  │ │                │ ╰───┴──╯           │                                  │ │",
            "│                    │ │                                  │ │                │ ╭───┬──╮           │                                  │ │",
            "│                    │ │                                  │ │ pre_execution  │ │ 0 │  │           │                                  │ │",
            "│                    │ │                                  │ │                │ ╰───┴──╯           │                                  │ │",
            "│                    │ │                                  │ │                │ ╭─────┬──────────╮ │                                  │ │",
            "│                    │ │                                  │ │ env_change     │ │     │ ╭───┬──╮ │ │                                  │ │",
            "│                    │ │                                  │ │                │ │ PWD │ │ 0 │  │ │ │                                  │ │",
            "│                    │ │                                  │ │                │ │     │ ╰───┴──╯ │ │                                  │ │",
            "│                    │ │                                  │ │                │ ╰─────┴──────────╯ │                                  │ │",
            "│                    │ │                                  │ │ display_output │                    │                                  │ │",
            "│                    │ │                                  │ ╰────────────────┴────────────────────╯                                  │ │",
            "│                    │ │                                  │ ╭───┬───────────────────────────┬────────────────────────┬─────╮         │ │",
            "│                    │ │ menus                            │ │ # │           name            │ only_buffer_difference │ ... │         │ │",
            "│                    │ │                                  │ ├───┼───────────────────────────┼────────────────────────┼─────┤         │ │",
            "│                    │ │                                  │ │ 0 │ completion_menu           │ false                  │ ... │         │ │",
            "│                    │ │                                  │ │ 1 │ history_menu              │ true                   │ ... │         │ │",
            "│                    │ │                                  │ │ 2 │ help_menu                 │ true                   │ ... │         │ │",
            "│                    │ │                                  │ │ 3 │ commands_menu             │ false                  │ ... │         │ │",
            "│                    │ │                                  │ │ 4 │ vars_menu                 │ true                   │ ... │         │ │",
            "│                    │ │                                  │ │ 5 │ commands_with_description │ true                   │ ... │         │ │",
            "│                    │ │                                  │ ╰───┴───────────────────────────┴────────────────────────┴─────╯         │ │",
            "│                    │ │                                  │ ╭────┬───────────────────────────┬──────────┬─────────┬──────────┬─────╮ │ │",
            "│                    │ │ keybindings                      │ │  # │           name            │ modifier │ keycode │   mode   │ ... │ │ │",
            "│                    │ │                                  │ ├────┼───────────────────────────┼──────────┼─────────┼──────────┼─────┤ │ │",
            "│                    │ │                                  │ │  0 │ completion_menu           │ none     │ tab     │ [list 3  │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ items]   │     │ │ │",
            "│                    │ │                                  │ │  1 │ completion_previous       │ shift    │ backtab │ [list 3  │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ items]   │     │ │ │",
            "│                    │ │                                  │ │  2 │ history_menu              │ control  │ char_r  │ emacs    │ ... │ │ │",
            "│                    │ │                                  │ │  3 │ next_page                 │ control  │ char_x  │ emacs    │ ... │ │ │",
            "│                    │ │                                  │ │  4 │ undo_or_previous_page     │ control  │ char_z  │ emacs    │ ... │ │ │",
            "│                    │ │                                  │ │  5 │ yank                      │ control  │ char_y  │ emacs    │ ... │ │ │",
            "│                    │ │                                  │ │  6 │ unix-line-discard         │ control  │ char_u  │ [list 3  │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ items]   │     │ │ │",
            "│                    │ │                                  │ │  7 │ kill-line                 │ control  │ char_k  │ [list 3  │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ items]   │     │ │ │",
            "│                    │ │                                  │ │  8 │ commands_menu             │ control  │ char_t  │ [list 3  │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ items]   │     │ │ │",
            "│                    │ │                                  │ │  9 │ vars_menu                 │ alt      │ char_o  │ [list 3  │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ items]   │     │ │ │",
            "│                    │ │                                  │ │ 10 │ commands_with_description │ control  │ char_s  │ [list 3  │ ... │ │ │",
            "│                    │ │                                  │ │    │                           │          │         │ items]   │     │ │ │",
            "│                    │ │                                  │ ╰────┴───────────────────────────┴──────────┴─────────┴──────────┴─────╯ │ │",
            "│                    │ ╰──────────────────────────────────┴──────────────────────────────────────────────────────────────────────────╯ │",
            "╰────────────────────┴─────────────────────────────────────────────────────────────────────────────────────────────────────────────────╯",
        ])
    );
}

#[test]
fn test_collapse_big_0() {
    Playground::setup("test_expand_big_0", |dirs, sandbox| {
        sandbox.with_files(vec![FileWithContent(
            "sample.toml",
            r#"
            [package]
            authors = ["The Nushell Project Developers"]
            default-run = "nu"
            description = "A new type of shell"
            documentation = "https://www.nushell.sh/book/"
            edition = "2021"
            exclude = ["images"]
            homepage = "https://www.nushell.sh"
            license = "MIT"
            name = "nu"
            repository = "https://github.com/nushell/nushell"
            rust-version = "1.60"
            version = "0.74.1"

            # See more keys and their definitions at https://doc.rust-lang.org/cargo/reference/manifest.html

            [package.metadata.binstall]
            pkg-url = "{ repo }/releases/download/{ version }/{ name }-{ version }-{ target }.{ archive-format }"
            pkg-fmt = "tgz"

            [package.metadata.binstall.overrides.x86_64-pc-windows-msvc]
            pkg-fmt = "zip"

            [workspace]
            members = [
                "crates/nu-cli",
                "crates/nu-engine",
                "crates/nu-parser",
                "crates/nu-system",
                "crates/nu-command",
                "crates/nu-protocol",
                "crates/nu-plugin",
                "crates/nu_plugin_inc",
                "crates/nu_plugin_gstat",
                "crates/nu_plugin_example",
                "crates/nu_plugin_query",
                "crates/nu_plugin_custom_values",
                "crates/nu-utils",
            ]

            [dependencies]
            chrono = { version = "0.4.23", features = ["serde"] }
            crossterm = "0.24.0"
            ctrlc = "3.2.1"
            log = "0.4"
            miette = { version = "5.5.0", features = ["fancy-no-backtrace"] }
            nu-ansi-term = "0.46.0"
            nu-cli = { path = "./crates/nu-cli", version = "0.74.1" }
            nu-engine = { path = "./crates/nu-engine", version = "0.74.1" }
            reedline = { version = "0.14.0", features = ["bashisms", "sqlite"] }

            rayon = "1.6.1"
            is_executable = "1.0.1"
            simplelog = "0.12.0"
            time = "0.3.12"

            [target.'cfg(not(target_os = "windows"))'.dependencies]
            # Our dependencies don't use OpenSSL on Windows
            openssl = { version = "0.10.38", features = ["vendored"], optional = true }
            signal-hook = { version = "0.3.14", default-features = false }


            [target.'cfg(windows)'.build-dependencies]
            winres = "0.1"

            [target.'cfg(target_family = "unix")'.dependencies]
            nix = { version = "0.25", default-features = false, features = ["signal", "process", "fs", "term"] }
            atty = "0.2"

            [dev-dependencies]
            nu-test-support = { path = "./crates/nu-test-support", version = "0.74.1" }
            tempfile = "3.2.0"
            assert_cmd = "2.0.2"
            criterion = "0.4"
            pretty_assertions = "1.0.0"
            serial_test = "0.10.0"
            hamcrest2 = "0.3.0"
            rstest = { version = "0.15.0", default-features = false }
            itertools = "0.10.3"

            [features]
            plugin = [
                "nu-plugin",
                "nu-cli/plugin",
                "nu-parser/plugin",
                "nu-command/plugin",
                "nu-protocol/plugin",
                "nu-engine/plugin",
            ]
            # extra used to be more useful but now it's the same as default. Leaving it in for backcompat with existing build scripts
            extra = ["default"]
            default = ["plugin", "which-support", "trash-support", "sqlite"]
            stable = ["default"]
            wasi = []

            # Enable to statically link OpenSSL; otherwise the system version will be used. Not enabled by default because it takes a while to build
            static-link-openssl = ["dep:openssl"]

            # Stable (Default)
            which-support = ["nu-command/which-support"]
            trash-support = ["nu-command/trash-support"]

            # Main nu binary
            [[bin]]
            name = "nu"
            path = "src/main.rs"

            # To use a development version of a dependency please use a global override here
            # changing versions in each sub-crate of the workspace is tedious
            [patch.crates-io]
            reedline = { git = "https://github.com/nushell/reedline.git", branch = "main" }

            # Criterion benchmarking setup
            # Run all benchmarks with `cargo bench`
            # Run individual benchmarks like `cargo bench -- <regex>` e.g. `cargo bench -- parse`
            [[bench]]
            name = "benchmarks"
            harness = false
            "#,
        )]);

        let actual = nu!(
            cwd: dirs.test(), pipeline(
            "open sample.toml | table --collapse"
        ));

        _print_lines(&actual.out, 80);

        let expected = join_lines([
            "╭──────────────────┬─────────┬─────────────────────────────────────────────────╮",
            "│ bench            │ harness │ name                                            │",
            "│                  ├─────────┼─────────────────────────────────────────────────┤",
            "│                  │ false   │ benchmarks                                      │",
            "├──────────────────┼──────┬──┴─────────────────────────────────────────────────┤",
            "│ bin              │ name │ path                                               │",
            "│                  ├──────┼────────────────────────────────────────────────────┤",
            "│                  │ nu   │ src/main.rs                                        │",
            "├──────────────────┼──────┴────────┬──────────┬────────────────────────────────┤",
            "│ dependencies     │ chrono        │ features │ serde                          │",
            "│                  │               ├──────────┼────────────────────────────────┤",
            "│                  │               │ version  │ 0.4.23                         │",
            "│                  ├───────────────┼──────────┴────────────────────────────────┤",
            "│                  │ crossterm     │ 0.24.0                                    │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ ctrlc         │ 3.2.1                                     │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ is_executable │ 1.0.1                                     │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ log           │ 0.4                                       │",
            "│                  ├───────────────┼──────────┬────────────────────────────────┤",
            "│                  │ miette        │ features │ fancy-no-backtrace             │",
            "│                  │               ├──────────┼────────────────────────────────┤",
            "│                  │               │ version  │ 5.5.0                          │",
            "│                  ├───────────────┼──────────┴────────────────────────────────┤",
            "│                  │ nu-ansi-term  │ 0.46.0                                    │",
            "│                  ├───────────────┼─────────┬─────────────────────────────────┤",
            "│                  │ nu-cli        │ path    │ ./crates/nu-cli                 │",
            "│                  │               ├─────────┼─────────────────────────────────┤",
            "│                  │               │ version │ 0.74.1                          │",
            "│                  ├───────────────┼─────────┼─────────────────────────────────┤",
            "│                  │ nu-engine     │ path    │ ./crates/nu-engine              │",
            "│                  │               ├─────────┼─────────────────────────────────┤",
            "│                  │               │ version │ 0.74.1                          │",
            "│                  ├───────────────┼─────────┴─────────────────────────────────┤",
            "│                  │ rayon         │ 1.6.1                                     │",
            "│                  ├───────────────┼──────────┬────────────────────────────────┤",
            "│                  │ reedline      │ features │ bashisms                       │",
            "│                  │               │          ├────────────────────────────────┤",
            "│                  │               │          │ sqlite                         │",
            "│                  │               ├──────────┼────────────────────────────────┤",
            "│                  │               │ version  │ 0.14.0                         │",
            "│                  ├───────────────┼──────────┴────────────────────────────────┤",
            "│                  │ simplelog     │ 0.12.0                                    │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ time          │ 0.3.12                                    │",
            "├──────────────────┼───────────────┴───┬───────────────────────────────────────┤",
            "│ dev-dependencies │ assert_cmd        │ 2.0.2                                 │",
            "│                  ├───────────────────┼───────────────────────────────────────┤",
            "│                  │ criterion         │ 0.4                                   │",
            "│                  ├───────────────────┼───────────────────────────────────────┤",
            "│                  │ hamcrest2         │ 0.3.0                                 │",
            "│                  ├───────────────────┼───────────────────────────────────────┤",
            "│                  │ itertools         │ 0.10.3                                │",
            "│                  ├───────────────────┼─────────┬─────────────────────────────┤",
            "│                  │ nu-test-support   │ path    │ ./crates/nu-test-support    │",
            "│                  │                   ├─────────┼─────────────────────────────┤",
            "│                  │                   │ version │ 0.74.1                      │",
            "│                  ├───────────────────┼─────────┴─────────────────────────────┤",
            "│                  │ pretty_assertions │ 1.0.0                                 │",
            "│                  ├───────────────────┼──────────────────┬────────────────────┤",
            "│                  │ rstest            │ default-features │ false              │",
            "│                  │                   ├──────────────────┼────────────────────┤",
            "│                  │                   │ version          │ 0.15.0             │",
            "│                  ├───────────────────┼──────────────────┴────────────────────┤",
            "│                  │ serial_test       │ 0.10.0                                │",
            "│                  ├───────────────────┼───────────────────────────────────────┤",
            "│                  │ tempfile          │ 3.2.0                                 │",
            "├──────────────────┼───────────────────┴─┬─────────────────────────────────────┤",
            "│ features         │ default             │ plugin                              │",
            "│                  │                     ├─────────────────────────────────────┤",
            "│                  │                     │ which-support                       │",
            "│                  │                     ├─────────────────────────────────────┤",
            "│                  │                     │ trash-support                       │",
            "│                  │                     ├─────────────────────────────────────┤",
            "│                  │                     │ sqlite                              │",
            "│                  ├─────────────────────┼─────────────────────────────────────┤",
            "│                  │ extra               │ default                             │",
            "│                  ├─────────────────────┼─────────────────────────────────────┤",
            "│                  │ plugin              │ nu-plugin                           │",
            "│                  │                     ├─────────────────────────────────────┤",
            "│                  │                     │ nu-cli/plugin                       │",
            "│                  │                     ├─────────────────────────────────────┤",
            "│                  │                     │ nu-parser/plugin                    │",
            "│                  │                     ├─────────────────────────────────────┤",
            "│                  │                     │ nu-command/plugin                   │",
            "│                  │                     ├─────────────────────────────────────┤",
            "│                  │                     │ nu-protocol/plugin                  │",
            "│                  │                     ├─────────────────────────────────────┤",
            "│                  │                     │ nu-engine/plugin                    │",
            "│                  ├─────────────────────┼─────────────────────────────────────┤",
            "│                  │ stable              │ default                             │",
            "│                  ├─────────────────────┼─────────────────────────────────────┤",
            "│                  │ static-link-openssl │ dep:openssl                         │",
            "│                  ├─────────────────────┼─────────────────────────────────────┤",
            "│                  │ trash-support       │ nu-command/trash-support            │",
            "│                  ├─────────────────────┼─────────────────────────────────────┤",
            "│                  │ wasi                │                                     │",
            "│                  ├─────────────────────┼─────────────────────────────────────┤",
            "│                  │ which-support       │ nu-command/which-support            │",
            "├──────────────────┼───────────────┬─────┴─────────────────────────────────────┤",
            "│ package          │ authors       │ The Nushell Project Developers            │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ default-run   │ nu                                        │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ description   │ A new type of shell                       │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ documentation │ https://www.nushell.sh/book/              │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ edition       │ 2021                                      │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ exclude       │ images                                    │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ homepage      │ https://www.nushell.sh                    │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ license       │ MIT                                       │",
            "│                  ├───────────────┼──────────┬───────────┬────────────────────┤",
            "│                  │ metadata      │ binstall │ overrides │ ...                │",
            "│                  │               │          ├───────────┼────────────────────┤",
            "│                  │               │          │ pkg-fmt   │ tgz                │",
            "│                  │               │          ├───────────┼────────────────────┤",
            "│                  │               │          │ pkg-url   │ { repo }/releases/ │",
            "│                  │               │          │           │ download/{ v       │",
            "│                  │               │          │           │ ersion             │",
            "│                  │               │          │           │  }/{ name }-{ vers │",
            "│                  │               │          │           │ ion }-             │",
            "│                  │               │          │           │ { target }.{       │",
            "│                  │               │          │           │  archive-format }  │",
            "│                  ├───────────────┼──────────┴───────────┴────────────────────┤",
            "│                  │ name          │ nu                                        │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ repository    │ https://github.com/nushell/nushell        │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ rust-version  │ 1.60                                      │",
            "│                  ├───────────────┼───────────────────────────────────────────┤",
            "│                  │ version       │ 0.74.1                                    │",
            "├──────────────────┼───────────┬───┴──────┬────────┬───────────────────────────┤",
            "│ patch            │ crates-io │ reedline │ branch │ main                      │",
            "│                  │           │          ├────────┼───────────────────────────┤",
            "│                  │           │          │ git    │ https://github.com/nushel │",
            "│                  │           │          │        │ l/reedline.git            │",
            "├──────────────────┼───────────┴──────────┴────────┴─┬──────────────┬──────────┤",
            "│ target           │ cfg(not(target_os = \"windows\")) │ dependencies │ ...      │",
            "│                  │                                 │              ├──────────┤",
            "│                  │                                 │              │ ...      │",
            "│                  ├─────────────────────────────────┼──────────────┼──────────┤",
            "│                  │ cfg(target_family = \"unix\")     │ dependencies │ ...      │",
            "│                  │                                 │              ├──────────┤",
            "│                  │                                 │              │ ...      │",
            "│                  ├─────────────────────────────────┼──────────────┴──────────┤",
            "│                  │ cfg(windows)                    │ ...                     │",
            "├──────────────────┼─────────┬───────────────────────┴─────────────────────────┤",
            "│ workspace        │ members │ crates/nu-cli                                   │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-engine                                │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-parser                                │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-system                                │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-command                               │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-protocol                              │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-plugin                                │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu_plugin_inc                            │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu_plugin_gstat                          │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu_plugin_example                        │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu_plugin_query                          │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu_plugin_custom_values                  │",
            "│                  │         ├─────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-utils                                 │",
            "╰──────────────────┴─────────┴─────────────────────────────────────────────────╯",
        ]);

        assert_eq!(actual.out, expected);

        let actual = nu!(
            cwd: dirs.test(), pipeline(
            "open sample.toml | table --collapse --width=160"
        ));

        _print_lines(&actual.out, 111);

        let expected = join_lines([
            "╭──────────────────┬─────────┬────────────────────────────────────────────────────────────────────────────────╮",
            "│ bench            │ harness │ name                                                                           │",
            "│                  ├─────────┼────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │ false   │ benchmarks                                                                     │",
            "├──────────────────┼──────┬──┴────────────────────────────────────────────────────────────────────────────────┤",
            "│ bin              │ name │ path                                                                              │",
            "│                  ├──────┼───────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │ nu   │ src/main.rs                                                                       │",
            "├──────────────────┼──────┴────────┬──────────┬───────────────────────────────────────────────────────────────┤",
            "│ dependencies     │ chrono        │ features │ serde                                                         │",
            "│                  │               ├──────────┼───────────────────────────────────────────────────────────────┤",
            "│                  │               │ version  │ 0.4.23                                                        │",
            "│                  ├───────────────┼──────────┴───────────────────────────────────────────────────────────────┤",
            "│                  │ crossterm     │ 0.24.0                                                                   │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ ctrlc         │ 3.2.1                                                                    │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ is_executable │ 1.0.1                                                                    │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ log           │ 0.4                                                                      │",
            "│                  ├───────────────┼──────────┬───────────────────────────────────────────────────────────────┤",
            "│                  │ miette        │ features │ fancy-no-backtrace                                            │",
            "│                  │               ├──────────┼───────────────────────────────────────────────────────────────┤",
            "│                  │               │ version  │ 5.5.0                                                         │",
            "│                  ├───────────────┼──────────┴───────────────────────────────────────────────────────────────┤",
            "│                  │ nu-ansi-term  │ 0.46.0                                                                   │",
            "│                  ├───────────────┼─────────┬────────────────────────────────────────────────────────────────┤",
            "│                  │ nu-cli        │ path    │ ./crates/nu-cli                                                │",
            "│                  │               ├─────────┼────────────────────────────────────────────────────────────────┤",
            "│                  │               │ version │ 0.74.1                                                         │",
            "│                  ├───────────────┼─────────┼────────────────────────────────────────────────────────────────┤",
            "│                  │ nu-engine     │ path    │ ./crates/nu-engine                                             │",
            "│                  │               ├─────────┼────────────────────────────────────────────────────────────────┤",
            "│                  │               │ version │ 0.74.1                                                         │",
            "│                  ├───────────────┼─────────┴────────────────────────────────────────────────────────────────┤",
            "│                  │ rayon         │ 1.6.1                                                                    │",
            "│                  ├───────────────┼──────────┬───────────────────────────────────────────────────────────────┤",
            "│                  │ reedline      │ features │ bashisms                                                      │",
            "│                  │               │          ├───────────────────────────────────────────────────────────────┤",
            "│                  │               │          │ sqlite                                                        │",
            "│                  │               ├──────────┼───────────────────────────────────────────────────────────────┤",
            "│                  │               │ version  │ 0.14.0                                                        │",
            "│                  ├───────────────┼──────────┴───────────────────────────────────────────────────────────────┤",
            "│                  │ simplelog     │ 0.12.0                                                                   │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ time          │ 0.3.12                                                                   │",
            "├──────────────────┼───────────────┴───┬──────────────────────────────────────────────────────────────────────┤",
            "│ dev-dependencies │ assert_cmd        │ 2.0.2                                                                │",
            "│                  ├───────────────────┼──────────────────────────────────────────────────────────────────────┤",
            "│                  │ criterion         │ 0.4                                                                  │",
            "│                  ├───────────────────┼──────────────────────────────────────────────────────────────────────┤",
            "│                  │ hamcrest2         │ 0.3.0                                                                │",
            "│                  ├───────────────────┼──────────────────────────────────────────────────────────────────────┤",
            "│                  │ itertools         │ 0.10.3                                                               │",
            "│                  ├───────────────────┼─────────┬────────────────────────────────────────────────────────────┤",
            "│                  │ nu-test-support   │ path    │ ./crates/nu-test-support                                   │",
            "│                  │                   ├─────────┼────────────────────────────────────────────────────────────┤",
            "│                  │                   │ version │ 0.74.1                                                     │",
            "│                  ├───────────────────┼─────────┴────────────────────────────────────────────────────────────┤",
            "│                  │ pretty_assertions │ 1.0.0                                                                │",
            "│                  ├───────────────────┼──────────────────┬───────────────────────────────────────────────────┤",
            "│                  │ rstest            │ default-features │ false                                             │",
            "│                  │                   ├──────────────────┼───────────────────────────────────────────────────┤",
            "│                  │                   │ version          │ 0.15.0                                            │",
            "│                  ├───────────────────┼──────────────────┴───────────────────────────────────────────────────┤",
            "│                  │ serial_test       │ 0.10.0                                                               │",
            "│                  ├───────────────────┼──────────────────────────────────────────────────────────────────────┤",
            "│                  │ tempfile          │ 3.2.0                                                                │",
            "├──────────────────┼───────────────────┴─┬────────────────────────────────────────────────────────────────────┤",
            "│ features         │ default             │ plugin                                                             │",
            "│                  │                     ├────────────────────────────────────────────────────────────────────┤",
            "│                  │                     │ which-support                                                      │",
            "│                  │                     ├────────────────────────────────────────────────────────────────────┤",
            "│                  │                     │ trash-support                                                      │",
            "│                  │                     ├────────────────────────────────────────────────────────────────────┤",
            "│                  │                     │ sqlite                                                             │",
            "│                  ├─────────────────────┼────────────────────────────────────────────────────────────────────┤",
            "│                  │ extra               │ default                                                            │",
            "│                  ├─────────────────────┼────────────────────────────────────────────────────────────────────┤",
            "│                  │ plugin              │ nu-plugin                                                          │",
            "│                  │                     ├────────────────────────────────────────────────────────────────────┤",
            "│                  │                     │ nu-cli/plugin                                                      │",
            "│                  │                     ├────────────────────────────────────────────────────────────────────┤",
            "│                  │                     │ nu-parser/plugin                                                   │",
            "│                  │                     ├────────────────────────────────────────────────────────────────────┤",
            "│                  │                     │ nu-command/plugin                                                  │",
            "│                  │                     ├────────────────────────────────────────────────────────────────────┤",
            "│                  │                     │ nu-protocol/plugin                                                 │",
            "│                  │                     ├────────────────────────────────────────────────────────────────────┤",
            "│                  │                     │ nu-engine/plugin                                                   │",
            "│                  ├─────────────────────┼────────────────────────────────────────────────────────────────────┤",
            "│                  │ stable              │ default                                                            │",
            "│                  ├─────────────────────┼────────────────────────────────────────────────────────────────────┤",
            "│                  │ static-link-openssl │ dep:openssl                                                        │",
            "│                  ├─────────────────────┼────────────────────────────────────────────────────────────────────┤",
            "│                  │ trash-support       │ nu-command/trash-support                                           │",
            "│                  ├─────────────────────┼────────────────────────────────────────────────────────────────────┤",
            "│                  │ wasi                │                                                                    │",
            "│                  ├─────────────────────┼────────────────────────────────────────────────────────────────────┤",
            "│                  │ which-support       │ nu-command/which-support                                           │",
            "├──────────────────┼───────────────┬─────┴────────────────────────────────────────────────────────────────────┤",
            "│ package          │ authors       │ The Nushell Project Developers                                           │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ default-run   │ nu                                                                       │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ description   │ A new type of shell                                                      │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ documentation │ https://www.nushell.sh/book/                                             │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ edition       │ 2021                                                                     │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ exclude       │ images                                                                   │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ homepage      │ https://www.nushell.sh                                                   │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ license       │ MIT                                                                      │",
            "│                  ├───────────────┼──────────┬───────────┬────────────────────────┬─────────┬────────────────┤",
            "│                  │ metadata      │ binstall │ overrides │ x86_64-pc-windows-msvc │ pkg-fmt │ zip            │",
            "│                  │               │          ├───────────┼────────────────────────┴─────────┴────────────────┤",
            "│                  │               │          │ pkg-fmt   │ tgz                                               │",
            "│                  │               │          ├───────────┼───────────────────────────────────────────────────┤",
            "│                  │               │          │ pkg-url   │ { repo }/releases/download/{ v                    │",
            "│                  │               │          │           │ ersion }/{ name }-{ version }-                    │",
            "│                  │               │          │           │ { target }.{ archive-format }                     │",
            "│                  ├───────────────┼──────────┴───────────┴───────────────────────────────────────────────────┤",
            "│                  │ name          │ nu                                                                       │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ repository    │ https://github.com/nushell/nushell                                       │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ rust-version  │ 1.60                                                                     │",
            "│                  ├───────────────┼──────────────────────────────────────────────────────────────────────────┤",
            "│                  │ version       │ 0.74.1                                                                   │",
            "├──────────────────┼───────────┬───┴──────┬────────┬──────────────────────────────────────────────────────────┤",
            "│ patch            │ crates-io │ reedline │ branch │ main                                                     │",
            "│                  │           │          ├────────┼──────────────────────────────────────────────────────────┤",
            "│                  │           │          │ git    │ https://github.com/nushell/reedline.git                  │",
            "├──────────────────┼───────────┴──────────┴────────┴─┬──────────────┬─────────────┬──────────┬────────────────┤",
            "│ target           │ cfg(not(target_os = \"windows\")) │ dependencies │ openssl     │ features │ vendored       │",
            "│                  │                                 │              │             ├──────────┼────────────────┤",
            "│                  │                                 │              │             │ optional │ true           │",
            "│                  │                                 │              │             ├──────────┼────────────────┤",
            "│                  │                                 │              │             │ version  │ 0.10.38        │",
            "│                  │                                 │              ├─────────────┼──────────┴───────┬────────┤",
            "│                  │                                 │              │ signal-hook │ default-features │ false  │",
            "│                  │                                 │              │             ├──────────────────┼────────┤",
            "│                  │                                 │              │             │ version          │ 0.3.14 │",
            "│                  ├─────────────────────────────────┼──────────────┼──────┬──────┴──────────────────┴────────┤",
            "│                  │ cfg(target_family = \"unix\")     │ dependencies │ atty │ 0.2                              │",
            "│                  │                                 │              ├──────┼──────────────────┬───────────────┤",
            "│                  │                                 │              │ nix  │ default-features │ false         │",
            "│                  │                                 │              │      ├──────────────────┼───────────────┤",
            "│                  │                                 │              │      │ features         │ signal        │",
            "│                  │                                 │              │      │                  ├───────────────┤",
            "│                  │                                 │              │      │                  │ process       │",
            "│                  │                                 │              │      │                  ├───────────────┤",
            "│                  │                                 │              │      │                  │ fs            │",
            "│                  │                                 │              │      │                  ├───────────────┤",
            "│                  │                                 │              │      │                  │ term          │",
            "│                  │                                 │              │      ├──────────────────┼───────────────┤",
            "│                  │                                 │              │      │ version          │ 0.25          │",
            "│                  ├─────────────────────────────────┼──────────────┴─────┬┴───────┬──────────┴───────────────┤",
            "│                  │ cfg(windows)                    │ build-dependencies │ winres │ 0.1                      │",
            "├──────────────────┼─────────┬───────────────────────┴────────────────────┴────────┴──────────────────────────┤",
            "│ workspace        │ members │ crates/nu-cli                                                                  │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-engine                                                               │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-parser                                                               │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-system                                                               │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-command                                                              │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-protocol                                                             │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-plugin                                                               │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu_plugin_inc                                                           │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu_plugin_gstat                                                         │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu_plugin_example                                                       │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu_plugin_query                                                         │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu_plugin_custom_values                                                 │",
            "│                  │         ├────────────────────────────────────────────────────────────────────────────────┤",
            "│                  │         │ crates/nu-utils                                                                │",
            "╰──────────────────┴─────────┴────────────────────────────────────────────────────────────────────────────────╯",
        ]);

        assert_eq!(actual.out, expected);
    })
}

fn join_lines(lines: impl IntoIterator<Item = impl AsRef<str>>) -> String {
    lines
        .into_iter()
        .map(|s| s.as_ref().to_string())
        .collect::<Vec<_>>()
        .join("")
}

// util function to easier copy && paste
#[allow(dead_code)]
fn _print_lines(s: &str, w: usize) {
    eprintln!("{:#?}", _split_str_by_width(s, w));
}

// util function to easier copy && paste
// todo: make UTF-8 friendly
fn _split_str_by_width(s: &str, w: usize) -> Vec<String> {
    let mut lines = vec![];
    let mut line = String::new();
    let mut i = 0;
    for c in s.chars() {
        if i < w {
            i += 1;
            line.push(c);
        } else {
            lines.push(line);
            line = String::new();
            line.push(c);
            i = 1;
        }
    }

    lines.push(line);

    lines
}

#[test]
fn table_expand_index_offset() {
    let actual = nu!(r#"1..1002 | table --expand"#);
    let suffix = "╭──────┬──────╮│ 1000 │ 1001 ││ 1001 │ 1002 │╰──────┴──────╯";
    let expected_suffix = actual.out.strip_suffix(suffix);
    assert!(expected_suffix.is_some(), "{:?}", actual.out);
}

#[test]
fn table_index_offset() {
    let actual = nu!(r#"1..1002 | table"#);
    let suffix = "╭──────┬──────╮│ 1000 │ 1001 ││ 1001 │ 1002 │╰──────┴──────╯";
    let expected_suffix = actual.out.strip_suffix(suffix);
    assert!(expected_suffix.is_some(), "{:?}", actual.out);
}
