use crate::command_input::{CommandInput, Move};
use crate::history::History;

use crate::fixed_length_grapheme_string::FixedLengthGraphemeString;
use crate::history::Command;
use crate::history_cleaner;
use crate::settings::KeyScheme;
use crate::settings::Settings;
use std::io::{stdin, stdout, Write};
use termion::color;
use termion::event::Key;
use termion::input::TermRead;
use termion::raw::IntoRawMode;
use termion::screen::AlternateScreen;
use termion::{clear, cursor, terminal_size};

pub struct Interface<'a> {
    history: &'a History,
    settings: &'a Settings,
    input: CommandInput,
    selection: usize,
    matches: Vec<Command>,
    debug: bool,
    run: bool,
    menu_mode: MenuMode,
    in_vim_insert_mode: bool,
}

pub struct SelectionResult {
    pub run: bool,
    pub selection: Option<String>,
}

pub enum MoveSelection {
    Up,
    Down,
}

#[derive(PartialEq)]
pub enum MenuMode {
    Normal,
    ConfirmDelete,
}

impl MenuMode {
    fn text(&self, interface: &Interface) -> &str {
        match *self {
            MenuMode::Normal => match interface.settings.key_scheme {
                KeyScheme::Emacs => "McFly | ESC - Exit | ⏎ - Run | TAB - Edit | F2 - Delete",
                KeyScheme::Vim => {
                    if interface.in_vim_insert_mode {
                        "McFly (Vim) | ESC - Exit | ⏎ - Run | TAB - Edit | F2 - Delete        -- INSERT --"
                    } else {
                        "McFly (Vim) | ESC - Exit | ⏎ - Run | TAB - Edit | F2 - Delete"
                    }
                }
            },
            MenuMode::ConfirmDelete => "Delete selected command from the history? (Y/N)",
        }
    }

    fn bg(&self) -> String {
        match *self {
            MenuMode::Normal => color::Bg(color::LightBlue).to_string(),
            MenuMode::ConfirmDelete => color::Bg(color::Red).to_string(),
        }
    }
}

const PROMPT_LINE_INDEX: u16 = 3;
const INFO_LINE_INDEX: u16 = 1;
const RESULTS_TOP_INDEX: u16 = 5;

impl<'a> Interface<'a> {
    pub fn new(settings: &'a Settings, history: &'a History) -> Interface<'a> {
        Interface {
            history,
            settings,
            input: CommandInput::from(settings.command.to_owned()),
            selection: 0,
            matches: Vec::new(),
            debug: settings.debug,
            run: false,
            menu_mode: MenuMode::Normal,
            in_vim_insert_mode: false,
        }
    }

    pub fn display(&mut self) -> SelectionResult {
        self.build_cache_table();
        self.select();

        let command = self.input.command.to_owned();

        if command.chars().any(|c| !c.is_whitespace()) {
            self.history.record_selected_from_ui(
                &command,
                &self.settings.session_id,
                &self.settings.dir,
            );
            SelectionResult {
                run: self.run,
                selection: Some(command),
            }
        } else {
            SelectionResult {
                run: self.run,
                selection: None,
            }
        }
    }

    fn build_cache_table(&self) {
        self.history.build_cache_table(
            &self.settings.dir.to_owned(),
            &Some(self.settings.session_id.to_owned()),
            None,
            None,
            None,
        );
    }

    fn menubar<W: Write>(&self, screen: &mut W) {
        let (width, _height): (u16, u16) = terminal_size().unwrap();
        write!(
            screen,
            "{hide}{cursor}{clear}{fg}{bg}{text:width$}{reset_bg}",
            hide = cursor::Hide,
            fg = color::Fg(color::LightWhite).to_string(),
            bg = self.menu_mode.bg(),
            cursor = cursor::Goto(1, INFO_LINE_INDEX),
            clear = clear::CurrentLine,
            text = self.menu_mode.text(self),
            reset_bg = color::Bg(color::Reset).to_string(),
            width = width as usize
        )
        .unwrap();
        screen.flush().unwrap();
    }

    fn prompt<W: Write>(&self, screen: &mut W) {
        write!(
            screen,
            "{}{}{}$ {}",
            if self.settings.lightmode {
                color::Fg(color::Black).to_string()
            } else {
                color::Fg(color::LightWhite).to_string()
            },
            cursor::Goto(1, PROMPT_LINE_INDEX),
            clear::CurrentLine,
            self.input
        )
        .unwrap();
        write!(
            screen,
            "{}{}",
            cursor::Goto(self.input.cursor as u16 + 3, PROMPT_LINE_INDEX),
            cursor::Show
        )
        .unwrap();
        screen.flush().unwrap();
    }

    fn debug_cursor<W: Write>(&self, screen: &mut W) {
        write!(
            screen,
            "{}{}",
            cursor::Hide,
            cursor::Goto(0, RESULTS_TOP_INDEX + self.settings.results + 1)
        )
        .unwrap();
        screen.flush().unwrap();
    }

    fn results<W: Write>(&mut self, screen: &mut W) {
        write!(
            screen,
            "{}{}{}",
            cursor::Hide,
            cursor::Goto(1, RESULTS_TOP_INDEX),
            clear::All
        )
        .unwrap();
        let (width, _height): (u16, u16) = terminal_size().unwrap();

        if !self.matches.is_empty() && self.selection > self.matches.len() - 1 {
            self.selection = self.matches.len() - 1;
        }

        for (index, command) in self.matches.iter().enumerate() {
            let mut fg = if self.settings.lightmode {
                color::Fg(color::Black).to_string()
            } else {
                color::Fg(color::LightWhite).to_string()
            };

            let mut highlight = if self.settings.lightmode {
                color::Fg(color::Blue).to_string()
            } else {
                color::Fg(color::Green).to_string()
            };

            let mut bg = color::Bg(color::Reset).to_string();

            if index == self.selection {
                if self.settings.lightmode {
                    fg = color::Fg(color::LightWhite).to_string();
                    bg = color::Bg(color::LightBlack).to_string();
                    highlight = color::Fg(color::White).to_string();
                } else {
                    fg = color::Fg(color::Black).to_string();
                    bg = color::Bg(color::LightWhite).to_string();
                    highlight = color::Fg(color::Green).to_string();
                }
            }

            write!(screen, "{}{}", fg, bg).unwrap();

            write!(
                screen,
                "{}{}",
                cursor::Goto(1, index as u16 + RESULTS_TOP_INDEX),
                Interface::truncate_for_display(
                    command,
                    &self.input.command,
                    width,
                    highlight,
                    fg,
                    self.debug
                )
            )
            .unwrap();

            write!(screen, "{}", color::Bg(color::Reset)).unwrap();
            write!(screen, "{}", color::Fg(color::Reset)).unwrap();
        }
        screen.flush().unwrap();
    }

    #[allow(unused)]
    fn debug<W: Write, S: Into<String>>(&self, screen: &mut W, s: S) {
        write!(
            screen,
            "{}{}{}",
            cursor::Goto(1, 2),
            clear::CurrentLine,
            s.into()
        )
        .unwrap();
        screen.flush().unwrap();
    }

    fn move_selection(&mut self, direction: MoveSelection) {
        match direction {
            MoveSelection::Up => {
                if self.selection > 0 {
                    self.selection -= 1;
                }
            }
            MoveSelection::Down => {
                self.selection += 1;
            }
        }
    }

    fn accept_selection(&mut self) {
        if !self.matches.is_empty() {
            self.input.set(&self.matches[self.selection].cmd);
        }
    }

    fn confirm(&mut self, confirmation: bool) {
        if confirmation {
            if let MenuMode::ConfirmDelete = self.menu_mode {
                self.delete_selection()
            }
        }
        self.menu_mode = MenuMode::Normal;
    }

    fn delete_selection(&mut self) {
        if !self.matches.is_empty() {
            {
                let command = &self.matches[self.selection];
                history_cleaner::clean(self.settings, self.history, &command.cmd);
            }
            self.build_cache_table();
            self.refresh_matches();
        }
    }

    fn refresh_matches(&mut self) {
        self.selection = 0;
        self.matches = self
            .history
            .find_matches(&self.input.command, self.settings.results as i16);
    }

    fn select(&mut self) {
        let stdin = stdin();
        let mut screen = AlternateScreen::from(stdout().into_raw_mode().unwrap());
        //        let mut screen = stdout().into_raw_mode().unwrap();
        write!(screen, "{}", clear::All).unwrap();

        self.refresh_matches();
        self.results(&mut screen);
        self.menubar(&mut screen);
        self.prompt(&mut screen);

        for c in stdin.keys() {
            self.debug_cursor(&mut screen);

            if self.menu_mode != MenuMode::Normal {
                match c.unwrap() {
                    Key::Ctrl('c')
                    | Key::Ctrl('d')
                    | Key::Ctrl('g')
                    | Key::Ctrl('z')
                    | Key::Ctrl('r') => {
                        self.run = false;
                        self.input.clear();
                        break;
                    }
                    Key::Char('y') | Key::Char('Y') => {
                        self.confirm(true);
                    }
                    Key::Char('n') | Key::Char('N') | Key::Esc => {
                        self.confirm(false);
                    }
                    _ => {}
                }
            } else {
                let early_out = match self.settings.key_scheme {
                    KeyScheme::Emacs => self.select_with_emacs_key_scheme(c.unwrap()),
                    KeyScheme::Vim => self.select_with_vim_key_scheme(c.unwrap()),
                };

                if early_out {
                    break;
                }
            }

            self.results(&mut screen);
            self.menubar(&mut screen);
            self.prompt(&mut screen);
        }

        write!(screen, "{}{}", clear::All, cursor::Show).unwrap();
    }

    fn select_with_emacs_key_scheme(&mut self, k: Key) -> bool {
        match k {
            Key::Char('\n') | Key::Char('\r') | Key::Ctrl('j') => {
                self.run = true;
                self.accept_selection();
                return true;
            }
            Key::Char('\t') => {
                self.run = false;
                self.accept_selection();
                return true;
            }
            Key::Ctrl('c')
            | Key::Ctrl('d')
            | Key::Ctrl('g')
            | Key::Ctrl('z')
            | Key::Esc
            | Key::Ctrl('r') => {
                self.run = false;
                self.input.clear();
                return true;
            }
            Key::Ctrl('b') => self.input.move_cursor(Move::Backward),
            Key::Ctrl('f') => self.input.move_cursor(Move::Forward),
            Key::Ctrl('a') => self.input.move_cursor(Move::BOL),
            Key::Ctrl('e') => self.input.move_cursor(Move::EOL),
            Key::Ctrl('w') | Key::Alt('\x08') | Key::Alt('\x7f') => {
                self.input.delete(Move::BackwardWord);
                self.refresh_matches();
            }
            Key::Alt('d') => {
                self.input.delete(Move::ForwardWord);
                self.refresh_matches();
            }
            Key::Ctrl('v') => {
                self.debug = !self.debug;
            }
            Key::Alt('b') => self.input.move_cursor(Move::BackwardWord),
            Key::Alt('f') => self.input.move_cursor(Move::ForwardWord),
            Key::Left => self.input.move_cursor(Move::Backward),
            Key::Right => self.input.move_cursor(Move::Forward),
            Key::Up | Key::PageUp | Key::Ctrl('p') => self.move_selection(MoveSelection::Up),
            Key::Down | Key::PageDown | Key::Ctrl('n') => self.move_selection(MoveSelection::Down),
            Key::Ctrl('k') => {
                self.input.delete(Move::EOL);
                self.refresh_matches();
            }
            Key::Ctrl('u') => {
                self.input.delete(Move::BOL);
                self.refresh_matches();
            }
            Key::Backspace | Key::Ctrl('h') => {
                self.input.delete(Move::Backward);
                self.refresh_matches();
            }
            Key::Delete => {
                self.input.delete(Move::Forward);
                self.refresh_matches();
            }
            Key::Home => self.input.move_cursor(Move::BOL),
            Key::End => self.input.move_cursor(Move::EOL),
            Key::Char(c) => {
                self.input.insert(c);
                self.refresh_matches();
            }
            Key::F(2) => {
                if !self.matches.is_empty() {
                    self.menu_mode = MenuMode::ConfirmDelete;
                }
            }
            _ => {}
        }

        false
    }

    fn select_with_vim_key_scheme(&mut self, k: Key) -> bool {
        if self.in_vim_insert_mode {
            match k {
                Key::Char('\n') | Key::Char('\r') | Key::Ctrl('j') => {
                    self.run = true;
                    self.accept_selection();
                    return true;
                }
                Key::Char('\t') => {
                    self.run = false;
                    self.accept_selection();
                    return true;
                }
                Key::Ctrl('c') | Key::Ctrl('g') | Key::Ctrl('z') | Key::Ctrl('r') => {
                    self.run = false;
                    self.input.clear();
                    return true;
                }
                Key::Left => self.input.move_cursor(Move::Backward),
                Key::Right => self.input.move_cursor(Move::Forward),
                Key::Up | Key::PageUp | Key::Ctrl('u') => self.move_selection(MoveSelection::Up),
                Key::Down | Key::PageDown | Key::Ctrl('d') => {
                    self.move_selection(MoveSelection::Down)
                }
                Key::Esc => self.in_vim_insert_mode = false,
                Key::Backspace => {
                    self.input.delete(Move::Backward);
                    self.refresh_matches();
                }
                Key::Delete => {
                    self.input.delete(Move::Forward);
                    self.refresh_matches();
                }
                Key::Home => self.input.move_cursor(Move::BOL),
                Key::End => self.input.move_cursor(Move::EOL),
                Key::Char(c) => {
                    self.input.insert(c);
                    self.refresh_matches();
                }
                Key::F(2) => {
                    if !self.matches.is_empty() {
                        self.menu_mode = MenuMode::ConfirmDelete;
                    }
                }
                _ => {}
            }
        } else {
            match k {
                Key::Char('\n') | Key::Char('\r') | Key::Ctrl('j') => {
                    self.run = true;
                    self.accept_selection();
                    return true;
                }
                Key::Char('\t') => {
                    self.run = false;
                    self.accept_selection();
                    return true;
                }
                Key::Ctrl('c')
                | Key::Ctrl('g')
                | Key::Ctrl('z')
                | Key::Esc
                | Key::Char('q')
                // TODO add ZZ as shortcut
                | Key::Ctrl('r') => {
                    self.run = false;
                    self.input.clear();
                    return true;
                }
                Key::Left | Key::Char('h') => self.input.move_cursor(Move::Backward),
                Key::Right | Key::Char('l') => self.input.move_cursor(Move::Forward),
                Key::Up | Key::PageUp | Key::Char('k') | Key::Ctrl('u') => self.move_selection(MoveSelection::Up),
                Key::Down | Key::PageDown | Key::Char('j') | Key::Ctrl('d') => self.move_selection(MoveSelection::Down),
                Key::Char('b') | Key::Char('e') => self.input.move_cursor(Move::BackwardWord),
                Key::Char('w') => self.input.move_cursor(Move::ForwardWord),
                Key::Char('0') | Key::Char('^') => self.input.move_cursor(Move::BOL),
                Key::Char('$') => self.input.move_cursor(Move::EOL),
                Key::Char('i') | Key::Char('a') => self.in_vim_insert_mode = true,
                Key::Backspace => {
                    self.input.delete(Move::Backward);
                    self.refresh_matches();
                }
                Key::Delete | Key::Char('x') => {
                    self.input.delete(Move::Forward);
                    self.refresh_matches();
                }
                Key::Home => self.input.move_cursor(Move::BOL),
                Key::End => self.input.move_cursor(Move::EOL),
                Key::Char(_c) => {

                }
                Key::F(2) => {
                    if !self.matches.is_empty() {
                        self.menu_mode = MenuMode::ConfirmDelete;
                    }
                }
                _ => {}
            }
        }

        false
    }

    fn truncate_for_display(
        command: &Command,
        search: &str,
        width: u16,
        highlight_color: String,
        base_color: String,
        debug: bool,
    ) -> String {
        let mut prev = 0;
        let debug_space = if debug { 90 } else { 0 };
        let max_grapheme_length = if width > debug_space {
            width - debug_space
        } else {
            2
        };
        let mut out = FixedLengthGraphemeString::empty(max_grapheme_length);

        let lowercase_search = search.to_lowercase();
        let lowercase_cmd = command.cmd.to_lowercase();
        let search_len = search.len();

        if !search.is_empty() {
            for (index, _) in lowercase_cmd.match_indices(&lowercase_search) {
                if prev != index {
                    out.push_grapheme_str(&command.cmd[prev..index]);
                }
                out.push_str(&highlight_color);
                out.push_grapheme_str(&command.cmd[index..(index + search_len)]);
                out.push_str(&base_color);
                prev = index + search_len;
            }
        }

        if prev != command.cmd.len() {
            out.push_grapheme_str(&command.cmd[prev..]);
        }

        if debug {
            out.max_grapheme_length += debug_space;
            out.push_grapheme_str("  ");
            out.push_str(&format!("{}", color::Fg(color::LightBlue)));
            out.push_grapheme_str(format!("rnk: {:.*} ", 2, command.rank));
            out.push_grapheme_str(format!("age: {:.*} ", 2, command.features.age_factor));
            out.push_grapheme_str(format!("lng: {:.*} ", 2, command.features.length_factor));
            out.push_grapheme_str(format!("ext: {:.*} ", 0, command.features.exit_factor));
            out.push_grapheme_str(format!(
                "r_ext: {:.*} ",
                0, command.features.recent_failure_factor
            ));
            out.push_grapheme_str(format!("dir: {:.*} ", 3, command.features.dir_factor));
            out.push_grapheme_str(format!(
                "s_dir: {:.*} ",
                3, command.features.selected_dir_factor
            ));
            out.push_grapheme_str(format!("ovlp: {:.*} ", 3, command.features.overlap_factor));
            out.push_grapheme_str(format!(
                "i_ovlp: {:.*} ",
                3, command.features.immediate_overlap_factor
            ));
            out.push_grapheme_str(format!(
                "occ: {:.*}",
                2, command.features.occurrences_factor
            ));
            out.push_grapheme_str(format!(
                "s_occ: {:.*} ",
                2, command.features.selected_occurrences_factor
            ));
            out.push_str(&base_color);
        }

        out.string
    }
}

// TODO:
// Ctrl('X') + Ctrl('U') => undo
// Ctrl('X') + Ctrl('G') => abort
// Meta('c') => capitalize word
// Meta('l') => downcase word
// Meta('t') => transpose words
// Meta('u') => upcase word
// Meta('y') => yank pop
// Ctrl('r') => reverse history search
// Ctrl('s') => forward history search
// Ctrl('t') => transpose characters
// Ctrl('q') | Ctrl('v') => quoted insert
// Ctrl('y') => yank
// Ctrl('_') => undo
