use std::rc::Rc;

use floem::{
    context::PaintCx,
    cosmic_text::{Attrs, AttrsList, FamilyOwned, TextLayout},
    id::Id,
    peniko::kurbo::{Point, Rect, Size},
    view::{ChangeFlags, View},
    Renderer,
};
use lapce_core::{buffer::rope_text::RopeText, mode::Mode};

use crate::{
    config::{color::LapceColor, LapceConfig},
    doc::Document,
};

use super::{view::changes_colors, EditorData};

pub struct EditorGutterView {
    id: Id,
    editor: Rc<EditorData>,
    width: f64,
}

pub fn editor_gutter_view(editor: Rc<EditorData>) -> EditorGutterView {
    let id = Id::next();

    EditorGutterView {
        id,
        editor,
        width: 0.0,
    }
}

impl EditorGutterView {
    fn paint_head_changes(
        &self,
        cx: &mut PaintCx,
        doc: Rc<Document>,
        viewport: Rect,
        is_normal: bool,
        config: &LapceConfig,
    ) {
        if !is_normal {
            return;
        }

        let changes = doc.head_changes.get_untracked();
        let line_height = config.editor.line_height() as f64;

        let min_line = (viewport.y0 / line_height).floor() as usize;
        let max_line = (viewport.y1 / line_height).ceil() as usize;

        let changes = changes_colors(changes, min_line, max_line, config);
        for (y, height, removed, color) in changes {
            let height = if removed {
                10.0
            } else {
                height as f64 * line_height
            };
            let mut y = y as f64 * line_height - viewport.y0;
            if removed {
                y -= 5.0;
            }
            cx.fill(
                &Size::new(3.0, height)
                    .to_rect()
                    .with_origin(Point::new(self.width + 7.0, y)),
                color,
                0.0,
            )
        }
    }

    fn paint_sticky_headers(
        &self,
        cx: &mut PaintCx,
        is_normal: bool,
        config: &LapceConfig,
    ) {
        if !is_normal {
            return;
        }

        if !config.editor.sticky_header {
            return;
        }
        let sticky_header_height = self.editor.sticky_header_height;
        let sticky_header_height = sticky_header_height.get_untracked();
        if sticky_header_height == 0.0 {
            return;
        }

        let sticky_area_rect =
            Size::new(self.width + 25.0 + 30.0, sticky_header_height)
                .to_rect()
                .with_origin(Point::new(-25.0, 0.0))
                .inflate(25.0, 0.0);
        cx.fill(
            &sticky_area_rect,
            config.get_color(LapceColor::LAPCE_DROPDOWN_SHADOW),
            3.0,
        );
        cx.fill(
            &sticky_area_rect,
            config.get_color(LapceColor::EDITOR_STICKY_HEADER_BACKGROUND),
            0.0,
        );
    }
}

impl View for EditorGutterView {
    fn id(&self) -> Id {
        self.id
    }

    fn child(&self, _id: Id) -> Option<&dyn View> {
        None
    }

    fn child_mut(&mut self, _id: Id) -> Option<&mut dyn View> {
        None
    }

    fn children(&self) -> Vec<&dyn View> {
        Vec::new()
    }

    fn children_mut(&mut self) -> Vec<&mut dyn View> {
        Vec::new()
    }

    fn update(
        &mut self,
        _cx: &mut floem::context::UpdateCx,
        _state: Box<dyn std::any::Any>,
    ) -> ChangeFlags {
        ChangeFlags::default()
    }

    fn layout(
        &mut self,
        cx: &mut floem::context::LayoutCx,
    ) -> floem::taffy::prelude::Node {
        cx.layout_node(self.id, false, |_| Vec::new())
    }

    fn compute_layout(
        &mut self,
        cx: &mut floem::context::LayoutCx,
    ) -> Option<floem::peniko::kurbo::Rect> {
        if let Some(width) = cx.get_layout(self.id).map(|l| l.size.width as f64) {
            self.width = width;
        }
        None
    }

    fn event(
        &mut self,
        _cx: &mut floem::context::EventCx,
        _id_path: Option<&[Id]>,
        _event: floem::event::Event,
    ) -> bool {
        false
    }

    fn paint(&mut self, cx: &mut floem::context::PaintCx) {
        let viewport = self.editor.viewport.get_untracked();
        let cursor = self.editor.cursor;
        let screen_lines = self.editor.screen_lines();
        let config = self.editor.common.config;

        let kind_is_normal = self
            .editor
            .view
            .kind
            .with_untracked(|kind| kind.is_normal());
        let (offset, mode) = cursor.with_untracked(|c| (c.offset(), c.get_mode()));
        let config = config.get_untracked();
        let line_height = config.editor.line_height() as f64;
        let last_line = self.editor.view.last_line();
        let current_line = self
            .editor
            .view
            .doc
            .get_untracked()
            .buffer
            .with_untracked(|buffer| buffer.line_of_offset(offset));

        let family: Vec<FamilyOwned> =
            FamilyOwned::parse_list(&config.editor.font_family).collect();
        let attrs = Attrs::new()
            .family(&family)
            .color(*config.get_color(LapceColor::EDITOR_DIM))
            .font_size(config.editor.font_size() as f32);
        let attrs_list = AttrsList::new(attrs);
        let current_line_attrs_list = AttrsList::new(
            attrs.color(*config.get_color(LapceColor::EDITOR_FOREGROUND)),
        );
        let show_relative = config.core.modal
            && config.editor.modal_mode_relative_line_numbers
            && mode != Mode::Insert
            && kind_is_normal;

        for line in &screen_lines.lines {
            let line = *line;
            if line > last_line {
                break;
            }

            let text = if show_relative {
                if line == current_line {
                    line + 1
                } else {
                    line.abs_diff(current_line)
                }
            } else {
                line + 1
            }
            .to_string();

            let info = screen_lines.info.get(&line).unwrap();
            let mut text_layout = TextLayout::new();
            if line == current_line {
                text_layout.set_text(&text, current_line_attrs_list.clone());
            } else {
                text_layout.set_text(&text, attrs_list.clone());
            }
            let size = text_layout.size();
            let height = size.height;
            let y = info.y;

            cx.draw_text(
                &text_layout,
                Point::new(
                    (self.width - (size.width)).max(0.0),
                    y as f64 + (line_height - height) / 2.0 - viewport.y0,
                ),
            );
        }

        self.paint_head_changes(
            cx,
            self.editor.view.doc.get_untracked(),
            viewport,
            kind_is_normal,
            &config,
        );
        self.paint_sticky_headers(cx, kind_is_normal, &config);
    }
}
