use crate::config::option::{LineMode, SortType};
use crate::io::FileOperationOptions;

use super::{Command, CommandComment};

impl CommandComment for Command {
    // These comments are displayed at the help page
    fn comment(&self) -> &'static str {
        match self {
            Self::SetLineMode(linemode) => match linemode {
                LineMode::Size => "Show files with size",
                LineMode::MTime => "Show files with modified time",
                LineMode::SizeMTime => "Show files with size and modified time",
            },
            Self::Escape => "Escape from visual mode (cancel)",
            Self::BulkRename => "Bulk rename",

            Self::ToggleVisualMode => "Toggle visual mode",

            Self::ChangeDirectory { .. } => "Change directory",
            Self::ParentDirectory => "CD to parent directory",
            Self::PreviousDirectory => "CD to the last dir in history",

            Self::NewTab { .. } => "Open a new tab",
            Self::CloseTab => "Close current tab",
            Self::CommandLine { prefix, .. } => match prefix.trim() {
                "cd" => "Change directory",
                "search" => "Open a search prompt",
                "search_glob" => "Glob search",
                "rename" => "Rename selected file",
                "touch" => "Touch file",
                "mkdir" => "Make a new directory",
                _ => "Open a command line",
            },

            Self::CutFiles => "Cut selected files",
            Self::CopyFiles => "Copy selected files",
            Self::CopyFileName => "Copy filename",
            Self::CopyFileNameWithoutExtension => "Copy filename without extension",
            Self::CopyFilePath => "Copy path to file",
            Self::CopyDirPath => "Copy directory name",
            Self::SymlinkFiles { .. } => "Symlink selected files",

            Self::PasteFiles {
                options:
                    FileOperationOptions {
                        overwrite,
                        skip_exist,
                        ..
                    },
            } => match (overwrite, skip_exist) {
                (true, false) => "Paste, overwrite",
                (false, true) => "Paste, skip existing files",
                _ => "Paste",
            },
            Self::DeleteFiles { .. } => "Delete selected files",

            Self::CursorMoveUp { .. } => "Move cursor up",
            Self::CursorMoveDown { .. } => "Move cursor down",
            Self::CursorMoveHome => "Move cursor to the very top",
            Self::CursorMoveEnd => "Move cursor to the ver bottom",
            Self::CursorMovePageUp(_) => "Move cursor one page up",
            Self::CursorMovePageDown(_) => "Move cursor one page down",

            Self::CursorMovePageHome => "Move cursor to top of page",
            Self::CursorMovePageMiddle => "Move cursor to middle of page",
            Self::CursorMovePageEnd => "Move cursor to bottom of page",

            Self::ParentCursorMoveUp { .. } => "Cursor up in parent list",
            Self::ParentCursorMoveDown { .. } => "Cursor down in parent list",

            Self::PreviewCursorMoveUp { .. } => "Cursor up in file preview",
            Self::PreviewCursorMoveDown { .. } => "Cursor down in file preview",

            Self::NewDirectory { .. } => "Make a new directory",
            Self::OpenFile => "Open a file",
            Self::OpenFileWith { .. } => "Open using selected program",

            Self::Quit(_) => "Quit the program",
            Self::ReloadDirList => "Reload current dir listing",
            Self::RenameFile { .. } => "Rename file",
            Self::TouchFile { .. } => "Touch file",
            Self::RenameFileAppend => "Rename a file",
            Self::RenameFilePrepend => "Rename a file",
            Self::RenameFileKeepExt => "Rename a file",

            Self::SearchString { .. } => "Search",
            Self::SearchIncremental { .. } => "Search as you type",
            Self::SearchGlob { .. } => "Search with globbing",
            Self::SearchNext => "Next search entry",
            Self::SearchPrev => "Previous search entry",

            Self::SelectFiles { .. } => "Select file",
            Self::SetMode => "Set file permissions",
            Self::SubProcess { spawn: false, .. } => "Run a shell command",
            Self::SubProcess { spawn: true, .. } => "Run command in background",
            Self::ShowTasks => "Show running background tasks",

            Self::ToggleHiddenFiles => "Toggle hidden files displaying",

            Self::SwitchLineNums(_) => "Switch line numbering",

            Self::Flat { .. } => "Flattern directory list",
            Self::NumberedCommand { .. } => "Jump via input number",

            Self::Sort(sort_type) => match sort_type {
                SortType::Lexical => "Sort lexically",
                SortType::Mtime => "Sort by modification time",
                SortType::Natural => "Sort naturally",
                SortType::Size => "Sort by size",
                SortType::Ext => "Sort by extension",
            },
            Self::SortReverse => "Reverse sort order",

            Self::Filter { .. } => "Filter directory list",

            Self::TabSwitch { .. } => "Switch to the next tab",
            Self::TabSwitchIndex { .. } => "Switch to a given tab",
            Self::Help => "Open this help page",

            Self::SearchFzf => "Search via fzf",
            Self::SubdirFzf => "Switch to a child directory via fzf",
            Self::Zoxide(_) => "Zoxide",
            Self::ZoxideInteractive => "Zoxide interactive",

            Self::BookmarkAdd => "Add a bookmark",
            Self::BookmarkChangeDirectory => "Navigate to a bookmark",
        }
    }
}
