#!/usr/bin/env python3
"""Build README for GitHub and crates.io.

This script uses Hurl doc to generate README suitable for GitHub and crates.io
The README are displayed on GitHub and crates.io. It's generated by concatenating

- home.md
- samples.md
- manual.md
- installation.md

Examples:
    $ python3 bin/docs/build_readme.py github > README.md
    $ python3 bin/docs/build_readme.py crates > packages/hurl/README.md

"""
import os
import re
import sys
from pathlib import Path

from markdown import parse_markdown, MarkdownDoc


def build_home_md(text: str) -> MarkdownDoc:
    """Returns the MarkDown representation of the Home"""

    # We adapt the "Why Hurl" part to transform h2 tag back to markdown
    def showcase_rep(m):
        return f"<li><b>{m.group(1)}:</b> {m.group(2).lower()}</li>"

    home = re.sub(
        r"""<li class="showcase-item"><h2 class="showcase-item-title">(.+)</h2>(.+)</li>""",
        showcase_rep,
        text,
    )
    return parse_markdown(text=home)


def build_sample_md(text: str) -> MarkdownDoc:
    """Returns the MarkDown representation of the sample"""
    return parse_markdown(text=text)


def build_manual_md(text: str) -> MarkdownDoc:
    return parse_markdown(text=text)


def build_installation_md(text: str) -> MarkdownDoc:
    return parse_markdown(text=text)


def replace(text: str, dest: str) -> str:
    # Do some replacements

    # Replace canonical links to hurl.dev links:
    text = re.sub(
        r"/docs/(.*)\.md",
        r"https://hurl.dev/docs/\1.html",
        text,
    )

    # Change some text snippets for GitHub and crates.io
    # - first param: pattern to look for in Markdown file
    # - second param: pattern to replace in GitHub README
    # - third param: pattern to replace in crates.io README
    snippets = [
        ("blog.md", "https://hurl.dev/blog/", "https://hurl.dev/blog/"),
        (
            """\
<div class="home-logo">
    <img class="u-theme-light" src="/docs/assets/img/logo-light.svg" width="277px" height="72px" alt="Hurl logo"/>
    <img class="u-theme-dark" src="/docs/assets/img/logo-dark.svg" width="277px" height="72px" alt="Hurl logo"/>
</div>
""",
            """\
<picture>
    <source media="(prefers-color-scheme: light)" srcset="/docs/assets/img/logo-light.svg?sanitize=true" > 
    <source media="(prefers-color-scheme: dark)" srcset="/docs/assets/img/logo-dark.svg?sanitize=true" > 
    <img src="/docs/assets/img/logo-light.svg?sanitize=true" width="264px" alt="Hurl Logo">
</picture>

[![deploy status](https://github.com/Orange-OpenSource/hurl/workflows/test/badge.svg)](https://github.com/Orange-OpenSource/hurl/actions)
[![coverage](https://Orange-OpenSource.github.io/hurl/coverage/badges/flat.svg)](https://Orange-OpenSource.github.io/hurl/coverage)
[![Crates.io](https://img.shields.io/crates/v/hurl.svg)](https://crates.io/crates/hurl)
[![documentation](https://img.shields.io/badge/-documentation-ff0288)](https://hurl.dev)
""",
            """\
<img src="https://hurl.dev/assets/img/logo-light.svg" width="264px" alt="Hurl Logo">

[![deploy status](https://github.com/Orange-OpenSource/hurl/workflows/test/badge.svg)](https://github.com/Orange-OpenSource/hurl/actions)
[![coverage](https://Orange-OpenSource.github.io/hurl/coverage/badges/flat.svg)](https://Orange-OpenSource.github.io/hurl/coverage)
[![Crates.io](https://img.shields.io/crates/v/hurl.svg)](https://crates.io/crates/hurl)
[![documentation](https://img.shields.io/badge/-documentation-ff0288)](https://hurl.dev)
""",
        ),
        (
            """<div id="home-demo"></div>""",
            """<a href="https://hurl.dev/player.html?id=hurl&speed=3"><img src="/docs/assets/img/poster-hurl.png" width="100%" alt="Hurl Demo"/></a>
""",
            """<a href="https://hurl.dev/player.html?id=hurl&speed=3"><img src="https://hurl.dev/assets/img/poster-hurl.png" width="100%" alt="Hurl Demo"/></a>
""",
        ),
        (
            """\
<div class="picture">
    <picture>
        <source srcset="/docs/assets/img/home-waterfall-light.avif" type="image/avif">
        <source srcset="/docs/assets/img/home-waterfall-light.webp" type="image/webp">
        <source srcset="/docs/assets/img/home-waterfall-light.png" type="image/png">
        <img class="u-theme-light u-drop-shadow u-border u-max-width-100" src="/docs/assets/img/home-waterfall-light.png" width="480" alt="HTML report"/>
    </picture>
    <picture>
        <source srcset="/docs/assets/img/home-waterfall-dark.avif" type="image/avif">
        <source srcset="/docs/assets/img/home-waterfall-dark.webp" type="image/webp">
        <source srcset="/docs/assets/img/home-waterfall-dark.png" type="image/png">
        <img class="u-theme-dark u-drop-shadow u-border u-max-width-100" src="/docs/assets/img/home-waterfall-dark.png" width="480" alt="HTML report"/>
    </picture>
</div>
""",
            """\
<picture>
    <source media="(prefers-color-scheme: light)" srcset="/docs/assets/img/home-waterfall-light.png">
    <source media="(prefers-color-scheme: dark)" srcset="/docs/assets/img/home-waterfall-dark.png">
    <img src="/docs/assets/img/home-waterfall-light.png" width="480" alt="HTML report"/>
</picture>
""",
            """\
<img src="https://hurl.dev/assets/img/home-waterfall-light.png" width="480" alt="HTML report"/>
""",
        ),
    ]
    for old, new_for_github, new_for_crates in snippets:
        if dest == "github":
            new = new_for_github
        else:
            new = new_for_crates
        text = text.replace(old, new)
    return text


def main(dest: str) -> int:
    if dest not in ["github", "crates"]:
        sys.stderr.write("build_readme.py [github, crates]\n")
        return os.EX_USAGE

    home = Path("docs/home.md").read_text()
    home_md = build_home_md(text=home)

    sample = Path("docs/samples.md").read_text()
    samples_md = build_sample_md(text=sample)

    manual = Path("docs/manual.md").read_text()
    manual_md = build_manual_md(text=manual)

    installation = Path("docs/installation.md").read_text()
    installation_md = parse_markdown(text=installation)

    # TOC construction
    body_md = MarkdownDoc()
    body_md.extend(samples_md)
    body_md.extend(manual_md)
    body_md.extend(installation_md)
    toc = body_md.toc()
    toc_md = parse_markdown(text=toc)

    # README construction
    readme_md = MarkdownDoc()
    readme_md.extend(home_md)
    readme_md.extend(toc_md)
    readme_md.extend(body_md)
    readme = readme_md.to_text()
    readme = replace(text=readme, dest=dest)

    print(readme)
    return os.EX_OK


if __name__ == "__main__":
    main(dest=sys.argv[1])
