#[derive(Debug)]
struct Test {
	name: String,
	args: Vec<String>,
	fixture: String,
	force_color: String,
	no_color: bool,
}

#[cfg(test)]
mod tests {
	extern crate temp_env;

	use super::*;
	use assert_cmd::prelude::*;
	use std::process::Command;

	#[test]
	fn running_all_e2e_tests() {
		let tests = get_all_tests();
		for test in tests {
			// appending what test this is so we can find them faster
			let appendix = &format!("########{}########", &test.name);

			let force_color_val = if test.force_color.is_empty() {
				None
			} else {
				Some(&test.force_color)
			};
			let no_color_val = if test.no_color { Some(appendix) } else { None };

			temp_env::with_vars(vec![("FORCE_COLOR", force_color_val), ("NO_COLOR", no_color_val)], || {
				let output =
					Command::cargo_bin("cfonts").unwrap().args(&test.args).output().expect("failed to execute process");

				assert_eq!(String::from_utf8_lossy(&output.stdout).to_string() + appendix, test.fixture.clone() + appendix);
			});
		}
	}
}

fn get_all_tests() -> Vec<Test> {
	let supported_characters: String = String::from("abcdefghijklmnopqrstuvwxyz0123456789!?.+-_=@#$%&()/:;,'\\\"");

	let tests = vec![
		Test {
			name: String::from("Default font"),
			args: vec!["test".to_string()],
			fixture: concat!(
				"\n\n",
				" ████████╗ ███████╗ ███████╗ ████████╗\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"    ██║    █████╗   ███████╗    ██║   \n",
				"    ██║    ██╔══╝   ╚════██║    ██║   \n",
				"    ██║    ███████╗ ███████║    ██║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\n"
			)
			.to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Single color"),
			args: vec!["test".to_string(), "-c".to_string(), "red".to_string()],
			fixture: concat!(
				"\n\n",
				" \x1B[31m████████\x1B[39m╗ \x1B[31m███████\x1B[39m╗ \x1B[31m███████\x1B[39m╗ \x1B[31m████████\x1B[39m╗\n",
				" ╚══\x1B[31m██\x1B[39m╔══╝ \x1B[31m██\x1B[39m╔════╝ \x1B[31m██\x1B[39m╔════╝ ╚══\x1B[31m██\x1B[39m╔══╝\n",
				" \x1B[31m   ██\x1B[39m║    \x1B[31m█████\x1B[39m╗   \x1B[31m███████\x1B[39m╗ \x1B[31m   ██\x1B[39m║   \n",
				" \x1B[31m   ██\x1B[39m║    \x1B[31m██\x1B[39m╔══╝   ╚════\x1B[31m██\x1B[39m║ \x1B[31m   ██\x1B[39m║   \n",
				" \x1B[31m   ██\x1B[39m║    \x1B[31m███████\x1B[39m╗ \x1B[31m███████\x1B[39m║ \x1B[31m   ██\x1B[39m║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\n"
			)
			.to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Two colors"),
			args: vec!["test".to_string(), "-c".to_string(), "red,blue".to_string()],
			fixture: concat!("\n\n",
				" \x1B[31m████████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m████████\x1B[39m\x1B[34m╗\x1B[39m\n",
				" \x1B[34m╚══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔══╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔════╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔════╝\x1B[39m \x1B[34m╚══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔══╝\x1B[39m\n",
				" \x1B[31m   ██\x1B[39m\x1B[34m║   \x1B[39m \x1B[31m█████\x1B[39m\x1B[34m╗  \x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m║   \x1B[39m\n",
				" \x1B[31m   ██\x1B[39m\x1B[34m║   \x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔══╝  \x1B[39m \x1B[34m╚════\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m║   \x1B[39m\n",
				" \x1B[31m   ██\x1B[39m\x1B[34m║   \x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m║\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m║   \x1B[39m\n",
				" \x1B[34m   ╚═╝   \x1B[39m \x1B[34m╚══════╝\x1B[39m \x1B[34m╚══════╝\x1B[39m \x1B[34m   ╚═╝   \x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Background colors"),
			args: vec!["test".to_string(), "-b".to_string(), "blue".to_string()],
			fixture: concat!(
				"\x1B[44m\n\n\n",
				" ████████╗ ███████╗ ███████╗ ████████╗\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"    ██║    █████╗   ███████╗    ██║   \n",
				"    ██║    ██╔══╝   ╚════██║    ██║   \n",
				"    ██║    ███████╗ ███████║    ██║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\x1B[49m\n"
			)
			.to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Background hex colors"),
			args: vec!["test".to_string(), "-b".to_string(), "#8f67e9".to_string()],
			fixture: concat!(
				"\x1B[48;2;143;103;233m\n\n\n",
				" ████████╗ ███████╗ ███████╗ ████████╗\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"    ██║    █████╗   ███████╗    ██║   \n",
				"    ██║    ██╔══╝   ╚════██║    ██║   \n",
				"    ██║    ███████╗ ███████║    ██║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\x1B[49m\n"
			)
			.to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Respect FORCE_COLOR=0"),
			args: vec![
				"test".to_string(),
				"-c".to_string(),
				"#ff8800".to_string(),
				"-b".to_string(),
				"#76429f".to_string(),
			],
			fixture: concat!(
				"\n\n\n",
				" ████████╗ ███████╗ ███████╗ ████████╗\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"    ██║    █████╗   ███████╗    ██║   \n",
				"    ██║    ██╔══╝   ╚════██║    ██║   \n",
				"    ██║    ███████╗ ███████║    ██║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\n"
			)
			.to_string(),
			force_color: String::from("0"),
			no_color: false,
		},
		Test {
			name: String::from("Respect FORCE_COLOR=1"),
			args: vec![
				"test".to_string(),
				"-c".to_string(),
				"#ff8800".to_string(),
				"-b".to_string(),
				"#76429f".to_string(),
			],
			fixture: concat!(
				"\x1B[45m\n\n\n",
				" \x1B[93m████████\x1B[39m╗ \x1B[93m███████\x1B[39m╗ \x1B[93m███████\x1B[39m╗ \x1B[93m████████\x1B[39m╗\n",
				" ╚══\x1B[93m██\x1B[39m╔══╝ \x1B[93m██\x1B[39m╔════╝ \x1B[93m██\x1B[39m╔════╝ ╚══\x1B[93m██\x1B[39m╔══╝\n",
				" \x1B[93m   ██\x1B[39m║    \x1B[93m█████\x1B[39m╗   \x1B[93m███████\x1B[39m╗ \x1B[93m   ██\x1B[39m║   \n",
				" \x1B[93m   ██\x1B[39m║    \x1B[93m██\x1B[39m╔══╝   ╚════\x1B[93m██\x1B[39m║ \x1B[93m   ██\x1B[39m║   \n",
				" \x1B[93m   ██\x1B[39m║    \x1B[93m███████\x1B[39m╗ \x1B[93m███████\x1B[39m║ \x1B[93m   ██\x1B[39m║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\x1B[49m\n"
			)
			.to_string(),
			force_color: String::from("1"),
			no_color: false,
		},
		Test {
			name: String::from("Respect FORCE_COLOR=2"),
			args: vec!["test".to_string(), "-c".to_string(), "#ff8800".to_string(), "-b".to_string(), "#76429f".to_string()],
			fixture: concat!("\x1B[48;5;97m\n\n\n",
				" \x1B[38;5;214m████████\x1B[39m╗ \x1B[38;5;214m███████\x1B[39m╗ \x1B[38;5;214m███████\x1B[39m╗ \x1B[38;5;214m████████\x1B[39m╗\n",
				" ╚══\x1B[38;5;214m██\x1B[39m╔══╝ \x1B[38;5;214m██\x1B[39m╔════╝ \x1B[38;5;214m██\x1B[39m╔════╝ ╚══\x1B[38;5;214m██\x1B[39m╔══╝\n",
				" \x1B[38;5;214m   ██\x1B[39m║    \x1B[38;5;214m█████\x1B[39m╗   \x1B[38;5;214m███████\x1B[39m╗ \x1B[38;5;214m   ██\x1B[39m║   \n",
				" \x1B[38;5;214m   ██\x1B[39m║    \x1B[38;5;214m██\x1B[39m╔══╝   ╚════\x1B[38;5;214m██\x1B[39m║ \x1B[38;5;214m   ██\x1B[39m║   \n",
				" \x1B[38;5;214m   ██\x1B[39m║    \x1B[38;5;214m███████\x1B[39m╗ \x1B[38;5;214m███████\x1B[39m║ \x1B[38;5;214m   ██\x1B[39m║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\x1B[49m\n").to_string(),
			force_color: String::from("2"),
			no_color: false,
		},
		Test {
			name: String::from("Respect FORCE_COLOR=3"),
			args: vec!["test".to_string(), "-c".to_string(), "#ff8800".to_string(), "-b".to_string(), "#76429f".to_string()],
			fixture: concat!("\x1B[48;2;118;66;159m\n\n\n",
				" \x1B[38;2;255;136;0m████████\x1B[39m╗ \x1B[38;2;255;136;0m███████\x1B[39m╗ \x1B[38;2;255;136;0m███████\x1B[39m╗ \x1B[38;2;255;136;0m████████\x1B[39m╗\n",
				" ╚══\x1B[38;2;255;136;0m██\x1B[39m╔══╝ \x1B[38;2;255;136;0m██\x1B[39m╔════╝ \x1B[38;2;255;136;0m██\x1B[39m╔════╝ ╚══\x1B[38;2;255;136;0m██\x1B[39m╔══╝\n",
				" \x1B[38;2;255;136;0m   ██\x1B[39m║    \x1B[38;2;255;136;0m█████\x1B[39m╗   \x1B[38;2;255;136;0m███████\x1B[39m╗ \x1B[38;2;255;136;0m   ██\x1B[39m║   \n",
				" \x1B[38;2;255;136;0m   ██\x1B[39m║    \x1B[38;2;255;136;0m██\x1B[39m╔══╝   ╚════\x1B[38;2;255;136;0m██\x1B[39m║ \x1B[38;2;255;136;0m   ██\x1B[39m║   \n",
				" \x1B[38;2;255;136;0m   ██\x1B[39m║    \x1B[38;2;255;136;0m███████\x1B[39m╗ \x1B[38;2;255;136;0m███████\x1B[39m║ \x1B[38;2;255;136;0m   ██\x1B[39m║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\x1B[49m\n").to_string(),
			force_color: String::from("3"),
			no_color: false,
		},
		Test {
			name: String::from("Respect NO_COLOR"),
			args: vec![
				"test".to_string(),
				"-c".to_string(),
				"#ff8800".to_string(),
				"-b".to_string(),
				"#76429f".to_string(),
			],
			fixture: concat!(
				"\n\n\n",
				" ████████╗ ███████╗ ███████╗ ████████╗\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"    ██║    █████╗   ███████╗    ██║   \n",
				"    ██║    ██╔══╝   ╚════██║    ██║   \n",
				"    ██║    ███████╗ ███████║    ██║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\n"
			)
			.to_string(),
			force_color: String::from(""),
			no_color: true,
		},
		Test {
			name: String::from("FORCE_COLOR overrides NO_COLOR"),
			args: vec![
				"test".to_string(),
				"-c".to_string(),
				"#ff8800".to_string(),
			],
			fixture: concat!(
				"\n\n",
				" \x1B[38;2;255;136;0m████████\x1B[39m╗ \x1B[38;2;255;136;0m███████\x1B[39m╗ \x1B[38;2;255;136;0m███████\x1B[39m╗ \x1B[38;2;255;136;0m████████\x1B[39m╗\n",
				" ╚══\x1B[38;2;255;136;0m██\x1B[39m╔══╝ \x1B[38;2;255;136;0m██\x1B[39m╔════╝ \x1B[38;2;255;136;0m██\x1B[39m╔════╝ ╚══\x1B[38;2;255;136;0m██\x1B[39m╔══╝\n",
				" \x1B[38;2;255;136;0m   ██\x1B[39m║    \x1B[38;2;255;136;0m█████\x1B[39m╗   \x1B[38;2;255;136;0m███████\x1B[39m╗ \x1B[38;2;255;136;0m   ██\x1B[39m║   \n",
				" \x1B[38;2;255;136;0m   ██\x1B[39m║    \x1B[38;2;255;136;0m██\x1B[39m╔══╝   ╚════\x1B[38;2;255;136;0m██\x1B[39m║ \x1B[38;2;255;136;0m   ██\x1B[39m║   \n",
				" \x1B[38;2;255;136;0m   ██\x1B[39m║    \x1B[38;2;255;136;0m███████\x1B[39m╗ \x1B[38;2;255;136;0m███████\x1B[39m║ \x1B[38;2;255;136;0m   ██\x1B[39m║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\n"
			)
			.to_string(),
			force_color: String::from("3"),
			no_color: true,
		},
		Test {
			name: String::from("Align center"),
			args: vec!["test".to_string(), "-a".to_string() ,"center".to_string()],
			fixture: concat!("\n\n",
				"                      ████████╗ ███████╗ ███████╗ ████████╗\n",
				"                      ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"                         ██║    █████╗   ███████╗    ██║   \n",
				"                         ██║    ██╔══╝   ╚════██║    ██║   \n",
				"                         ██║    ███████╗ ███████║    ██║   \n",
				"                         ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Align right"),
			args: vec!["test".to_string(), "-a".to_string() ,"right".to_string()],
			fixture: concat!("\n\n",
				"                                           ████████╗ ███████╗ ███████╗ ████████╗\n",
				"                                           ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"                                              ██║    █████╗   ███████╗    ██║   \n",
				"                                              ██║    ██╔══╝   ╚════██║    ██║   \n",
				"                                              ██║    ███████╗ ███████║    ██║   \n",
				"                                              ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Align top"),
			args: vec!["test".to_string(), "-a".to_string() ,"top".to_string()],
			fixture: concat!("",
				" ████████╗ ███████╗ ███████╗ ████████╗\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"    ██║    █████╗   ███████╗    ██║   \n",
				"    ██║    ██╔══╝   ╚════██║    ██║   \n",
				"    ██║    ███████╗ ███████║    ██║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\n\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Align bottom"),
			args: vec!["test".to_string(), "-a".to_string() ,"bottom".to_string()],
			fixture: concat!("\n\n\n\n",
				" ████████╗ ███████╗ ███████╗ ████████╗\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"    ██║    █████╗   ███████╗    ██║   \n",
				"    ██║    ██╔══╝   ╚════██║    ██║   \n",
				"    ██║    ███████╗ ███████║    ██║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Letter spacing 2"),
			args: vec!["test".to_string(), "-l".to_string() ,"2".to_string()],
			fixture: concat!("\n\n",
				"  ████████╗  ███████╗  ███████╗  ████████╗\n",
				"  ╚══██╔══╝  ██╔════╝  ██╔════╝  ╚══██╔══╝\n",
				"     ██║     █████╗    ███████╗     ██║   \n",
				"     ██║     ██╔══╝    ╚════██║     ██║   \n",
				"     ██║     ███████╗  ███████║     ██║   \n",
				"     ╚═╝     ╚══════╝  ╚══════╝     ╚═╝   \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Letter spacing 10"),
			args: vec!["test".to_string(), "-l".to_string() ,"10".to_string()],
			fixture: concat!("\n\n",
				"          ████████╗          ███████╗          ███████╗          ████████╗\n",
				"          ╚══██╔══╝          ██╔════╝          ██╔════╝          ╚══██╔══╝\n",
				"             ██║             █████╗            ███████╗             ██║   \n",
				"             ██║             ██╔══╝            ╚════██║             ██║   \n",
				"             ██║             ███████╗          ███████║             ██║   \n",
				"             ╚═╝             ╚══════╝          ╚══════╝             ╚═╝   \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Line break"),
			args: vec!["test|line".to_string()],
			fixture: concat!("\n\n",
				" ████████╗ ███████╗ ███████╗ ████████╗\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"    ██║    █████╗   ███████╗    ██║   \n",
				"    ██║    ██╔══╝   ╚════██║    ██║   \n",
				"    ██║    ███████╗ ███████║    ██║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n",
				" ██╗      ██╗ ███╗   ██╗ ███████╗\n",
				" ██║      ██║ ████╗  ██║ ██╔════╝\n",
				" ██║      ██║ ██╔██╗ ██║ █████╗  \n",
				" ██║      ██║ ██║╚██╗██║ ██╔══╝  \n",
				" ███████╗ ██║ ██║ ╚████║ ███████╗\n",
				" ╚══════╝ ╚═╝ ╚═╝  ╚═══╝ ╚══════╝\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Line height"),
			args: vec!["test|line".to_string(), "-z".to_string(), "5".to_string()],
			fixture: concat!("\n\n",
				" ████████╗ ███████╗ ███████╗ ████████╗\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"    ██║    █████╗   ███████╗    ██║   \n",
				"    ██║    ██╔══╝   ╚════██║    ██║   \n",
				"    ██║    ███████╗ ███████║    ██║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n\n\n\n\n",
				" ██╗      ██╗ ███╗   ██╗ ███████╗\n",
				" ██║      ██║ ████╗  ██║ ██╔════╝\n",
				" ██║      ██║ ██╔██╗ ██║ █████╗  \n",
				" ██║      ██║ ██║╚██╗██║ ██╔══╝  \n",
				" ███████╗ ██║ ██║ ╚████║ ███████╗\n",
				" ╚══════╝ ╚═╝ ╚═╝  ╚═══╝ ╚══════╝\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Spaceless"),
			args: vec!["test".to_string(), "-s".to_string()],
			fixture: concat!("",
				" ████████╗ ███████╗ ███████╗ ████████╗\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝\n",
				"    ██║    █████╗   ███████╗    ██║   \n",
				"    ██║    ██╔══╝   ╚════██║    ██║   \n",
				"    ██║    ███████╗ ███████║    ██║   \n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Max length at 2"),
			args: vec!["test".to_string(), "-m".to_string(), "2".to_string()],
			fixture: concat!("\n\n",
				" ████████╗ ███████╗\n",
				" ╚══██╔══╝ ██╔════╝\n",
				"    ██║    █████╗  \n",
				"    ██║    ██╔══╝  \n",
				"    ██║    ███████╗\n",
				"    ╚═╝    ╚══════╝\n",
				"\n",
				" ███████╗ ████████╗\n",
				" ██╔════╝ ╚══██╔══╝\n",
				" ███████╗    ██║   \n",
				" ╚════██║    ██║   \n",
				" ███████║    ██║   \n",
				" ╚══════╝    ╚═╝   \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Max length at 3"),
			args: vec!["test".to_string(), "-m".to_string(), "3".to_string()],
			fixture: concat!("\n\n",
				" ████████╗ ███████╗ ███████╗\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝\n",
				"    ██║    █████╗   ███████╗\n",
				"    ██║    ██╔══╝   ╚════██║\n",
				"    ██║    ███████╗ ███████║\n",
				"    ╚═╝    ╚══════╝ ╚══════╝\n",
				"\n",
				" ████████╗\n",
				" ╚══██╔══╝\n",
				"    ██║   \n",
				"    ██║   \n",
				"    ██║   \n",
				"    ╚═╝   \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Gradient"),
			args: vec!["test".to_string(), "-g".to_string(), "red,green".to_string()],
			fixture: concat!("\n\n",
				" \x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;28m█\x1B[39m\x1B[38;2;255;0;56m█\x1B[39m\x1B[38;2;255;0;84m█\x1B[39m\x1B[38;2;255;0;113m█\x1B[39m\x1B[38;2;255;0;141m█\x1B[39m\x1B[38;2;255;0;170m█\x1B[39m\x1B[38;2;255;0;198m█\x1B[39m\x1B[38;2;255;0;226m╗\x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m█\x1B[39m\x1B[38;2;198;0;255m█\x1B[39m\x1B[38;2;170;0;255m█\x1B[39m\x1B[38;2;141;0;255m█\x1B[39m\x1B[38;2;113;0;255m█\x1B[39m\x1B[38;2;84;0;255m█\x1B[39m\x1B[38;2;56;0;255m█\x1B[39m\x1B[38;2;28;0;255m╗\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m█\x1B[39m\x1B[38;2;0;56;255m█\x1B[39m\x1B[38;2;0;85;255m█\x1B[39m\x1B[38;2;0;113;255m█\x1B[39m\x1B[38;2;0;141;255m█\x1B[39m\x1B[38;2;0;169;255m█\x1B[39m\x1B[38;2;0;198;255m█\x1B[39m\x1B[38;2;0;226;255m╗\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m█\x1B[39m\x1B[38;2;0;255;198m█\x1B[39m\x1B[38;2;0;255;169m█\x1B[39m\x1B[38;2;0;255;141m█\x1B[39m\x1B[38;2;0;255;113m█\x1B[39m\x1B[38;2;0;255;85m█\x1B[39m\x1B[38;2;0;255;56m█\x1B[39m\x1B[38;2;0;255;28m█\x1B[39m\x1B[38;2;0;255;0m╗\x1B[39m\n",
				" \x1B[38;2;255;0;0m╚\x1B[39m\x1B[38;2;255;0;28m═\x1B[39m\x1B[38;2;255;0;56m═\x1B[39m\x1B[38;2;255;0;84m█\x1B[39m\x1B[38;2;255;0;113m█\x1B[39m\x1B[38;2;255;0;141m╔\x1B[39m\x1B[38;2;255;0;170m═\x1B[39m\x1B[38;2;255;0;198m═\x1B[39m\x1B[38;2;255;0;226m╝\x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m█\x1B[39m\x1B[38;2;198;0;255m█\x1B[39m\x1B[38;2;170;0;255m╔\x1B[39m\x1B[38;2;141;0;255m═\x1B[39m\x1B[38;2;113;0;255m═\x1B[39m\x1B[38;2;84;0;255m═\x1B[39m\x1B[38;2;56;0;255m═\x1B[39m\x1B[38;2;28;0;255m╝\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m█\x1B[39m\x1B[38;2;0;56;255m█\x1B[39m\x1B[38;2;0;85;255m╔\x1B[39m\x1B[38;2;0;113;255m═\x1B[39m\x1B[38;2;0;141;255m═\x1B[39m\x1B[38;2;0;169;255m═\x1B[39m\x1B[38;2;0;198;255m═\x1B[39m\x1B[38;2;0;226;255m╝\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m╚\x1B[39m\x1B[38;2;0;255;198m═\x1B[39m\x1B[38;2;0;255;169m═\x1B[39m\x1B[38;2;0;255;141m█\x1B[39m\x1B[38;2;0;255;113m█\x1B[39m\x1B[38;2;0;255;85m╔\x1B[39m\x1B[38;2;0;255;56m═\x1B[39m\x1B[38;2;0;255;28m═\x1B[39m\x1B[38;2;0;255;0m╝\x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;28m \x1B[39m\x1B[38;2;255;0;56m \x1B[39m\x1B[38;2;255;0;84m█\x1B[39m\x1B[38;2;255;0;113m█\x1B[39m\x1B[38;2;255;0;141m║\x1B[39m\x1B[38;2;255;0;170m \x1B[39m\x1B[38;2;255;0;198m \x1B[39m\x1B[38;2;255;0;226m \x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m█\x1B[39m\x1B[38;2;198;0;255m█\x1B[39m\x1B[38;2;170;0;255m█\x1B[39m\x1B[38;2;141;0;255m█\x1B[39m\x1B[38;2;113;0;255m█\x1B[39m\x1B[38;2;84;0;255m╗\x1B[39m\x1B[38;2;56;0;255m \x1B[39m\x1B[38;2;28;0;255m \x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m█\x1B[39m\x1B[38;2;0;56;255m█\x1B[39m\x1B[38;2;0;85;255m█\x1B[39m\x1B[38;2;0;113;255m█\x1B[39m\x1B[38;2;0;141;255m█\x1B[39m\x1B[38;2;0;169;255m█\x1B[39m\x1B[38;2;0;198;255m█\x1B[39m\x1B[38;2;0;226;255m╗\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m \x1B[39m\x1B[38;2;0;255;198m \x1B[39m\x1B[38;2;0;255;169m \x1B[39m\x1B[38;2;0;255;141m█\x1B[39m\x1B[38;2;0;255;113m█\x1B[39m\x1B[38;2;0;255;85m║\x1B[39m\x1B[38;2;0;255;56m \x1B[39m\x1B[38;2;0;255;28m \x1B[39m\x1B[38;2;0;255;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;28m \x1B[39m\x1B[38;2;255;0;56m \x1B[39m\x1B[38;2;255;0;84m█\x1B[39m\x1B[38;2;255;0;113m█\x1B[39m\x1B[38;2;255;0;141m║\x1B[39m\x1B[38;2;255;0;170m \x1B[39m\x1B[38;2;255;0;198m \x1B[39m\x1B[38;2;255;0;226m \x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m█\x1B[39m\x1B[38;2;198;0;255m█\x1B[39m\x1B[38;2;170;0;255m╔\x1B[39m\x1B[38;2;141;0;255m═\x1B[39m\x1B[38;2;113;0;255m═\x1B[39m\x1B[38;2;84;0;255m╝\x1B[39m\x1B[38;2;56;0;255m \x1B[39m\x1B[38;2;28;0;255m \x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m╚\x1B[39m\x1B[38;2;0;56;255m═\x1B[39m\x1B[38;2;0;85;255m═\x1B[39m\x1B[38;2;0;113;255m═\x1B[39m\x1B[38;2;0;141;255m═\x1B[39m\x1B[38;2;0;169;255m█\x1B[39m\x1B[38;2;0;198;255m█\x1B[39m\x1B[38;2;0;226;255m║\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m \x1B[39m\x1B[38;2;0;255;198m \x1B[39m\x1B[38;2;0;255;169m \x1B[39m\x1B[38;2;0;255;141m█\x1B[39m\x1B[38;2;0;255;113m█\x1B[39m\x1B[38;2;0;255;85m║\x1B[39m\x1B[38;2;0;255;56m \x1B[39m\x1B[38;2;0;255;28m \x1B[39m\x1B[38;2;0;255;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;28m \x1B[39m\x1B[38;2;255;0;56m \x1B[39m\x1B[38;2;255;0;84m█\x1B[39m\x1B[38;2;255;0;113m█\x1B[39m\x1B[38;2;255;0;141m║\x1B[39m\x1B[38;2;255;0;170m \x1B[39m\x1B[38;2;255;0;198m \x1B[39m\x1B[38;2;255;0;226m \x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m█\x1B[39m\x1B[38;2;198;0;255m█\x1B[39m\x1B[38;2;170;0;255m█\x1B[39m\x1B[38;2;141;0;255m█\x1B[39m\x1B[38;2;113;0;255m█\x1B[39m\x1B[38;2;84;0;255m█\x1B[39m\x1B[38;2;56;0;255m█\x1B[39m\x1B[38;2;28;0;255m╗\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m█\x1B[39m\x1B[38;2;0;56;255m█\x1B[39m\x1B[38;2;0;85;255m█\x1B[39m\x1B[38;2;0;113;255m█\x1B[39m\x1B[38;2;0;141;255m█\x1B[39m\x1B[38;2;0;169;255m█\x1B[39m\x1B[38;2;0;198;255m█\x1B[39m\x1B[38;2;0;226;255m║\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m \x1B[39m\x1B[38;2;0;255;198m \x1B[39m\x1B[38;2;0;255;169m \x1B[39m\x1B[38;2;0;255;141m█\x1B[39m\x1B[38;2;0;255;113m█\x1B[39m\x1B[38;2;0;255;85m║\x1B[39m\x1B[38;2;0;255;56m \x1B[39m\x1B[38;2;0;255;28m \x1B[39m\x1B[38;2;0;255;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;28m \x1B[39m\x1B[38;2;255;0;56m \x1B[39m\x1B[38;2;255;0;84m╚\x1B[39m\x1B[38;2;255;0;113m═\x1B[39m\x1B[38;2;255;0;141m╝\x1B[39m\x1B[38;2;255;0;170m \x1B[39m\x1B[38;2;255;0;198m \x1B[39m\x1B[38;2;255;0;226m \x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m╚\x1B[39m\x1B[38;2;198;0;255m═\x1B[39m\x1B[38;2;170;0;255m═\x1B[39m\x1B[38;2;141;0;255m═\x1B[39m\x1B[38;2;113;0;255m═\x1B[39m\x1B[38;2;84;0;255m═\x1B[39m\x1B[38;2;56;0;255m═\x1B[39m\x1B[38;2;28;0;255m╝\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m╚\x1B[39m\x1B[38;2;0;56;255m═\x1B[39m\x1B[38;2;0;85;255m═\x1B[39m\x1B[38;2;0;113;255m═\x1B[39m\x1B[38;2;0;141;255m═\x1B[39m\x1B[38;2;0;169;255m═\x1B[39m\x1B[38;2;0;198;255m═\x1B[39m\x1B[38;2;0;226;255m╝\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m \x1B[39m\x1B[38;2;0;255;198m \x1B[39m\x1B[38;2;0;255;169m \x1B[39m\x1B[38;2;0;255;141m╚\x1B[39m\x1B[38;2;0;255;113m═\x1B[39m\x1B[38;2;0;255;85m╝\x1B[39m\x1B[38;2;0;255;56m \x1B[39m\x1B[38;2;0;255;28m \x1B[39m\x1B[38;2;0;255;0m \x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Gradient independent"),
			args: vec!["test|x".to_string(), "-g".to_string(), "red,green".to_string(), "-i".to_string()],
			fixture: concat!("\n\n",
				" \x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;28m█\x1B[39m\x1B[38;2;255;0;56m█\x1B[39m\x1B[38;2;255;0;84m█\x1B[39m\x1B[38;2;255;0;113m█\x1B[39m\x1B[38;2;255;0;141m█\x1B[39m\x1B[38;2;255;0;170m█\x1B[39m\x1B[38;2;255;0;198m█\x1B[39m\x1B[38;2;255;0;226m╗\x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m█\x1B[39m\x1B[38;2;198;0;255m█\x1B[39m\x1B[38;2;170;0;255m█\x1B[39m\x1B[38;2;141;0;255m█\x1B[39m\x1B[38;2;113;0;255m█\x1B[39m\x1B[38;2;84;0;255m█\x1B[39m\x1B[38;2;56;0;255m█\x1B[39m\x1B[38;2;28;0;255m╗\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m█\x1B[39m\x1B[38;2;0;56;255m█\x1B[39m\x1B[38;2;0;85;255m█\x1B[39m\x1B[38;2;0;113;255m█\x1B[39m\x1B[38;2;0;141;255m█\x1B[39m\x1B[38;2;0;169;255m█\x1B[39m\x1B[38;2;0;198;255m█\x1B[39m\x1B[38;2;0;226;255m╗\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m█\x1B[39m\x1B[38;2;0;255;198m█\x1B[39m\x1B[38;2;0;255;169m█\x1B[39m\x1B[38;2;0;255;141m█\x1B[39m\x1B[38;2;0;255;113m█\x1B[39m\x1B[38;2;0;255;85m█\x1B[39m\x1B[38;2;0;255;56m█\x1B[39m\x1B[38;2;0;255;28m█\x1B[39m\x1B[38;2;0;255;0m╗\x1B[39m\n",
				" \x1B[38;2;255;0;0m╚\x1B[39m\x1B[38;2;255;0;28m═\x1B[39m\x1B[38;2;255;0;56m═\x1B[39m\x1B[38;2;255;0;84m█\x1B[39m\x1B[38;2;255;0;113m█\x1B[39m\x1B[38;2;255;0;141m╔\x1B[39m\x1B[38;2;255;0;170m═\x1B[39m\x1B[38;2;255;0;198m═\x1B[39m\x1B[38;2;255;0;226m╝\x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m█\x1B[39m\x1B[38;2;198;0;255m█\x1B[39m\x1B[38;2;170;0;255m╔\x1B[39m\x1B[38;2;141;0;255m═\x1B[39m\x1B[38;2;113;0;255m═\x1B[39m\x1B[38;2;84;0;255m═\x1B[39m\x1B[38;2;56;0;255m═\x1B[39m\x1B[38;2;28;0;255m╝\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m█\x1B[39m\x1B[38;2;0;56;255m█\x1B[39m\x1B[38;2;0;85;255m╔\x1B[39m\x1B[38;2;0;113;255m═\x1B[39m\x1B[38;2;0;141;255m═\x1B[39m\x1B[38;2;0;169;255m═\x1B[39m\x1B[38;2;0;198;255m═\x1B[39m\x1B[38;2;0;226;255m╝\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m╚\x1B[39m\x1B[38;2;0;255;198m═\x1B[39m\x1B[38;2;0;255;169m═\x1B[39m\x1B[38;2;0;255;141m█\x1B[39m\x1B[38;2;0;255;113m█\x1B[39m\x1B[38;2;0;255;85m╔\x1B[39m\x1B[38;2;0;255;56m═\x1B[39m\x1B[38;2;0;255;28m═\x1B[39m\x1B[38;2;0;255;0m╝\x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;28m \x1B[39m\x1B[38;2;255;0;56m \x1B[39m\x1B[38;2;255;0;84m█\x1B[39m\x1B[38;2;255;0;113m█\x1B[39m\x1B[38;2;255;0;141m║\x1B[39m\x1B[38;2;255;0;170m \x1B[39m\x1B[38;2;255;0;198m \x1B[39m\x1B[38;2;255;0;226m \x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m█\x1B[39m\x1B[38;2;198;0;255m█\x1B[39m\x1B[38;2;170;0;255m█\x1B[39m\x1B[38;2;141;0;255m█\x1B[39m\x1B[38;2;113;0;255m█\x1B[39m\x1B[38;2;84;0;255m╗\x1B[39m\x1B[38;2;56;0;255m \x1B[39m\x1B[38;2;28;0;255m \x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m█\x1B[39m\x1B[38;2;0;56;255m█\x1B[39m\x1B[38;2;0;85;255m█\x1B[39m\x1B[38;2;0;113;255m█\x1B[39m\x1B[38;2;0;141;255m█\x1B[39m\x1B[38;2;0;169;255m█\x1B[39m\x1B[38;2;0;198;255m█\x1B[39m\x1B[38;2;0;226;255m╗\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m \x1B[39m\x1B[38;2;0;255;198m \x1B[39m\x1B[38;2;0;255;169m \x1B[39m\x1B[38;2;0;255;141m█\x1B[39m\x1B[38;2;0;255;113m█\x1B[39m\x1B[38;2;0;255;85m║\x1B[39m\x1B[38;2;0;255;56m \x1B[39m\x1B[38;2;0;255;28m \x1B[39m\x1B[38;2;0;255;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;28m \x1B[39m\x1B[38;2;255;0;56m \x1B[39m\x1B[38;2;255;0;84m█\x1B[39m\x1B[38;2;255;0;113m█\x1B[39m\x1B[38;2;255;0;141m║\x1B[39m\x1B[38;2;255;0;170m \x1B[39m\x1B[38;2;255;0;198m \x1B[39m\x1B[38;2;255;0;226m \x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m█\x1B[39m\x1B[38;2;198;0;255m█\x1B[39m\x1B[38;2;170;0;255m╔\x1B[39m\x1B[38;2;141;0;255m═\x1B[39m\x1B[38;2;113;0;255m═\x1B[39m\x1B[38;2;84;0;255m╝\x1B[39m\x1B[38;2;56;0;255m \x1B[39m\x1B[38;2;28;0;255m \x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m╚\x1B[39m\x1B[38;2;0;56;255m═\x1B[39m\x1B[38;2;0;85;255m═\x1B[39m\x1B[38;2;0;113;255m═\x1B[39m\x1B[38;2;0;141;255m═\x1B[39m\x1B[38;2;0;169;255m█\x1B[39m\x1B[38;2;0;198;255m█\x1B[39m\x1B[38;2;0;226;255m║\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m \x1B[39m\x1B[38;2;0;255;198m \x1B[39m\x1B[38;2;0;255;169m \x1B[39m\x1B[38;2;0;255;141m█\x1B[39m\x1B[38;2;0;255;113m█\x1B[39m\x1B[38;2;0;255;85m║\x1B[39m\x1B[38;2;0;255;56m \x1B[39m\x1B[38;2;0;255;28m \x1B[39m\x1B[38;2;0;255;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;28m \x1B[39m\x1B[38;2;255;0;56m \x1B[39m\x1B[38;2;255;0;84m█\x1B[39m\x1B[38;2;255;0;113m█\x1B[39m\x1B[38;2;255;0;141m║\x1B[39m\x1B[38;2;255;0;170m \x1B[39m\x1B[38;2;255;0;198m \x1B[39m\x1B[38;2;255;0;226m \x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m█\x1B[39m\x1B[38;2;198;0;255m█\x1B[39m\x1B[38;2;170;0;255m█\x1B[39m\x1B[38;2;141;0;255m█\x1B[39m\x1B[38;2;113;0;255m█\x1B[39m\x1B[38;2;84;0;255m█\x1B[39m\x1B[38;2;56;0;255m█\x1B[39m\x1B[38;2;28;0;255m╗\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m█\x1B[39m\x1B[38;2;0;56;255m█\x1B[39m\x1B[38;2;0;85;255m█\x1B[39m\x1B[38;2;0;113;255m█\x1B[39m\x1B[38;2;0;141;255m█\x1B[39m\x1B[38;2;0;169;255m█\x1B[39m\x1B[38;2;0;198;255m█\x1B[39m\x1B[38;2;0;226;255m║\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m \x1B[39m\x1B[38;2;0;255;198m \x1B[39m\x1B[38;2;0;255;169m \x1B[39m\x1B[38;2;0;255;141m█\x1B[39m\x1B[38;2;0;255;113m█\x1B[39m\x1B[38;2;0;255;85m║\x1B[39m\x1B[38;2;0;255;56m \x1B[39m\x1B[38;2;0;255;28m \x1B[39m\x1B[38;2;0;255;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;28m \x1B[39m\x1B[38;2;255;0;56m \x1B[39m\x1B[38;2;255;0;84m╚\x1B[39m\x1B[38;2;255;0;113m═\x1B[39m\x1B[38;2;255;0;141m╝\x1B[39m\x1B[38;2;255;0;170m \x1B[39m\x1B[38;2;255;0;198m \x1B[39m\x1B[38;2;255;0;226m \x1B[39m\x1B[38;2;255;0;255m \x1B[39m\x1B[38;2;226;0;255m╚\x1B[39m\x1B[38;2;198;0;255m═\x1B[39m\x1B[38;2;170;0;255m═\x1B[39m\x1B[38;2;141;0;255m═\x1B[39m\x1B[38;2;113;0;255m═\x1B[39m\x1B[38;2;84;0;255m═\x1B[39m\x1B[38;2;56;0;255m═\x1B[39m\x1B[38;2;28;0;255m╝\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;0;28;255m╚\x1B[39m\x1B[38;2;0;56;255m═\x1B[39m\x1B[38;2;0;85;255m═\x1B[39m\x1B[38;2;0;113;255m═\x1B[39m\x1B[38;2;0;141;255m═\x1B[39m\x1B[38;2;0;169;255m═\x1B[39m\x1B[38;2;0;198;255m═\x1B[39m\x1B[38;2;0;226;255m╝\x1B[39m\x1B[38;2;0;255;255m \x1B[39m\x1B[38;2;0;255;226m \x1B[39m\x1B[38;2;0;255;198m \x1B[39m\x1B[38;2;0;255;169m \x1B[39m\x1B[38;2;0;255;141m╚\x1B[39m\x1B[38;2;0;255;113m═\x1B[39m\x1B[38;2;0;255;85m╝\x1B[39m\x1B[38;2;0;255;56m \x1B[39m\x1B[38;2;0;255;28m \x1B[39m\x1B[38;2;0;255;0m \x1B[39m\n",
				"\n",
				" \x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;145m█\x1B[39m\x1B[38;2;218;0;255m╗\x1B[39m\x1B[38;2;72;0;255m \x1B[39m\x1B[38;2;0;72;255m \x1B[39m\x1B[38;2;0;218;255m█\x1B[39m\x1B[38;2;0;255;145m█\x1B[39m\x1B[38;2;0;255;0m╗\x1B[39m\n",
				" \x1B[38;2;255;0;0m╚\x1B[39m\x1B[38;2;255;0;145m█\x1B[39m\x1B[38;2;218;0;255m█\x1B[39m\x1B[38;2;72;0;255m╗\x1B[39m\x1B[38;2;0;72;255m█\x1B[39m\x1B[38;2;0;218;255m█\x1B[39m\x1B[38;2;0;255;145m╔\x1B[39m\x1B[38;2;0;255;0m╝\x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;145m╚\x1B[39m\x1B[38;2;218;0;255m█\x1B[39m\x1B[38;2;72;0;255m█\x1B[39m\x1B[38;2;0;72;255m█\x1B[39m\x1B[38;2;0;218;255m╔\x1B[39m\x1B[38;2;0;255;145m╝\x1B[39m\x1B[38;2;0;255;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;145m█\x1B[39m\x1B[38;2;218;0;255m█\x1B[39m\x1B[38;2;72;0;255m╔\x1B[39m\x1B[38;2;0;72;255m█\x1B[39m\x1B[38;2;0;218;255m█\x1B[39m\x1B[38;2;0;255;145m╗\x1B[39m\x1B[38;2;0;255;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;145m█\x1B[39m\x1B[38;2;218;0;255m╔\x1B[39m\x1B[38;2;72;0;255m╝\x1B[39m\x1B[38;2;0;72;255m \x1B[39m\x1B[38;2;0;218;255m█\x1B[39m\x1B[38;2;0;255;145m█\x1B[39m\x1B[38;2;0;255;0m╗\x1B[39m\n",
				" \x1B[38;2;255;0;0m╚\x1B[39m\x1B[38;2;255;0;145m═\x1B[39m\x1B[38;2;218;0;255m╝\x1B[39m\x1B[38;2;72;0;255m \x1B[39m\x1B[38;2;0;72;255m \x1B[39m\x1B[38;2;0;218;255m╚\x1B[39m\x1B[38;2;0;255;145m═\x1B[39m\x1B[38;2;0;255;0m╝\x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Gradient transition"),
			args: vec!["test".to_string(), "-g".to_string(), "red,#ff8800".to_string(), "-t".to_string()],
			fixture: concat!("\n\n",
				" \x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;3;0m█\x1B[39m\x1B[38;2;255;7;0m█\x1B[39m\x1B[38;2;255;11;0m█\x1B[39m\x1B[38;2;255;15;0m█\x1B[39m\x1B[38;2;255;18;0m█\x1B[39m\x1B[38;2;255;22;0m█\x1B[39m\x1B[38;2;255;26;0m█\x1B[39m\x1B[38;2;255;30;0m╗\x1B[39m\x1B[38;2;255;34;0m \x1B[39m\x1B[38;2;255;37;0m█\x1B[39m\x1B[38;2;255;41;0m█\x1B[39m\x1B[38;2;255;45;0m█\x1B[39m\x1B[38;2;255;49;0m█\x1B[39m\x1B[38;2;255;52;0m█\x1B[39m\x1B[38;2;255;56;0m█\x1B[39m\x1B[38;2;255;60;0m█\x1B[39m\x1B[38;2;255;64;0m╗\x1B[39m\x1B[38;2;255;68;0m \x1B[39m\x1B[38;2;255;71;0m█\x1B[39m\x1B[38;2;255;75;0m█\x1B[39m\x1B[38;2;255;79;0m█\x1B[39m\x1B[38;2;255;83;0m█\x1B[39m\x1B[38;2;255;86;0m█\x1B[39m\x1B[38;2;255;90;0m█\x1B[39m\x1B[38;2;255;94;0m█\x1B[39m\x1B[38;2;255;98;0m╗\x1B[39m\x1B[38;2;255;102;0m \x1B[39m\x1B[38;2;255;105;0m█\x1B[39m\x1B[38;2;255;109;0m█\x1B[39m\x1B[38;2;255;113;0m█\x1B[39m\x1B[38;2;255;117;0m█\x1B[39m\x1B[38;2;255;120;0m█\x1B[39m\x1B[38;2;255;124;0m█\x1B[39m\x1B[38;2;255;128;0m█\x1B[39m\x1B[38;2;255;132;0m█\x1B[39m\x1B[38;2;255;136;0m╗\x1B[39m\n",
				" \x1B[38;2;255;0;0m╚\x1B[39m\x1B[38;2;255;3;0m═\x1B[39m\x1B[38;2;255;7;0m═\x1B[39m\x1B[38;2;255;11;0m█\x1B[39m\x1B[38;2;255;15;0m█\x1B[39m\x1B[38;2;255;18;0m╔\x1B[39m\x1B[38;2;255;22;0m═\x1B[39m\x1B[38;2;255;26;0m═\x1B[39m\x1B[38;2;255;30;0m╝\x1B[39m\x1B[38;2;255;34;0m \x1B[39m\x1B[38;2;255;37;0m█\x1B[39m\x1B[38;2;255;41;0m█\x1B[39m\x1B[38;2;255;45;0m╔\x1B[39m\x1B[38;2;255;49;0m═\x1B[39m\x1B[38;2;255;52;0m═\x1B[39m\x1B[38;2;255;56;0m═\x1B[39m\x1B[38;2;255;60;0m═\x1B[39m\x1B[38;2;255;64;0m╝\x1B[39m\x1B[38;2;255;68;0m \x1B[39m\x1B[38;2;255;71;0m█\x1B[39m\x1B[38;2;255;75;0m█\x1B[39m\x1B[38;2;255;79;0m╔\x1B[39m\x1B[38;2;255;83;0m═\x1B[39m\x1B[38;2;255;86;0m═\x1B[39m\x1B[38;2;255;90;0m═\x1B[39m\x1B[38;2;255;94;0m═\x1B[39m\x1B[38;2;255;98;0m╝\x1B[39m\x1B[38;2;255;102;0m \x1B[39m\x1B[38;2;255;105;0m╚\x1B[39m\x1B[38;2;255;109;0m═\x1B[39m\x1B[38;2;255;113;0m═\x1B[39m\x1B[38;2;255;117;0m█\x1B[39m\x1B[38;2;255;120;0m█\x1B[39m\x1B[38;2;255;124;0m╔\x1B[39m\x1B[38;2;255;128;0m═\x1B[39m\x1B[38;2;255;132;0m═\x1B[39m\x1B[38;2;255;136;0m╝\x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;3;0m \x1B[39m\x1B[38;2;255;7;0m \x1B[39m\x1B[38;2;255;11;0m█\x1B[39m\x1B[38;2;255;15;0m█\x1B[39m\x1B[38;2;255;18;0m║\x1B[39m\x1B[38;2;255;22;0m \x1B[39m\x1B[38;2;255;26;0m \x1B[39m\x1B[38;2;255;30;0m \x1B[39m\x1B[38;2;255;34;0m \x1B[39m\x1B[38;2;255;37;0m█\x1B[39m\x1B[38;2;255;41;0m█\x1B[39m\x1B[38;2;255;45;0m█\x1B[39m\x1B[38;2;255;49;0m█\x1B[39m\x1B[38;2;255;52;0m█\x1B[39m\x1B[38;2;255;56;0m╗\x1B[39m\x1B[38;2;255;60;0m \x1B[39m\x1B[38;2;255;64;0m \x1B[39m\x1B[38;2;255;68;0m \x1B[39m\x1B[38;2;255;71;0m█\x1B[39m\x1B[38;2;255;75;0m█\x1B[39m\x1B[38;2;255;79;0m█\x1B[39m\x1B[38;2;255;83;0m█\x1B[39m\x1B[38;2;255;86;0m█\x1B[39m\x1B[38;2;255;90;0m█\x1B[39m\x1B[38;2;255;94;0m█\x1B[39m\x1B[38;2;255;98;0m╗\x1B[39m\x1B[38;2;255;102;0m \x1B[39m\x1B[38;2;255;105;0m \x1B[39m\x1B[38;2;255;109;0m \x1B[39m\x1B[38;2;255;113;0m \x1B[39m\x1B[38;2;255;117;0m█\x1B[39m\x1B[38;2;255;120;0m█\x1B[39m\x1B[38;2;255;124;0m║\x1B[39m\x1B[38;2;255;128;0m \x1B[39m\x1B[38;2;255;132;0m \x1B[39m\x1B[38;2;255;136;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;3;0m \x1B[39m\x1B[38;2;255;7;0m \x1B[39m\x1B[38;2;255;11;0m█\x1B[39m\x1B[38;2;255;15;0m█\x1B[39m\x1B[38;2;255;18;0m║\x1B[39m\x1B[38;2;255;22;0m \x1B[39m\x1B[38;2;255;26;0m \x1B[39m\x1B[38;2;255;30;0m \x1B[39m\x1B[38;2;255;34;0m \x1B[39m\x1B[38;2;255;37;0m█\x1B[39m\x1B[38;2;255;41;0m█\x1B[39m\x1B[38;2;255;45;0m╔\x1B[39m\x1B[38;2;255;49;0m═\x1B[39m\x1B[38;2;255;52;0m═\x1B[39m\x1B[38;2;255;56;0m╝\x1B[39m\x1B[38;2;255;60;0m \x1B[39m\x1B[38;2;255;64;0m \x1B[39m\x1B[38;2;255;68;0m \x1B[39m\x1B[38;2;255;71;0m╚\x1B[39m\x1B[38;2;255;75;0m═\x1B[39m\x1B[38;2;255;79;0m═\x1B[39m\x1B[38;2;255;83;0m═\x1B[39m\x1B[38;2;255;86;0m═\x1B[39m\x1B[38;2;255;90;0m█\x1B[39m\x1B[38;2;255;94;0m█\x1B[39m\x1B[38;2;255;98;0m║\x1B[39m\x1B[38;2;255;102;0m \x1B[39m\x1B[38;2;255;105;0m \x1B[39m\x1B[38;2;255;109;0m \x1B[39m\x1B[38;2;255;113;0m \x1B[39m\x1B[38;2;255;117;0m█\x1B[39m\x1B[38;2;255;120;0m█\x1B[39m\x1B[38;2;255;124;0m║\x1B[39m\x1B[38;2;255;128;0m \x1B[39m\x1B[38;2;255;132;0m \x1B[39m\x1B[38;2;255;136;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;3;0m \x1B[39m\x1B[38;2;255;7;0m \x1B[39m\x1B[38;2;255;11;0m█\x1B[39m\x1B[38;2;255;15;0m█\x1B[39m\x1B[38;2;255;18;0m║\x1B[39m\x1B[38;2;255;22;0m \x1B[39m\x1B[38;2;255;26;0m \x1B[39m\x1B[38;2;255;30;0m \x1B[39m\x1B[38;2;255;34;0m \x1B[39m\x1B[38;2;255;37;0m█\x1B[39m\x1B[38;2;255;41;0m█\x1B[39m\x1B[38;2;255;45;0m█\x1B[39m\x1B[38;2;255;49;0m█\x1B[39m\x1B[38;2;255;52;0m█\x1B[39m\x1B[38;2;255;56;0m█\x1B[39m\x1B[38;2;255;60;0m█\x1B[39m\x1B[38;2;255;64;0m╗\x1B[39m\x1B[38;2;255;68;0m \x1B[39m\x1B[38;2;255;71;0m█\x1B[39m\x1B[38;2;255;75;0m█\x1B[39m\x1B[38;2;255;79;0m█\x1B[39m\x1B[38;2;255;83;0m█\x1B[39m\x1B[38;2;255;86;0m█\x1B[39m\x1B[38;2;255;90;0m█\x1B[39m\x1B[38;2;255;94;0m█\x1B[39m\x1B[38;2;255;98;0m║\x1B[39m\x1B[38;2;255;102;0m \x1B[39m\x1B[38;2;255;105;0m \x1B[39m\x1B[38;2;255;109;0m \x1B[39m\x1B[38;2;255;113;0m \x1B[39m\x1B[38;2;255;117;0m█\x1B[39m\x1B[38;2;255;120;0m█\x1B[39m\x1B[38;2;255;124;0m║\x1B[39m\x1B[38;2;255;128;0m \x1B[39m\x1B[38;2;255;132;0m \x1B[39m\x1B[38;2;255;136;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;3;0m \x1B[39m\x1B[38;2;255;7;0m \x1B[39m\x1B[38;2;255;11;0m╚\x1B[39m\x1B[38;2;255;15;0m═\x1B[39m\x1B[38;2;255;18;0m╝\x1B[39m\x1B[38;2;255;22;0m \x1B[39m\x1B[38;2;255;26;0m \x1B[39m\x1B[38;2;255;30;0m \x1B[39m\x1B[38;2;255;34;0m \x1B[39m\x1B[38;2;255;37;0m╚\x1B[39m\x1B[38;2;255;41;0m═\x1B[39m\x1B[38;2;255;45;0m═\x1B[39m\x1B[38;2;255;49;0m═\x1B[39m\x1B[38;2;255;52;0m═\x1B[39m\x1B[38;2;255;56;0m═\x1B[39m\x1B[38;2;255;60;0m═\x1B[39m\x1B[38;2;255;64;0m╝\x1B[39m\x1B[38;2;255;68;0m \x1B[39m\x1B[38;2;255;71;0m╚\x1B[39m\x1B[38;2;255;75;0m═\x1B[39m\x1B[38;2;255;79;0m═\x1B[39m\x1B[38;2;255;83;0m═\x1B[39m\x1B[38;2;255;86;0m═\x1B[39m\x1B[38;2;255;90;0m═\x1B[39m\x1B[38;2;255;94;0m═\x1B[39m\x1B[38;2;255;98;0m╝\x1B[39m\x1B[38;2;255;102;0m \x1B[39m\x1B[38;2;255;105;0m \x1B[39m\x1B[38;2;255;109;0m \x1B[39m\x1B[38;2;255;113;0m \x1B[39m\x1B[38;2;255;117;0m╚\x1B[39m\x1B[38;2;255;120;0m═\x1B[39m\x1B[38;2;255;124;0m╝\x1B[39m\x1B[38;2;255;128;0m \x1B[39m\x1B[38;2;255;132;0m \x1B[39m\x1B[38;2;255;136;0m \x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Gradient transition, multiple colors"),
			args: vec!["test".to_string(), "-g".to_string(), "red,#ff8800,blue,red,red".to_string(), "-t".to_string()],
			fixture: concat!("\n\n",
				" \x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;15;0m█\x1B[39m\x1B[38;2;255;30;0m█\x1B[39m\x1B[38;2;255;45;0m█\x1B[39m\x1B[38;2;255;60;0m█\x1B[39m\x1B[38;2;255;75;0m█\x1B[39m\x1B[38;2;255;90;0m█\x1B[39m\x1B[38;2;255;105;0m█\x1B[39m\x1B[38;2;255;120;0m╗\x1B[39m\x1B[38;2;255;136;0m \x1B[39m\x1B[38;2;226;120;28m█\x1B[39m\x1B[38;2;198;105;56m█\x1B[39m\x1B[38;2;170;90;85m█\x1B[39m\x1B[38;2;141;75;113m█\x1B[39m\x1B[38;2;113;60;141m█\x1B[39m\x1B[38;2;85;45;170m█\x1B[39m\x1B[38;2;56;30;198m█\x1B[39m\x1B[38;2;28;15;226m╗\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;28;0;226m█\x1B[39m\x1B[38;2;56;0;198m█\x1B[39m\x1B[38;2;85;0;170m█\x1B[39m\x1B[38;2;113;0;141m█\x1B[39m\x1B[38;2;141;0;113m█\x1B[39m\x1B[38;2;170;0;85m█\x1B[39m\x1B[38;2;198;0;56m█\x1B[39m\x1B[38;2;226;0;28m╗\x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m╗\x1B[39m\n",
				" \x1B[38;2;255;0;0m╚\x1B[39m\x1B[38;2;255;15;0m═\x1B[39m\x1B[38;2;255;30;0m═\x1B[39m\x1B[38;2;255;45;0m█\x1B[39m\x1B[38;2;255;60;0m█\x1B[39m\x1B[38;2;255;75;0m╔\x1B[39m\x1B[38;2;255;90;0m═\x1B[39m\x1B[38;2;255;105;0m═\x1B[39m\x1B[38;2;255;120;0m╝\x1B[39m\x1B[38;2;255;136;0m \x1B[39m\x1B[38;2;226;120;28m█\x1B[39m\x1B[38;2;198;105;56m█\x1B[39m\x1B[38;2;170;90;85m╔\x1B[39m\x1B[38;2;141;75;113m═\x1B[39m\x1B[38;2;113;60;141m═\x1B[39m\x1B[38;2;85;45;170m═\x1B[39m\x1B[38;2;56;30;198m═\x1B[39m\x1B[38;2;28;15;226m╝\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;28;0;226m█\x1B[39m\x1B[38;2;56;0;198m█\x1B[39m\x1B[38;2;85;0;170m╔\x1B[39m\x1B[38;2;113;0;141m═\x1B[39m\x1B[38;2;141;0;113m═\x1B[39m\x1B[38;2;170;0;85m═\x1B[39m\x1B[38;2;198;0;56m═\x1B[39m\x1B[38;2;226;0;28m╝\x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m╚\x1B[39m\x1B[38;2;255;0;0m═\x1B[39m\x1B[38;2;255;0;0m═\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m╔\x1B[39m\x1B[38;2;255;0;0m═\x1B[39m\x1B[38;2;255;0;0m═\x1B[39m\x1B[38;2;255;0;0m╝\x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;15;0m \x1B[39m\x1B[38;2;255;30;0m \x1B[39m\x1B[38;2;255;45;0m█\x1B[39m\x1B[38;2;255;60;0m█\x1B[39m\x1B[38;2;255;75;0m║\x1B[39m\x1B[38;2;255;90;0m \x1B[39m\x1B[38;2;255;105;0m \x1B[39m\x1B[38;2;255;120;0m \x1B[39m\x1B[38;2;255;136;0m \x1B[39m\x1B[38;2;226;120;28m█\x1B[39m\x1B[38;2;198;105;56m█\x1B[39m\x1B[38;2;170;90;85m█\x1B[39m\x1B[38;2;141;75;113m█\x1B[39m\x1B[38;2;113;60;141m█\x1B[39m\x1B[38;2;85;45;170m╗\x1B[39m\x1B[38;2;56;30;198m \x1B[39m\x1B[38;2;28;15;226m \x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;28;0;226m█\x1B[39m\x1B[38;2;56;0;198m█\x1B[39m\x1B[38;2;85;0;170m█\x1B[39m\x1B[38;2;113;0;141m█\x1B[39m\x1B[38;2;141;0;113m█\x1B[39m\x1B[38;2;170;0;85m█\x1B[39m\x1B[38;2;198;0;56m█\x1B[39m\x1B[38;2;226;0;28m╗\x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m║\x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;15;0m \x1B[39m\x1B[38;2;255;30;0m \x1B[39m\x1B[38;2;255;45;0m█\x1B[39m\x1B[38;2;255;60;0m█\x1B[39m\x1B[38;2;255;75;0m║\x1B[39m\x1B[38;2;255;90;0m \x1B[39m\x1B[38;2;255;105;0m \x1B[39m\x1B[38;2;255;120;0m \x1B[39m\x1B[38;2;255;136;0m \x1B[39m\x1B[38;2;226;120;28m█\x1B[39m\x1B[38;2;198;105;56m█\x1B[39m\x1B[38;2;170;90;85m╔\x1B[39m\x1B[38;2;141;75;113m═\x1B[39m\x1B[38;2;113;60;141m═\x1B[39m\x1B[38;2;85;45;170m╝\x1B[39m\x1B[38;2;56;30;198m \x1B[39m\x1B[38;2;28;15;226m \x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;28;0;226m╚\x1B[39m\x1B[38;2;56;0;198m═\x1B[39m\x1B[38;2;85;0;170m═\x1B[39m\x1B[38;2;113;0;141m═\x1B[39m\x1B[38;2;141;0;113m═\x1B[39m\x1B[38;2;170;0;85m█\x1B[39m\x1B[38;2;198;0;56m█\x1B[39m\x1B[38;2;226;0;28m║\x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m║\x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;15;0m \x1B[39m\x1B[38;2;255;30;0m \x1B[39m\x1B[38;2;255;45;0m█\x1B[39m\x1B[38;2;255;60;0m█\x1B[39m\x1B[38;2;255;75;0m║\x1B[39m\x1B[38;2;255;90;0m \x1B[39m\x1B[38;2;255;105;0m \x1B[39m\x1B[38;2;255;120;0m \x1B[39m\x1B[38;2;255;136;0m \x1B[39m\x1B[38;2;226;120;28m█\x1B[39m\x1B[38;2;198;105;56m█\x1B[39m\x1B[38;2;170;90;85m█\x1B[39m\x1B[38;2;141;75;113m█\x1B[39m\x1B[38;2;113;60;141m█\x1B[39m\x1B[38;2;85;45;170m█\x1B[39m\x1B[38;2;56;30;198m█\x1B[39m\x1B[38;2;28;15;226m╗\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;28;0;226m█\x1B[39m\x1B[38;2;56;0;198m█\x1B[39m\x1B[38;2;85;0;170m█\x1B[39m\x1B[38;2;113;0;141m█\x1B[39m\x1B[38;2;141;0;113m█\x1B[39m\x1B[38;2;170;0;85m█\x1B[39m\x1B[38;2;198;0;56m█\x1B[39m\x1B[38;2;226;0;28m║\x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m█\x1B[39m\x1B[38;2;255;0;0m║\x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\n",
				" \x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;15;0m \x1B[39m\x1B[38;2;255;30;0m \x1B[39m\x1B[38;2;255;45;0m╚\x1B[39m\x1B[38;2;255;60;0m═\x1B[39m\x1B[38;2;255;75;0m╝\x1B[39m\x1B[38;2;255;90;0m \x1B[39m\x1B[38;2;255;105;0m \x1B[39m\x1B[38;2;255;120;0m \x1B[39m\x1B[38;2;255;136;0m \x1B[39m\x1B[38;2;226;120;28m╚\x1B[39m\x1B[38;2;198;105;56m═\x1B[39m\x1B[38;2;170;90;85m═\x1B[39m\x1B[38;2;141;75;113m═\x1B[39m\x1B[38;2;113;60;141m═\x1B[39m\x1B[38;2;85;45;170m═\x1B[39m\x1B[38;2;56;30;198m═\x1B[39m\x1B[38;2;28;15;226m╝\x1B[39m\x1B[38;2;0;0;255m \x1B[39m\x1B[38;2;28;0;226m╚\x1B[39m\x1B[38;2;56;0;198m═\x1B[39m\x1B[38;2;85;0;170m═\x1B[39m\x1B[38;2;113;0;141m═\x1B[39m\x1B[38;2;141;0;113m═\x1B[39m\x1B[38;2;170;0;85m═\x1B[39m\x1B[38;2;198;0;56m═\x1B[39m\x1B[38;2;226;0;28m╝\x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m╚\x1B[39m\x1B[38;2;255;0;0m═\x1B[39m\x1B[38;2;255;0;0m╝\x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\x1B[38;2;255;0;0m \x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Browser env"),
			args: vec!["test".to_string(), "-e".to_string(), "browser".to_string()],
			fixture: concat!("",
				"<div style=\"font-family:monospace;white-space:pre;text-align:left;max-width:100%;overflow:scroll;background:transparent\">\n\n",
				" ████████╗ ███████╗ ███████╗ ████████╗<br>\n",
				" ╚══██╔══╝ ██╔════╝ ██╔════╝ ╚══██╔══╝<br>\n",
				"    ██║    █████╗   ███████╗    ██║   <br>\n",
				"    ██║    ██╔══╝   ╚════██║    ██║   <br>\n",
				"    ██║    ███████╗ ███████║    ██║   <br>\n",
				"    ╚═╝    ╚══════╝ ╚══════╝    ╚═╝   \n",
				"\n",
				"</div>\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Browser env with color"),
			args: vec!["test".to_string(), "-e".to_string(), "browser".to_string(), "-c".to_string(), "red,blue,green".to_string()],
			fixture: concat!("",
				"<div style=\"font-family:monospace;white-space:pre;text-align:left;max-width:100%;overflow:scroll;background:transparent\">\n\n",
				" <span style=\"color:#ea3223\">████████</span><span style=\"color:#0020f5\">╗</span> <span style=\"color:#ea3223\">███████</span><span style=\"color:#0020f5\">╗</span> <span style=\"color:#ea3223\">███████</span><span style=\"color:#0020f5\">╗</span> <span style=\"color:#ea3223\">████████</span><span style=\"color:#0020f5\">╗</span><br>\n",
				" <span style=\"color:#0020f5\">╚══</span><span style=\"color:#ea3223\">██</span><span style=\"color:#0020f5\">╔══╝</span> <span style=\"color:#ea3223\">██</span><span style=\"color:#0020f5\">╔════╝</span> <span style=\"color:#ea3223\">██</span><span style=\"color:#0020f5\">╔════╝</span> <span style=\"color:#0020f5\">╚══</span><span style=\"color:#ea3223\">██</span><span style=\"color:#0020f5\">╔══╝</span><br>\n",
				" <span style=\"color:#ea3223\">   ██</span><span style=\"color:#0020f5\">║   </span> <span style=\"color:#ea3223\">█████</span><span style=\"color:#0020f5\">╗  </span> <span style=\"color:#ea3223\">███████</span><span style=\"color:#0020f5\">╗</span> <span style=\"color:#ea3223\">   ██</span><span style=\"color:#0020f5\">║   </span><br>\n",
				" <span style=\"color:#ea3223\">   ██</span><span style=\"color:#0020f5\">║   </span> <span style=\"color:#ea3223\">██</span><span style=\"color:#0020f5\">╔══╝  </span> <span style=\"color:#0020f5\">╚════</span><span style=\"color:#ea3223\">██</span><span style=\"color:#0020f5\">║</span> <span style=\"color:#ea3223\">   ██</span><span style=\"color:#0020f5\">║   </span><br>\n",
				" <span style=\"color:#ea3223\">   ██</span><span style=\"color:#0020f5\">║   </span> <span style=\"color:#ea3223\">███████</span><span style=\"color:#0020f5\">╗</span> <span style=\"color:#ea3223\">███████</span><span style=\"color:#0020f5\">║</span> <span style=\"color:#ea3223\">   ██</span><span style=\"color:#0020f5\">║   </span><br>\n",
				" <span style=\"color:#0020f5\">   ╚═╝   </span> <span style=\"color:#0020f5\">╚══════╝</span> <span style=\"color:#0020f5\">╚══════╝</span> <span style=\"color:#0020f5\">   ╚═╝   </span>\n",
				"\n",
				"</div>\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"block]\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "block".to_string()],
			fixture: concat!("\n\n",
				"  █████╗  ██████╗   ██████╗ ██████╗  ███████╗ ███████╗  ██████╗  ██╗  ██╗ ██╗\n",
				" ██╔══██╗ ██╔══██╗ ██╔════╝ ██╔══██╗ ██╔════╝ ██╔════╝ ██╔════╝  ██║  ██║ ██║\n",
				" ███████║ ██████╔╝ ██║      ██║  ██║ █████╗   █████╗   ██║  ███╗ ███████║ ██║\n",
				" ██╔══██║ ██╔══██╗ ██║      ██║  ██║ ██╔══╝   ██╔══╝   ██║   ██║ ██╔══██║ ██║\n",
				" ██║  ██║ ██████╔╝ ╚██████╗ ██████╔╝ ███████╗ ██║      ╚██████╔╝ ██║  ██║ ██║\n",
				" ╚═╝  ╚═╝ ╚═════╝   ╚═════╝ ╚═════╝  ╚══════╝ ╚═╝       ╚═════╝  ╚═╝  ╚═╝ ╚═╝\n",
				"\n",
				"      ██╗ ██╗  ██╗ ██╗      ███╗   ███╗ ███╗   ██╗  ██████╗  ██████╗   ██████╗ \n",
				"      ██║ ██║ ██╔╝ ██║      ████╗ ████║ ████╗  ██║ ██╔═══██╗ ██╔══██╗ ██╔═══██╗\n",
				"      ██║ █████╔╝  ██║      ██╔████╔██║ ██╔██╗ ██║ ██║   ██║ ██████╔╝ ██║   ██║\n",
				" ██   ██║ ██╔═██╗  ██║      ██║╚██╔╝██║ ██║╚██╗██║ ██║   ██║ ██╔═══╝  ██║▄▄ ██║\n",
				" ╚█████╔╝ ██║  ██╗ ███████╗ ██║ ╚═╝ ██║ ██║ ╚████║ ╚██████╔╝ ██║      ╚██████╔╝\n",
				"  ╚════╝  ╚═╝  ╚═╝ ╚══════╝ ╚═╝     ╚═╝ ╚═╝  ╚═══╝  ╚═════╝  ╚═╝       ╚══▀▀═╝ \n",
				"\n",
				" ██████╗  ███████╗ ████████╗ ██╗   ██╗ ██╗   ██╗ ██╗    ██╗ ██╗  ██╗ ██╗   ██╗\n",
				" ██╔══██╗ ██╔════╝ ╚══██╔══╝ ██║   ██║ ██║   ██║ ██║    ██║ ╚██╗██╔╝ ╚██╗ ██╔╝\n",
				" ██████╔╝ ███████╗    ██║    ██║   ██║ ██║   ██║ ██║ █╗ ██║  ╚███╔╝   ╚████╔╝ \n",
				" ██╔══██╗ ╚════██║    ██║    ██║   ██║ ╚██╗ ██╔╝ ██║███╗██║  ██╔██╗    ╚██╔╝  \n",
				" ██║  ██║ ███████║    ██║    ╚██████╔╝  ╚████╔╝  ╚███╔███╔╝ ██╔╝ ██╗    ██║   \n",
				" ╚═╝  ╚═╝ ╚══════╝    ╚═╝     ╚═════╝    ╚═══╝    ╚══╝╚══╝  ╚═╝  ╚═╝    ╚═╝   \n",
				"\n",
				" ███████╗  ██████╗   ██╗ ██████╗  ██████╗  ██╗  ██╗ ███████╗  ██████╗  ███████╗\n",
				" ╚══███╔╝ ██╔═████╗ ███║ ╚════██╗ ╚════██╗ ██║  ██║ ██╔════╝ ██╔════╝  ╚════██║\n",
				"   ███╔╝  ██║██╔██║ ╚██║  █████╔╝  █████╔╝ ███████║ ███████╗ ███████╗      ██╔╝\n",
				"  ███╔╝   ████╔╝██║  ██║ ██╔═══╝   ╚═══██╗ ╚════██║ ╚════██║ ██╔═══██╗    ██╔╝ \n",
				" ███████╗ ╚██████╔╝  ██║ ███████╗ ██████╔╝      ██║ ███████║ ╚██████╔╝    ██║  \n",
				" ╚══════╝  ╚═════╝   ╚═╝ ╚══════╝ ╚═════╝       ╚═╝ ╚══════╝  ╚═════╝     ╚═╝  \n",
				"\n",
				"  █████╗   █████╗  ██╗ ██████╗                                       ██████╗ \n",
				" ██╔══██╗ ██╔══██╗ ██║ ╚════██╗       ██╗                   ██████╗ ██╔═══██╗\n",
				" ╚█████╔╝ ╚██████║ ██║   ▄███╔╝     ██████╗ █████╗          ╚═════╝ ██║██╗██║\n",
				" ██╔══██╗  ╚═══██║ ╚═╝   ▀▀══╝       ╚██╔═╝ ╚════╝          ██████╗ ██║██║██║\n",
				" ╚█████╔╝  █████╔╝ ██╗   ██╗    ██╗   ╚═╝          ███████╗ ╚═════╝ ╚█║████╔╝\n",
				"  ╚════╝   ╚════╝  ╚═╝   ╚═╝    ╚═╝                ╚══════╝          ╚╝╚═══╝ \n",
				"\n",
				"  ██╗ ██╗  ▄▄███▄▄· ██╗ ██╗    ██╗     ██╗ ██╗      ██╗             █╗  █╗█╗ \n",
				" ████████╗ ██╔════╝ ╚═╝██╔╝    ██║    ██╔╝ ╚██╗    ██╔╝ ██╗         ╚╝  ╚╝╚╝ \n",
				" ╚██╔═██╔╝ ███████╗   ██╔╝  ████████╗ ██║   ██║   ██╔╝  ╚═╝ ██╗              \n",
				" ████████╗ ╚════██║  ██╔╝   ██╔═██╔═╝ ██║   ██║  ██╔╝   ██╗ ╚═╝              \n",
				" ╚██╔═██╔╝ ███████║ ██╔╝██╗ ██████║   ╚██╗ ██╔╝ ██╔╝    ╚═╝ ▄█╗ ▄█╗          \n",
				"  ╚═╝ ╚═╝  ╚═▀▀▀══╝ ╚═╝ ╚═╝ ╚═════╝    ╚═╝ ╚═╝  ╚═╝         ▀═╝ ▀═╝          \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"block\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "block".to_string(), "-c".to_string(), "red,blue".to_string()],
			fixture: concat!("\n\n",
				"  \x1B[31m█████\x1B[39m\x1B[34m╗\x1B[39m  \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m  \x1B[31m██████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m  \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\n",
				" \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔════╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔════╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔════╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔════╝ \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\n",
				" \x1B[31m███████\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║     \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m█████\x1B[39m\x1B[34m╗  \x1B[39m \x1B[31m█████\x1B[39m\x1B[34m╗  \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m  ███\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\n",
				" \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║     \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔══╝  \x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔══╝  \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\n",
				" \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║     \x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\n",
				" \x1B[34m╚═╝  ╚═╝\x1B[39m \x1B[34m╚═════╝ \x1B[39m \x1B[34m ╚═════╝\x1B[39m \x1B[34m╚═════╝ \x1B[39m \x1B[34m╚══════╝\x1B[39m \x1B[34m╚═╝     \x1B[39m \x1B[34m ╚═════╝ \x1B[39m \x1B[34m╚═╝  ╚═╝\x1B[39m \x1B[34m╚═╝\x1B[39m\n",
				"\n",
				" \x1B[31m     ██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗     \x1B[39m \x1B[31m███\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m   ███\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m███\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m╗\x1B[39m  \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m  \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m\n",
				" \x1B[31m     ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m ██\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║     \x1B[39m \x1B[31m████\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m ████\x1B[39m\x1B[34m║\x1B[39m \x1B[31m████\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔═══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔═══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\n",
				" \x1B[31m     ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m█████\x1B[39m\x1B[34m╔╝ \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║     \x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔\x1B[39m\x1B[31m████\x1B[39m\x1B[34m╔\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m║\x1B[39m\n",
				" \x1B[31m██   ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔═\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║     \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔╝\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔═══╝ \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m▄▄ ██\x1B[39m\x1B[34m║\x1B[39m\n",
				" \x1B[34m╚\x1B[39m\x1B[31m█████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║ ╚═╝\x1B[39m\x1B[31m ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║ ╚\x1B[39m\x1B[31m████\x1B[39m\x1B[34m║\x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║     \x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m\n",
				" \x1B[34m ╚════╝ \x1B[39m \x1B[34m╚═╝  ╚═╝\x1B[39m \x1B[34m╚══════╝\x1B[39m \x1B[34m╚═╝     ╚═╝\x1B[39m \x1B[34m╚═╝  ╚═══╝\x1B[39m \x1B[34m ╚═════╝ \x1B[39m \x1B[34m╚═╝     \x1B[39m \x1B[34m ╚══\x1B[39m\x1B[31m▀▀\x1B[39m\x1B[34m═╝ \x1B[39m\n",
				"\n",
				" \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m████████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗    \x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m╗\x1B[39m\n",
				" \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔════╝\x1B[39m \x1B[34m╚══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔══╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║    \x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m ██\x1B[39m\x1B[34m╔╝\x1B[39m\n",
				" \x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m║   \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m █\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m ██\x1B[39m\x1B[34m║\x1B[39m \x1B[34m ╚\x1B[39m\x1B[31m███\x1B[39m\x1B[34m╔╝ \x1B[39m \x1B[34m ╚\x1B[39m\x1B[31m████\x1B[39m\x1B[34m╔╝ \x1B[39m\n",
				" \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[34m╚════\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m║   \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m   ██\x1B[39m\x1B[34m║\x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m ██\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m███\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m  \x1B[31m██\x1B[39m\x1B[34m╔\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗ \x1B[39m \x1B[34m  ╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔╝  \x1B[39m\n",
				" \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m║\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m║   \x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[34m ╚\x1B[39m\x1B[31m████\x1B[39m\x1B[34m╔╝ \x1B[39m \x1B[34m╚\x1B[39m\x1B[31m███\x1B[39m\x1B[34m╔\x1B[39m\x1B[31m███\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔╝\x1B[39m\x1B[31m ██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m║   \x1B[39m\n",
				" \x1B[34m╚═╝  ╚═╝\x1B[39m \x1B[34m╚══════╝\x1B[39m \x1B[34m   ╚═╝   \x1B[39m \x1B[34m ╚═════╝ \x1B[39m \x1B[34m  ╚═══╝  \x1B[39m \x1B[34m ╚══╝╚══╝ \x1B[39m \x1B[34m╚═╝  ╚═╝\x1B[39m \x1B[34m   ╚═╝   \x1B[39m\n",
				"\n",
				" \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m  \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m  \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m  \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m\n",
				" \x1B[34m╚══\x1B[39m\x1B[31m███\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔═\x1B[39m\x1B[31m████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m███\x1B[39m\x1B[34m║\x1B[39m \x1B[34m╚════\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[34m╚════\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m  ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔════╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔════╝ \x1B[39m \x1B[34m╚════\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\n",
				" \x1B[31m  ███\x1B[39m\x1B[34m╔╝ \x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m  \x1B[31m█████\x1B[39m\x1B[34m╔╝\x1B[39m  \x1B[31m█████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m║\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m    ██\x1B[39m\x1B[34m╔╝\x1B[39m\n",
				"  \x1B[31m███\x1B[39m\x1B[34m╔╝  \x1B[39m \x1B[31m████\x1B[39m\x1B[34m╔╝\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m  \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔═══╝ \x1B[39m \x1B[34m ╚═══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[34m╚════\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[34m╚════\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔═══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m╔╝ \x1B[39m\n",
				" \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m  \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m     ██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m║\x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m║  \x1B[39m\n",
				" \x1B[34m╚══════╝\x1B[39m \x1B[34m ╚═════╝ \x1B[39m \x1B[34m ╚═╝\x1B[39m \x1B[34m╚══════╝\x1B[39m \x1B[34m╚═════╝ \x1B[39m \x1B[34m     ╚═╝\x1B[39m \x1B[34m╚══════╝\x1B[39m \x1B[34m ╚═════╝ \x1B[39m \x1B[34m   ╚═╝  \x1B[39m\n",
				"\n",
				"  \x1B[31m█████\x1B[39m\x1B[34m╗ \x1B[39m  \x1B[31m█████\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m                                      \x1B[31m██████\x1B[39m\x1B[34m╗ \x1B[39m\n",
				" \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[34m╚════\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m     \x1B[31m  ██\x1B[39m\x1B[34m╗  \x1B[39m                 \x1B[31m██████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔═══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\n",
				" \x1B[34m╚\x1B[39m\x1B[31m█████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██████\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m  ▄███\x1B[39m\x1B[34m╔╝\x1B[39m     \x1B[31m██████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m█████\x1B[39m\x1B[34m╗\x1B[39m          \x1B[34m╚═════╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\n",
				" \x1B[31m██\x1B[39m\x1B[34m╔══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[34m ╚═══\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[34m╚═╝\x1B[39m \x1B[31m  ▀▀\x1B[39m\x1B[34m══╝ \x1B[39m     \x1B[34m ╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔═╝\x1B[39m \x1B[34m╚════╝\x1B[39m          \x1B[31m██████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m\n",
				" \x1B[34m╚\x1B[39m\x1B[31m█████\x1B[39m\x1B[34m╔╝\x1B[39m  \x1B[31m█████\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m  ██\x1B[39m\x1B[34m╗   \x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[34m  ╚═╝  \x1B[39m        \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[34m╚═════╝\x1B[39m \x1B[34m╚\x1B[39m\x1B[31m█\x1B[39m\x1B[34m║\x1B[39m\x1B[31m████\x1B[39m\x1B[34m╔╝\x1B[39m\n",
				" \x1B[34m ╚════╝ \x1B[39m \x1B[34m ╚════╝ \x1B[39m \x1B[34m╚═╝\x1B[39m \x1B[34m  ╚═╝   \x1B[39m \x1B[34m╚═╝\x1B[39m                \x1B[34m╚══════╝\x1B[39m         \x1B[34m ╚╝╚═══╝ \x1B[39m\n",
				"\n",
				"  \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m ██\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m▄▄███▄▄\x1B[39m\x1B[34m·\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m ██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m╗   \x1B[39m  \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗ \x1B[39m \x1B[31m    ██\x1B[39m\x1B[34m╗\x1B[39m             \x1B[31m█\x1B[39m\x1B[34m╗\x1B[39m  \x1B[31m█\x1B[39m\x1B[34m╗\x1B[39m\x1B[31m█\x1B[39m\x1B[34m╗\x1B[39m \n",
				" \x1B[31m████████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔════╝\x1B[39m \x1B[34m╚═╝\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m║   \x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m   ██\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m         \x1B[34m╚╝\x1B[39m  \x1B[34m╚╝╚╝\x1B[39m \n",
				" \x1B[34m╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔═\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m  ██\x1B[39m\x1B[34m╔╝ \x1B[39m \x1B[31m████████\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║   \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m \x1B[31m  ██\x1B[39m\x1B[34m╔╝ \x1B[39m \x1B[34m╚═╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m              \n",
				" \x1B[31m████████\x1B[39m\x1B[34m╗\x1B[39m \x1B[34m╚════\x1B[39m\x1B[31m██\x1B[39m\x1B[34m║\x1B[39m  \x1B[31m██\x1B[39m\x1B[34m╔╝  \x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔═\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔═╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m║   \x1B[31m██\x1B[39m\x1B[34m║\x1B[39m  \x1B[31m██\x1B[39m\x1B[34m╔╝  \x1B[39m \x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[34m╚═╝\x1B[39m              \n",
				" \x1B[34m╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔═\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m███████\x1B[39m\x1B[34m║\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔╝\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██████\x1B[39m\x1B[34m║  \x1B[39m \x1B[34m╚\x1B[39m\x1B[31m██\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔╝\x1B[39m \x1B[31m██\x1B[39m\x1B[34m╔╝   \x1B[39m \x1B[34m╚═╝\x1B[39m \x1B[31m▄█\x1B[39m\x1B[34m╗\x1B[39m \x1B[31m▄█\x1B[39m\x1B[34m╗\x1B[39m          \n",
				" \x1B[34m ╚═╝ ╚═╝ \x1B[39m \x1B[34m╚═\x1B[39m\x1B[31m▀▀▀\x1B[39m\x1B[34m══╝\x1B[39m \x1B[34m╚═╝ ╚═╝\x1B[39m \x1B[34m╚═════╝  \x1B[39m \x1B[34m ╚═╝\x1B[39m \x1B[34m╚═╝ \x1B[39m \x1B[34m╚═╝    \x1B[39m     \x1B[31m▀\x1B[39m\x1B[34m═╝\x1B[39m \x1B[31m▀\x1B[39m\x1B[34m═╝\x1B[39m          \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"console\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "console".to_string()],
			fixture: concat!("\n\n",
				"abcdefghijklmnopqrstuvwxyz0123456789!?.+-_=@#$%&()/:;,'\"\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"console\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "console".to_string(), "-c".to_string(), "red".to_string()],
			fixture: concat!("\n\n",
				"\u{1b}[31ma\u{1b}[39m\u{1b}[31mb\u{1b}[39m\u{1b}[31mc\u{1b}[39m\u{1b}[31md\u{1b}[39m\u{1b}[31me\u{1b}[39m\u{1b}[31mf\u{1b}[39m\u{1b}[31mg\u{1b}[39m\u{1b}[31mh\u{1b}[39m\u{1b}[31mi\u{1b}[39m\u{1b}[31mj\u{1b}[39m\u{1b}[31mk\u{1b}[39m\u{1b}[31ml\u{1b}[39m\u{1b}[31mm\u{1b}[39m\u{1b}[31mn\u{1b}[39m\u{1b}[31mo\u{1b}[39m\u{1b}[31mp\u{1b}[39m\u{1b}[31mq\u{1b}[39m\u{1b}[31mr\u{1b}[39m\u{1b}[31ms\u{1b}[39m\u{1b}[31mt\u{1b}[39m\u{1b}[31mu\u{1b}[39m\u{1b}[31mv\u{1b}[39m\u{1b}[31mw\u{1b}[39m\u{1b}[31mx\u{1b}[39m\u{1b}[31my\u{1b}[39m\u{1b}[31mz\u{1b}[39m\u{1b}[31m0\u{1b}[39m\u{1b}[31m1\u{1b}[39m\u{1b}[31m2\u{1b}[39m\u{1b}[31m3\u{1b}[39m\u{1b}[31m4\u{1b}[39m\u{1b}[31m5\u{1b}[39m\u{1b}[31m6\u{1b}[39m\u{1b}[31m7\u{1b}[39m\u{1b}[31m8\u{1b}[39m\u{1b}[31m9\u{1b}[39m\u{1b}[31m!\u{1b}[39m\u{1b}[31m?\u{1b}[39m\u{1b}[31m.\u{1b}[39m\u{1b}[31m+\u{1b}[39m\u{1b}[31m-\u{1b}[39m\u{1b}[31m_\u{1b}[39m\u{1b}[31m=\u{1b}[39m\u{1b}[31m@\u{1b}[39m\u{1b}[31m#\u{1b}[39m\u{1b}[31m$\u{1b}[39m\u{1b}[31m%\u{1b}[39m\u{1b}[31m&\u{1b}[39m\u{1b}[31m(\u{1b}[39m\u{1b}[31m)\u{1b}[39m\u{1b}[31m/\u{1b}[39m\u{1b}[31m:\u{1b}[39m\u{1b}[31m;\u{1b}[39m\u{1b}[31m,\u{1b}[39m\u{1b}[31m'\u{1b}[39m\u{1b}[31m\"\u{1b}[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"chrome\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "chrome".to_string()],
			fixture: concat!("\n\n",
				" ╔═╗ ╔╗  ╔═╗ ╔╦╗ ╔═╗ ╔═╗ ╔═╗ ╦ ╦ ╦  ╦ ╦╔═ ╦   ╔╦╗ ╔╗╔ ╔═╗ ╔═╗ ╔═╗  ╦═╗ ╔═╗ ╔╦╗\n",
				" ╠═╣ ╠╩╗ ║    ║║ ║╣  ╠╣  ║ ╦ ╠═╣ ║  ║ ╠╩╗ ║   ║║║ ║║║ ║ ║ ╠═╝ ║═╬╗ ╠╦╝ ╚═╗  ║ \n",
				" ╩ ╩ ╚═╝ ╚═╝ ═╩╝ ╚═╝ ╚   ╚═╝ ╩ ╩ ╩ ╚╝ ╩ ╩ ╩═╝ ╩ ╩ ╝╚╝ ╚═╝ ╩   ╚═╝╚ ╩╚═ ╚═╝  ╩ \n",
				"\n",
				" ╦ ╦ ╦  ╦ ╦ ╦  ╦ ╦ ╦ ╦ ╔═╗ ╔═╗ ╗  ═╗ ╔═╗ ╦   ╔═  ╔═╗ ╔═╗ ╔═╗ ╔═╗ ╦ ╔═╗         \n",
				" ║ ║ ╚╗╔╝ ║║║ ╔╩╦╝ ╚╦╝ ╔═╝ ║═║ ║ ╔═╝  ╠║ ╚╬╝ ╚═╗ ╠═╗  ═╣ ╠═╣ ╚═╣ ║  ╔╝   ═╬═ ──\n",
				" ╚═╝  ╚╝  ╚╩╝ ╩ ╩   ╩  ╚═╝ ╚═╝ ║ ╚══ ╚═╝  ╩  ══╝ ╚═╝   ╩ ╚═╝ ╚═╝ o  o  o       \n",
				"\n",
				"       ╔═╗    ╔╬╗ o╔  ╦  ╔ ╗  ╔       ╗ ╗╗\n",
				"    ══ ║╚╝ ╬╬ ╚╬╗ ╔╝ ╔╬═ ║ ║ ╔╝ o o       \n",
				" ──    ╚══ ╬╬ ╚╬╝ ╝o ╚╝  ╚ ╝ ╝  o ╔ ╔     \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"chrome\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "chrome".to_string(), "-c".to_string(), "red,blue,green".to_string()],
			fixture: concat!("\n\n",
				" \x1B[31m╔═╗\x1B[39m \x1B[31m╔╗ \x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╔╦╗\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╦ ╦\x1B[39m \x1B[31m╦\x1B[39m \x1B[31m ╦\x1B[39m \x1B[31m╦╔═\x1B[39m \x1B[31m╦  \x1B[39m \x1B[31m╔╦╗\x1B[39m \x1B[31m╔╗╔\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╔═╗ \x1B[39m \x1B[31m╦═╗\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╔╦╗\x1B[39m\n",
				" \x1B[34m╠═╣\x1B[39m \x1B[34m╠╩╗\x1B[39m \x1B[34m║  \x1B[39m \x1B[34m ║║\x1B[39m \x1B[34m║╣ \x1B[39m \x1B[34m╠╣ \x1B[39m \x1B[34m║ ╦\x1B[39m \x1B[34m╠═╣\x1B[39m \x1B[34m║\x1B[39m \x1B[34m ║\x1B[39m \x1B[34m╠╩╗\x1B[39m \x1B[34m║  \x1B[39m \x1B[34m║║║\x1B[39m \x1B[34m║║║\x1B[39m \x1B[34m║ ║\x1B[39m \x1B[34m╠═╝\x1B[39m \x1B[34m║═╬╗\x1B[39m \x1B[34m╠╦╝\x1B[39m \x1B[34m╚═╗\x1B[39m \x1B[34m ║ \x1B[39m\n",
				" \x1B[32m╩ ╩\x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32m═╩╝\x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32m╚  \x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32m╩ ╩\x1B[39m \x1B[32m╩\x1B[39m \x1B[32m╚╝\x1B[39m \x1B[32m╩ ╩\x1B[39m \x1B[32m╩═╝\x1B[39m \x1B[32m╩ ╩\x1B[39m \x1B[32m╝╚╝\x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32m╩  \x1B[39m \x1B[32m╚═╝╚\x1B[39m \x1B[32m╩╚═\x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32m ╩ \x1B[39m\n",
				"\n",
				" \x1B[31m╦ ╦\x1B[39m \x1B[31m╦  ╦\x1B[39m \x1B[31m╦ ╦\x1B[39m \x1B[31m ╦ ╦\x1B[39m \x1B[31m╦ ╦\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╗\x1B[39m \x1B[31m ═╗\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╦  \x1B[39m \x1B[31m╔═ \x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m╦\x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m \x1B[39m \x1B[31m   \x1B[39m \x1B[31m  \x1B[39m\n",
				" \x1B[34m║ ║\x1B[39m \x1B[34m╚╗╔╝\x1B[39m \x1B[34m║║║\x1B[39m \x1B[34m╔╩╦╝\x1B[39m \x1B[34m╚╦╝\x1B[39m \x1B[34m╔═╝\x1B[39m \x1B[34m║═║\x1B[39m \x1B[34m║\x1B[39m \x1B[34m╔═╝\x1B[39m \x1B[34m ╠║\x1B[39m \x1B[34m╚╬╝\x1B[39m \x1B[34m╚═╗\x1B[39m \x1B[34m╠═╗\x1B[39m \x1B[34m ═╣\x1B[39m \x1B[34m╠═╣\x1B[39m \x1B[34m╚═╣\x1B[39m \x1B[34m║\x1B[39m \x1B[34m ╔╝\x1B[39m \x1B[34m \x1B[39m \x1B[34m═╬═\x1B[39m \x1B[34m──\x1B[39m\n",
				" \x1B[32m╚═╝\x1B[39m \x1B[32m ╚╝ \x1B[39m \x1B[32m╚╩╝\x1B[39m \x1B[32m╩ ╩ \x1B[39m \x1B[32m ╩ \x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32m║\x1B[39m \x1B[32m╚══\x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32m ╩ \x1B[39m \x1B[32m══╝\x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32m  ╩\x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32m╚═╝\x1B[39m \x1B[32mo\x1B[39m \x1B[32m o \x1B[39m \x1B[32mo\x1B[39m \x1B[32m   \x1B[39m \x1B[32m  \x1B[39m\n",
				"\n",
				" \x1B[31m  \x1B[39m \x1B[31m  \x1B[39m \x1B[31m╔═╗\x1B[39m \x1B[31m  \x1B[39m \x1B[31m╔╬╗\x1B[39m \x1B[31mo╔\x1B[39m \x1B[31m ╦ \x1B[39m \x1B[31m╔\x1B[39m \x1B[31m╗\x1B[39m \x1B[31m ╔\x1B[39m \x1B[31m \x1B[39m \x1B[31m \x1B[39m \x1B[31m \x1B[39m \x1B[31m╗\x1B[39m \x1B[31m╗╗\x1B[39m\n",
				" \x1B[34m  \x1B[39m \x1B[34m══\x1B[39m \x1B[34m║╚╝\x1B[39m \x1B[34m╬╬\x1B[39m \x1B[34m╚╬╗\x1B[39m \x1B[34m╔╝\x1B[39m \x1B[34m╔╬═\x1B[39m \x1B[34m║\x1B[39m \x1B[34m║\x1B[39m \x1B[34m╔╝\x1B[39m \x1B[34mo\x1B[39m \x1B[34mo\x1B[39m \x1B[31m \x1B[39m \x1B[31m \x1B[39m \x1B[31m  \x1B[39m\n",
				" \x1B[32m──\x1B[39m \x1B[32m  \x1B[39m \x1B[32m╚══\x1B[39m \x1B[32m╬╬\x1B[39m \x1B[32m╚╬╝\x1B[39m \x1B[32m╝o\x1B[39m \x1B[32m╚╝ \x1B[39m \x1B[32m╚\x1B[39m \x1B[32m╝\x1B[39m \x1B[32m╝ \x1B[39m \x1B[32mo\x1B[39m \x1B[32m╔\x1B[39m \x1B[32m╔\x1B[39m \x1B[32m \x1B[39m \x1B[32m  \x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"3d\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "3d".to_string()],
			fixture: concat!("\n\n",
				"_____/\\\\\\\\\\\\\\\\\\_____/\\\\\\\\\\\\\\\\\\\\\\\\\\__________/\\\\\\\\\\\\\\\\\\__/\\\\\\\\\\\\\\\\\\\\\\\\___\n",
				" ___/\\\\\\\\\\\\\\\\\\\\\\\\\\__\\/\\\\\\/////////\\\\\\_____/\\\\\\////////__\\/\\\\\\////////\\\\\\_\n",
				"  __/\\\\\\/////////\\\\\\_\\/\\\\\\_______\\/\\\\\\___/\\\\\\/___________\\/\\\\\\______\\//\\\\\\\n",
				"   _\\/\\\\\\_______\\/\\\\\\_\\/\\\\\\\\\\\\\\\\\\\\\\\\\\\\___/\\\\\\_____________\\/\\\\\\_______\\/\\\\\\\n",
				"    _\\/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\_\\/\\\\\\/////////\\\\\\_\\/\\\\\\_____________\\/\\\\\\_______\\/\\\\\\\n",
				"     _\\/\\\\\\/////////\\\\\\_\\/\\\\\\_______\\/\\\\\\_\\//\\\\\\____________\\/\\\\\\_______\\/\\\\\\\n",
				"      _\\/\\\\\\_______\\/\\\\\\_\\/\\\\\\_______\\/\\\\\\__\\///\\\\\\__________\\/\\\\\\_______/\\\\\\_\n",
				"       _\\/\\\\\\_______\\/\\\\\\_\\/\\\\\\\\\\\\\\\\\\\\\\\\\\/_____\\////\\\\\\\\\\\\\\\\\\_\\/\\\\\\\\\\\\\\\\\\\\\\\\/__\n",
				"        _\\///________\\///__\\/////////////__________\\/////////__\\////////////____\n",
				"\n",
				"__/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\__/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\_____/\\\\\\\\\\\\\\\\\\\\\\\\__/\\\\\\________/\\\\\\\n",
				" _\\/\\\\\\///////////__\\/\\\\\\///////////____/\\\\\\//////////__\\/\\\\\\_______\\/\\\\\\\n",
				"  _\\/\\\\\\_____________\\/\\\\\\______________/\\\\\\_____________\\/\\\\\\_______\\/\\\\\\\n",
				"   _\\/\\\\\\\\\\\\\\\\\\\\\\_____\\/\\\\\\\\\\\\\\\\\\\\\\_____\\/\\\\\\____/\\\\\\\\\\\\\\_\\/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\n",
				"    _\\/\\\\\\///////______\\/\\\\\\///////______\\/\\\\\\___\\/////\\\\\\_\\/\\\\\\/////////\\\\\\\n",
				"     _\\/\\\\\\_____________\\/\\\\\\_____________\\/\\\\\\_______\\/\\\\\\_\\/\\\\\\_______\\/\\\\\\\n",
				"      _\\/\\\\\\_____________\\/\\\\\\_____________\\/\\\\\\_______\\/\\\\\\_\\/\\\\\\_______\\/\\\\\\\n",
				"       _\\/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\_\\/\\\\\\_____________\\//\\\\\\\\\\\\\\\\\\\\\\\\/__\\/\\\\\\_______\\/\\\\\\\n",
				"        _\\///////////////__\\///_______________\\////////////____\\///________\\///_\n",
				"\n",
				"__/\\\\\\\\\\\\\\\\\\\\\\______/\\\\\\\\\\\\\\\\\\\\\\__/\\\\\\________/\\\\\\__/\\\\\\____________\n",
				" _\\/////\\\\\\///______\\/////\\\\\\///__\\/\\\\\\_____/\\\\\\//__\\/\\\\\\____________\n",
				"  _____\\/\\\\\\_____________\\/\\\\\\_____\\/\\\\\\__/\\\\\\//_____\\/\\\\\\____________\n",
				"   _____\\/\\\\\\_____________\\/\\\\\\_____\\/\\\\\\\\\\\\//\\\\\\_____\\/\\\\\\____________\n",
				"    _____\\/\\\\\\_____________\\/\\\\\\_____\\/\\\\\\//_\\//\\\\\\____\\/\\\\\\____________\n",
				"     _____\\/\\\\\\_____________\\/\\\\\\_____\\/\\\\\\____\\//\\\\\\___\\/\\\\\\____________\n",
				"      _____\\/\\\\\\______/\\\\\\___\\/\\\\\\_____\\/\\\\\\_____\\//\\\\\\__\\/\\\\\\____________\n",
				"       __/\\\\\\\\\\\\\\\\\\\\\\_\\//\\\\\\\\\\\\\\\\\\______\\/\\\\\\______\\//\\\\\\_\\/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\n",
				"        _\\///////////___\\/////////_______\\///________\\///__\\///////////////_\n",
				"\n",
				"__/\\\\\\\\____________/\\\\\\\\__/\\\\\\\\\\_____/\\\\\\_______/\\\\\\\\\\_____\n",
				" _\\/\\\\\\\\\\\\________/\\\\\\\\\\\\_\\/\\\\\\\\\\\\___\\/\\\\\\_____/\\\\\\///\\\\\\___\n",
				"  _\\/\\\\\\//\\\\\\____/\\\\\\//\\\\\\_\\/\\\\\\/\\\\\\__\\/\\\\\\___/\\\\\\/__\\///\\\\\\_\n",
				"   _\\/\\\\\\\\///\\\\\\/\\\\\\/_\\/\\\\\\_\\/\\\\\\//\\\\\\_\\/\\\\\\__/\\\\\\______\\//\\\\\\\n",
				"    _\\/\\\\\\__\\///\\\\\\/___\\/\\\\\\_\\/\\\\\\\\//\\\\\\\\/\\\\\\_\\/\\\\\\_______\\/\\\\\\\n",
				"     _\\/\\\\\\____\\///_____\\/\\\\\\_\\/\\\\\\_\\//\\\\\\/\\\\\\_\\//\\\\\\______/\\\\\\_\n",
				"      _\\/\\\\\\_____________\\/\\\\\\_\\/\\\\\\__\\//\\\\\\\\\\\\__\\///\\\\\\__/\\\\\\___\n",
				"       _\\/\\\\\\_____________\\/\\\\\\_\\/\\\\\\___\\//\\\\\\\\\\____\\///\\\\\\\\\\/____\n",
				"        _\\///______________\\///__\\///_____\\/////_______\\/////______\n",
				"\n",
				"__/\\\\\\\\\\\\\\\\\\\\\\\\\\__________/\\\\\\__________/\\\\\\\\\\\\\\\\\\____\n",
				" _\\/\\\\\\/////////\\\\\\_____/\\\\\\\\/\\\\\\\\_____/\\\\\\///////\\\\\\__\n",
				"  _\\/\\\\\\_______\\/\\\\\\___/\\\\\\//\\////\\\\\\__\\/\\\\\\_____\\/\\\\\\__\n",
				"   _\\/\\\\\\\\\\\\\\\\\\\\\\\\\\/___/\\\\\\______\\//\\\\\\_\\/\\\\\\\\\\\\\\\\\\\\\\/___\n",
				"    _\\/\\\\\\/////////____\\//\\\\\\______/\\\\\\__\\/\\\\\\//////\\\\\\___\n",
				"     _\\/\\\\\\______________\\///\\\\\\\\/\\\\\\\\/___\\/\\\\\\____\\//\\\\\\__\n",
				"      _\\/\\\\\\________________\\////\\\\\\//_____\\/\\\\\\_____\\//\\\\\\_\n",
				"       _\\/\\\\\\___________________\\///\\\\\\\\\\\\__\\/\\\\\\______\\//\\\\\\\n",
				"        _\\///______________________\\//////___\\///________\\///_\n",
				"\n",
				"_____/\\\\\\\\\\\\\\\\\\\\\\____/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\__/\\\\\\________/\\\\\\\n",
				" ___/\\\\\\/////////\\\\\\_\\///////\\\\\\/////__\\/\\\\\\_______\\/\\\\\\\n",
				"  __\\//\\\\\\______\\///________\\/\\\\\\_______\\/\\\\\\_______\\/\\\\\\\n",
				"   ___\\////\\\\\\_______________\\/\\\\\\_______\\/\\\\\\_______\\/\\\\\\\n",
				"    ______\\////\\\\\\____________\\/\\\\\\_______\\/\\\\\\_______\\/\\\\\\\n",
				"     _________\\////\\\\\\_________\\/\\\\\\_______\\/\\\\\\_______\\/\\\\\\\n",
				"      __/\\\\\\______\\//\\\\\\________\\/\\\\\\_______\\//\\\\\\______/\\\\\\_\n",
				"       _\\///\\\\\\\\\\\\\\\\\\\\\\/_________\\/\\\\\\________\\///\\\\\\\\\\\\\\\\\\/__\n",
				"        ___\\///////////___________\\///___________\\/////////____\n",
				"\n",
				"__/\\\\\\________/\\\\\\__/\\\\\\______________/\\\\\\__/\\\\\\_______/\\\\\\\n",
				" _\\/\\\\\\_______\\/\\\\\\_\\/\\\\\\_____________\\/\\\\\\_\\///\\\\\\___/\\\\\\/_\n",
				"  _\\//\\\\\\______/\\\\\\__\\/\\\\\\_____________\\/\\\\\\___\\///\\\\\\\\\\\\/___\n",
				"   __\\//\\\\\\____/\\\\\\___\\//\\\\\\____/\\\\\\____/\\\\\\______\\//\\\\\\\\_____\n",
				"    ___\\//\\\\\\__/\\\\\\_____\\//\\\\\\__/\\\\\\\\\\__/\\\\\\________\\/\\\\\\\\_____\n",
				"     ____\\//\\\\\\/\\\\\\_______\\//\\\\\\/\\\\\\/\\\\\\/\\\\\\_________/\\\\\\\\\\\\____\n",
				"      _____\\//\\\\\\\\\\_________\\//\\\\\\\\\\\\//\\\\\\\\\\________/\\\\\\////\\\\\\__\n",
				"       ______\\//\\\\\\___________\\//\\\\\\__\\//\\\\\\_______/\\\\\\/___\\///\\\\\\\n",
				"        _______\\///_____________\\///____\\///_______\\///_______\\///_\n",
				"\n",
				"__/\\\\\\________/\\\\\\__/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\_____/\\\\\\\\\\\\\\_________/\\\\\\\n",
				" _\\///\\\\\\____/\\\\\\/__\\////////////\\\\\\____/\\\\\\/////\\\\\\___/\\\\\\\\\\\\\\\n",
				"  ___\\///\\\\\\/\\\\\\/______________/\\\\\\/____/\\\\\\____\\//\\\\\\_\\/////\\\\\\\n",
				"   _____\\///\\\\\\/______________/\\\\\\/_____\\/\\\\\\_____\\/\\\\\\_____\\/\\\\\\\n",
				"    _______\\/\\\\\\_____________/\\\\\\/_______\\/\\\\\\_____\\/\\\\\\_____\\/\\\\\\\n",
				"     _______\\/\\\\\\___________/\\\\\\/_________\\/\\\\\\_____\\/\\\\\\_____\\/\\\\\\\n",
				"      _______\\/\\\\\\_________/\\\\\\/___________\\//\\\\\\____/\\\\\\______\\/\\\\\\\n",
				"       _______\\/\\\\\\________/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\__\\///\\\\\\\\\\\\\\/_______\\/\\\\\\\n",
				"        _______\\///________\\///////////////_____\\///////_________\\/// \n",
				"\n",
				"____/\\\\\\\\\\\\\\\\\\_________/\\\\\\\\\\\\\\\\\\\\_____________/\\\\\\___\n",
				" __/\\\\\\///////\\\\\\_____/\\\\\\///////\\\\\\__________/\\\\\\\\\\___\n",
				"  _\\///______\\//\\\\\\___\\///______/\\\\\\_________/\\\\\\/\\\\\\___\n",
				"   ___________/\\\\\\/___________/\\\\\\//________/\\\\\\/\\/\\\\\\___\n",
				"    ________/\\\\\\//____________\\////\\\\\\_____/\\\\\\/__\\/\\\\\\___\n",
				"     _____/\\\\\\//__________________\\//\\\\\\__/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\n",
				"      ___/\\\\\\/____________/\\\\\\______/\\\\\\__\\///////////\\\\\\//_\n",
				"       __/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\_\\///\\\\\\\\\\\\\\\\\\/_____________\\/\\\\\\___\n",
				"        _\\///////////////____\\/////////_______________\\///____\n",
				"\n",
				"___/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\ ____________/\\\\\\\\\\__/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\n",
				" __\\/\\\\\\///////////_ ________/\\\\\\\\////__\\/////////////\\\\\\\n",
				"  __\\/\\\\\\____________ _____/\\\\\\///__________________/\\\\\\/_\n",
				"   __\\/\\\\\\\\\\\\\\\\\\\\\\\\_______/\\\\\\\\\\\\\\\\\\\\\\_____________/\\\\\\/___\n",
				"    __\\////////////\\\\\\____/\\\\\\\\///////\\\\\\_________/\\\\\\/_____\n",
				"     _____________\\//\\\\\\__\\/\\\\\\______\\//\\\\\\______/\\\\\\/_______\n",
				"      __/\\\\\\________\\/\\\\\\__\\//\\\\\\______/\\\\\\_____/\\\\\\/_________\n",
				"       _\\//\\\\\\\\\\\\\\\\\\\\\\\\\\/____\\///\\\\\\\\\\\\\\\\\\/____/\\\\\\/___________\n",
				"        __\\/////////////________\\/////////_____\\///_____________\n",
				"\n",
				"_____/\\\\\\\\\\\\\\\\\\_________/\\\\\\\\\\\\\\\\\\________/\\\\\\_______/\\\\\\\\\\\\\\___ ______\n",
				" ___/\\\\\\///////\\\\\\_____/\\\\\\///////\\\\\\____/\\\\\\\\\\\\\\__/\\\\\\//////\\\\\\_ ______\n",
				"  __\\/\\\\\\_____\\/\\\\\\____/\\\\\\______\\//\\\\\\__/\\\\\\\\\\\\\\\\\\_\\///_____\\//\\\\\\______\n",
				"   __\\///\\\\\\\\\\\\\\\\\\/____\\//\\\\\\_____/\\\\\\\\\\_\\//\\\\\\\\\\\\\\___________/\\\\\\_ ______\n",
				"    ___/\\\\\\///////\\\\\\____\\///\\\\\\\\\\\\\\\\/\\\\\\__\\//\\\\\\\\\\_________/\\\\\\\\/__ ______\n",
				"     __/\\\\\\______\\//\\\\\\_____\\////////\\/\\\\\\___\\//\\\\\\_________/\\\\\\/____ ______\n",
				"      _\\//\\\\\\______/\\\\\\____/\\\\________/\\\\\\_____\\///_________\\///______ ______\n",
				"       __\\///\\\\\\\\\\\\\\\\\\/____\\//\\\\\\\\\\\\\\\\\\\\\\/_______/\\\\\\_________/\\\\\\_____ __/\\\\\\\n",
				"        ____\\/////////_______\\///////////________\\///_________\\///______ _\\///_\n",
				"\n",
				"___________________________________________________________\n",
				" ___________________________________________________________\n",
				"  ______/\\\\\\_________________________________________________\n",
				"   _____\\/\\\\\\______/\\\\\\\\\\\\\\\\\\\\\\____________________/\\\\\\\\\\\\\\\\\\\\\n",
				"    __/\\\\\\\\\\\\\\\\\\\\\\_\\///////////____________________\\//////////_\n",
				"     _\\/////\\\\\\///___________________________________/\\\\\\\\\\\\\\\\\\\\\n",
				"      _____\\/\\\\\\_____________________________________\\//////////_\n",
				"       _____\\///_____________________/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\_____________\n",
				"        _____________________________\\///////////////______________\n",
				"\n",
				"____/\\\\\\\\\\\\\\\\\\\\_______/\\\\\\__/\\\\\\__________/\\\\\\__________/\\\\\\________/\\\\\\\n",
				" __/\\\\\\////////\\\\\\____\\/\\\\\\_\\/\\\\\\______/\\\\\\\\\\\\\\\\\\\\\\_____/\\\\/\\\\_____/\\\\\\/_\n",
				"  _\\/\\\\\\__/\\\\\\\\\\\\\\\\\\__/\\\\\\\\\\\\\\\\\\\\\\\\\\__/\\\\\\///\\\\\\////\\\\__\\//\\\\\\____/\\\\\\/___\n",
				"   _\\/\\\\\\_\\/\\\\\\///\\\\\\_\\///\\\\\\///\\\\\\/__\\////\\\\\\\\\\\\__\\//____\\///___/\\\\\\/_____\n",
				"    _\\/\\\\\\_\\/\\\\\\\\\\\\\\\\____\\/\\\\\\_\\/\\\\\\______\\////\\\\\\\\\\\\___________/\\\\\\/_______\n",
				"     _\\/\\\\\\_\\////////____/\\\\\\\\\\\\\\\\\\\\\\\\\\__/\\\\__\\/\\\\\\///\\\\\\______/\\\\\\/___/\\\\\\__\n",
				"      _\\//\\\\\\____________\\///\\\\\\///\\\\\\/__\\///\\\\\\\\\\\\\\\\\\\\\\/_____/\\\\\\/____/\\\\/\\\\_\n",
				"       __\\///\\\\\\\\\\\\\\\\\\\\\\\\___\\/\\\\\\_\\/\\\\\\_____\\/////\\\\\\///__ __/\\\\\\/_____\\//\\\\\\__\n",
				"        ____\\////////////____\\///__\\///__________\\///________\\///________\\///__ \n",
				"\n",
				"___________/\\\\\\____________/\\\\\\__/\\\\\\____________________/\\\\\\______\n",
				" ________/\\\\\\//\\\\\\________/\\\\\\/__\\///\\\\\\________________/\\\\\\/_______\n",
				"  _______/\\\\\\_/\\\\\\________/\\\\\\______\\//\\\\\\_____________/\\\\\\/_________\n",
				"   ______\\//\\\\\\\\//________/\\\\\\________\\//\\\\\\__________/\\\\\\/___________\n",
				"    _____/\\\\\\///\\\\\\_______\\/\\\\\\_________\\/\\\\\\________/\\\\\\/_________/\\\\\\\n",
				"     ___/\\\\\\/__\\///\\\\\\/\\\\\\_\\//\\\\\\________/\\\\\\_______/\\\\\\/__________\\///_\n",
				"      __/\\\\\\______\\//\\\\\\//___\\//\\\\\\______/\\\\\\______/\\\\\\/_________________\n",
				"       _\\//\\\\\\\\\\\\\\\\\\\\\\//\\\\\\____\\///\\\\\\__/\\\\\\/_____/\\\\\\/_______________/\\\\\\\n",
				"        __\\///////////_\\///_______\\///__\\///______\\///________________\\///_\n",
				"\n",
				"________________/\\\\\\___/\\\\\\/\\\\\\\n",
				" _______________\\/\\\\\\__\\/\\\\\\/\\\\\\\n",
				"  _______________\\///___\\///\\///_\n",
				"   __/\\\\\\_________________________\n",
				"    _\\///__________________________\n",
				"     __/\\\\\\\\__/\\\\\\\\_________________\n",
				"      _\\///\\\\_\\///\\\\_________________\n",
				"       __/\\\\/___/\\\\/__________________\n",
				"        _\\//____\\//____________________\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"3d\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "3d".to_string(), "-c".to_string(), "red,blue".to_string()],
			fixture: concat!("\n\n",
				"\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\/////////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\////////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\/////////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m__________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\/////////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\/////////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m___________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\x1B[39m\x1B[34m_________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\////\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/////////////\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\/////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\////////////\x1B[39m\x1B[34m____\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\///////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\///////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\//////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\///////\x1B[39m\x1B[34m_____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\///////\x1B[39m\x1B[34m_____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\/////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\/////////\\\\\\\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[31m\\///////////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\////////////\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[31m\\/////\\\\\\///\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/////\\\\\\///\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\//\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\//\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\//\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\//\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[31m\\///////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/////////\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///////////////\x1B[39m\x1B[34m_\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[31m/\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\x1B[39m\x1B[34m________\x1B[39m\x1B[31m/\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\///\\\\\\\x1B[39m\x1B[34m___\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\//\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///\\\\\\\x1B[39m\x1B[34m_\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\///\\\\\\/\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///\\\\\\/\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\//\\\\\\\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____________\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\//\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____________\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\//\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\///\\\\\\\\\\/\x1B[39m\x1B[34m____\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m______________\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\/////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\/////\x1B[39m\x1B[34m______\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m____\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\/////////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\\/\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\///////\\\\\\\x1B[39m\x1B[34m__\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\//\\////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m___\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\/////////\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\//////\\\\\\\x1B[39m\x1B[34m___\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\\/\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m__\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\////\\\\\\//\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\///\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\//////\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\/////////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///////\\\\\\/////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\////\\\\\\\x1B[39m\x1B[34m________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\////\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\////\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///////////\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\/////////\x1B[39m\x1B[34m____\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m______________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____________\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____________\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///\\\\\\\\\\\\/\x1B[39m\x1B[34m___\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\//\\\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\/\\\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\//\\\\\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\//\\\\\\/\\\\\\/\\\\\\/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\\\\\\\\x1B[39m\x1B[34m____\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\//\\\\\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\//\\\\\\\\\\\\//\\\\\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\////\\\\\\\x1B[39m\x1B[34m__\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\///\\\\\\\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\////////////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\/////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///\\\\\\/\\\\\\/\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__________\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/////\\\\\\\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\///\\\\\\/\x1B[39m\x1B[34m_____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m________\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m__________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///////////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\///////\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/// \n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\///////\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\///////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________\x1B[39m\x1B[31m/\\\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[34m__________\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m________\x1B[39m\x1B[31m/\\\\\\//\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\/\\/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\//\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m\\////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\//\x1B[39m\x1B[34m_______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__________\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m__________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///////////\\\\\\//\x1B[39m\x1B[34m_\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__________\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[31m\\///////////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\/////////\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__________\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m____\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m \x1B[34m_\x1B[39m\x1B[34m___________\x1B[39m\x1B[31m/\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\///////////\x1B[39m\x1B[34m_\x1B[39m \x1B[34m_\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\\\////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/////////////\\\\\\\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____________\x1B[39m \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\///\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___________\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m___\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\////////////\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\///////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_____\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[34m____________\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_______\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_________\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\\\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m___________\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/////////////\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\/////////\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_____________\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m \x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\///////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\///////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\//////\\\\\\\x1B[39m\x1B[34m_\x1B[39m \x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m \x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\///////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///\\\\\\\\\\\\\\\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m \x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\////////\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m____\x1B[39m \x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\x1B[39m\x1B[34m________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m______\x1B[39m \x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_____\x1B[39m \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\/////////\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///////////\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m______\x1B[39m \x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____________\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____________\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____________\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//////////\x1B[39m\x1B[34m_\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[31m\\/////\\\\\\///\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//////////\x1B[39m\x1B[34m_\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____________\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///////////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____________\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\////////\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\/\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\///\\\\\\////\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m___\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\///\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\///\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\////\\\\\\\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\//\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_____\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\////\\\\\\\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_______\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\////////\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\/\\\\\\///\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m___________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\///\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\\\\\\\\\\\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\/\\\\\x1B[39m\x1B[34m_\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\\\\\\\\\\\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\/////\\\\\\///\x1B[39m\x1B[34m__\x1B[39m \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m__\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\////////////\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m________\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m__\x1B[39m \n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m__________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_______________\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\//\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____________\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___________\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m\\//\\\\\\\\//\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\///\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m____\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_______\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///\\\\\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m______\x1B[39m\x1B[31m\\//\\\\\\//\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m___\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m___________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_____\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[31m\\//\\\\\\\\\\\\\\\\\\\\\\//\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m__\x1B[39m\x1B[31m\\///\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\/\x1B[39m\x1B[34m_____________\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///////////\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m____\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_____\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_______________\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m_\x1B[39m\n",
				"\n",
				"\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\/\\\\\\\x1B[39m\n",
				" \x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\/\\\\\\/\\\\\\\x1B[39m\n",
				"  \x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\///\x1B[39m\x1B[34m_\x1B[39m\n",
				"   \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________\x1B[39m\n",
				"    \x1B[34m_\x1B[39m\x1B[31m\\///\x1B[39m\x1B[34m__\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________\x1B[39m\n",
				"     \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________\x1B[39m\n",
				"      \x1B[34m_\x1B[39m\x1B[31m\\///\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\///\\\\\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________\x1B[39m\n",
				"       \x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[31m/\\\\/\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________\x1B[39m\n",
				"        \x1B[34m_\x1B[39m\x1B[31m\\//\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[31m\\//\x1B[39m\x1B[34m___\x1B[39m\x1B[34m_\x1B[39m\x1B[34m______\x1B[39m\x1B[34m_\x1B[39m\x1B[34m_________\x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"huge\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "huge".to_string()],
			fixture: concat!("\n\n",
				"  ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄    ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄    ▄▄▄▄▄▄▄▄▄▄▄ \n",
				" ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░▌  ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░▌  ▐░░░░░░░░░░░▌\n",
				" ▐░█▀▀▀▀▀▀▀█░▌ ▐░█▀▀▀▀▀▀▀█░▌ ▐░█▀▀▀▀▀▀▀▀▀  ▐░█▀▀▀▀▀▀▀█░▌ ▐░█▀▀▀▀▀▀▀▀▀ \n",
				" ▐░▌       ▐░▌ ▐░▌       ▐░▌ ▐░▌           ▐░▌       ▐░▌ ▐░▌          \n",
				" ▐░█▄▄▄▄▄▄▄█░▌ ▐░█▄▄▄▄▄▄▄█░▌ ▐░▌           ▐░▌       ▐░▌ ▐░█▄▄▄▄▄▄▄▄▄ \n",
				" ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░▌  ▐░▌           ▐░▌       ▐░▌ ▐░░░░░░░░░░░▌\n",
				" ▐░█▀▀▀▀▀▀▀█░▌ ▐░█▀▀▀▀▀▀▀█░▌ ▐░▌           ▐░▌       ▐░▌ ▐░█▀▀▀▀▀▀▀▀▀ \n",
				" ▐░▌       ▐░▌ ▐░▌       ▐░▌ ▐░▌           ▐░▌       ▐░▌ ▐░▌          \n",
				" ▐░▌       ▐░▌ ▐░█▄▄▄▄▄▄▄█░▌ ▐░█▄▄▄▄▄▄▄▄▄  ▐░█▄▄▄▄▄▄▄█░▌ ▐░█▄▄▄▄▄▄▄▄▄ \n",
				" ▐░▌       ▐░▌ ▐░░░░░░░░░░▌  ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░▌  ▐░░░░░░░░░░░▌\n",
				"  ▀         ▀   ▀▀▀▀▀▀▀▀▀▀    ▀▀▀▀▀▀▀▀▀▀▀   ▀▀▀▀▀▀▀▀▀▀    ▀▀▀▀▀▀▀▀▀▀▀ \n",
				"\n",
				"  ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄         ▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄    ▄ \n",
				" ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░▌       ▐░▌ ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░▌  ▐░▌\n",
				" ▐░█▀▀▀▀▀▀▀▀▀  ▐░█▀▀▀▀▀▀▀▀▀  ▐░▌       ▐░▌  ▀▀▀▀█░█▀▀▀▀   ▀▀▀▀▀█░█▀▀▀  ▐░▌ ▐░▌ \n",
				" ▐░▌           ▐░▌           ▐░▌       ▐░▌      ▐░▌            ▐░▌     ▐░▌▐░▌  \n",
				" ▐░█▄▄▄▄▄▄▄▄▄  ▐░▌ ▄▄▄▄▄▄▄▄  ▐░█▄▄▄▄▄▄▄█░▌      ▐░▌            ▐░▌     ▐░▌░▌   \n",
				" ▐░░░░░░░░░░░▌ ▐░▌▐░░░░░░░░▌ ▐░░░░░░░░░░░▌      ▐░▌            ▐░▌     ▐░░▌    \n",
				" ▐░█▀▀▀▀▀▀▀▀▀  ▐░▌ ▀▀▀▀▀▀█░▌ ▐░█▀▀▀▀▀▀▀█░▌      ▐░▌            ▐░▌     ▐░▌░▌   \n",
				" ▐░▌           ▐░▌       ▐░▌ ▐░▌       ▐░▌      ▐░▌            ▐░▌     ▐░▌▐░▌  \n",
				" ▐░▌           ▐░█▄▄▄▄▄▄▄█░▌ ▐░▌       ▐░▌  ▄▄▄▄█░█▄▄▄▄   ▄▄▄▄▄█░▌     ▐░▌ ▐░▌ \n",
				" ▐░▌           ▐░░░░░░░░░░░▌ ▐░▌       ▐░▌ ▐░░░░░░░░░░░▌ ▐░░░░░░░▌     ▐░▌  ▐░▌\n",
				"  ▀             ▀▀▀▀▀▀▀▀▀▀▀   ▀         ▀   ▀▀▀▀▀▀▀▀▀▀▀   ▀▀▀▀▀▀▀       ▀    ▀ \n",
				"\n",
				"  ▄             ▄▄       ▄▄   ▄▄        ▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄▄ \n",
				" ▐░▌           ▐░░▌     ▐░░▌ ▐░░▌      ▐░▌ ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌\n",
				" ▐░▌           ▐░▌░▌   ▐░▐░▌ ▐░▌░▌     ▐░▌ ▐░█▀▀▀▀▀▀▀█░▌ ▐░█▀▀▀▀▀▀▀█░▌\n",
				" ▐░▌           ▐░▌▐░▌ ▐░▌▐░▌ ▐░▌▐░▌    ▐░▌ ▐░▌       ▐░▌ ▐░▌       ▐░▌\n",
				" ▐░▌           ▐░▌ ▐░▐░▌ ▐░▌ ▐░▌ ▐░▌   ▐░▌ ▐░▌       ▐░▌ ▐░█▄▄▄▄▄▄▄█░▌\n",
				" ▐░▌           ▐░▌  ▐░▌  ▐░▌ ▐░▌  ▐░▌  ▐░▌ ▐░▌       ▐░▌ ▐░░░░░░░░░░░▌\n",
				" ▐░▌           ▐░▌   ▀   ▐░▌ ▐░▌   ▐░▌ ▐░▌ ▐░▌       ▐░▌ ▐░█▀▀▀▀▀▀▀▀▀ \n",
				" ▐░▌           ▐░▌       ▐░▌ ▐░▌    ▐░▌▐░▌ ▐░▌       ▐░▌ ▐░▌          \n",
				" ▐░█▄▄▄▄▄▄▄▄▄  ▐░▌       ▐░▌ ▐░▌     ▐░▐░▌ ▐░█▄▄▄▄▄▄▄█░▌ ▐░▌          \n",
				" ▐░░░░░░░░░░░▌ ▐░▌       ▐░▌ ▐░▌      ▐░░▌ ▐░░░░░░░░░░░▌ ▐░▌          \n",
				"  ▀▀▀▀▀▀▀▀▀▀▀   ▀         ▀   ▀        ▀▀   ▀▀▀▀▀▀▀▀▀▀▀   ▀           \n",
				"\n",
				"  ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄         ▄ \n",
				" ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░▌       ▐░▌\n",
				" ▐░█▀▀▀▀▀▀▀█░▌ ▐░█▀▀▀▀▀▀▀█░▌ ▐░█▀▀▀▀▀▀▀▀▀   ▀▀▀▀█░█▀▀▀▀  ▐░▌       ▐░▌\n",
				" ▐░▌       ▐░▌ ▐░▌       ▐░▌ ▐░▌                ▐░▌      ▐░▌       ▐░▌\n",
				" ▐░▌       ▐░▌ ▐░█▄▄▄▄▄▄▄█░▌ ▐░█▄▄▄▄▄▄▄▄▄       ▐░▌      ▐░▌       ▐░▌\n",
				" ▐░▌       ▐░▌ ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌      ▐░▌      ▐░▌       ▐░▌\n",
				" ▐░█▄▄▄▄▄▄▄█░▌ ▐░█▀▀▀▀█░█▀▀   ▀▀▀▀▀▀▀▀▀█░▌      ▐░▌      ▐░▌       ▐░▌\n",
				" ▐░░░░░░░░░░░▌ ▐░▌     ▐░▌             ▐░▌      ▐░▌      ▐░▌       ▐░▌\n",
				"  ▀▀▀▀▀▀█░█▀▀  ▐░▌      ▐░▌   ▄▄▄▄▄▄▄▄▄█░▌      ▐░▌      ▐░█▄▄▄▄▄▄▄█░▌\n",
				"         ▐░▌   ▐░▌       ▐░▌ ▐░░░░░░░░░░░▌      ▐░▌      ▐░░░░░░░░░░░▌\n",
				"          ▀     ▀         ▀   ▀▀▀▀▀▀▀▀▀▀▀        ▀        ▀▀▀▀▀▀▀▀▀▀▀ \n",
				"\n",
				"  ▄               ▄   ▄         ▄   ▄       ▄   ▄         ▄   ▄▄▄▄▄▄▄ \n",
				" ▐░▌             ▐░▌ ▐░▌       ▐░▌ ▐░▌     ▐░▌ ▐░▌       ▐░▌ ▐░░░░░░░▌\n",
				"  ▐░▌           ▐░▌  ▐░▌       ▐░▌  ▐░▌   ▐░▌  ▐░▌       ▐░▌  ▀▀▀▀▀█░▌\n",
				"   ▐░▌         ▐░▌   ▐░▌       ▐░▌   ▐░▌ ▐░▌   ▐░▌       ▐░▌      ▐░▌ \n",
				"    ▐░▌       ▐░▌    ▐░▌   ▄   ▐░▌    ▐░▐░▌    ▐░█▄▄▄▄▄▄▄█░▌     ▐░▌  \n",
				"     ▐░▌     ▐░▌     ▐░▌  ▐░▌  ▐░▌     ▐░▌     ▐░░░░░░░░░░░▌    ▐░▌   \n",
				"      ▐░▌   ▐░▌      ▐░▌ ▐░▌░▌ ▐░▌    ▐░▌░▌     ▀▀▀▀█░█▀▀▀▀    ▐░▌    \n",
				"       ▐░▌ ▐░▌       ▐░▌▐░▌ ▐░▌▐░▌   ▐░▌ ▐░▌        ▐░▌       ▐░▌     \n",
				"        ▐░▐░▌        ▐░▌░▌   ▐░▐░▌  ▐░▌   ▐░▌       ▐░▌      ▐░█▄▄▄▄▄ \n",
				"         ▐░▌         ▐░░▌     ▐░░▌ ▐░▌     ▐░▌      ▐░▌      ▐░░░░░░░▌\n",
				"          ▀           ▀▀       ▀▀   ▀       ▀        ▀        ▀▀▀▀▀▀▀ \n",
				"\n",
				"   ▄▄▄▄▄▄▄▄▄       ▄▄▄▄       ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄         ▄ \n",
				"  ▐░░░░░░░░░▌    ▄█░░░░▌     ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░▌       ▐░▌\n",
				" ▐░█░█▀▀▀▀▀█░▌  ▐░░▌▐░░▌      ▀▀▀▀▀▀▀▀▀█░▌  ▀▀▀▀▀▀▀▀▀█░▌ ▐░▌       ▐░▌\n",
				" ▐░▌▐░▌    ▐░▌   ▀▀ ▐░░▌               ▐░▌           ▐░▌ ▐░▌       ▐░▌\n",
				" ▐░▌ ▐░▌   ▐░▌      ▐░░▌               ▐░▌  ▄▄▄▄▄▄▄▄▄█░▌ ▐░█▄▄▄▄▄▄▄█░▌\n",
				" ▐░▌  ▐░▌  ▐░▌      ▐░░▌      ▄▄▄▄▄▄▄▄▄█░▌ ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌\n",
				" ▐░▌   ▐░▌ ▐░▌      ▐░░▌     ▐░░░░░░░░░░░▌  ▀▀▀▀▀▀▀▀▀█░▌  ▀▀▀▀▀▀▀▀▀█░▌\n",
				" ▐░▌    ▐░▌▐░▌      ▐░░▌     ▐░█▀▀▀▀▀▀▀▀▀            ▐░▌           ▐░▌\n",
				" ▐░█▄▄▄▄▄█░█░▌  ▄▄▄▄█░░█▄▄▄  ▐░█▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄█░▌           ▐░▌\n",
				"  ▐░░░░░░░░░▌  ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌           ▐░▌\n",
				"   ▀▀▀▀▀▀▀▀▀    ▀▀▀▀▀▀▀▀▀▀▀   ▀▀▀▀▀▀▀▀▀▀▀   ▀▀▀▀▀▀▀▀▀▀▀             ▀ \n",
				"\n",
				"  ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄▄▄▄▄▄▄▄▄▄▄   ▄▄ \n",
				" ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░░▌\n",
				" ▐░█▀▀▀▀▀▀▀▀▀  ▐░█▀▀▀▀▀▀▀▀▀   ▀▀▀▀▀▀▀▀▀█░▌ ▐░█▀▀▀▀▀▀▀█░▌ ▐░█▀▀▀▀▀▀▀█░▌ ▐░░▌\n",
				" ▐░█▄▄▄▄▄▄▄▄▄  ▐░▌                    ▐░▌  ▐░▌       ▐░▌ ▐░▌       ▐░▌ ▐░░▌\n",
				" ▐░░░░░░░░░░░▌ ▐░█▄▄▄▄▄▄▄▄▄          ▐░▌   ▐░█▄▄▄▄▄▄▄█░▌ ▐░█▄▄▄▄▄▄▄█░▌ ▐░░▌\n",
				"  ▀▀▀▀▀▀▀▀▀█░▌ ▐░░░░░░░░░░░▌        ▐░▌     ▐░░░░░░░░░▌  ▐░░░░░░░░░░░▌ ▐░░▌\n",
				"           ▐░▌ ▐░█▀▀▀▀▀▀▀█░▌       ▐░▌     ▐░█▀▀▀▀▀▀▀█░▌  ▀▀▀▀▀▀▀▀▀█░▌ ▐░░▌\n",
				"           ▐░▌ ▐░▌       ▐░▌      ▐░▌      ▐░▌       ▐░▌           ▐░▌  ▀▀ \n",
				"  ▄▄▄▄▄▄▄▄▄█░▌ ▐░█▄▄▄▄▄▄▄█░▌     ▐░▌       ▐░█▄▄▄▄▄▄▄█░▌  ▄▄▄▄▄▄▄▄▄█░▌  ▄▄ \n",
				" ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌    ▐░▌        ▐░░░░░░░░░░░▌ ▐░░░░░░░░░░░▌ ▐░░▌\n",
				"  ▀▀▀▀▀▀▀▀▀▀▀   ▀▀▀▀▀▀▀▀▀▀▀      ▀          ▀▀▀▀▀▀▀▀▀▀▀   ▀▀▀▀▀▀▀▀▀▀▀   ▀▀ \n",
				"\n",
				"     ▄▄▄▄▄▄▄                                            ▄▄▄▄▄▄▄▄▄▄▄▄ \n",
				"   ▄█░░░░░░█▄                                          ▐░░░░░░░░░░░░▌\n",
				"  ▐░░▌▀▀▀▀█░░▌          ▄▄                      ▄▄▄▄▄  ▐░█▀▀▀▀▀▀▀▀█░▌\n",
				"   ▀▀  ▄▄▄█░░▌         ▐░░▌                    ▐░░░░░▌ ▐░▌        ▐░▌\n",
				"     ▄█░░░░░█        ▄▄█░░█▄▄   ▄▄▄▄▄           ▀▀▀▀▀  ▐░▌  ▄▄▄▄  ▐░▌\n",
				"    ▐░░▌▀▀▀▀        ▐░░░░░░░░▌ ▐░░░░░▌                 ▐░▌ █░░░░█ ▐░▌\n",
				"    ▐░░▌             ▀▀█░░█▀▀   ▀▀▀▀▀           ▄▄▄▄▄  ▐░▌▐░████░▄█░▌\n",
				"     ▀▀                ▐░░▌             ▄▄▄▄▄  ▐░░░░░▌ ▐░▌ █░░░░░░░░▌\n",
				"     ▄▄         ▄▄      ▀▀             ▐░░░░░▌  ▀▀▀▀▀  ▐░▌▄▄███████░▌\n",
				"    ▐░░▌       ▐░░▌                     ▀▀▀▀▀          ▐░░░░░░░░░░░░▌\n",
				"     ▀▀         ▀▀                                      ▀▀▀▀▀▀▀▀▀▀▀▀ \n",
				"\n",
				"    ▄         ▄          ▄                ▄   ▄▄▄▄▄▄▄        ▄▄▄▄▄   ▄▄▄▄▄  \n",
				"   ▐░▌       ▐░▌    ▄▄▄▄█░█▄▄▄▄    ▄     ▐░▌ ▐░░░░░░░▌      ▐░░░░░▌ ▐░░░░░▌ \n",
				"  ▄█░█▄▄▄▄▄▄▄█░█▄  ▐░░░░░░░░░░░▌  ▐░▌   ▐░▌  ▐░█▀▀▀█░▌     ▐░░█▀▀▀   ▀▀▀█░░▌\n",
				" ▐░░░░░░░░░░░░░░░▌ ▐░█▀▀█░█▀▀▀▀    ▀   ▐░▌   ▐░▌   ▐░▌     ▐░░▌         ▐░░▌\n",
				"  ▀█░█▀▀▀▀▀▀▀█░█▀  ▐░█▄▄█░█▄▄▄▄       ▐░▌    ▐░█▄▄▄█░▌     ▐░░▌         ▐░░▌\n",
				"   ▐░▌       ▐░▌   ▐░░░░░░░░░░░▌     ▐░▌      ▐░░░░░░▌     ▐░░▌         ▐░░▌\n",
				"  ▄█░█▄▄▄▄▄▄▄█░█▄   ▀▀▀▀█░█▀▀█░▌    ▐░▌      ▐░█▀▀▀▀█░▌ ▄  ▐░░▌         ▐░░▌\n",
				" ▐░░░░░░░░░░░░░░░▌  ▄▄▄▄█░█▄▄█░▌   ▐░▌   ▄   ▐░▌     ▐░█░▌ ▐░░▌         ▐░░▌\n",
				"  ▀█░█▀▀▀▀▀▀▀█░█▀  ▐░░░░░░░░░░░▌  ▐░▌   ▐░▌  ▐░█▄▄▄▄█░▌    ▐░░█▄▄▄   ▄▄▄█░░▌\n",
				"   ▐░▌       ▐░▌    ▀▀▀▀█░█▀▀▀▀  ▐░▌     ▀   ▐░░░░░░▌▐░▌    ▐░░░░░▌ ▐░░░░░▌ \n",
				"    ▀         ▀          ▀        ▀           ▀▀▀▀▀▀  ▀      ▀▀▀▀▀   ▀▀▀▀▀  \n",
				"\n",
				"          ▄                  ▄   ▄  ▄ \n",
				"         ▐░▌                ▐░▐ ▐░▐▐░▐\n",
				"        ▐░▌   ▄▄   ▄▄       ▐░▐ ▐░▐▐░▐\n",
				"       ▐░▌   ▐░░▌ ▐░░▌       ▀   ▀  ▀ \n",
				"      ▐░▌     ▀▀   ▀▀                 \n",
				"     ▐░▌                              \n",
				"    ▐░▌       ▄▄   ▄▄                 \n",
				"   ▐░▌       ▐░░▌ ▐░░▌  ▄▄            \n",
				"  ▐░▌         ▀▀   ▀▌  ▐░░▌           \n",
				" ▐░▌               ▀    ▀▌            \n",
				"  ▀                     ▀             \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"huge\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "huge".to_string(), "-c".to_string(), "red,blue".to_string()],
			fixture: concat!("\n\n",
				"  \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄\x1B[39m    \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄\x1B[39m    \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀▀\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m          \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄▄▄\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀▀\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m          \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄▄▄\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄▄▄\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"  \x1B[31m▀         ▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀▀▀▀\x1B[39m    \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀▀▀▀\x1B[39m    \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m \n",
				"\n",
				"  \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄         ▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄    ▄\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌  ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀\x1B[39m   \x1B[31m▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m            \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄▄▄\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▄▄▄▄▄▄▄▄\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m            \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m            \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m    \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m            \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m            \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌  ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"  \x1B[31m▀\x1B[39m             \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀         ▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀\x1B[39m       \x1B[31m▀    ▀\x1B[39m \n",
				"\n",
				"  \x1B[31m▄\x1B[39m             \x1B[31m▄▄       ▄▄\x1B[39m   \x1B[31m▄▄        ▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌     ▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌      ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌     ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌    ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌  ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌  ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌  ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌  ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▀   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀▀\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌    ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m          \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄▄▄\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌     ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m          \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌      ▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m          \n",
				"  \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀         ▀\x1B[39m   \x1B[31m▀        ▀▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀\x1B[39m           \n",
				"\n",
				"  \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄         ▄\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m                \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄▄▄\x1B[39m       \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌     ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m             \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"  \x1B[31m▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌      ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"         \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"          \x1B[31m▀\x1B[39m     \x1B[31m▀         ▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m        \x1B[31m▀\x1B[39m        \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m \n",
				"\n",
				"  \x1B[31m▄               ▄\x1B[39m   \x1B[31m▄         ▄\x1B[39m   \x1B[31m▄       ▄\x1B[39m   \x1B[31m▄         ▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌             ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌     ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌           ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌         ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \n",
				"    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▄   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \n",
				"     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌     ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌  ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌  ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \n",
				"      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀\x1B[39m    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m    \n",
				"       \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m       \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m        \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m       \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \n",
				"        \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m        \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m       \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄\x1B[39m \n",
				"         \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m         \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌     ▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌     ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"          \x1B[31m▀\x1B[39m           \x1B[31m▀▀       ▀▀\x1B[39m   \x1B[31m▀       ▀\x1B[39m        \x1B[31m▀\x1B[39m        \x1B[31m▀▀▀▀▀▀▀\x1B[39m \n",
				"\n",
				"   \x1B[31m▄▄▄▄▄▄▄▄▄\x1B[39m       \x1B[31m▄▄▄▄\x1B[39m       \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄         ▄\x1B[39m \n",
				"  \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▄█\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▀▀▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▀▀▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌    ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▀▀ ▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m               \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m               \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▄▄▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌  ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌  ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▄▄▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▀▀▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▀▀▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌    ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀▀\x1B[39m            \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▄▄▄▄█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█▄▄▄\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"  \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"   \x1B[31m▀▀▀▀▀▀▀▀▀\x1B[39m    \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m             \x1B[31m▀\x1B[39m \n",
				"\n",
				"  \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄▄▄▄▄\x1B[39m   \x1B[31m▄▄\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄▄▄\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m                    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄▄▄\x1B[39m          \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"  \x1B[31m▀▀▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m        \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m       \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▀▀▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m           \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▀▀\x1B[39m \n",
				"  \x1B[31m▄▄▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m       \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▄▄▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▄▄\x1B[39m \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m        \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"  \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m      \x1B[31m▀\x1B[39m          \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀▀▀▀▀▀▀▀▀▀▀\x1B[39m   \x1B[31m▀▀\x1B[39m \n",
				"\n",
				"     \x1B[31m▄▄▄▄▄▄▄\x1B[39m                                            \x1B[31m▄▄▄▄▄▄▄▄▄▄▄▄\x1B[39m \n",
				"   \x1B[31m▄█\x1B[39m\x1B[34m░░░░░░\x1B[39m\x1B[31m█▄\x1B[39m                                          \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"  \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌▀▀▀▀█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m          \x1B[31m▄▄\x1B[39m                      \x1B[31m▄▄▄▄▄\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"   \x1B[31m▀▀  ▄▄▄█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m         \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m                    \x1B[31m▐\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌        ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"     \x1B[31m▄█\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[31m█\x1B[39m        \x1B[31m▄▄█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█▄▄\x1B[39m   \x1B[31m▄▄▄▄▄\x1B[39m           \x1B[31m▀▀▀▀▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌  ▄▄▄▄  ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"    \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌▀▀▀▀\x1B[39m        \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[31m▌\x1B[39m                 \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ █\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[31m█ ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"    \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m             \x1B[31m▀▀█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█▀▀\x1B[39m   \x1B[31m▀▀▀▀▀\x1B[39m           \x1B[31m▄▄▄▄▄\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"     \x1B[31m▀▀\x1B[39m                \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m             \x1B[31m▄▄▄▄▄\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ █\x1B[39m\x1B[34m░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"     \x1B[31m▄▄\x1B[39m         \x1B[31m▄▄\x1B[39m      \x1B[31m▀▀\x1B[39m             \x1B[31m▐\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▀▀▀▀▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▄▄███████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"    \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m       \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m                     \x1B[31m▀▀▀▀▀\x1B[39m          \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"     \x1B[31m▀▀\x1B[39m         \x1B[31m▀▀\x1B[39m                                      \x1B[31m▀▀▀▀▀▀▀▀▀▀▀▀\x1B[39m \n",
				"\n",
				"    \x1B[31m▄         ▄\x1B[39m          \x1B[31m▄\x1B[39m                \x1B[31m▄\x1B[39m   \x1B[31m▄▄▄▄▄▄▄\x1B[39m        \x1B[31m▄▄▄▄▄\x1B[39m   \x1B[31m▄▄▄▄▄\x1B[39m  \n",
				"   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄\x1B[39m    \x1B[31m▄     ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[31m▌\x1B[39m \n",
				"  \x1B[31m▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█▀▀▀\x1B[39m   \x1B[31m▀▀▀█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀\x1B[39m    \x1B[31m▀   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m         \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"  \x1B[31m▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄\x1B[39m       \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m         \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░░░░░░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m         \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"  \x1B[31m▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄\x1B[39m   \x1B[31m▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌ ▄\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m         \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				" \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▄\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌     ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m         \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"  \x1B[31m▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░░░░░░░░░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌   ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▄▄▄▄█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█▄▄▄\x1B[39m   \x1B[31m▄▄▄█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\n",
				"   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌       ▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▀▀▀▀█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█▀▀▀▀\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌     ▀\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░░░░░░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m    \x1B[31m▐\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[31m▌\x1B[39m \n",
				"    \x1B[31m▀         ▀\x1B[39m          \x1B[31m▀\x1B[39m        \x1B[31m▀\x1B[39m           \x1B[31m▀▀▀▀▀▀  ▀\x1B[39m      \x1B[31m▀▀▀▀▀\x1B[39m   \x1B[31m▀▀▀▀▀\x1B[39m  \n",
				"\n",
				"          \x1B[31m▄\x1B[39m                  \x1B[31m▄\x1B[39m   \x1B[31m▄  ▄\x1B[39m \n",
				"         \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m                \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐\x1B[39m\n",
				"        \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▄▄\x1B[39m   \x1B[31m▄▄\x1B[39m       \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐\x1B[39m\n",
				"       \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m   \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m       \x1B[31m▀\x1B[39m   \x1B[31m▀  ▀\x1B[39m \n",
				"      \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m     \x1B[31m▀▀\x1B[39m   \x1B[31m▀▀\x1B[39m                 \n",
				"     \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m                              \n",
				"    \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m       \x1B[31m▄▄\x1B[39m   \x1B[31m▄▄\x1B[39m                 \n",
				"   \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m       \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m  \x1B[31m▄▄\x1B[39m            \n",
				"  \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m         \x1B[31m▀▀\x1B[39m   \x1B[31m▀▌\x1B[39m  \x1B[31m▐\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m           \n",
				" \x1B[31m▐\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌\x1B[39m               \x1B[31m▀\x1B[39m    \x1B[31m▀▌\x1B[39m            \n",
				"  \x1B[31m▀\x1B[39m                     \x1B[31m▀\x1B[39m             \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"shade\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "shade".to_string()],
			fixture: concat!("\n\n",
				"░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░\n",
				"░░██░░███░░████░███░░████░████░░███░█░░█░███░███░░█░░█░█░░░░█░░█░█░░█░░██░░███░\n",
				"░█  █░█  █░█   ░█  █░█   ░█   ░█   ░█░░█░ █ ░  █░░█░░█░█░░░░████░██░█░█  █░█  █\n",
				"░████░███ ░█░░░░█░░█░███░░███░░█░██░████░░█░░░░█░░███ ░█░░░░█  █░█ ██░█░░█░███ \n",
				"░█  █░█  █░█░░░░█░░█░█  ░░█  ░░█░ █░█  █░░█░░█░█░░█  █░█░░░░█░░█░█░ █░█░░█░█  ░\n",
				"░█░░█░███ ░████░███ ░████░█░░░░███ ░█░░█░███░███░░█░░█░████░█░░█░█░░█░ ██ ░█░░░\n",
				"░ ░░ ░   ░░    ░   ░░    ░ ░░░░   ░░ ░░ ░   ░   ░░ ░░ ░    ░ ░░ ░ ░░ ░░  ░░ ░░░\n",
				"░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░\n",
				"\n",
				"░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░\n",
				"░░██░░███░░░███░███░█░░█░█░░░█░█░░█░█░░█░█░█░████░░██░░██░░░▐██░░████░█░░░░████\n",
				"░█  █░█  █░█   ░ █ ░█░░█░█░░░█░█░░█░█░░█░███░  █ ░█  █░ █░░░   █░   █░█░█░░█   \n",
				"░█░░█░███ ░ ██░░░█░░█░░█░█░░░█░█░░█░ ██ ░ █ ░░█ ░░█░▌█░░█░░░░░█ ░░░██░████░███░\n",
				"░█░░█░█  █░░  █░░█░░█░░█░ █░█ ░████░█  █░░█░░█ ░░░█░ █░░█░░░░█ ░░░░ █░   █░   █\n",
				"░ ███░█░░█░███ ░░█░░ ██ ░░ █ ░░█  █░█░░█░░█░░████░ ██ ░███░░████░████░░░░█░███ \n",
				"░░   ░ ░░ ░   ░░░ ░░░  ░░░░ ░░░ ░░ ░ ░░ ░░ ░░    ░░  ░░   ░░    ░    ░░░░ ░   ░\n",
				"░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░\n",
				"\n",
				"░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░\n",
				"░░███░████░░██░░░██░░░██░░▐██░░░░░░░░░░░░░░░░░░░░░░░░░░░░██░░░▌▐░░░░▌░░░░░█\n",
				"░█   ░   █░█  █░█  █░░██░░   █░░░░░░░░█░░░░░░░░░░░░████░█  █░████░░███░█░█ \n",
				"░███░░████░ ██ ░ ███░░██░░░░█ ░░░░░░░███░████░░░░░░    ░█░▌█░ ▌▐ ░█ █ ░ ░█░\n",
				"░█  █░ █  ░█  █░░  █░░  ░░░░ ░░░░░░░░ █ ░    ░░░░░░████░█░█ ░████░ ██░░░█ ░\n",
				"░ ██ ░█ ░░░ ██ ░░░█ ░░██░░░░█░░░░█░░░░ ░░░░░░░░░░░░    ░ ███░ ▌▐ ░░░▌█░░█░█\n",
				"░░  ░░ ░░░░░  ░░░░ ░░░  ░░░░ ░░░░ ░░░░░░░░░░░░████░░░░░░    ░░  ░░███ ░█ ░ \n",
				"░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░    ░░░░░░░░░░░░░░░░   ░░ ░░░\n",
				"\n",
				"░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░░\n",
				"░░█░░░░░░█░░░█░░░░░░█░░░░░░░░░░░░░░░░░█░░░░█░█░░\n",
				"░░█░░░░░█ ░░░ █░░░░█ ░░█░░░░░░░░░░░░░░ ░░░░ ░ ░░\n",
				"░█████░█ ░░░░░ █░░░█░░░ ░░░░░░░░░░░░░░░░░░░░░░░░\n",
				"░█  █ ░█░░░░░░░█░░█ ░░░░░░░░█░░░░░░░░░░░░░░░░░░░\n",
				"░████░░ █░░░░░█ ░░█░░░░█░░░░ ░░░░░░░░░░░░░░░░░░░\n",
				"░     ░░ █░░░█ ░░█ ░░░░ ░░░░█░░░░░█░░░░░░░░░░░░░\n",
				"░░░░░░░░░ ░░░ ░░░ ░░░░░░░░░░█░░░░█ ░░░░░░░░░░░░░\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"shade\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "shade".to_string(), "-c".to_string(), "red,blue".to_string()],
			fixture: concat!("\n\n",
				"\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m   \x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m   \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m   \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m   \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m \x1B[34m░\x1B[39m  \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█ ██\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m \n",
				"\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m  \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m  \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m  \x1B[34m░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m \x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m \x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m \x1B[31m██\x1B[39m \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m \x1B[34m░░\x1B[39m \x1B[34m░\x1B[39m   \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m    \x1B[34m░\x1B[39m   \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m    \x1B[34m░\x1B[39m \x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m   \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m \x1B[34m░░\x1B[39m \x1B[34m░\x1B[39m   \x1B[34m░\x1B[39m   \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m \x1B[34m░░\x1B[39m \x1B[34m░\x1B[39m    \x1B[34m░\x1B[39m \x1B[34m░░\x1B[39m \x1B[34m░\x1B[39m \x1B[34m░░\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░  ░\x1B[39m\x1B[34m░\x1B[39m \x1B[34m░░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\n",
				"\n",
				"\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m   \x1B[34m░\x1B[39m \x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m  \x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m   \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m   \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m   \n",
				"\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m \x1B[34m░\x1B[39m \x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m \x1B[31m██\x1B[39m \x1B[34m░\x1B[39m \x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m  \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m   \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m   \x1B[31m█\x1B[39m\n",
				"\x1B[34m░\x1B[39m \x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m \x1B[31m██\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m \x1B[31m██\x1B[39m \x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m \n",
				"\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m   \x1B[34m░\x1B[39m \x1B[34m░░\x1B[39m \x1B[34m░\x1B[39m   \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░ ░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░  ░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░ ░░\x1B[39m\x1B[34m░\x1B[39m \x1B[34m░░\x1B[39m \x1B[34m░\x1B[39m \x1B[34m░░\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░ ░\x1B[39m\x1B[34m░\x1B[39m    \x1B[34m░\x1B[39m\x1B[34m░  ░\x1B[39m\x1B[34m░\x1B[39m   \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m    \x1B[34m░\x1B[39m    \x1B[34m░\x1B[39m\x1B[34m░░░\x1B[39m \x1B[34m░\x1B[39m   \x1B[34m░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\n",
				"\n",
				"\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▐██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌▐\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░\x1B[39m\x1B[31m█\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m   \x1B[34m░\x1B[39m   \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m   \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \n",
				"\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m \x1B[31m██\x1B[39m \x1B[34m░\x1B[39m \x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m    \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m▌█\x1B[39m\x1B[34m░\x1B[39m \x1B[31m▌▐\x1B[39m \x1B[34m░\x1B[39m\x1B[31m█ █\x1B[39m \x1B[34m░\x1B[39m \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m  \x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m  \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░  ░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░ ░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m \x1B[34m░\x1B[39m    \x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m \x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\n",
				"\x1B[34m░\x1B[39m \x1B[31m██\x1B[39m \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m \x1B[31m██\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m██\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░ ░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m    \x1B[34m░\x1B[39m \x1B[31m███\x1B[39m\x1B[34m░\x1B[39m \x1B[31m▌▐\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m▌█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[34m░  ░\x1B[39m\x1B[34m░\x1B[39m \x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░  ░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░ ░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░  ░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░ ░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░ ░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m    \x1B[34m░\x1B[39m\x1B[34m░  ░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m███\x1B[39m \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m \n",
				"\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m    \x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m   \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m \x1B[34m░░░\x1B[39m\n",
				"\n",
				"\x1B[34m░\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░ ░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░ ░ ░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[31m█████\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░ ░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[31m█  █\x1B[39m \x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[31m████\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░ ░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[34m    \x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m \x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░ ░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░░░\x1B[39m\n",
				"\x1B[34m░\x1B[39m\x1B[34m░░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░ ░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░ ░░\x1B[39m\x1B[34m░\x1B[39m \x1B[34m░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m\x1B[34m░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[31m█\x1B[39m \x1B[34m░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░\x1B[39m\x1B[34m░\x1B[39m\x1B[34m░░░░░░\x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"simple\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "simple".to_string()],
			fixture: concat!("\n\n",
				"    _     ___     __   ___    ___   ___    ___   _  _   ___      _   _  _ \n",
				"   /_\\   | _ )  / __| |   \\  | __| | __|  / __| | || | |_ _|  _ | | | |/ /\n",
				"  / _ \\  | _ \\ | (__  | |) | | _|  | _|  | (_ | | __ |  | |  | || | | ' < \n",
				" /_/ \\_\\ |___/  \\___| |___/  |___| |_|    \\___| |_||_| |___|  \\__/  |_|\\_\\\n",
				"\n",
				"  _      _    _   _  _    ___    ___    ___    ___   ___   _____   _   _ \n",
				" | |    | \\  / | | \\| |  / _ \\  | _ \\  / _ \\  | _ \\ / __| |_   _| | | | |\n",
				" | |__  | |\\/| | | .` | | (_) | |  _/ | (_) | |   / \\__ \\   | |   | |_| |\n",
				" |____| |_|  |_| |_|\\_|  \\___/  |_|    \\__\\_\\ |_|_\\ |___/   |_|    \\___/ \n",
				"\n",
				" _    _  __      __ _   _  _    _   ____   __    _   ___   ___   _ _    ___ \n",
				" \\ \\ / / \\ \\    / / \\ \\/ / \\ \\ / / |_  /  /  \\  / | |_  ) |__ / | | |  | __|\n",
				"  \\ V /   \\ \\/\\/ /   >  <   \\ V /   / /  | () | | |  / /   |_ \\ |_  _| |__ \\\n",
				"   \\_/     \\_/\\_/   /_/\\_\\   |_|   /___|  \\__/  |_| /___| |___/   |_|  |___/\n",
				"\n",
				"   __   ____   ___   ___   _   ___         _                 ___   / __ \\ \n",
				"  / /  |__  | ( _ ) / _ \\ | | |__ \\      _| |_   ___        |___| / / _` |\n",
				" / _ \\   / /  / _ \\ \\_, / |_|  /_/      |_   _| |___|  ___   ___  \\ \\__,_|\n",
				" \\___/  /_/   \\___/  /_/  (_) (_)   (_)   |_|         |___| |___|  \\____/ \n",
				"\n",
				"  _| | |_   ||_     __   __       / / \\ \\     __             |-| |-||-|\n",
				" |_  .  _| (_-< (_)/ /  / _|___  | |   | |   / / (_) (_)     |/  |/ |/ \n",
				" |_     _| / _/   / /_  > _|_ _| | |   | |  / /      ( ) ( )           \n",
				"   |_|_|    ||   /_/(_) \\_____|   \\_\\ /_/  /_/   (_) |/  |/            \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"simple\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "simple".to_string(), "-c".to_string(), "red".to_string()],
			fixture: concat!("\n\n",
				"\x1B[31m \x1B[39m\x1B[31m   _   \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m   __ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___  \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m  ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m _  _ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m    _ \x1B[39m\x1B[31m \x1B[39m\x1B[31m _  _ \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m  /_\\  \x1B[39m\x1B[31m \x1B[39m\x1B[31m| _ )\x1B[39m\x1B[31m \x1B[39m\x1B[31m / __|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|   \\ \x1B[39m\x1B[31m \x1B[39m\x1B[31m| __|\x1B[39m\x1B[31m \x1B[39m\x1B[31m| __|\x1B[39m\x1B[31m \x1B[39m\x1B[31m / __|\x1B[39m\x1B[31m \x1B[39m\x1B[31m| || |\x1B[39m\x1B[31m \x1B[39m\x1B[31m|_ _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _ | |\x1B[39m\x1B[31m \x1B[39m\x1B[31m| |/ /\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m / _ \\ \x1B[39m\x1B[31m \x1B[39m\x1B[31m| _ \\\x1B[39m\x1B[31m \x1B[39m\x1B[31m| (__ \x1B[39m\x1B[31m \x1B[39m\x1B[31m| |) |\x1B[39m\x1B[31m \x1B[39m\x1B[31m| _| \x1B[39m\x1B[31m \x1B[39m\x1B[31m| _| \x1B[39m\x1B[31m \x1B[39m\x1B[31m| (_ |\x1B[39m\x1B[31m \x1B[39m\x1B[31m| __ |\x1B[39m\x1B[31m \x1B[39m\x1B[31m | | \x1B[39m\x1B[31m \x1B[39m\x1B[31m| || |\x1B[39m\x1B[31m \x1B[39m\x1B[31m| ' < \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m/_/ \\_\\\x1B[39m\x1B[31m \x1B[39m\x1B[31m|___/\x1B[39m\x1B[31m \x1B[39m\x1B[31m \\___|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|___/ \x1B[39m\x1B[31m \x1B[39m\x1B[31m|___|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m \\___|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|_||_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|___|\x1B[39m\x1B[31m \x1B[39m\x1B[31m \\__/ \x1B[39m\x1B[31m \x1B[39m\x1B[31m|_|\\_\\\x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m _    \x1B[39m\x1B[31m \x1B[39m\x1B[31m _    _ \x1B[39m\x1B[31m \x1B[39m\x1B[31m _  _ \x1B[39m\x1B[31m \x1B[39m\x1B[31m  ___  \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m  ___  \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m _____ \x1B[39m\x1B[31m \x1B[39m\x1B[31m _   _ \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m| |   \x1B[39m\x1B[31m \x1B[39m\x1B[31m| \\  / |\x1B[39m\x1B[31m \x1B[39m\x1B[31m| \\| |\x1B[39m\x1B[31m \x1B[39m\x1B[31m / _ \\ \x1B[39m\x1B[31m \x1B[39m\x1B[31m| _ \\\x1B[39m\x1B[31m \x1B[39m\x1B[31m / _ \\ \x1B[39m\x1B[31m \x1B[39m\x1B[31m| _ \\\x1B[39m\x1B[31m \x1B[39m\x1B[31m/ __|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|_   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m| | | |\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m| |__ \x1B[39m\x1B[31m \x1B[39m\x1B[31m| |\\/| |\x1B[39m\x1B[31m \x1B[39m\x1B[31m| .` |\x1B[39m\x1B[31m \x1B[39m\x1B[31m| (_) |\x1B[39m\x1B[31m \x1B[39m\x1B[31m|  _/\x1B[39m\x1B[31m \x1B[39m\x1B[31m| (_) |\x1B[39m\x1B[31m \x1B[39m\x1B[31m|   /\x1B[39m\x1B[31m \x1B[39m\x1B[31m\\__ \\\x1B[39m\x1B[31m \x1B[39m\x1B[31m  | |  \x1B[39m\x1B[31m \x1B[39m\x1B[31m| |_| |\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m|____|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|_|  |_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|_|\\_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m \\___/ \x1B[39m\x1B[31m \x1B[39m\x1B[31m|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m \\__\\_\\\x1B[39m\x1B[31m \x1B[39m\x1B[31m|_|_\\\x1B[39m\x1B[31m \x1B[39m\x1B[31m|___/\x1B[39m\x1B[31m \x1B[39m\x1B[31m  |_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m \\___/ \x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m_    _ \x1B[39m\x1B[31m \x1B[39m\x1B[31m__      __\x1B[39m\x1B[31m \x1B[39m\x1B[31m_   _ \x1B[39m\x1B[31m \x1B[39m\x1B[31m_    _ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ____\x1B[39m\x1B[31m \x1B[39m\x1B[31m  __  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m _ _  \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m\\ \\ / /\x1B[39m\x1B[31m \x1B[39m\x1B[31m\\ \\    / /\x1B[39m\x1B[31m \x1B[39m\x1B[31m\\ \\/ /\x1B[39m\x1B[31m \x1B[39m\x1B[31m\\ \\ / /\x1B[39m\x1B[31m \x1B[39m\x1B[31m|_  /\x1B[39m\x1B[31m \x1B[39m\x1B[31m /  \\ \x1B[39m\x1B[31m \x1B[39m\x1B[31m/ |\x1B[39m\x1B[31m \x1B[39m\x1B[31m|_  )\x1B[39m\x1B[31m \x1B[39m\x1B[31m|__ /\x1B[39m\x1B[31m \x1B[39m\x1B[31m| | | \x1B[39m\x1B[31m \x1B[39m\x1B[31m| __|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m \\ V / \x1B[39m\x1B[31m \x1B[39m\x1B[31m \\ \\/\\/ / \x1B[39m\x1B[31m \x1B[39m\x1B[31m >  < \x1B[39m\x1B[31m \x1B[39m\x1B[31m \\ V / \x1B[39m\x1B[31m \x1B[39m\x1B[31m / / \x1B[39m\x1B[31m \x1B[39m\x1B[31m| () |\x1B[39m\x1B[31m \x1B[39m\x1B[31m| |\x1B[39m\x1B[31m \x1B[39m\x1B[31m / / \x1B[39m\x1B[31m \x1B[39m\x1B[31m |_ \\\x1B[39m\x1B[31m \x1B[39m\x1B[31m|_  _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|__ \\\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m  \\_/  \x1B[39m\x1B[31m \x1B[39m\x1B[31m  \\_/\\_/  \x1B[39m\x1B[31m \x1B[39m\x1B[31m/_/\\_\\\x1B[39m\x1B[31m \x1B[39m\x1B[31m  |_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m/___|\x1B[39m\x1B[31m \x1B[39m\x1B[31m \\__/ \x1B[39m\x1B[31m \x1B[39m\x1B[31m|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m/___|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|___/\x1B[39m\x1B[31m \x1B[39m\x1B[31m  |_| \x1B[39m\x1B[31m \x1B[39m\x1B[31m|___/\x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m  __ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ____ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m _ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _   \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m / __ \\ \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m / / \x1B[39m\x1B[31m \x1B[39m\x1B[31m|__  |\x1B[39m\x1B[31m \x1B[39m\x1B[31m( _ )\x1B[39m\x1B[31m \x1B[39m\x1B[31m/ _ \\\x1B[39m\x1B[31m \x1B[39m\x1B[31m| |\x1B[39m\x1B[31m \x1B[39m\x1B[31m|__ \\\x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m _| |_ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\x1B[31m \x1B[39m\x1B[31m|___|\x1B[39m\x1B[31m \x1B[39m\x1B[31m/ / _` |\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m/ _ \\\x1B[39m\x1B[31m \x1B[39m\x1B[31m  / / \x1B[39m\x1B[31m \x1B[39m\x1B[31m/ _ \\\x1B[39m\x1B[31m \x1B[39m\x1B[31m\\_, /\x1B[39m\x1B[31m \x1B[39m\x1B[31m|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m /_/ \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m|_   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|___|\x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m\\ \\__,_|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m\\___/\x1B[39m\x1B[31m \x1B[39m\x1B[31m /_/  \x1B[39m\x1B[31m \x1B[39m\x1B[31m\\___/\x1B[39m\x1B[31m \x1B[39m\x1B[31m /_/ \x1B[39m\x1B[31m \x1B[39m\x1B[31m(_)\x1B[39m\x1B[31m \x1B[39m\x1B[31m(_)  \x1B[39m\x1B[31m \x1B[39m\x1B[31m(_)\x1B[39m\x1B[31m \x1B[39m\x1B[31m  |_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\x1B[31m \x1B[39m\x1B[31m|___|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|___|\x1B[39m\x1B[31m \x1B[39m\x1B[31m \\____/ \x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m _| | |_ \x1B[39m\x1B[31m \x1B[39m\x1B[31m ||_\x1B[39m\x1B[31m \x1B[39m\x1B[31m    __ \x1B[39m\x1B[31m \x1B[39m\x1B[31m __     \x1B[39m\x1B[31m \x1B[39m\x1B[31m / /\x1B[39m\x1B[31m \x1B[39m\x1B[31m\\ \\ \x1B[39m\x1B[31m \x1B[39m\x1B[31m   __\x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m|-|\x1B[39m\x1B[31m \x1B[39m\x1B[31m|-||-|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m|_  .  _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m(_-<\x1B[39m\x1B[31m \x1B[39m\x1B[31m(_)/ / \x1B[39m\x1B[31m \x1B[39m\x1B[31m/ _|___ \x1B[39m\x1B[31m \x1B[39m\x1B[31m| | \x1B[39m\x1B[31m \x1B[39m\x1B[31m | |\x1B[39m\x1B[31m \x1B[39m\x1B[31m  / /\x1B[39m\x1B[31m \x1B[39m\x1B[31m(_)\x1B[39m\x1B[31m \x1B[39m\x1B[31m(_)\x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m|/ \x1B[39m\x1B[31m \x1B[39m\x1B[31m|/ |/ \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m|_     _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m/ _/\x1B[39m\x1B[31m \x1B[39m\x1B[31m  / /_ \x1B[39m\x1B[31m \x1B[39m\x1B[31m> _|_ _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m| | \x1B[39m\x1B[31m \x1B[39m\x1B[31m | |\x1B[39m\x1B[31m \x1B[39m\x1B[31m / / \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m( )\x1B[39m\x1B[31m \x1B[39m\x1B[31m( )\x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m      \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m  |_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m || \x1B[39m\x1B[31m \x1B[39m\x1B[31m /_/(_)\x1B[39m\x1B[31m \x1B[39m\x1B[31m\\_____| \x1B[39m\x1B[31m \x1B[39m\x1B[31m \\_\\\x1B[39m\x1B[31m \x1B[39m\x1B[31m/_/ \x1B[39m\x1B[31m \x1B[39m\x1B[31m/_/  \x1B[39m\x1B[31m \x1B[39m\x1B[31m(_)\x1B[39m\x1B[31m \x1B[39m\x1B[31m|/ \x1B[39m\x1B[31m \x1B[39m\x1B[31m|/ \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m      \x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"simple3d\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "simple3d".to_string()],
			fixture: concat!("\n\n",
				"           __                 __               ___     __      __              \n",
				"   __     /\\ \\        ___    /\\ \\      __    /\'___\\  / _ `\\   /\\ \\        __   \n",
				" /\'__`\\   \\ \\ \\____  /\'___\\  \\_\\ \\   / ,.`\\ /\\ \\__/ /\\ \\_\\ \\  \\ \\ \\___   /\\_\\  \n",
				"/\\ \\_\\.\\_  \\ \\  ,. \\/\\ \\__/ /\\ ,. \\ /\\  __/ \\ \\ ,__\\\\ \\____ \\  \\ \\  _ `\\ \\/\\ \\ \n",
				"\\ \\__/.\\_\\  \\ \\____/\\ \\____\\\\ \\____\\\\ \\____\\ \\ \\_\\_/ \\/___/\\ \\  \\ \\_\\ \\_\\ \\ \\ \\\n",
				" \\/__/\\/_/   \\/___/  \\/____/ \\/___ / \\/____/  \\/_/     /\\____/   \\/_/\\/_/  \\/_/\n",
				"                                                       \\_/__/                  ",
				"\n\n",
				"  __      __  __     ___                                                      \n",
				" /\\_\\    /\\ \\/  \\   /\\_ \\      ___ ___      ___      ___    _____      __     \n",
				" \\/\\ \\   \\ \\    <   \\//\\ \\   /\' __` __`\\  /\' _ `\\   / __`\\ /\\ \'__`\\  /\'__`\\   \n",
				" _\\ \\ \\   \\ \\  ^  \\   \\_\\ \\_ /\\ \\/\\ \\/\\ \\ /\\ \\/\\ \\ /\\ \\_\\ \\\\ \\ \\_\\ \\/\\ \\L\\ \\  \n",
				"/\\ \\_\\ \\   \\ \\_\\ \\_\\  /\\____\\\\ \\_\\ \\_\\ \\_\\\\ \\_\\ \\_\\\\ \\____/ \\ \\ ,__/\\ \\___, \\ \n",
				"\\ \\____/    \\/_/\\/_/  \\/____/ \\/_/\\/_/\\/_/ \\/_/\\/_/ \\/___/   \\ \\ \\/  \\/___/\\_\\\n",
				" \\/___/                                                       \\/_/        \\/_/",
				"\n\n                __                                                       \n",
				" _ __    ____  /\\ \\__    __  __   __  __  __  __  __   __  _    __  __   \n",
				"/\\` __\\ / ,__\\ \\ \\ ,_\\  /\\ \\/\\ \\ /\\ \\/\\ \\/\\ \\/\\ \\/\\ \\ /\\ \\/\'\\  /\\ \\_\\ \\  \n",
				"\\ \\ \\/ /\\__, `\\ \\ \\ \\/  \\ \\ \\_\\ \\\\ \\ \\/ |\\ \\ \\_/ \\_/ \\\\/>  </  \\/`____ \\ \n",
				" \\ \\_\\ \\/\\____/  \\ \\ \\_  \\ \\____/ \\ \\___/ \\ \\___^___/\' /\\_/\\_\\  `/___/> \\\n",
				"  \\/_/  \\/___/    \\ \\__\\  \\/___/   \\/__/   \\/__//__/   \\//\\/_/     /\\___/\n",
				"                   \\/__/                                           \\/__/ ",
				"\n\n            __        _        ___        __      __ __       ______    \n",
				" ____     /\'__`\\    /\' \\     /\'___`\\    /\'__`\\   /\\ \\\\ \\     /\\  ___\\   \n",
				"/\\_ ,`\\  /\\ \\/\\ \\  /\\_, \\   /\\_\\ /\\ \\  /\\_\\L\\ \\  \\ \\ \\\\ \\    \\ \\ \\__/   \n",
				"\\/_/  /_ \\ \\ \\ \\ \\ \\/_/\\ \\  \\/_/// /__ \\/_/_\\_<_  \\ \\ \\\\ \\_   \\ \\___``\\ \n",
				"  /\\____\\ \\ \\ \\_\\ \\   \\ \\ \\    // /_\\ \\  /\\ \\L\\ \\  \\ \\__ ,__\\  \\/\\ \\L\\ \\\n",
				"  \\/____/  \\ \\____/    \\ \\_\\  /\\______/  \\ \\____/   \\/_/\\_\\_/   \\ \\____/\n",
				"            \\/___/      \\/_/  \\/_____/    \\/___/       \\/_/      \\/___/ ",
				"\n\n  ____     ________    __        __       __        _          __               \n",
				" /\'___\\   /\\____   \\ /\' _`\\    /\'_ `\\    /\\ \\     /\'_`\\       /\\ \\              \n",
				"/\\ \\__/   \\/___/\' /\'/\\ \\L\\ \\  /\\ \\L\\ \\   \\ \\ \\   /\\_\\/\\`\\     \\_\\ \\___  _______ \n",
				"\\ \\  _``\\     /\' /\' \\/_> _ <_ \\ \\___, \\   \\ \\ \\  \\/_//\'/\'    /\\___  __\\/\\______\\\n",
				" \\ \\ \\L\\ \\   /\' /\'    /\\ \\L\\ \\ \\/__,/\\ \\   \\ \\_\\    /\\_\\  __ \\/__/\\ \\_/\\/______/\n",
				"  \\ \\____/  /\\_/      \\ \\____/      \\ \\_\\   \\/\\_\\   \\/\\_\\/\\_\\    \\ \\_\\          \n",
				"   \\/___/   \\//        \\/___/        \\/_/    \\/_/    \\/_/\\/_/     \\/_/          ",
				"\n\n                        __        __ __          _      __    __       ____     \n",
				"           _______     /\'_`\\_    _\\ \\\\ \\__     /|_\\_   /\\_\\  /\\_\\    /|  _ \\    \n",
				"          /\\______\\   /\'/\'_` \\  /\\__  _  _\\   /\'  _ `\\ \\/_/ / / /    |/\\ ` |    \n",
				"          \\/______/_ /\\ \\ \\L\\ \\ \\/_L\\ \\\\ \\L_  \\ \\___  \\    / / /     \\ / __`\\/\\ \n",
				"  _______   /\\______\\\\ \\ `\\__,_\\  /\\_   _  _\\  \\ `\\_ _/   / / /  __  /|  \\L>  <_\n",
				" /\\______\\  \\/______/ \\ `\\_____\\  \\/_/\\_\\\\_\\/   `\\_/\\_\\  / / /  /\\_\\ | \\_____/\\/\n",
				" \\/______/             `\\/_____/     \\/_//_/       \\/_/  \\/_/   \\/_/  \\/____/\\/ ",
				"\n\n   _      __          __                    _    _   _  \n",
				" /\' \\    /\\ `\\       /\\_\\ __    __        /\\ \\ /\\ \\/\\ \\ \n",
				"/\\ ,/    \\`\\  \\     / / //\\_\\  /\\_\\       \\ \\/ \\ \\/\\ \\/ \n",
				"\\ \\ \\     `\\`\\ \\   / / / \\/_/_ \\/_/_    _  \\/   \\/  \\/  \n",
				" \\ \\ `\\    `\\/\' \\ / / /    /\\_\\  /\\ \\ /\\ \\              \n",
				"  \\ `\\__\\   /\\__// / /     \\/_/  \\ \\/ \\ \\/              \n",
				"   `\\/_/    \\/_/ \\/_/             \\/   \\/               \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"simple3d\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "simple3d".to_string(), "-c".to_string(), "red".to_string()],
			fixture: concat!("\n\n",
				"\x1B[31m          \x1B[39m\x1B[31m __       \x1B[39m\x1B[31m        \x1B[39m\x1B[31m  __    \x1B[39m\x1B[31m        \x1B[39m\x1B[31m   ___  \x1B[39m\x1B[31m   __     \x1B[39m\x1B[31m __        \x1B[39m\x1B[31m      \x1B[39m\n",
				"\x1B[31m   __     \x1B[39m\x1B[31m/\\ \\      \x1B[39m\x1B[31m  ___   \x1B[39m\x1B[31m /\\ \\   \x1B[39m\x1B[31m   __   \x1B[39m\x1B[31m /'___\\ \x1B[39m\x1B[31m / _ `\\   \x1B[39m\x1B[31m/\\ \\       \x1B[39m\x1B[31m __   \x1B[39m\n",
				"\x1B[31m /'__`\\   \x1B[39m\x1B[31m\\ \\ \\____ \x1B[39m\x1B[31m /'___\\ \x1B[39m\x1B[31m \\_\\ \\  \x1B[39m\x1B[31m / ,.`\\ \x1B[39m\x1B[31m/\\ \\__/ \x1B[39m\x1B[31m/\\ \\_\\ \\  \x1B[39m\x1B[31m\\ \\ \\___   \x1B[39m\x1B[31m/\\_\\  \x1B[39m\n",
				"\x1B[31m/\\ \\_\\.\\_ \x1B[39m\x1B[31m \\ \\  ,. \\\x1B[39m\x1B[31m/\\ \\__/ \x1B[39m\x1B[31m/\\ ,. \\ \x1B[39m\x1B[31m/\\  __/ \x1B[39m\x1B[31m\\ \\ ,__\\\x1B[39m\x1B[31m\\ \\____ \\ \x1B[39m\x1B[31m \\ \\  _ `\\ \x1B[39m\x1B[31m\\/\\ \\ \x1B[39m\n",
				"\x1B[31m\\ \\__/.\\_\\\x1B[39m\x1B[31m  \\ \\____/\x1B[39m\x1B[31m\\ \\____\\\x1B[39m\x1B[31m\\ \\____\\\x1B[39m\x1B[31m\\ \\____\\\x1B[39m\x1B[31m \\ \\_\\_/\x1B[39m\x1B[31m \\/___/\\ \\\x1B[39m\x1B[31m  \\ \\_\\ \\_\\\x1B[39m\x1B[31m \\ \\ \\\x1B[39m\n",
				"\x1B[31m \\/__/\\/_/\x1B[39m\x1B[31m   \\/___/ \x1B[39m\x1B[31m \\/____/\x1B[39m\x1B[31m \\/___ /\x1B[39m\x1B[31m \\/____/\x1B[39m\x1B[31m  \\/_/  \x1B[39m\x1B[31m   /\\____/\x1B[39m\x1B[31m   \\/_/\\/_/\x1B[39m\x1B[31m  \\/_/\x1B[39m\n",
				"\x1B[31m          \x1B[39m\x1B[31m          \x1B[39m\x1B[31m        \x1B[39m\x1B[31m        \x1B[39m\x1B[31m        \x1B[39m\x1B[31m        \x1B[39m\x1B[31m   \\_/__/ \x1B[39m\x1B[31m           \x1B[39m\x1B[31m      \x1B[39m\n",
				"\n",
				"\x1B[31m  __    \x1B[39m\x1B[31m  __  __    \x1B[39m\x1B[31m ___     \x1B[39m\x1B[31m             \x1B[39m\x1B[31m         \x1B[39m\x1B[31m        \x1B[39m\x1B[31m         \x1B[39m\x1B[31m          \x1B[39m\n",
				"\x1B[31m /\\_\\   \x1B[39m\x1B[31m /\\ \\/  \\   \x1B[39m\x1B[31m/\\_ \\    \x1B[39m\x1B[31m  ___ ___    \x1B[39m\x1B[31m  ___    \x1B[39m\x1B[31m  ___   \x1B[39m\x1B[31m _____   \x1B[39m\x1B[31m   __     \x1B[39m\n",
				"\x1B[31m \\/\\ \\  \x1B[39m\x1B[31m \\ \\    <   \x1B[39m\x1B[31m\\//\\ \\   \x1B[39m\x1B[31m/' __` __`\\  \x1B[39m\x1B[31m/' _ `\\  \x1B[39m\x1B[31m / __`\\ \x1B[39m\x1B[31m/\\ '__`\\ \x1B[39m\x1B[31m /'__`\\   \x1B[39m\n",
				"\x1B[31m _\\ \\ \\ \x1B[39m\x1B[31m  \\ \\  ^  \\ \x1B[39m\x1B[31m  \\_\\ \\_ \x1B[39m\x1B[31m/\\ \\/\\ \\/\\ \\ \x1B[39m\x1B[31m/\\ \\/\\ \\ \x1B[39m\x1B[31m/\\ \\_\\ \\\x1B[39m\x1B[31m\\ \\ \\_\\ \\\x1B[39m\x1B[31m/\\ \\L\\ \\  \x1B[39m\n",
				"\x1B[31m/\\ \\_\\ \\\x1B[39m\x1B[31m   \\ \\_\\ \\_\\\x1B[39m\x1B[31m  /\\____\\\x1B[39m\x1B[31m\\ \\_\\ \\_\\ \\_\\\x1B[39m\x1B[31m\\ \\_\\ \\_\\\x1B[39m\x1B[31m\\ \\____/\x1B[39m\x1B[31m \\ \\ ,__/\x1B[39m\x1B[31m\\ \\___, \\ \x1B[39m\n",
				"\x1B[31m\\ \\____/\x1B[39m\x1B[31m    \\/_/\\/_/\x1B[39m\x1B[31m  \\/____/\x1B[39m\x1B[31m \\/_/\\/_/\\/_/\x1B[39m\x1B[31m \\/_/\\/_/\x1B[39m\x1B[31m \\/___/ \x1B[39m\x1B[31m  \\ \\ \\/ \x1B[39m\x1B[31m \\/___/\\_\\\x1B[39m\n",
				"\x1B[31m \\/___/ \x1B[39m\x1B[31m            \x1B[39m\x1B[31m         \x1B[39m\x1B[31m             \x1B[39m\x1B[31m         \x1B[39m\x1B[31m        \x1B[39m\x1B[31m   \\/_/  \x1B[39m\x1B[31m      \\/_/\x1B[39m\n",
				"\n",
				"\x1B[31m       \x1B[39m\x1B[31m        \x1B[39m\x1B[31m __      \x1B[39m\x1B[31m         \x1B[39m\x1B[31m        \x1B[39m\x1B[31m             \x1B[39m\x1B[31m        \x1B[39m\x1B[31m           \x1B[39m\n",
				"\x1B[31m _ __  \x1B[39m\x1B[31m  ____  \x1B[39m\x1B[31m/\\ \\__   \x1B[39m\x1B[31m __  __  \x1B[39m\x1B[31m __  __ \x1B[39m\x1B[31m __  __  __  \x1B[39m\x1B[31m __  _  \x1B[39m\x1B[31m  __  __   \x1B[39m\n",
				"\x1B[31m/\\` __\\\x1B[39m\x1B[31m / ,__\\ \x1B[39m\x1B[31m\\ \\ ,_\\  \x1B[39m\x1B[31m/\\ \\/\\ \\ \x1B[39m\x1B[31m/\\ \\/\\ \\\x1B[39m\x1B[31m/\\ \\/\\ \\/\\ \\ \x1B[39m\x1B[31m/\\ \\/'\\ \x1B[39m\x1B[31m /\\ \\_\\ \\  \x1B[39m\n",
				"\x1B[31m\\ \\ \\/ \x1B[39m\x1B[31m/\\__, `\\\x1B[39m\x1B[31m \\ \\ \\/  \x1B[39m\x1B[31m\\ \\ \\_\\ \\\x1B[39m\x1B[31m\\ \\ \\/ |\x1B[39m\x1B[31m\\ \\ \\_/ \\_/ \\\x1B[39m\x1B[31m\\/>  </ \x1B[39m\x1B[31m \\/`____ \\ \x1B[39m\n",
				"\x1B[31m \\ \\_\\ \x1B[39m\x1B[31m\\/\\____/\x1B[39m\x1B[31m  \\ \\ \\_ \x1B[39m\x1B[31m \\ \\____/\x1B[39m\x1B[31m \\ \\___/\x1B[39m\x1B[31m \\ \\___^___/'\x1B[39m\x1B[31m /\\_/\\_\\\x1B[39m\x1B[31m  `/___/> \\\x1B[39m\n",
				"\x1B[31m  \\/_/ \x1B[39m\x1B[31m \\/___/ \x1B[39m\x1B[31m   \\ \\__\\\x1B[39m\x1B[31m  \\/___/ \x1B[39m\x1B[31m  \\/__/ \x1B[39m\x1B[31m  \\/__//__/  \x1B[39m\x1B[31m \\//\\/_/\x1B[39m\x1B[31m     /\\___/\x1B[39m\n",
				"\x1B[31m       \x1B[39m\x1B[31m        \x1B[39m\x1B[31m    \\/__/\x1B[39m\x1B[31m         \x1B[39m\x1B[31m        \x1B[39m\x1B[31m             \x1B[39m\x1B[31m        \x1B[39m\x1B[31m     \\/__/ \x1B[39m\n",
				"\n",
				"\x1B[31m         \x1B[39m\x1B[31m   __     \x1B[39m\x1B[31m   _     \x1B[39m\x1B[31m   ___     \x1B[39m\x1B[31m   __     \x1B[39m\x1B[31m __ __      \x1B[39m\x1B[31m ______    \x1B[39m\n",
				"\x1B[31m ____    \x1B[39m\x1B[31m /'__`\\   \x1B[39m\x1B[31m /' \\    \x1B[39m\x1B[31m /'___`\\   \x1B[39m\x1B[31m /'__`\\   \x1B[39m\x1B[31m/\\ \\\\ \\     \x1B[39m\x1B[31m/\\  ___\\   \x1B[39m\n",
				"\x1B[31m/\\_ ,`\\  \x1B[39m\x1B[31m/\\ \\/\\ \\  \x1B[39m\x1B[31m/\\_, \\   \x1B[39m\x1B[31m/\\_\\ /\\ \\  \x1B[39m\x1B[31m/\\_\\L\\ \\  \x1B[39m\x1B[31m\\ \\ \\\\ \\    \x1B[39m\x1B[31m\\ \\ \\__/   \x1B[39m\n",
				"\x1B[31m\\/_/  /_ \x1B[39m\x1B[31m\\ \\ \\ \\ \\ \x1B[39m\x1B[31m\\/_/\\ \\  \x1B[39m\x1B[31m\\/_/// /__ \x1B[39m\x1B[31m\\/_/_\\_<_ \x1B[39m\x1B[31m \\ \\ \\\\ \\_  \x1B[39m\x1B[31m \\ \\___``\\ \x1B[39m\n",
				"\x1B[31m  /\\____\\\x1B[39m\x1B[31m \\ \\ \\_\\ \\\x1B[39m\x1B[31m   \\ \\ \\ \x1B[39m\x1B[31m   // /_\\ \\\x1B[39m\x1B[31m  /\\ \\L\\ \\\x1B[39m\x1B[31m  \\ \\__ ,__\\\x1B[39m\x1B[31m  \\/\\ \\L\\ \\\x1B[39m\n",
				"\x1B[31m  \\/____/\x1B[39m\x1B[31m  \\ \\____/\x1B[39m\x1B[31m    \\ \\_\\\x1B[39m\x1B[31m  /\\______/\x1B[39m\x1B[31m  \\ \\____/\x1B[39m\x1B[31m   \\/_/\\_\\_/\x1B[39m\x1B[31m   \\ \\____/\x1B[39m\n",
				"\x1B[31m         \x1B[39m\x1B[31m   \\/___/ \x1B[39m\x1B[31m     \\/_/\x1B[39m\x1B[31m  \\/_____/ \x1B[39m\x1B[31m   \\/___/ \x1B[39m\x1B[31m      \\/_/  \x1B[39m\x1B[31m    \\/___/ \x1B[39m\n",
				"\n",
				"\x1B[31m  ____    \x1B[39m\x1B[31m ________ \x1B[39m\x1B[31m   __     \x1B[39m\x1B[31m   __      \x1B[39m\x1B[31m __     \x1B[39m\x1B[31m   _    \x1B[39m\x1B[31m    \x1B[39m\x1B[31m  __      \x1B[39m\x1B[31m         \x1B[39m\n",
				"\x1B[31m /'___\\   \x1B[39m\x1B[31m/\\____   \\\x1B[39m\x1B[31m /' _`\\   \x1B[39m\x1B[31m /'_ `\\    \x1B[39m\x1B[31m/\\ \\    \x1B[39m\x1B[31m /'_`\\  \x1B[39m\x1B[31m    \x1B[39m\x1B[31m /\\ \\     \x1B[39m\x1B[31m         \x1B[39m\n",
				"\x1B[31m/\\ \\__/   \x1B[39m\x1B[31m\\/___/' /'\x1B[39m\x1B[31m/\\ \\L\\ \\  \x1B[39m\x1B[31m/\\ \\L\\ \\   \x1B[39m\x1B[31m\\ \\ \\   \x1B[39m\x1B[31m/\\_\\/\\`\\\x1B[39m\x1B[31m    \x1B[39m\x1B[31m \\_\\ \\___ \x1B[39m\x1B[31m _______ \x1B[39m\n",
				"\x1B[31m\\ \\  _``\\ \x1B[39m\x1B[31m    /' /' \x1B[39m\x1B[31m\\/_> _ <_ \x1B[39m\x1B[31m\\ \\___, \\  \x1B[39m\x1B[31m \\ \\ \\  \x1B[39m\x1B[31m\\/_//'/'\x1B[39m\x1B[31m    \x1B[39m\x1B[31m/\\___  __\\\x1B[39m\x1B[31m/\\______\\\x1B[39m\n",
				"\x1B[31m \\ \\ \\L\\ \\\x1B[39m\x1B[31m   /' /'  \x1B[39m\x1B[31m  /\\ \\L\\ \\\x1B[39m\x1B[31m \\/__,/\\ \\ \x1B[39m\x1B[31m  \\ \\_\\ \x1B[39m\x1B[31m   /\\_\\ \x1B[39m\x1B[31m __ \x1B[39m\x1B[31m\\/__/\\ \\_/\x1B[39m\x1B[31m\\/______/\x1B[39m\n",
				"\x1B[31m  \\ \\____/\x1B[39m\x1B[31m  /\\_/    \x1B[39m\x1B[31m  \\ \\____/\x1B[39m\x1B[31m      \\ \\_\\\x1B[39m\x1B[31m   \\/\\_\\\x1B[39m\x1B[31m   \\/\\_\\\x1B[39m\x1B[31m/\\_\\\x1B[39m\x1B[31m    \\ \\_\\ \x1B[39m\x1B[31m         \x1B[39m\n",
				"\x1B[31m   \\/___/ \x1B[39m\x1B[31m  \\//     \x1B[39m\x1B[31m   \\/___/ \x1B[39m\x1B[31m       \\/_/\x1B[39m\x1B[31m    \\/_/\x1B[39m\x1B[31m    \\/_/\x1B[39m\x1B[31m\\/_/\x1B[39m\x1B[31m     \\/_/ \x1B[39m\x1B[31m         \x1B[39m\n",
				"\n",
				"\x1B[31m          \x1B[39m\x1B[31m           \x1B[39m\x1B[31m   __      \x1B[39m\x1B[31m  __ __      \x1B[39m\x1B[31m    _     \x1B[39m\x1B[31m __    __    \x1B[39m\x1B[31m   ____     \x1B[39m\n",
				"\x1B[31m          \x1B[39m\x1B[31m _______   \x1B[39m\x1B[31m  /'_`\\_   \x1B[39m\x1B[31m _\\ \\\\ \\__   \x1B[39m\x1B[31m  /|_\\_   \x1B[39m\x1B[31m/\\_\\  /\\_\\   \x1B[39m\x1B[31m /|  _ \\    \x1B[39m\n",
				"\x1B[31m          \x1B[39m\x1B[31m/\\______\\  \x1B[39m\x1B[31m /'/'_` \\  \x1B[39m\x1B[31m/\\__  _  _\\  \x1B[39m\x1B[31m /'  _ `\\ \x1B[39m\x1B[31m\\/_/ / / /   \x1B[39m\x1B[31m |/\\ ` |    \x1B[39m\n",
				"\x1B[31m          \x1B[39m\x1B[31m\\/______/_ \x1B[39m\x1B[31m/\\ \\ \\L\\ \\ \x1B[39m\x1B[31m\\/_L\\ \\\\ \\L_ \x1B[39m\x1B[31m \\ \\___  \\\x1B[39m\x1B[31m    / / /    \x1B[39m\x1B[31m \\ / __`\\/\\ \x1B[39m\n",
				"\x1B[31m  _______ \x1B[39m\x1B[31m  /\\______\\\x1B[39m\x1B[31m\\ \\ `\\__,_\\\x1B[39m\x1B[31m  /\\_   _  _\\\x1B[39m\x1B[31m  \\ `\\_ _/\x1B[39m\x1B[31m   / / /  __ \x1B[39m\x1B[31m /|  \\L>  <_\x1B[39m\n",
				"\x1B[31m /\\______\\\x1B[39m\x1B[31m  \\/______/\x1B[39m\x1B[31m \\ `\\_____\\\x1B[39m\x1B[31m  \\/_/\\_\\\\_\\/\x1B[39m\x1B[31m   `\\_/\\_\\\x1B[39m\x1B[31m  / / /  /\\_\\\x1B[39m\x1B[31m | \\_____/\\/\x1B[39m\n",
				"\x1B[31m \\/______/\x1B[39m\x1B[31m           \x1B[39m\x1B[31m  `\\/_____/\x1B[39m\x1B[31m     \\/_//_/ \x1B[39m\x1B[31m      \\/_/\x1B[39m\x1B[31m  \\/_/   \\/_/\x1B[39m\x1B[31m  \\/____/\\/ \x1B[39m\n",
				"\n",
				"\x1B[31m   _     \x1B[39m\x1B[31m __     \x1B[39m\x1B[31m     __ \x1B[39m\x1B[31m      \x1B[39m\x1B[31m      \x1B[39m\x1B[31m     \x1B[39m\x1B[31m  _  \x1B[39m\x1B[31m  _   _  \x1B[39m\n",
				"\x1B[31m /' \\    \x1B[39m\x1B[31m/\\ `\\   \x1B[39m\x1B[31m    /\\_\\\x1B[39m\x1B[31m __   \x1B[39m\x1B[31m __   \x1B[39m\x1B[31m     \x1B[39m\x1B[31m/\\ \\ \x1B[39m\x1B[31m/\\ \\/\\ \\ \x1B[39m\n",
				"\x1B[31m/\\ ,/    \x1B[39m\x1B[31m\\`\\  \\  \x1B[39m\x1B[31m   / / /\x1B[39m\x1B[31m/\\_\\  \x1B[39m\x1B[31m/\\_\\  \x1B[39m\x1B[31m     \x1B[39m\x1B[31m\\ \\/ \x1B[39m\x1B[31m\\ \\/\\ \\/ \x1B[39m\n",
				"\x1B[31m\\ \\ \\    \x1B[39m\x1B[31m `\\`\\ \\ \x1B[39m\x1B[31m  / / / \x1B[39m\x1B[31m\\/_/_ \x1B[39m\x1B[31m\\/_/_ \x1B[39m\x1B[31m   _ \x1B[39m\x1B[31m \\/  \x1B[39m\x1B[31m \\/  \\/  \x1B[39m\n",
				"\x1B[31m \\ \\ `\\  \x1B[39m\x1B[31m  `\\/' \\\x1B[39m\x1B[31m / / /  \x1B[39m\x1B[31m  /\\_\\\x1B[39m\x1B[31m  /\\ \\\x1B[39m\x1B[31m /\\ \\\x1B[39m\x1B[31m     \x1B[39m\x1B[31m         \x1B[39m\n",
				"\x1B[31m  \\ `\\__\\\x1B[39m\x1B[31m   /\\__/\x1B[39m\x1B[31m/ / /   \x1B[39m\x1B[31m  \\/_/\x1B[39m\x1B[31m  \\ \\/\x1B[39m\x1B[31m \\ \\/\x1B[39m\x1B[31m     \x1B[39m\x1B[31m         \x1B[39m\n",
				"\x1B[31m   `\\/_/ \x1B[39m\x1B[31m   \\/_/ \x1B[39m\x1B[31m\\/_/    \x1B[39m\x1B[31m      \x1B[39m\x1B[31m   \\/ \x1B[39m\x1B[31m  \\/ \x1B[39m\x1B[31m     \x1B[39m\x1B[31m         \x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"simpleBlock\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "simpleBlock".to_string()],
			fixture: concat!("\n\n",
				"                                                                                \n",
				"    _|_|    _|_|_|      _|_|_|  _|_|_|    _|_|_|_|  _|_|_|_|    _|_|_|  _|    _|\n",
				"  _|    _|  _|    _|  _|        _|    _|  _|        _|        _|        _|    _|\n",
				"  _|_|_|_|  _|_|_|    _|        _|    _|  _|_|_|    _|_|_|    _|  _|_|  _|_|_|_|\n",
				"  _|    _|  _|    _|  _|        _|    _|  _|        _|        _|    _|  _|    _|\n",
				"  _|    _|  _|_|_|      _|_|_|  _|_|_|    _|_|_|_|  _|          _|_|_|  _|    _|\n",
				"                                                                                \n",
				"\n",
				"                                                                        \n",
				"  _|_|_|        _|  _|    _|  _|        _|      _|  _|      _|    _|_|  \n",
				"    _|          _|  _|  _|    _|        _|_|  _|_|  _|_|    _|  _|    _|\n",
				"    _|          _|  _|_|      _|        _|  _|  _|  _|  _|  _|  _|    _|\n",
				"    _|    _|    _|  _|  _|    _|        _|      _|  _|    _|_|  _|    _|\n",
				"  _|_|_|    _|_|    _|    _|  _|_|_|_|  _|      _|  _|      _|    _|_|  \n",
				"                                                                        \n",
				"\n",
				"                                                                            \n",
				"  _|_|_|      _|_|      _|_|_|      _|_|_|  _|_|_|_|_|  _|    _|  _|      _|\n",
				"  _|    _|  _|    _|    _|    _|  _|            _|      _|    _|  _|      _|\n",
				"  _|_|_|    _|  _|_|    _|_|_|      _|_|        _|      _|    _|  _|      _|\n",
				"  _|        _|    _|    _|    _|        _|      _|      _|    _|    _|  _|  \n",
				"  _|          _|_|  _|  _|    _|  _|_|_|        _|        _|_|        _|    \n",
				"                                                                            \n",
				"\n",
				"                                                                            \n",
				"  _|          _|  _|      _|  _|      _|  _|_|_|_|_|    _|      _|    _|_|  \n",
				"  _|          _|    _|  _|      _|  _|          _|    _|  _|  _|_|  _|    _|\n",
				"  _|    _|    _|      _|          _|          _|      _|  _|    _|      _|  \n",
				"    _|  _|  _|      _|  _|        _|        _|        _|  _|    _|    _|    \n",
				"      _|  _|      _|      _|      _|      _|_|_|_|_|    _|      _|  _|_|_|_|\n",
				"                                                                            \n",
				"\n",
				"                                                                            \n",
				"  _|_|_|    _|  _|    _|_|_|_|    _|_|_|  _|_|_|_|_|    _|_|      _|_|    _|\n",
				"        _|  _|  _|    _|        _|                _|  _|    _|  _|    _|  _|\n",
				"    _|_|    _|_|_|_|  _|_|_|    _|_|_|          _|      _|_|      _|_|_|  _|\n",
				"        _|      _|          _|  _|    _|      _|      _|    _|        _|    \n",
				"  _|_|_|        _|    _|_|_|      _|_|      _|          _|_|    _|_|_|    _|\n",
				"                                                                            \n",
				"\n",
				"                                                                 _|_|_|_|_|  \n",
				"  _|_|           _|                                            _|          _|\n",
				"      _|         _|                              _|_|_|_|_|  _|    _|_|_|  _|\n",
				"  _|_|       _|_|_|_|_|  _|_|_|_|_|                          _|  _|    _|  _|\n",
				"                 _|                              _|_|_|_|_|  _|    _|_|_|_|  \n",
				"  _|             _|                  _|_|_|_|_|                _|            \n",
				"         _|                                                      _|_|_|_|_|_|\n",
				"\n",
				"                _|                              _|  _|                       \n",
				"    _|  _|    _|_|_|  _|_|    _|    _|        _|      _|          _|         \n",
				"  _|_|_|_|_|  _|_|    _|_|  _|    _|  _|      _|      _|        _|   _|      \n",
				"    _|  _|      _|_|      _|        _|_|  _|  _|      _|      _|           _|\n",
				"  _|_|_|_|_|  _|_|_|    _|  _|_|  _|    _|    _|      _|    _|       _|      \n",
				"    _|  _|      _|    _|    _|_|    _|_|  _|  _|      _|  _|               _|\n",
				"                                                _|  _|                   _|  \n",
				"\n",
				"        _|  _|_|\n",
				"        _|  _|_|\n",
				"                \n",
				"                \n",
				"                \n",
				"    _|          \n",
				"  _|            \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"simpleBlock\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "simpleBlock".to_string(), "-c".to_string(), "red".to_string()],
			fixture: concat!("\n\n",
				"\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m       \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m       _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m   _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m       _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|  _|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m   _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m       _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|  _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|  _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m   _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m       \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m       _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|  _|  \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m               \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m       \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|          _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|          _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m       _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|    _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|  \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m   _|  _|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|    \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m     _|  _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m               \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m       \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m       _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m         _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m       _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m       _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m       _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m       _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m         \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m       \x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|_|_|_|_|  \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|          _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m     _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|_|_|  _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|    _|  _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m       \x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|_|_|_|  \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|            \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m       \x1B[39m\x1B[31m \x1B[39m\x1B[31m_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|_|_|_|_|_|\x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m   _|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|    _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m         _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|_|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m       _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m   _|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m     _|    \x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|_|_|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|  _|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|      \x1B[39m\x1B[31m \x1B[39m\x1B[31m_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m   _|  _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|    _|_|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|_|  _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|        \x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m       \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m           \x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|  \x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\x1B[31m \x1B[39m\x1B[31m _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m _|_|\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m   _|\x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m _|  \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\x1B[31m \x1B[39m\x1B[31m     \x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"slick\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "slick".to_string()],
			fixture: concat!("\n\n",
				"╱╭━━━╮╱╭━━╮╱╱╭━━━╮╱╭━━━╮╱╭━━━╮╱╭━━━╮╱╭━━━╮╱╭╮╱╭╮╱╭━━╮╱╱╱╭╮╱╭╮╭━╮╱╭╮╱╱╱╱╭━╮╭━╮\n",
				"╱┃╭━╮┃╱┃╭╮┃╱╱┃╭━╮┃╱╰╮╭╮┃╱┃╭━━╯╱┃╭━━╯╱┃╭━╮┃╱┃┃╱┃┃╱╰┫┣╯╱╱╱┃┃╱┃┃┃╭╯╱┃┃╱╱╱╱┃┃╰╯┃┃\n",
				"╱┃┃╱┃┃╱┃╰╯╰╮╱┃┃╱╰╯╱╱┃┃┃┃╱┃╰━━╮╱┃╰━━╮╱┃┃╱╰╯╱┃╰━╯┃╱╱┃┃╱╱╱╱┃┃╱┃╰╯╯╱╱┃┃╱╱╱╱┃╭╮╭╮┃\n",
				"╱┃╰━╯┃╱┃╭━╮┃╱┃┃╱╭╮╱╱┃┃┃┃╱┃╭━━╯╱┃╭━━╯╱┃┃╭━╮╱┃╭━╮┃╱╱┃┃╱╱╭╮┃┃╱┃╭╮┃╱╱┃┃╱╭╮╱┃┃┃┃┃┃\n",
				"╱┃╭━╮┃╱┃╰━╯┃╱┃╰━╯┃╱╭╯╰╯┃╱┃╰━━╮╱┃┃╱╱╱╱┃╰┻━┃╱┃┃╱┃┃╱╭┫┣╮╱┃╰╯┃╱┃┃┃╰╮╱┃╰━╯┃╱┃┃┃┃┃┃\n",
				"╱╰╯╱╰╯╱╰━━━╯╱╰━━━╯╱╰━━━╯╱╰━━━╯╱╰╯╱╱╱╱╰━━━╯╱╰╯╱╰╯╱╰━━╯╱╰━━╯╱╰╯╰━╯╱╰━━━╯╱╰╯╰╯╰╯\n",
				"\n",
				"╱╭━╮╱╭╮╱╭━━━╮╱╭━━━╮╱╭━━━╮╱╱╭━━━╮╱╭━━━╮╱╭━━━━╮╱╭╮╱╭╮╱╭╮╱╱╭╮╱╭╮╭╮╭╮╱╭━╮╭━╮╱╭╮╱╱╭╮\n",
				"╱┃┃╰╮┃┃╱┃╭━╮┃╱┃╭━╮┃╱┃╭━╮┃╱╱┃╭━╮┃╱┃╭━╮┃╱┃╭╮╭╮┃╱┃┃╱┃┃╱┃╰╮╭╯┃╱┃┃┃┃┃┃╱╰╮╰╯╭╯╱┃╰╮╭╯┃\n",
				"╱┃╭╮╰╯┃╱┃┃╱┃┃╱┃╰━╯┃╱┃┃╱┃┃╱╱┃╰━╯┃╱┃╰━━╮╱╰╯┃┃╰╯╱┃┃╱┃┃╱╰╮┃┃╭╯╱┃┃┃┃┃┃╱╱╰╮╭╯╱╱╰╮╰╯╭╯\n",
				"╱┃┃╰╮┃┃╱┃┃╱┃┃╱┃╭━━╯╱┃┃╱┃┃╱╱┃╭╮╭╯╱╰━━╮┃╱╱╱┃┃╱╱╱┃┃╱┃┃╱╱┃╰╯┃╱╱┃╰╯╰╯┃╱╱╭╯╰╮╱╱╱╰╮╭╯╱\n",
				"╱┃┃╱┃┃┃╱┃╰━╯┃╱┃┃╱╱╱╱┃╰━╯┃╮╱┃┃┃╰╮╱┃╰━╯┃╱╱╱┃┃╱╱╱┃╰━╯┃╱╱╰╮╭╯╱╱╰╮╭╮╭╯╱╭╯╭╮╰╮╱╱╱┃┃╱╱\n",
				"╱╰╯╱╰━╯╱╰━━━╯╱╰╯╱╱╱╱╰━━━━╯╱╰╯╰━╯╱╰━━━╯╱╱╱╰╯╱╱╱╰━━━╯╱╱╱╰╯╱╱╱╱╰╯╰╯╱╱╰━╯╰━╯╱╱╱╰╯╱╱\n",
				"\n",
				"╱╭━━━━╮╱╭━━━╮╱╱╭╮╱╱╭━━━╮╱╭━━━╮╱╭╮╱╭╮╱╭━━━╮╱╭━━━╮╱╭━━━╮╱╭━━━╮╱╭━━━╮╱╭╮╱╭━━━╮╱╱╱\n",
				"╱╰━━╮━┃╱┃╭━╮┃╱╭╯┃╱╱┃╭━╮┃╱┃╭━╮┃╱┃┃╱┃┃╱┃╭━━╯╱┃╭━━╯╱┃╭━╮┃╱┃╭━╮┃╱┃╭━╮┃╱┃┃╱┃╭━╮┃╱╱╱\n",
				"╱╱╱╭╯╭╯╱┃┃┃┃┃╱╰╮┃╱╱╰╯╭╯┃╱╰╯╭╯┃╱┃╰━╯┃╱┃╰━━╮╱┃╰━━╮╱╰╯╭╯┃╱┃╰━╯┃╱┃╰━╯┃╱┃┃╱╰╯╭╯┃╱╱╱\n",
				"╱╱╭╯╭╯╱╱┃┃┃┃┃╱╱┃┃╱╱╭━╯╭╯╱╭╮╰╮┃╱╰━━╮┃╱╰━━╮┃╱┃╭━╮┃╱╱╱┃╭╯╱┃╭━╮┃╱╰━━╮┃╱╰╯╱╱╱┃╭╯╱╱╱\n",
				"╱╭╯━╰━╮╱┃╰━╯┃╱╭╯╰╮╱┃╰━━╮╱┃╰━╯┃╱╱╱╱┃┃╱╭━━╯┃╱┃╰━╯┃╱╱╱┃┃╱╱┃╰━╯┃╱╭━━╯┃╱╭╮╱╱╱╭╮╱╱╭╮\n",
				"╱╰━━━━╯╱╰━━━╯╱╰━━╯╱╰━━━╯╱╰━━━╯╱╱╱╱╰╯╱╰━━━╯╱╰━━━╯╱╱╱╰╯╱╱╰━━━╯╱╰━━━╯╱╰╯╱╱╱╰╯╱╱╰╯\n",
				"\n",
				"╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╭━━━━╮╱╱╱╭━━╮╱╱╱╱╭╮╱╱╭╮╱╱╭━╮╱╱╭━━╮╱╱╱╭━╮╱╭━╮╱╱╱╱╱╱╱╭━╮╱╱╱\n",
				"╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱╱┃╭━━╮┃╱╱╭╯╭╮╰╮╱╭━╯╰╮╱╰╯╱╭╯╭╯╱╱┃╭━╯╱╱╭╯╭╯╱╰╮╰╮╱╱╱╱╱╭╯╭╯╱╭╮\n",
				"╱╱╭╮╱╱╱╱╱╱╱╱╱╱╱╱╭━━━╮╱┃┃╭━┃┃╱╱╰╮┃┃╭╯╱┃╰━━╮╱╱╱╭╯╭╯╱╱╭╯╰╮╱╱╭╯╭╯╱╱╱╰╮╰╮╱╱╱╭╯╭╯╱╱╰╯\n",
				"╱╭╯╰╮╱╭━━╮╱╱╱╱╱╱╰━━━╯╱┃┃╰╯┃┃╱╱╭╯┃┃╰╮╱╰━━╮┃╱╱╭╯╭╯╱╱╱┃╭╮┃╱╱╰╮╰╮╱╱╱╭╯╭╯╱╱╭╯╭╯╱╱╱╭╮\n",
				"╱╰╮╭╯╱╰━━╯╱╭━━╮╱╭━━━╮╱┃╰━━╯━╮╱╰╮╰╯╭╯╱╰╮╭━╯╱╭╯╭╯╱╭╮╱┃╰╯┃╮╱╱╰╮╰╮╱╭╯╭╯╱╱╭╯╭╯╱╱╱╱╰╯\n",
				"╱╱╰╯╱╱╱╱╱╱╱╰━━╯╱╰━━━╯╱╰━━━━━╯╱╱╰━━╯╱╱╱╰╯╱╱╱╰━╯╱╱╰╯╱╰━━━╯╱╱╱╰━╯╱╰━╯╱╱╱╰━╯╱╱╱╱╱╱╱\n",
				"\n",
				"╱╭╮╱╱╱╱╭╮╱╭╮╭╮\n",
				"╱┃┃╱╱╱╱╰╯╱╰╯╰╯\n",
				"╱╰╯╱╱╱╱╱╱╱╱╱╱╱\n",
				"╱╭╮╱╭╮╱╱╱╱╱╱╱╱\n",
				"╱╰┫╱╰┫╱╱╱╱╱╱╱╱\n",
				"╱╱╯╱╱╯╱╱╱╱╱╱╱╱\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"slick\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "slick".to_string(), "-c".to_string(), "red,green".to_string()],
			fixture: concat!("\n\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮╭━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━╮╭━╮\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╭╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰┫┣╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃╰╯┃┃\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰╯╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰╯╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭╮╭╮┃\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃╭━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃┃┃┃\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╰╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰┻━┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭┫┣╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃┃┃┃\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯╰━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯╰╯╰╯\x1B[39m\n",
				"\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╭━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮╭╮╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━╮╭━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╭╮\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m┃┃╰╮┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭╮╭╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰╮╭╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃┃┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╰╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰╮╭╯┃\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m┃╭╮╰╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯┃┃╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮┃┃╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃┃┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╰╯╭╯\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m┃┃╰╮┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭╮╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰╯╰╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╭╯\x1B[39m\x1B[32m╱\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╭╮╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╮╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱╱\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯╰━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━╯╰━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱╱\x1B[39m\n",
				"\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╭━━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╰━━╮━┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯╭╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯╭╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯╭╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯╭╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃┃┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮╰╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m┃╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m┃╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╭╯━╰━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━╯┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╰━━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\n",
				"\n",
				"\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╭━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╭━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━╮\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[31m╭━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╮╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━╯╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃╭━┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮┃┃╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╭╯╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃┃╰╯┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯┃┃╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╭╮┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╰╮╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭━━━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰━━╯━╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╰╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╭━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m┃╰╯┃╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╮╰╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╯╭╯\x1B[39m\x1B[32m╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━╯\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━━━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[31m╰━╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━╯\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰━╯\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\n",
				"\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮╭╮\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰╯╰╯\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╰╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╭╮\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[31m╰┫\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╰┫\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\n",
				"\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[31m╯\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱\x1B[39m\x1B[32m╱\x1B[39m\x1B[32m╱╱╱╱\x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"pallet\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "pallet".to_string()],
			fixture: concat!("\n\n",
				"─╔═══╗─╔══╗──╔═══╗─╔═══╗─╔═══╗─╔═══╗─╔═══╗─╔╗─╔╗─╔══╗───╔╗─╔╗╔═╗─╔╗────╔═╗╔═╗\n",
				"─║╔═╗║─║╔╗║──║╔═╗║─╚╗╔╗║─║╔══╝─║╔══╝─║╔═╗║─║║─║║─╚╣╠╝───║║─║║║╔╝─║║────║║╚╝║║\n",
				"─║║─║║─║╚╝╚╗─║║─╚╝──║║║║─║╚══╗─║╚══╗─║║─╚╝─║╚═╝║──║║────║║─║╚╝╝──║║────║╔╗╔╗║\n",
				"─║╚═╝║─║╔═╗║─║║─╔╗──║║║║─║╔══╝─║╔══╝─║║╔═╗─║╔═╗║──║║──╔╗║║─║╔╗║──║║─╔╗─║║║║║║\n",
				"─║╔═╗║─║╚═╝║─║╚═╝║─╔╝╚╝║─║╚══╗─║║────║╚╩═║─║║─║║─╔╣╠╗─║╚╝║─║║║╚╗─║╚═╝║─║║║║║║\n",
				"─╚╝─╚╝─╚═══╝─╚═══╝─╚═══╝─╚═══╝─╚╝────╚═══╝─╚╝─╚╝─╚══╝─╚══╝─╚╝╚═╝─╚═══╝─╚╝╚╝╚╝\n",
				"\n",
				"─╔═╗─╔╗─╔═══╗─╔═══╗─╔═══╗──╔═══╗─╔═══╗─╔════╗─╔╗─╔╗─╔╗──╔╗─╔╗╔╗╔╗─╔═╗╔═╗─╔╗──╔╗\n",
				"─║║╚╗║║─║╔═╗║─║╔═╗║─║╔═╗║──║╔═╗║─║╔═╗║─║╔╗╔╗║─║║─║║─║╚╗╔╝║─║║║║║║─╚╗╚╝╔╝─║╚╗╔╝║\n",
				"─║╔╗╚╝║─║║─║║─║╚═╝║─║║─║║──║╚═╝║─║╚══╗─╚╝║║╚╝─║║─║║─╚╗║║╔╝─║║║║║║──╚╗╔╝──╚╗╚╝╔╝\n",
				"─║║╚╗║║─║║─║║─║╔══╝─║║─║║──║╔╗╔╝─╚══╗║───║║───║║─║║──║╚╝║──║╚╝╚╝║──╔╝╚╗───╚╗╔╝─\n",
				"─║║─║║║─║╚═╝║─║║────║╚═╝╠╗─║║║╚╗─║╚═╝║───║║───║╚═╝║──╚╗╔╝──╚╗╔╗╔╝─╔╝╔╗╚╗───║║──\n",
				"─╚╝─╚═╝─╚═══╝─╚╝────╚════╝─╚╝╚═╝─╚═══╝───╚╝───╚═══╝───╚╝────╚╝╚╝──╚═╝╚═╝───╚╝──\n",
				"\n",
				"─╔════╗─╔═══╗──╔╗──╔═══╗─╔═══╗─╔╗─╔╗─╔═══╗─╔═══╗─╔═══╗─╔═══╗─╔═══╗─╔╗─╔═══╗───\n",
				"─╚══╗═║─║╔═╗║─╔╝║──║╔═╗║─║╔═╗║─║║─║║─║╔══╝─║╔══╝─║╔═╗║─║╔═╗║─║╔═╗║─║║─║╔═╗║───\n",
				"───╔╝╔╝─║║║║║─╚╗║──╚╝╔╝║─╚╝╔╝║─║╚═╝║─║╚══╗─║╚══╗─╚╝╔╝║─║╚═╝║─║╚═╝║─║║─╚╝╔╝║───\n",
				"──╔╝╔╝──║║║║║──║║──╔═╝╔╝─╔╗╚╗║─╚══╗║─╚══╗║─║╔═╗║───║╔╝─║╔═╗║─╚══╗║─╚╝───║╔╝───\n",
				"─╔╝═╚═╗─║╚═╝║─╔╝╚╗─║║╚═╗─║╚═╝║────║║─╔══╝║─║╚═╝║───║║──║╚═╝║─╔══╝║─╔╗───╔╗──╔╗\n",
				"─╚════╝─╚═══╝─╚══╝─╚═══╝─╚═══╝────╚╝─╚═══╝─╚═══╝───╚╝──╚═══╝─╚═══╝─╚╝───╚╝──╚╝\n",
				"\n",
				"──────────────────────╔════╗───╔╩╩╩╗──╔╝╩╚╗─╔╗──╔═╗───╔╗────╔═╗─╔═╗───────╔═╗───\n",
				"────────────────╔═══╗─║╔══╗║──╔╝╔═╗╚╗─║╔═╗║─╚╝─╔╝╔╝───║║───╔╝╔╝─╚╗╚╗─────╔╝╔╝─╔╗\n",
				"──╔╗──╔══╗──────╚═══╝─║║╔═║║──╚╗╠═╣╔╝─║╚══╗───╔╝╔╝──╔═╝╚╗─╔╝╔╝───╚╗╚╗───╔╝╔╝──╚╝\n",
				"─╔╝╚╗─╚══╝──────╔═══╗─║║╚╝║║──╔╝╠═╣╚╗─╚══╗║──╔╝╔╝───║╔╗╔╝─╚╗╚╗───╔╝╔╝──╔╝╔╝───╔╗\n",
				"─╚╗╔╝──────╔══╗─╚═══╝─║╚══╝╠╗─╚╗╚═╝╔╝─║╚═╝║─╔╝╔╝─╔╗─║╚╝║───╚╗╚╗─╔╝╔╝──╔╝╔╝────╚╝\n",
				"──╚╝───────╚══╝───────╚═════╝──╚╦╦╦╝──╚╗╦╔╝─╚═╝──╚╝─╚══╝────╚═╝─╚═╝───╚═╝───────\n",
				"\n",
				"────╔╗─╔╗─╔╗╔╗\n",
				"────║║─║║─║║║║\n",
				"────╚╝─╚╝─╚╝╚╝\n",
				"─╔╗─╔╗────────\n",
				"─╚╣─╚╣────────\n",
				"──╝──╝────────\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"pallet\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "pallet".to_string(), "-c".to_string(), "red,blue,green".to_string()],
			fixture: concat!("\n\n",
				"\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗╔═╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m───\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═╗╔═╗\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╔╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╣╠╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m───\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║╚╝║║\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚╝╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚╝╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m───\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔╗╔╗║\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║╔═╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║║║║\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╚╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m───\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚╩═║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╣╠╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║║║║\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m───\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝╚═╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝╚╝╚╝\x1B[39m\n",
				"\n",
				"\x1B[34m─\x1B[39m\x1B[31m╔═╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔════╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗╔╗╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═╗╔═╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╔╗\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m║║╚╗║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔╗╔╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚╗╔╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║║║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╚╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚╗╔╝║\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m║╔╗╚╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝║║╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗║║╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║║║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╚╝╔╝\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m║║╚╗║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔╗╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚══╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚╝╚╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╔╝\x1B[39m\x1B[34m─\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m───\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝╠╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╔╗╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔╗╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m──\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m───\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚════╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝╚═╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═╝╚═╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m──\x1B[39m\n",
				"\n",
				"\x1B[34m─\x1B[39m\x1B[31m╔════╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m╚══╗═║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝╔╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝╔╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝╔╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝╔╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗╚╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚══╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚══╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m║╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚══╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m║╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m╔╝═╚═╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║╚═╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m───\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔══╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔══╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m╚════╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m───\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\n",
				"\n",
				"\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─────\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔════╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╩╩╩╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╩╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╔═╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╔═╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═╗\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\x1B[31m╔═╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔══╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔═╗╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔═╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[34m───\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║╔═║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╠═╣╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═╝╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m╔╝╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔═══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║╚╝║║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╠═╣╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚══╗║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╔╗╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m╚╗╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔══╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚══╝╠╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╚═╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚═╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║╚╝║\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╚╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╝╔╝\x1B[39m\x1B[34m───\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─────\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═════╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╦╦╦╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╗╦╔╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═╝\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚══╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[31m╚═╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═╝\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚═╝\x1B[39m\x1B[34m────\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\n",
				"\n",
				"\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗╔╗\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║\x1B[39m\x1B[34m─\x1B[39m\x1B[31m║║║║\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╝╚╝\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╔╗\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[31m╚╣\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╚╣\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\n",
				"\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m─\x1B[39m\x1B[31m╝\x1B[39m\x1B[34m─\x1B[39m\x1B[34m──\x1B[39m\x1B[34m─\x1B[39m\x1B[34m────\x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"grid\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "grid".to_string()],
			fixture: concat!("\n\n",
				"╋╋╋╋╋╋┏┓╋╋╋╋╋╋╋╋╋╋┏┓╋╋╋╋╋╋╋┏━┓╋╋╋╋╋╋┏┓╋╋╋┏┓╋╋┏┓╋┏┓╋╋╋┏┓╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋\n",
				"╋┏━━┓╋┃┗━┓╋┏━━┓╋┏━┛┃╋┏━━┓╋┏┛┗┓╋┏━━┓╋┃┗━┓╋┗┛╋╋┗┛╋┃┃┏┓╋┃┃╋╋┏┓┏┓╋┏━┓╋╋┏━━┓╋┏━━┓\n",
				"╋┃┏┓┃╋┃┏┓┃╋┃┏━┛╋┃┏┓┃╋┃┃━┫╋┗┓┏┛╋┃┏┓┃╋┃┏┓┃╋┏┓╋╋┏┓╋┃┗┛┛╋┃┃╋╋┃┗┛┃╋┃┏┓┓╋┃┏┓┃╋┃┏┓┃\n",
				"╋┃┏┓┃╋┃┗┛┃╋┃┗━┓╋┃┗┛┃╋┃┃━┫╋╋┃┃╋╋┃┗┛┃╋┃┃┃┃╋┃┃╋╋┃┃╋┃┏┓┓╋┃┗┓╋┃┃┃┃╋┃┃┃┃╋┃┗┛┃╋┃┗┛┃\n",
				"╋┗┛┗┛╋┗━━┛╋┗━━┛╋┗━━┛╋┗━━┛╋╋┗┛╋╋┗━┓┃╋┗┛┗┛╋┗┛╋┏┛┃╋┗┛┗┛╋┗━┛╋┗┻┻┛╋┗┛┗┛╋┗━━┛╋┃┏━┛\n",
				"╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋┗━━┛╋╋╋╋╋╋╋╋╋┗━┛╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋┗┛╋╋\n",
				"\n",
				"╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋┏┓╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋┏━━━┓╋╋┏┓╋╋┏━━━┓╋┏━━━┓\n",
				"╋┏━━┓╋┏━┓╋┏━━┓╋┏┛┗┓╋┏┓┏┓╋┏┓┏┓╋┏┓┏┓┏┓╋┏┓┏┓╋┏┓╋┏┓╋┏━━━┓╋┃┏━┓┃╋┏┛┃╋╋┃┏━┓┃╋┃┏━┓┃\n",
				"╋┃┏┓┃╋┃┏┛╋┃━━┫╋┗┓┏┛╋┃┃┃┃╋┃┗┛┃╋┃┗┛┗┛┃╋┗╋╋┛╋┃┗━┛┃╋┣━━┃┃╋┃┃┃┃┃╋┗┓┃╋╋┗┛┏┛┃╋┗┛┏┛┃\n",
				"╋┃┗┛┃╋┃┃╋╋┣━━┃╋╋┃┗┓╋┃┗┛┃╋┗┓┏┛╋┗┓┏┓┏┛╋┏╋╋┓╋┗━┓┏┛╋┃┃━━┫╋┃┃┃┃┃╋╋┃┃╋╋┏━┛┏┛╋┏┓┗┓┃\n",
				"╋┗━┓┃╋┗┛╋╋┗━━┛╋╋┗━┛╋┗━━┛╋╋┗┛╋╋╋┗┛┗┛╋╋┗┛┗┛╋┗━━┛╋╋┗━━━┛╋┃┗━┛┃╋┏┛┗┓╋┃┗━┻┓╋┃┗━┛┃\n",
				"╋╋╋┗┛╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋┗━━━┛╋┗━━┛╋┗━━━┛╋┗━━━┛\n",
				"\n",
				"╋┏┓╋┏┓╋┏━━━┓╋┏━━━┓╋┏━━━┓╋┏━━━┓╋┏━━━┓╋┏┓╋┏━━━┓╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋╋┏━━━━┓╋\n",
				"╋┃┃╋┃┃╋┃┏━━┛╋┃┏━━┛╋┃┏━┓┃╋┃┏━┓┃╋┃┏━┓┃╋┃┃╋┃┏━┓┃╋╋╋╋╋┏┓╋╋╋╋╋╋╋╋╋╋╋╋┏━━━┓╋┃┏━━┓┃╋\n",
				"╋┃┗━┛┃╋┃┗━━┓╋┃┗━━┓╋┗┛┏┛┃╋┃┗━┛┃╋┃┗━┛┃╋┃┃╋┗┛┏┛┃╋╋╋╋┏┛┗┓╋┏━━┓╋╋╋╋╋╋┗━━━┛╋┃┃┏━┃┃╋\n",
				"╋┗━━┓┃╋┗━━┓┃╋┃┏━┓┃╋╋╋┃┏┛╋┃┏━┓┃╋┗━━┓┃╋┗┛╋╋╋┃┏┛╋╋╋╋┗┓┏┛╋┗━━┛╋╋╋╋╋╋┏━━━┓╋┃┃┗┛┃┃╋\n",
				"╋╋╋╋┃┃╋┏━━┛┃╋┃┗━┛┃╋╋╋┃┃╋╋┃┗━┛┃╋┏━━┛┃╋┏┓╋╋╋┏┓╋╋┏┓╋╋┗┛╋╋╋╋╋╋╋┏━━┓╋┗━━━┛╋┃┗━━┛┗┓\n",
				"╋╋╋╋┗┛╋┗━━━┛╋┗━━━┛╋╋╋┗┛╋╋┗━━━┛╋┗━━━┛╋┗┛╋╋╋┗┛╋╋┗┛╋╋╋╋╋╋╋╋╋╋╋┗━━┛╋╋╋╋╋╋╋┗━━━━━┛\n",
				"\n",
				"╋╋┏━━━┓╋╋╋┏┓╋╋┏┓╋╋┏━┓╋╋╋┏┓╋╋╋╋┏━┓╋┏━┓╋╋╋╋╋╋╋┏━┓╋╋╋╋╋╋╋╋╋╋┏┓╋┏┓┏┓\n",
				"╋┏┛┏━┓┗┓╋┏┛┗┓╋┗┛╋┏┛┏┛╋╋╋┃┃╋╋╋┏┛┏┛╋┗┓┗┓╋╋╋╋╋┏┛┏┛╋┏┓╋┏┓╋╋╋╋┗┛╋┗┛┗┛\n",
				"╋┗┓┃┃┃┏┛╋┃━━┫╋╋╋┏┛┏┛╋╋┏━┛┗┓╋┏┛┏┛╋╋╋┗┓┗┓╋╋╋┏┛┏┛╋╋┗┛╋┗┛╋╋╋╋╋╋╋╋╋╋╋\n",
				"╋┏┛┃┃┃┗┓╋┣━━┃╋╋┏┛┏┛╋╋╋┃┏┓┏┛╋┗┓┗┓╋╋╋┏┛┏┛╋╋┏┛┏┛╋╋╋┏┓╋╋╋╋╋╋╋╋╋╋╋╋╋╋\n",
				"╋┗┓┗━┛┏┛╋┗┓┏┛╋┏┛┏┛╋┏┓╋┃┗┛┃╋╋╋┗┓┗┓╋┏┛┏┛╋╋┏┛┏┛╋╋╋╋┗┛╋┏┓╋┏┓╋╋╋╋╋╋╋╋\n",
				"╋╋┗━━━┛╋╋╋┗┛╋╋┗━┛╋╋┗┛╋┗━━┛╋╋╋╋┗━┛╋┗━┛╋╋╋┗━┛╋╋╋╋╋╋╋╋┗┫╋┗┫╋╋╋╋╋╋╋╋\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"grid\" - with color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "grid".to_string(), "-c".to_string(), "red,blue,green".to_string()],
			fixture: concat!("\n\n",
				"\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┏━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━┓\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃━┫\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┓┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┃\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃━┫\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┃\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┗┛┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┻┻┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━┛\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋╋\x1B[39m\n",
				"\n",
				"\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┏━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓┏┓┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━┓┃\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃━━┫\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┓┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┗┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗╋╋┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┣━━┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃┃┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛┏┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛┏┛┃\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┣━━┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┓┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┓┏┓┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏╋╋┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━┓┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃━━┫\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃┃┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━┛┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓┗┓┃\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┗━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━┻┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━┛┃\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━┛\x1B[39m\n",
				"\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━━┓\x1B[39m\x1B[34m╋\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━━┓┃\x1B[39m\x1B[34m╋\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┃┗━┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛┏┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛┏┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃┏━┃┃\x1B[39m\x1B[34m╋\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┗━━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┃┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┓┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┃┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┓┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┃┗┛┃┃\x1B[39m\x1B[34m╋\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[34m╋╋╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗━━┛┗┓\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[34m╋╋╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━━━┛\x1B[39m\n",
				"\n",
				"\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━━━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┏━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┏━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━┓\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[31m┏━┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓┏┓\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┏┛┏━┓┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┃┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┓┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛┗┛\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┗┓┃┃┃┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃━━┫\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏━┛┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┓┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┏┛┃┃┃┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┣━━┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┏┓┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┓┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[31m┗┓┗━┛┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┓┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┃┗┛┃\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┓┗┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┛┏┛\x1B[39m\x1B[34m╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┏┓\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\n",
				"\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━┛\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┗┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[31m┗━┛\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━┛\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗━┛\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┫\x1B[39m\x1B[34m╋\x1B[39m\x1B[31m┗┫\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋\x1B[39m\x1B[34m╋\x1B[39m\x1B[34m╋╋╋╋\x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"tiny\" - no color"),
			args: vec![supported_characters.clone(), "--font".to_string(), "tiny".to_string()],
			fixture: concat!("\n\n",
				" ▄▀█ █▄▄ █▀▀ █▀▄ █▀▀ █▀▀ █▀▀ █ █ █   █ █▄▀ █   █▀▄▀█ █▄ █ █▀█ █▀█ █▀█ █▀█ █▀▀\n",
				" █▀█ █▄█ █▄▄ █▄▀ ██▄ █▀  █▄█ █▀█ █ █▄█ █ █ █▄▄ █ ▀ █ █ ▀█ █▄█ █▀▀ ▀▀█ █▀▄ ▄▄█\n",
				"\n",
				" ▀█▀ █ █ █ █ █ █ █ ▀▄▀ █▄█ ▀█ ▞█▚ ▄█ ▀█ ▀▀█ █ █ █▀ █▄▄ ▀▀█ ███ █▀█ █ ▀█   ▄█▄ ▄▄\n",
				"  █  █▄█ ▀▄▀ ▀▄▀▄▀ █ █  █  █▄ ▚█▞  █ █▄ ▄██ ▀▀█ ▄█ █▄█   █ █▄█ ▀▀█ ▄  ▄ ▄  ▀    \n",
				"\n",
				"    ▀▀ ▛█▜ ▟▄▙ ▖█▗ ▀ ▄▀ ▄▄█ ▄▀ ▀▄   ▄▀ ▀      ▀ ▛ ▜\n",
				" ▄▄ ▀▀ ▙▟▃ ▜▀▛ ▘█▝ ▄▀ ▄ █▄█ ▀▄ ▄▀ ▄▀   ▄ ▄▀ █      \n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
		Test {
			name: String::from("Font test: \"tiny\" - with color"),
			args: vec![supported_characters, "--font".to_string(), "tiny".to_string(), "-c".to_string(), "red".to_string()],
			fixture: concat!("\n\n",
				"\x1B[31m \x1B[39m\x1B[31m▄▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m█ █\x1B[39m\x1B[31m \x1B[39m\x1B[31m█\x1B[39m\x1B[31m \x1B[39m\x1B[31m  █\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m█  \x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀▄▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄ █\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀▀\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m█▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m██▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀ \x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█ █\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m█ ▀ █\x1B[39m\x1B[31m \x1B[39m\x1B[31m█ ▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄▄█\x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m▀█▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m█ █\x1B[39m\x1B[31m \x1B[39m\x1B[31m█ █\x1B[39m\x1B[31m \x1B[39m\x1B[31m█ █ █\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▄▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m▞█▚\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█ █\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m███\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m \x1B[39m\x1B[31m \x1B[39m\x1B[31m▄█▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄▄\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m █ \x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▄▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▄▀▄▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m█ █\x1B[39m\x1B[31m \x1B[39m\x1B[31m █ \x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m▚█▞\x1B[39m\x1B[31m \x1B[39m\x1B[31m █\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄██\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m  █\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▀█\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m ▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m ▀ \x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\n",
				"\n",
				"\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m▛█▜\x1B[39m\x1B[31m \x1B[39m\x1B[31m▟▄▙\x1B[39m\x1B[31m \x1B[39m\x1B[31m▖█▗\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀ ▄▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m  ▄▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m  \x1B[39m\x1B[31m \x1B[39m\x1B[31m \x1B[39m\x1B[31m \x1B[39m\x1B[31m▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m▛ ▜\x1B[39m\n",
				"\x1B[31m \x1B[39m\x1B[31m▄▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m▙▟▃\x1B[39m\x1B[31m \x1B[39m\x1B[31m▜▀▛\x1B[39m\x1B[31m \x1B[39m\x1B[31m▘█▝\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄▀ ▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m█▄█\x1B[39m\x1B[31m \x1B[39m\x1B[31m▀▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄▀  \x1B[39m\x1B[31m \x1B[39m\x1B[31m▄\x1B[39m\x1B[31m \x1B[39m\x1B[31m▄▀\x1B[39m\x1B[31m \x1B[39m\x1B[31m█\x1B[39m\x1B[31m \x1B[39m\x1B[31m \x1B[39m\x1B[31m \x1B[39m\x1B[31m   \x1B[39m\n",
				"\n\n").to_string(),
			force_color: String::from(""),
			no_color: false,
		},
	];

	tests
}
