//! Internal implementation details for the host-guest interface.
//!
//! Note that the vast majority of this module is just providing FFI-safe
//! versions of common `std` types (e.g. `Vec`, `String`, and `Box<dyn Error>`),
//! or FFI-safe trait objects.
//!
/// If the macro generated the wrong code, this doctest would fail.
///
/// ```rust
/// use fj::{abi::INIT_FUNCTION_NAME, models::Metadata};
///
/// fj::register_model!(|_| {
///     Ok(Metadata::new("My Model", "1.2.3"))
/// });
///
/// mod x {
///     extern "C" {
///         pub fn fj_model_init(_: *mut fj::abi::Host<'_>) -> fj::abi::InitResult;
///     }
/// }
///
/// // make sure our function has the right signature
/// let func: fj::abi::InitFunction = fj_model_init;
///
/// // We can also make sure the unmangled name is correct by calling it.
///
/// let metadata: fj::models::Metadata = unsafe {
///     let mut host = Host;
///     let mut host = fj::abi::Host::from(&mut host);
///     x::fj_model_init(&mut host).unwrap().into()
/// };
///
/// assert_eq!(metadata.name, "My Model");
///
/// struct Host;
/// impl fj::models::Host for Host {
///     fn register_boxed_model(&mut self, model: Box<dyn fj::models::Model>) { todo!() }
/// }
/// ```
mod context;
pub mod ffi_safe;
mod host;
mod metadata;
mod model;

use backtrace::Backtrace;
use std::{any::Any, fmt::Display, panic, sync::Mutex};

pub use self::{
    context::Context,
    host::Host,
    metadata::{Metadata, ModelMetadata},
    model::Model,
};

/// Define the initialization routine used when registering models.
///
/// See the [`crate::model`] macro if your model can be implemented as a pure
/// function.
///
/// # Examples
///
/// ```rust
/// use fj::models::*;
///
/// fj::register_model!(|host: &mut dyn Host| {
///     host.register_model(MyModel::default());
///
///     Ok(Metadata::new(env!("CARGO_PKG_NAME"), env!("CARGO_PKG_VERSION")))
/// });
///
/// #[derive(Default)]
/// struct MyModel { }
///
/// impl Model for MyModel {
///     fn metadata(&self) -> std::result::Result<fj::models::ModelMetadata, Box<(dyn std::error::Error + Send + Sync + 'static)>> { todo!() }
///
///     fn shape(&self, ctx: &dyn Context) -> Result<fj::Shape, Error> {
///         todo!()
///     }
/// }
/// ```
#[macro_export]
macro_rules! register_model {
    ($init:expr) => {
        #[no_mangle]
        unsafe extern "C" fn fj_model_init(
            mut host: *mut $crate::abi::Host<'_>,
        ) -> $crate::abi::InitResult {
            let init: fn(
                &mut dyn $crate::models::Host,
            ) -> Result<
                $crate::models::Metadata,
                $crate::models::Error,
            > = $init;

            match init(&mut *host) {
                Ok(meta) => $crate::abi::InitResult::Ok(meta.into()),
                Err(e) => $crate::abi::InitResult::Err(e.into()),
            }
        }
    };
}

/// The signature of the function generated by [`register_model`].
///
/// ```rust
/// fj::register_model!(|_| { todo!() });
///
/// const _: fj::abi::InitFunction = fj_model_init;
/// ```
pub type InitFunction = unsafe extern "C" fn(*mut Host<'_>) -> InitResult;
pub type InitResult = ffi_safe::Result<Metadata, ffi_safe::BoxedError>;
pub type ShapeResult = ffi_safe::Result<crate::Shape, ffi_safe::BoxedError>;
pub type ModelMetadataResult =
    ffi_safe::Result<ModelMetadata, ffi_safe::BoxedError>;

/// The name of the function generated by [`register_model`].
///
pub const INIT_FUNCTION_NAME: &str = "fj_model_init";

// Contains details about a panic that we need to pass back to the application from the panic hook.
struct PanicInfo {
    message: Option<String>,
    location: Option<Location>,
    backtrace: Backtrace,
}

impl Display for PanicInfo {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        let message = self
            .message
            .as_ref()
            .map_or("No error given", |message| message.as_str());

        write!(f, "\"{}\", ", message)?;

        if let Some(location) = self.location.as_ref() {
            write!(f, "{}", location)?;
        } else {
            write!(f, "no location given")?;
        }

        writeln!(f, "\nBacktrace:\n{:?}", self.backtrace)?;

        Ok(())
    }
}

struct Location {
    file: String,
    line: u32,
    column: u32,
}

impl Display for Location {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}:{}:{}", self.file, self.line, self.column)
    }
}

static LAST_PANIC: Mutex<Option<PanicInfo>> = Mutex::new(None);

/// Capturing panics is something Rust really doesn't want you to do, and as such, they make it convoluted.
/// This sets up all the machinery in the background to pull it off.
///
/// It's okay to call this multiple times.
pub fn initialize_panic_handling() {
    panic::set_hook(Box::new(|panic_info| {
        let mut last_panic =
            LAST_PANIC.lock().expect("Panic queue was poisoned."); // FIXME that can probably overflow the stack.
        let message = if let Some(s) =
            panic_info.payload().downcast_ref::<std::string::String>()
        {
            Some(s.as_str())
        } else {
            panic_info.payload().downcast_ref::<&str>().copied()
        }
        .map(|s| s.to_string());

        let location = panic_info.location().map(|location| Location {
            file: location.file().to_string(),
            line: location.line(),
            column: location.column(),
        });

        let backtrace = backtrace::Backtrace::new();
        *last_panic = Some(PanicInfo {
            message,
            location,
            backtrace,
        });
    }));
}

fn on_panic(_payload: Box<dyn Any + Send>) -> crate::abi::ffi_safe::String {
    // The payload is technically no longer needed, but I left it there just in case a change to `catch_unwind` made
    // it useful again.
    if let Ok(mut panic_info) = LAST_PANIC.lock() {
        if let Some(panic_info) = panic_info.take() {
            crate::abi::ffi_safe::String::from(format!(
                "Panic in model: {}",
                panic_info
            ))
        } else {
            crate::abi::ffi_safe::String::from(
                "Panic in model: No details were given.".to_string(),
            )
        }
    } else {
        crate::abi::ffi_safe::String::from(
            "Panic in model, but due to a poisoned panic queue the information could not be collected."
        .to_string())
    }
}
