(* 
  HereDoc
  Alain Frisch  <Alain.Frisch@ens.fr>
*)

{
  type loc = int * int

  type here_clause =
    | Literal of string
    | Expr of (string * loc)
    | ExtVar of (string * loc)
    | Conditional of (string * loc)
    | Abstraction of (string * loc)
    | Special of (string * loc)
    | Textend

  let antiquot_buffer = Buffer.create 512

  let extract lexbuf strip_begin strip_end =
    let s = Lexing.lexeme lexbuf in
    let l = String.length s in
    let loc = 
      Lexing.lexeme_start lexbuf + strip_begin,
      Lexing.lexeme_end lexbuf - strip_end in

    (String.sub s strip_begin (l - strip_begin - strip_end), loc)

}

let letter = [ 'A' - 'Z' 'a' - 'z' '_' ] 
let acletter = letter | ['0'-'9' '\''] 

rule token = parse
  | '$' ( letter acletter* '.' )* letter acletter*
      { Expr (extract lexbuf 1 0) }
  | '$' '$'
      { 
	let beg = (Lexing.lexeme_start lexbuf + 2) in
	let finish = antiquot lexbuf in
	let s = Buffer.contents antiquot_buffer in
	Buffer.clear antiquot_buffer;
	Expr (s, (beg, finish))
      }

  | '$' '?'
      {
	let beg = (Lexing.lexeme_start lexbuf + 2) in
	let finish = condition lexbuf in
	let s = Buffer.contents antiquot_buffer in
	Buffer.clear antiquot_buffer;
	Conditional (s, (beg, finish))
      }

  | '$' '['
      {
	let beg = (Lexing.lexeme_start lexbuf + 2) in
	let finish = abstraction lexbuf in
	let s = Buffer.contents antiquot_buffer in
	Buffer.clear antiquot_buffer;
	Abstraction (s, (beg, finish))
      }

  | '$' '{' [^ '}']*  '}'      { Special (extract lexbuf 2 1) }

  | '$' '(' [^ ')']*  ')'      { ExtVar (extract lexbuf 2 1) }

  | '$'             { failwith "Bad $ expander" }

  | '\\' '\n'       { Literal "" }
  | '\\' '$'        { Literal "$" }
  | '\\' '\\'       { Literal "\\\\" }
  | '\\' '$' '$'    { Literal "$$" }
  | '\\' _          { Literal (Lexing.lexeme lexbuf) }
  | [^ '$' '\\']+   { Literal (Lexing.lexeme lexbuf) }
  | eof             { Textend }
  | _               { Literal (Lexing.lexeme lexbuf) }

and antiquot = parse
  | '\\' '$' '$'
      {
	Buffer.add_string antiquot_buffer "$$";
	antiquot lexbuf
      }
  | '\\' '\\'
      {
	Buffer.add_char antiquot_buffer '\\';
	antiquot lexbuf
      }
  | '$' '$'
      { Lexing.lexeme_start lexbuf }
  | _
      { 
	Buffer.add_string antiquot_buffer (Lexing.lexeme lexbuf);
        antiquot lexbuf
      }

and condition = parse
  | '\\' ':'
      {
	Buffer.add_string antiquot_buffer ":";
	condition lexbuf
      }
  | '\\' '\\'
      {
	Buffer.add_char antiquot_buffer '\\';
	condition lexbuf
      }
  | ':'
      { Lexing.lexeme_start lexbuf }
  | _
      { 
	Buffer.add_string antiquot_buffer (Lexing.lexeme lexbuf);
        condition lexbuf
      }

and abstraction = parse
  | '\\' ']' '$'
      {
	Buffer.add_string antiquot_buffer "]$";
	abstraction lexbuf
      }
  | '\\' '\\'
      {
	Buffer.add_char antiquot_buffer '\\';
	abstraction lexbuf
      }
  | ']' '$'
      { Lexing.lexeme_start lexbuf }
  | _
      { 
	Buffer.add_string antiquot_buffer (Lexing.lexeme lexbuf);
        abstraction lexbuf
      }


